#!/usr/bin/perl -w
# ------------------------------------------------------------------------------
#   ident "@(#)smct.pl 1.9     02/10/18 SMI"
# ------------------------------------------------------------------------------

BEGIN { 
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_softrepo.pm";  # software repository
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_swlrepo.pm";   # swl repository
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_package.pm";   # pkg class
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_software.pm";  # pkg class
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_swl.pm";       # swl class
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_utils.pl";     # tools
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_parser.pl"; # conf. parser
   require "$ENV{NHAS_PROD_DIR}/nhsmct/lib/sl_traces.pl";        # jumpstart
}

# ------------------------------------------------------------------------------

package main;

# SLB environment variables (point to working env)
#
my @SLBENV = ( "SMCT_SOL_DIR",
	       "SMCT_ENV_DIR",
	       "NHAS_PROD_DIR",
	       "SMCT_CONFIG_DIR",
	       "SMCT_DEFAULT_CONFIG_DIR",
	       "SMCT_SCRIPT_DIR",
	       "SMCT_EXPORT_DIR",
	       "SMCT_SOFTREP_DIR",
	       "SMCT_SWLREP_DIR",
	       "SMCT_IMPORT_DIR",
	       "SMCT_JUMPSTART_DIR",
	       "SMCT_FLASH_DIR",
	       "SMCT_TMP_DIR" );

# Operations Type
$CREATE = "create";
$CONFIG = "config";
$DEPLOY = "deploy";

# Input parameters
#
   $OPERATION = undef;   # Operation to perform
my $VERBOSE = 1;         # Default verbosity level
my $DEBUG = 1;           # Default debug level
my $REMOTE = undef;      # Default export mode
my $CONFIG_DIR = undef;  # directory where swl configuration files are stored
my $EXPORT_DIR = undef;  # Export directory
my $IMPORT_DIR = undef;  # Import directory
my $SWL_VERSION = undef; # Software load version to export or delete
my $SWL_NAME = undef;    # Software load name to export or delete
my $FORCE = undef;       # Force the swl data removal
my $BRIEF = undef;       # Force a short listing to be displayed
my $ALL = undef;         # Force a whole listing to be displayed
my $SWL = undef;         # SWL repo flag
my $SOFT = undef;        # Software repo flag
my $LOG_FILE = undef;    # log file

# Global variables
my $OPSTAT = undef;     # slb operation status

# DB objects mapping on the repositories databases
#
$SOFTREP = undef;    # DBM database associated to the software repository
$SWLREP = undef;     # DBM database associated to the software load repository

# ------------------------------------------------------------------------------
# 'usage' Displays usage of the smct commands
#
# Parameter(s)
#
# @_ : array of error messages
#
sub usage {
   ( $errorMsg, $errorLine,  @paramLst ) = @_;
   $OPERATION = 'help' unless ( defined $OPERATION );
   &sl_traces::message( "\nUsage:\n\n" );
   if ( ( $OPERATION eq $CREATE ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\tslcreate -n <swl-name> -v <swl-version> ".
                   "[ -c <config-dir> ] [ -f ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   }
   if ( ( $OPERATION eq 'export' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tslexport -n <swl-name> -v <swl-version> ".
                   "[ -e <export-dir> ] [ -r ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   } 
   if ( ( $OPERATION eq 'import' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tslimport -n <swl-name> -v <swl-version> ".
                   "[ -i <import-dir> ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   } 
   if ( ( $OPERATION eq 'delete' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tsldelete -n <swl-name> -v <swl-version> [ -f ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   }
   if ( ( $OPERATION eq $CONFIG ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tslconfig -n <swl-name> -v <swl-version> ".
                   "[ -c <config-dir> ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   }
   if ( ( $OPERATION eq $DEPLOY ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tsldeploy -n <swl-name> -v <swl-version> ".
                   "[ -c <config-dir> ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   }
   if ( ( $OPERATION eq 'list' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "\n\tsllist -s ".
                   "[ -n <swl-name> -v <swl-version> ] [ -b ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
     &sl_traces::message(  "\n\tsllist -p [ -b ] ".
                   "\n\t\t[ -l <log-file> ] [ -h ] [ -V <verbosity-level> ]\n" );
   }
  &sl_traces::message(  "\nParameters:\n\n" );
   if ( ( $OPERATION eq 'delete' ) or
        ( $OPERATION eq $CREATE ) or
        ( $OPERATION eq 'export' ) or
        ( $OPERATION eq $DEPLOY ) or
        ( $OPERATION eq $CONFIG ) or
        ( $OPERATION eq 'import' ) or
	( $OPERATION eq 'list' ) or 
	( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-n <swl-name>\n\tsoftware load name.\n\n" );
      &sl_traces::message(  "-v <swl-version>\n\tsoftware load version.\n\n" );
   }
   if ( ( $OPERATION eq $CREATE ) or 
        ( $OPERATION eq $CONFIG ) or 
        ( $OPERATION eq $DEPLOY ) or 
        ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-c <config-dir>\n\tDirectory where the configuration ".
                   "files are stored.\n\n" );
   }
   if ( ( $OPERATION eq 'export' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-e <export-dir>\n\tdirectory where the software load\n" );
      &sl_traces::message(  "\tdata are exported.\n\n" );
      &sl_traces::message(  "-r\n\tif this option is set the " );
      &sl_traces::message(  "software\n\tis copied to <export-dir>.\n\n" );
   }
   if ( ( $OPERATION eq 'import' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-i <import-dir>\n\tdirectory where the software load \n" );
      &sl_traces::message(  "\tdata to export are stored.\n\n" );
   }
   if ( ( $OPERATION eq 'list' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-s\n\tselect the software load repository.\n\n" );
      &sl_traces::message(  "-p\n\tselect the software repository.\n\n" );
   }
   if ( ( $OPERATION eq 'delete' ) or
        ( $OPERATION eq $CREATE ) or
        ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-f\n\tif this option is set all the the\n" );
      &sl_traces::message(  "\tsoftware load data are deleted.\n\n" );
   }
   if ( ( $OPERATION eq 'list' ) or ( $OPERATION eq 'help' ) ) {
      &sl_traces::message(  "-b\n\tif this option is set a short listing\n" );
      &sl_traces::message(  "\tis displayed.\n\n" );
   }
   &sl_traces::message(  "-l <log-file>\n\tspecifies the name of the file ".
                "receiving \n\tinformation messages (default ".
                "is stdout).\n\tError messages are output to stderr.\n\n" );
   &sl_traces::message(  "-V <verbosity-level>\n\tset the verbosity level. Verbosity ".
                "level is a value between 1 and 3.\n\n" );
   &sl_traces::message(  "-h\n\tprint this help.\n\n" );

   &sl_traces::error( $errorMsg, $errorLine, @paramLst ) 
      if ( defined  $errorMsg );
   exit 0;
}

# ------------------------------------------------------------------------------
# 'checkEnv' reads the environment variables set by the ssmct setup
#  returns "" if all the variables are set ; the first unset variable
#  in the other cases.
#
# Parameter(s)
#
# -
#
sub checkEnv {
   foreach $key ( @SLBENV ) {
      return $key unless ( defined $ENV{$key} );
   }
   return "";
}

# ------------------------------------------------------------------------------
# 'retrieveSoftware' retrieve packages. Only packages that have changed or that
# do not exist are retrieved.
#
# Parameter(s)
#
# $softLst : list of software to retrieve
# $swlId   : current software load ID 
#
sub retrieveSoftware {
   my ( $swlId, %softLst ) = @_;
   my $status = 0;
   &sl_traces::info( 2, "Retrieving software load < $swlId > ".
                     "software" );
   foreach $key ( keys %softLst ) {
      $soft = $softLst{$key};
      # Look if the software exists
      $softInst = $SOFTREP->lookUp( $soft );
      unless ( defined $softInst ) {
	 # Software does not exist: let's fetch it
	 if ( !$soft->fetch ) {
            # Reference current swl within the software object
	    $soft->setState( $sl_software::SOFTWARE_STATE{'TRANSFERED'} );
	    # Add software to the software repository
	    $SOFTREP->add( $soft );
	 }
	 else {
	    # Software retrieval has failed
            # Note: It is acceptable to fail when fetching a software
            # This code authorizes several trials in oder to 
            # gather all the software included in a software load
	    $status = 1;  
	 }
      }
   }
   return $status;
}

# ------------------------------------------------------------------------------
# 'sl_config' configure the software load with user defined config. data
#
sub sl_config {
   my ( $swlName, $swlVersion ) = @_;
   my $status = 0;
   &sl_traces::info( 1, "Configuring software load < %s > ".
                     "version < %s >",
		     ( $swlName, $swlVersion ) );
   # Create a swl bootstrap object
   $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
   # Check if the software load exists in the repository
   my $swlInst = $SWLREP->lookUp( $swl );
   &sl_traces::error( "Software Load < %s > ".
	              "version < %s > does not exist",
		      "- smct::sl_config -".__LINE__,
		      ( $swlName, $swlVersion ) )
      unless ( defined $swlInst );
   # Create a cluster object from the configuration files
   my @configFileList = ( ${sl_parser::MACHINE_CONF},
                          ${sl_parser::CLUSTER_CONF},
                          ${sl_parser::NETWORK_CONF} );
   &sl_parser::load( @configFileList );
   my $cluster = undef;
   ( $cluster ) = &sl_parser::getObjects( "sl_cluster", ( 0 ) );
   &sl_traces::error( "Unable to get cluster definition ".
                      "from the configuration files",
		      "- smct::sl_config -".__LINE__ )
      unless ( defined $cluster );
   # Check that the cluster generic definition is consistent with the current 
   # configuration data
   my $clusterRef = $swlInst->getCluster;
   my $differences = sl_cluster->compare( $clusterRef, $cluster );
   &sl_traces::error( "Current cluster definition is different from ".
                      "reference cluster definition ".
		      "\n%s",
		      "- smct::sl_config -".__LINE__,
		      ( $differences ) )   
      if ( defined $differences );
   # Update cluster reference definition with group config. data
   $clusterRef->update( $cluster, $OPERATION );
   # Update swl definition
   $swlInst->setCluster( $clusterRef );
   # Clean the configuration swlrepo part
   $SWLREP->clean( $sl_swlrepo::SWLREPO_PART{'CONFIG'}, $swlInst );   
   # Get configuration files
   $status = $swlInst->configure;
   return $status;
}

# ------------------------------------------------------------------------------
# 'sl_deploy' configure the FS associated to a software load
#
sub sl_deploy {
   my ( $swlName, $swlVersion ) = @_;
   my $status = 0;
   &sl_traces::info( 1, "Deploying software load < %s > ".
                     "version < %s >", 
		     ( $swlName, $swlVersion ) );
   # Create a swl bootstrap object
   $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
   # Check if the software load exists in the repository
   my $swlInst = $SWLREP->lookUp( $swl );
   &sl_traces::error( "Software Load < %s > ".
	              "version < %s > does not exist",
		      "- smct::sl_deploy -".__LINE__, 
		      ( $swlName, $swlVersion ) )
      unless ( defined $swlInst );
   # Create a cluster object from the configuration files
   my @configFileList = ( ${sl_parser::MACHINE_CONF},
                          ${sl_parser::CLUSTER_CONF},
                          ${sl_parser::NETWORK_CONF} );
   &sl_parser::load( @configFileList );
   my $cluster = undef;
   ( $cluster ) = &sl_parser::getObjects( "sl_cluster", ( 0 ) );
   &sl_traces::error( "Unable to get cluster definition ".
                      "from the configuration files",
		      "- smct::sl_deploy -".__LINE__ )
      unless ( defined $cluster );
   # Check that the cluster generic definition is consistent with the current 
   # configuration data
   my $clusterRef = $swlInst->getCluster;
   my $differences = sl_cluster->compare( $clusterRef, $cluster );
   &sl_traces::error( "Current cluster definition is different from ".
                      "reference cluster definition ".
		      "\n%s",
		      "- smct::sl_deploy -".__LINE__,
		      ( $differences ) )   
      if ( defined $differences );
   # Update cluster reference definition with group config. data
   $clusterRef->update( $cluster, $OPERATION );
   # Update swl definition
   $swlInst->setCluster( $cluster );
   # Clean the deployment swlrepo part
   $SWLREP->clean( $sl_swlrepo::SWLREPO_PART{'DEPLOY'}, $swlInst );   
   # Create NHAS configuration files
   $status = $swlInst->deploy;
   return $status;
}

# ------------------------------------------------------------------------------
# 'sl_create' creates a software load from the model given as parameter
#
sub sl_create {
   my ( $swlName, $swlVersion, $force ) = @_;
   my $status = 0;
   my $failure = undef;
   my $swl = undef;
   my $cluster = undef;
   my %softLst = ();
   &sl_traces::info( 1, "Creating software load < %s > ".
                     "version < %s >",
		     ( $swlName, $swlVersion ) );
   # Create a swl object
   $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
   # Check if the software load already exists in the repository
   my $swlInst = $SWLREP->lookUp( $swl );
   if ( defined $swlInst ) {
      if ( ! defined $force ) {
         # Ask to the user for removal
	 &sl_traces::warning( 1, "Software load < %s > < %s > ".
			      "already exists.",
			      ( $swl->{'Name'}, $swl->{'Version'} ) );
	 $question = "Do you want to remove it [ y | n ] ? : ";
	 if ( &sl_utils::ask( $question ) ) {
	    $status = &sl_delete( $swl->{'Name'}, $swl->{'Version'}, undef );
	 }
	 else {
	    $status = 1;
	 }
      }
      else {
         # Force the swl removal (keep the software)
	 $SWLREP->remove( $swlInst );
      }
   }
   unless ( $status ) {
      # Create a cluster object from the configuration files
      @configFileList = ( ${sl_parser::MACHINE_CONF},
                          ${sl_parser::CLUSTER_CONF},
                          ${sl_parser::NETWORK_CONF} );
      &sl_parser::load( @configFileList );
      ( $cluster ) = &sl_parser::getObjects( "sl_cluster", ( 0 ) );
      if ( defined $cluster ) {
	 # Complete swl definition with cluster definition
	 $swl->setCluster( $cluster );
      }
      else {
	 $status = 1;
      }
   }
   unless ( $status ) {
      # Read software list from configuration file
      $cluster = $swl->getCluster;
      %softLst = $cluster->getSoftLst;
      # Gather the software
      $status = &retrieveSoftware( $swl->iD, %softLst );
   }
   unless ( $status ) {
      # Create swl installation data
      $swl->install;
   }
   # Save the swl into the swl-repository if all the software 
   # has been retrieved and transfered into the software repository
   $swl->commit unless ( $status );
   return $status;
}

# ------------------------------------------------------------------------------
# 'sl_export' export platform update data
#
# Parameter(s)
#
# $pfuName    : pfu name
# $pfuVersion : pfu version
# $exportDir  : directory where the pfu data are exported
# $remote     : if set indicates that the packages and patches must alos be 
#               copied to the export directory.
#
sub sl_export {
   my ( $swlName, $swlVersion, $exportDir, $remote ) = @_;
   my $status = 0;
   &sl_traces::info( 1, "Exporting software load < %s > ".
                     "version < %s > to < %s >",
		     ( $swlName, $swlVersion, $exportDir ) );
   my $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
   my $swlInst = $SWLREP->lookUp( $swl ); # Look if the swl exists
   if ( defined $swlInst ) {
      $status = $SWLREP->export( $swlInst, $exportDir, $remote );
   }
   else {
      &sl_traces::error( "Software Load < %s > version ".
        	         "< %s > does not exist",  
			 "- smct::slexport - ".__LINE__,
			 ( $swlName, $swlVersion ) );
   }      
   return $status;  
}

# ------------------------------------------------------------------------------
# 'sl_delete' delete existing platform update/software load from the 
#  pfurepo/swlrepo
#
# Parameter(s)
#
# $force : if set forces the removal of the packages and patches associated
#          to the swl
#
sub sl_delete {
   my ( $swlName, $swlVersion, $force ) = @_;
   my $full = "y";
   &sl_traces::info( 1, "Deleting software load < %s > version < %s >",
                     ( $swlName, $swlVersion ) );
   my $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
   my $swlInst = $SWLREP->lookUp( $swl ); # Look if the swl exists
   if ( defined $swlInst ) {
      $SWLREP->remove( $swlInst, $full, $force );
   }
   else {
      &sl_traces::error( "Software Load < $swlName > ".
	                 "version < $swlVersion > does not exist", 
			 "- smct::sldelete - ".__LINE__ );
   }      
   return 0;
}

# ------------------------------------------------------------------------------
# 'sl_list' displays the swl data (either for the swl given as parameter or for
#  all the swl stored in the repository).
#
# Parameter(s)
#
# $swlName    : swl name
# $swlVersion : swl version
# $all        : if set all the data stored in the repository are displayed
# $brief      : to display a short listing
#
sub sl_list {
   my ( $swlName, $swlVersion, $all, $brief, $swl, $soft ) = @_;
   my $status = 0;
   if ( defined $all ) { 
      if ( defined $swl ) {
	 &sl_traces::info( 1, "Listing software load(s)" );
	 $SWLREP->display( undef, $brief );
      }
      elsif ( defined $soft ) {
	 &sl_traces::info( 1, "Listing software" );
	 $SOFTREP->display( undef, $brief );
      }
   }
   else {
      &sl_traces::info( 1, "Listing software load < %s > version < %s >",
			( $swlName, $swlVersion ) );
      my $swl = sl_swl->new( 'Name' => $swlName, 'Version' => $swlVersion );
      my $swlInst = $SWLREP->lookUp( $swl ); # Look if the swl exists
      if ( defined $swlInst ) {
	 $SWLREP->display( $swlInst, $brief );
      }
      else {
	 &sl_traces::error( "Software Load < %s > ".
	                    "version < %s > does not exist",
			    "- smct::sl_list - ".__LINE__,
			    ( $swlName, $swlVersion ) );
         $status = 1;
      }   
   }
   return $status; 
}

# ------------------------------------------------------------------------------
# Main starts here

# ------------------------------------------------------------------------------
# Read Input parameters
#

&sl_traces::init;

# Look for the operation requested
for ( $i=0; $i <= $#ARGV; $i++ ) {
   if ( ( $ARGV[$i] eq $CREATE ) or
    ( $ARGV[$i] eq $DEPLOY ) or
    ( $ARGV[$i] eq 'export' ) or
    ( $ARGV[$i] eq 'import' ) or
    ( $ARGV[$i] eq 'delete' ) or
    ( $ARGV[$i] eq $CONFIG ) or
    ( $ARGV[$i] eq 'list' ) or
    ( $ARGV[$i] eq 'help' ) ) { $OPERATION=$ARGV[$i]; last } 
}

# Check the operation parameters
if ( $OPERATION eq $DEPLOY ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-c' ) { $CONFIG_DIR=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq $DEPLOY ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq "export" ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-e' ) { $EXPORT_DIR=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-r' ) { $REMOTE="y" }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq 'export' ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq "import" ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-i' ) { $IMPORT_DIR=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-r' ) { $REMOTE="y" }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq 'export' ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq "delete" ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-f' ) { $FORCE="y" }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq 'delete' ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq $CONFIG ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-c' ) { $CONFIG_DIR=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq $CONFIG ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq "list" ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq '-b' ) { $BRIEF="y" }
      elsif ( $ARGV[$i] eq '-p' ) { $SOFT="y" }
      elsif ( $ARGV[$i] eq '-s' ) { $SWL="y" }
      elsif ( $ARGV[$i] eq 'list' ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
elsif( $OPERATION eq $CREATE ) {
   for ( $i=0; $i <= $#ARGV; $i++ ) {
      if    ( $ARGV[$i] eq '-V' ) { $VERBOSE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-l' ) { $LOG_FILE=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-d' ) { $DEBUG++ }
      elsif ( $ARGV[$i] eq '-c' ) { $CONFIG_DIR=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-v' ) { $SWL_VERSION=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-n' ) { $SWL_NAME=$ARGV[++$i] }
      elsif ( $ARGV[$i] eq '-h' ) { &usage; }
      elsif ( $ARGV[$i] eq '-f' ) { $FORCE="y" }
      elsif ( $ARGV[$i] eq $CREATE ) { next; }
      else {
	 &usage( "Illegal option %s", "- smct - ".__LINE__,
	         ( "< $ARGV[$i] >" ) );
      }
   }
}
else {
   $OPERATION = 'help';
   &usage( "Unknown operation", "- smct - ".__LINE__ );
}

# ------------------------------------------------------------------------------
# Check input Parameters
#

( $OPERATION eq $CREATE ) || 
( $OPERATION eq 'export' ) || 
( $OPERATION eq 'import' ) || 
( $OPERATION eq $DEPLOY ) || 
( $OPERATION eq 'delete' ) ||
( $OPERATION eq 'list' ) ||
( $OPERATION eq $CONFIG ) ||
   &usage( "No operation has been specified", "- smct - ".__LINE__ );
if ( ( $OPERATION eq $CREATE ) ||
     ( $OPERATION eq $DEPLOY ) ||
     ( $OPERATION eq $CONFIG ) ) {
   # Override default configuration directory
   if ( defined $CONFIG_DIR ) {
     $CONFIG_DIR = &sl_utils::setAbsDir( $CONFIG_DIR );
     $ENV{'SMCT_CONFIG_DIR'} = $CONFIG_DIR;
   }
}
if ( $OPERATION eq 'export' ) {
   $EXPORT_DIR = $ENV{'SMCT_EXPORT_DIR'}
      unless ( defined $EXPORT_DIR );
   $EXPORT_DIR = &sl_utils::setAbsDir( $EXPORT_DIR );
}
if ( $OPERATION eq 'import' ) {
   $IMPORT_DIR = $ENV{'SMCT_IMPORT_DIR'}
      unless ( defined $IMPORT_DIR );
   $IMPORT_DIR = &sl_utils::setAbsDir( $IMPORT_DIR );
}
if ( ( $OPERATION eq 'delete' ) ||
     ( $OPERATION eq $CREATE ) || 
     ( $OPERATION eq 'export' ) || 
     ( $OPERATION eq 'import' ) || 
     ( $OPERATION eq $CONFIG ) ||
     ( $OPERATION eq $DEPLOY ) ) { 
   &usage( "Missing option %s", "- smct - ".__LINE__,
           ( "<swl-name>" ) ) 
      unless ( defined $SWL_NAME );
   &usage( "Missing option %s", "- smct - ".__LINE__,
           ( "<swl-version>" ) ) 
      unless ( defined $SWL_VERSION );
}
if ( $OPERATION eq 'list' ) {
   if ( defined $SWL ) {
      unless ( ( defined $SWL_VERSION ) &&
	       ( defined $SWL_NAME ) ) {
         $ALL = "yes";
      }
   }
   elsif ( defined $SOFT ) {
         $ALL = "yes";
   }
   else {
      &usage( "Select software or software load repository", 
              "- smct - ".__LINE__ );
   }
}
&sl_traces::setVerbosity( $VERBOSE ) if ( defined $VERBOSE );
&sl_traces::setDebug( $DEBUG ) if ( defined $DEBUG );
&sl_traces::setLogFile( $LOG_FILE ) if ( defined $LOG_FILE );

# ------------------------------------------------------------------------------
# Read SLB environment variables
#
$VAR_NOT_SET = &checkEnv;
if ( $VAR_NOT_SET ) { 
   &sl_traces::error( "Variable < %s > is not defined",
                      "- smct - ".__LINE__,
		      ( $VAR_NOT_SET ) );
}
# -----------------------------------------------------------------------------
# Open the dbm data bases associated to repositories
# 
$SOFTREP = sl_softrepo->new( 'DB' => "$ENV{'SMCT_SOFTREP_DIR'}/softdb" );
$SWLREP = sl_swlrepo->new( 'DB' => "$ENV{'SMCT_SWLREP_DIR'}/swldb" );

# -----------------------------------------------------------------------------
# Start of the operation processing
#
 
if ( $OPERATION eq $CREATE ) {
   $OPSTAT = &sl_create( $SWL_NAME, $SWL_VERSION, $FORCE );
}
elsif ( $OPERATION eq 'export' ) {
   $OPSTAT = &sl_export( $SWL_NAME, $SWL_VERSION, $EXPORT_DIR, $REMOTE );
}
elsif ( $OPERATION eq 'delete' ) {
   $OPSTAT = &sl_delete( $SWL_NAME, $SWL_VERSION, $FORCE );
}
elsif ( $OPERATION eq 'list' ) {
   $OPSTAT = &sl_list( $SWL_NAME, $SWL_VERSION, $ALL, $BRIEF, $SWL, $SOFT );
}
elsif ( $OPERATION eq $DEPLOY ) {
   $OPSTAT = &sl_deploy( $SWL_NAME, $SWL_VERSION );
}
elsif ( $OPERATION eq $CONFIG ) {
   $OPSTAT = &sl_config( $SWL_NAME, $SWL_VERSION );
}
else {
   &sl_traces::error( "The operation < %s > is not yet ".
                      "implemented", "- smct - ".__LINE__, 
		      ( $OPERATION ) );
}

# -----------------------------------------------------------------------------
# Close the dbm data bases associated to the software and swl repositories
# 
$SOFTREP->save;
$SWLREP->save;

# -----------------------------------------------------------------------------
# Epilog
# 
if ( !$OPSTAT ) {
   &sl_traces::info( 1, "has succeeded" );
}
else {
   &sl_traces::info( 1,"has failed" );
}

exit $OPSTAT;
