#!/usr/bin/perl
# ------------------------------------------------------------------------------
#  ident "@(#)nhinstall.pl 1.14 03/04/18 SMI"
# ------------------------------------------------------------------------------
#
#  Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# ------------------------------------------------------------------------------

package nhinstall ;

use variables;

require "$ENV{NHINSTALL_LIB}/tools/lib/conf.pl";
require "$ENV{NHINSTALL_LIB}/tools/lib/common.pl";
require "$ENV{NHINSTALL_LIB}/tools/lib/sequencer.pl";

#------------------------------------------------------------------------------
#
# check for change since last run
# check the checksum of cluster_definition.conf
#
#------------------------------------------------------------------------------

sub check_for_change {

  my $cluster_checksum ;
  my $expected_cluster_checksum ;
  my $filename = $ENV{NH_CLUSTER_FILE_FULLNAME};

  #
  # check that the installation (if re-launch is compatible with the
  # previous run (i.e. can't launch for adding if the cluster installation
  # is not complete
  #
  my $previous_install = SEQUENCER::get_data("INSTALL_TYPE");
  if (($previous_install ne "") && ($previous_install ne $INSTALL_TYPE)) {
    COMMON::error("Required installation type is not compatible with the previous one ('reset' or 'clear' is required)");
  }
  SEQUENCER::set_data("INSTALL_TYPE", $INSTALL_TYPE);


  # check for change of configuration files since last run

  $cluster_checksum = COMMON::get_checksum($filename);
  $expected_cluster_checksum = SEQUENCER::get_data("CLUSTER_CHECKSUM");
  if ($expected_cluster_checksum ne "") {
    # it's not the first time
    $RECOVERY = "1";
    if ($expected_cluster_checksum ne $cluster_checksum) {
      my $filename = $ENV{NH_CLUSTER_FILE_FULLNAME};
      print "$filename has been changed since last run.\n";
      print "This may cause the installation to be incorrect.\n";
      print "Perform a reset to restart the installation from The beginning.\n";
      my $c = "";
      while (($c ne "y") && ($c ne "Y") && ($c ne "n") && ($c ne "N")) {
	print "Do you want to continue the installation anyway [y/n]: ";
	chomp($c = <STDIN>) ;
      }
      if (($c eq "n") || ($c eq "N")) {
	exit(0);
      }
    }
  }

  SEQUENCER::set_data("CLUSTER_CHECKSUM", $cluster_checksum);
}

#------------------------------------------------------------------------------
#
# installation of Solaris on MEN
#
#------------------------------------------------------------------------------

sub install_solaris_on_men {

  if ($SOLARIS_INSTALL eq "YES") {
    # create the Jumpstart environment
    SEQUENCER::execute_action("men_jumpstart", $MEN1_ID);
    SEQUENCER::execute_action("men_jumpstart", $MEN2_ID);
    
    # detect the end of the installation
    if ($AUTO_REBOOT eq "YES") {
      # ask the user to boot both nodes
      SEQUENCER::execute_action("ask_for_boot",0);
      SEQUENCER::execute_action("men_wait", $MEN1_ID);
      SEQUENCER::execute_action("men_wait", $MEN2_ID);
    } else {
      # ask the user to notify when finished
      SEQUENCER::execute_action("ask_for_solaris_installation",0);
    }
    
    #  configure SVM if it has been requested
    if ($USE_SVM eq "YES") {
      SEQUENCER::execute_action("men_svm", $MEN1_ID);
      SEQUENCER::execute_action("men_svm", $MEN2_ID);
    }

  }

  # add the Solaris patches even if Solaris is not installed
  SEQUENCER::execute_action("men_addon_solaris", $MEN1_ID);
  SEQUENCER::execute_action("men_addon_solaris", $MEN2_ID);

}

#------------------------------------------------------------------------------
#
# installation of Solaris on diskless
#
#------------------------------------------------------------------------------

sub install_solaris_on_diskless {

  if ($INSTALL_TYPE eq "CLUSTER") {
    # DCSS
    if ($USE_DCSS eq "YES") {
      SEQUENCER::execute_action("men_dcss", $MEN1_ID);
      SEQUENCER::execute_action("men_dcss", $MEN2_ID);
    }
    
    # OS services
    SEQUENCER::execute_action("nmen_solaris", $MEN1_ID);
    SEQUENCER::execute_action("nmen_solaris", $MEN2_ID);
  }

  # diskless environment
  SEQUENCER::execute_action("nmen_diskless", $MEN1_ID);
  SEQUENCER::execute_action("nmen_diskless", $MEN2_ID);

  # diskless packages on master (hardware specific)
  SEQUENCER::execute_action("nmen_disklesspkg", $MEN1_ID);

  # add the Solaris patches
  SEQUENCER::execute_action("nmen_addon_solaris", $MEN1_ID);
}

#------------------------------------------------------------------------------
#
# installation of Foundation Services on MEN
#
#------------------------------------------------------------------------------

sub install_fs_on_men {

  # NHADM
  SEQUENCER::execute_action("men_checker", $MEN1_ID);
  SEQUENCER::execute_action("men_checker", $MEN2_ID);

  # CGTP
  SEQUENCER::execute_action("men_cgtp", $MEN1_ID);
  SEQUENCER::execute_action("men_cgtp", $MEN2_ID);

  # CGTP patch
  SEQUENCER::execute_action("men_cgtppatch", $MEN1_ID);
  SEQUENCER::execute_action("men_cgtppatch", $MEN2_ID);

  # CMM
  SEQUENCER::execute_action("men_cmm", $MEN1_ID);
  SEQUENCER::execute_action("men_cmm", $MEN2_ID);

  # RNFS
  SEQUENCER::execute_action("men_rnfs", $MEN1_ID);
  SEQUENCER::execute_action("men_rnfs", $MEN2_ID);

  # PMD
  SEQUENCER::execute_action("men_pmd", $MEN1_ID);
  SEQUENCER::execute_action("men_pmd", $MEN2_ID);

  # NSM
  SEQUENCER::execute_action("men_nsm", $MEN1_ID);
  SEQUENCER::execute_action("men_nsm", $MEN2_ID);

  # WDT
  SEQUENCER::execute_action("men_wdt", $MEN1_ID);
  SEQUENCER::execute_action("men_wdt", $MEN2_ID);

  # Foundation Services configuration file creation
  SEQUENCER::execute_action("men_fsconf", $MEN1_ID);
  SEQUENCER::execute_action("men_fsconf", $MEN2_ID);

  # JDK
  SEQUENCER::execute_action("men_jdk", $MEN1_ID);
  SEQUENCER::execute_action("men_jdk", $MEN2_ID);

  # MA - JDMK
  SEQUENCER::execute_action("men_ma_jdmk", $MEN1_ID);
  SEQUENCER::execute_action("men_ma_jdmk", $MEN2_ID);

  # manpages
  SEQUENCER::execute_action("men_manpages", $MEN1_ID);
  SEQUENCER::execute_action("men_manpages", $MEN2_ID);

  # add-on at the end of the installation
  SEQUENCER::execute_action("men_addon_fs", $MEN1_ID);
  SEQUENCER::execute_action("men_addon_fs", $MEN2_ID);
}

#------------------------------------------------------------------------------
#
# installation of Foundation Services on diskless
#
#------------------------------------------------------------------------------

sub install_fs_on_diskless {


  if ($INSTALL_TYPE eq "CLUSTER") {

    # RBS
    SEQUENCER::execute_action("nmen_rbs", $MEN1_ID);
    SEQUENCER::execute_action("nmen_rbs", $MEN2_ID);

    if ($USE_NLS eq "YES") {
      SEQUENCER::execute_action("nmen_nls", $MEN1_ID);
      SEQUENCER::execute_action("nmen_nls", $MEN2_ID);
    }

  } else {
    if ($USE_NLS eq "YES") {
      SEQUENCER::execute_action("nmen_nls_add_nodes", $MEN1_ID);
    } else {
      # update table to add diskless configuration
      SEQUENCER::execute_action("nmen_rbs_add_nodes", $MEN1_ID);
    }
  }

  # NHADM
  SEQUENCER::execute_action("nmen_checker", $MEN1_ID);

  # CGTP
  SEQUENCER::execute_action("nmen_cgtp", $MEN1_ID);

  # CGTP patch
  SEQUENCER::execute_action("nmen_cgtppatch", $MEN1_ID);

  # CMM
  SEQUENCER::execute_action("nmen_cmm", $MEN1_ID);

  # PMD
  SEQUENCER::execute_action("nmen_pmd", $MEN1_ID);

  # WDT
  SEQUENCER::execute_action("nmen_wdt", $MEN1_ID);

  # Foundation Services configuration file creation
  # this also update the MEN nhfs.conf
  SEQUENCER::execute_action("nmen_fsconf", $MEN1_ID);
  SEQUENCER::execute_action("nmen_fsconf", $MEN2_ID);

  # reset the /etc/dfs/dfstab on MEN1
  SEQUENCER::execute_action("nmen_reset_men1_dfstab", $MEN1_ID);

  # JDK
  SEQUENCER::execute_action("nmen_jdk", $MEN1_ID);

  # MA
  SEQUENCER::execute_action("nmen_ma", $MEN1_ID);

  # add-on at the end of the installation
  SEQUENCER::execute_action("nmen_addon_fs", $MEN1_ID);

  # post-installation tasks
  SEQUENCER::execute_action("nmen_end", $MEN1_ID);
}

#------------------------------------------------------------------------------
#
# describes the stage for installation a full cluster
#
#------------------------------------------------------------------------------

sub install_cluster {

  # set the flag "concerned" to TRUE for all nodes
  for my $i (0..$#NODE_LIST) {
    $NODE_LIST[$i]{concerned} = "TRUE";
  }

  # install Solaris for MEN maybe partially
  install_solaris_on_men();

  # status is not defined when just launched
  my $status = SEQUENCER::get_data("MEN_ENV_IN_PROGRESS");
  if (($status eq "YES") || ($status eq "")) {
    SEQUENCER::force_action("mount_men",$MEN1_ID);
  }
  SEQUENCER::set_data("MEN_ENV_IN_PROGRESS", "YES");

  # some checking about Solaris version and NHAS already installed
  SEQUENCER::execute_action("men_check_solaris", $MEN1_ID);
  SEQUENCER::execute_action("men_check_solaris", $MEN2_ID);
  SEQUENCER::execute_action("men_check_nhas", $MEN1_ID);
  SEQUENCER::execute_action("men_check_nhas", $MEN2_ID);

  # retrieve some infos (or set missing infos) on MEN
  SEQUENCER::execute_action("men_getinfo", $MEN1_ID);
  SEQUENCER::execute_action("men_getinfo", $MEN2_ID);
  
  # install the Foundation Services
  install_fs_on_men();
  
  SEQUENCER::set_data("MEN_ENV_IN_PROGRESS", "NO");

  # if the diskless environment is required,
  # . installation of Solaris environment
  # . installation of Foundation Services
  if (SEQUENCER::get_data("DISKLESS_ENV") eq "YES") {
    if (SEQUENCER::get_data("DISKLESS_ENV_IN_PROGRESS") eq "YES") {
      SEQUENCER::force_action("mount_nmen",$MEN1_ID);
    }
    SEQUENCER::set_data("DISKLESS_ENV_IN_PROGRESS", "YES");

    # smosservice needs the men to be rebooted
    if ($USE_DCSS eq "NO") {
      if ($AUTO_REBOOT eq "YES") {
	# MEN1 is rebooted (waiting for Solaris to be ready)
	$REBOOT= "SOL";
	SEQUENCER::execute_action("men_reboot_for_diskless", $MEN1_ID);
      } else {
	SEQUENCER::execute_action("men_manual_reboot_for_diskless", $MEN1_ID);
      }
    }

    # set here to be sure it's done even if Solaris is not
    # installed on diskless
    SEQUENCER::execute_action("mount_nmen", $MEN1_ID);

    install_solaris_on_diskless();
    install_fs_on_diskless();

    # reset progress indicator for diskless installation
    SEQUENCER::set_data("DISKLESS_ENV_IN_PROGRESS", "NO");
  }

  # remove the installation indicator before the final reboot
  # set the correct node name
  SEQUENCER::execute_action("men_end", $MEN1_ID);
  SEQUENCER::execute_action("men_end", $MEN2_ID);

  # final reboot
  if ($AUTO_REBOOT eq "YES") {
    # MEN are rebooted (waiting for FS to be ready)
    $REBOOT= "FS";
    SEQUENCER::execute_action("men_reboot_end", $MEN1_ID);
    SEQUENCER::execute_action("men_reboot_end", $MEN2_ID);
  } else {
    SEQUENCER::execute_action("men_manual_reboot_end", $MEN1_ID);
    SEQUENCER::execute_action("men_manual_reboot_end", $MEN2_ID);
  }

  if (SEQUENCER::get_data("DISKLESS_ENV") eq "YES") {
    SEQUENCER::execute_action("boot_diskless", 0);
  }

  COMMON::print_stage("The software installation is complete.");
  COMMON::print_stage("The cluster has started synchronizing the replicated filesystems.\nWait until all these filesystems are in the REP state before performing switchovers or failovers.\nUse /usr/opt/SUNWesm/sbin/sndrstat -l to see the filesystem states.");
}

#------------------------------------------------------------------------------
#
# describes the stage for adding a diskless
#
#------------------------------------------------------------------------------

sub add_diskless {

  COMMON::print_stage("Processing nodes information");


  # set the flag "concerned" to TRUE for diskless to add
  for my $id (split(" ",$ENV{NHINSTALL_NODE_LIST})) {
    my $found = "FALSE";
    for my $i (0..$#NODE_LIST) {
      if ($NODE_LIST[$i]{id} eq $id) {
	$NODE_LIST[$i]{concerned} = "TRUE";
	if ($NODE_LIST[$i]{type} eq "MEN") {
	  COMMON::error("Node $id is a master-eligible node: only diskless nodes can be added");
	}
	$found = TRUE;
      }
    }
    if ($found eq "FALSE") {
      COMMON::error("Node with id $id not found in cluster_definition.conf");
    }
  }

  #
  # check that the node list has not been changed since the last run
  #
  my $list = "";
  for my $i (0..$#NODE_LIST) {
    if ($NODE_LIST[$i]{concerned} eq "TRUE") {
      $list = join " ",$list, $NODE_LIST[$i]{id}
    }
  }
  my $previous_list = SEQUENCER::get_data("ADDED_NODES");
  if (($previous_list ne "") && ($previous_list ne $list)) {
    COMMON::error("List of nodes to be added has changed since the last run (previous list:$previous_list)");
  }
  SEQUENCER::set_data("ADDED_NODES", $list);

  #
  # display nodes information
  #
  COMMON::print_action("Nodes");
  for my $i (0..$#NODE_LIST) {
    if ($NODE_LIST[$i]{concerned} eq "TRUE") {
      my $nodename = COMMON::build_node_name($NODE_LIST[$i]{id}, "");
      COMMON::print_subaction("$nodename");
    }
  }

  # retrieve some infos (or set missing infos) on MEN
  SEQUENCER::execute_action("men_getinfo", $MEN1_ID);
  SEQUENCER::execute_action("men_getinfo", $MEN2_ID);

  install_solaris_on_diskless();
  install_fs_on_diskless();

  # update /etc/hosts on all old nodes
  SEQUENCER::execute_action("men_cgtp_add_nodes", $MEN1_ID);
  SEQUENCER::execute_action("men_cgtp_add_nodes", $MEN2_ID);
  SEQUENCER::execute_action("nmen_cgtp_add_nodes", $MEN1_ID);

  # update cluster_nodes_table
  SEQUENCER::execute_action("men_cmm_add_nodes", $MEN1_ID);
  SEQUENCER::execute_action("men_cmm_add_nodes", $MEN2_ID);

  # force CMM to reload cluster_nodes_table on the master
  # otherwise maybe superseded with the version in memory
  SEQUENCER::execute_action("men_cmm_reload_file", $MEN1_ID);

  # update fs.conf on master (regarding share commands)
  SEQUENCER::execute_action("men_fsconf_add_nodes", $MEN1_ID);
  SEQUENCER::execute_action("men_fsconf_add_nodes", $MEN2_ID);
  
  # final reboot
  if ($AUTO_REBOOT eq "YES") {
    # MEN are rebooted (waiting for FS to be ready)
    $REBOOT= "FS";
    SEQUENCER::execute_action("men_reboot_end", $MEN1_ID);
    SEQUENCER::execute_action("men_reboot_end", $MEN2_ID);
  } else {
    SEQUENCER::execute_action("men_manual_reboot_end", $MEN1_ID);
    SEQUENCER::execute_action("men_manual_reboot_end", $MEN2_ID);
  }

  SEQUENCER::execute_action("boot_diskless", 0);

  COMMON::print_log("Your cluster is ready\n");
}

#------------------------------------------------------------------------------
#
# MAIN
#
#------------------------------------------------------------------------------

{
  $INSTALL_TYPE = "CLUSTER";
  if (defined($ENV{NHINSTALL_NODE_LIST})) {
    $INSTALL_TYPE = "ADD_DISKLESS";
  }

  #
  # ACTIONS ALWAYS DONE WHEN STARTING THE SCRIPT
  #

  COMMON::init();
  SEQUENCER::init();

  # recovery is set to true if the checksum already exists
  # (indicator reset when recovery has been performed)
  check_for_change();
  CONF::load_conf();

  #
  # ACTIONS BASED ON PROGRESS INDICATOR AND RECOVERY
  #

  if ($INSTALL_TYPE eq "CLUSTER") {
    # ask if the diskless environment has to be installed if there is no diskless
    SEQUENCER::execute_action("ask_for_diskless",0);
  } else {
    SEQUENCER::set_data("DISKLESS_ENV", "YES");
  }

  # all information have been gathered, check the configuration files
  CONF::check_conf();

  # exit when testing the configuration loading
  if ($TESTING eq "conf") {
    exit 0;
  }

  #
  # always done because the directories can be unshared
  # - export: Solaris and NHAS distribution
  #
  SEQUENCER::force_action("prepare_directories", 0);

  #
  # get the installation server info
  # (info are stored and re-read from the status file after a restart)
  #

  SEQUENCER::execute_action("get_install_server_info",0);
  #
  # Always configure the interface: it may be broken
  #
  SEQUENCER::force_action("configure_nic",0);


  if ($INSTALL_TYPE eq "CLUSTER") {
    # MEN1_ID and MEN2_ID set during configuration files loading

    if ($SOLARIS_INSTALL eq "YES") {
      SEQUENCER::force_action("check_solaris_distrib", 0);
    }
    install_cluster() ;

  } elsif ($INSTALL_TYPE eq "ADD_DISKLESS") {
    # MEN1_ID and MEN2_ID supersede with actual master and vice-master node ids
    $MEN1_ID = COMMON::get_id("master");
    $MEN2_ID = COMMON::get_id("vice");

    add_diskless();
  }

}
