#!/usr/bin/perl
# -----------------------------------------------------------------------------
#  ident "@(#)conf.pl 1.37 03/04/24 SMI"
# -----------------------------------------------------------------------------
#
#  Copyright 2001-2003 Sun Microsystems, Inc. All rights reserved.
#
# -----------------------------------------------------------------------------

package CONF;

use variables;

#-----------------------------------------------------------------------------
#
#  extract string
#
#-----------------------------------------------------------------------------

sub get_string
  {
    my ($string) = @_;

    if ($string =~ m/\"([^\"]*)\"/) {
      return $1;
    }
    return $string;
  }

#------------------------------------------------------------------------------
#
#  check package/patch parameter
#
#------------------------------------------------------------------------------

sub check_addon_directive {

  my ($directive, $name, $line, $filename, $phase, $scope, $men, $nmen) = @_;

  # phase
  if (($phase ne "F") && ($phase ne "D") && ($phase ne "S") && ($phase ne "I")) {
    COMMON::error("$directive $name: phase must be I, S, F or D (line $line, file $filename)") ;
  }

  #scope
  if (($scope ne "LOCAL") && ($scope ne "SHARED") && ($scope ne "USR_SPECIFIC")) {
    COMMON::error("$directive $name: scope must be LOCAL, SHARED or USR_SPECIFIC (line $line, file $filename)") ;
  }

  # coherency
  if (($scope eq "LOCAL") || ($scope eq "USR_SPECIFIC")) {
    if (($men ne "Y") && ($men ne "N")) {
      COMMON::error("$directive $name: MEN must be Y or N (line $line, file $filename)") ;
    }
    if (($nmen ne "Y") && ($nmen ne "N")) {
      COMMON::error("$directive $name: NMEN must be Y or N (line $line, file $filename)") ;
    }
  }
	
  if (($scope eq "SHARED") and ($phase eq "S")) {
    COMMON::error("$directive $name: shared $directive can be installed only after the installation of the Foundation Service on MEN");
  } 
}

#------------------------------------------------------------------------------
#
#  load cluster definition
#
#------------------------------------------------------------------------------

sub load_cluster
  {
    my ($filename) = @_ ;

    my $node_count = 0 ;
    
    if ($G_verbose eq TRUE) {
      print "Loading cluster definition from $filename\n" ;
    }

    open(FCLUSTER, "<$filename") or die("Can't access $filename") ;
    while (<FCLUSTER>) {
      chomp($line = $_) ;
      $line =~ s/#.*// ;
      if ($line eq "") {
	next ;
      }
      if ($line !~ m/[\s\t]*([^\s\t]*)[\s\t]*=[\s\t]*(.*)/) {
	COMMON::error("$line: invalid syntax (line $., file $filename)");
      }

      if ($1 eq "CLUSTER_ID") {
	#
	#   CLUSTER_ID=<cluster-id>
	#
	if (($2 <= 0) || ($2 > 255)) {
	  COMMON::error("Wrong cluster Id ($2): must be between 1 and 255 (line $., file $filename)") ;
	}
	$CLUSTER=$2;
      }

      elsif ($1 eq "ARCH") {
	#
	#   ARCH=<arch>
	#
	$ARCH=$2 ;
      } 

      elsif ($1 eq "MEN_INTERFACES") {
	#
	#   MEN_INTERFACES=<nic0> <nic1>
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	$MEN_NIC0= $1 ;
	$MEN_NIC1= $2 ;
      } 

      elsif ($1 eq "NMEN_INTERFACES") {
	#
	#   NMEN_INTERFACES=<nic0> <nic1>
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	$NMEN_NIC0= $1 ;
	$NMEN_NIC1= $2 ;
      } 

      elsif ($1 eq "NIC0_POSTFIX") {
	#
	#   NIC0_POSTFIX=<string>
	#
	$POST_NIC0=get_string($2);
      } 
      elsif ($1 eq "NIC1_POSTFIX") {
	#
	#   NIC1_POSTFIX=<string>
	#
	$POST_NIC1=get_string($2);
      } 
      elsif ($1 eq "CGTP_POSTFIX") {
	#
	#   CGTP_POSTFIX=<string>
	#
	$POST_CGTP=get_string($2);
      } 

      elsif ($1 eq "NODE") {

	#
	# NODE=<Node id> { <MAC address 0> | - }  { <MAC address 1> | - } [ <hostname> | - ] [ <nic0> | - ] [ <nic1> | - ] [ public IP address | - ] [ public nic | - ]
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($1 eq "") {
	  COMMON::error("Node id is missing (line $., file $filename)") ;
	} elsif ($2 eq "") {
	  COMMON::error("MAC address is missing (line $., file $filename)") ;
	}
	if ($1 <= 0) {
	  COMMON::error("Wrong node Id ($1):must be greater than 0 (line $., file $filename)") ;
	}
	if ($1 eq $MASTER_ID) {
	  COMMON::error("Wrong node Id ($1): cannot equal to master node Id (line $., file $filename)") ;
	}
	if ($1 eq $SERVER_NODE) {
	  COMMON::error("Wrong node Id ($1): already used as the installation server id (line $., file $filename)")
	}

	my $node = {} ;
	$node->{id} = $1 ;
	$node->{eth0} = $2 ;
	$node->{eth0} =~ tr/A-F/a-f/; # convert MAC address into lowercase
	$node->{eth1} = $3 ;
	$node->{eth1} =~ tr/A-F/a-f/; # convert MAC address into lowercase
	$node->{name} = $4 ;
	$node->{nic0} = $5 ;
	$node->{nic1} = $6 ;
	$node->{pubName} = $7;
	$node->{pubIp} = $8 ;
	$node->{pubNic} = $9 ;
	if ($node->{eth0} eq "-") {
	  $node->{eth0} = "";
	}
	if ($node->{eth1} eq "-") {
	  $node->{eth1} = "";
	}
	if ($node->{name} eq "-") {
	  $node->{name} = "";
	}
	if ($node->{nic0} eq "-") {
	  $node->{nic0} = "";
	}
	if ($node->{nic1} eq "-") {
	  $node->{nic1} = "";
	}
	if ($node->{pubName} eq "-") {
	  $node->{pubName} = "";
	}
	if ($node->{pubIp} eq "-") {
	  $node->{pubIp} = "";
	}
	if ($node->{pubNic} eq "-") {
	  $node->{pubNic} = "";
	}

	$node->{type} = "DISKLESS";
	$node->{concerned} = "FALSE";
	if ($node_count eq $MEN1) {
	  $MEN1_ID = $node->{id};
	  $node->{type} = "MEN";
	} elsif ($node_count eq $MEN2) {
	  $MEN2_ID = $node->{id} ;
	  $node->{type} = "MEN";
	}
	push @NODE_LIST,$node;

	$node_count++ ;
      }

      elsif ($1 eq "VENDOR_TYPE") {
	#
	# VENDOR_TYPE=string
	#
	if ($2 eq "") {
	  COMMON::error("VENDOR_TYPE: invalid syntax (line $., file $filename)") ;
	}
	push @VENDOR_TYPE, $2 ;
      }


      elsif ($1 eq "DISKLESS_BOOT_POLICY") {
	#
	# DISKLESS_BOOT_POLICY = { DHCP_DYNAMIC | DHCP_STATIC | NLS_DYNAMIC | NLS_MAC_ADDR | NLS_CPCI_SLOT | NLS_SWITCH_PORT }
	#
	if (($2 ne "DHCP_DYNAMIC")
	    && ($2 ne "DHCP_STATIC") 
#	    && ($2 ne "NLS_DYNAMIC")
#	    && ($2 ne "NLS_MAC_ADDR") 
#	    && ($2 ne "NLS_CPCI_SLOT")
#	    && ($2 ne "NLS_SWITCH_PORT")
	   ) {
	  COMMON::error("wrong DISKLESS_BOOT_POLICY ($2) (line $., file $filename)") ;
	}
	$DISKLESS_BOOT_POLICY=$2;
	
	$USE_NLS = "YES";
	if (($DISKLESS_BOOT_POLICY eq "DHCP_DYNAMIC") || 
	    ($DISKLESS_BOOT_POLICY eq "DHCP_STATIC")) {
	  $USE_NLS = "NO";
	}
      }

      elsif ($1 eq "USE_CGTP") {
	#
	# USE_CGTP={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong USE_CGTP ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$USE_CGTP=$2;
      }

      elsif ($1 eq "USE_WDT") {
	#
	# USE_WDT={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong USE_WDT ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$USE_WDT=$2;
      }

      elsif ($1 eq "RESTRICT_RHOSTS") {
	#
	# RESTRICT_RHOSTS={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong RESTRICT_RHOSTS ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$RESTRICT_RHOSTS=$2;
      }

      elsif ($1 eq "LOGICAL_SLICE_SUPPORT") {
	#
	# LOGICAL_SLICE_SUPPORT={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong LOGICAL_SLICE_SUPPORT ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$USE_SVM=$2;
      }

      elsif ($1 eq "IDE_SUPPORT") {
	#
	# IDE_SUPPORT={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong IDE_SUPPORT ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$IDE_SUPPORT=$2;
      }

      elsif ($1 eq "USE_DCSS") {
	#
	# USE_DCSS={ YES | NO }
	#
	if (($2 ne "YES") && ($2 ne "NO")) {
	  COMMON::error("wrong USE_DCSS ($2): must be YES OR NO (line $., file $filename)") ;
	}
	$USE_DCSS=$2;
      }

      elsif ($1 eq "PASSWORD") {
	#
	# PASSWORD=<string>
	#
	if ($2 eq "") {
	  COMMON::error("PASSWORD: invalid syntax (line $., file $filename)") ;
	}
	$PASSWORD=$2;
	$ENCRYPTED_PASSWORD = crypt ($PASSWORD, "NH");
      }

      elsif ($1 eq "EXPORTED") {
	#
	# EXPORTED=dir
	#
	if ($2 eq "") {
	  COMMON::error("EXPORTED: invalid syntax (line $., file $filename)") ;
	}
	push @DIR_EXPORTED, $2 ;
      }

      elsif ($1 eq "MOUNTED") {
	#
	# MOUNTED=<local> <remote>
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($2 eq "") {
	  COMMON::error("MOUNTED: invalid syntax (line $., file $filename)") ;
	}
	push @DIR_MOUNTED, { "remote" => "$2", "local" => "$1" } ;
      }

      elsif ($1 eq "SLICE") {
	#
	#  SLICE=<slice> { <size> | free } <mounting point> <bitmap slice> <options>
	#
	$value=$2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($5 eq "") {
	  COMMON::error("slice $1: invalid number of parameters (line $., file $filename)") ;
	}
	my $slice = {} ;
	$slice->{name} = $1 ;
	$slice->{size} = $2 ;
	$slice->{mounting_point} = $3 ;
	if ($4 eq "-") {
	  $slice->{bitmap} = "";
	} else {
	  $slice->{bitmap} = $4;
	}
	if ($5 eq "-") {
	  $slice->{option} = "";
	} else {
	  $slice->{option} = $5;
	}
	if (($slice->{size} eq "free") && ($slice->{bitmap} ne "")) {
	  COMMON::error("slice $1: exact size required for a replicated slice (bitmap slice defined)\n") ;
	}

	# determine raw and block device name
	my $slice_name = $slice->{name};
	if ($slice_name =~ m/c[0-9]*t[0-9]*d[0-9]s[0-9]*/) {
	# if the slice looks like a physical slice
	  $slice->{raw} = "/dev/rdsk/$slice_name";
	  $slice->{block} = "/dev/dsk/$slice_name";
	} else {
	  # probably a metadevice (usage of SVM other that automatically through the installer
	  # for customer's specific configuration
	  $slice->{raw} = "/dev/md/rdsk/$slice_name";
	  $slice->{block} = "/dev/md/dsk/$slice_name";
	}
	$slice->{metadevice} = "";
	push @SLICE_LIST, $slice;
      }

      elsif ($1 eq "CLUSTER_NETWORK") {
	#
	#  CLUSTER_NETWORK=<netmask> <nic0 net> <nic1 net> <cgtp net>
	#
	$value=$2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($2 eq "") {
	  COMMON::error("CLUSTER_NETWORK: invalid number of parameters (line $., file $filename)") ;
	}
	$USE_CLUSTER_NETWORK="TRUE";
	$CLUSTER_DOTTED_NETMASK  = $1;
	$CLUSTER_DOTTED_NIC0_NET = $2;
	if (($3 eq "") || ($3 eq "-")) {
	  $CLUSTER_DOTTED_NIC1_NET = "";
	} else {
	  $CLUSTER_DOTTED_NIC1_NET  = $3;
	}
	if (($4 eq "") || ($4 eq "-")) {
	  $CLUSTER_DOTTED_CGTP_NET = "";
	} else {
	  $CLUSTER_DOTTED_CGTP_NET  = $4;
	}
      }

      elsif ($1 eq "PUBLIC_NETWORK") {
	#
	#  PUBLIC_NETWORK=<netmask> <network>
	#
	$value=$2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($2 eq "") {
	  COMMON::error("PUBLIC_NETWORK: invalid number of parameters (line $., file $filename)") ;
	}
        $USE_PUBLIC_NETWORK = "YES";
	$PUBLIC_BIT_NETMASK   = COMMON::convertDottedToBitmask($1);
	$PUBLIC_DOTTED_NETMASK =
	  COMMON::convertBitmaskToDotted($PUBLIC_BIT_NETMASK);
	$PUBLIC_BIT_NETWORK   = COMMON::convertDottedToBitmask($2);
	$PUBLIC_DOTTED_NETWORK =
	  COMMON::convertBitmaskToDotted($PUBLIC_BIT_NETWORK);
      }

      elsif ($1 eq "EXTERNAL_ACCESS") {
	#
	#  EXTERNAL_ACCESS=<host name> <IP address> <NIC>
	#
	$value=$2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($3 eq "") {
	  COMMON::error("EXTERNAL_ACCESS: invalid number of parameters (line $., file $filename)") ;
	}
	$USE_NSM = "YES";
        $USE_EXTERNAL_ACCESS = "YES";
	$EXTERNAL_HOSTNAME  = $1;
	$EXTERNAL_IP  = $2;
	$EXTERNAL_NIC = $3;
      }
      
      else {
	COMMON::error("$1: unknown parameter (line $., file $filename)") ;
      }
    }
    
    close(FCLUSTER) ;
  }

#------------------------------------------------------------------------------
#
#  load addon definition
#
#------------------------------------------------------------------------------

sub load_addon
  {
    my ($filename) = @_ ;

    if (! -f $filename) {
      $DO_ADDON = FALSE ;
      return ;
    }

    $DO_ADDON = TRUE;

    if ($G_verbose eq TRUE) {
      print "Loading Add-on definition from $filename\n" ;
    }

    open(FADDON, "<$filename") or die("Can't access $filename") ;
    while (<FADDON>) {
      chomp($line = $_) ;
      $line =~ s/#.*// ;
      if ($line eq "") {
	next ;
      }
      if ($line !~ m/[\s\t]*([^\s\t]*)[\s\t]*=[\s\t]*(.*)/) {
	COMMON::error("$line: invalid syntax (line $., file $filename)");
      }

      if ($1 eq "PATCH") {
	#
	#   PATCH=<reference> <dir> <subdir> <phase> <scope> [ <MEN> <NMEN> ]
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($5 eq "") {
	  COMMON::error("patch $1: invalid number of parameters (line $., file $filename)") ;
	}
	my $patch = {} ;
	$patch->{reference} = $1 ;
	$patch->{dir} = $2 ;
	$patch->{subdir} = $3 ;
	if ($3 eq "-") {
	  $patch->{subdir} = "";
	}
	$patch->{phase} = $4 ;
	$patch->{scope} = $5 ;
	$patch->{men} = $6 ;
	$patch->{nmen} = $7;
	check_addon_directive("patch",
			      $patch->{reference},
			      $.,
			      $filename,
			      $patch->{phase},
			      $patch->{scope},
			      $patch->{men},
			      $patch->{nmen}
			     ) ;
	push @PATCH_LIST, $patch;

      } elsif ($1 eq "PACKAGE") {
	#
	#   PACKAGE=<name> <dir> <subdir> <phase> <scope> [ <MEN> <NMEN> ]
	#
	$value = $2 ;
	$value =~ m/[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)[\s\t]*([^\s\t]*)/;
	if ($5 eq "") {
	  COMMON::error("package $1: invalid number of parameters (line $., file $filename)") ;
	}
	my $package = {} ;
	$package->{reference} = $1 ;
	$package->{dir} = $2 ;
	$package->{subdir} = $3 ;
	if ($3 eq "-") {
	  $package->{subdir} = "";
	}
	$package->{phase} = $4 ;
	$package->{scope} = $5 ;
	$package->{men} = $6 ;
	$package->{nmen} = $7;
	check_addon_directive("package",
			      $package->{reference},
			      $.,
			      $filename,
			      $package->{phase},
			      $package->{scope},
			      $package->{men},
			      $package->{nmen}
			     ) ;
	push @PACKAGE_LIST, $package;
      }
      else {
	COMMON::error("$1: unknown parameter (line $., file $filename)") ;
      }
    }

    close(FADDON) ;
  }

#------------------------------------------------------------------------------
#
#  check cluster configuration
#
#------------------------------------------------------------------------------

sub check_cluster
  {
    my $i;
    my $j;

    # check mandatory information
    if ($CLUSTER eq "") {
      COMMON::error("CLUSTER_ID not defined") ;
    }

    # check that node ids are all different
    for ($i = 0; $i <= ($#NODE_LIST - 1); $i++) {
      for ($j = $i+1; $j <= $#NODE_LIST; $j++) {
	if ($NODE_LIST[$i]{id} == $NODE_LIST[$j]{id}) {
	  COMMON::error("Node Id $NODE_LIST[$i]{id} is defined twice in the cluster definition file");
	}
      }
    }

    # check coherency and set default value

    # check default network interfaces
    if ($USE_CGTP eq "YES") {
      if (($MEN_NIC0 eq "") || ($MEN_NIC1 eq "")) {
	  COMMON::error("Both interfaces must be defined for MEN nodes (MEN_INTERFACES directive)") ;
	}	
      if (($NMEN_NIC0 eq "") || ($NMEN_NIC1 eq "")) {
	  COMMON::error("Both interfaces must be defined for NMEN nodes (NMEN_INTERFACES directive)") ;
	}	
      if ($USE_CLUSTER_NETWORK eq "TRUE") {
	if ($CLUSTER_DOTTED_NIC1_NET eq "") {
	  COMMON::error("NIC1 sub-network must be defined (CLUSTER_NETWORK directive)") ;
	}
	if ($CLUSTER_DOTTED_CGTP_NET eq "") {
	  COMMON::error("CGTP sub-network must be defined (CLUSTER_NETWORK directive)") ;
	}
      }	
    } else {
      if ($MEN_NIC0 eq "") {
	  COMMON::error("The first interface must be defined for MEN nodes (MEN_INTERFACES directive)") ;
	}	
      if ($NMEN_NIC0 eq "") {
	  COMMON::error("The first interface must be defined for NMEN nodes (NMEN_INTERFACES directive)") ;
	}	
    }
    
    # check that postfixes are all different
    if ($POST_NIC0 eq $POST_NIC1) {
      COMMON::error("postfix for NIC0 and NIC1 must be different"); 
    }
    if ($POST_NIC0 eq $POST_CGTP) {
      COMMON::error("postfix for NIC0 and CGTP must be different"); 
    }
    if ($POST_NIC1 eq $POST_CGTP) {
      COMMON::error("postfix for NIC1 and CGTP must be different"); 
    }

    # check master eligible nodes
    for $i (0..1) {
      if ($NODE_LIST[$i]{eth0} eq "") {
	COMMON::error("MAC address is required for node $NODE_LIST[$i]{id}\n") ;
      }
      if ($NODE_LIST[$i]{nic0} eq "") {
	# no nic0 defined, set to the default value
	$NODE_LIST[$i]{nic0} = $MEN_NIC0;
      }
      if ($NODE_LIST[$i]{nic1} eq "") {
	# no nic1 defined, set to the default value
	$NODE_LIST[$i]{nic1} = $MEN_NIC1;
      }
    }

    # check public interfaces
    if ($USE_PUBLIC_NETWORK eq "YES") {
      for $i (0..$#NODE_LIST) {
	if (($NODE_LIST[$i]{pubName} eq "") || ($NODE_LIST[$i]{pubIp} eq "") || ($NODE_LIST[$i]{pubNic} eq "")) {
	  COMMON::error("Node id $NODE_LIST[$i]{id}, full definition for public network required (name, IP address and interface)");
	}
      }
    } else {
      for $i (0..$#NODE_LIST) {
	if (($NODE_LIST[$i]{pubName} ne "") || ($NODE_LIST[$i]{pubIp} ne "") || ($NODE_LIST[$i]{pubNic} ne "")) {
	  COMMON::warning("Node id $NODE_LIST[$i]{id}, attributes for the public network are ignored (PUBLIC_NETWORK not defined)");
	}
      }
    }
    
    # check diskless boot
    my $need_arg = FALSE;
    my $arg_type = "";
    my $need_interface = FALSE;
    if (($DISKLESS_BOOT_POLICY eq "DHCP_DYNAMIC") || 
	($DISKLESS_BOOT_POLICY eq "NLS_DYNAMIC")) {
      $need_arg = FALSE;
      $arg_type = "";
      $need_interface = FALSE;
    } elsif (($DISKLESS_BOOT_POLICY eq "DHCP_STATIC") ||
	     ($DISKLESS_BOOT_POLICY eq "NLS_MAC_ADDR")) {
      $need_arg = TRUE;
      $arg_type = "MAC addresses";
      $need_interface = TRUE;
    } elsif ($DISKLESS_BOOT_POLICY eq "NLS_CPCI_SLOT") {
      COMMON::error("Boot policy based on CPCI slots not yet supported by the installation tool");
#      $need_arg = TRUE;
#      $arg_type = "";
#      $need_interface = TRUE;
    } elsif ($DISKLESS_BOOT_POLICY eq "NLS_SWITCH_PORT") {
      COMMON::error("Boot policy based on switch ports not yet supported by the installation tool");
#      $need_arg = TRUE;
#      $arg_type = "switch port numbers";
#      $need_interface = TRUE;
    }
      
    # check diskless nodes
    for $i (2..$#NODE_LIST) {

      if ($need_arg eq "TRUE") {
	if (($NODE_LIST[$i]{eth0} eq "") || ($NODE_LIST[$i]{eth1} eq "")) {
	  COMMON::error("NODE $NODE_LIST[$i]{id}: $arg_type mandatory for the required boot policy ($DISKLESS_BOOT_POLICY)\n") ;
	}
      } else {
	if (($NODE_LIST[$i]{eth0} ne "") || ($NODE_LIST[$i]{eth1} ne "")) {
	  COMMON::warning("NODE $NODE_LIST[$i]{id}: Client ID ignored for the required boot policy ($DISKLESS_BOOT_POLICY)");
	}
      }
      
      if ($need_interface eq "TRUE"){
	if ($NODE_LIST[$i]{nic0} eq "") {
	  # no nic0 defined, set to the default value
	  $NODE_LIST[$i]{nic0} = $NMEN_NIC0;
	}
	if ($NODE_LIST[$i]{nic1} eq "") {
	  # no nic1 defined, set to the default value
	  $NODE_LIST[$i]{nic1} = $NMEN_NIC1;
	}
	
      } else {
	if (($NODE_LIST[$i]{nic0} ne "") || ($NODE_LIST[$i]{nic1} ne "")) {
	  COMMON::warning("NODE $NODE_LIST[$i]{id}: NIC0 and NIC1 ignored for the required policy ($DISKLESS_BOOT_POLICY)");
	}
	# set to the default value
	$NODE_LIST[$i]{nic0} = $NMEN_NIC0;
	$NODE_LIST[$i]{nic1} = $NMEN_NIC1;
      }
    }
   
  }
    
#------------------------------------------------------------------------------
#
#  load environment variables used for configuration
#
#------------------------------------------------------------------------------

sub load_variables {

  # debug mode
  if (defined($ENV{NHINSTALL_DEBUG})) {
    my $debug_mode = "$ENV{NHINSTALL_DEBUG}";
    if ($debug_mode eq "all") {
      $DEBUG_FILE = TRUE;
      $DEBUG_CMD  = TRUE;
      $DEBUG_ACTION  = TRUE;
      $DEBUG_CONFIG = TRUE;
    } else {
      foreach my $mode (split(/,/, $debug_mode)) {
	if ($mode eq "action") {
	  $DEBUG_ACTION = TRUE;
	} elsif ($mode eq "file") {
	  $DEBUG_FILE = TRUE;
	} elsif ($mode eq "cmd") {
	  $DEBUG_CMD = TRUE;
	} elsif ($mode eq "conf") {
	  $DEBUG_CONFIG = TRUE;
	} else {
	  COMMON::warning("debug mode $mode unknown: ignored");
	}
      }
    }
  }
  
  # process variables
  if (!defined($ENV{SERVER_INTERFACE}) || ($ENV{SERVER_INTERFACE} eq "")) {
    COMMON::error("SERVER_INTERFACE (network interface used to access cluster) is not defined") ;
  }
  $IFACE=$ENV{SERVER_INTERFACE};
  
  if (!defined($ENV{SERVER_NODE}) || ($ENV{SERVER_NODE} eq "")) {
    COMMON::error("SERVER_NODE (node id of the installation server) is not defined") ;
  }
  $SERVER_NODE = $ENV{SERVER_NODE} ;
  if ($SERVER_NODE <= 0) {
    COMMON::error("Wrong SERVER_NODE value ($SERVER_NODE): must be greater than 0") ;
  }
  if ($SERVER_NODE == $MASTER_ID) {
    COMMON::error("Wrong SERVER_NODE value ($SERVER_NODE): cannot be equal to master node id") ;
  }
  
  $SOLARIS_INSTALL = $ENV{SOLARIS_INSTALL} ;
  if (($SOLARIS_INSTALL ne "YES") && ($SOLARIS_INSTALL ne "NO")) {
    COMMON::error("SOLARIS_INSTALL ($SOLARIS_INSTALL) must be set to YES or NO") ;
  }
  
  $AUTO_REBOOT = $ENV{AUTO_REBOOT} ;
  if (($AUTO_REBOOT ne "YES") && ($AUTO_REBOOT ne "NO")) {
    COMMON::error("AUTO_REBOOT ($AUTO_REBOOT) must be set to YES or NO") ;
  }
  
  $TARGET_OS = $ENV{TARGET_OS} ;
  if (($TARGET_OS ne "Solaris_8") && ($TARGET_OS ne "Solaris_9")) {
    COMMON::error("TARGET_OS ($TARGET_OS) must be set to Solaris_8 or Solaris_9") ;
  }
  
  $SOLARIS_DIR             = $ENV{SOLARIS_DIR};
  $SOLARIS_PACKAGE_SUBDIR  = $ENV{SOLARIS_PACKAGE_SUBDIR};
  $SOLARIS_PATCH_SUBDIR    = $ENV{SOLARIS_PATCH_SUBDIR};
  $JUMPSTART               = $ENV{JUMPSTART_DIR};
  $PROFILE                 = $ENV{NHINSTALL_PROFILE_FILE};
  $WORKING_DIR             = $ENV{WORKING_DIR};
  $ROOT_DIR                = $ENV{ROOT_DIR};
  $PACKAGE_SUBDIR          = $ENV{PACKAGE_SUBDIR};
  $PATCH_SUBDIR            = $ENV{PATCH_SUBDIR};

  if (!defined($ENV{NHINSTALL_TESTING})) {
    $TESTING = "";
  } else {
    $TESTING = $ENV{NHINSTALL_TESTING};
  }
}

#------------------------------------------------------------------------------
#
#  look for a slice with a particular attribute
#
#------------------------------------------------------------------------------

sub slice_search {
  my ($attribute, $value, $idx) = @_ ;
  
  for my $i (0..$#SLICE_LIST) {
    if ($SLICE_LIST[$i]{$attribute} eq $value) {
      $$idx = $i ;
      return TRUE;
    }
  }
  
  return FALSE;
}

#------------------------------------------------------------------------------
#
#  check slice configuration
#
#------------------------------------------------------------------------------

sub check_slice {
  my $idx ;
  
  if (slice_search(mounting_point, $SLICE_ROOT, \$idx) ne "TRUE") {
    COMMON::error("A slice with $SLICE_ROOT (root) as mounting point is required") ;
  }
  
  if ($SLICE_LIST[$idx]{size} < $SLICE_ROOT_MIN_SIZE) {
    COMMON::error("Minimum size for $SLICE_ROOT (root) must be $SLICE_ROOT_MIN_SIZE") ;
  }
  
  if (slice_search(mounting_point, $SLICE_SUNWCGHA, \$idx) ne "TRUE") {
    COMMON::error("A slice with $SLICE_SUNWCGHA as mounting point is required") ;
  }
  
  if ($SLICE_LIST[$idx]{size} < $SLICE_SUNWCGHA_MIN_SIZE) {
    COMMON::error("Minimum size for $SLICE_SUNWCGHA must be $SLICE_SUNWCGHA_MIN_SIZE") ;
  }
  
  # if the user ask for no diskless, SLICE_EXPORT is not mandatory
  my $diskless_required = SEQUENCER::get_data("DISKLESS_ENV");
  if ((scalar(@NODE_LIST) > 2) || ($diskless_required eq "YES")) {
    if (slice_search(mounting_point, $SLICE_EXPORT, \$idx) ne "TRUE") {
      COMMON::error ("A slice with $SLICE_EXPORT as mounting point is required") ;
    }
    my $DISKLESS_NUMBER = scalar(@NODE_LIST) - 2;
    my $MIN_SIZE = $SLICE_EXPORT_MIN_SIZE + $DISKLESS_NUMBER * $SLICE_EXPORT_MIN_SIZE_PER_DISKLESS;
    if ($SLICE_LIST[$idx]{size} < $MIN_SIZE) {
      COMMON::error("Minimum size for $SLICE_EXPORT must be $MIN_SIZE") ;
    }
  }
  
  # check the bitmap slice
  for my $i (0..$#SLICE_LIST) {
    if ($SLICE_LIST[$i]{bitmap} ne "") {

      # the bitmap slice must exist
      if (slice_search("name", $SLICE_LIST[$i]{bitmap}, \$idx) ne "TRUE") {
	COMMON::error("Bitmap slice $SLICE_LIST[$i]{bitmap} for slice $SLICE_LIST[$i]{name} not defined");
      }

      # it can't replicated itself
      if ($idx == $i) {
	COMMON::error("Slice $SLICE_LIST[$i]{name} can not be replicated on itself");
      }

      # it is defined as "unnamed"
      if ($SLICE_LIST[$idx]{mounting_point} ne $SLICE_UNMOUNTED) {
	COMMON::error("Bitmap slice $SLICE_LIST[$i]{bitmap} for slice $SLICE_LIST[$i]{name} is not an $SLICE_UNMOUNTED slice");
      }

      # it is not used for another replicated slice
      for my $j (0..$#SLICE_LIST) {
	if (($i ne $j) && ($SLICE_LIST[$j]{bitmap} ne "")) {
	  if ($SLICE_LIST[$i]{bitmap} eq $SLICE_LIST[$j]{bitmap}) {
	    COMMON::error("Bitmap slice $SLICE_LIST[$i]{bitmap} is used twice for different slices");
	  }
	}
      }
    }
  }
}

#------------------------------------------------------------------------------
#
#  check exported directories
#
#------------------------------------------------------------------------------

sub check_exported {

  my $i;
  my $j;
  my $found;
  my $idx;

  #
  # each exported directory must be a child of a replicated partition
  #
  for $i (0..$#DIR_EXPORTED) {
    $found = FALSE ;
    for $j (0..$#SLICE_LIST) {
      if ($SLICE_LIST[$j]{bitmap} ne "") {
	# look for slice name as a subtring of directory name
	# add a slash at the end of the exported dir and mounting point name
	# if not already present to avoid problem due to similar partial name
	# ex: exported=/SUNWcgha/local/foodoo and mounting point /SUNWcgha/local/foo
	my $exported = $DIR_EXPORTED[$i] ;
	if (substr($exported, length($exported)-1, 1) ne "/") {
	  $exported = $exported . "/" ;
	}
	my $mounting_point = $SLICE_LIST[$j]{mounting_point} ;
	if (substr($mounting_point, length($mounting_point)-1, 1) ne "/") {
	  $mounting_point = $mounting_point . "/" ;
	}
	
	my $lg = length($mounting_point) ;
	if (substr($exported, 0, $lg) eq $mounting_point) {
	  $found = TRUE ;
	  last ;
	}
      }
    }
    if ($found eq FALSE) {
      COMMON::error("Exported directory $DIR_EXPORTED[$i] is not located on a replicated slice") ;
    }
  }
  
  #
  # check that an exported directory is not the child of an already exported directory
  #
  for $i (0..$#DIR_EXPORTED) {
    $found = FALSE ;
    for $j (0..$#DIR_EXPORTED) {
      if ($i == $j) {
	next ;
      }
      
      my $exported1 = $DIR_EXPORTED[$i] ;
      if (substr($exported1, length($exported1)-1, 1) ne "/") {
	$exported1 = $exported1 . "/" ; 
      }
      my $exported2 = $DIR_EXPORTED[$j] ;
      if (substr($exported2, length($exported2)-1, 1) ne "/") {
	$exported2 = $exported2 . "/" ;
      }
      
      my $lg = length($exported1) ;
      if (substr($exported2, 0, $lg) eq $exported1) {
	$found = TRUE ;
	$idx = $j; # problem with scope of variable
	last ;
      }
    }

    if ($found eq "TRUE") {
      COMMON::error("Exported directory $DIR_EXPORTED[$idx] is child of an already exported directory ($DIR_EXPORTED[$i])") ;
    }
  }
}

#------------------------------------------------------------------------------
#
#  check remote mounted directories
#
#------------------------------------------------------------------------------

sub check_remote
  {
    #
    # each remote mounted directory must be a child of an exported directory
    #
    for $i (0..$#DIR_MOUNTED) {
      my $found = FALSE ;
      for $j (0..$#DIR_EXPORTED) {
	  # look for exported directory name as a subtring of remote directory name
	  # add a slash at the end of the exported dir and remote directory
	  # if not already present to avoid problem due to similar partial name
	  # ex: exported=/user1 and mounting point /user12
	  my $remote = $DIR_MOUNTED[$i]{remote} ;
	  if (substr($remote, length($remote)-1, 1) ne "/") {
	    $remote = $remote . "/" ;
	  }
	  my $exported = $DIR_EXPORTED[$j] ;
	  if (substr($exported, length($exported)-1, 1) ne "/") {
	    $exported = $exported . "/" ;
	  }

	my $lg = length($exported);
	if (substr($remote, 0, $lg) eq $exported) {
	  $found = TRUE ;
	  last ;
	}
      }
      if ($found eq FALSE) {
	COMMON::error("Remote directory $DIR_MOUNTED[$i]{remote} for mounting point $DIR_MOUNTED[$i]{local} is not child of an exported directory") ;
      }
    }
  }
#------------------------------------------------------------------------------
#
#  check slices when SVM is selected
#
#------------------------------------------------------------------------------

sub check_svm {

  my $idx ;

  if ($USE_SVM ne "YES") {
    return ;
  }

  if ($SOLARIS_INSTALL eq "NO") {
    COMMON::error("LOGICAL_SLICE_SUPPORT must be set to NO when Solaris installation is skipped (SVM configuration is under your responsibility)");
  }

  if ($ENV{OPTIMIZED_INSTALL} ne "YES") {
    COMMON::error("OPTIMIZED_INSTALL must be set to YES when configuring slices with DiskSuite");
  }

  # for each slice replicated or for storing bitmap slice
  # a concat metadevice will be created
  # raw device and block device are changed accordingly
  for my $i (0..$#SLICE_LIST) {
    if (($SLICE_LIST[$i]{bitmap} ne "") || ($SLICE_LIST[$i]{mounting_point} eq $SLICE_UNMOUNTED)) {
      my $meta_number = $i + 1;
      $SLICE_LIST[$i]{metadevice} = "d$meta_number";
      $SLICE_LIST[$i]{raw} = "/dev/md/rdsk/d$meta_number";
      $SLICE_LIST[$i]{block} = "/dev/md/dsk/d$meta_number";
    }
  }

  # check slice used for meta database
  if (slice_search(mounting_point, $SLICE_TAG_REPLICA, \$idx) ne "TRUE") {
    COMMON::error("A slice with $SLICE_TAG_REPLICA attribute (as mounting point) is required") ;
  }
  # its name is changed (only unnamed is supported by jumpstart)
  $SLICE_REPLICA = $SLICE_LIST[$idx]{block};
  $SLICE_LIST[$idx]{mounting_point} = $SLICE_UNMOUNTED ;
}

#------------------------------------------------------------------------------
#
#  display_conf
#
#    display loaded configuration when debug is requested
#
#------------------------------------------------------------------------------

sub display_conf {


  my $network;
  my $netmask;
  if ($DEBUG_CONFIG ne "TRUE") {
    return ;
  }

  # display general information
  COMMON::print_debug(DEBUG_CONFIG, "General setting");
  COMMON::print_debug(DEBUG_CONFIG, "---------------");

  COMMON::print_debug(DEBUG_CONFIG, "    Cluster Id:          $CLUSTER");
  COMMON::print_debug(DEBUG_CONFIG, "    OS:                  $TARGET_OS");
  COMMON::print_debug(DEBUG_CONFIG, "    Solaris intallation: $SOLARIS_INSTALL");
  COMMON::print_debug(DEBUG_CONFIG, "    Arch:                $ARCH");
  COMMON::print_debug(DEBUG_CONFIG, "    Nic0 postfix:        $POST_NIC0");
  COMMON::print_debug(DEBUG_CONFIG, "    Nic1 postfix:        $POST_NIC1");
  COMMON::print_debug(DEBUG_CONFIG, "    CGTP postfix:        $POST_CGTP");
  COMMON::print_debug(DEBUG_CONFIG, "    Boot policy:         $DISKLESS_BOOT_POLICY");
  COMMON::print_debug(DEBUG_CONFIG, "    CGTP used:           $USE_CGTP");
  COMMON::print_debug(DEBUG_CONFIG, "    SVM used:            $USE_SVM");
  COMMON::print_debug(DEBUG_CONFIG, "    DCSS used:           $USE_DCSS");
  COMMON::print_debug(DEBUG_CONFIG, "    WDT used:            $USE_WDT");      

  COMMON::print_debug(DEBUG_CONFIG, "    Netmask:             $CLUSTER_DOTTED_NETMASK");
  COMMON::print_debug(DEBUG_CONFIG, "    NIC0 subnet:         $CLUSTER_DOTTED_NIC0_NET");
  if ($USE_CGTP eq "YES") {
    COMMON::print_debug(DEBUG_CONFIG, "    NIC1 subnet:         $CLUSTER_DOTTED_NIC1_NET");
    COMMON::print_debug(DEBUG_CONFIG, "    CGTP subnet:         $CLUSTER_DOTTED_CGTP_NET");
  }
  COMMON::print_debug(DEBUG_CONFIG, "");
  
  # display server information
  COMMON::print_debug(DEBUG_CONFIG, "Installation server");
  COMMON::print_debug(DEBUG_CONFIG, "-------------------");
  COMMON::print_debug(DEBUG_CONFIG, "");
  COMMON::print_debug(DEBUG_CONFIG, "    Node Id:           $SERVER_NODE");
  COMMON::print_debug(DEBUG_CONFIG, "    IP:                $SERVER_IP");
  COMMON::print_debug(DEBUG_CONFIG, "    Network interface: $IFACE");
  COMMON::print_debug(DEBUG_CONFIG, "    Network:           $DOTTED_NETWORK");
  COMMON::print_debug(DEBUG_CONFIG, "    Netmask:           $DOTTED_NETMASK");
  COMMON::print_debug(DEBUG_CONFIG, "");

  # display the node list
  COMMON::print_debug(DEBUG_CONFIG, "Node list");
  COMMON::print_debug(DEBUG_CONFIG, "---------");
  
 
  for my $node (0..$#NODE_LIST) {
    my $node_id  = $NODE_LIST[$node]{id};
    my $nodename = COMMON::build_node_name($node_id, "");
    COMMON::print_debug(DEBUG_CONFIG, "");
    COMMON::print_debug(DEBUG_CONFIG, 
			"  Node ID: $NODE_LIST[$node]{id}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    name:              $nodename");
    COMMON::print_debug(DEBUG_CONFIG,
			"    MAC address #0:    $NODE_LIST[$node]{eth0}");
    COMMON::print_debug(DEBUG_CONFIG, 
			"    MAC address #1:    $NODE_LIST[$node]{eth1}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    net interface #0:  $NODE_LIST[$node]{nic0}");
    COMMON::print_debug(DEBUG_CONFIG, 
			"    net interface #1:  $NODE_LIST[$node]{nic1}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    IP #0:             $NODE_LIST[$node]{nic0Ip}");
    if ($USE_CGTP eq "YES") {
      COMMON::print_debug(DEBUG_CONFIG,
			  "    IP #1:             $NODE_LIST[$node]{nic1Ip}");
      COMMON::print_debug(DEBUG_CONFIG,
			  "    IP CGTP:           $NODE_LIST[$node]{cgtpIp}");
    }
    if ($USE_PUBLIC_NETWORK eq "YES") {
      COMMON::print_debug(DEBUG_CONFIG, 
			  "    public host name:  $NODE_LIST[$node]{pubName}");
      COMMON::print_debug(DEBUG_CONFIG, 
			  "    public interface:  $NODE_LIST[$node]{pubNic}");
      COMMON::print_debug(DEBUG_CONFIG, 
			  "    public IP address: $NODE_LIST[$node]{pubIp}");
    }
  }

  COMMON::print_debug(DEBUG_CONFIG, "");

  # display the slice list
  COMMON::print_debug(DEBUG_CONFIG, "Slice list");
  COMMON::print_debug(DEBUG_CONFIG, "----------");

  for my $slice (0..$#SLICE_LIST) {
    COMMON::print_debug(DEBUG_CONFIG, "");
    COMMON::print_debug(DEBUG_CONFIG,
			"  Name: $SLICE_LIST[$slice]{name}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    mounting point: $SLICE_LIST[$slice]{mounting_point}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    size:           $SLICE_LIST[$slice]{size}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    option:         $SLICE_LIST[$slice]{option}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    raw device:     $SLICE_LIST[$slice]{raw}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    block device:   $SLICE_LIST[$slice]{block}");
    COMMON::print_debug(DEBUG_CONFIG,
			"    metadevice:     $SLICE_LIST[$slice]{metadevice}");
  }

  COMMON::print_debug(DEBUG_CONFIG, "");

  # display the list of exported directories
  COMMON::print_debug(DEBUG_CONFIG, "Exported directories");
  COMMON::print_debug(DEBUG_CONFIG, "--------------------");
  COMMON::print_debug(DEBUG_CONFIG, "");
  for my $i (0..$#DIR_EXPORTED) {
    COMMON::print_debug(DEBUG_CONFIG, "    $DIR_EXPORTED[$i]");
  }

  COMMON::print_debug(DEBUG_CONFIG, "");

  # display the list of mounted directories
  COMMON::print_debug(DEBUG_CONFIG, "Mounted directories");
  COMMON::print_debug(DEBUG_CONFIG, "-------------------");
  for my $i (0..$#DIR_MOUNTED) {
    COMMON::print_debug(DEBUG_CONFIG, "");
    COMMON::print_debug(DEBUG_CONFIG, "    local:  $DIR_MOUNTED[$i]{local}");
    COMMON::print_debug(DEBUG_CONFIG, "    remote: $DIR_MOUNTED[$i]{remote}");
  }

  COMMON::print_debug(DEBUG_CONFIG, "");

  # display the vendor list
  if (scalar(@VENDOR_TYPE) > 0) {
    COMMON::print_debug(DEBUG_CONFIG, "Vendor type");
    COMMON::print_debug(DEBUG_CONFIG, "-----------");
    COMMON::print_debug(DEBUG_CONFIG, "");
    for my $i (0..$#VENDOR_TYPE) {
      COMMON::print_debug(DEBUG_CONFIG, "    $VENDOR_TYPE[$i]");
    }
  }
  

  # display the patch list
  if (scalar(@PATCH_LIST) > 0) {
    COMMON::print_debug(DEBUG_CONFIG, "");
    COMMON::print_debug(DEBUG_CONFIG, "Patch list");
    COMMON::print_debug(DEBUG_CONFIG, "-----------");
    for my $i (0..$#PATCH_LIST) {
      COMMON::print_debug(DEBUG_CONFIG, "");
      COMMON::print_debug(DEBUG_CONFIG, "  ID: $PATCH_LIST[$i]{reference}");
      COMMON::print_debug(DEBUG_CONFIG, "    Directory:     $PATCH_LIST[$i]{dir}");
      COMMON::print_debug(DEBUG_CONFIG, "    Sub-directory: $PATCH_LIST[$i]{dir}");
      COMMON::print_debug(DEBUG_CONFIG, "    Phase:         $PATCH_LIST[$i]{phase}");
      COMMON::print_debug(DEBUG_CONFIG, "    Scope:         $PATCH_LIST[$i]{scope}");
      COMMON::print_debug(DEBUG_CONFIG, "    On MEN:        $PATCH_LIST[$i]{men}");
      COMMON::print_debug(DEBUG_CONFIG, "    On NMEN:       $PATCH_LIST[$i]{nmen}");
    }
  }

  # display the package list
  if (scalar(@PACKAGE_LIST) > 0) {
    COMMON::print_debug(DEBUG_CONFIG, "");
    COMMON::print_debug(DEBUG_CONFIG, "Package list");
    COMMON::print_debug(DEBUG_CONFIG, "------------");
    for my $i (0..$#PACKAGE_LIST) {
      COMMON::print_debug(DEBUG_CONFIG, "");
      COMMON::print_debug(DEBUG_CONFIG, "  Name: $PACKAGE_LIST[$i]{reference}");
      COMMON::print_debug(DEBUG_CONFIG, "    Directory:     $PACKAGE_LIST[$i]{dir}");
      COMMON::print_debug(DEBUG_CONFIG, "    Sub-directory: $PACKAGE_LIST[$i]{dir}");
      COMMON::print_debug(DEBUG_CONFIG, "    Phase:         $PACKAGE_LIST[$i]{phase}");
      COMMON::print_debug(DEBUG_CONFIG, "    Scope:         $PACKAGE_LIST[$i]{scope}");
      COMMON::print_debug(DEBUG_CONFIG, "    On MEN:        $PACKAGE_LIST[$i]{men}");
      COMMON::print_debug(DEBUG_CONFIG, "    On NMEN:       $PACKAGE_LIST[$i]{nmen}");
    }
  }
  
  COMMON::print_debug(DEBUG_CONFIG, "");
}

#------------------------------------------------------------------------------
#
#  checking of the configuration
#
#------------------------------------------------------------------------------

sub common_check
  {
    
    if (scalar(@NODE_LIST) < 2) {
      COMMON::error("At least 2 nodes must be defined (2 master eligible nodes)") ;
    }
    if (scalar(@NODE_LIST) > $MAX_NODES) {
      COMMON::error("Maximum $MAX_NODES nodes are supported (including 2 master eligible nodes)") ;
    }

    check_cluster ;
    check_slice ;
    check_exported ;
    check_remote ;
    check_svm;
  }

#------------------------------------------------------------------------------
#
#  check that a Host ID is compatible with a subnetwork mask
#
#------------------------------------------------------------------------------

sub checkHostId {
  my ($hostId, $bitmask) = @_;

  if (($hostId & $bitmask) != 0) {
    COMMON::error("Node id $hostId is not compatible with the cluster netmask");
  }
}
  
#------------------------------------------------------------------------------
#
#  configure global variables that are deducted from the loaded info
#  used by check_conf.pl
#
#------------------------------------------------------------------------------

sub configure_global {
  
  # set the default cluster network characteristics
  if ($USE_CLUSTER_NETWORK eq "FALSE") {
    $CLUSTER_DOTTED_NETMASK  = "255.255.255.0";
    $CLUSTER_DOTTED_NIC0_NET = "10.$CLUSTER.1.0";

    if ($USE_CGTP eq "YES") {
      $CLUSTER_DOTTED_NIC1_NET = "10.$CLUSTER.2.0";
      $CLUSTER_DOTTED_CGTP_NET = "10.$CLUSTER.3.0";
    }
  }

  # compute the bitmask from the dotted form
  $CLUSTER_BIT_NETMASK =
    COMMON::convertDottedToBitmask($CLUSTER_DOTTED_NETMASK);
  $CLUSTER_BIT_NIC0_NET = 
    COMMON::convertDottedToBitmask($CLUSTER_DOTTED_NIC0_NET);
  if ($USE_CGTP eq "YES") {
    $CLUSTER_BIT_NIC1_NET =
      COMMON::convertDottedToBitmask($CLUSTER_DOTTED_NIC1_NET);
    $CLUSTER_BIT_CGTP_NET =
      COMMON::convertDottedToBitmask($CLUSTER_DOTTED_CGTP_NET);
  }

  # reformat the dotted form from the bitmask that they had all the same
  # format
  $CLUSTER_DOTTED_NETMASK =
    COMMON::convertBitmaskToDotted($CLUSTER_BIT_NETMASK);
  $CLUSTER_DOTTED_NIC0_NETMASK =
    COMMON::convertBitmaskToDotted($CLUSTER_BIT_NIC0_NET);
  if ($USE_CGTP eq "YES") {
    $CLUSTER_DOTTED_NIC1_NETMASK =
      COMMON::convertBitmaskToDotted($CLUSTER_BIT_NIC1_NET);
    $CLUSTER_DOTTED_CGTP_NETMASK =
      COMMON::convertBitmaskToDotted($CLUSTER_BIT_CGTP_NET);
  }

  #information depending on network
  if ($USE_PUBLIC_NETWORK eq "YES") {
    if (! defined($ENV{SERVER_IP})) {
      COMMON::error("SERVER_IP not defined (required when a public network is used)");
    }
    $SERVER_IP = $ENV{SERVER_IP};
    $BIT_NETWORK   = $PUBLIC_BIT_NETWORK;
    $BIT_NETMASK   = $PUBLIC_BIT_NETMASK;
  } else {
    if (defined($ENV{SERVER_IP})) {
      COMMON::warning("SERVER_IP ignored (only required when a public network is used)");
    }
    
    $BIT_NETWORK   = $CLUSTER_BIT_NIC0_NET;
    $BIT_NETMASK   = $CLUSTER_BIT_NETMASK;
    $SERVER_IP = COMMON::computeIp($SERVER_NODE, 
				   $CLUSTER_BIT_NIC0_NET,
				   $CLUSTER_BIT_NETMASK);
  }
  $DOTTED_NETMASK = COMMON::convertBitmaskToDotted($BIT_NETMASK);
  $DOTTED_NETWORK = COMMON::convertBitmaskToDotted($BIT_NETWORK);


  # compute the IP address of each node
  for my $i (0..$#NODE_LIST) {

    checkHostId($NODE_LIST[$i]{id},
		$CLUSTER_BIT_NETMASK);

    $NODE_LIST[$i]{nic0Ip} = COMMON::computeIp($NODE_LIST[$i]{id}, 
					       $CLUSTER_BIT_NIC0_NET,
					       $CLUSTER_BIT_NETMASK);
    if ($USE_CGTP eq "YES") {
      $NODE_LIST[$i]{nic1Ip} =  COMMON::computeIp($NODE_LIST[$i]{id},
						  $CLUSTER_BIT_NIC1_NET,
						  $CLUSTER_BIT_NETMASK);
      $NODE_LIST[$i]{cgtpIp} =  COMMON::computeIp($NODE_LIST[$i]{id},
						  $CLUSTER_BIT_CGTP_NET,
						  $CLUSTER_BIT_NETMASK);
    }
  }
    

  # diskless environment definition for service, packages and patches
  if ($USE_DCSS eq "NO") {
    $SERVICE_USR_PATH            = "/export/$TARGET_OS/usr_sparc.all";
    $SERVICE_USR_BASEDIR         = "default";
  } else {
    $SERVICE_USR_PATH            = "/export/exec/$TARGET_OS/";
    $SERVICE_USR_BASEDIR         = "default";
  }
}


#------------------------------------------------------------------------------
#
#  check configuration
#
#------------------------------------------------------------------------------

sub check_conf
  {
    common_check ;

    configure_global;

    display_conf ;
  }

#------------------------------------------------------------------------------
#
#  load configuration
#
#  used by all sources except check_conf and diskless_found.pl
#
#------------------------------------------------------------------------------

sub load_conf
  {
    $G_verbose = FALSE ;

    load_variables ;
    load_cluster($ENV{NH_CLUSTER_FILE_FULLNAME}) ;
    load_addon($ENV{NH_ADDON_FILE_FULLNAME}) ;
  }

{
}
