#!/usr/bin/perl
# ------------------------------------------------------------------------------
#   ident "@(#)nmen_rbs.pl 1.31 02/11/13 SMI"
# ------------------------------------------------------------------------------
#
#  Copyright 2001 Sun Microsystems, Inc. All rights reserved.
#
# ------------------------------------------------------------------------------

package NMEN_RBS;

use variables;

#-------------------------------------------------------------------------------
#
#  compute the post-fix file name
#-------------------------------------------------------------------------------

sub getPostFix {
  my ($dotted_network) = @_;
  
  my $postfix = "";
  foreach my $byte (split('\.',$dotted_network)) {
    $postfix = "${postfix}_${byte}";
  }

  return $postfix;
}

#-------------------------------------------------------------------------------
#
# special command execution without standard error device redirection
#
#-------------------------------------------------------------------------------

sub exec_cmd_local {

    my ($command) = @_;

    COMMON::print_debug("DEBUG_CMD", "$command");

    @result = qx{$command 2> /dev/null};
  
    # in case of error, display output and exit
    if ( $? ne 0 ) {
      $status = $? / 256 ;
      print_log("-----------------------------------------------------------------------------\n");
      print_log("Error when executing: $command (status=$status)\n") ;
      foreach $line (@result)
	{
	  print_log("  $line\n") ;
	}
      print_log("-----------------------------------------------------------------------------\n");
       exit 1;
    }

   # if it's OK, display if tracing is required

    foreach $line (@result)
      {
	COMMON::print_debug("DEBUG_CMD", "  $line");
      }
  
    return 1;
}

#-------------------------------------------------------------------------------
#
# replace a entry in dhcpsvc.conf
#
#-------------------------------------------------------------------------------

sub replace_in_file {
   my ( $file, $update_entry_input, $update_value_input ) = @_ ;
   my $count=0;
   my $update_entry;
   my $update_value;
   my $garbage;
     
   open (FILE, $file ) || COMMON::error("Unable to open $file");
	open (FILE_TMP, ">${file}.tmp" ) || COMMON::error("Unable to open ${file}.tmp");
	
   while ( <FILE> ) {
      
      chop();
      ($update_entry, $update_value, $garbage) = split (/=/, $_) ;
            
      if ( "$update_entry" eq "$update_entry_input" ) {
			print FILE_TMP "${update_entry}=${update_value_input}\n";
			$count++;
		} else {
			print FILE_TMP "$_\n";
		}
		next;
       	        
   }
   close (FILE);
	close (FILE_TMP);
   
   unless ( $count ) {
      open (FILE_TMP, ">>${file}.tmp" ) || COMMON::error("Unable to update $file");
      print FILE_TMP "$update_entry_input=$update_value_input\n";
      close (FILE_TMP);
   }
	
	COMMON::exec_cmd ( "$MV ${file}.tmp ${file}") ||
	  COMMON::error ( "Unable to move ${file}.tmp ${file}");
   
}

#-------------------------------------------------------------------------------
#
# Management of /etc/inet/dhcpsvc.conf file
#
#-------------------------------------------------------------------------------
sub update_dhcpsvcconf {

   my ($proto_node_id) = @_;

   my $proto_name = COMMON::build_node_name($proto_node_id, "");
   my $proto_ip   = COMMON::get_external_ip($proto_node_id);
   my $men2_ip =    COMMON::get_external_ip($MEN2_ID);
	
	
   my $local_dhcpsvc_file = "$WORKING_DIR/dhcpsvc_${proto_name}";
   my $remote_dhcpsvc_file = "/etc/inet/dhcpsvc.conf";

   my $dhcp_container_dir="/SUNWcgha/remote/var/dhcp";

   if ( "$WORKAROUND_RBSCONFIG_SCRIPT" eq "FALSE" ) {

     COMMON::print_action("Updating DHCP service configuration parameters");
     #
     # retrieve  dhcpsvc.conf file
     #
     COMMON::get_file($proto_ip, "$remote_dhcpsvc_file", "$local_dhcpsvc_file");

     #
     # locally update dhcpsvc.conf file
     #
     open (DHCPSVC, ">>$local_dhcpsvc_file" ) || COMMON::error("Unable to open $local_dhcpsvc_file");
     if ($USE_CGTP eq "YES") {
       print  DHCPSVC "INTERFACES=$MEN_NIC0,$MEN_NIC1\n";
     } else {
       print  DHCPSVC "INTERFACES=$MEN_NIC0\n";
     }
     print  DHCPSVC "OFFER_CACHE_TIMEOUT=30\n";
     close (DHCPSVC);

     replace_in_file($local_dhcpsvc_file, "RESOURCE", "SUNWnhrbs") ||
       COMMON::error( "Unable to update RESOURCE line in $local_dhcpsvc_file" );

     replace_in_file($local_dhcpsvc_file, "PATH", "$dhcp_container_dir") ||
       COMMON::error( "Unable to update PATH line in $local_dhcpsvc_file" );

     #
     # remote copy dhcpsvc.conf file
     #
     COMMON::put_file($proto_ip, "$local_dhcpsvc_file", "$remote_dhcpsvc_file");
	
     #
     # update MEN2 with this file
     #
     COMMON::put_file($men2_ip, "$local_dhcpsvc_file", "$remote_dhcpsvc_file");
				
   }
}

#------------------------------------------------------------------------------------
# 
# Workaround to have on the second master eligible node the same environment for TFTP
# as the one on the first master eligible node:
#  - /tftpboot with the diskless hexadecimal address files linked to the inetboot file
#  - the service tftp is configured in /etc/inet/inetd.conf
# without this workaround diskless cannot be booted from this node when is elected as
# the master after a switch over/ fail over
#
#------------------------------------------------------------------------------------
sub rbs_vicemaster_workaround {

  my ($men1_ip, $men2_ip) = @_;

  # Copy /etc/inet/inetd.conf from first master eligible node to the second
  # master eligible node to have the needed entry of tftp when this latest node
  # becomes the master
  #
  COMMON::print_action("Transferring inetd.conf");
  COMMON::transfer_file($men1_ip, $men2_ip, "/etc/inet/inetd.conf");
    
  #
  # Create /tftpboot on the second master eligible node and populate it
  # with the files got from the /tftpboot on the first master eligible node
  # 
  COMMON::print_action("Copying /tftpboot directory from first MEN to second one");

  COMMON::remote_exec("$men2_ip", "$RRM -fr /tftpboot") ;
  COMMON::remote_exec("$men2_ip", "$RMKDIR /tftpboot") ||
    COMMON::error( "Unable to create remotely /tftpboot on ${men2_ip}" );
  
  # can not use standard exec_cmd command because of invalid output redirection
  exec_cmd_local("$RSH -n $men1_ip \"$RFIND /tftpboot | $RCPIO -omB\" > $WORKING_DIR/cpio.tmp") ||
    COMMON::error("creating cpio dump of /tftpboot from men1");
  COMMON::exec_cmd("$CAT $WORKING_DIR/cpio.tmp | $RSH $men2_ip $RCPIO -idumvB") ||
    COMMON::error("creating /tftpboot from cpio dump on men2");
    
}

#-------------------------------------------------------------------------------
#
# build Client ID
#
#-------------------------------------------------------------------------------

sub build_cid {

  my ($eth) = @_ ;

  # build the Client ID from the MAC address
  my $cid = "01" ;
  foreach $string (split(':',$eth)) { 
    if (length($string) == 0) {
      $cid .= "00" ;
    } elsif (length($string) == 1) {
      $cid .= "0$string" ;
    } else {
      $cid .= $string;
    }
  }

  # be sure that letters are uppercase
  $cid =~ tr/a-z/A-Z/;

  return $cid;
}

#-------------------------------------------------------------------------------
#
# configure diskless entries
#
#-------------------------------------------------------------------------------

sub diskless_configure {

  my ($proto_ip) = @_;
  my $node_ip_if0_template;
  my $node_ip_if1_template;
  my $node_ip_cgtp_template;
  
  my $node_name_template;
  my $node_name_if0_template;
  my $node_name_if1_template;

  my $net_if0 = $CLUSTER_DOTTED_NIC0_NET;
  my $net_if1 = $CLUSTER_DOTTED_NIC1_NET;

  my $cmd;

  COMMON::print_action("Adding diskless definition");
  #
  # remotely add DHCP network entries in DHCP containers by invoking pntadm -C
  #
  for my $node (0..$#NODE_LIST) {
    my $node_id = $NODE_LIST[$node]{id};
    
    if (($NODE_LIST[$node]{type} eq "DISKLESS") 
	&& ($NODE_LIST[$node]{concerned} eq "TRUE")) {
      
      $node_name_template = COMMON::build_node_name($node_id, "");
      $node_name_if0_template = COMMON::build_node_name($node_id, "NIC0");
      
      $node_ip_if0_template = COMMON::build_node_ip($node_id, "NIC0");

      COMMON::print_subaction("for $node_name_template ...");
      # the following two commands generate warning
      # so do not display them to avoid the customer to be confused
      
      my $subcmd0 = "-f PERMANENT" ;
      my $subcmd1 = "-f PERMANENT" ;
      if ($DISKLESS_BOOT_POLICY eq "DHCP_STATIC") {
	my $cid = build_cid($NODE_LIST[$node]{eth0}) ;
	$subcmd0 = "-f PERMANENT+MANUAL -i $cid" ;
	$cid = build_cid($NODE_LIST[$node]{eth1}) ;
	$subcmd1 = "-f PERMANENT+MANUAL -i $cid" ;
      }
      
      $cmd = "$RPNTADM -A $node_ip_if0_template $subcmd0 -h $node_name_if0_template -m $node_ip_if0_template $net_if0";
      COMMON::remote_exec_ignore ( "$proto_ip", $cmd) ||
	COMMON::print_debug(DEBUG_CMD," Cannot remotely execute $cmd but we don't care..." );

      if ($USE_CGTP eq "YES") {
      
	$node_name_if1_template = COMMON::build_node_name($node_id, "NIC1");;
	$node_ip_if1_template = COMMON::build_node_ip($node_id, "NIC1");
	$node_ip_cgtp_template = COMMON::build_node_ip($node_id, "CGTP");

	$cmd = "$RPNTADM -A $node_ip_if1_template $subcmd1 -h $node_name_if1_template -m $node_ip_if1_template $net_if1";
	COMMON::remote_exec_ignore ( "$proto_ip", $cmd) ||
	  COMMON::print_debug(DEBUG_CMD," Cannot remotely execute $cmd but we don't care" );
	

	$cmd = "$RDHTADM -A -m $node_ip_if0_template -d \':NhCgtpAddr=$node_ip_cgtp_template:Include=Common:Include=$net_if0:SrootPTH=\"\/export\/root\/$node_name_template\":SswapPTH=\"\/export\/swap\/$node_name_template\":\'";
	COMMON::remote_exec ( "$proto_ip", $cmd) ||
	  COMMON::error( " Cannot remotely execute $cmd" );
	
	$cmd = "$RDHTADM -A -m $node_ip_if1_template -d \':NhCgtpAddr=$node_ip_cgtp_template:Include=Common:Include=$net_if1:SrootPTH=\"\/export\/root\/$node_name_template\":SswapPTH=\"\/export\/swap\/$node_name_template\":\'";
	COMMON::remote_exec ( "$proto_ip", $cmd) ||
	  COMMON::error( " Cannot remotely execute $cmd" );

      } else {

	# do not define NhCgtpAddr

	$cmd = "$RDHTADM -A -m $node_ip_if0_template -d \':Include=Common:Include=$net_if0:SrootPTH=\"\/export\/root\/$node_name_template\":SswapPTH=\"\/export\/swap\/$node_name_template\":\'";
	COMMON::remote_exec ( "$proto_ip", $cmd) ||
	  COMMON::error( " Cannot remotely execute $cmd" );
      }
    }
  }
}

#-------------------------------------------------------------------------------
#
# Configure RBS 
#
#-------------------------------------------------------------------------------
sub rbs_configure {

  my ($proto_node_id) = @_;

  my $proto_ip   = COMMON::get_external_ip($proto_node_id);
	
  my $net_if0 = $CLUSTER_DOTTED_NIC0_NET;
  my $net_if1 = $CLUSTER_DOTTED_NIC1_NET;
	
  my $broadcast_if0;
  my $broadcast_if1;
	
  my $master_ip;
  my $master_ip_if0;
  my $master_ip_if1;
  my $master_ip_cgtp0;

  my $dhcp_container_dir="/SUNWcgha/local/export/data/var/dhcp";
  my $defaultdomain = "/etc/defaultdomain" ;
	
  my $dhcmd;
  
  my $platform;	
	
  my $men2_ip= COMMON::get_external_ip($MEN2_ID);
      
  $master_ip_if0   = COMMON::build_node_ip($MASTER_ID, "NIC0");
  $broadcast_if0 = COMMON::computeBroadcast($CLUSTER_BIT_NIC0_NET, $CLUSTER_BIT_NETMASK);
  if ($USE_CGTP eq "YES") {
    $master_ip_if1   = COMMON::build_node_ip($MASTER_ID, "NIC1");
    $master_ip_cgtp0 = COMMON::build_node_ip($MASTER_ID, "CGTP");
    $master_ip = $master_ip_cgtp0;
    $broadcast_if1 = COMMON::computeBroadcast($CLUSTER_BIT_NIC1_NET, $CLUSTER_BIT_NETMASK);;
  } else {
    $master_ip = $master_ip_if0;
  }


  #
  # remotely stop dhcp
  #
  COMMON::print_action("Stopping DHCP daemon");
  COMMON::remote_exec ( "$proto_ip", "/etc/rc2.d/K21dhcp stop") ||
    COMMON::error( " Unable to stop dhcp daemon\n" );

  if ( "$WORKAROUND_RBSCONFIG_SCRIPT" eq "FALSE" ) {
    
    #
    # remotely execute dhcpconfig
    #
    COMMON::print_action("Configuring the DHCP service");
    COMMON::remote_exec ( "$proto_ip", "$RDHCPCONFIG -D -r SUNWfiles -p $dhcp_container_dir -n" ) ||
      COMMON::error( " Unable to execute $RDHCPCONFIG -D -r SUNWfiles -p $dhcp_container_dir -n" );
    
    if ($USE_NLS eq "NO") {
      # 
      # remotely create DHCP containers by invoking pntadm -C
      #
      COMMON::print_action("Creating the DHCP network tables");
      COMMON::remote_exec ( "$proto_ip", "$RPNTADM -C 	$net_if0" ) ||
	COMMON::error( " Unable to execute $RPNTADM -C 	$net_if0" );

      if ($USE_CGTP eq "YES") {
	COMMON::remote_exec ( "$proto_ip", "$RPNTADM -C 	$net_if1" ) ||
	  COMMON::error( " Unable to execute $RPNTADM -C 	$net_if1" );
      }
      
      COMMON::print_action("Adding symbols and macros in the DHCP table");
      $platform = "";
      for $i (0..$#VENDOR_TYPE) {
	if ($platform eq "") {
	  $platform = $VENDOR_TYPE[$i] ;
	} else {
	  $platform .= " " . $VENDOR_TYPE[$i];
	}
      }
      $platform = "\"Vendor=$platform\"" ;
      
      $dhcmd ="$RDHTADM -A -s SbootFIL -d \'$platform,7,ASCII,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SswapPTH -d \'$platform,6,ASCII,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SswapIP4 -d \'$platform,5,IP,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SrootPTH -d \'$platform,4,ASCII,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SrootNM -d \'$platform,3,ASCII,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SrootIP4 -d \'$platform,2,IP,1,1\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s SrootOpt -d \'$platform,1,ASCII,1,0\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      $dhcmd ="$RDHTADM -A -s NhCgtpAddr -d \'Site,128,IP,1,1\'";
      COMMON::remote_exec_bis ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );
      #
      # remotely add	macros to dhcptab container
      #
      $dhcmd ="$RDHTADM -A -m $net_if0 -d \':Broadcst=$broadcast_if0:MTU=1500:Router=$master_ip_if0:\'";
      COMMON::remote_exec ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );

      if ($USE_CGTP eq "YES") {
	
	$dhcmd ="$RDHTADM -A -m $net_if1 -d \':Broadcst=$broadcast_if1:MTU=1500:Router=$master_ip_if1:\'";
	COMMON::remote_exec ( "$proto_ip", $dhcmd) ||
	  COMMON::error( " Cannot remotely execute $dhcmd" );
      }

      $dhcmd ="$RDHTADM -A -m Common -d \':Include=Locale:BootSrvA=$master_ip:SrootIP4=$master_ip:BootSrvN=$master_ip:SrootNM=$master_ip:BootFile=inetboot.$ARCH.$TARGET_OS:Subnet=$CLUSTER_DOTTED_NETMASK:\'";	
      COMMON::remote_exec ( "$proto_ip", $dhcmd) ||
	COMMON::error( " Cannot remotely execute $dhcmd" );	
      
      
      #
      # create a entry for each concerned diskless
      #
      diskless_configure($proto_ip) ;
      
      #
      # specific modification for RBS
      # rename containers SUNWfiles_x_x_x_x in SUNWnhrbs_x_x_x_x
      #
      COMMON::print_action("Renaming the DHCP network tables");

      my $postfix0 = getPostFix($CLUSTER_DOTTED_NIC0_NET);

      COMMON::remote_exec( "$proto_ip", "$RMV $dhcp_container_dir/SUNWfiles1${postfix0} $dhcp_container_dir/SUNWnhrbs1${postfix0}") ||
	COMMON::error( "Unable to rename container dhcp files" );

      if ($USE_CGTP eq "YES") {
	my $postfix1 = getPostFix($CLUSTER_DOTTED_NIC1_NET);

	COMMON::remote_exec( "$proto_ip", "$RMV $dhcp_container_dir/SUNWfiles1${postfix1} $dhcp_container_dir/SUNWnhrbs1${postfix1}") ||
	  COMMON::error( "Unable to rename container dhcp files" );
      }

      COMMON::remote_exec( "$proto_ip", "$RMV $dhcp_container_dir/SUNWfiles1_dhcptab $dhcp_container_dir/SUNWnhrbs1_dhcptab ") ||
	COMMON::error( "Unable to rename container dhcp files" );
      
      
      #
      # apply the workaround for boot server on second master eligible node when elected as the master
      #
    }

    if ( "$WORKAROUND_RBS_VICEMASTER_TFTP" eq "TRUE" ) {
      rbs_vicemaster_workaround ($proto_ip,$men2_ip);
    }
  }
}

#-------------------------------------------------------------------------------
#
# Undo RBS 
#
#-------------------------------------------------------------------------------

sub rbs_cleanup
  {
    my $node_ip   = COMMON::get_external_ip($NODE_ID);
    COMMON::remote_exec_ignore("$node_ip", "$RDHCPCONFIG -U -f -x -h") ;
  }


#-------------------------------------------------------------------------------
#
# installation and configuration of RBS on MEN and NMEN
#
#-------------------------------------------------------------------------------

sub nmen_rbs {

  ($NODE_ID) = @_;

  COMMON::init_and_check([]);
  
  my $rbs_packages = $ENV{RBS_PACKAGES};

  COMMON::print_stage_node("Reliable Boot Server (RBS) installation") ;
  COMMON::install_men_pkg($NODE_ID, $ROOT_DIR, $PACKAGE_SUBDIR, $rbs_packages, "LOCAL", "") ;

  
  if ( $NODE_ID eq $MEN1_ID ) {

    rbs_configure ($NODE_ID);
    update_dhcpsvcconf($NODE_ID);
    
  }
}

#-------------------------------------------------------------------------------
#
# recovery after failure
#
#-------------------------------------------------------------------------------

sub recovery_nmen_rbs {

  ($NODE_ID) = @_;

  COMMON::init_and_check([]);
  
  if ($NODE_ID eq $MEN1_ID) {
    COMMON::print_recovery_node("Reliable Boot Server (RBS) clean-up") ;
    rbs_cleanup() ;
  }
}

#-------------------------------------------------------------------------------
#
# adding a diskless
#
#-------------------------------------------------------------------------------

sub nmen_rbs_add_nodes {

  ($NODE_ID) = @_;
  my $node_ip = COMMON::get_external_ip($NODE_ID, "");
  my $men2_ip= COMMON::get_external_ip($MEN2_ID);

  COMMON::init_and_check([]);

  COMMON::print_stage_node("Reliable Boot Server (RBS): configuration of additional nodes") ;
  diskless_configure($node_ip);
  rbs_vicemaster_workaround ($node_ip,$men2_ip);
}

{
}

