#************************************************************ 
#
# Component      CNF  
#
# Synopsis       Reliable Boot Server
#
#
# Copyright 2001 Sun Microsystems, Inc. All rights reserved.
#
#
#************************************************************
#
# #ident "@(#)RBS.pm 1.7 02/10/15 SMI"
#
#************************************************************

#=head1 Reliable Boot Server
#
#=over 4
#
#=cut

package RBS ;

use LOG ;

$E_dhcp_configuration_file = "/etc/inet/dhcpsvc.conf" ;
$E_dhcp_inittab            = "/etc/dhcp/inittab";
$E_dhcp_symbol             = "NhCgtpAddr";

my $G_status = OK ;


#--------------------------------------------------------------
#
#  rbs_installed
#
#--------------------------------------------------------------

sub rbs_installed
  {
    my $version ;
    my $name ;
    my $basedir ;

    my $return = OS::get_package("SUNWnhrbs", "LOCAL", 
				 \$version, \$name, \$basedir) ;
    if ($return eq OK) {
      return TRUE ;
    }
    if (($return eq NA) || ($return eq NOK)) {
      return FALSE ;
    }
  }
#--------------------------------------------------------------
#
#  get_parameter
#
#  1 : parameter name
#
#  return value of the parameter
#
#--------------------------------------------------------------

sub get_parameter
  {
    ($L_parameter) = @_ ;

    chomp($result = qx{/usr/bin/grep "[ \t]*${L_parameter}[ \t]*=" $E_dhcp_configuration_file}) ;

    LOG::printMsg(3, "dhcpsvc.conf grep: %s", $result) ;
    if ($result eq "")
      {
	return "";
      }

    $result =~ s/#.*// ;
    $result =~ m/[\s\t]*([a-zA-Z0-9_]*)[\s\t]*=[\s\t]*([^\s\t]*)/ ;
    if ($1 ne $L_parameter)
      {
	return "" ;
      }

    LOG::printMsg(3, "%s value = %s", $L_parameter, $2) ;

    return $2 ;
}

#--------------------------------------------------------------
#
#  check_parameter
#
#  1 : parameter name
#  2 : parameter value
#
#  return value of the parameter
#
#--------------------------------------------------------------

sub check_parameter
  {
    ($L_parameter, $L_value) = @_ ;

    $value = get_parameter($L_parameter) ;
    if ($L_value eq "")
      {

#=item <parameter> not defined
#
#    The specified parameter must be defined in /etc/inet/dhcpsvc.conf.
#
#=cut

	LOG::printError("%s not defined", $L_parameter) ;
	$G_status = FAILED ;
	return ;
      }

    if ($L_value ne $value)
      {

#=item <parameter>: <current> instead of <expected>
#
#    Please set the correct value for the specified parameter in
#    /etc/inet/dhcpsvc.conf.
#
#=cut

	LOG::printError("%s: %s instead of %s", 
			$L_parameter,  $value, $L_value) ;
	$G_status = FAILED ;
      }
}

#--------------------------------------------------------------
#
#  check_dhcpsvc
#
#--------------------------------------------------------------

sub check_dhcpsvc {

  $L_status = OK ;
  
  if (OS::check_file($E_dhcp_configuration_file) ne TRUE)
    {
      $L_status = "FAILED";
      LOG::printResult($L_status, "check %s",$E_dhcp_configuration_file) ;
      return ;
    } 
  
  #
  # parameter checking
  # 
  check_parameter("DAEMON_ENABLED", TRUE) ;
  check_parameter("RUN_MODE", "server") ;
  check_parameter("RESOURCE", "SUNWnhrbs") ;
  check_parameter("CONVER", 1) ;
  
  $L_value = get_parameter("PATH") ;
  if ($L_value eq "")
    {
      
#=item PATH not defined
#
#    PATH is not defined in /etc/inet/dhcpsvc.conf
#
#=cut
      
	LOG::printError("PATH not defined") ;
	$G_status = FAILED
      }
    else
      {
	if (! -e $L_value)
	  {

#=item PATH does not reference a valid file (<file>)
#
#    Please check that the file referenced by PATH in /etc/inet/dhcpsvc.conf
#    exists or set PATH with the correct path name.
#
#=cut

	    LOG::printError("PATH does not reference a valid file (%s)", $L_value) ;
	    $G_status = FAILED
	  }
      }

    LOG::printResult($G_status, "check %s", $E_dhcp_configuration_file) ;
}

#--------------------------------------------------------------
#
#  check_inittab
#
#--------------------------------------------------------------

sub check_inittab {

  my $L_status = "OK";

  if (OS::check_file($E_dhcp_inittab) ne TRUE)
    {
      $L_status = "FAILED";
      LOG::printResult($L_status, "check %s",$E_dhcp_inittab) ;
      return ;
    } 

  # check if NhCgtpAddr is defined in /etc/dhcp/inittab
  chomp($result = qx{/usr/bin/grep "[ \t]*${E_dhcp_symbol}[ \t]*" $E_dhcp_inittab}) ;
  
  LOG::printMsg(3, "/etc/dhcp/initab grep: %s***", $result) ;  
  $result =~ s/#.*// ;
  $result =~ m/[\s\t]*([a-zA-Z0-9_]*)[\s\t]*/ ;
  if ($1 ne $E_dhcp_symbol)
    {

#=item symbol <symbol> not defined in /etc/dhcp/inittab
#
#    The specified symbol is required to allow information to
#    be retrieved from the DHCP requests on a diskless node.
#
#=cut

	LOG::printError("symbol %s not defined in %s", $E_dhcp_symbol, $E_dhcp_inittab) ;
	$L_status = "FAILED" ;
      }
  
  LOG::printResult($L_status, "check %s",$E_dhcp_inittab) ;
}

#--------------------------------------------------------------
#
#  check_configuration
#
#--------------------------------------------------------------

sub check_configuration
  {
    if (OS::guess_node_role ne "DISKFULL") {
      return ;
    }
    
    if (rbs_installed eq "FALSE") {
      return ;
    }

    LOG::printMsg(0,"");
    LOG::printMsg(0,"Reliable Boot Server checking");
    LOG::printMsg(0,"");

    check_dhcpsvc;
    check_inittab;

  }

#--------------------------------------------------------------
#
#  display
#
#      display information regarding Reliable Boot Server
#
#--------------------------------------------------------------

sub display
 { 
   return;
   printLine("") ;
   printLine("Reliable Boot Server") ;
   printLine("") ;    
 } 

#--------------------------------------------------------------
#
#  main
#
#--------------------------------------------------------------

#=back
#
#=cut

{
}
