#************************************************************ 
#
# Component       CNF  
#
# Synopsis        Cluster Membership Monitor
#
# Copyright 2001 Sun Microsystems, Inc. All rights reserved.
#
#
#************************************************************
#
# #ident "@(#)CMM.pm 1.9 02/10/15 SMI"
#
#************************************************************

#=head1 Cluster configuration
#
#=over 4
#
#=cut

package CMM ;

use VAR;
use LOG ;


my $G_cmmrole = "/opt/SUNWcgha/sbin/nhcmmrole" ;
my $G_not_configured = "/etc/opt/SUNWcgha/not_configured" ;
my $G_category = "CMM" ;

my $G_display_status = TRUE ;

#
#  valid values
#
my @G_valid_role = (
		    "MASTER" ,
		    "VICEMASTER" ,
		    "NONE"
		    ) ;


my @G_valid_parameter = (
			 "domain_id" ,
			 "attributes" ,
			 "role" ,
			 "election" ,
			) ;

#
#  information about nodes read into node-specific files
#
my %G_node_data ;




#==============================================================
#  used by other modules to get information
#==============================================================

#--------------------------------------------------------------
#
#  get_actual_role
#
#    return the role by using cmmrole
#
#--------------------------------------------------------------

sub get_actual_role
  {
    $result = system("$G_cmmrole -to 5") ;
    $result = $result / 256 ;
    LOG::printMsg(2, "result of cmmrole: %s", $result) ;
    
    if ($result eq 1)
      {
	return MASTER ;
      }
    if ($result eq 2)
      {
	return VICEMASTER ;
      }
    return UNKNOWN ;
  }

#--------------------------------------------------------------
#
#  get_node_id
#
#    return the local id of the node
#
#--------------------------------------------------------------

sub get_local_node_id
  {
    return $E_variables{NODE_ID} ;
  }

#==============================================================
#  functions used for loading various files
#  handled by this module
#==============================================================


#--------------------------------------------------------------
#
#  load_node_specific_file
#
#      load the CMM node specific file
#
#--------------------------------------------------------------

sub load_node_specific_file
  {
    $L_status = OK ;
    
    my $filename = "$E_variables{MINI_CONFIG_DIR}/$E_variables{MINI_CONFIG_FILE}";
    LOG::printMsg(0,"Loading Minimum Configuration File %s ...", $filename) ;

    if (OS::check_file($filename) eq FALSE)
	{
	  exit ;
	}

    open(FH,"<$filename") or die "$filename";
    
    while (<FH>)
      {
	chomp($line =$_) ;
	$initial = $line ;
	# remove comments
	$line =~ /([^\#]*)/;
	$line=$1 ;
	if ($line eq "")
	  {
	    next ;
	  }
	
	# retrieve parameter and its value
	$line =~ /[\s\t]*([^:\s\t]*)[\s\t]*:[\s\t]*([^:\s\t]*)/ ;
	$parameter = $1 ;
	$value = $2 ;

	if (($parameter eq "") || ($value eq ""))
	  {
	    # badly-formatted line : error

#=item ill-formatted line (<line number>)
#
#    The line contained in the file being processing is incorrect.
#    Please check its syntax.
#
#=cut

	    LOG::printError("ill-formatted line (%s)", $.);
	    $error = TRUE ;
	    next ;
	  }
	
	#
	#  check that the parameter name is valid
	#
	if (MAIN::isInArray(\@G_valid_parameter,scalar(@G_valid_parameter),
			    $parameter) ne TRUE)
	  {

#=item node <node name> <parameter>: invalid parameter
#
#    The specified parameter is not known.
#    Please check the syntax of the specified file.
#
#=cut

	    LOG::printError("node %s %s: invalid parameter", $E_cluster_node[$E_cluster_local_node]{node_name}, $parameter) ;
	    $L_status = FAILED ;
	  }
	
	#
	#  check the role value is valid
	#
	if ($parameter eq "role")
	  {
	    if (MAIN::isInArray(\@G_valid_role,scalar(@G_valid_role),
				$value) ne TRUE)
	      {

#=item node <node name>: invalid role (<value>)
#
#    The specified role is not correct.
#
#=cut

		LOG::printError("node %s: invalid role (%s)",
				$E_cluster_node[$E_cluster_local_node]{node_name}, $value) ;
		$L_status = FAILED ;
	      }
	  }
	
	$G_node_data{$parameter} = $value ;
      }
    close(FH) ;
    
    if (($G_display_status eq TRUE) || ($L_status ne OK))
      {
	LOG::printResult($L_status, "node %s: configuration file syntax",
			 $E_cluster_node[$E_cluster_local_node]{node_name});
      }
  }

#--------------------------------------------------------------
#
#  load_cluster_file
#
#--------------------------------------------------------------

sub load_cluster_file
  {
    my $filename = "$E_variables{LOCAL_CONFIG_DIR}/$E_cmm_cluster_file_name";
      
    if (! -r $filename) {
      LOG::printError("Cluster nodes table file %s not accessible", $filename) ;
      exit ;
    }

    LOG::printMsg(0,"Loading cluster nodes table file %s ...", $filename) ;

    open(FH,"<$filename");

    while (<FH>)
      {
	chomp($line =$_) ;
	$initial = $line ;
	# remove comments
	$line =~ /([^\#]*)/;
	$line=$1 ;
	if ($line eq "")
	  {
	    next ;
	  }
	
	$line =~ m/[\s\t]*([0-9]*)[\s\t]*([0-9]*)[\s\t]*([^\s]*)[\s\t]*([^\s]*)/o ;
	LOG::printMsg(2, "node id=%s, domain_id=%s, node name=%s, attributes=%s", $1, $2, $3, $4);
	my $elem = {};
	$elem->{node_id} = $1;
	$elem->{domain_id} = $2;
	$elem->{node_name} = $3;
	$elem->{attributes} = $4;
	push @E_cluster_node, $elem;
	if ($elem->{node_id} == $E_local_node_id)
	  {
	    $E_cluster_local_node = $E_cluster_node_count;
	  }
	$E_cluster_node_count++ ;
      }

    if ($E_cluster_local_node < 0)
      {

#=item local node (id=<node_id>) not defined in the cluster nodes table
#
#    The local node has no entry in the cluster nodes table.
#    Please add an entry for this node.
#
#=cut

	LOG::printError("Local node (id=%s) not defined in the cluster nodes table", 
			$E_local_node_id) ;
	exit ;
      }

    close(FH) ;

    $E_cluster_file_loaded = "TRUE";
  }


#--------------------------------------------------------------
#
#  check_local_node_specific_file
#
#      check the consistency of information of the local node
#      specific file
#
#--------------------------------------------------------------

sub check_local_node_specific_file
  {

    $G_display_success = TRUE ;

    #  check that all cluster data are identical for each node
    #  (be aware that subnet masks are not mandatory
    $L_status = OK ;
    if ($E_variables{DOMAIN_ID} ne $G_node_data{domain_id})
      {
	$L_status = FAILED ;
      }
    LOG::printResult($L_status, "Domain id coherency in node specific file") ;
  }


#--------------------------------------------------------------
#
#  check_cluster_file
#
#--------------------------------------------------------------

sub check_cluster_file
  {
    #
    # check that the domain id in the cluster file
    # is equal to the one in the cmm configuration file
    #
    $L_status = OK ;
    for ($i = 0 ; $i < $E_cluster_node_count ; $i++)
      {
	LOG::printMsg(2,"check domain_id CMM_DOMAIN_ID=%s node domain id=%s",
		      $E_variables{DOMAIN_ID}, $E_cluster_node[$i]{domain_id}) ;
	if ($E_cluster_node[$i]{domain_id} != $E_variables{DOMAIN_ID})
	  {

#=item incorrect domain id for node <node name>
#
#    The domain id for the specified node is not equal to the value
#    indicated in the CMM configuration file (CMM_DOMAIN_ID variable).
#
#=cut

	    LOG::printError("incorrect domain id for node %s",  $G_cluster_node{node_name}[$i]) ;
	    $L_status = FAILED ;
	  }
      }
    LOG::printResult($L_status, "Domain ID coherency in cluster file") ;

    #
    # check that node_id of each node is unique
    # and is not equal to nodes id reserved for master and vice master logical address
    #
    $L_status = OK ;
    for ($i = 0 ; $i < $E_cluster_node_count - 1 ; $i++)
      {

	if ($E_cluster_node[$i]{node_id} == $E_master_node_id)
	  {

#=item Invalid node id for node <node>: <node id> reserved for master logical address
#
#    The specified node id is reserved for the master logical address.
#    It cannot be used as a node id for the specified node. Please change it.
#
#=cut

		LOG::printError("Invalid node id for %s: %s reserved for master logical address",
				$E_cluster_node[$i]{node_name}, $E_cluster_node[$i]{node_id}) ;
		$L_status = FAILED ;
	  }


	if ($E_cluster_node[$i]{node_id} == $E_master_node_id)
	  {
	  }
	for ($j = $i+1 ; $j < $E_cluster_node_count ; $j++)
	  {
	    if ($E_cluster_node[$i]{node_id} == $E_cluster_node[$j]{node_id})
	      {

#=item <node1> and <node2> have the same node_id
#
#    Both nodes have the same node id in the cluster nodes table.
#    Please check that all nodes have a unique node id.
#
#=cut

		LOG::printError("%s and %s have the same node_id", $E_cluster_node[$i]{node_name}, $E_cluster_node[$j]{node_name}) ;
		$L_status = FAILED ;
	      }
	  }
      }
    LOG::printResult($L_status, "Node Id uniqueness in cluster file") ;
  }

#--------------------------------------------------------------
#
#  check_not_configured
#
#  . check that configuration is available
#
#--------------------------------------------------------------

sub check_not_configured
  {
    $L_status = OK ;
    if (-f $G_not_configured)
      {

#=item Netra HA Suite services will not start
#
#    NHAS2 services will not started since the not_configured file exists.
#    Delete this file to enable the starting of services.
#
#=cut

	LOG::printError("Netra HA Suite services will not start") ;
	$L_status = FAILED ;
      }
    LOG::printResult($L_status, "File not_configured removed"); 
  }

#--------------------------------------------------------------
#
#  check_configuration
#
#      check the CMM configuration by:
#      . reading all the CMM node specific file
#        -> verify the consistency
#      . reading the CMM cluster file
#        -> verify the consistency
#
#--------------------------------------------------------------

sub check_configuration
  {

    if ($E_cluster_file_loaded eq "TRUE") {
      LOG::printMsg(0,"") ;
      LOG::printMsg(0,"Cluster file consistency check") ;
      LOG::printMsg(0,"") ;
      check_cluster_file ;
    }
    
    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      LOG::printMsg(0,"") ;
      LOG::printMsg(0,"Node-specific file check") ;
      LOG::printMsg(0,"") ;
      check_local_node_specific_file ;
    }


    
    LOG::printMsg(0,"") ;
    LOG::printMsg(0,"Foundation Services checking") ;
    LOG::printMsg(0,"") ;
    check_not_configured ;

    #
    # check that /etc/services is correctly configured
    #
    LOG::printMsg(0,"") ;
    LOG::printMsg(0,"Services checking") ;
    LOG::printMsg(0,"") ;
    OS::check_service("cmm-api","tcp") ;
    OS::check_service("cmm-config-push","udp") ;
    OS::check_service("cmm-config-pull","tcp") ;
    OS::check_service("cmm-election","udp") ;
    OS::check_service("cmm-cgtp-probe","tcp") ;
    OS::check_service("cmm-cgtp-hb","udp") ;
  }

#--------------------------------------------------------------
#
#  load_all_files
#
#    load all files that are required for checking
#    (expect for installation)
#
#--------------------------------------------------------------

sub load_all_files
  {
    $E_local_node_id = get_local_node_id() ;

    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      # the file is mandatory, load it
      load_cluster_file ;
    } else {
      # load it if it is present
      if (-r "$E_variables{LOCAL_CONFIG_DIR}/$E_cmm_cluster_file_name" ) {
	load_cluster_file ;
      }
    }

    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      load_node_specific_file ;
    }
  }

#--------------------------------------------------------------
#
#  display
#
#--------------------------------------------------------------

sub display
  {
    $G_display_status = FALSE ;

    #
    #  cluster information
    #
    printLine("") ;
    printLine("Cluster information") ;
    printLine("");
    for ($i = 0 ; $i < $E_cluster_node_count ; $i++)
      {
	printLine("    Node name:        %s", $E_cluster_node[$i]{node_name}) ;
	printLine("    Node id:          %s", $E_cluster_node[$i]{node_id}) ;
	printLine("    Domain id:        %s", $E_cluster_node[$i]{domain_id}) ;
	printLine("    Attributes:       %s", $E_cluster_node[$i]{attributes}) ;
	printLine("") ;
      }

    #
    #  node-specific information
    #

    if ($E_variables{NODE_TYPE} eq "DISKFULL")
      {
	printLine("") ;
	printLine("Node-specific information") ;
	printLine("");
	printLine("    Domain Id:        %s", $G_node_data{domain_id}) ;
	printLine("    Role:             %s", $G_node_data{role}) ;
	printLine("    Attributes:       %s", $G_node_data{attributes}) ;
	printLine("    Election:         %s", $G_node_data{election}) ;  
      }
  }

#=back
#
#=cut

{
}

