#************************************************************ 
#
# Component       CNF  
#
# Synopsis        Network and CGTP interfaces
#
#
# Copyright 2001 Sun Microsystems, Inc. All rights reserved.
#
#
#************************************************************
#
# #ident "@(#)CGTP.pm 1.9 02/11/07 SMI"
#
#************************************************************

#=head1 Network and CGTP interfaces
#
#=over 4
#
#=cut

package CGTP ;

use VAR;
use LOG;

#
#
#
my $G_category = "CGTP" ;

#
#  network devices related information
#
my @G_net_device_array ;     # array of network device names
my $G_net_device_count = 0 ; # # of network devices
my %G_net_device_type ;      # !
my %G_net_device_flags ;     # !
my %G_net_device_mtu ;       # !
my %G_net_device_index ;     # !
my %G_net_device_inet ;      # ! hashes indexed by network device name
my %G_net_device_ipaddr ;    # ! 
my %G_net_device_netmask ;   # ! 
my %G_net_device_broadcast ; # !
my %G_net_device_ether ;     # !
my %G_net_device_hostname ;  # !
my $G_physical_net_device_count = 0 ; # # of physical network devices

#
#  netstat related information
#
my @G_netstat ;
my $G_netstat_count ;

#
#  updated when loading information
#
my $G_physical_up_and_running_count = 0 ;
my $G_MAC_list ;                     # list of all different MAC addresses
my $G_MAC_count = 0 ;                # # of different MAC addresses
my $G_MAC_twice = FALSE ;            # true if same address found twice

#
#  configuration data
#

my %G_net_interface_number;
my $G_net_physical_interfaces ;
my $G_net_devices_for_master ;
my $G_net_devices_for_others ;
my $G_net_devices_to_test ;
my $G_hostname_to_test ;

my $G_load_done = FALSE ;
my $G_load_hostnames_done = FALSE ;

my $G_cgtp_interface = "cgtp0" ;

#------------------------------------------------------------------------------
#
#  convertDottedToBitmask
#
#    convert a dotted form to a bit mask
#
#------------------------------------------------------------------------------

sub convertDottedToBitmask {

  my ($dotted) = @_;

  my $mask = 0;
  foreach $value (split('\.', $dotted)) {
    $mask = $mask * 256 + $value;
  }

  return $mask;
}

#------------------------------------------------------------------------------
#
#  convertBitmaskToDotted
#
#    convert a bit mask to a dotted form
#
#------------------------------------------------------------------------------

sub convertBitmaskToDotted {

  my ($bitmask) = @_;

  my $dotted = "";
  for (my $i=0; $i < 4; $i++) {
    if ($i == 0) {
      $dotted = sprintf("%d", $bitmask % 256);
    } else {
      $dotted = sprintf("%d.%s", $bitmask % 256, $dotted)
    }
    $bitmask = $bitmask / 256;
  }

  return $dotted;
}

#--------------------------------------------------------------
#
#  build_IP_address
#
#      1 : interface
#
#--------------------------------------------------------------

sub build_IP_address
  {
    my ($L_node_id, $L_interface) = @_ ;

    # build the expected IP address of The node
    my $netmask = hex($G_net_device_netmask{$L_interface});
    my $subnet  = convertDottedToBitmask($G_net_device_ipaddr{$L_interface});
    my $address = convertBitmaskToDotted(($subnet & $netmask) + $L_node_id);

    return $address ;
  }

#--------------------------------------------------------------
#
#  prepare
#
#      look for some information regarding network devices
#      and set some indicators
#
#--------------------------------------------------------------

sub prepare
  {
    my $L_up ;
    my $L_running ;

    #
    #  count and flag all ethernet devices up and running
    #  check at the same time that the same MAC address is not used twice
    #  get the status of CGTP device
    #

    LOG::printMsg(2, "    -> analysis of data") ;
    foreach $L_device (@G_net_device_array)
      {
	$L_up = FALSE;
	$L_running = FALSE;
	foreach $i (split(',',$G_net_device_flags{$L_device}))
	  {
	    if ($i eq UP )
	      {
		$L_up = TRUE ;
		LOG::printMsg(2, "    -> %s is UP", $L_device) ;
	      }
	    if ($i eq RUNNING )
	      {
		$L_running = TRUE ;
		LOG::printMsg(2, "    -> %s is RUNNING", $L_device) ;
	      }
	  }

	if ( ($L_up eq TRUE) && ($L_running eq TRUE) )
	  {
	    if (($L_device !~ /^cgtp/) && ($G_net_device_type{$L_device} eq PHYSICAL))
	      {
		#
		#  physical devices
		#

		$G_physical_up_and_running_count += 1;
		LOG::printMsg(2, "    -> %s is a physical device UP and RUNNING", $L_device) ;

		#
		# add it in the list of used MAC addresses
		# (if we are root)
		if ($E_isRoot eq TRUE)
		  {
#		    LOG::printMsg(2, "    -> checking if %s already used", $G_net_device_ether{$L_device}) ;
		    for ($i = 0, $found = FALSE ;
			 ($i < $G_MAC_count) && ($found ne TRUE) ;
			 $i++)
		      {
			LOG::printMsg(2, "        compare with %s", $G_MAC_list[$i]) ;
			if ($G_MAC_list[$i] eq $G_net_device_ether{$L_device})
			  {
			    LOG::printMsg(2, "        -> found twice") ;
			    $found = TRUE ;
			    $G_MAC_twice = TRUE ;
			  }
		      }
		    if ($found eq FALSE)
		      {
			$G_MAC_list[$G_MAC_count] =
			  $G_net_device_ether{$L_device} ;
			$G_MAC_count += 1 ;
		      }
		  }
	      }
	  }
      }
  }

#--------------------------------------------------------------
#
#  build_interface_lists
#
#
#--------------------------------------------------------------

sub build_interface_lists
  {
    
    LOG::printMsg(1, "USE_CGTP: %s", $E_variables{USE_CGTP}) ;

    if ($E_variables{USE_CGTP} ne "FALSE") {

      $G_net_physical_interfaces = "$E_variables{NIC0} $E_variables{NIC1}" ;
      $G_net_devices_to_test = "$G_cgtp_interface $E_variables{NIC0} $E_variables{NIC1}" ;
      $G_net_devices_for_master = "$G_cgtp_interface $G_cgtp_interface:1 $E_variables{NIC0} $E_variables{NIC0}:1 $E_variables{NIC1} $E_variables{NIC1}:1" ;
      $G_net_devices_for_others = "$G_cgtp_interface $E_variables{NIC0} $E_variables{NIC1}" ;
      %G_net_interface_number = ( "$G_cgtp_interface" => 3,
				  "$E_variables{NIC0}" => 1,
				  "$E_variables{NIC1}" => 2
				) ;

      if ($E_variables{NODE_TYPE} eq "DISKLESS") {
	$G_hostname_to_test = "$E_variables{NIC0} $E_variables{NIC1}";
      } else {
	$G_hostname_to_test = "$G_cgtp_interface $E_variables{NIC0} $E_variables{NIC1}";
      }
    } else {

      $G_net_physical_interfaces = "$E_variables{NIC0}" ;
      $G_net_devices_to_test = "$E_variables{NIC0}" ;
      $G_net_devices_for_master = "$E_variables{NIC0} $E_variables{NIC0}:1" ;
      $G_net_devices_for_others = "$E_variables{NIC0}" ;
      %G_net_interface_number = ( "$E_variables{NIC0}" => 1) ;
      $G_hostname_to_test = "$E_variables{NIC0}";

    }


    LOG::printMsg(1, "physical device to test:   %s", $G_net_devices_to_test) ;
    LOG::printMsg(1, "interface for master:      %s", $G_net_devices_for_master) ;
    LOG::printMsg(1, "interface for other nodes: %s", $G_net_devices_for_others) ;
    LOG::printMsg(1, "hostname to test:          %s", $G_hostname_to_test) ;
  }

#--------------------------------------------------------------
#
#  load
#
#      load information regarding network devices and cgtp
#
#--------------------------------------------------------------

sub load
  {
    if ($G_load_done eq TRUE)
      {
	return ;
      }

    $G_load_done = TRUE ;

    #
    # get the complete list of network device
    #

    LOG::printMsg(2, "    execution '/sbin/ifconfig -a") ;
    @L_result = qx'/sbin/ifconfig -a';

    foreach $L_line (@L_result)
      {
	chomp($L_line) ;
	LOG::printMsg(2,"    analyzing %s", $L_line) ;
	#
	# check if the line contains an ethernet device reference
	#
	if (($L_line =~ m/^([a-z0-9]*): /o) ||
	    ($L_line =~ m/^([a-z0-9]*:[0-9]+): /o))
	  {

	    #
	    # keep only the ethernet device reference
	    # remove alias and loopback
	    #
	    $L_device = $1;
	    LOG::printMsg(2, "    --> device: %s", $L_device) ;
	    if ($L_device ne "lo0")
		{
		  if (MAIN::isInArray(\@G_net_device_array,
				$G_net_device_count,$L_device) eq FALSE)
		    {
		      $G_net_device_array[$G_net_device_count] = $L_device ;
		      if ($L_line =~ m/^([a-z0-9]*): /o)
			{
			  $G_net_device_type{$L_device} = PHYSICAL ;
			  $G_physical_net_device_count++ ;
			  LOG::printMsg(2, "    --> %s is a physical device", $L_device) ;
			}
		      else
			{
			  $G_net_device_type{$L_device} = ALIAS ;
			  LOG::printMsg(2, "    --> %s is an alias", $L_device) ;
			}
		      $G_net_device_count++;
		    }
		}
	  }
      }

    #
    #  for each network interface, get the information
    # that can be retrieved by the 'ifconfig -a' command
    #
    foreach $L_device (@G_net_device_array)
      {
	@L_result = qx{/sbin/ifconfig $L_device};
	foreach $L_line (@L_result)
	  {
	    if ($L_line =~ /flags=[0-9]*<([^>]*)>/o)
	      {
	        $G_net_device_flags{$L_device} = $1 ;
	      }
	    if ($L_line =~ /mtu ([0-9]*)/o)
	      {
		$G_net_device_mtu{$L_device} = $1;
	      }
	    if ($L_line =~ /index ([0-9]*)/o)
	      {
		$G_net_device_index{$L_device} = $1;
	      }
	    if ($L_line =~ /inet ([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)/o)
	      {
		$G_net_device_ipaddr{$L_device} = $1;
	      }
	    if ($L_line =~ /netmask ([0-9a-f]*)/o)
	      {
		$G_net_device_netmask{$L_device} = $1;
	      }
	    if ($L_line =~ /broadcast ([0-9]*\.[0-9]*\.[0-9]*\.[0-9]*)/o)
	      {
		$G_net_device_broadcast{$L_device} = $1;
	      }
	    if ($L_line =~ /ether ([0-9aa-f:]*)/o)
	      {
		$G_net_device_ether{$L_device} = $1;
	      }
	  }
      }

    prepare ;
  }

#--------------------------------------------------------------
#
#  load_hostnames
#
#      load hostname by reading content of /etc/hostname.<nic>
#
#--------------------------------------------------------------

sub load_hostnames
  {
    if ($G_load_hostnames_done eq TRUE)
      {
	return ;
      }
    $G_load_hostnames_done = TRUE ;

    foreach $L_device (@G_net_device_array)
      {
	LOG::printMsg(1,"    --> access to /etc/hostname.%s", $L_device) ;
	if (-e "/etc/hostname.$L_device")
	  {
	    open(FH, "/etc/hostname.$L_device") ;
	    chomp($line = <FH>) ;
	    close(FH) ;
	    $G_net_device_hostname{$L_device} = $line ;
	    LOG::printMsg(1,"    --> /etc/hostname.%s contains %s", $L_device, $G_net_device_hostname{$L_device}) ;
	  } else {
	    LOG::printMsg(1,"    --> no file /etc/hostname.%s", $L_device) ;
	  }
      }
  }

#--------------------------------------------------------------
#
#  exist_device
#
#      return TRUE if the device specified as an argument
#      exists in the device list
#
#--------------------------------------------------------------

sub exist_device
  {
    my ($L_device) = @_ ;

    foreach $device (@G_net_device_array)
      {
	if ($device eq $L_device)
	  {
	    return TRUE ;
	  }
      }
    return FALSE ;
  }

#--------------------------------------------------------------
#
#  ready
#
#      return TRUE if the device specified as an argument
#      is up and running
#      assumption: existence has been previously tested
#
#--------------------------------------------------------------

sub ready
  {
    my ($L_device) = @_ ;

    $L_up = FALSE;
    $L_running = FALSE;
    foreach $i (split(',',$G_net_device_flags{$L_device}))
      {
	if ($i eq UP )
	  {
	    $L_up = TRUE ;
	  }
	if ($i eq RUNNING )
	  {
	    $L_running = TRUE ;
	  }
      }

    if ( ($L_up eq TRUE) && ($L_running eq TRUE) )
      {
	return TRUE ;
      }
    return FALSE ;
  }

#--------------------------------------------------------------
#
#  load_netstat_info
#
#      load netstat information in order to allow further analysis
#
#--------------------------------------------------------------

sub load_netstat_info
  {
    @L_result = qx{netstat -rn} ;
    $G_netstat_count = 0 ;

    foreach $line (@L_result)
      {
	chomp($line) ;
	if ($line =~ m/([0-9\.]+)[\s\t]+([0-9\.]+)[\s\t]+([A-Z]*)[\s\t]+([0-9]+)[\s\t]+([0-9]+)[\s\t]+([0-9a-z:]*)/)
	  {
	    LOG::printMsg(2,"destination=%s gateway=%s flags=%s ref=%s use=%s interface=%s", $1, $2, $3, $4, $5, $6) ;
	    $G_netstat{destination}[$G_netstat_count] = $1 ;
	    $G_netstat{gateway}[$G_netstat_count]     = $2 ;
	    $G_netstat{flags}[$G_netstat_count]       = $3 ;
	    $G_netstat{ref}[$G_netstat_count]         = $4 ;
	    $G_netstat{use}[$G_netstat_count]         = $5 ;
	    $G_netstat{interface}[$G_netstat_count]   = $6 ;
	    $G_netstat_count++ ;
	  }
      }
  }

#--------------------------------------------------------------
#
# search_route
#
#--------------------------------------------------------------

sub search_route
  {
    ($destination, $gateway) = @_ ;

    for (my $i = 0 ; $i < $G_netstat_count ; $i++)
      {
	if (($G_netstat{destination}[$i] eq $destination) && ($G_netstat{gateway}[$i] eq $gateway))
	  {
	    return TRUE ;
	  }
      }
    return FALSE ;
  }


#--------------------------------------------------------------
#
#  check_route
#
#--------------------------------------------------------------

sub check_route
  {

    my $L_status = OK ;
    my $interface ;
    my $result ;
    my $gateway ;
    my $cgtpip ;
    my $interfaceip ;
    my $i ;
    my $node_id ;
    my $node_name ;

    if (exist_device($G_cgtp_interface) eq "FALSE") {

#=item CGTP interface is not present
#
#    Foundations Services are probably not being started.
#
#=cut
			
      LOG::printError("CGTP interface is not present") ;
      $L_status = FAILED_SKIP ;
      LOG::printResult($L_status, "CGTP route checking") ;
      return ;
    }


    # check that there is a cgtp address of each node is routed to
    # each address of physical interface of this node

    for ($i = 0 ; $i < $E_cluster_node_count ; $i++)
      {
	if ($i != $E_cluster_local_node)
	  {
	    foreach $interface (split(" ",$G_net_physical_interfaces))
	      {
		$cgtpip = build_IP_address($E_cluster_node[$i]{node_id}, $G_cgtp_interface) ;
		$interfaceip = build_IP_address($E_cluster_node[$i]{node_id}, $interface) ;
		$result = search_route($cgtpip, $interfaceip) ;
		if ($result ne TRUE)
		  {

#=item Destination <CGTP ip>, gateway <interface IP>: no route defined
#
#    To be routed, CGTP protocol need a route to be defined for each interface.
#
#=cut
			
		    LOG::printError("Destination %s, gateway %s: no route defined", $cgtpip, $interfaceip) ;
		    $L_status = FAILED ;
		  }
		LOG::printResult($L_status, "CGTP route for %s through %s", $E_cluster_node[$i]{node_name}, $interface) ;
	      }
	  }
      }

    # check that there is a route also for the master and vice-master address
    foreach $node_id ($E_master_node_id)
      {
	foreach $interface (split(" ",$G_net_physical_interfaces))
	  {
	    $cgtpip = build_IP_address($node_id, $G_cgtp_interface) ;
	    $interfaceip = build_IP_address($node_id, $interface) ;
	    $result = search_route($cgtpip, $interfaceip) ;
	    if ($result ne TRUE)
	      {
		LOG::printError("Destination %s, gateway %s: no route defined", $cgtpip, $interfaceip) ;
		$L_status = FAILED ;
	      }
	    $nodename = "master" ;
	    LOG::printResult($L_status, "CGTP route for %s through %s", $nodename, $interface) ;
	  }
      }
  }

#--------------------------------------------------------------
#
#  check_configuration
#
#      check network information to eliminate potential problem
#
#--------------------------------------------------------------

sub check_configuration
  {
    my $L_status = OK ;

    LOG::printMsg(0, "") ;
    if ($E_variables{USE_CGTP} eq "TRUE") {
      LOG::printMsg(0, "CGTP and network configuration") ;
    } else {
      LOG::printMsg(0, "Network configuration") ;
    }
    LOG::printMsg(0, "") ;

    load ;
    load_hostnames ;
    build_interface_lists ;

    #
    # test the existence of /etc/hostname.<NICx>
    #
    $L_status = OK ;
    foreach $L_interface (split(" ",$G_hostname_to_test))
      {
	LOG::printMsg(1,"    test existence of /etc/hostname.%s", $L_interface);
	if (! -e "/etc/hostname.$L_interface")
	  {

#=item /etc/hostname.<interface> does not exists
#
#    This file is required and should contain the name of the node
#    for the specified interface.
#
#=cut
	    LOG::printError("/etc/hostname.%s does not exist", $L_interface);
	    $L_status = FAILED ;
	  }
      }
    LOG::printResult($L_status, "/etc/hostname.* must exist for each interface") ;

    #
    #
    # hostname and IP address of each interface must be defined in /etc/hosts
    #
    #  (host names have been update by "load" function)
    #
    $L_status = OK ;
    foreach $L_interface (split(" ",$G_hostname_to_test))
      {
        if (exist_device($L_interface) eq FALSE)
          {
            next ;
          }
	$hostname = $G_net_device_hostname{$L_interface} ;


	if ($hostname eq "")
	  {

#=item host name not defined for <interface>
#
#    Either /etc/hostname does not exist or does not contain a name.
#
#=cut

	    LOG::printError("host name not defined for %s", $L_interface);
	    $L_status = FAILED ;
	    next ;
	  }
	$ip = build_IP_address($E_local_node_id, $L_interface) ;
	$found = FALSE ;
	open(FH, "</etc/hosts") ;
	while (<FH>)
	  {
	    chomp($line = $_) ;
	    $line =~ s/#.*// ;
	    #
	    # <IP address> <hostname> ...
	    #
	    if ($line =~ m/[\s\t]*([0-9.]*)[\s\t]*(.*)/)
              {
		if ($1 eq $ip)
		  {
		    LOG::printMsg(3, "definition IP: %s hostname: %s being processed", $1, $2) ;
		    if ($found eq TRUE)
		      {

#=item <IP address> defined twice in /etc/hosts
#
#    Please check entries to be sure that only one hostname is defined
#    for the specified IP address.
#
#=cut
			
			LOG::printError("%s defined twice in /etc/hosts", $ip) ;
			$L_status = FAILED ;
		      }
		    else
		      {
			$found = TRUE ;

			# test each host name for the given IP address
			my $one_found = FALSE ;
			foreach $current_name (split(" ",$2)) {
			  if ($current_name eq $hostname) {
			    $one_found = TRUE ;
			    last ;
			  }
			}
			if ($one_found ne TRUE)
			  {

#=item invalid host name for <interface> (<IP address>): (<current> instead of <expected>)
#
#    The host name specified for <interface> does not match the name found
#    in /etc/hostname.<interface> for a physical interface or in the
#    cluster nodes table for the cgtp interface.
#
#=cut

			    LOG::printError("invalid host name for %s (%s): (%s instead of %s)", $L_interface, $ip, $2, $hostname) ;
			    $L_status = FAILED ;
			  }
		      }
		  }
	      }
	  }
	if ($found eq FALSE)
	  {

#=item <interface> (<IP address>) has no associated host name in /etc/hosts
#
#    The <IP address> corresponding to the <interface> is not defined
#    in /etc/hosts.
#
#=cut
    
	    LOG::printError("%s (%s) has no associated host name in /etc/hosts", $L_interface, $ip) ;
	    $L_status = FAILED ;
	  }
	close(FH) ;
      }
    LOG::printResult($L_status, "node name for each interface defined in /etc/hosts") ;
  }

#--------------------------------------------------------------
#
#  check_starting
#
#      check when CGTP has been started
#
#--------------------------------------------------------------

sub check_starting
  {
    my $L_status ;
    my $L_list ;
    my $i ;
    my $expected ;

    LOG::printMsg(0, "") ;
    if ($E_variables{USE_CGTP} eq "TRUE") {
      LOG::printMsg(0, "CGTP availability") ;
    } else {
      LOG::printMsg(0, "Network availability") ;
    }
    LOG::printMsg(0, "") ;

    load ;
    load_hostnames ;
    build_interface_lists ;

    if (CMM::get_actual_role eq MASTER)
      {
	$L_list = $G_net_devices_for_master ;
      }
    else
      {
	$L_list = $G_net_devices_for_others ;
      }

    LOG::printMsg(2, "    list of devices to check: %s", $L_list) ;

    foreach $L_device (split(" ",$L_list))
      {
	$L_status = OK ;
	if (exist_device($L_device) ne TRUE)
	  {
	    $L_status = FAILED ;
	  }
	LOG::printResult($L_status, "%s device exists", $L_device) ;
    
	if ($L_status eq OK)
	  {
	    $L_status = FAILED ;
	    if (ready("$L_device") eq TRUE)
	      {
		$L_status = OK ;
	      }
	    LOG::printResult($L_status,"%s device up and running", $L_device ) ;
	  }
      }

    $L_status = OK ;
    if ($E_isRoot eq TRUE)
      {
	if ($G_MAC_twice eq TRUE)
	  {
	    $L_status = FAILED ;
	  }
      }
    else
      {
	$L_status = "must be root";
      }
    LOG::printResult($L_status, "not the same MAC address used twice") ;

    #
    #  check the IP address
    #
    #  IP address must look like subnet0.{domain_id}.{interface}.{node_id}
    #
    $L_status = OK ;
    foreach $L_interface (split(" ",$G_net_devices_to_test))
      { 
	if (exist_device($L_interface) eq TRUE)
	  {
#	    $expected = build_IP_address($E_cluster_node[$E_cluster_local_node]{node_id},
#$L_interface) ;
	    $expected = build_IP_address($E_local_node_id, $L_interface) ;

	    LOG::printMsg(2, "    test IP address of %s: expected:%s actual=%s", 
			  $L_interface, $expected, $G_net_device_ipaddr{$L_interface}) ;
	    if ($expected ne $G_net_device_ipaddr{$L_interface})
	      {

#=item <interface>: invalid IP address (<current> instead of <expected>)
#
#    The IP address configured for the specified interface is not correct.
#
#=cut

		$L_status = FAILED ;
		LOG::printError("%s: invalid IP address (%s instead of %s)",
				$L_interface, $G_net_device_ipaddr{$L_interface}, $expected) ;
	      }
	  }
      }
    LOG::printResult($L_status, "Node Id and IP address coherency") ;


    if ($E_cluster_file_loaded eq "TRUE") {
      #
      # check accessibility
      #
      LOG::printMsg(0,"Following tests perform network access through available interfaces");
      LOG::printMsg(0,"Can take a long time ...") ;
      
      #
      #  perform network access through cgtp, eth0 and eth1
      #
      
      $L_list = $G_net_devices_to_test ;
      for ($i = 0 ; $i < $E_cluster_node_count ; $i++) {
	if ($i ne $E_cluster_local_node) {
	  LOG::printMsg(1,"check accessibility of node %s", $E_cluster_node[$i]{node_name}) ;
	  foreach $L_device (split(" ", $G_net_devices_to_test)) {
	    if (exist_device($L_device) eq FALSE) {
	      next ;
	    }
	    if (ready($L_device) eq FALSE) {
	      LOG::printMsg(2,"    %s not ready: not tested", $L_device);
	      next ;
	    }
	    LOG::printMsg(2,"    %s ready: to be tested", $L_device);
	    $L_status = OK ;
	    #
	    #  build the address
	    #
	    $address=build_IP_address($E_cluster_node[$i]{node_id}, $L_device) ;
	    
	    LOG::printMsg(1,"    ping node %s (%s)", $E_cluster_node[$i]{node_name}, $address) ;
	    $command = "/usr/sbin/ping -i $L_device $address 3" ;
	    LOG::printMsg(2, "  execute: %s", $command) ;
	    $L_result = qx{$command} ;
	    LOG::printMsg(2, "      %s", $L_result) ;
	    if ($L_result =~ m/no answer/)
	      {
		$L_status = FAILED ;
	      }
	    LOG::printResult($L_status, "%s reachable through %s", 
			     $E_cluster_node[$i]{node_name}, $L_device) ;
	  }
	}
      }
    }

    if ($E_variables{USE_CGTP} ne "FALSE") {
      LOG::printMsg(0,"") ;
      LOG::printMsg(0,"Routing") ;
      LOG::printMsg(0,"") ;

      # check route
      if ($E_variables{NODE_TYPE} eq "DISKFULL") {
	$L_status = OK ;
	if (! -f "/etc/notrouter") 
	  {
	    $L_status = FAILED ;
	  }
	LOG::printResult($L_status, "Routing mechanism disabled (/etc/notrouter exists)") ;
      }

      load_netstat_info ;
      check_route ;
    }

  }
#--------------------------------------------------------------
#
#  display
#
#--------------------------------------------------------------

sub display
  {
    printLine("") ;
    printLine("Network interfaces") ;
    printLine("") ;

    load ;
    load_hostnames ;
    build_interface_lists ;

    if ($E_variables{NODE_TYPE} eq "DISKFULL") {
      $L_list = $G_net_devices_for_master ;
    } else {
      $L_list = $G_net_devices_for_others ;
    }

    foreach $L_interface (split(" ",$L_list))
      {
	printLine("    Interface:             %s", $L_interface) ;
	if (exist_device($L_interface) eq FALSE)
	    {
	      printLine("    -> not available") ;
	      printLine("");
	      next ;
	    }
	printLine("    IP address:            %s", $G_net_device_ipaddr{$L_interface}) ;
	printLine("    Broadcast address:     %s", $G_net_device_broadcast{$L_interface}) ;
	printLine("    Flags:                 %s", $G_net_device_flags{$L_interface}) ;
	printLine("    Netmask:               %s", $G_net_device_netmask{$L_interface}) ;

	if ($G_net_device_type{$L_interface} ne PHYSICAL)
	  {
	    printLine("") ;
	    next ;
	  }
	printLine("    MAC address:           %s", $G_net_device_ether{$L_interface}) ;
	printLine("    Associated host name:  %s", $G_net_device_hostname{$L_interface}) ;
	printLine("") ;
      }
  }
#--------------------------------------------------------------
#
#  main
#
#--------------------------------------------------------------

#=back
#
#=cut

{
}
