#!/bin/sh
#
#  info.sh
#
#  info.sh is a script that gathers information about the installed system
#  and places it in a file for SAM-FS customer support.
#
#  NOTE: this script must be run as root.
#
#
# $Id: info.sh,v 1.19.2.12 2002/10/17 14:58:06 tja Exp $
#

#  CONFIGURATION VARIABLES

#  Where the report goes
REPORT=${1:-/tmp/SAMreport}

#  How many lines from log files, etc., to show in the output report.
#  Default is 1,000, but your support provider may ask you to change
#  this to provide additional historical log messages.
HOW_MANY_LINES=${2:-1000}

#  Locale for messages
LOCALE=C

#  Options for dump_cat
DUMPCATOPTS="-V"

#  END OF CONFIGURATION VARIABLES - NO CHANGES BELOW HERE, PLEASE
 
#  Directories that SAM-FS uses

ETCDIR=/etc/opt/SUNWsamfs
VARDIR=/var/opt/SUNWsamfs
USERDIR=/opt/SUNWsamfs/bin
EXECDIR=/opt/SUNWsamfs/sbin
SHFSDIR=/var/opt/SUNWsamfs/sharefsd
CATDIR=/var/opt/SUNWsamfs/catalog
FSDDIR=/var/opt/SUNWsamfs/fsd
FTPDIR=/var/opt/SUNWsamfs/ftp

#  Default trace directories
TRCDIR=$VARDIR/trace
TRCTMP=/tmp/$$.trcctls

#  A tag emitted on lines indicating serious problems
TAG=":-( "

#  The prompt we fake on echoed command lines
PROMPT="`uname -n`# "

#  Function definitions

#  How many "."s on the current line?  999 means "too many".
COUNT=999

#  Note:  if you find the lines of dots aren't the same length, or you
#         get extra "."s in your output file, then check to make sure you
#         haven't done something like "verbose ls >>$REPORT", because then
#         twiddle's dot will go to the report file instead of the screen!
#         Also avoid "verbose /opt/SUNWsamfs/sbin/samcmd u 2>&1 | head -100",
#         because then twiddle's dot will go to the screen but won't get
#         counted.
twiddle()
{
    COUNT=`expr $COUNT + 1`
	if [ $COUNT -ge 45 ]; then
		COUNT=1
        echo
		/bin/echo Please wait.\\c
	fi
    /bin/echo .\\c
}

#  Warn the user that the info.sh script really should be run as root.
rerun_as_root()
{
	echo " "
	echo " "
	echo Cannot read ${1} ...  Please rerun this script as root.
	echo Cannot read $1 >>$REPORT 2>&1
	echo " "
	/bin/echo Please wait.\\c
}

#  define a function to execute a command, echoing it as it does so, and
#  indent everything.

verbosei(){
    twiddle
    echo ""		>>$REPORT 2>&1
    echo "    " $PROMPT $*		>>$REPORT 2>&1
    (eval $*		| sed -e "s/^/     /" )>>$REPORT 2>&1
    echo ""		>>$REPORT 2>&1
}

#  show a file, with title and verbose echoing of the cat command, with
#  some indentation.

showifiles(){
for file in $*; do
	echo "     ------------------- $file -------------------" >>$REPORT 2>&1
	if [ -d $file -o -c $file -o -b $file -o -p $file ]; then
		verbosei /bin/ls -l $file
		echo $TAG  "     $file is a directory, char special, block special or pipe.">>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	elif [ -r $file ]; then
		verbosei /bin/ls -l $file
		verbosei /usr/bin/cat $file
	elif [ -f $file ]; then
		verbosei /bin/ls -l $file
		echo "    " $file is not readable >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
    else
		echo "    " $file does not exist >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	fi
done
}

#  Non-indenting versions of the above functions

verbose(){
    twiddle
    echo ""		>>$REPORT 2>&1
    echo $PROMPT $*		>>$REPORT 2>&1
    eval $*		>>$REPORT 2>&1
    echo ""		>>$REPORT 2>&1
}

showfiles(){
for file in $*; do
	echo "------------------- $file -------------------" >>$REPORT 2>&1
	if [ -d $file -o -c $file -o -b $file -o -p $file ]; then
		verbose /bin/ls -l $file
		echo $TAG "$file is a directory, char special, block special or pipe.">>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	elif [ -r $file ]; then
		verbose /bin/ls -l $file
		verbose /usr/bin/cat $file
	elif [ -f $file ]; then
		verbose /bin/ls -l $file
		echo $file is not readable >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
    else
		echo $file does not exist >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	fi
done
}

# show at least $HOW_MANY_LINES lines of a group of files

showenuf(){
ENUF=0
REM=$HOW_MANY_LINES
for file in $*; do
	echo "------------------- $file -------------------" >>$REPORT 2>&1
	if [ -d $file -o -c $file -o -b $file -o -p $file ]; then
		verbose /bin/ls -l $file
		echo $TAG "$file is a directory, char special, block special or pipe.">>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	elif [ x`echo $file | /bin/egrep '.gz$|.Z$'` = x$file ]; then
		verbose /bin/ls -l $file
		echo $file is compressed >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	elif [ -r $file ]; then
		verbose /bin/ls -l $file
		ENUF=`/bin/tail -$REM $file | /usr/ucb/wc | cut -c1-8`
		verbose /bin/tail -$REM $file
		if [ $ENUF -ge $REM ]; then
			return
		else
			REM=`expr $REM - $ENUF`
		fi
	elif [ -f $file ]; then
		verbose /bin/ls -l $file
		echo $file is not readable >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
    else
		echo $file does not exist >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	fi
done
}

# show just the last $HOW_MANY_LINES lines of a file

showtails(){
for file in $*; do
	echo "------------------- $file -------------------" >>$REPORT 2>&1
	if [ -d $file -o -c $file -o -b $file -o -p $file ]; then
		verbose /bin/ls -l $file
		echo $TAG "$file is a directory, char special, block special or pipe.">>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	elif [ -r $file ]; then
		verbose /bin/ls -l $file
		verbose /bin/tail -$HOW_MANY_LINES $file
	elif [ -f $file ]; then
		verbose /bin/ls -l $file
		echo $file is not readable >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
    else
		echo $file does not exist >>$REPORT 2>&1
		echo ""		>>$REPORT 2>&1
	fi
done
}

verbosehead(){
    twiddle
    echo ""		>>$REPORT 2>&1
    echo $PROMPT $*		>>$REPORT 2>&1
    eval $* 2>&1 | head -100		>>$REPORT 2>&1
    echo ""		>>$REPORT 2>&1
}

settrace(){
  eval ${1}TRC=`/bin/grep $2 < $TRCTMP | sed -e "s/$2 *//" | sed -e "s/^off$//"`
}

#  End of function definitions

#
#  now collect up everything...
#
echo Report is being produced into ${REPORT} ...

rm -f $REPORT
echo "SAM-FS Software Support" >$REPORT
echo " " >>$REPORT
echo "Run at `/bin/date` on `/bin/uname -n` by `/bin/id`" >>$REPORT
echo " " >>$REPORT

#  Find the log file locations in the various configuration files
if [ -f $ETCDIR/defaults.conf ]; then
	if [ -r $ETCDIR/defaults.conf ]; then
		LOGIS=`grep LOCAL $ETCDIR/defaults.conf | grep -v "#" | sed -e "s/^.*LOG_//"`
		if [ x$LOGIS = x ]; then
			LOGIS=local7
		fi
	else
		rerun_as_root $ETCDIR/defaults.conf
		LOGIS=local7
	fi
else
	LOGIS=local7
fi

# Get the samfs log file.
# Skip non-sam entries with patterns like "local1,...,local7.none".

if [ -f /etc/syslog.conf ]; then
	if [ -r /etc/syslog.conf ]; then
		# We decided to use the first possible entry, versus the last.
		SAMLL=`grep -i $LOGIS /etc/syslog.conf | grep -v "#" | grep -v $LOGIS'[^;]*.none' | head -1`
		# Verify that there's a tab character in the line.
		# Watch out for invisible tab character in following line!
		SAMLM=`echo "$SAMLL" | grep "	"`
		if [ "X" = "X$SAMLM" ]; then
			echo $TAG /etc/syslog.conf entry for SAM-FS missing tab >> $REPORT 2>&1
			echo " " >> $REPORT 2>&1
			# Sometimes there will be a second entry defined properly,
			# so try the first entry with spaces versus tabs.
			SAMLOG=`echo "$SAMLL" | sed -e "s/^.* //" -e "1q"`
		else
			# Watch out for invisible tab character in following line!
			SAMLOG=`echo "$SAMLL" | sed -e "s/^.*	//" -e "1q"`
		fi
	else
		rerun_as_root /etc/syslog.conf
		SAMLOG=""
	fi
else
	SAMLOG=""
fi

#	beware - archiver.cmd et al. allow "#" anywhere to start a comment.
#   strip anything starting with "#" to the end of the line.

if [ -f $ETCDIR/archiver.cmd ]; then
	if [ -r $ETCDIR/archiver.cmd ]; then
		ARCHLOGS=`sed -e "/#/s/#.*//" < $ETCDIR/archiver.cmd|grep "logfile.*="| sed -e "/logfile/,/logfile/s/^.*= *//"`
	else
		rerun_as_root $ETCDIR/archiver.cmd
		ARCHLOGS=""
	fi
else
	ARCHLOGS=""
fi

if [ -f $ETCDIR/recycler.cmd ]; then
	if [ -r $ETCDIR/recycler.cmd ]; then
		RECLLOGS=`sed -e "/#/s/#.*//" < $ETCDIR/recycler.cmd|grep "logfile.*=" | sed -e "/logfile/,/logfile/s/^.*= *//"`
	else
		rerun_as_root $ETCDIR/recycler.cmd
		RECLLOGS=""
	fi
else
	RECLLOGS=""
fi

if [ -f $ETCDIR/releaser.cmd ]; then
	if [ -r $ETCDIR/releaser.cmd ]; then
		RELLOG=`sed -e "/#/s/#.*//" < $ETCDIR/releaser.cmd|grep "logfile.*="| sed -e "/logfile/,/logfile/s/^.*= *//"`
	else
		rerun_as_root $ETCDIR/releaser.cmd
		RELLOG=$VARDIR/releaser.log
	fi
else
	RELLOG=$VARDIR/releaser.log
fi

if [ -f $ETCDIR/samlogd.cmd ]; then
	if [ -r $ETCDIR/samlogd.cmd ]; then
		STAGELOG=`sed -e "/#/s/#.*//" < $ETCDIR/samlogd.cmd|grep "stage.*="| sed -e "/stage/,/stage/s/stage *= *//" -e "s/ .*$//"`
	else
		rerun_as_root $ETCDIR/samlogd.cmd
		STAGELOG=""
	fi
else
	STAGELOG=""
fi

if [ -f $ETCDIR/stager.cmd ]; then
	if [ -r $ETCDIR/stager.cmd ]; then
		STAGER_LOG=`sed -e "/#/s/#.*//" < $ETCDIR/stager.cmd|grep "logfile.*="| sed -e "/logfile/,/logfile/s/logfile *= *//" -e "s/ .*$//"`
	else
		rerun_as_root $ETCDIR/stager.cmd
		STAGER_LOG=""
	fi
else
	STAGER_LOG=""
fi

#  These log files' locations are fixed
DEVLOGS=/var/opt/SUNWsamfs/devlog/*

#  These data directory locations are fixed
ARCHDATA="$VARDIR/archiver"
STAGER_DATA="$VARDIR/stager"

#  Find the trace files.

/opt/SUNWsamfs/sbin/samcmd d 2>/dev/null > $TRCTMP

settrace ARCH sam-archiverd
settrace CAT  sam-catserverd
settrace FSD  sam-fsd
settrace FTP  sam-ftpd
settrace RCY  sam-recycler
settrace SHFS sam-sharefsd
settrace STG  sam-stagerd

/bin/rm $TRCTMP

#  Make sure we're running as root.   Although most things will work,
#  without being root, we should warn the user that the report is
#  incomplete.

UID=`/bin/id | sed -e "s/^[^=]*=//" -e "s/(.*//"`
if [ $UID -ne 0 ] ; then
	echo "Warning:  This script should be run as 'root'."
	echo " "
    echo "    But, since you have started it as uid ${UID}, it will make the"
	echo "    best effort it can.  The output report will be missing some"
    echo "    important pieces of information, and you will probably see some"
    echo "    error messages while this script is running.  If you have access"
    echo "    to root, you should re-run this script as root."
	echo " "
	echo $TAG NOT RUN AS ROOT >> $REPORT 2>&1
	echo $TAG NOT RUN AS ROOT >> $REPORT 2>&1
	echo $TAG NOT RUN AS ROOT >> $REPORT 2>&1
	echo " " >> $REPORT 2>&1
fi

echo "This report generated into: $REPORT"   >>$REPORT
echo ""   >>$REPORT 2>&1

echo "syslogd facility code:      $LOGIS"    >>$REPORT
echo ""   >>$REPORT 2>&1

echo "SAM-FS Log:                 $SAMLOG"   >>$REPORT
echo "Archiver Logs:              $ARCHLOGS" >>$REPORT
echo "Archiver Traces:            $ARCHTRC"  >>$REPORT
echo "Archiver Data:              $ARCHDATA" >>$REPORT
echo "Catalog Trace:              $CATTRC"   >>$REPORT
echo "Device Logs:                $DEVLOGS"  >>$REPORT
echo "FSD Trace:                  $FSDTRC"   >>$REPORT
echo "FTPD Trace:                 $FTPTRC"   >>$REPORT
echo "Recycler Log:               $RECLLOGS" >>$REPORT
echo "Recycler Trace:             $RCYTRC"   >>$REPORT
echo "Releaser Log:               $RELLOG"   >>$REPORT
echo "Share FS Trace:             $SHFSTRC"  >>$REPORT
echo "Stage Log:                  $STAGELOG" >>$REPORT
echo "Stager Log:                 $STAGER_LOG"   >>$REPORT
echo "Stager Trace:               $STGTRC"   >>$REPORT
echo "Stager Data:                $STAGER_DATA"  >>$REPORT
echo ""   >>$REPORT 2>&1

echo "------------------- locale --------------------------" >>$REPORT
verbose /bin/locale
LC_ALL=$LOCALE
export LC_ALL

echo "------------------- boot time -----------------------" >>$REPORT
verbose /bin/who -b

echo "------------------- uptime --------------------------" >>$REPORT
verbose /bin/uptime 

echo "------------------- system type ---------------------" >>$REPORT
SYS_TYPE=`/bin/grep -i nexus /var/adm/messages* 2>&1 | head -1 | sed -e "s/^.*nexus = //"`
if [ X = X$SYS_TYPE ]; then
    SYS_TYPE="Not Available"
fi
echo "\n$SYS_TYPE" >>$REPORT
echo "" >>$REPORT 2>&1

echo "------------------- uname ---------------------------" >>$REPORT
verbose /bin/uname -a

#
#  Check for some Solaris patches we know are required for proper operation
#  of SAM-FS.  There may be others; this list is subject to change.
#  Issue a warning if the patch isn't there; otherwise echo the full patch id.
#
SVER=`uname -r | cut -c1-5`
if [ $SVER = "5.6" ]; then
  NEEDED="105181 105568"
elif [ $SVER = "5.7" ]; then
  NEEDED="106541"
  KERN_ARCH=`/bin/isainfo -kv`
  KERN_VER=` echo $KERN_ARCH | cut -c1-2`
elif [ $SVER = "5.8" ]; then
  NEEDED="NONE"
  KERN_ARCH=`/bin/isainfo -kv`
  KERN_VER=` echo $KERN_ARCH | cut -c1-2`
elif [ $SVER = "5.9" ]; then
  NEEDED="NONE"
  KERN_ARCH=`/bin/isainfo -kv`
  KERN_VER=` echo $KERN_ARCH | cut -c1-2`
else
  NEEDED="UNKNOWN"
  echo $TAG "Unknown Solaris revision $SVER"       >>$REPORT
fi

echo "------------------- kernel architecture info --------" >>$REPORT

if [ X"$KERN_VER" = X64 ]; then
    echo "\nSolaris is running in 64-bit kernel mode."  >>$REPORT
else
    echo "\nSolaris is running in 32-bit kernel mode."  >>$REPORT
fi
if [ X"$KERN_ARCH" != X ]; then
    echo $KERN_ARCH                                  >>$REPORT
fi
echo "" >>$REPORT 2>&1

showfiles /etc/release
echo "" >>$REPORT 2>&1

echo "------------------- check system patches ------------" >>$REPORT
echo "\nPatches needed for Solaris ${SVER}: $NEEDED\n" >>$REPORT
if [ X"$NEEDED" = "XNONE" -o X"$NEEDED" = "UNKNOWN" ]; then
    NEEDED=""
fi

/bin/showrev -p 2>/dev/null | /bin/sed -e 's/Obso.*//' | /bin/sort > /tmp/$$.patchlist
for patch in $NEEDED; do
    PATCHFOUND=`/bin/grep -w $patch /tmp/$$.patchlist`
    if [ "X$PATCHFOUND" = "X" ]; then
      echo $TAG "Solaris patch " $patch " not installed.\n" >>$REPORT
    else
      echo "Solaris " $PATCHFOUND " installed.\n" >>$REPORT
    fi
done
echo "/bin/showrev -p | /bin/sed -e 's/Obso.*//' | /bin/sort > /tmp/$$.patchlist\n" >>$REPORT
showifiles /tmp/$$.patchlist
rm /tmp/$$.patchlist

#
# SUNWqfs   - Sun QFS Solaris (system)
# SUNWsamfs - Sun SAM-FS and Sun SAM-QFS software Solaris (system)
# SUNWsamtp - Sun SAM-FS tools (application)
# SUNWmd    - Solstice DiskSuite (system)
# SUNWmdg   - Solstice DiskSuite Tool (system)
# SUNWfcp   - Sun FCP SCSI Device Driver (system)
# SUNWfcpx  - Sun FCP SCSI Device Driver (64-bit) (system)
# SUNWfctl  - Sun Fibre Channel Transport layer (system)
# SUNWfctlx - Sun Fibre Channel Transport layer (64-bit) (system)
# SUNWqlc   - Qlogic ISP 2200/2202 Fibre Channel Device Driver (system)
# SUNWqlcx  - Qlogic ISP 2200/2202 Fibre Channel Device Driver (64 bit) (system)
# SUNWsmgr  - SANsurfer, QLogic Fibre Channel Administration (system)
# SANergy   - SANergy solaris sparc (application)
# AMPXdst   - Ampex DST Tape Drive Device Driver (system)
# JNIC      - JNI Fibre Channel SCSI HBA Driver (system)
# QLA2200-3 - QLogic QLA2200 driver (system)
#
echo "------------------- package info --------------------" >>$REPORT

verbose "/bin/pkginfo | grep LSC"

for PACKAGE in SUNWqfs SUNWsamfs SUNWsamtp SUNWmd SUNWmdg SUNWfcp SUNWfcpx SUNWfctl SUNWfctlx SUNWqlc SUNWqlcx SUNWsmgr SANergy AMPXdst JNIC QLA2200-3
do
    twiddle
    echo "" >>$REPORT 2>&1
    verbose /bin/pkginfo -l $PACKAGE
    verbose /usr/sbin/pkgchk  $PACKAGE
done

showfiles /opt/SUNWsamfs/include/version.h

echo "------------------- /kernel info ---------------------" >>$REPORT
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -l /kernel/drv/sam*"           >>$REPORT 2>&1
/bin/ls -l /kernel/drv/sam*                  >>$REPORT 2>&1
verbose /bin/ls -l /kernel/fs/sam*
verbose /bin/ls -l /kernel/sys/sam*

echo "/bin/ls -l /kernel/drv/sparcv9/sam*"   >>$REPORT 2>&1
/bin/ls -l /kernel/drv/sparcv9/sam*          >>$REPORT 2>&1
verbose /bin/ls -l /kernel/fs/sparcv9/sam*
verbose /bin/ls -l /kernel/sys/sparcv9/sam*

echo "------------------- name_to_sysnum ------------------" >>$REPORT
verbose /bin/ls -l /etc/name_to_sysnum
verbose "/bin/grep sam /etc/name_to_sysnum"

echo "------------------- modinfo output ------------------" >>$REPORT
verbose "/usr/sbin/modinfo | /bin/grep sam | /bin/grep -v sampling"
verbose "/usr/sbin/modinfo | /bin/grep ' sd '"
verbose "/usr/sbin/modinfo | /bin/grep ' st '"
verbose "/usr/sbin/modinfo | /bin/egrep -i 'fcp|fctl|fp|qlc|ssd|disk|tape'"

echo "------------------- driver output -------------------" >>$REPORT
/bin/grep Driver /var/adm/messages* 2>&1 | head -10 >>$REPORT 2>&1
echo ""   >>$REPORT 2>&1

echo "------------------- storage array info ------------------" >>$REPORT
if [ -f /usr/sbin/luxadm ]; then
	verbose /usr/sbin/luxadm qlgc
fi
#
# Disable cfgadm until bugid 4736456 is resolved.  We only
# want to run it when the proper patch is installed.
#
#if [ -f /usr/sbin/cfgadm ]; then
#	verbose /usr/sbin/cfgadm -al
#fi

echo "------------------- ls -lR /opt/SUNWsamfs ------------" >>$REPORT
verbose /bin/ls -lR  /opt/SUNWsamfs
verbose /bin/ls -lRL /opt/SUNWsamfs

echo "------------------- $ETCDIR directory ---------------" >>$REPORT
verbose /bin/ls -ld  $ETCDIR
verbose /bin/ls -lLd  $ETCDIR
verbose /bin/ls -l  $ETCDIR
verbose /bin/ls -lL $ETCDIR

echo "------------------- $VARDIR directory ---------------" >>$REPORT
verbose /bin/ls -ld  $VARDIR
verbose /bin/ls -lLd  $VARDIR
verbose /bin/ls -l  $VARDIR
verbose /bin/ls -lL $VARDIR

showfiles $ETCDIR/mcf
showfiles $ETCDIR/hosts*
showfiles $ETCDIR/defaults.conf
showfiles $ETCDIR/inquiry.conf
showfiles $ETCDIR/mount.sh
showfiles $EXECDIR/dev_down.sh
showfiles $EXECDIR/save_core.sh
showfiles $ETCDIR/archiver.cmd
showfiles $EXECDIR/ar_notify.sh
showfiles $ETCDIR/preview.cmd
showfiles $ETCDIR/recycler.cmd
showfiles $EXECDIR/recycler.sh
showfiles $ETCDIR/releaser.cmd
showfiles $ETCDIR/samfs.cmd
showfiles $ETCDIR/samlogd.cmd
showfiles $ETCDIR/stager.cmd
showfiles $ETCDIR/diskvols.conf
showfiles /etc/syslog.conf 
showfiles /kernel/drv/samst.conf
showfiles /kernel/drv/samrd.conf
showfiles /kernel/drv/st.conf
showfiles /kernel/drv/fp.conf
showfiles /kernel/drv/fcp.conf
showfiles /kernel/drv/qlc.conf
showfiles /kernel/drv/qla.conf
showfiles /kernel/drv/qla2200.conf
showfiles /usr/kernel/drv/dst.conf 
showfiles /etc/driver_classes
showfiles /etc/driver_aliases
showfiles /etc/system

echo "------------------- /etc/system flags ---------------" >>$REPORT
if grep sam_statvfs_bias /etc/system > /dev/null; then
     echo $TAG found sam_statvfs_bias in /etc/system
fi
echo ""   >>$REPORT 2>&1

echo "------------------- samset --------------------------" >>$REPORT
echo ""   >>$REPORT 2>&1
verbose /opt/SUNWsamfs/sbin/samset
echo ""   >>$REPORT 2>&1
verbose /opt/SUNWsamfs/sbin/samset debug
echo ""   >>$REPORT 2>&1
verbose /opt/SUNWsamfs/sbin/samset devlog all

echo "------------------- hostid --------------------------" >>$REPORT
echo ""   >>$REPORT 2>&1
echo "Hostid is  `/usr/ucb/hostid`" >>$REPORT 2>&1
echo ""   >>$REPORT 2>&1
showfiles $ETCDIR/LICENSE.4.0
showfiles $ETCDIR/LICENSE.dat

echo "------------------- jre info ------------------------" >>$REPORT
if [ -r /opt/SUNWsamfs/jre/bin/jre -a -x /opt/SUNWsamfs/jre/bin/jre ]; then
    verbose "/opt/SUNWsamfs/jre/bin/jre -v 2>&1 | grep -i Version"
else
    echo ""   >>$REPORT 2>&1
fi

echo "------------------- rpc info ------------------------" >>$REPORT
verbose /bin/grep rpc      /etc/nsswitch.conf
verbose /bin/grep services /etc/nsswitch.conf
verbose /bin/grep sam      /etc/rpc
verbose "/bin/ypcat rpc      | grep sam"
verbose /bin/grep sam      /etc/services
verbose /bin/grep sam      /etc/yp/src/services
verbose "/bin/ypcat services | grep sam"

echo "------------------- ipcs info -----------------------" >>$REPORT
verbose /bin/ipcs -a

echo "------------------- /dev/samst ----------------------" >>$REPORT
verbose /bin/ls -l /dev/samst
verbose /bin/ls -Ll /dev/samst

echo "------------------- /dev/samrd ----------------------" >>$REPORT
verbose /bin/ls -l /dev/samrd
verbose /bin/ls -Ll /dev/samrd

echo "------------------- /dev/rmt ------------------------" >>$REPORT
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -l /dev/rmt/*[0-9]"            >>$REPORT 2>&1
/bin/ls -l /dev/rmt/*[0-9]                   >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -Ll /dev/rmt/*[0-9]"           >>$REPORT 2>&1
/bin/ls -Ll /dev/rmt/*[0-9]                  >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1

echo "------------------- /dev/rdst -----------------------" >>$REPORT
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -l /dev/rdst*"                 >>$REPORT 2>&1
/bin/ls -l /dev/rdst*                        >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -Ll /dev/rdst*"                >>$REPORT 2>&1
/bin/ls -Ll /dev/rdst*                       >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1

echo "------------------- /dev/dsk ------------------------" >>$REPORT
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -l /dev/dsk/*s2"               >>$REPORT 2>&1
/bin/ls -l /dev/dsk/*s2                      >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -Ll /dev/dsk/*s2"              >>$REPORT 2>&1
/bin/ls -Ll /dev/dsk/*s2                     >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1

echo "------------------- /dev/rdsk -----------------------" >>$REPORT
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -l /dev/rdsk/*s2"              >>$REPORT 2>&1
/bin/ls -l /dev/rdsk/*s2                     >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1
echo "/bin/ls -Ll /dev/rdsk/*s2"             >>$REPORT 2>&1
/bin/ls -Ll /dev/rdsk/*s2                    >>$REPORT 2>&1
echo ""                                      >>$REPORT 2>&1

echo "------------------- SANergy info --------------------" >>$REPORT
verbose '/bin/ls -l /opt/SANergy/lib /opt/SANergy/lib/sparcv9'
verbose '/bin/ls -Ll /opt/SANergy/lib /opt/SANergy/lib/sparcv9'

echo "------------------- group entry ---------------------" >>$REPORT
verbose /bin/ls -l /etc/group
verbose /bin/ls -Ll /etc/group
verbose /usr/bin/grep SAM_FS /etc/group
verbose /usr/bin/ypmatch SAM_FS group

if [ "x$SAMLOG" = x ]; then
	echo $TAG " No sam log file defined."  >>$REPORT
else
	showenuf $SAMLOG\*
fi

for FILES in $ARCHLOGS $ARCHTRC $RECLLOGS $RCYTRC $DEVLOGS $RELLOG $STAGELOG $STAGER_LOG $STGTRC $CATTRC $FSDTRC $FTPTRC $SHFSTRC
do
	showtails $FILES
done

showenuf /var/adm/messages\*

showtails $VARDIR/.grau/graulog*
showtails /tmp/.grau

showfiles /var/spool/cron/crontabs/root

#
# Do not run adb threadlist on running system for E10K/E15K systems
# due to the send_mondo panic problem.
#
echo "------------------- threadlist of running system ----" >>$REPORT
UNAME="`uname -a`"
if [ `echo $UNAME | grep -c "SUNW,Ultra-Enterprise-1.000"` -eq 0 ]; then
	echo '=n"---- date/time ----"
		time/Y
		lbolt/X
		=n"---- utsname ----"
		$<utsname
		=n"---- threadlist ----"
		$<threadlist
		=n"---- sam_ adb end ----"' | /bin/adb -k    >> $REPORT 2>&1
else
	echo "Skipping adb threadlist for E10K/E15K systems.\n"  >>$REPORT
fi

echo "------------------- stack tracebacks ----------------" >>$REPORT
if [ $UID -eq 0 ]; then
	if [ -f /usr/proc/bin/pstack -a -f /usr/proc/bin/pflags -a -f /usr/proc/bin/ptree ]; then
		# ps -e only returns 8 characters, so match on first 8.
		SAM_FSD=`/usr/bin/ps -e | grep sam-fsd | /bin/awk '{print $1}'`
		if [ -n "$SAM_FSD" ]; then
			echo "\n/usr/proc/bin/ptree $SAM_FSD"            >>$REPORT
			/usr/proc/bin/ptree $SAM_FSD                     >>$REPORT
			/usr/proc/bin/ptree $SAM_FSD > /tmp/$$.procs
			SAMPROCS=`/bin/cat /tmp/$$.procs | /bin/awk '{print $1}'`
			for PID in $SAMPROCS ; do
				verbose "/usr/proc/bin/pstack $PID; /usr/proc/bin/pflags $PID"
			done
			/bin/rm -f /tmp/$$.procs
		else
			echo "Sorry, sam-fsd process not running.\n"   >>$REPORT
		fi
	else
		echo "Sorry, no pstack, pflags, or ptree available.\n"   >>$REPORT
	fi
else
	rerun_as_root pstack
fi

echo "------------------- samtrace ------------------------" >>$REPORT
verbose "/opt/SUNWsamfs/sbin/samtrace -v | /bin/compress -c | /bin/uuencode samtrace.Z"

echo "------------------- sam-fsd information -------------" >>$REPORT
verbose /usr/lib/fs/samfs/sam-fsd

echo "------------------- filesystem information ----------" >>$REPORT
# Background this command so it doesn't hang info.sh on fs problems.
echo ""     >>$REPORT 2>&1
echo "    " $PROMPT /bin/df -kl >>$REPORT 2>&1
/bin/df -kl > /tmp/$$.bg.out 2>&1 &
sleep 5
twiddle
sleep 5
if [ -s /tmp/$$.bg.out ]; then
    cat /tmp/$$.bg.out | sed -e "s/^/     /"  >>$REPORT 2>&1
else
    echo "     $TAG /bin/df -kl timed out" >>$REPORT 2>&1
fi
echo ""     >>$REPORT 2>&1
rm -f /tmp/$$.bg.out

verbosei /etc/mount 

showifiles /etc/mnttab /etc/vfstab /etc/dfs/dfstab /etc/inittab

# accumulate samfs filesystem mount points
SAMFSMP=`sed -e "/#/s/#.*//" < /etc/vfstab | /bin/awk '$4 == "samfs" { print $3 " "}'`

# accumulate samfs filesystem names
SAMFSES=`sed -e "/#/s/#.*//" < /etc/vfstab | /bin/awk '$4 == "samfs" { print $1 " "}'`

echo "------------------- filesystem inodes ---------------" >>$REPORT
# Background this command so it doesn't hang info.sh on fs problems.
for dir in $SAMFSMP; do
    echo ""     >>$REPORT 2>&1
    echo $PROMPT /bin/ls -al $dir/.inodes >>$REPORT 2>&1
    /bin/ls -al $dir/.inodes > /tmp/$$.bg.out 2>&1 &
    sleep 5
    twiddle
    if [ -s /tmp/$$.bg.out ]; then
        cat /tmp/$$.bg.out >>$REPORT 2>&1
    else
        echo "$TAG /bin/ls -al $dir/.inodes timed out" >>$REPORT 2>&1
    fi
    echo ""     >>$REPORT 2>&1
    rm -f /tmp/$$.bg.out
done

echo "------------------- filesystem info -----------------" >>$REPORT
for fs in $SAMFSES; do
    verbose /opt/SUNWsamfs/sbin/samfsinfo $fs
done

echo "------------------- shared filesystem info ----------" >>$REPORT
for fs in $SAMFSES; do
    if [ `mount -v | grep "type samfs" | cut -d" " -f1 | egrep -c "^$fs$"` -eq 0 ]; then
        verbose /opt/SUNWsamfs/sbin/samsharefs -R $fs
    else
        verbose /opt/SUNWsamfs/sbin/samsharefs $fs
    fi
done

echo "------------------- filesystem parameters -----------" >>$REPORT
for SAMFS in $SAMFSMP; do
    verbose /opt/SUNWsamfs/sbin/samcmd N $SAMFS
done

echo "------------------- samu filesystems info -----------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd f

echo "------------------- samu daemon trace info ----------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd d

echo "------------------- samu mass storage info ----------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd m

echo "------------------- samu preview info ---------------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd p

echo "------------------- samu pending stage queue --------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd w

echo "------------------- samu staging queue info ---------" >>$REPORT
verbosehead /opt/SUNWsamfs/sbin/samcmd u

echo "------------------- samu removable media info -------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd r

echo "------------------- samu staging activity info ------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/samcmd n

echo "------------------- samu archiver info --------------" >>$REPORT
for SAMFS in $SAMFSES; do
    verbose /opt/SUNWsamfs/sbin/samcmd a $SAMFS
done

echo "------------------- archiver -lv --------------------" >>$REPORT
echo ""   >>$REPORT 2>&1
verbose /bin/grep wait $ETCDIR/archiver.cmd
verbose /opt/SUNWsamfs/sbin/archiver -lv
echo ""   >>$REPORT 2>&1

echo "------------------- showqueue -v --------------------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/showqueue -v

echo "------------------- system panic dumps --------------" >>$REPORT
if [ $SVER != "5.6" ]; then
  if [ $UID -eq 0 ]; then
    verbose /usr/sbin/dumpadm 
    dumpdir=`/usr/sbin/dumpadm | grep "core dir" | sed -e "s/Savecore directory: //" | /bin/awk '{print $1}'`
    enabled=`/usr/sbin/dumpadm | grep "core ena" | sed -e "s/Savecore enabled: //" | /bin/awk '{print $1}'`
	if [ -n "$enabled" ]; then
		if [ $enabled = "no" ]; then
			echo $TAG Core dump program \'savecore\' disabled by /usr/sbin/dumpadm >>$REPORT
		fi
	fi
  else
    rerun_as_root dumpadm
  fi
else
  if [ -r /etc/rc2.d/S20sysetup ]; then
	grep savecore /etc/rc2.d/S20sysetup >/dev/null 2>&1
	if [ $? ]; then
		dumpdir=`grep savecore /etc/rc2.d/S20sysetup|grep -v "^#"|sed -e "s/savecore//"`
	else
		echo $TAG Core dump program \'savecore\' commented out in /etc/rc2.d/S20sysetup >>$REPORT
	fi
  else
		echo $TAG Core dump program \'savecore\' not found in /etc/rc2.d/S20sysetup >>$REPORT
  fi
fi

if [ -n "$dumpdir" ]; then
	coredir=`eval echo $dumpdir`
	verbose /bin/ls -ld $coredir
	verbose /bin/ls -l $coredir
	for unix in `ls $coredir/unix.[0-9]* 2>/dev/null`
	do
		twiddle
		vmcore=`echo $unix | sed -e "s./unix./vmcore."`
		verbose /bin/ls -l $unix $vmcore
		echo '=n"---- panic string ----"
			*panicstr/s
			=n"---- date/time ----"
			time/Y
			lbolt/X
			=n"---- utsname ----"
			$<utsname
			=n"---- stack traceback ----"
			$c
			=n"---- threadlist ----"
			$<threadlist
			=n"---- sam_ adb end ----"' | adb -k $unix $vmcore >>$REPORT 2>&1
	done
fi

echo "------------------- ps -edalf -----------------------" >>$REPORT
verbose /usr/bin/ps -edalf

if [ $SVER != "5.6" -a $SVER != "5.7" ]; then
    echo "------------------- prstat --------------------------" >>$REPORT
    verbose /usr/bin/prstat 1 1
fi

echo "------------------- prtconf -------------------------" >>$REPORT
verbose /usr/sbin/prtconf -V
verbose /usr/sbin/prtconf

echo "------------------- shared memory -------------------" >>$REPORT
verbose /opt/SUNWsamfs/sbin/dmpshm

echo "------------------- archiver datadir ----------------" >>$REPORT
verbose /bin/ls -ld $VARDIR/archiver
verbose /bin/ls -lLd $VARDIR/archiver
verbose /bin/ls -lLa $VARDIR/archiver
verbose /bin/ls -ld $ARCHDATA
verbose /bin/ls -lLd $ARCHDATA
verbose /bin/ls -lLa $ARCHDATA

if [ -d $ARCHDATA ]; then
  if [ -r $ARCHDATA -a -x $ARCHDATA ]; then
	(cd $ARCHDATA; tar cf /tmp/$$.tar .; compress /tmp/$$.tar) >>$REPORT 2>&1
	if [ -r /tmp/$$.tar.Z ]; then
		uuencode archive.tar.Z </tmp/$$.tar.Z > /tmp/$$.tarz
		head -$HOW_MANY_LINES /tmp/$$.tarz >>$REPORT 2>&1
		TARN=`/usr/ucb/wc /tmp/$$.tarz | cut -c1-8`
		if [  $TARN -gt $HOW_MANY_LINES ]; then
			echo ""   >>$REPORT 2>&1
			echo ""   >>$REPORT 2>&1
			echo "end" >>$REPORT 2>&1
		fi
		rm -f /tmp/$$.tar.Z /tmp/$$.tarz
	else
		echo $TAG Something is wrong in $ARCHDATA >>$REPORT 2>&1
	fi
  else
	rerun_as_root $ARCHDATA
  fi
fi

echo "------------------- stager datadir ----------------" >>$REPORT
verbose /bin/ls -ld $STAGER_DATA
verbose /bin/ls -lLd $STAGER_DATA
verbose /bin/ls -lLa $STAGER_DATA

if [ -d $STAGER_DATA ]; then
  if [ -r $STAGER_DATA -a -x $STAGER_DATA ]; then
	(cd $STAGER_DATA; tar cf /tmp/$$.tar .; compress /tmp/$$.tar) >>$REPORT 2>&1
	if [ -r /tmp/$$.tar.Z ]; then
		uuencode stager.tar.Z </tmp/$$.tar.Z > /tmp/$$.tarz
		head -$HOW_MANY_LINES /tmp/$$.tarz >>$REPORT 2>&1
		TARN=`/usr/ucb/wc /tmp/$$.tarz | cut -c1-8`
		if [  $TARN -gt $HOW_MANY_LINES ]; then
			echo ""   >>$REPORT 2>&1
			echo ""   >>$REPORT 2>&1
			echo "end" >>$REPORT 2>&1
		fi
		rm -f /tmp/$$.tar.Z /tmp/$$.tarz
	else
		echo $TAG Something is wrong in $STAGER_DATA >>$REPORT 2>&1
	fi
  else
	rerun_as_root $STAGER_DATA
  fi
fi

echo "------------------- mcf analysis --------------------" >>$REPORT
if [ -r $ETCDIR/mcf ]; then
    #  Check to see if two device names (/dev/rmt/0 and /dev/samst/c1t2u0, for example) 
    #  refer to the same hardware device.
	compare_devices() {
		if [ $# -ne 2 -o X$1 = X- -o X$2 = X- ]; then
			echo Only $# device name\(s\), or device name was a dash - no comparison done >>$REPORT 2>&1
        else 
			if [ ! -r $1 ]; then
				echo $TAG $1 does not exist >>$REPORT 2>&1
				return
			fi
			if [ ! -r $2 ]; then
				echo $TAG $2 does not exist >>$REPORT 2>&1
				return
			fi
			dev1=`ls -l $1 | sed -e 's,.*/devices,,' | sed -e 's,\(.*/\)[^@]*@,\1@,' -e 's,:.*,,'`
			dev2=`ls -l $2 | sed -e 's,.*/devices,,' | sed -e 's,\(.*/\)[^@]*@,\1@,' -e 's,:.*,,'`
			if [ $dev1 != $dev2 ]; then
				echo $TAG $1 and $2 do not match >>$REPORT 2>&1
				echo $TAG $1 evaluates to $dev1 >>$REPORT 2>&1
				echo $TAG $2 evaluates to $dev2 >>$REPORT 2>&1
				echo >>$REPORT 2>&1
			else
				echo $1 and $2 seem to match >>$REPORT 2>&1
			fi
		fi
	}

    lineno=0
	exec <$ETCDIR/mcf
	while read eqid eqord eqtype famset state additional
	do
		# strip comments
		case $eqid in
		\#*)
			continue
			;;
		esac

		if [ X != X$additional ]; then
			catpath=$additional
			catp1=`echo $catpath | /bin/cut -c1`
			if [ X$catp1 != "X/" ]; then
				catpath=$CATDIR/$additional
			fi
		else
			catpath=$CATDIR/$famset
		fi

        lineno=`expr $lineno + 1`
		echo "    " $lineno ":" $eqid $eqord $eqtype $famset $state $additional >>$REPORT

		case $eqtype in
		'ma'|'ms')    
			;;
		'md'|'mm'|'mr'|'gx')    
			hope=`echo $eqid | sed -e "s/dsk/rdsk/"`
			if [ X != X$additional -a X- != X$additional ]; then
				if [ X$hope != X$additional ]; then
					echo $TAG raw and cooked devices do not match >>$REPORT
				else
					echo $eqid and $additional seem to match >>$REPORT
				fi
				verbosei /etc/prtvtoc $additional
			else
				verbosei /etc/prtvtoc $hope
			fi
			;;
		'rb'|'hp'|'ml'|'cy'|'dm'|'ds'|'me'|'ac'|'eb'|'ad'|'a1'|'a2'|'sl'|'s9'|'ic')
			echo "$eqtype dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
			echo "" >>$REPORT 2>&1
			;;
		'fj')    
			echo "Fujitsu LT300 parameter file follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid
			echo "Fujitsu LT300 dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
			echo "" >>$REPORT 2>&1
			;;
		'gr')    
			echo "GRAU/EMASS parameter file follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid
			echo "GRAU/EMASS dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
			echo "" >>$REPORT 2>&1
			;;
		'im')    
			echo "IBM parameter files follow" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid /etc/ibmatl.conf
			echo "IBM dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath 
			echo "" >>$REPORT 2>&1
			;;
		'sk')    
			echo "STK parameter file follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid
			echo "" >>$REPORT 2>&1
			showifiles $ETCDIR/ssi.sh
			echo "STK dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
			echo "" >>$REPORT 2>&1
			;;
		'hy')    
			echo "HY dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			hist_cat=$catpath
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
			echo "" >>$REPORT 2>&1
			;;
		'od'|'wo'|'mo'|'mf')    
			catpath=$CATDIR/man$eqord
			if [ -f $catpath ]; then
				echo "$eqtype$eqord dump_cat follows" >>$REPORT 2>&1
				echo "" >>$REPORT 2>&1
				verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
				echo "" >>$REPORT 2>&1
			fi
			;;
		'tp'|'lt'|'st'|'vt'|'dt'|'fd'|'se'|'sg'|'d3'|'d2'|'xt'|'ib'|'i7'|'so'|'at')
			catpath=$CATDIR/man$eqord
			if [ -f $catpath ]; then
				echo "$eqtype$eqord dump_cat follows" >>$REPORT 2>&1
				echo "" >>$REPORT 2>&1
				verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath
				echo "" >>$REPORT 2>&1
			fi

			compare_devices $eqid $additional
			;;
		'rd')
			;;
		'sc')
			echo "sam remote client parameter file follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid
			echo "remote client dump_cat follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $catpath 
			echo "" >>$REPORT 2>&1
			echo ""
			echo "Please run /opt/SUNWsamfs/sbin/info.sh on any remote servers"
			echo "and include the SAMreports with this one. "
			COUNT=999
			;;
		'ss')
			echo "sam remote server parameter file follows" >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			showifiles $eqid
			echo ""
			echo "Please run /opt/SUNWsamfs/sbin/info.sh on any remote clients"
			echo "and include the SAMreports with this one. "
			COUNT=999
			;;
		z?)
			echo "sam migkit library " >>$REPORT 2>&1
			echo "" >>$REPORT 2>&1
			verbosei /bin/ls -l $eqid
			verbosei /bin/ls -Ll $eqid
			;;
		default)
			echo $TAG "Unknown equipment type" >>$REPORT 2>&1
			;;
		esac
	done
elif [ -f $ETCDIR/mcf ]; then
		rerun_as_root $ETCDIR/mcf
elif [ -d $ETCDIR ]; then
		echo $TAG missing $ETCDIR/mcf >>$REPORT 2>&1
fi

if [ x$hist_cat = x ]; then
	echo "default HY dump_cat follows" >>$REPORT 2>&1
	echo "" >>$REPORT 2>&1
	verbose /opt/SUNWsamfs/sbin/dump_cat $DUMPCATOPTS $CATDIR/historian
	echo "" >>$REPORT 2>&1
fi

echo "------------------- dump_log ------------------------" >>$REPORT
verbose /bin/ls -l /var/adm/log/fs*log
echo /opt/SUNWsamfs/sbin/dump_log >>$REPORT 2>&1 
/opt/SUNWsamfs/sbin/dump_log >>$REPORT 2>&1

echo "------------------- core files ----------------------" >>$REPORT
echo "" >>$REPORT 2>&1

ONLINE_CORE_FILES=""
for DIR in $ETCDIR $VARDIR $USERDIR $EXECDIR; do

    CORE_FILES=""
    if [ -d $DIR ]; then
        if [ -r $DIR -a -x $DIR ]; then
            echo "CORE_FILES=/bin/find $DIR -follow -name \"core*\" -type f -print" >>$REPORT
            CORE_FILES=`/bin/find $DIR -follow -name "core*" -type f -print` 
            for file in $CORE_FILES; do
                verbose "/bin/ls -tlLd $file; /bin/file $file"
            done
        else
            rerun_as_root $DIR
        fi
    else
        echo $TAG missing $DIR >>$REPORT 2>&1
    fi

    if [ x"$CORE_FILES" != x ]; then
        ONLINE_CORE_FILES="$ONLINE_CORE_FILES $CORE_FILES"
    else
        echo "" >>$REPORT 2>&1
    fi

done

if [ -f /core ]; then
    verbose "/bin/ls -tlLd /core; /bin/file /core"
    ONLINE_CORE_FILES="$ONLINE_CORE_FILES /core"
fi

if [ $SVER != "5.6" -a $SVER != "5.7" ]; then
	echo "------------------- core file stack tracebacks ------" >>$REPORT
	if [ $UID -eq 0 ]; then
		if [ -f /usr/proc/bin/pstack -a -f /usr/proc/bin/pflags ]; then
			for CORE in $ONLINE_CORE_FILES; do
				verbose "/usr/proc/bin/pstack $CORE; /usr/proc/bin/pflags $CORE"
			done
		else
			echo "Sorry, no pstack/pflags."   >>$REPORT
		fi
	else
		rerun_as_root pstack
	fi
fi

echo "" >>$REPORT 2>&1
echo "End at `/bin/date` on `/bin/uname -n` by `/bin/id`" >>$REPORT
echo " " >>$REPORT
echo "------------------- THE END -------------------------" >>$REPORT
chmod 444 $REPORT
echo " "
echo " "
echo "The report in $REPORT should now be ftp'ed to your"
echo "support provider as a binary file.  So that you do not experience"
echo "unnecessary delays, please DO NOT encode or compress the report"
echo "in any way when sending it."
echo " "
echo "However, if you still feel you must compress the file, use only"
echo "the standard Solaris 'compress' utility."
echo " "
if [ x"$ONLINE_CORE_FILES" != x ]; then
    echo "This script found the following core files.  Please ftp them as"
    echo "binary files to your support provider only if they appear to be"
    echo "from one of our products."
    echo " "
    for file in $ONLINE_CORE_FILES; do
        /bin/file $file | sed -e "s/:.*,/ :/" ; /bin/ls -tlLd $file
        echo
    done
fi
