#! /usr/bin/sh
#
# ident	"@(#)db-setup.sh	1.35	03/04/11 SMI"
#
# Copyright 2002-2003 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#

prm_cleanupAndExit () {
    rcode=$1

    echolog ''
    echolog 'Aborting PRM database setup.'
    echolog ''

    # Clean up any temp files belonging to us -- just in case
    /usr/bin/rm -f $TEMPDIR/*.$$

    # Make sure DB is shutdown properly
    prm_exitStopDb $rcode
}

# Check for prerequisites (eg. system utilities required by setup)
prm_checkPrereq () {
    echologverbose 'Checking PRM prerequisites'

    if [ ! -f /usr/ccs/bin/make -o ! -f /usr/ccs/bin/ld -o \
         ! -f /usr/ccs/bin/ar ] ; then
        echolog ""
        echolog "The PRM Add-On requires the following executables: make, ld, ar"
        echolog "to be present on the host machine in the directory  /usr/ccs/bin"
        echolog "Please install the SUNWsprot, SUNWtoo, SUNWbtool  packages before" 
        echolog "attempting to execute the es-setup again."

        return 1
    fi

    return 0
}


# NOTE: This MUST be called before any actual setup operations (database
# changes, etc.) are performed. The existence of the configuration file
# indicates an existing (possibly incomplete) setup.
prm_createCfg () {
    echologverbose 'Creating PRM database config file'

    if [ ! -f "$PRM_CFGFILE" ] ; then
        echologverbose 'Creating PRM database configuration file'

        PRM_USER=report
        PRM_PASSWD=report

        if [ -z "$INSTALL_TYPE" ] ; then
            echolog 'Internal error: PRM installation type not defined!'
            return 3
        fi

        /usr/bin/cat<<-EOF >"$PRM_CFGFILE"
user=$PRM_USER
password=$PRM_PASSWD
type=$INSTALL_TYPE
EOF
        if [ $? -ne 0 ] ; then
            echolog 'Failed to create PRM database config file.'
            return 1
        fi
    fi
    return 0
}

#
# Check for a previous PRM setup - even a partial setup is considered
# to be a "previous setup".
#
# Inputs:
#   none
#
# Returns:
#   0 - no previous setup
#   1 - found a previous setup
#
prm_checkSetup () {
    echologverbose 'Checking for existing PRM setup'

    # Check for existing PRM setup

    /usr/bin/pkginfo SUNWesdrg >/dev/null 2>&1
    if [ $? -ne 0 ] ; then
        echolog 'No existing PRM installation found.'
        return 0
    fi

    # Check for existence of PRM database config file. The presence of this
    # file is an indicator of whether a (possibly incomplete) setup already
    # exists.
    if [ -f "$PRM_CFGFILE" ] ; then
        # Found previous setup
        installtype=`/usr/bin/egrep '^type=' $PRM_CFGFILE | /usr/bin/sed -e 's/^type=//' `
        if [ ! -z "$installtype" ] ; then
            prm_cfgInstallationType $installtype
        fi

        return 1
    fi

    return 0
}

# Checks if all tablespaces for the PRM user are still intact
# Returns:
#	0	All tablespaces still appear to be intact
#	1	One or more tablespaces are missing
#
# NOTE: will abort if a serious database error occurred.
prm_checkTablespaces () {
    echologverbose 'Checking for existing PRM tablespaces'

    tslist=`prm_getCfg tablespaces`
    if [ $? -ne 0 ] ; then
        echolog 'Unable to get tablespace list from $2' "$PRM_TABLESPACE_CFGFILE"
        prm_cleanupAndExit 1
    fi

    for ts in $tslist ; do
        db_checkTablespace $ts $SYS_USER $SYS_PASSWD $LOGFILE
        rcode=$?
        if [ $rcode -eq 1 ] ; then
            # Tablespace seems to be missing
            return 1
        fi
        if [ $rcode -eq 2 ] ; then
            prm_cleanupAndExit 2
        fi
    done

    # PRM tablespaces appear to be OK
    return 0
}

prm_recompileInvalidObjects () {
    tmpfile=$TEMPDIR/prm-recomp.$$
    /usr/bin/rm -f $tmpfile

    echologverbose 'Recompiling invalid database objects...'
    db_runRecompInv $SYS_USER $SYS_PASSWD $tmpfile $LOGFILE
    if [ $? -ne 0 ] ; then
        echolog 'Failed to recompile all database objects'
        /usr/bin/rm -f $tmpfile
        return 1
    fi

    /usr/bin/rm -f $tmpfile
    return 0
}

# Re-create SunMC synonyms and permissions (in case they were lost)
prm_recreateSynonyms () {
    steps_file=$SQL_SCRIPTS/recreate_synonyms.txt
    /usr/bin/egrep -v '^--|^$' "$steps_file" | while read line ; do
        prm_runStep "$line" 2
        if [ $? -ne 0 ] ; then
            return 1
        fi
    done
    return $?
}

# Drop user for reference schema
prm_dropRefUser () {
    db_checkUser $SYS_USER $SYS_PASSWD rep $LOGFILE
    rcode=$?
    if [ $rcode -eq 0 ] ; then
        db_dropUser rep $SYS_USER $SYS_PASSWD $LOGFILE
        if [ $? -ne 0 ] ; then
            echolog 'Error while dropping reference user'
#NOTE: should we abort, or is it OK to continue with a failure code?
            prm_cleanupAndExit 2
        fi
    elif [ $rcode -eq 2 ] ; then
        echolog 'Error while checking for reference user'
        prm_cleanupAndExit 2
    fi

    return 0
}

prm_createRefSchema () {
    echologverbose 'Creating reference PRM schema'
    if [ ! -f "$STEPS_FILE" ] ; then
        echolog 'Unable to find SQL steps file: $2' "$STEPS_FILE"
        return 1
    fi

    # Parse steps file, ignoring comment lines (starts with "--") and
    # blank lines.
    /usr/bin/egrep -v '^--|^$' $STEPS_FILE | while read line ; do
        prm_runStep "$line" 3
        if [ $? -ne 0 ] ; then
            return 1
        fi
    done

    return $?
}

prm_compareRefSchema () {
    tmpfile=$TEMPDIR/prm-cmpref.$$
    /usr/bin/rm -f $tmpfile

    v_obj_script=$SQL_SCRIPTS/verify-object.sql
    v_schema_script=$SQL_SCRIPTS/verify-schema.sql
    if [ ! -f "$v_obj_script" -o ! -f "$v_schema_script" ] ; then
        echolog 'PRM schema validation scripts missing'
        prm_cleanupAndExit 2
    fi

    #
    # Check database objects
    #
    db_runDbScript "$v_obj_script report rep sunmc" $SYS_USER $SYS_PASSWD $tmpfile
    db_checkErr $tmpfile $LOGFILE
    if [ $? -ne 0 ] ; then
        echolog 'Error executing validation script $2' "$v_obj_script"
        /usr/bin/rm -f $tmpfile
        prm_cleanupAndExit 2
    fi

    /usr/bin/egrep -i "FATAL ERROR" $tmpfile >/dev/null
    if [ $? -eq 0 ] ; then
        echolog 'Found discrepancies in existing PRM schema objects'
        /usr/bin/rm -f $tmpfile
        return 1
    fi
    /usr/bin/rm -f $tmpfile

    #
    # Check schema
    #
    db_runDbScript "$v_schema_script report rep sunmc" $SYS_USER $SYS_PASSWD $tmpfile
    db_checkErr $tmpfile $LOGFILE
    if [ $? -ne 0 ] ; then
        echolog 'Error executing validation script $2' "$v_schema_script"
        /usr/bin/rm -f $tmpfile
        prm_cleanupAndExit 2
    fi

    /usr/bin/egrep -i "FATAL ERROR" $tmpfile >/dev/null
    if [ $? -eq 0 ] ; then
        echolog 'Found discrepancies in existing PRM schema'
        /usr/bin/rm -f $tmpfile
        return 1
    fi
    /usr/bin/rm -f $tmpfile

    return 0
}


prm_compareSynonyms () {
    orig_synonyms=$TEMPDIR/prm-validate-orig.$$
    tmpfile=$TEMPDIR/prm-validate-tmp.$$
    actual_synonyms=$TEMPDIR/prm-validate-act.$$
    /usr/bin/rm -f $orig_synonyms $tmpfile $actual_synonyms

    crt_synonyms_script=$SQL_SCRIPTS/create_sunmc_synonyms.sql
    if [ ! -f "$crt_synonyms_script" ] ; then
        echolog 'Missing script: $2' "$crt_synonyms_script"
        prm_cleanupAndExit 2
    fi

    # What the synonyms should look like
    /usr/bin/nawk '{ if ($0 !~ /^--.*/) { print $3 " " toupper($5); }}' $crt_synonyms_script \
	| /usr/bin/sed -n -e 's/;//' -e 's/\./ /p' \
	| /usr/bin/sort -k 1,1 >$orig_synonyms

    # What the synonyms actually are currently
    sql="SELECT RTRIM(LTRIM(synonym_name))||' '||RTRIM(LTRIM(table_owner))||' '||RTRIM(LTRIM(table_name)) FROM dba_synonyms WHERE owner = 'SUNMC' and table_owner = 'REPORT' ORDER BY synonym_name"
    db_runSql "$sql" $SYS_USER $SYS_PASSWD $tmpfile
    db_checkErr $tmpfile $LOGFILE
    if [ $? -ne 0 ] ; then
        echolog 'Error while checking current synonyms'
        /usr/bin/rm -f $orig_synonyms $tmpfile
        prm_cleanupAndExit 2
    fi
    /usr/bin/sed '/^$/d' $tmpfile > $actual_synonyms

    # Check for any differences between them
    /usr/bin/diff $orig_synonyms $actual_synonyms >/dev/null
    if [ $? -ne 0 ] ; then
        /usr/bin/rm -f $orig_synonyms $tmpfile $actual_synonyms
        return 1
    fi

    /usr/bin/rm -f $orig_synonyms $tmpfile $actual_synonyms
    return 0
}


#
# Validate an existing PRM database setup
#
# Inputs:
#   none
#
# Returns:
#   0 = Setup is valid
#   1 = Setup is invalid
#   
prm_validateSetup () {
    echolog ''
    echolog 'A previous PRM setup has been detected.'
    echologverbose 'Checking consistency of existing PRM setup...'

    # Sanity checks
    if [ -z "$PRM_USER" -o -z "$PRM_PASSWD" -o -z "$INSTALL_TYPE" ] ; then
        # Something is wrong with the setup
        echolog 'WARNING: unable to determine PRM database configuration parameters'
        echolog 'PRM setup may not be able to recover from this problem.'

        # NOTE: for now, just return invalid status and hope unsetup recovers
        # We really should have a last-resort recovery mode in db-unsetup that
        # will attempt to clean up the database based on known PRM user login/
        # passwords and all possible tablespace names, etc., so that we can
        # at least proceed with a new installation. Currently, if we ever get
        # here, it probably means it's impossible to recover.

        return 1
    fi


    #
    # Relink Oracle with partitioning option (required by PRM)
    #
    prm_relinkDb
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi


    # BUGFIX: because current SunMC setup scripts run the addons' setup first,
    # the subsequent core SunMC database setup will overwrite any PRM's custom
    # initfile. The published solution to this is to re-run es-setup with the
    # '-p PRM' option, which eventually calls this script.
    # Here at this point in this script, we have detected an existing PRM
    # installation which potentially has the wrong initfile -- hence, we will
    # re-apply the new initfile here, and also re-create SunMC synonyms for
    # PRM objects further down below.
    prm_applyNewInitfile
    if [ $? -ne 0 ] ; then
        return 1
    fi

    # Start database
    db_start

    # Make sure all tablespaces still exist
    prm_checkTablespaces
    if [ $? -ne 0 ] ; then
        return 1
    fi
  
    # Recreate SunMC synonyms
    prm_recreateSynonyms
    if [ $? -ne 0 ] ; then
        return 2
    fi

    # Fix any potential problems with invalid objects. At the end, if we still
    # have a discrepancy, then we know it's something seriously wrong that
    # cannot be fixed by recompiling invalid objects.
    prm_recompileInvalidObjects
    if [ $? -ne 0 ] ; then
        return 2
    fi
    # Discard reference schema user, if any
    prm_dropRefUser

    # Create reference schema as comparison to check against current schema
    prm_createRefSchema
    if [ $? -ne 0 ] ; then
        echolog 'Unable to create reference schema'
        prm_cleanupAndExit 2
    fi

    prm_compareRefSchema
    if [ $? -ne 0 ] ; then
        prm_dropRefUser
        db_stop
        return 1
    fi

    prm_compareSynonyms
    if [ $? -ne 0 ] ; then
        prm_dropRefUser
        db_stop
        return 1
    fi

    # Done with reference schema, discard it
    prm_dropRefUser

    # Stop database
    db_stop

    return 0
}


#
# Prompt user for PRM installation type (small/medium/large)
#
prm_getInstallationType () {
    echolog 'Please choose one of the following PRM installation sizes:'
    echolog ''
    echolog '[1] Small - supports up to 100 agents monitoring 500 properties each.'
    echolog '    This installation requires 4GB of free disk space.'
    echolog ''
    echolog '[2] Medium - supports up to 400 agents monitoring 600 properties each.'
    echolog '    This installation requires 12GB of free disk space.'
    echolog ''
    echolog '[3] Large - supports up to 750 agents monitoring 650 properties each.'
    echolog '    This installation requires 22GB of free disk space.'
    echolog ''

    while [ 1 ] ; do
        /usr/bin/gettext 'Please enter your choice: [1|2|3] '

        read inst_type
        if [ $? -ne 0 ] ; then
            return 1
        fi

        case $inst_type in
          1)
            prm_cfgInstallationType small
            if [ $? -ne 0 ] ; then
                return 2
            else
                return 0
            fi
            ;;

          2)
            prm_cfgInstallationType medium
            if [ $? -ne 0 ] ; then
                return 2
            else
                return 0
            fi
            ;;

          3)
            prm_cfgInstallationType large
            if [ $? -ne 0 ] ; then
                return 2
            else
                return 0
            fi
            ;;

          *)
            echolog "Please enter one of '1', '2', or '3'."
            continue
            ;;
        esac
    done

    # Should never reach here
    return 3
}


#
# Prompt user for the PRM report directories and ensure that there is 
# sufficient disk space for PRM in the directories
#
# Inputs:
#   none
#
# Outputs:
#   prm_dir_l__DIR=<directory>
#   prm_dir_m__DIR=<directory>
#   prm_dir_i__DIR=<directory>
#
#   where:
#      PRM_DIRECTORIES=prm_dir_l prm_dir_m prm_dir_i 
#
# Returns:
#   0 = Success
#   1 = Error
#
prm_getTablespaceDirectories () {

    echolog ''
    echolog 'PRM setup now needs you to specify the directories where PRM data '
    echolog 'files will be kept.'
    echolog ''

    #
    # get list of PRM directories
    #
    dirs=`prm_getCfg directories` 
    if [ $? -ne 0 ]; then
        return 1
    fi

    #
    # foreach directory
    #
    prmDirSpecs=""
    for dirspec in $dirs
    do
        #
        # Get required available space 
        #
        size=`prm_getCfg ${dirspec}:size`
        if [ $? -ne 0 ]; then
            return 1
        fi

        #
        # Prompt user for directory with sufficient space 
        #
        prm_askDir $size "$prmDirSpecs"
        if [ $? -ne 0 ]; then
            return 1
        fi
        prmDir=$dir

        #
        # set the PRM directory
        #
        eval ${dirspec}__DIR=$prmDir/SUNWsymon/PRM/db

        # Strip leading space from $size, otherwise we get embedded spaces in
        # $prmDirSpecs which will cause parsing code to break.
        size=`/bin/echo $size`

        prmDirSpecs="$prmDirSpecs $prmDir:$size"
    done

    return 0
}

#
# Prompt user for directory with sufficient free disk space.
#
# Ensure that specified directory is located on a local disk 
# partition that does not also include /opt or /tmp.
#
# Also account for multiple PRM directories on the same 
# partition. 
#
# Inputs:
#   $1 = required size
#   $2 = list of current PRM directory-req'd size pairs 
#
# Returns:
#   0 = Success
#   1 = Error
#
# Sets $dir to the chosen directory, if success.
#
prm_askDir () {

    reqspace=$1
    dirspecs="$2"
    
    dir=""
    while [ "$dir" = "" ] ; do
        echolog 'Please enter a directory with at least $2 MB of free disk space: [q to abort]' "$reqspace"
        read dir
        if [ "$dir" = "" ] ; then 
            continue
        fi

        if [ "$dir" = "q" ] ; then
            return 1
        fi

        prm_validateDir $dir $reqspace "$dirspecs"
        if [ $? -ne 0 ] ; then
            dir=""
            continue
        fi
    done
    return 0
}

# prm_validateDir $dir $reqdSpace $prmDirSpecs
#
# Ensure the following: 
#
# - directory exists
# - directory doesn't already contain previous PRM files & directories
# - directory is not /
# - directory is on a local UFS partition
# - directory in not under /opt
# - directory starts with /
# - directory is writeable
# - directory has sufficient free space
#
# Returns 0 if directory is valid, 1 otherwise.
prm_validateDir () {
    dir=$1
    reqdSpace=$2
    prmDirSpecs="$3"

    if [ ! -d "$dir" ]; then 
        echolog 'Directory $2 not found.' $dir
        return 1
    fi

    if [ -d "$dir/SUNWsymon/PRM/db" ] ; then
        echolog 'The directory $2 is not empty.' "$dir"
        echolog 'Please select a different directory or manually remove the $2 subdirectory and its contents.' "$dir"
        return 1
    fi

    if [ "$dir" = '/' ]; then
        echolog 'The directory cannot be /.' $dir
        return 1
    fi

    fstype=`/usr/bin/df -n $dir | /usr/bin/awk '{print $3}'`
    if [ $? -ne 0 -o "$fstype" != "ufs" ]; then
        echolog 'The directory must be on a local filesystem.' 
        return 1
    fi 

    mnt=`/usr/bin/df -l $dir | /usr/bin/awk '{print $1}'`
    if [ $? -ne 0 -o "$mnt" = "/opt" ]; then
        echolog 'The directory cannot be on the /opt filesystem.' 
        return 1
    fi

    /bin/echo $dir | /usr/bin/egrep '^/' >/dev/null
    if [ $? -ne 0 ] ; then
        echolog 'The directory path must be absolute (must start with a /)'
        return 1
    fi

    /usr/bin/touch $dir/test.$$
    if [ $? -ne 0 ]; then
        /usr/bin/rm -f $dir/test.$$
        echolog 'The directory is not writeable.' 
        return 1
    fi
    /usr/bin/rm -f $dir/test.$$

    prm_checkFreeSpace $dir $reqdSpace "$prmDirSpecs"
    rcode=$?
    if [ $rcode -eq 1 ]; then
        echolog 'The directory does not have sufficient free space.' 
        return 1
    elif [ $rcode -eq 2 ] ; then
        echolog 'Unable to check disk space'
        return 2
    fi

    return 0
}

#
# Check directory for available space
#
# Inputs:
#   $1 = directory
#   $2 = required space (in MB)
#   $3 = current PRM directory specs (e.g. dir:size dir:size ...)
#
# Returns:
#   0 - Sufficient space
#   1 - Insufficient space
#
prm_checkFreeSpace () {
    dir=$1
    reqd=$2
    specs="$3"

    # Convert MB to KB for ease of comparison with df output
    reqd=`/usr/bin/expr "$reqd" \* 1024`
    if [ $? -ne 0 ] ; then
        echolog 'Failed to check available disk space'
        return 1
    fi

    dfinfo=`/usr/bin/df -k $dir | /usr/bin/tail -1 | /usr/bin/awk '{print $6, $4}'`
    if [ $? -ne 0 ]; then
        echolog 'Unable to determine available space on $2.' "$1"
        return 1
    fi

    mount=`/bin/echo $dfinfo | /usr/bin/awk '{print $1}'`
    avail=`/bin/echo $dfinfo | /usr/bin/awk '{print $2}'`

    #
    # Account for other PRM directories
    #
    for spec in $specs
    do
        d=`/bin/echo $spec | /usr/bin/cut -f1 -d:` 
        m=`/usr/bin/df -k $d | /usr/bin/tail -1 | /usr/bin/awk '{print $6}'`
        if [ $? -ne 0 ]; then
            return 2
        fi
        if [ "$mount" = "$m" ]; then
            s=`/bin/echo $spec | /usr/bin/cut -f2 -d: | /usr/bin/xargs /usr/bin/expr 1024 \*`
            if [ $? -ne 0 ] ; then
                echolog 'Error while calculating disk space required'
                return 2
            fi
            avail=`/usr/bin/expr $avail - $s`
            if [ $? -ne 0 ] ; then
                echolog 'Error while calculating disk space required'
                return 2
            fi
        fi
    done
    
    if [ "$avail" -gt "$reqd" ]; then
        return 0
    fi
    return 1
} 


# Relink Oracle with partitioning option
# NOTE: should make sure database is not running when calling this.
prm_relinkDb () {
    echologverbose 'Relinking database with partitioning option'

    db_linkOracle "part"
    if [ $? -ne 0 ] ; then
        echolog 'Failed to link DB partitioning option required by PRM.'
        return 1
    fi

    return 0
}

# prm_backupInitfile $initfile $bakdir
prm_backupInitfile () {
    initfile=$1; bakdir=$2

    echologverbose 'Backing up original initfile'

    if [ ! -d "$bakdir" ] ; then
        /usr/bin/mkdir -p $bakdir
    fi

    # Sanity check: if there's already a copy of the initfile in the backup
    # directory, it probably means the current initfile is a PRM initfile, not
    # a SunMC initfile; in which case we do NOT want to replace the copy in the
    # backup directory with the current initfile.
    if [ -f "$bakdir/$initfile" ] ; then
        echologverbose 'Initfile already in backup directory, will NOT replace it'
        return 0
    fi

    # Backup original init file so that unsetup can restore original settings
    /usr/bin/cp -p $ORA_INITFILE $bakdir
    if [ $? -ne 0 ] ; then
        echolog 'Unable to backup database init file'
        return 1
    fi

    return 0
}

# prm_applyNewInitfile $initfile
prm_applyNewInitfile () {
    initfile=$1

    echologverbose 'Applying replacement database initfile'

    destdir=`/usr/bin/dirname $ORA_INITFILE`
    if [ $? -ne 0 ] ; then
        return 2
    fi

    # Replace file
    new_initfile=$SQL_SCRIPTS/`prm_getCfg initfile`
    if [ ! -f "$new_initfile" ] ; then
        echolog 'Unable to find replacement database initfile: $2' "$new_initfile"
        return 1
    fi

    # Instantiate variable parameters in template initfile and copy result
    # into real initfile.
    hostname=`/usr/bin/hostname`
    /usr/bin/cat $new_initfile | /usr/bin/sed \
	-e "s@__HOSTNAME__@${hostname}@g" \
	-e "s@__VAR_ORA_DATADIR__@${VAR_ORA_DATADIR}@g" \
	-e "s@__ORACLE_HOME__@${ORACLE_HOME}@g" \
	-e "s@__ORACLE_SID__@${ORACLE_SID}@g" \
	>$ORA_INITFILE

    return $?
}

# Adjust initSunMC.ora to accomodate for larger PRM requirements.
# NOTE: this should be called only when database is not running.
prm_adjustInitfile () {
    bakdir=$PRM_UNINSTALLDIR

    if [ ! -f "$ORA_INITFILE" ] ; then
        echolog 'Unable to locate database initialization file'
        return 1
    fi

    prm_backupInitfile $ORA_INITFILE $bakdir
    if [ $? -ne 0 ] ; then
        return 1
    fi

    prm_applyNewInitfile $ORA_INITFILE
    if [ $? -ne 0 ] ; then
        return 1
    fi

    return 0
}


#
# Create the PRM tablespaces according to the configuration specified
# in the prm-report-tablespaces.cfg file and the corresponding PRM 
# directories specified by the user as set to the following variables:
#
# - prm_dir_l__DIR=<directory> 
# - prm_dir_m__DIR=<directory>
# - prm_dir_i__DIR=<directory> 
#
# Inputs:
#   none
#
# Returns:
#   0 = Success
#   1 = Error
#
prm_createTablespaces () {
    echologverbose 'Creating tablespaces'

    #
    # get the tablespace directories
    #
    tsdirs=`prm_getCfg directories`
    if [ $? -ne 0 ]; then
        echolog "Unable to get tablespace directories."
        return 1
    fi

    for tsdir in $tsdirs
    do
        #
        # get the tablespaces for the directory
        #
        tablespaces=`prm_getCfg ${tsdir}:tablespaces`
        if [ $? -ne 0 ]; then
            echolog "Unable to get directory tablespaces."
            return 1
        fi
        for ts in $tablespaces
        do
            eval prm_createTablespace $ts \$${tsdir}__DIR
            if [ $? -ne 0 ]; then
                echolog "Unable to create tablespace."
                return 1
            fi
        done
    done
    return 0
}

#
# Create a tablespace
#
# Inputs:
#   $1 = tablespace name                   (e.g. report_data_l)
#   $2 = fullpath for tablespace directory (e.g. /PRM/...)
#
# Returns:
#   0 - Success
#   1 - Error
#
prm_createTablespace () {
    ts=$1
    dir=$2

    # Create tablespace directories if they aren't already there
    if [ ! -d "$dir" ] ; then
        /usr/bin/mkdir -p $dir
        if [ $? -ne 0 ] ; then
            echolog 'Unable to create directory: $2' "$dir"
            return 1
        fi
    fi

    # Make sure tablespace directories are owned by smcorau, and that it has
    # the correct permissions
    /usr/bin/chown smcorau:smcorag $dir && /usr/bin/chmod 775 $dir
    if [ $? -ne 0 ] ; then
        echolog 'Failed to set permissions on $2' "$dir"
        return 1
    fi

    filespecs=`prm_getCfg ${ts}:files`
    if [ $? -ne 0 ]; then
        echolog "Unable to get directory tablespaces."
        return 1
    fi

    storeparms=`prm_getCfg ${ts}:store_params`
    if [ $? -ne 0 ]; then
        echolog "Unable to get storage parameters."
        return 1
    fi

    extentparms=`prm_getCfg ${ts}:extent_params`
    if [ $? -ne 0 ]; then
        echolog "Unable to get extent parameters."
        return 1
    fi

    # Prepend full path to file specs
    specs=""
    for file in $filespecs ; do
        filepath="$dir/$file"
        if [ -f "$filepath" ] ; then
            echolog 'The database device file $2 already exists.' "$filepath"
            return 1
        fi
        specs="$specs $filepath"
    done

    db_createTblsp $ts $SYS_USER $SYS_PASSWD "$storeparms" "$extentparms" $LOGFILE $specs
    if [ $? -ne 0 ] ; then
        echolog 'Failed to create tablespace $2.' "$ts"
        return 1
    fi

    return 0
}

prm_moveRbs () {
    rbsts_revscript="$PRM_REVSQL_DIR/revRbsTblsp.sql"
    rbs_revscript="$PRM_REVSQL_DIR/revRbsSegment.sql"
    srcdir=$ORA_DATADIR

    # Determine destination dir for RBS tablespace
    destdir_key=`prm_getCfg rbs:directory`
    if [ $? -ne 0 ] ; then
        echolog 'Unable to determine target directory for RBS tablespace'
        return 1
    fi
    eval destdir=\$${destdir_key}__DIR
    if [ ! -d "$destdir" ] ; then
        echolog 'Target directory for RBS does not exist: $2' "$destdir"
        return 1
    fi

    # Sanity checks
    if [ -h "$srcdir/rbs01.dbf" -a -f "$destdir/rbs01.dbf" ] ; then
        echologverbose 'RBS tablespace already moved, skipping'
        return 0
    fi
    if [ ! -f "$srcdir/rbs01.dbf" ] ; then
        echolog 'RBS tablespace not in expected location $2: aborting' "$srcdir"
        return 2
    fi

    # Reverse-engineer RBS tablespace and RBS segments definitions so that we
    # can restore it at uninstall time.

    if [ ! -f "$rbsts_revscript" ] ; then
        db_revSqlTblsp "RBS" "$rbsts_revscript" $SYS_USER $SYS_PASSWD $LOGFILE
        if [ $? -ne 0 ] ; then
            echolog 'Error while running extraction for tablespace: $2' "RBS"
            return 1
        fi
    fi

    if [ ! -f "$rbs_revscript" ] ; then
        db_revRbsSeg "RBS" "$rbs_revscript" $SYS_USER $SYS_PASSWD $LOGFILE
        if [ $? -ne 0 ] ; then
            echolog 'Error while running RBS extraction for tablespace $2' "RBS"
            return 1
        fi
    fi

    # Move RBS tablespace to location with more space
    # Shutdown Oracle before attempting to move any files
    db_stop

    if [ -f $destdir/rbs01.dbf ] ; then
        echolog 'Cannot not move RBS: $2 already exists' "$destdir/rbs01.dbf"
        return 1
    fi
    /usr/bin/cp -p $srcdir/rbs01.dbf $destdir/rbs01.dbf
    if [ $? -ne 0 ] ; then
        echolog 'Error while relocating RBS tablespace'
        return 1
    fi

    # Rename original file to reserve space in /var/opt
    /usr/bin/mv $srcdir/rbs01.dbf $srcdir/rbs01.dbx

    # Rename the original ORACLE file to new ORACLE file
    db_renameDbFile "$VAR_ORA_DATADIR/rbs01.dbf" "$destdir/rbs01.dbf"
    if [ $? -ne 0 ] ; then
        echolog 'Failed to rename database file $2 to $3' "$VAR_ORA_DATADIR/rbs01.dbf" "$destdir/rbs01.dbf"
        return 1
    fi
    db_stop

    # Restart DB to ensure that new RBS location works properly
    db_start

    return 0
}

prm_moveTempTs () {
    temp_revscript="$PRM_REVSQL_DIR/revTempTblsp.sql"
    srcdir=$ORA_DATADIR

    # Determine destination dir for RBS tablespace
    destdir_key=`prm_getCfg tmp:directory`
    if [ $? -ne 0 ] ; then
        echolog 'Unable to determine target directory for RBS tablespace'
        return 1
    fi
    eval destdir=\$${destdir_key}__DIR
    if [ ! -d "$destdir" ] ; then
        echolog 'Target directory for TEMP does not exist: $2' "$destdir"
        return 1
    fi

    # Sanity checks
    if [ -h "$srcdir/temp01.dbf" -a -f "$destdir/temp01.dbf" ] ; then
        echologverbose 'TEMP tablespace already relocated, skipping'
        return 0
    fi
    if [ ! -f "$srcdir/temp01.dbf" ] ; then
        echolog 'TEMP tablespace not in expected location $2: aborting' "$srcdir"
        return 2
    fi

    # Reverse-engineer TEMP tablespace definition so that we can restore it
    # at uninstall time.

    if [ ! -f "$temp_revscript" ] ; then
        db_revSqlTblsp "TEMP" "$temp_revscript" $SYS_USER $SYS_PASSWD $LOGFILE
        if [ $? -ne 0 ] ; then
            echolog 'Error while running extraction for tablespace: $2' "TEMP"
            return 1
        fi
    fi

    # Move TEMP tablespace to location with more space
    # Shutdown database before attempting to move any files
    db_stop

    if [ -f $destdir/temp01.dbf ] ; then
        echolog 'Cannot move TEMP: $2 already exists' "$destdir/temp01.dbf"
        return 1
    fi
    /usr/bin/cp -p $srcdir/temp01.dbf $destdir/temp01.dbf
    if [ $? -ne 0 ] ; then
        echolog 'Error while relocating TEMP'
        return 1
    fi

    # Rename original file to reserve space
    /usr/bin/mv $srcdir/temp01.dbf $srcdir/temp01.dbx

    # Rename the original ORACLE file to new ORACLE file
    db_renameDbFile "$VAR_ORA_DATADIR/temp01.dbf" "$destdir/temp01.dbf"
    if [ $? -ne 0 ] ; then
        echolog 'Failed to rename database file $2 to $3' "$VAR_ORA_DATADIR/temp01.dbf" "$destdir/temp01.dbf"
        return 1
    fi
    db_stop

    # Restart database to ensure that everything works with new location
    db_start

    return 0
}

prm_moveVarTablespaces () {
    PRM_REVSQL_DIR=$PRM_UNINSTALLDIR

    # Create uninstall directory if it doesn't already exist
    if [ ! -d "$PRM_REVSQL_DIR" ] ; then
        /usr/bin/mkdir -p $PRM_REVSQL_DIR
        /usr/bin/chmod 775 $PRM_REVSQL_DIR
    fi

    echologverbose 'Moving RBS tablespace'
    prm_moveRbs
    if [ $? -ne 0 ] ; then
        return 1
    fi

    echologverbose 'Moving TEMP tablespace'
    prm_moveTempTs
    if [ $? -ne 0 ] ; then
        return 1
    fi

    return 0
}

# prm_runStep $stepline $whichspec
# - parses $stepline from steps.txt and runs the command if a user is defined
#   for the column indicated by $whichspec
# - $whichspec can be:
#	2	for schema creation
#	3	for schema validation
prm_runStep () {
    stepline="$1"
    whichspec=$2

    tmpfile=$TEMPDIR/prm-steps.$$
    /usr/bin/rm -f $tmpfile

    scriptname=`/bin/echo $stepline | /usr/bin/cut -d: -f1 | /usr/bin/sed -e "s/__SUNMC_OWNER__/${SUNMC_SCHEMA_USER}/" `
    userspec=`/bin/echo $stepline   | /usr/bin/cut -d: -f$whichspec`

    if [ "$scriptname" = "" ] ; then
        echolog 'Syntax error in $2: $3' "$STEPS_FILE" "$line"
        return 1
    fi

    # Ignore step if there is no user spec for schema creation
    if [ ! -z "$userspec" ] ; then

        # Parse user spec
        case $userspec in
          __SUNMC_OWNER__) login=$SUNMC_SCHEMA_USER; passwd=$SUNMC_SCHEMA_PASSWD ;;
          rep)    login=rep;         passwd=rep           ;;
          report) login=$PRM_USER;   passwd=$PRM_PASSWD   ;;
          sunmc)  login=$SUNMC_USER; passwd=$SUNMC_PASSWD ;;
          sys)    login=$SYS_USER;   passwd=$SYS_PASSWD   ;;
          *)      echolog 'Illegal userspec $2 in $3' "$userspec" "$STEPS_FILE"
                  return 1
                  ;;
        esac

        # execute script
        script=$BASEDIR/${scriptname}
        echologverbose 'Running script: $2' "$script"
        db_runDbScript "$script" $login $passwd $tmpfile
        db_checkErr $tmpfile $LOGFILE
        if [ $? -ne 0 ] ; then
            echolog 'Error while executing script: $2' "$script"
            /usr/bin/rm -f $tmpfile
            return 1
        fi
        /usr/bin/rm -f $tmpfile
    else
        echologverbose 'No userspec, skipping script: $2' "$scriptname"
    fi

    return 0
}

prm_createSchema () {
    #
    # Create schema using the scripts specified in the steps.txt file
    #

    echologverbose 'Creating PRM schema...'
    if [ ! -f "$STEPS_FILE" ] ; then
        echolog 'Unable to find SQL steps file: $2' "$STEPS_FILE"
        return 1
    fi

    # Parse steps file, ignoring comment lines (starts with "--") and
    # blank lines.
    /usr/bin/egrep -v '^--|^$' $STEPS_FILE | while read line ; do
        prm_runStep "$line" 2
        if [ $? -ne 0 ] ; then
            return 1
        fi
    done
    if [ $? -ne 0 ] ; then
        echolog 'Error occurred while creating PRM database schema'
        return 1
    fi

    # Recompile any invalid DB objects
    prm_recompileInvalidObjects
    return $?
}


prm_adjustRedoLogs () {
    echologverbose 'Adjusting database redo log sizes'

    newsize=`prm_getCfg redo:size`
    if [ -z "$newsize" ] ; then
        echolog 'Unable to determine redo log size'
        return 1
    fi

    db_replaceRedoLogs $newsize
    if [ $? -ne 0 ] ; then
        echolog 'Failed to resize database redo logs'
        return 1
    fi

    return 0
}


prm_createCronJobs () {
    tmpfile=$TEMPDIR/prm-setup-cron.$$
    /usr/bin/rm -f $tmpfile

    echologverbose 'Creating PRM cron jobs...'

    /usr/bin/crontab -l | /usr/bin/egrep -v "schanalyze.sh" | /usr/bin/sed '/^$/d' > $tmpfile
    /usr/bin/cat<<-EOF>> $tmpfile
0 0 * * * $BASEDIR/addons/PRM/lib/sbin/schanalyze.sh -s report -e 30 -x $BASEDIR/addons/PRM/lib/sbin/xparstable.txt    1>&2 2>/dev/null
EOF
    /usr/bin/crontab $tmpfile
    rcode=$?

    /usr/bin/rm -f $tmpfile
    return $rcode
}

# Returns:
#	0 - no existing PRM setup found
#	1 - existing, valid PRM setup found
#	2 - existing PRM setup found, but validation failed
#	3 - error occurred
prm_checkCurrentSetup () {
    prm_checkPrereq
    if [ $? -ne 0 ] ; then
        return 3
    fi

    prm_checkSetup
    if [ $? -eq 1 ] ; then
        # Existing setup found
        prm_validateSetup
        if [ $? -ne 0 ] ; then
            return 2
        else
            return 1
        fi
    fi

    return 0
}

prm_doActualSetup () {

    echolog ''
    echolog 'Setting up PRM database environment, please wait ...'
    echolog ''

    #
    # Create PRM database configuration file
    #
    prm_createCfg
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    #
    # Relink Oracle with partitioning option (required by PRM)
    #
    prm_relinkDb
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    #
    # Adjust initSunMC.ora parameters
    #
    prm_adjustInitfile
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    # Startup database
    db_start

    #
    # Create PRM tablespaces in PRM directories
    #
    prm_createTablespaces
    if [ $? -ne 0 ]; then
        prm_cleanupAndExit 1
    fi

    #
    # Move temp and rbs from /var to the PRM directories
    #
    prm_moveVarTablespaces
    if [ $? -ne 0 ]; then
        prm_cleanupAndExit 1
    fi

    #
    # Create PRM schema
    #
    prm_createSchema
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    # Adjust redo log sizes
    prm_adjustRedoLogs
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    #
    # Create backup directory and analyze crontab entry
    #
    prm_createCronJobs
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    fi

    # Final cleanup
    db_stop

    echolog 'PRM database successfully setup.'

    return 0
}

# prm_removeCurSetup $rcode
# $rcode is the (non-zero) return code from checkCurrentSetup
prm_removeCurSetup () {
    rcode="$1"

    echolog ''
    if [ "$rcode" -eq 1 ] ; then
        echolog "The PRM setup appears to be valid."
    else
        echolog "The PRM setup appears to be invalid."
    fi

    echolog "To remove the existing PRM data and restore the PRM factory"
    echolog "settings, continue with the setup." 
    echolog ""
    echolog "To preserve the PRM data, please perform the following steps:"
    echolog ""
    echolog "  1. Exit current PRM setup session."
    echolog "  2. Backup Sun MC data via es-backup."
    echolog "  3. Run PRM setup to remove PRM data and restore factory settings."
    echolog "  4. Restore the backup via es-restore."
    echolog ""

    #
    # Ask user whether they want to:
    #
    #    - keep old PRM data and exit 
    #    - reset to "factory default" via unsetup and setup 
    #
    question="Do you want to keep the existing PRM data and exit? [y/n] " 
    gen_askYesNo "$question"
    if [ $? -eq 1 ]; then
        echolog "Exiting PRM setup."
        prm_cleanupAndExit 0
    fi

    #
    # Double-check to confirm that it is ok to remove PRM data   
    #
    echolog "Proceeding to remove PRM data to reset PRM to factory defaults."
    question="Please confirm that you want to remove the PRM data [y/n] " 
    gen_askYesNo "$question"
    if [ $? -eq 0 ]; then
        echolog "Exiting PRM setup."
        prm_cleanupAndExit 0
    fi

    #
    # Remove PRM data via db-unsetup
    #
    $PRM_BASEDIR/db-unsetup.sh
    if [ $? -ne 0 ] ; then
        echolog 'Unable to clean up previous PRM setup'
        prm_cleanupAndExit 2
    fi

    # Allow user to choose another installation type from previous one
    if [ "$INSTALL_TYPE" != "dev" ] ; then
        INSTALL_TYPE=""
    fi
}


###########################################################################
#
# Main entry point
#
###########################################################################

#
# setup environment
#

VERBOSE=OFF; export VERBOSE
KEEP_TABLESPACES=0
BASEDIR=`/usr/bin/pkgparam SUNWescom BASEDIR`
if [ -z "${BASEDIR}" ]; then
    /usr/bin/gettext "SunMC is not installed."
    exit 1
fi

. ${BASEDIR}/SUNWsymon/sbin/es-common.sh
set_basedir
set_db_env
setup_textdomain SUNW_SUNMC_PRM

# (Use logfile passed from calling program if any; otherwise log to
#  install/db-setup.$timestamp)
check_logfile prm-db-setup

. ${BASEDIR}/addons/PRM/sbin/prm-db-common.sh
prm_setupEnvironment

STEPS_FILE=$SQL_SCRIPTS/steps.txt
INSTALL_TYPE=""


#
# Process command-line options
#
CMD=normal
while getopts cdefkm:v OPT ; do
    case $OPT in
      c)
        # Check PRM setup prerequisites and validate any existing PRM setup
        # Exits with status codes:
        #	0 - no existing PRM setup was found
        #	1 - an existing, valid PRM setup was found
        #	2 - an existing PRM setup was found, but validation failed
        #	3 - an error occurred while performing this operation
        CMD=checksetup
        ;;

      d)
        # Check the given directory argument for validity
        # For this option, db-setup.sh should be invoked as follows:
        #	db-setup.sh -d <directory> <size required> <list of existing dir:size pairs>
        # Exit status codes:
        #	0 - directory is valid
        #	1 - directory is invalid
        CMD=checkdir
        ;;

      e)
        # Do everything else setup needs to do, given the PRM installation
        # type and the list of mappings between PRM directory keys and actual
        # directory pathnames.
        # This should be invoked this way:
        #	db-setup.sh -m <small|medium|large> -e prm_dir_l:<dir1> prm_dir_m:<dir2> prm_dir_s:<dir3>
        # Exit status codes:
        #	0        - PRM setup successful
        #	non-zero - error occurred
        CMD=dosetup
        ;;

      f)
        # Prints to STDOUT the PRM tablespace configuration File to be used
        # for the PRM installation type chosen by the -m option. Note that
        # the -m option is mandatory.
        if [ -z "$INSTALL_TYPE" -o ! -f "$PRM_TABLESPACE_CFGFILE" ] ; then
            echolog 'A valid -m option must be provided before -f.'
            exit 3
        fi
        /bin/echo $PRM_TABLESPACE_CFGFILE
        exit 0
        ;;

      k)
        # This is only ever used by es-restore.
        # All we need to do here is to recreate the PRM schema (unsetup will
        # have dropped only the schema and nothing else at this point)
        CMD=recreate_schema
        ;;

      m)
        # specify PRM installation type (small/medium/large)
        prm_cfgInstallationType $OPTARG
        if [ $? -ne 0 ] ; then
            exit 2
        fi
        ;;

      v)
        VERBOSE=1
        ;;

      *)
        exit 3
        ;;
    esac
done


#
# Perform commands based on options
# (We have to do it here because it's difficult to have multi-parameter options
#  inside the getopts loop)
case $CMD in

  checksetup)
    prm_checkCurrentSetup
    exit $?
    ;;

  checkdir)
    dir=$1;  shift
    size=$1; shift
    specs="$*"

    prm_validateDir $dir $size "$specs"
    exit $?
    ;;

  dosetup)
    # Parse directory mappings into global vars
    for mapping in $* ; do
        key=`/usr/bin/echo $mapping | /usr/bin/cut -f1 -d:`
        val=`/ust/bin/echo $mapping | /usr/bin/cut -f2 -d:`

        eval ${key}__DIR=$val
    done

    if [ -z "$INSTALL_TYPE" ] ; then
        echolog 'The PRM installation type needs to be specified with the -m option.'
        exit 1
    fi

    prm_doActualSetup
    exit $?
    ;;

  normal)
    # Normal interactive setup procedure

    prm_checkCurrentSetup
    rcode=$?
    if [ $rcode -ne 0 ] ; then
        prm_removeCurSetup $rcode
    fi

    # Prompt user for PRM size selection (small/medium/large)
    if [ -z "$INSTALL_TYPE" ] ; then
        prm_getInstallationType
        if [ $? -ne 0 ] ; then
            prm_cleanupAndExit 1
        fi
    fi

    # Prompt user for PRM directories
    prm_getTablespaceDirectories
    if [ $? -ne 0 ]; then
        prm_cleanupAndExit 1
    fi

    # Perform actual setup
    prm_doActualSetup
    exit $?
    ;;

  recreate_schema)
    prm_createSchema
    if [ $? -ne 0 ] ; then
        prm_cleanupAndExit 1
    else
        exit 0
    fi
    ;;

  *)
    echolog 'Internal error: unknown operating mode'
    exit 2
    ;;
esac

# Should not reach here.
exit 3

