/**********************************************************************/
/*                                                                    */
/* Copyright (c) 2001 by Sun Microsystems, Inc.                       */
/* All rights reserved.                                               */
/*                                                                    */
/**********************************************************************/

//
// File: Sample3.java
//
// IMPORTANT NOTE: To execute this program successfully, you must have 
// available an MTP Telnet3270 end-system on which the ACCT transaction
// is available.
//
// This program searches for ACCT account numbers for the given Surname on the
// given host. It demonstrates navigation through a series of emulator screens.
// This program also requires the class Sample3Waiter.
//
// This program takes the parameters:
//   The host name	 e.g. crippen
//   The port number	 e.g. 9993
//   The search string	 e.g. Greenfield 
//     NOTE: The search string has a maximum length of 12 characters.
//
// e.g. java Sample3 crippen 9993 Greenfield
//
// This program also demonstrates two techniques for interrogating the 
// KixEmulator screen:
//
// Technique 1 - Requesting a field.
// This technique asks the KixEmulator to return the field at the specified
// point on the KixEmulator. This is a KixField which can be interrogated with
// a variety of methods, explained further in the JavaDoc API of KixField.
//
// Technique 2 - Requesting the display buffer.
// Requesting the display buffer from the KixEmulator results in a character
// array of the screen. This can be manipulated in the usual manner.
//

import unikix.clientapis.emulator.KixEmulator;
import unikix.clientapis.emulator.KixField;
import java.util.Vector;

public class Sample3
{
    public static void main(String[] args)
    {
	//
	// If the number of supplied parameters is not 3 or the 3rd parameter
	// is greater that 12 characters in length then display the usage
	// message.
	//
	if ((args.length != 3) || (args[2].length() > 12))
	{
	    System.out.println("Usage: java Sample3 <host> <port> <searchstring>");
	    System.out.println("  Where the search string has a max length of 12 characters");
	    System.exit(0);
	}

	System.out.println("------------------------------------------------");

	//
	// Run the method to obtain the account numbers and store the results
	// in a vector.
	//
	Vector v = getAccountNumbers(args);

	//
	// Did we get any results (is the vector empty) ?
	//
	if (v.isEmpty())
	{
	    System.out.println("Sorry, no matches for " + args[2]);
	}
	else
	{
	    //
	    // Display the results on the screen.
	    //
	    for ( int i=0;i<v.size();i++ )
	    {
		System.out.println(" Account number: "+v.elementAt(i));
	    }
	}

	System.out.println("------------------------------------------------");
	System.exit(0);
    }

    //
    // This method returns the account numbers (as a vector) for the
    // specified searchName.
    //
    // It achieves this by logging on to the specified host,
    // then navigates a series of screens, enters the given searchName
    // into the emulator and returns the account numbers (if any) for that
    // searchName.
    //
    // This method assumes that:
    //   args[0] is the host
    //   args[1] is the port (NOTE: This should be numeric only)
    //   args[2] is the searchName
    //
    private static Vector getAccountNumbers(String[] args)
    {
	//
	// Setup the KixEmulator and returnVector.
	//
	KixEmulator ke = new KixEmulator();
	Vector returnVector = new Vector();

	//
	// Set the KixEmulator host and port settings.
	//
	ke.setTN3270Host(args[0]);
	ke.setTN3270Port(Integer.parseInt(args[1]));
	String searchName = args[2];

	//
	// Connect the KixEmulator
	//
	ke.connect();
	ke.waitUntilConnected();
	ke.waitUntilKeyboardUnlocked();

	//
	// The KixEmulator should now be at the "Unikix welcome screen".
	//
	ke.pressClear();
	ke.waitUntilKeyboardUnlocked();

	//
	// The KixEmulator should now be on a blank screen.
	//
	ke.typeString("ACCT");
	ke.pressEnter();
	ke.waitUntilKeyboardUnlocked();

	//
	// The KixEmulator should now be on the ACCT screen.
	//
	ke.typeString(searchName);
	ke.pressEnter();
	ke.waitUntilKeyboardUnlocked();

	//
	// Get the account numbers.
	//
	while(true)
	{
	    for (int i=0; i<6; i++)
	    {
		//
		// Get the appropriate field from the emulator.
		//
		KixField kf = ke.findField(ke.offsetFromRowColumn(17+i,2));

		//
		// Try to get a substring of the first 5 characters of the
		// field. A failure indicates an empty field.
		//
		try
		{
		    String accountNumber  = kf.getText().substring(0,5);
		    //
		    // Store the resultant String in the vector.
		    //
		    returnVector.addElement(accountNumber);
		}
		catch(IndexOutOfBoundsException e)
		{
		    //
		    // An empty field, so exit the loop.
		    //
		    break;
		}
	    }

	    //
	    // Check to see if we have reached the end of the list.
	    // This technique inspects a substring of the display buffer.
	    //
	    String s = new String(ke.getDisplayBuffer(),
					ke.offsetFromRowColumn(24,39), 3);

	    //
	    // If PA2 is on the screen we need to go round the loop again.
	    // If it isn't then we have reached the end of the list.
	    // 
	    if ( !(s.equals("PA2")) )
	    {
	    	break;
	    }

	    //
	    // There are more results, so go round the loop again. PA2 moves
	    // to the next set of results.
	    //
	    ke.pressPA(2);
	    ke.waitUntilKeyboardUnlocked();
	}

	//
	// Test to see if we obtained any results. If not, we need to perform
	// the clear operation an extra time.
	//
	if (returnVector.isEmpty())
	{
	    ke.pressClear();
	    ke.waitUntilKeyboardUnlocked();
	}

	//
	// Now perform the logging-off procedure
	//
	ke.pressClear();
	ke.waitUntilKeyboardUnlocked();

	//
	// The KixEmulator should have returned to the blank screen
	//
	ke.typeString("CSSF LOGOFF");
	ke.pressEnter();
	ke.waitUntilDisconnected();

	//
	// Return the results
	//
	return returnVector;
    }
}
