#!/usr/bin/sh
#Copyright 02/27/01 Sun Microsystems, Inc. All rights reserved.
#pragma ident "@(#)em_cgw_services	1.11 01/02/27 Sun Microsystems"

###############################################################################
# em_cgw_services
###############################################################################
#
# Start/stop Solstice EM CORBA gateway services.
#
###############################################################################

O_SERVICES_LOG="/var/opt/SUNWconn/em/debug/orbix_services.log"
ORBIX_STARTUP_SCRIPT=${IO_INSTALL_DIR}/bin/start_${IT_DOMAIN_NAME}_services
ORBIX_STOP_SCRIPT=${IO_INSTALL_DIR}/bin/stop_${IT_DOMAIN_NAME}_services

###############################################################################
# errmsg
###############################################################################
#
# Echo arguments.
#
###############################################################################

errmsg()
{
    if [ "$1" = "-n" ] ; then
	shift
	gettext em_cgw_services "$*"		# no newline
    else
	gettext em_cgw_services "$*\\n"
    fi
}

###############################################################################
# get_local_host
###############################################################################
#
# Return the name of the local host.  In a cluster, this is the logical
# host name.  Outside of a cluster, it's the physical host name.
#
###############################################################################

#get_local_host()
#{
#    if /usr/bin/pkginfo -q SUNWemha ; then  # cluster software installed
#	cluster_dir=`/usr/bin/pkgparam SUNWemha BASEDIR`
#	hosts=`$cluster_dir/SUNWcluster/bin/haget -f all_logical_hosts 2> /dev/null`

#	if [ $? -ne 0 ]; then  # something went wrong
#	    /usr/bin/uname -n  # ...just use the physical host
#	else
#	    set $hosts
#	    echo $1            # report only the 1st one
#	fi
#    else
#	/usr/bin/uname -n      # physical host name
#    fi
#}

###############################################################################
# Environment
###############################################################################
#
# Set up the environment.
#
###############################################################################

trap 'errmsg "\\nem_cgw_services terminated."; exit 1' 1 3 15
trap 'interrupt' 2

#
# To debug, switch the shell to ksh and uncomment out the following.
#
# single_step()
# {
#     echo "${PS2}\c"
#     read
# }
# 
#set -x
# trap 'single_step' DEBUG

umask 002					# make it something sane

#
# Set local variables.  These variables control the behavior of em_cgw_services.
#

#
# If $EM_RUNTIME has a capacity >= diskspace_threshold, a warning is
# generated.  diskspace_threshold is the percentage free.
#

diskspace_threshold=98

#
# Set environment variables.
#

if /usr/bin/pkginfo -q SUNWemmis; then
	installdir=`/usr/bin/pkgparam SUNWemmis BASEDIR`
	. $installdir/SUNWconn/em/bin/emenv.sh
	export PATH  # bug in emenv.sh, it does not export PATH
fi

if /usr/bin/pkginfo -q SUNWemcgr; then
	installdir=`/usr/bin/pkgparam SUNWemcgr BASEDIR`
	. $installdir/SUNWconn/em/bin/em_corba_env.sh
	ORBIX_STARTUP_SCRIPT=${IO_INSTALL_DIR}/bin/start_${IT_DOMAIN_NAME}_services
	ORBIX_STOP_SCRIPT=${IO_INSTALL_DIR}/bin/stop_${IT_DOMAIN_NAME}_services
fi

SYSTEMID=`$EM_HOME/bin/get_local_host`; export SYSTEMID
RESTARTDIR=$EM_HOME/install/em_platform

#
# Set path to java

JAVA_HOME=/usr/java1.1
CLASSPATH=$JAVA_HOME/classes:$JAVA_HOME/lib/classes.zip:$JAVA_HOME/lib/rt.jar:$JAVA_HOME/lib/i18n.jar

export JAVA_HOME CLASSPATH

#
# Make sure the daemons started by em_cgw_services connect to the MIS we start.
# The user may have EM_MIS_SERVER set to another host.
#

EM_MIS_SERVER="${EM_CORBA_MIS_HOSTNAME:-$SYSTEMID}"; export EM_MIS_SERVER

#
# work from EM_HOME
#

if [ -d $EM_HOME ]; then
    cd $EM_HOME 
fi


###############################################################################
# interrupt  
###############################################################################
#
# When the user interrupts em_cgw_services, ask him what he wants to do.
#
###############################################################################

interrupt()
{
    errmsg '\n\nem_cgw_services interrupted.  What would you like to do?\n'
    errmsg '	c	continue'
    errmsg '	d <opt>	start em_debug with options <opt> and continue'
    errmsg '	q	quit without changing state (default)'
    errmsg '	s	stop all CORBA gateway processes and quit'
    errmsg -n '\n> '
    read answer options

    case "$answer" in
	c) 
	    errmsg 'Continuing...'
	    return
	    ;;

	d)
	    errmsg -n "Starting em_debug "
	    errmsg "$options..."
	    eval em_debug $options &
	    return
	    ;;

	s)
	    errmsg 'Shutting down CORBA gateway services ...'
	    kill_procs -HUP
	    exit 2
	    ;;

	*)
	    errmsg 'Quitting...'
	    exit 2
	    ;;
    esac
}



###############################################################################
# info  
###############################################################################
#
# Echo the arguments only if the -q option hasn't been given. All errors 
# should be displayed using echo.  Use info only for informative messages.
#
###############################################################################

info() 
{
    if [ "$1" = "-n" ] ; then
	shift
	gettext em_cgw_services "$*" > $display_device 2>&1  # no newline
    else
	gettext em_cgw_services "$*\\n" > $display_device 2>&1
    fi
}



###############################################################################
# debugmsg  
###############################################################################
#
# Echo the arguments only if the -d option hasn't been given. 
#
###############################################################################

debugmsg()
{
    if [ "$1" = "-n" ] ; then
	shift
	gettext em_cgw_services "$*" > $debug_device 2>&1  # no newline
    else
	gettext em_cgw_services "$*\\n" > $debug_device 2>&1
    fi
}


###############################################################################
# progress
###############################################################################
#
# Mark the passage of time to convince the user we haven't really hung at all.
#
###############################################################################

progress()
{
    # marker for monitor
    cp /dev/null /tmp/emha_progress > /dev/null 2>&1

    if [ "$quiet" = TRUE -o "$debug" = TRUE ] ; then
	return
    fi

    progress_counter=`expr $progress_counter + 1`   
    position=`expr $progress_counter % 4`

    if [ $position -eq 0 ] ; then
	echo '|\b\c'
    elif [ $position -eq 1 ] ; then
	echo '/\b\c'
    elif [ $position -eq 2 ] ; then
	echo '-\b\c'
    else
	echo '\\\b\c'
    fi
}


###############################################################################
# ask_confirm
###############################################################################
#
# Prompt the user with $1.  If they answer anything other than 'y', echo $2 
# and exit.  Note that a -f on the command line is equivalent to automatically
# answering 'y' to all questions.
#
###############################################################################

ask_confirm() 
{
    if [ $force = TRUE ] ; then			# don't bother asking
	return
    fi

    if [ -z "$1" ] ; then
	errmsg -n 'Do you really want to do this (y/n)? '	
    else
	errmsg -n "$1 "
    fi

    read answer

    if [ "$answer" != "y" ] && [ "$answer" != "Y" ] ; then
	if [ -z "$2" ]; then
	    errmsg "No action has been taken."
	else
	    errmsg "$2"
	fi
	exit 1
    fi
}


###############################################################################
# getpid
###############################################################################
#
# Return the pid(s) of the processes with the given name.  A null string is
# returned if the process isn't running.
#
###############################################################################

getpid()
{
    short_daemon=$1 
    ps -eo pid,comm  | grep "$short_daemon" | sed -e 's/^  *//' -e 's/ .*//'
}



###############################################################################
# kill_procs
###############################################################################
#
# Kill all the corba daemons.
#
###############################################################################

kill_procs() 
{
    info "Terminating CORBA Gateway services ..."

    # check if monitor is running
    em_mon=`ps -e | grep em_monit`
    if [ -n "$em_mon" ]; then
       # kill monitor so that it does not restart the corba daemons
       em_monitor -stop
    fi

    progress
    
    pid=`getpid itconfig_rep`
    if [ -n "$pid" ] ; then
    	debugmsg "Terminating Orbix services..."    
	if [ -x ${ORBIX_STOP_SCRIPT} ] ; then
		${ORBIX_STOP_SCRIPT} > ${O_SERVICES_LOG} 2>&1 &
	else
		errmsg "Unable to stop Orbix services coz script '${ORBIX_STOP_SCRIPT}' not found."
		exit 1
	fi
    fi
    sleep 2
        
    for daemon in em_io_corba_eds em_io_corba_mgw em_io_corba_epr em_io_corba_rgw 
    do
	pid=`getpid $daemon`
    
        for p in $pid ; do
            debugmsg "Terminating PID=$p"
	    #debugmsg "$daemon: PID=$p "
	    progress
	    kill $1  $p > $debug_device 2>&1
        done
    done

    progress

    # Wait for all the processes to die before exiting the shell
    # If any process is alive wait for 2 seconds before resuming
    # existance check

    process_is_alive=1
    while [ $process_is_alive -eq 1 ]
    do
        process_is_alive=0
        # Assumption : All the processes checked in this loop donot restart
        # after dying
        for daemon in em_io_corba_rgw em_io_corba_mgw em_io_corba_epr em_io_corba_eds 
        do
            pid=`getpid $daemon`
            if [ -n "$pid" ] ; then
                process_is_alive=1
                progress
                break
        fi
        done
        sleep 2
    done

    if [ -n "$em_mon" ]; then
    # start monitoring the remaining daemons
       em_monitor -restart
    fi  
}

###############################################################################
# cleanup_corba_gateway_files() will cleanup any log files, debug files, lock
# files, etc.
###############################################################################
cleanup_corba_gateway_files() {
    /bin/rm -f /tmp/.em_corba_eds.lock
    /bin/rm -f /tmp/.em_corba_epr.lock
    /bin/rm -f /tmp/.em_corba_mgw.lock
    /bin/rm -f /tmp/.em_corba_rgw.lock

    /bin/rm -f /var/opt/SUNWconn/em/debug/cos_naming.log
}
    
###############################################################################
# stat_procs
###############################################################################
#
# Display the process status.
#
###############################################################################

stat_procs() 
{
    for daemon in em_io_corba_rgw em_io_corba_mgw em_io_corba_epr em_io_corba_eds
    do
	pid=`getpid $daemon`
    
	if [ -z "$pid" ] ; then
           errmsg -n "$daemon:" | awk '{ printf "%-15s ", $1 }'
	   errmsg "not running"
	else
	    for p in $pid ; do
                echo "$daemon:\tPID=$p " | awk '{ printf "%-15s %s\n", $1, $2 }'
	    done
	fi
    done
}

################################################################################
# Start Event Gateway
# The default behavior is to start two instances of CORBA EDS. You can change
# this, depending on your performance and configuration needs, by modifying
# start_corba_event_gw()
#
################################################################################
start_corba_eds() {
    # $1 is log file, $2 is debug port number and $3 is number of threads

    # Create the lock file.  When em_croba_eds is done initializing, it'll
    # remove the lock file.

    cp /dev/null /tmp/.em_corba_eds.lock
    chmod 666    /tmp/.em_corba_eds.lock
    
    em_io_corba_eds -mis $EM_MIS_SERVER -log $1 -port $2 -threads $3 "-ORBdomain_name ${IT_DOMAIN_NAME} -ORBconfig_domains_dir ${IT_CONFIG_DOMAINS_DIR}" &
    EDS_PID=$!
    
    progress
    
    # Wait for the lock file to be removed.  This tells us em_corba_eds is
    # done initializing.
    
    while [ -f /tmp/.em_corba_eds.lock ] ; do
	ps -p $EDS_PID > /dev/null 2>&1
	if [ $? -ne 0 ]; then
	    # em_corba_eds is not running so we must abort startup
	    kill_procs -ABRT
	    exit 1
	fi
	sleep 3
	progress
    done
}

start_corba_event_gw() {
    start_corba_eds em_corba_eds1.log 6500 3
    debugmsg "Started the first CORBA EDS"

    start_corba_eds em_corba_eds2.log 6502 3
    debugmsg "Started the second CORBA EDS"
}

################################################################################

###############################################################################
# start_procs
###############################################################################
#
# Start the corba daemons.
#
###############################################################################

start_procs()
{
    info "Starting CORBA Gateway services ..."
    progress

    #
    # Check the diskspace in $EM_RUNTIME.  If it's close to full, put out a
    # warning.  
    #

    capacity=`df -k $EM_RUNTIME | tail -1 | 	
	      awk '{ print substr($5, 0, length($5) - 1) }'`

    if [ "$capacity" -ge "$diskspace_threshold" ] ; then
	errmsg -n "\\nWarning: "
	errmsg -n "$EM_RUNTIME "
	errmsg -n "is "
	errmsg -n "${capacity}% "
	errmsg "full.\\n"
	ask_confirm 'Start Solstice EM CORBA Gateway (y/n)?' "Solstice EM CORBA Gateway not started"
	errmsg "\\n"
    fi

    #if [ $mis_installed = TRUE ] ; then
	#if [ -x /etc/opt/licenses/lmstat ]; then
	#    rm -f /tmp/all_licenses_combined

	#    for LM_FILE in `echo $LM_LICENSE_FILE | awk -F: '{print $1 " " $2 " " $3 " " $4 " " $5 " " $6 }'`
	#    do
	#	if [ -n "$LM_FILE" ] && [ -r "$LM_FILE" ]; then
	#	    cat $LM_FILE >> /tmp/all_licenses_combined
	#	fi
	#    done
	#    /etc/opt/licenses/lmstat -i ssp.em.mis -c /tmp/all_licenses_combined | grep ssp.em.mis > $debug_device 2>&1
   	#    if [ $? -ne 0 ]; then
	#	errmsg "A usage license is not available."
	#	kill_procs -ABRT
	#	rm -f /tmp/all_licenses_combined
	#	exit 1
	#    fi
	#else
	#    errmsg "Warning: Unable to verify license, continuing anyway."
	#fi

	#rm -f /tmp/sysmgr.pids /tmp/dmuxd.pid /tmp/all_licenses_combined
    #fi

    ##############################################
     
    #  starting orbix services  
    
    debugmsg "Checking if Orbix services are running ..."
    N_SERVER_LOG="/var/opt/SUNWconn/em/debug/cos_naming.log"
    
    pid=`getpid itnaming`

    if [ -z "$pid" ] ; then
	debugmsg "Starting Orbix services ..."
	if [ -x $ORBIX_STOP_SCRIPT ] ; then
	  	${ORBIX_STARTUP_SCRIPT} > ${O_SERVICES_LOG} 2>&1 &
	 	 
	 	sleep 5
    		while [ `/usr/bin/du $O_SERVICES_LOG | cut -f1` = 0 ]; do
       			progress
       			sleep 10
    	 	done
	 
	 	debugmsg "Logging Name Service IOR...."
	 	${IO_INSTALL_DIR}/bin/itnaming -ORBdomain_name ${IT_DOMAIN_NAME} -ORBconfig_domains_dir \
	 	${IT_CONFIG_DOMAINS_DIR} prepare -publish_to_file=$N_SERVER_LOG > ${O_SERVICES_LOG} 2>&1 &
	 
	 	sleep 2
    	 	while [ `/usr/bin/du $N_SERVER_LOG | cut -f1` = 0 ]; do
	 		debugmsg "Waiting for Name server to come up..."
       			progress
			sleep 5
    	 	done
	else
	 	errmsg "Unable to start Orbix services coz startup script '${ORBIX_STARTUP_SCRIPT}' not found."
		exit 1
	fi
    fi

    progress
    
    ############################################

    info -n "Default CORBA Root Naming Context is: "
    info $root_naming_context

    debugmsg "Starting em_io_corba_epr..."

    #
    # Create the lock file.  When em_corba_epr is done initializing, it'll
    # remove the lock file.
    #

    cp /dev/null /tmp/.em_corba_epr.lock
    chmod 666    /tmp/.em_corba_epr.lock

    em_io_corba_epr "-ORBdomain_name ${IT_DOMAIN_NAME} -ORBconfig_domains_dir ${IT_CONFIG_DOMAINS_DIR}" &
    EPR_PID=$!

    progress

    #
    # Wait for the lock file to be removed.  This tells us em_io_corba_epr is
    # done initializing.
    #

    while [ -f /tmp/.em_corba_epr.lock ] ; do
	ps -p $EPR_PID > /dev/null 2>&1
        if [ $? -ne 0 ]; then
	    # em_corba_epr is not running so we must abort startup
	    kill_procs -ABRT
	    exit 1
	fi
	sleep 3
	progress
    done

    debugmsg "Starting em_io_corba_eds..."
    start_corba_event_gw

    debugmsg "Starting em_io_corba_mgw..."

    #
    # Create the lock file.  When em_corba_mgw is done initializing, it'll
    # remove the lock file.
    #
    
    cp /dev/null /tmp/.em_corba_mgw.lock
    chmod 666    /tmp/.em_corba_mgw.lock
    
    em_io_corba_mgw "-ORBdomain_name ${IT_DOMAIN_NAME} -ORBconfig_domains_dir ${IT_CONFIG_DOMAINS_DIR}"&
    MGW_PID=$!
    
    progress
    
    #
    # Wait for the lock file to be removed.  This tells us em_io_corba_mgw is
    # done initializing.
    #
    
    while [ -f /tmp/.em_corba_mgw.lock ] ; do
	ps -p $MGW_PID > /dev/null 2>&1
	if [ $? -ne 0 ]; then
	    # em_corba_mgw is not running so we must abort startup
	    kill_procs -ABRT
	    exit 1
	fi
	sleep 3
	progress
    done
    
    debugmsg "Starting em_corba_rgw ..."
    
    #
    # Create the lock file.  When em_corba_rgw is done initializing, it'll
    # remove the lock file.
    #
    
    cp /dev/null /tmp/.em_corba_rgw.lock
    chmod 666    /tmp/.em_corba_rgw.lock
    
    em_io_corba_rgw "-ORBdomain_name ${IT_DOMAIN_NAME} -ORBconfig_domains_dir ${IT_CONFIG_DOMAINS_DIR}" &
    RGW_PID=$!
    
    progress
    
    #
    # Wait for the lock file to be removed.  This tells us em_corba_rgw is
    # done initializing.
    #
    
    while [ -f /tmp/.em_corba_rgw.lock ] ; do
	ps -p $RGW_PID > /dev/null 2>&1
	if [ $? -ne 0 ]; then
	    # em_corba_rgw is not running so we must abort startup
	    kill_procs -ABRT        
	    exit 1
	fi
	sleep 3
	progress
    done
    progress
}



###############################################################################
# usage
###############################################################################
#
# Print a usage message.
#
###############################################################################

usage()
{
    errmsg ""
    errmsg "em_cgw_services starts or stops all the necessary processes to run"
    errmsg "Solstice Enterprise Manager CORBA Gateway."
    errmsg ""
    errmsg "Usage: em_cgw_services <command>"
    errmsg ""
    errmsg "<command> instructs em_cgw_services what operation to perform:"
    errmsg "default <command> is -start"
    errmsg ""
    errmsg "-debug    enable debugging for that command"
    errmsg "-force    force the command to execute without prompting for confirmations"
    errmsg "-help     print this message"
    errmsg "-quiet    suppress all output for that command"
    errmsg "-start    start EM CORBA Gateway services without reinitalizing data"
    errmsg "-init     reinitialize data and start EM CORBA Gateway services"
    errmsg "-status   display the status of the EM CORBA Gateway services"
    errmsg "-stop     stop EM CORBA Gateway services"
    errmsg "-version  display the current version number of Solstice Enterprise Manager"
    errmsg ""
    errmsg "-debug, -force, and -quiet only work in conjunction with another option,"
    errmsg 'for example, "em_cgw_services -debug -start".'
    errmsg ""
    errmsg "Environment variable listing -"
    errmsg "EM_CORBA_MIS_HOSTNAME specifies which MIS host to connect to"
    errmsg "EM_CORBA_ROOT_NAMING_CONTEXT specifies the default root naming context"
    errmsg "By default, root naming context is same as EM_CORBA_MIS_HOSTNAME"
    errmsg ""
    errmsg "Before starting em_cgw_services, ensure that -"
    errmsg "a) Orbix installation is complete"
    errmsg "b) Orbix lib and RogueWave lib directories are included in LD_LIBRARY_PATH"
    errmsg ""
}


###############################################################################
# main
###############################################################################
#
# The main part of this script.
#
###############################################################################

#
# Check if SUNWemmis is installed.
#

if pkginfo -q SUNWemmis.* ; then
    mis_installed=TRUE
else
    mis_installed=FALSE
fi

#
# Parse command-line arguments
#

quiet=FALSE; export quiet
force=FALSE; export force
debug=FALSE; export debug
progress_counter=0; export progress_counter

if tty -s ; then
   display_device=/dev/tty
else
   display_device=/dev/console
fi
export display_device

debug_device=/dev/null;  export debug_device

root_naming_context="${EM_CORBA_ROOT_NAMING_CONTEXT:-$EM_MIS_SERVER}"; export root_naming_context

if [ $# -eq 0 ] ; then
    command=start
else
    arg=$1
    while [ ! -z "$arg" ] ; do
	case $arg in
	    -f | -force )  
		force=TRUE
		;;
	    -q | -quiet )  
		quiet=TRUE
		display_device=/dev/null
		;;
	    -d | -debug )
		debug=TRUE
		debug_device=/dev/tty
		;;
	    -s | -stop )
		command=stop
		;;

	    -i | -init )
                command=init
                ;;

	    -init_no_kill)
		command=init_no_kill
		;;

	    -start )
		command=start
		;;
	    -start_no_kill )
	        command=start_no_kill
                ;;
	    -status )
		command=status
		;;
	    -V | -version )
		errmsg -n "Solstice (TM) Enterprise Manager (TM) "
		cat $EM_HOME/.version
		errmsg ""
		exit 0
		;;
	    -h | -help | * )
		usage
		exit 2
		;;
	esac
	shift 1
	arg=$1
    done
fi

#
# Check if root user.
#

my_uid=`/usr/xpg4/bin/id -u`

if [ $my_uid -ne 0 ] ; then
    errmsg "You must be the root user to run em_cgw_services."
    exit 1
fi

case $command in
    start) 
	kill_procs -HUP
        start_procs
	;;

    start_no_kill)
        start_procs
        ;;

    init)
        if [ $force = FALSE -a $quiet = FALSE ]; then
            ask_confirm 'Warning: All current CORBA Gateway information will be lost.\nDo you want to continue (y/n)? ' 'CORBA Gateway information has not been altered.'
        fi

        kill_procs -HUP
	cleanup_corba_gateway_files
        start_procs
        ;;

    init_no_kill)
	cleanup_corba_gateway_files
        start_procs
        ;;

    stop)
	if [ $force = FALSE ]; then
	    ask_confirm 'Do you want to gracefully terminate EM CORBA Gateway services (y/n)?: ' 'EM CORBA Gateway services not terminated.'
	fi
	kill_procs -HUP
	debugmsg "em_cgw_services shutdown complete."
	exit 0
	;;

    status)
	stat_procs
	exit 0
	;;

    *) 
	usage
	exit 1
	;;
esac

info "CORBA Gateway services started."
exit 0
