/*
 * @(#)file      SnmpSession.java
 * @(#)author    Sun Microsystems, Inc.
 * @(#)version   3.39
 * @(#)date      01/08/24
 *
 * Copyright 2000 Sun Microsystems, Inc. All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2000 Sun Microsystems, Inc. Tous droits rservs.
 * Ce logiciel est propriet de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation. 
 *
 */
// Copyright (c) 1995-96 by Cisco Systems, Inc.

package javax.management.snmp.manager ;


import java.io.Serializable;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Stack;

import javax.management.snmp.SnmpOid;
import javax.management.snmp.SnmpTimeticks;
import javax.management.snmp.SnmpVarBind;
import javax.management.snmp.SnmpVarBindList;
import javax.management.snmp.SnmpDefinitions;
import javax.management.snmp.SnmpStatusException;

// import debug stuff
//
import com.sun.jdmk.Trace;

/**
 * Creates, controls, and manages one or more requests. A session can be bound to a peer so that all
 * requests created (without specifying a peer) will use this as the default.
 * Alternatively, there can be a mixture of peers, that is, two requests can talk to
 * two different peers.  
 * <P>
 * Each <CODE>SnmpSession</CODE> has a dispatcher that is a thread used to service
 * all the requests it creates and each <CODE>SnmpSession</CODE> uses a separate socket for sending requests.
 * The socket imposes an upper limit on size of response packet. Any packet which exceeds this limit is truncated.
 * By default, this limit is {@link SnmpPeer#defaultSnmpResponsePktSize}. It can be changed using {@link #setResponsePktSize}.
 * <P>
 * This class implements an SNMP commands interface which provides a variety 
 * of convenience methods that are used to create requests. Each of the requests
 * perform a specific SNMP operation.
 * These methods are simple to use and automatically start the request.
 * The session maintains the list of all active requests and responses.
 * <P>
 * Users can explicitly create and start the requests.
 * This is usually not required for most applications.
 * <P>
 * Each method provides specific SNMP operations like <CODE>SnmpGet</CODE>, 
 * <CODE>SnmpGetNext</CODE> and <CODE>SnmpSet</CODE>.  The <CODE>SnmpGet</CODE>
 * and <CODE>SnmpGetNext</CODE> operations can be either poll or non-poll.
 * In addition, there is a convenient method which automatically walks
 * through a set of instances starting at any arbitrary place in the MIB
 * hierarchy and terminates when a specified condition becomes <CODE>true</CODE>.
 * <P>
 * The requests are retried when the peer does not respond within a
 * specified time. There is a maximum try limit. Certain options
 * of sessions can be configured using the <CODE>SnmpOptions</CODE> object.
 * This allows enabling and disabling of features like fixing protocol data units (PDU)
 * on error, handling <CODE>SnmpTooBig</CODE> errors and multiplexing requests.
 * If not specified, the default value sets the following options:
 * <UL>
 * <LI>multiplex allowed
 * <LI>fix PDU on error
 * <LI>handle too big PDU
 * </UL>
 * <P>
 * Once a session is created the user can perform one or more SNMP operations
 * in a one-at-a-time sequence.  Usually this sequence is a set of logical
 * operations which together form a complete operation.  
 * For example, a <CODE>SnmpSet</CODE> operation, which involves an initial
 * query to the agent for obtaining a valid index. This is followed by one
 * or more <CODE>SnmpSet</CODE> commands to complete the set operation.
 * <P>
 * At any time, one or more requests active in a session can be cancelled.
 *
 * @see javax.management.snmp.manager.SnmpRequest
 */

final public class SnmpSession implements SnmpDefinitions, Runnable, Serializable {

    /* NPCTE fix for bug 4468685, esc 530909 MR 13-June-2001 */
    String usedThread = null;
    /* end of NPCTE fix for bugid 4468685 */
    /**
     * Constructor for creating a new session.  
     * @param name	The session name.
     * @param apeer	The default peer to which to send requests.
     *
     * @exception SnmpStatusException Unable to initialize the Snmp Datagram Socket.
     */
    public SnmpSession (String name, SnmpPeer apeer) throws SnmpStatusException {
        this(name) ;
        _defaultPeer = apeer ;
    }

    /**
     * Constructor for creating a new session.  
     * @param name	The session name.
     *
     * @exception SnmpStatusException Unable to initialize the Snmp Datagram Socket.
     */
    public SnmpSession (String name) throws SnmpStatusException {
        super() ;
        sessionName = name ;
        initialize();
    }

    /**
     * Gets the session name (often used in identification).
     * @return The session name.
     */
    final public String getName() {
        return sessionName ;
    }

    /**
     * Sets the session name (often used in identification).
     * @param name The session name.
     */
    final public void setName(String name) {
        if (name == null)
            name = "UnNamedSession" ;
        else
            sessionName = name ;
    }

    /**
     * Gets a default peer (if any) bound to this session.
     * @return The default peer object.
     */
    final public SnmpPeer getDefaultPeer() {
        return _defaultPeer ;
    }
    
    /**
     * Sets peer as default. This peer is used when an SNMP command is
     * issued without specifying the destination.
     * @param apeer The default destination.
     */
    final public void setDefaultPeer(SnmpPeer apeer) {
        _defaultPeer = apeer ;
    }

    /**
     * Gets the maximum size allowed for response packet.
     * Any packet which exceeds this limit will be truncated.
     * The default value is {@link SnmpPeer#defaultSnmpResponsePktSize}.
     * @return The maximum size allowed for response packet.
     */
    final public int getResponsePktSize() {
        return _theSocket.getResponsePktSize() ;
    }

    /**
     * Sets the maximum size allowed for response packet.
     * @param size The maximum size allowed for response packet.
     */
    final public void setResponsePktSize(int size) {
        _theSocket.setResponsePktSize(size);
    }
    
    /**
     * Gets the number of errors that occured.
     * @return The number of errors that occured.
     */
    final public synchronized int getPktsErrors() {
        return _theSocket.getPktsErrors() ;
    }

    /**
     * Gets the number of packets received.
     * @return The number of packets received.
     */
    final public synchronized int getInPkts() {
        return _theSocket.getInPkts() ;
    }
    
    /**
     * Gets the number of packets sent.
     * @return The number of packets sent.
     */
    final public synchronized int getOutPkts() {
        return _theSocket.getOutPkts() ;
    }
    
    /**
     * Reset all the counters.
     */
    final public synchronized void performResetPktStatistics() {
        _theSocket.performResetPktStatistics() ;
    }
    
    /**
     * Gets a <CODE>String</CODE> representation of the session.
     * @return A <CODE>String</CODE> representation of the session.
     */
    final public String toString() {
        return "SnmpSession : " + sessionName ;
    }

    /**
     * Gets the <CODE>SnmpSocket</CODE> which will be used by requests
     * created in this session.
     * @return The socket which will be used in this session.
     */
    final SnmpSocket getSocket() {
        return _theSocket ;
    }

    /**
     * Sets the <CODE>SnmpSocket</CODE> which will be used by requests
     * created in this session.
     * @param s The socket which will be used in this session.
     * @exception SocketException A socket could not be created.
     */
    final void setSocket(SnmpSocket socket) throws java.net.SocketException {
        if (socket.isValid()) {
            _theSocket = socket ;
        } else
            throw new java.net.SocketException("Cannot set an invalid socket.") ;
    }

    /**
     * Two sessions are equivalent if they are equal.
     * @param arg The session to compare <CODE>this</CODE> with.
     * @return <CODE>true</CODE> if they are equal, <CODE>false</CODE> otherwise.
     */
    final synchronized boolean isEquivalent(SnmpSession arg) {
        if (this == arg)
            return true ;
        //         if (getSocket().equals(arg.getSocket())) {
        //             return true ;
        //         }

        return false ;
    }

    /**
     * Indicates whether the thread for this session is active.
     * @return <CODE>true</CODE> if active, <CODE>false</CODE> otherwise.
     */
    final public synchronized boolean isSessionActive() {
        return (_myThread != null && _myThread.isAlive()) ;
    }

    /**
     * Indicates whether this session is performing synchronous operation for a request.
     * @return <CODE>true</CODE> if the session is performing synchronous operation, <CODE>false</CODE> otherwise.
     */
    public synchronized boolean syncInProgress() {
        return _syncReq != null ;
    }

    /**
     * Destroys any pending requests and then stops the session.
     * The session will not be usable after this method returns.
     */
    final public void destroySession() {
        
        // Remove the clause synchronized of the destroySession method because
        // all the methods called in destroySession are already synchronized.
        // Synchronization is isolated as possible to avoid thread lock.
        // fix bug jaw.00392.B
        //
        isBeingDestroyed = true;
        _theSocket.isBeingDestroyed = true;
        cancelAllRequests() ;
        cancelAllResponses() ;
        synchronized(this) {
            _theSocket.close() ;
            _theSocket = null ;
        }
        snmpQman.decNbSessions();
        killSessionThread() ;
    }

    /**
     * Gets a list of requests which have the specified destination.
     * @param apeer The <CODE>SnmpPeer</CODE>.
     * @return Vector of requests to the specified peer.
     */
    public synchronized Vector getAllRequestsForPeer(SnmpPeer apeer) {
        
        Vector vec = new Vector() ;
        for (Enumeration e = _requestList.elements() ; e.hasMoreElements(); ) {
            SnmpRequest req = (SnmpRequest)e.nextElement() ;
            if (req.getPeer() == apeer)
                vec.addElement(req) ;
        }
        return vec ;
    }

    /**
     * Cancels all pending requests in this session.
     */
    final public void cancelAllRequests() {
        
        // Remove the clause synchronized of the cancelAllRequests method.
        // Synchronization is isolated as possible to avoid thread lock.
        // fix bug jaw.00392.B
        //
        if (_requestList.isEmpty()) {
            return ;
        }
        
        for (Enumeration e = _requestList.elements() ; e.hasMoreElements(); ) {
            ((SnmpRequest)e.nextElement()).cancelRequest() ;
        }
        _requestList.clear() ;
    }

    /**
     * Indicates whether a response was received for this request.
     * The status of the response is maintained in the request object.
     * @param req The request.
     * @return <CODE>true</CODE> if the request has a pending response, <CODE>false</CODE> otherwise.
     */
    public boolean checkResponseFor(SnmpRequest req) {
        return _respq.contains(req) ;
    }

    /**
     * Returns <CODE>true</CODE> if the current executing thread is this session's dispatcher.
     * Typically used to detect whether the user is doing a sync operation from
     * this dispatcher context. For instance, a user gives a sync command
     * from within a request callback using its associated session.
     * @return <CODE>true</CODE> if current thread is this session's dispatcher, <CODE>false</CODE> otherwise.
     */
    public boolean thisSessionContext() {
        return (Thread.currentThread() == getDispatcher()) ;
    }

    /**
     * Performs a single SNMP <CODE>get</CODE> request on the variable binding list.
     * When the command completes, the appropriate callback method is invoked.
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that gets invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler  How to get response data.
     */
    final public SnmpRequest snmpGetRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst) 
        throws SnmpStatusException {
        return makeAsyncRequest(peer, pduGetRequestPdu, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpGet(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpGetRequest(peer, new_cb, new_vblst) ;
    }

    /**
     * Performs a single SNMP <CODE>get</CODE> request on the variable binding list.
     * Uses the default peer.
     *
     * @param cb The callback that gets invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetRequest(SnmpRequestHandler cb, SnmpVarBindList vblst) 
        throws SnmpStatusException {
        return snmpGetRequest(_defaultPeer, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetRequest(SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpGet(SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        return snmpGet(_defaultPeer, cb, vblst) ;
    }
    
    /**
     * Performs a single SNMP <CODE>getnext</CODE> request on the variable binding list.
     * When the command completes, the appropriate callback method is invoked.
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that is invoked when a request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetNextRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst)
        throws SnmpStatusException {
        return makeAsyncRequest(peer, pduGetNextRequestPdu, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetNextRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpGetNext(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpGetNextRequest(peer, new_cb, new_vblst) ;
    }
    
    /**
     * Performs a single SNMP <CODE>getnext</CODE> request on the variable binding list.
     * Uses the default peer.
     *
     * @param cb The callback that is invoked when a request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetNextRequest(SnmpRequestHandler cb, SnmpVarBindList vblst) 
        throws SnmpStatusException {
        return snmpGetNextRequest(_defaultPeer, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetNextRequest(SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpGetNext(SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        return snmpGetNext(_defaultPeer, cb, vblst) ;
    }
    
    /**
     * Performs a single SNMP <CODE>getbulk</CODE> request on the variable binding list.
     * When the command completes, the appropriate callback method is invoked.
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that is invoked when a request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param nonRepeat Number of variable bindings to get one time.
     * @param maxRepeat Number of repetitions for the variable bindings to get multiple times.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler  How to get response data.
     */
    final public SnmpRequest snmpGetBulkRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst, int nonRepeat, int maxRepeat)
        throws SnmpStatusException {
        return makeAsyncBulkRequest(peer, cb, vblst, nonRepeat, maxRepeat) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetBulkRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList, int, int)}
     */
    final public SnmpRequest snmpGetBulk(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst, int nonRepeat, int maxRepeat) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpGetBulkRequest(peer, new_cb, new_vblst, nonRepeat, maxRepeat) ;
    }

    /**
     * Performs a single SNMP <CODE>getbulk</CODE> request on the variable binding list.
     * Uses the default peer.
     *
     * @param cb The callback that is invoked when a request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param nonRepeat Number of variable bindings to get one time.
     * @param maxRepeat Number of repetitions for the variable bindings to get multiple times.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetBulkRequest(SnmpRequestHandler cb, SnmpVarBindList vblst, int nonRepeat, int maxRepeat) 
        throws SnmpStatusException {
        return snmpGetBulkRequest(_defaultPeer, cb, vblst, nonRepeat, maxRepeat) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetBulkRequest(SnmpRequestHandler, SnmpVarBindList, int, int)}
     */
    final public SnmpRequest snmpGetBulk(SnmpHandler cb, SnmpVarbindList vblst, int nonRepeat, int maxRepeat) 
        throws SnmpStatusException {
        return snmpGetBulk(_defaultPeer, cb, vblst, nonRepeat, maxRepeat) ;
    }
    
    /**
     * Performs a single SNMP <CODE>set</CODE> request on the peer for the
     * specified variable binding list. Because retries on <CODE>set</CODE>
     * operations are not performed, the user needs to specify the timeout value
     * within which a response is requested.
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpSetRequest(SnmpPeer peer, SnmpRequestHandler cb,  SnmpVarBindList vblst)
        throws SnmpStatusException {
        
        if (vblst.isEmpty())
            throw new SnmpStatusException("VarBindList is empty for SNMP Set.") ;

        // checks if community is not null
        if (! peer.allowSnmpSets()) {
            throw new SnmpStatusException("Parameters may not have been configured"
                                          + " for doing SNMP SET's.") ;
        }

        if (! vblst.checkForValidValues())
            throw new SnmpStatusException("One or more variable binding has no valid value.") ;

        return makeAsyncRequest(peer, pduSetRequestPdu, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpSetRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpSet(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpSetRequest(peer, new_cb, new_vblst) ;
    }

    /**
     * Performs a single SNMP <CODE>set</CODE> request on the peer for the
     * specified variable binding list.
     * Uses the default peer.
     *
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpSetRequest(SnmpRequestHandler cb, SnmpVarBindList vblst) 
        throws SnmpStatusException {
        return snmpSetRequest(_defaultPeer, cb, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpSetRequest(SnmpRequestHandler, SnmpVarBindList)}
     */
    final public SnmpRequest snmpSet(SnmpHandler cb, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        return snmpSet(_defaultPeer, cb, vblst) ;
    }
    
    /**
     * Starts polling the peer at the specified intervals,
     * for the MIB variables in the variable binding list.
     * Performs an SNMP <CODE>get</CODE> request on the variable binding list.
     * After each poll, the appropriate callback method is invoked.
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param intrvl  The frequency of polling (in seconds).
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetPollRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst, int intrvl)
        throws SnmpStatusException {
        return makeAsyncPollRequest(peer, pduGetRequestPdu, cb, intrvl, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetPollRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList, int)}
     */
    final public SnmpRequest snmpGetPoll(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst, int intrvl) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpGetPollRequest(peer, new_cb, new_vblst, intrvl) ;
    }

    /**
     * Starts polling the peer at the specified intervals,
     * for the MIB variables in the variable binding list.
     * Uses the default peer.
     *
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param intrvl  The frequency of polling (in seconds).
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetPollRequest(SnmpRequestHandler cb, SnmpVarBindList vblst, int intrvl)
        throws SnmpStatusException {
        return snmpGetPollRequest(_defaultPeer, cb, vblst, intrvl) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetPollRequest(SnmpRequestHandler, SnmpVarBindList, int)}
     */
    final public SnmpRequest snmpGetPoll(SnmpHandler cb, SnmpVarbindList vblst, int intrvl) 
        throws SnmpStatusException {
        return snmpGetPoll(_defaultPeer, cb, vblst, intrvl) ;
    }
    
    /**
     * Starts polling the peer at the specified intervals,
     * for the MIB variables in the variable binding list.
     * It saves the original variable binding list that is used during each poll.
     * It performs an SNMP <CODE>getnext</CODE> request on the saved original
     * variable binding list and provides the response to the user via the callback
     * mechanism.
     * This method is similar to <CODE>snmpGetPoll</CODE> except that it uses
     * <CODE>getnext</CODE> on the original variable binding list instead of
     * <CODE>get</CODE>.
     *   
     * @param peer The SNMP peer object.
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param intrvl  The frequency of polling (in seconds).
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetNextPollRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst, int intrvl)
        throws SnmpStatusException {
        return makeAsyncPollRequest(peer, pduGetNextRequestPdu, cb, intrvl, vblst) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetNextPollRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList, int)}
     */
    final public SnmpRequest snmpGetNextPoll(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst, int intrvl) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpGetNextPollRequest(peer, new_cb, new_vblst, intrvl) ;
    }

    /**
     * Starts polling the peer at the specified intervals,
     * for the MIB variables in the variable binding list.
     * Uses the default peer.
     *   
     * @param cb The callback that is invoked when the request is complete.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param intrvl  The frequency of polling (in seconds).
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpGetNextPollRequest(SnmpRequestHandler cb, SnmpVarBindList vblst, int intrvl) 
        throws SnmpStatusException {
        return snmpGetNextPollRequest(_defaultPeer, cb, vblst, intrvl) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpGetNextPollRequest(SnmpRequestHandler, SnmpVarBindList, int)}
     */
    final public SnmpRequest snmpGetNextPoll(SnmpHandler cb, SnmpVarbindList vblst, int intrvl) 
        throws SnmpStatusException {
        return snmpGetNextPoll(_defaultPeer, cb, vblst, intrvl) ;
    }
    
    /**
     * Performs a single SNMP <CODE>inform</CODE> request on the variable binding list.
     * When the command completes, the appropriate callback method is invoked.
     * The variable list included in the outgoing inform request is composed of the following items:
     * <UL>
     * <LI><CODE>sysUpTime.0</CODE> with its current value.
     * <LI><CODE>snmpTrapOid.0</CODE> with the value specified by <CODE>trapOid</CODE>.
     * <LI><CODE>all the (oid, values)</CODE> from the specified variable binding list.
     * </UL>
     *
     * @param peer The SNMP peer object.
     * @param cb The callback that gets invoked when request is complete.
     * @param trapOid The OID identifying the inform request.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */

    final public SnmpRequest snmpInformRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpOid trapOid, SnmpVarBindList vblst) 
        throws SnmpStatusException {

        // First, make an SNMP V2 inform request pdu:
        // We clone varBindList and insert sysUpTime and snmpTrapOid variables.
        //
        SnmpVarBindList fullVbl ;    
        if (vblst != null)
            fullVbl = (SnmpVarBindList)vblst.clone() ;
        else 
            fullVbl = new SnmpVarBindList() ;
        
        SnmpTimeticks sysUpTimeValue = new SnmpTimeticks(getSysUpTime()) ;
        fullVbl.insertElementAt(new SnmpVarBind(snmpTrapOidOid, trapOid), 0) ;
        fullVbl.insertElementAt(new SnmpVarBind(sysUpTimeOid, sysUpTimeValue), 0) ;
                      
        // Next, send the pdu to the specified snmp peer.
        //
        return makeAsyncRequest(peer, pduInformRequestPdu, cb, fullVbl) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpInformRequest(SnmpPeer, SnmpRequestHandler, SnmpOid, SnmpVarBindList)}
     */
    final public SnmpRequest snmpInform(SnmpPeer peer, SnmpHandler cb, SnmpOid trapOid, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null) 
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpInformRequest(peer, new_cb, trapOid, new_vblst) ;
    }

    /**
     * Performs a single SNMP <CODE>inform</CODE> request on the variable binding list.
     * Uses the default peer.
     * <P>Note:
     * <BR> Take care when using the default SNMP peer in your requests.
     * <BR>Indeed, the default port for sending inform requests (162) is not the same
     * used for the <CODE>get</CODE>, <CODE>getnext</CODE>, <CODE>set</CODE> etc... operations (161).
     * So the peer to be used should not be the same.
     * <BR>To avoid any confusion, you can:
     * <UL>
     * <LI>either specify explicitely the SNMP peer to be used in the request.
     * <LI>or define 2 sessions, one dealing with operations and using the default peer/port 161,
     * the other dealing with inform requests and using the default peer/port 162.
     * </UL>
     *
     * @param cb The callback that gets invoked when request is complete.
     * @param trapOid The OID identifying the inform request.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @exception SnmpStatusException An error occured during the operation.
     *
     * @see javax.management.snmp.manager.SnmpRequestHandler How to get response data.
     */
    final public SnmpRequest snmpInformRequest(SnmpRequestHandler cb, SnmpOid trapOid, SnmpVarBindList vblst) 
        throws SnmpStatusException {
        return snmpInformRequest(_defaultPeer, cb, trapOid, vblst) ;
    }
    
    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpInformRequest(SnmpRequestHandler, SnmpOid, SnmpVarBindList)}
     */
    final public SnmpRequest snmpInform(SnmpHandler cb, SnmpOid trapOid, SnmpVarbindList vblst) 
        throws SnmpStatusException {
        return snmpInform(_defaultPeer, cb, trapOid, vblst) ;
    }
    
    /**
     * Walks through the lexicographic ordering of the agent MIB. It starts at
     * the specified variable binding list and continues until the OID <VAR>key</VAR>
     * specified as a parameter is "smaller" than the first variable in the
     * response variable binding list or an error occurs.
     * Typically, this feature is used to walk through an arbitrary table and obtain
     * all rows satisfying a specific condition. The condition may be
     * a specific instance combination for a MIB variable or a MIB variable
     * OID <CODE>entry</CODE> object.
     *
     * <PRE>
     * String var[] = {"sysDescr"} ;
     * SnmpVarBindList vblst = new SnmpVarBindList() ;
     * vblst.addVarBind(var) ;
     * SnmpRequest request = session.snmpWalkUntil(null, vblst, new SnmpOid("sysServices")) ;
     * </PRE>
     * This example walks through the <CODE>system</CODE> group and gets all the variables
     * in this group.
     * <PRE>
     * SnmpRequest request = session.snmpWalkUntil(null, vblst, new SnmpOid("1.4")) ;
     * </PRE>
     * This example walks through the entire agent MIB starting at <CODE>sysDescr</CODE>.
     * <PRE>
     * String var[] = {"sysDescr.0"} ;
     * SnmpVarBindList vblst = new SnmpVarBindList() ;
     * vblst.addVarBind(var) ;
     * SnmpRequest request = session.snmpWalkUntil(null, vblst, new SnmpOid("sysServices.0")) ;
     * </PRE>
     * This example walks through the <CODE>system</CODE> group beginning after the 
     * <CODE>sysDescr</CODE> variable (not included) and ending with the <CODE>sysServices</CODE> 
     * variable (included).
     * <P>
     * If there is more than one variable in the specified variable binding list,
     * the request will start at the first variable of the list.
     * <P>
     * The <CODE>snmpWalkUntil</CODE> request can be called in asynchronous mode only.
     *
     * @param peer The peer object to talk to.
     * @param cb The callback that is invoked when the request completes.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param key  Condition key that must be a subset of the first variable in
     * the response variable binding list. Walk terminates if this condition is <CODE>false</CODE>.
     * @exception SnmpStatusException An error occured during the operation.
     * 
     */
    final public SnmpRequest snmpWalkUntilRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst, SnmpOid key)
        throws SnmpStatusException {
        
        if (! isSessionActive()) {
            throw new SnmpStatusException(sessionName + " : Session is dead...") ;
        }
        SnmpPollRequest snmpreq = new SnmpPollRequest(this, peer, cb, pduWalkRequest) ;
        snmpreq.setOptions(snmpOptions.getOptions()) ;
        snmpreq.startPoll(vblst, key, true, 0) ;
        return snmpreq ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpWalkUntilRequest(SnmpPeer, SnmpRequestHandler, SnmpVarBindList, SnmpOid)}
     */
    final public SnmpRequest snmpWalkUntil(SnmpPeer peer, SnmpHandler cb, SnmpVarbindList vblst, SnmpOid key) 
        throws SnmpStatusException {
        
        SnmpRequestHandlerTranslator new_cb = null;
        SnmpVarBindList new_vblst = null;
        
        // Constructs the new SnmpRequestHandler.
        //
        if (cb != null)
            new_cb = new SnmpRequestHandlerTranslator(cb);
        
        // Constructs the new SnmpVarBindList.
        //
        if (vblst != null)
            new_vblst = new SnmpVarBindList(vblst);
        
        return snmpWalkUntilRequest(peer, new_cb, new_vblst, key) ;
    }

    /**
     * Walks through the lexicographic ordering of the agent MIB.
     * Uses the default peer.
     * <P>
     * The <CODE>snmpWalkUntil</CODE> request can be called in asynchronous mode only.
     *
     * @param cb The callback that is invoked when the request completes.
     * @param vblst A list of <CODE>SnmpVarBind</CODE> objects.
     * @param key  Condition key that must be a subset of the first variable in
     * the response variable binding list. Walk terminates if this condition is <CODE>false</CODE>.
     * @exception SnmpStatusException An error occured during the operation.
     *  
     */
    final public SnmpRequest snmpWalkUntilRequest(SnmpRequestHandler cb, SnmpVarBindList vblst, SnmpOid key) 
        throws SnmpStatusException {
        return snmpWalkUntilRequest(_defaultPeer, cb, vblst, key) ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #snmpWalkUntilRequest(SnmpRequestHandler, SnmpVarBindList, SnmpOid)}
     */
    final public SnmpRequest snmpWalkUntil(SnmpHandler cb, SnmpVarbindList vblst, SnmpOid key) 
        throws SnmpStatusException {
        return snmpWalkUntil(_defaultPeer, cb, vblst, key) ;
    }
    
    /**
     * Adds a request.
     * @param reqc The request to add.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    final synchronized void addRequest(SnmpRequest reqc) throws SnmpStatusException {
        
        // If the session is being destroyed, stop adding requests.
        //
        if (isBeingDestroyed == true)
            return;
        if (! isSessionActive()) {
            throw new SnmpStatusException(sessionName + " : Session is dead...") ;
        }
        _requestList.put(reqc, reqc) ;
    }

    /**
     * Deletes a request.
     * @param snmpreq The request to delete.
     */
    final synchronized void deleteRequest(SnmpRequest snmpreq) {
        _requestList.remove(snmpreq) ;
        if (_syncReq != null && _syncReq == snmpreq) {
            resetSyncMode() ;
        }
    }

    private synchronized void setSyncMode(SnmpRequest req) {
        _syncReq = req ;
    }

    private synchronized void resetSyncMode() {
        if (_syncReq == null)
            return ;
        _syncReq = null ;
        if (thisSessionContext())
            return ;
        this.notifyAll() ;
    }

    final SnmpRequest makeAsyncRequest(SnmpPeer peer, int cmd, SnmpRequestHandler cb, SnmpVarBindList vblst)
        throws SnmpStatusException {
        
        if (! isSessionActive()) {
            throw new SnmpStatusException(sessionName + " : Session is dead...") ;
        }
        SnmpRequest snmpreq = new SnmpRequest(this, peer, cb, cmd) ;
        snmpreq.setOptions(snmpOptions.getOptions()) ;
        snmpreq.start(vblst, true, 0) ;
        return snmpreq ;
    }

    final SnmpRequest makeAsyncBulkRequest(SnmpPeer peer, SnmpRequestHandler cb, SnmpVarBindList vblst, int nonRepeat, int maxRepeat)
        throws SnmpStatusException {
        
        if (! isSessionActive()) {
            throw new SnmpStatusException(sessionName + " : Session is dead...") ;
        }
        SnmpRequest snmpreq = new SnmpRequest(this, peer, cb, nonRepeat, maxRepeat) ;
        snmpreq.setOptions(snmpOptions.getOptions()) ;
        snmpreq.start(vblst, true, 0) ;
        return snmpreq ;
    }

    final SnmpPollRequest makeAsyncPollRequest(SnmpPeer peer, int cmd, SnmpRequestHandler cb, int intrvl, SnmpVarBindList vblst)
        throws SnmpStatusException {
        
        if (! isSessionActive()) {
            throw new SnmpStatusException(sessionName + " : Session is dead...") ;
        }
        SnmpPollRequest snmpreq = new SnmpPollRequest(this, peer, cb, cmd) ;
        snmpreq.setPollFrequency(intrvl) ;
        snmpreq.setOptions(snmpOptions.getOptions()) ;
        snmpreq.startPoll(vblst, true, 0) ;
        return snmpreq ;
    }

    /**
     * Indicates whether there are any responses available for processing.
     * These responses belong to the requests made using this session.
     * @return <CODE>true</CODE> if a response is available, <CODE>false</CODE> otherwise.
     */
    public boolean anyPendingResponses() {
        return (! _respq.isEmpty()) ;
    }

    /**
     * Dispatcher method for this session thread. This is the dispatcher method
     * which goes in an endless-loop and waits for servicing requests
     * which received a reply from the agent.
     */
    public void run() {
	
	/* NPCTE fix for bug 4468685, esc 530909 MR 13-June-2001 */ 
	synchronized(this) {
	    if(_myThread == null) return;
	    _myThread = Thread.currentThread() ;
	    usedThread = _myThread.toString();
	    _myThread.setPriority(Thread.NORM_PRIORITY);
	}
	/* end of NPCTE fix for bugid 4468685 */ 
        
	SnmpRequest reqc = null ;
       /* NPCTE fix for bug 4468685, esc 530909 MR 13-June-2001 */ 
	while(true) {
	    synchronized(this) {
		if(_myThread == null)
		    break;
		else
		    try {
			reqc = nextResponse(0) ;  
		    } catch (ThreadDeath d) {
			_myThread = null ;
			if (isDebugOn()) {
			    debug("run", 
				  "Session thread unexpectedly shutting down "+
				  sessionName);
			}
			throw d ;
		    }
	    }
	    if (reqc != null)
		processResponse(reqc);
        }
	/* end of NPCTE fix for bugid 4468685 */

        if (isTraceOn()) {
            trace("run", "Session thread shutting down " + sessionName);
        }
        _myThread = null ;
    }

    private void processResponse(SnmpRequest reqc) {
        
        while (reqc != null && _myThread != null) {
            try {
                if (reqc != null) {
                    if (isTraceOn()) {
                        trace("processResponse", "Processing response to req = " + reqc.getRequestId());
                    }
                    reqc.processResponse() ;  // Handles out of memory.
                    reqc = null ;  // finished processing.
                }
				
            } catch (Exception e) {
                if (isDebugOn()) {
                    debug("processResponse", e);
                }
                reqc = null ;
            } catch (OutOfMemoryError ome) {
                if (isDebugOn()) {
                    debug("processResponse", "Out of memory error in session thread " + sessionName);
                    debug("processResponse", ome);
                }
                Thread.currentThread().yield();
                continue ;   // re-process the request.
            }
        }
    }

    /**
     * Finalizer of the <CODE>SnmpSession</CODE> objects.
     * This method is called by the garbage collector on an object 
     * when garbage collection determines that there are no more references to the object.
     * <P>Removes all the requests for this SNMP session, closes the socket and
     * sets all the references to the <CODE>SnmpSession</CODE> object to <CODE>null</CODE>.
     */
    public void finalize() {
        
        if (_respq != null)
            _respq.removeAllElements() ;
        _respq = null ;
        if (_theSocket != null)
            _theSocket.close() ;
        _theSocket = null ;

        if (isTraceOn()) {
            trace("finalize", "Shutting all servers");
        }
        snmpQman = null ;
    }

    /**
     * Performs sync operations on active requests. Any number of requests
     * can be done in sync mode but only one per thread.
     * This method is not accessible outside of this package.
     * The user can do synchronous operation using the request handle only.
     *
     * @see SnmpRequest.waitForCompletion
     */ 
    void waitForResponse(SnmpRequest req, long waitTime) {
        
        if (! req.inProgress())
            return ;
        setSyncMode(req) ;

        if (isTraceOn()) {
            trace("waitForResponse", "Session switching to sync mode for request " + req.getRequestId());
        }
        long maxTime ;
        if (waitTime <= 0)
            maxTime = System.currentTimeMillis() + 6000 * 1000 ;
        else
            maxTime = System.currentTimeMillis() + waitTime ;

        while (req.inProgress() || syncInProgress()) {
            waitTime = maxTime - System.currentTimeMillis() ;
            if (waitTime <= 0)
                break ;
            synchronized (this) {
                if (! _respq.removeElement(req)) {
                    try {
                        this.wait(waitTime) ;
                    } catch(InterruptedException e) {
                    }
                    continue ;
                }
            }
            try {
                processResponse(req) ;
            } catch (Exception e) {
                if (isDebugOn()) {
                    debug("waitForResponse", e);
                }
            }
        }
        resetSyncMode() ;
        return ;
    }

    /**
     * Adds the request object which received a response to a request 
     * generated by the session.  This is added to a private store, which
     * will be eventually picked up by the dispatcher for processing.
     * @param reqc The request that received the response from the agent.
     */
    void addResponse(SnmpRequest reqc) {
        
        // If the session is being destroyed, stop adding responses.
        //
        if (isBeingDestroyed == true)
            return;
        
        SnmpRequest snmpreq = (SnmpRequest) reqc ;
        
        if (isSessionActive()) {
            if (syncInProgress()) {
                // process internal requests only.
                snmpreq = (SnmpRequest) reqc ;
                if (snmpreq.isInternalRequest()) {
                    if (isTraceOn()) {
                        trace("addResponse", "Session in sync mode. processing internal request only: " + reqc.getRequestId());
                    }
                    processResponse(reqc) ;
                    return ;
                } 
            }
            synchronized(this) {
                _respq.push(reqc) ;
                this.notifyAll() ;
            }
        } else {
            if (isDebugOn()) {
                debug("addResponse", "Peer thread dead. So response is dropped..." + reqc.getRequestId());
            }
        }
        return ;
    }

    private synchronized void cancelAllResponses() {
        if (_respq != null) {
            _syncReq = null ;
            _respq.removeAllElements() ;
            this.notifyAll() ;
        }
    }

    /**
     * Gets the dispatcher thread associated with the session.
     * Used to check whether the sync request was done from within the
     * dispatcher. You should prevent this from happening.
     * @return The dispatcher thread associated with the session.
     */
    final private Thread getDispatcher() {
        return _myThread ;
    }

    private synchronized void initialize() throws SnmpStatusException {

        if (isTraceOn()) {
            trace("initialize", "Initializing SNMP session internals");
        }
        
        if (_theSocket == null) {

            snmpQman = SnmpQManager.getTheInstance() ;
            snmpQman.incNbSessions();
            SnmpResponseHandler snmpRespHdlr = new SnmpResponseHandler(snmpQman) ;
		
            try {
                SnmpSocket asocket = new SnmpSocket(snmpRespHdlr) ;
                setSocket(asocket);
            } catch (java.net.SocketException se) {
                if (isDebugOn()) {
                    debug("initialize", "Unable to initialize Snmp Datagram socket");
                    debug("initialize", se);
                }
		    
                throw new SnmpStatusException("Unable to initialize Snmp Datagram Socket") ; 
            }
        }

        if (_myThread == null)
            _myThread = new Thread(this) ;
        if (!_myThread.isAlive())
            _myThread.start() ;
        
        // For the inform request sysUpTime, use the time the session started ...
        //
        startUpTime= java.lang.System.currentTimeMillis();
        
        if (isTraceOn()) {
            trace("initialize", "SNMP session internals initialized OK");
        }
    }

    synchronized SnmpRequest nextResponse(long time) {
        
        if (_respq.isEmpty()) {
            try {
                if (isTraceOn()) {
                    trace("nextResponse", "Blocking for response");
                }
                this.wait(time) ;
            } catch(InterruptedException e) {
            }
        }
        if (_respq.isEmpty())
            return null ;
        SnmpRequest reqc = (SnmpRequest) _respq.firstElement() ;
        _respq.removeElementAt(0) ;
        return reqc ;
    }

    /**
     * Make sure you are killing the thread when it is active.  Instead
     * prepare for a graceful exit.
     */
    private synchronized void killSessionThread() {
        
        if (isSessionActive()) {
            if (isTraceOn()) {
                trace("killSessionThread", "Destroying session " + sessionName);
            }
            if (Thread.currentThread() != _myThread) {
                _myThread = null ;
				//synchronized (_respq) {
                this.notifyAll() ;
				//}
            } else
                _myThread = null ;
        }
    }

    /**
     * Returns the time (in hundreths of second) elapsed since the session has been initialized.
     */
    private long getSysUpTime() {
        return (java.lang.System.currentTimeMillis() - startUpTime) / 10 ;
    }
    
    // TRACES & DEBUG
    //---------------
    
    boolean isTraceOn() {
        return Trace.isSelected(Trace.LEVEL_TRACE, Trace.INFO_SNMP);
    }

    void trace(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_TRACE, Trace.INFO_SNMP, clz, func, info);
    }

    void trace(String func, String info) {
        trace(dbgTag, func, info);
    }
    
    boolean isDebugOn() {
        return Trace.isSelected(Trace.LEVEL_DEBUG, Trace.INFO_SNMP);
    }

    void debug(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, info);
    }

    void debug(String clz, String func, Throwable exception) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, exception);
    }

    void debug(String func, String info) {
        debug(dbgTag, func, info);
    }
    
    void debug(String func, Throwable exception) {
        debug(dbgTag, func, exception);
    }
    
    // PROPERTIES
    //-----------
    
    // This boolean is used to stop adding requests/responses while the session
    // is being destroyed.
    //
    boolean isBeingDestroyed = false;
    
    /**
     * The dispatcher that will service all responses to requests generated
     * using this session object.  A session object creates one or more requests.
     * Thus it services all requests, which are created by this object, 
     * when a response arrives for a request generated by the session.
     * @serial
     */
    private  Thread _myThread = null ;

    /**
     * A place to store the responses from the agent.
     * A FIFO queue is needed here.
     * @serial
     */
    private Stack _respq = new Stack() ;

    /**
     * The <CODE>SnmpSocket</CODE> to be used to communicate with the peer
     * by all requests created in this session.
     * @serial
     */
    private SnmpSocket _theSocket = null ;

    /**
     * The default peer to be used in this session.
     * @serial
     */
    private SnmpPeer _defaultPeer = null ;

    /**
     * Request being synchronized from session thread.  This happens when
     * a user does sync operation from a callback.
     * @serial
     */
    private SnmpRequest _syncReq ;

    /**
     * The name identifying this session. Useful in debugging a session.
     * @serial
     */
    public String sessionName ;
    
    /**
     */
    private Hashtable _requestList = new Hashtable() ;

    /**
     * Set of options used to configure all <CODE>SnmpRequests</CODE> created 
     * in this session. The user can manipulate the options directly.
     * Note that this variable is public.
     * @serial
     */
    public SnmpOptions snmpOptions = new SnmpOptions() ;

    /**
     * For the inform request.
     */
    private static final SnmpOid sysUpTimeOid = new SnmpOid("1.3.6.1.2.1.1.3.0") ;
    private static final SnmpOid snmpTrapOidOid = new SnmpOid("1.3.6.1.6.3.1.1.4.1.0") ;
    private transient long startUpTime = 0;
    
    /**
     */
    SnmpQManager snmpQman = null ;
    /**
     */
    String dbgTag = "SnmpSession";
}
