/*
 * @(#)file      SnmpRequest.java
 * @(#)author    Sun Microsystems, Inc.
 * @(#)version   4.26
 * @(#)date      01/08/24
 *
 * Copyright 2000 Sun Microsystems, Inc. All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2000 Sun Microsystems, Inc. Tous droits rservs.
 * Ce logiciel est propriet de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation. 
 *
 */
// Copyright (c) 1995-96 by Cisco Systems, Inc.

package javax.management.snmp.manager ;


import java.io.Serializable;
import java.util.Date;

import javax.management.snmp.SnmpDataTypeEnums;
import javax.management.snmp.SnmpMessage;
import javax.management.snmp.SnmpNull;
import javax.management.snmp.SnmpVarBind;
import javax.management.snmp.SnmpVarBindList;
import javax.management.snmp.SnmpPduFactory;
import javax.management.snmp.SnmpPduPacket;
import javax.management.snmp.SnmpPduRequest;
import javax.management.snmp.SnmpPduBulk;
import javax.management.snmp.SnmpDefinitions;
import javax.management.snmp.SnmpStatusException;
import javax.management.snmp.SnmpTooBigException;
import javax.management.snmp.Timestamp;

// import debug stuff
//
import com.sun.jdmk.Trace;



/**
 * Creates a request that is used to perform one or more SNMP operations such as 
 * <CODE>Get</CODE>, <CODE>GetNext</CODE>, <CODE>Set</CODE>, <CODE>Inform</CODE> and 
 * <CODE>GetBulk</CODE> on a specified <CODE>SnmpVarBindList</CODE>.
 * <P>
 * This class provides basic functions that enable you to fire requests, handle retries, 
 * timeouts, and process responses from the agent.  
 * It then notifies the user upon completion of requests by invoking the callback interface 
 * implementation provided by the user.
 * <P>
 * An <CODE>SnmpRequest</CODE> can be created by the session object.
 * <P>
 * The peer object determines the destination of the request and controls
 * what and how many <CODE>SnmpVarBind</CODE>s to pack into a single request.
 * For example, the protocol data units (PDU) packet size, the number of
 * <CODE>SnmpVarBind</CODE>s allowed in a packet, the peer address and port, 
 * the <CODE>SnmpParameters</CODE> and probably some flow control mechanisms.
 * <P>
 * The session object provides resources such as the authentication mechanism, 
 * controlling all requests created by it, and finally the response to the user.
 * <P>
 * Each request can be configured with various options that influence
 * how a request handles a correctable error situation.
 * For example, <CODE>snmpTooBig</CODE> error and <CODE>snmpRspNoSuchName</CODE>
 * exception in the SNMPv1 context.
 * <P>
 * Each request, when ready to be sent, is assigned a unique identifier which helps
 * in identifying the request with matching responses to the protocol engine lying transparently underneath. 
 * The engine does the job of retrying the requests when the timer expires and calls the session when a timeout
 * occurs after exhausting the maximum number of tries.
 * <P>
 * When a user submits a request, it joins the pool of waiting requests waiting to be sent.  
 * When the request becomes ready to be sent, the underlying engine constructs a PDU packet and sends it out 
 * after starting the wait timer.  
 * The <CODE>SnmpVarBindList</CODE> with a request can be multiplexed with an <CODE>SnmpVarBindList</CODE> 
 * of other requests (passing multiplexing rules).  
 * The responses will be de-multiplexed. The multiplex operation remains transparent to the user.  
 * A user can choose to disallow this multiplexing of certain specific requests by providing the options 
 * when the request is created. This may become necessary to optimize performance of certain requests.
 * <P>
 * An application can cancel one or more requests at any time.
 * <P>
 * The requests that are sent are automatically retried if a response does not arrive within a specified interval.  
 * If the agent responds with an error, the request object uses the options to determine what should be done.
 * <P>
 * The request object provides the method, {@link #waitForCompletion waitForCompletion(long time)},
 * which enables a user to operate in a synchronous mode with a request.
 * This is done by blocking the user thread for the desired time interval.
 * The user thread gets notified whenever a request reaches completion, independently of the status of the response. 
 * If the user blocks on the session thread context, an exception is thrown (you should not block the session thread).
 * <P>
 * A request becomes active when a user submits the request successfully.
 * When an event happens that moves the request into the done state, the request becomes inactive.
 *
 * At any time, one or more requests active in a session can be cancelled.
 *
 * @see javax.management.snmp.manager.SnmpSession
 * @see javax.management.snmp.manager.SnmpPeer
 */

// **** Removed from Javadoc *** (feature to be verified and tested)

//It can also be created
//* directly by a user. The user would have to initialize the request manually
//* using a few methods. This is recommended for advanced users only.
//* A request is created in the context of a peer and a session.  

public class SnmpRequest implements Serializable, SnmpDefinitions {
    
    // PUBLIC VARIABLES
    //-----------------
      
    /**
     * Base status of a request.
     */
    final static private int stBase 		= 1 ;
    
    /**
     * Status of a request: in progress.
     */
    final static public int stInProgress 		= stBase ;
  
    /**
     * Status of a request: waiting to be sent.
     */
    final static public int stWaitingToSend 	= (stBase << 1) | stInProgress;
  
    /**
     * Status of a request: waiting for reply.
     */
    final static public int stWaitingForReply 	= (stBase << 2) | stInProgress;
  
    /**
     * Status of a request: reply received.
     */
    final static public int stReceivedReply 	= (stBase << 3) | stInProgress;
  
    /**
     * Status of a request: request aborted.
     */
    final static public int stAborted 			= (stBase << 4) ;
  
    /**
     * Status of a request: timeout.
     */
    final static public int stTimeout 			= (stBase << 5) ;
  
    /**
     * Status of a request: internal error occured.
     */
    final static public int stInternalError 	= (stBase << 6) ;
  
    /**
     * Status of a request: result available for the request.
     */
    final static public int stResultsAvailable 	= (stBase << 7) ;
  
    /**
     * Status of a request: request never used.
     */
    final static public int stNeverUsed 		= (stBase << 8) ;
    
    
    // PUBLIC VARIABLES
    //-----------------
      
    /**
     * This method maintains a global counter for the request ID.
     */
    private static RequestCounter requestCounter = new RequestCounter() ;
        
    /**
     * The peer object associated with this request.
     */
    private SnmpPeer thePeer ;
    
    /**
     * The session object associated with this request.
     */
    private SnmpSession session ;
    
    /**
     * The user implementation of the callback interface for this session.
     */
    private SnmpRequestHandler callback = null ;
    
    /**
     * The <CODE>nonRepeaters</CODE> field for the <CODE>bulk</CODE> operation.
     * This variable is used only for <CODE>pduGetBulkRequestPdu</CODE>.
     */
    private int nonRepeaters = 0 ;
    
    /**
     * The <CODE>maxRepetitions</CODE> field for the <CODE>bulk</CODE> operation.
     * This variable is used only for <CODE>pduGetBulkRequestPdu</CODE>.
     */
    private int maxRepetitions = 0 ;
    
    /**
     * Indicates whether this request is being serviced by another.
     * If this is not null, then there will be another object which will service this request. 
     */
    private ReqRedirectSrv proxy ;
    
    /**
     * The request PDU.
     */
    private SnmpPduPacket requestPdu ;

    /**
     * The response PDU.
     */
    private SnmpPduRequest responsePdu ;
    
    private SnmpVarBind internalVarBind[] = null ;
    private String reason = null ;

    /**
     * By default the following options are set:
     * <UL>
     * <LI> optAllowMultiplexing
     * <LI> optFixPduOnError
     * <LI> optHandleTooBigError
     * </UL>
     */
    private int options = SnmpOptions.defaultOption ;
    
    // indicates this request is created internally to handle a condition like
    // multiplxing, too-big etc.
    private final static int CREATED_BY_USER 	   = 1 ;  // normal user created.
    private final static int CREATED_INTERNALLY = 2 ;  // created internally.
    
    /**
     * A non-zero value indicates an ordinary request created by the user.
     * The value indicates an internal request is created by the package to solve a
     * special situation such as multiplexing requests or a <CODE>SnmpTooBig</CODE> handler.
     */
    private int mode = CREATED_BY_USER ;
                
    /**
     * Number of tries performed for the current polling operation.
     */
    private int numTries = 0 ;

    /**
     * Timeout. The default value is 5 seconds.
     */
    private int timeout = 5000 ; // 5 seconds.

    /**
     */
    private int reqState = stNeverUsed ;

    // Polling control parameters.
    private long  prevPollTime = 0 ;	// value of 0 means poll never happened.
    private long  nextPollTime = 0 ;
    private long  waitTimeForResponse ;
    private Date debugDate = new Date() ;

    /**
     * The request ID for an active request.
     */
    private int requestId = 0 ;	
    
    
    // PACKAGE VARIABLES
    //------------------
    
    /**
     * This contains a list of <CODE>SnmpVarBind</CODE> objects for making
     * the SNMP requests. A private copy of the user specified
     * <CODE>SnmpVarBindList</CODE> is made by cloning the
     * <CODE>SnmpVarBindList</CODE> to ensure that the user does not modify the
     * <CODE>SnmpVarBindList</CODE> or the contents while the polling is in progress.
     */
    SnmpVarBindList varBindList = null ;
    
    /**
     * The time stamp information for the agent response.
     */
    javax.management.snmp.Timestamp responseTimestamp = null ;  // timestamp info for agent response.
    
    /**
     * The error status associated with the response packet.
     */
    int errorStatus = 0 ;	
    
    /**
     * The index in <CODE>SnmpVarBindList</CODE> that caused the exception.
     */
    int errorIndex = 0 ;	
    
    /**
     * The SNMP command (only valid when session is active).
     */
    int command = 0 ;
    
    String dbgTag = "SnmpRequest";
    
    
    // TRACES & DEBUG
    //---------------
    
    boolean isTraceOn() {
        return Trace.isSelected(Trace.LEVEL_TRACE, Trace.INFO_SNMP);
    }

    void trace(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_TRACE, Trace.INFO_SNMP, clz, func, info);
    }

    void trace(String func, String info) {
        trace(dbgTag, func, info);
    }
    
    boolean isDebugOn() {
        return Trace.isSelected(Trace.LEVEL_DEBUG, Trace.INFO_SNMP);
    }

    void debug(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, info);
    }

    void debug(String clz, String func, Throwable exception) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, exception);
    }
    
    void debug(String func, String info) {
        debug(dbgTag, func, info);
    }
    
    void debug(String func, Throwable exception) {
        debug(dbgTag, func, exception);
    }
    
    
    // CONSTRUCTORS
    //-------------

    /**
     * For Java DMK internal use only.
     * Constructor for creating new request. This object can be created only by a session object. 
     * You must instantiate a <CODE>SnmpSession</CODE> object to create a request.
     * @param group	<CODE>SnmpSession</CODE> object for this request.
     * @param peer <CODE>SnmpPeer</CODE> object for this request.
     * @param requestCB Callback interface for the request.
     * @param cmd The command type for this request.
     * @see javax.management.snmp.manager.SnmpPeer
     * @see javax.management.snmp.manager.SnmpSession
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    SnmpRequest(SnmpSession group, SnmpPeer peer,SnmpRequestHandler requestCB, int cmd) throws SnmpStatusException {
        session = group ;
        thePeer = peer ;
        callback = requestCB ;
    
        switch (cmd) {
        case pduSetRequestPdu : 
        case pduGetRequestPdu : 
        case pduGetNextRequestPdu : 
        case pduGetBulkRequestPdu : 
        case pduInformRequestPdu : 
            command = cmd ;
            break ;
        case pduWalkRequest : 
            command = pduGetNextRequestPdu ; 	
            break ;
        default :
            throw  new SnmpStatusException("Unsupported Command. " + cmd) ;
        }
        session.addRequest(this) ;  // add to session queue.
        setTimeout(thePeer.getTimeout()) ;
    }

    /**
     * For Java DMK internal use only.
     * Constructor for creating new bulk request.
     * This constructor initializes the nonRepeaters and maxRepeaters with the
     * specified values.
     * @param group	<CODE>SnmpSession</CODE> object for this request.
     * @param requestCB	Callback interface for the request.
     * @param nonRepeat	Value for nonRepeater field.
     * @param maxRepeat	Value for maxRepeater field.
     * @see javax.management.snmp.manager.SnmpPeer
     * @see javax.management.snmp.manager.SnmpSession
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    SnmpRequest(SnmpSession group, SnmpPeer peer,SnmpRequestHandler requestCB, int nonRepeat, int maxRepeat) throws SnmpStatusException {
        this(group, peer, requestCB, pduGetBulkRequestPdu) ;
        nonRepeaters = nonRepeat ;
        maxRepetitions = maxRepeat ;
    }
    
    // PUBLIC METHODS
    //---------------

    // VARIABLES METHODS INHERITED BEFORE FROM class Request 

    /**
     * Gets the number of tries performed for the current request.
     * @return The number of tries performed.
     */
    final public synchronized int getNumTries () {
        return numTries ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #getNumTries}
     */
    final public synchronized int getRetryCount () {
        return getNumTries() ;
    }

    /**
     * Gets the request id (invoke identifier) of the current request.
     * @return The request id.
     */
    final public synchronized int getRequestId () {
        return requestId ;
    }

    /**
     * Gets the current status of the request.
     * @return The current status of the request.
     */
    final public synchronized int getRequestStatus() {
        return reqState ;
    }

    /**
     * For Java DMK internal use only.
     */
    final synchronized void setTimeout(int value) {
        timeout = value ;
    }

    /**
     * Indicates whether or not the request was aborted.
     * @return <CODE>true</CODE> if the request was aborted, <CODE>false</CODE> otherwise.
     */
    final public synchronized boolean isAborted() {
        return ((reqState & stAborted) == stAborted) ;
    }

    /**
     * Indicates whether or not the request is in progress.
     * @return <CODE>true</CODE> if the request is in progress, <CODE>false</CODE> otherwise.
     */
    final public synchronized boolean inProgress() {
        return ((reqState & stInProgress) == stInProgress) ;
    }

    /**
     * Indicates whether or not the request result is available.
     * @return <CODE>true</CODE> if the request result is available, <CODE>false</CODE> otherwise.
     */
    final public synchronized boolean isResultAvailable() {
        return (reqState == stResultsAvailable);
    }
        
    /**
     * Gets absolute time in milliseconds (based on epoch time) when the next
     * polling activity will begin.
     * @return The absolute time when polling will begin.
     */
    final public synchronized long getAbsNextPollTime () {
        return nextPollTime ;
    }

    /**
     * Gets absolute time in milliseconds (based on epoch time) before which a
     * response is expected from an agent.
     * @return The absolute time within which a response is expected.
     */
    final public synchronized long getAbsMaxTimeToWait() {
        if (prevPollTime == 0) {
            return System.currentTimeMillis() ;  // should never happen.
        } else {
            return waitTimeForResponse ;
        }
    }

    /**
     * Gets the absolute time (based on epoch time) when the poll started.
     * @return The absolute time the polling started.
     */
    final public synchronized long getPollTimestamp() {
        return prevPollTime ;
    }
    
    /**
     * Gets the session object for this request.
     * @return The session object for this request.
     */
    final public SnmpSession getSnmpSession() {
        return session ;
    }

    /**
     * Gets the peer object for this request.
     * @return The peer object for this request.
     */
    final public SnmpPeer getPeer() {
        return thePeer ;
    }

    /**
     * Gets the session parameter object for this request.
     * @return The session parameter object for this request.
     */
    final public SnmpParameters getParam() {
        return thePeer.getSnmpParam() ;
    }

    /**
     * Gets the SNMP command for this request.
     * @return The SNMP command for this request.
     */
    final public int getCommand() {
        return command ;
    }

    /**
     * Gets the status associated with the <CODE>SnmpVarBindList</CODE>.
     * @return The error status.
     */
    final public synchronized int getErrorStatus() {
        return errorStatus ;
    }

    /**
     * Gets the index.
     * <P>NOTE: this value is equal to the <CODE>errorIndex</CODE> field minus 1. 
     * @return The error index.
     */
    final public synchronized int getErrorIndex() {
        return errorIndex ;
    }

    /**
     * Gets the maximum number of tries before declaring that the peer 
     * is not responding. <CODE>SnmpSet</CODE> operations are never retried.
     * @return The maximum number of times a request should be tried.
     */
    final public int getMaxTries() {
        if (command == pduSetRequestPdu) 
            return 1 ;
        else
            return getPeer().getMaxTries() ;
    }
    
    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #getMaxTries}
     */
    final public int getMaxRetry() {
        return getMaxTries() ;
    }
    
    /**
     * Gets the options set for this request. Certain options are 
     * overridden.
     * @return The options set for this request.
     */
    final public int getOptions() {
        return options ;
    }

    /**
     * Allows the user to configure the behavior of the request when certain errors occur, 
     * such as <CODE>SnmpTooBig</CODE> and <CODE>optFixPduOnError</CODE>. 
     * These should be carefully chosen, as they can improve or degrade performance. 
     * This method can only be set when a request is inactive.
     * @param opt The option as an integer.
     */
    public final synchronized void setOptions(int opt) {
        if (! inProgress()) {
            options = opt ;
            if (!fixPduOnError() || !fixTooBigError()) {
                options &= ~SnmpOptions.optAllowMultiplexing ;
            }
        }
    }

    /**
     * Gets the active <CODE>SnmpVarBindList</CODE>. The contents of it are not guaranteed 
     * to be consistent when the request is active. It is dangerous to modify when the request is active.
     * @return The <CODE>SnmpVarBindList</CODE> when the request was fired.
     */
    public final synchronized SnmpVarBindList getRequestVarBindList() {
        return varBindList ;
    }

    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #getRequestVarBindList}
     */
    public final synchronized SnmpVarbindList getRequestVbList() {
        return new SnmpVarbindList(varBindList) ;
    }

    /**
     * Gets the <CODE>SnmpVarBindList</CODE> that is the response obtained when an agent is queried.
     * It returns a null value if the request is in progress.
     * This ensures accidental manipulation does not occur when a request is in progress. 
     * In case of an error, <CODE>SnmpVarBindList</CODE> is the copy of the original <CODE>SnmpVarBindList</CODE> 
     * at the time of making the request.
     * @return The <CODE>SnmpVarBindList</CODE> returned by the agent or the null value if the request
     * is in progress.
     */
    public final synchronized SnmpVarBindList getResponseVarBindList() {
        if (inProgress())
            return null ;
        return varBindList ;
    }
            
    /**
     * @deprecated As of Java Dynamic Management Kit 4.2, replaced by 
     * {@link #getResponseVarBindList}
     */
    public final synchronized SnmpVarbindList getResponseVbList() {
        if (inProgress())
            return null ;
        return new SnmpVarbindList(varBindList) ;
    }
            
    /**
     * Returns <CODE>true</CODE> if the request should be retried for <CODE>SnmpTooBig</CODE> error.
     * @return <CODE>true</CODE> if the request should be retried for <CODE>SnmpTooBig</CODE> error,
     * <CODE>false</CODE> otherwise.
     */
    public boolean fixTooBigError() {
        // FIXME: for get-bulk we don't fix too-big errors.
        // To we want to keep this behaviour ?
        return (((options & SnmpOptions.optHandleTooBigError) == 
                 SnmpOptions.optHandleTooBigError) &&
                (command != pduGetBulkRequestPdu));
    }

    /**
     * Returns <CODE>true</CODE> if the PDU should be fixed for recoverable error.
     * @return <CODE>true</CODE> if PDU should be fixed for recoverable error,
     * <CODE>false</CODE> otherwise.
     */
    public boolean fixPduOnError() {
        // FIXME: for get-bulk we don't fix errors.
        // To we want to keep this behaviour ?
        return (((options & SnmpOptions.optFixPduOnError) == 
                 SnmpOptions.optFixPduOnError) &&
                (command != pduGetBulkRequestPdu));
    }
    
    /**
     * Checks to see whether this request can be multiplexed with other requests. 
     * <CODE>SnmpSet</CODE> requests cannot be multiplexed.
     * A request already multiplexed cannot be further multiplexed.
     * @return <CODE>true</CODE> if the request is allowed to be multiplexed, <CODE>false</CODE> otherwise.
     */
    final public boolean allowMultiplex() {
        return ((options & SnmpOptions.optAllowMultiplexing)  == 
                SnmpOptions.optAllowMultiplexing ) ;
    }

    /**
     * Indicates whether this request was created internally by the package to handle error conditions.
     * @return <CODE>true</CODE> if this request was created internally, <CODE>false</CODE> otherwise.
     */
    public boolean isInternalRequest() {
        return mode == CREATED_INTERNALLY ;
    }
    
    /**
     * Used in synchronous mode only.
     * Provides a hook that enables a synchronous operation on a previously sent request.  
     * Only one request can be in sync mode on a given thread.
     * The thread that is blocked is notified when the request state reaches completion.  
     * The thread can be a session thread (when done from a callback) or a user thread.  
     * Internally, the blocking mechanism happens differently.
     * If a request is not active, the method returns immediately.
     * The user must get the error status of the request to determine the exact status of the request.
     * 
     * @param time The amount of time to wait. Zero means block until complete.
     * @return <CODE>true</CODE> if the request has completed, <CODE>false</CODE> if it is still active.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    final public boolean waitForCompletion(long time) 
        throws SnmpStatusException {
        if (! inProgress()) 			// check if request is in progress.
            return true ;

        if (this instanceof SnmpPollRequest) {
            throw new SnmpStatusException("Not supported for SnmpPollRequest objects") ; 
        }

        if (session.thisSessionContext()) {
            /* We can manipulate callback safely as we are in session thread.*/
            SnmpRequestHandler savedCallback = callback ;
            callback = null ;
            session.waitForResponse(this, time) ;
            callback = savedCallback ;
        } else {
            /* This is being done from a different thread. so notifyClient will
               do the notification.
            */
            synchronized (this) {
                SnmpRequestHandler savedCallback = callback ;
                try {
                    callback = null ;
                    this.wait(time) ;
                } catch (InterruptedException e) {
                }
                callback = savedCallback ;
            }
        }
		
        return (! inProgress()) ; // true if request completed.
    }

    /**
     * Cancels the active request and removes itself from the polling list.
     * It then marks the session as inactive. This method is not synchronized.
     */
    final public void cancelRequest() {
        if (proxy != null) {
            ReqRedirectSrv tmpproxy = proxy ;
            proxy = null ;
            tmpproxy.cancel(this) ;  // first cancel the proxy object.
        }
        errorStatus = snmpReqAborted ;  
        stopRequest() ;
        deleteRequest() ;
        notifyClient() ;
    }

    /**
     * Notifies the registered client about the completion of an operation.
     */
    final public synchronized void notifyClient() {
        this.notifyAll() ;
    }

    /**
     * Finalizer of the <CODE>SnmpRequest</CODE> objects.
     * This method is called by the garbage collector on an object 
     * when garbage collection determines that there are no more references to the object.
     * <P>Sets all the references to this SNMP request object to <CODE>null</CODE>.
     */
    public void finalize() {
        callback = null ;
        varBindList = null ;
        session = null ;
        thePeer = null ;
        requestPdu = responsePdu = null ;
        internalVarBind = null ;
    }

    /**
     * Gives a status report of the request.
     * @return The status report of the request.
     */
    public synchronized String toString() {
        StringBuffer s = new StringBuffer(300) ;
        s.append(tostring() + " " + getRequestCreationType()) ;
	/* NPCTE fix for bugId 4496037, esc 0, MR 24 August 2001 */
	if (isTraceOn()) {
	    s.append("\n" + thePeer.toString()) ;
	}
	/* end of NPCTE fix for bugId 4496037 */
        s.append(" SnmpCmd = " + SnmpPduPacket.pduTypeToString(command)) ;
        s.append("  RequestOptions = " + options) ;

        return s.toString() ;
    }
    
    /**
     * Returns the <CODE>String</CODE> representation of an error code.
     * @param errcode The error code as an integer.
     * @return The error code as a <CODE>String</CODE>.
     */
    public static String snmpErrorToString(int errcode) {
        switch (errcode) {
        case snmpRspNoError :
            return "noError" ;
        case snmpRspTooBig :
            return "tooBig" ;
        case snmpRspNoSuchName :
            return "noSuchName" ;
        case snmpRspBadValue :
            return "badValue" ;
        case snmpRspReadOnly :
            return "readOnly" ;
        case snmpRspGenErr :
            return "genErr" ;
        case snmpRspNoAccess :
            return "noAccess" ;
        case snmpRspWrongType :
            return "wrongType" ;
        case snmpRspWrongLength :
            return "wrongLength" ;
        case snmpRspWrongEncoding :
            return "wrongEncoding" ;
        case snmpRspWrongValue :
            return "wrongValue" ;
        case snmpRspNoCreation :
            return "noCreation" ;
        case snmpRspInconsistentValue :
            return "inconsistentValue" ;
        case snmpRspResourceUnavailable :
            return "resourceUnavailable" ;
        case snmpRspCommitFailed :
            return "commitFailed" ;
        case snmpRspUndoFailed :
            return "undoFailed" ;
        case snmpRspAuthorizationError :
            return "authorizationError" ;
        case snmpRspNotWritable :
            return "notWritable" ;
        case snmpRspInconsistentName :
            return "inconsistentName" ;
        case snmpReqTimeout :
            return "reqTimeout" ;
        case snmpReqAborted :
            return "reqAborted" ;
        case snmpRspDecodingError :
            return "rspDecodingError" ;
        case snmpReqEncodingError :
            return "reqEncodingError" ;
        case snmpReqPacketOverflow :
            return "reqPacketOverflow" ;
        case snmpRspEndOfTable :
            return "rspEndOfTable" ;
        case snmpReqRefireAfterVbFix :
            return "reqRefireAfterVbFix" ;
        case snmpReqHandleTooBig :
            return "reqHandleTooBig" ;
        case snmpReqTooBigImpossible :
            return "reqTooBigImpossible" ;
        case snmpReqInternalError :
            return "reqInternalError" ;
        case snmpReqSocketIOError :
            return "reqSocketIOError" ;
        case snmpReqUnknownError :
            return "reqUnknownError" ;
        case snmpWrongSnmpVersion :
            return "wrongSnmpVersion" ;
        }
        return "Unknown Error = " + errcode ;
    }
    
    
    // PRIVATE METHODS
    //----------------
    
    private synchronized String tostring() {
        StringBuffer s = new StringBuffer("RequestId = " + requestId) ;
        s.append("   " + "Status = " + statusDescription(reqState)) ;
        s.append("  Timeout/MaxTries/NumTries = " + timeout*numTries + "/" +
                 + getMaxTries() + "/" + numTries) ;

        if (prevPollTime > 0) {
            debugDate.setTime(prevPollTime) ;
            s.append("\nPrevPolled = " + debugDate.toString()) ;
        } else
            s.append("\nNeverPolled") ;
        s.append(" / RemainingTime(millis) = " + 
                 timeRemainingForAction(System.currentTimeMillis())) ;

        return s.toString() ;
    }

    private synchronized void initializeAndFire(long when) {
        requestPdu = responsePdu = null ;
        reason = null ;
        proxy = null ;  // erase previous state. new request starts.  
        if (when == 0) 
            startRequest(System.currentTimeMillis()) ;
        else
            startRequest(when) ;
        setErrorStatusAndIndex(0, 0) ;
    }
    
    /**
     * Calls the user implementation of the <CODE>SnmpRequestHandler</CODE> interface.
     */
    private void handleTimeout() {
        setRequestStatus(stTimeout) ;
        if (isDebugOn()) {
            debug("handleTimeout", "Snmp error/index = " + snmpErrorToString(errorStatus) + "/" +
                  errorIndex + ". Invoking timeout user defined callback...");
        }
        deleteRequest() ;
        notifyClient() ;

        requestPdu = responsePdu = null ;
        internalVarBind = null ;

        try {
            if (callback != null) 
                callback.processSnmpPollTimeout(this) ;
        } catch (Exception e) {  // catch any exception a user might not handle.
            if (isDebugOn()) {
                debug("handleTimeout", "Exception generated by user callback");
                debug("handleTimeout", e);
            }
        } catch (OutOfMemoryError ome) {
            if (isDebugOn()) {
                debug("handleTimeout", "OutOfMemory Error generated by user callback");
                debug("handleTimeout", ome);
            }
            Thread.currentThread().yield();
        }
        return ;
    }
  
    /**
     * Calls the user implementation of the <CODE>SnmpRequestHandler</CODE> interface.
     */
    private void handleError(String msg) {
        setRequestStatus(stResultsAvailable) ;
        if (isDebugOn()) {
            debug("handleError", "Snmp error/index = " + snmpErrorToString(errorStatus) + "/" +
                  errorIndex + ". Invoking error user defined callback...\n" + getVarBindList().varBindListToString());
        }
        
        deleteRequest() ;
        notifyClient() ;
		
        requestPdu = responsePdu = null ;
        internalVarBind = null ;
		
        try {
            if (callback != null)
                callback.processSnmpPollData(this, getErrorStatus(), getErrorIndex(), getVarBindList()) ;
        } catch (Exception e) {  // catch any exception a user might not handle.
            if (isDebugOn()) {
                debug("handleError", "Exception generated by user callback");
                debug("handleError", e);
            }
        } catch (OutOfMemoryError ome) {
            if (isDebugOn()) {
                debug("handleError", "OutOfMemory Error generated by user callback");
                debug("handleError", ome);
            }
            Thread.currentThread().yield();
        }
    }

    /**
     * Constructs a request PDU.
     */
    final private synchronized SnmpPduPacket constructPduPacket() { 
    
        SnmpPduPacket reqpdu = null ;
        Exception excep = null ;
        try {
            if (command == pduGetBulkRequestPdu) {
                SnmpPduBulk bulkPdu = new SnmpPduBulk() ;
                bulkPdu.nonRepeaters = nonRepeaters ;
                bulkPdu.maxRepetitions = maxRepetitions ;
                reqpdu = bulkPdu ;
            }
            else {
                SnmpPduRequest simplePdu = new SnmpPduRequest() ;
                reqpdu = simplePdu ;
                reqpdu.type= command;
            }
            // We allow an SNMP v1 manager to send inform request
            // but the protocol version PDU must be version 2.
            //
            if (command == pduInformRequestPdu) {
                reqpdu.version = snmpVersionTwo ;
            } 
            else {
                reqpdu.version = getParam().getProtocolVersion() ;
            }
            reqpdu.community= getParam().encodeAuthentication(command);
            reqpdu.requestId= getRequestId();
            reqpdu.varBindList=internalVarBind ;

            if (isTraceOn()) {
                trace("constructPduPacket", "Packet built");
            }

        } catch (SnmpStatusException se) {
            excep = se ;
            errorStatus = snmpReqEncodingError ;
            reason = se.getMessage() ;
        } catch (Exception e) {
            excep = e ;
            errorStatus = snmpReqUnknownError ;
            reason = e.getMessage() ;
        } 
        if (excep != null) {
            if (isDebugOn()) {
                debug("constructPduPacket", excep);
            }
            reqpdu = null ;
            queueResponse() ;
        }
        return reqpdu ;
    }

    /**
     * Parses the response packet. If the agent responds with error set, it does not parse any further.
     */
    final private synchronized void parsePduPacket(SnmpPduRequest rpdu) throws SnmpStatusException {
        
        if (rpdu == null)
            return ;
     
        errorStatus = rpdu.errorStatus ;
        errorIndex = rpdu.errorIndex ;

        if (errorStatus == snmpRspNoError) {
            if (command == pduGetBulkRequestPdu)
                updateVarBindList(rpdu.varBindList) ;
            else
                updateInternalVarBindWithResult(command, rpdu.varBindList, rpdu.version);
            responseTimestamp = new javax.management.snmp.Timestamp() ;
            getVarBindList().setTimestamp(responseTimestamp) ;
            return ;
        }
	
        if (errorStatus != snmpRspNoError)  
            --errorIndex ;  // rationalize for index to start with 0.
	
        switch (errorStatus) {
        case snmpRspNoSuchName :
            if (fixPduOnError()) { // try to fix pdu and retry request.
                if (fixVarBindList(internalVarBind, errorIndex)) {
                    if (getVarBindList().checkForUnspecifiedValue()) {
                        // Imdtly refire request without varBind copy.
                        setErrorStatusAndIndex(snmpReqRefireAfterVbFix, 0) ;
                    } else
                        setErrorStatusAndIndex(snmpRspNoError, 0) ;
                } else
                    setErrorStatusAndIndex(snmpReqInternalError, 0) ;
            }
            break ;
        case snmpRspTooBig :
            if (fixTooBigError() && varBindList.size() > 1) {
                // prepare now for handling too-big error.
                setErrorStatusAndIndex(snmpReqHandleTooBig, 0) ;
            }
            break ;
        }
        if (isTraceOn()) {
            trace("parsePduPacket", "received response. ErrorStatus/ErrorIndex = " + errorStatus + "/" + errorIndex);
        }
    }
    
    private boolean handleTooBigError() {
        if (!fixTooBigError()) 
            return false ;
        if (varBindList.size() < 2) {
            setErrorStatusAndIndex(snmpReqTooBigImpossible, 0) ;
            return false ;
        }
        try {
            // prepare now for handling too-big error.
            //
            if (isTraceOn()) {
                trace("handleTooBigError", "Handling Too big error");
            }
            proxy = new SnmpTooBig(this, getVarBindList()) ;
            return true ;
        } catch (Exception e) {
            reason = e.getMessage() ;
            setErrorStatusAndIndex(snmpReqInternalError, 0) ;
        }
        return false ;
    }
    
    private boolean sendPdu() {
        try {
            responseTimestamp = null ;  // set timestamp to null.
            responsePdu = null ;
      
            SnmpPduFactory pduFactory = thePeer.getPduFactory() ;
            SnmpMessage msg = pduFactory.encodePdu(requestPdu, thePeer.getMaxSnmpPktSize()) ;

            if (msg == null) {
                if (isDebugOn()) {
                    debug("sendPdu", "pdu factory returned a null value");
                }
                throw new SnmpStatusException(snmpReqUnknownError) ;
                // This exception will caught hereafter and reported as an snmpReqUnknownError
                // FIXME: may be it's not the best behaviour ?
            }

            int maxPktSize = thePeer.getMaxSnmpPktSize() ;
            byte[] encoding = new byte[maxPktSize] ;
            int encodingLength = msg.encodeMessage(encoding) ;
      
            if (isTraceOn()) {
                trace("sendPdu", "Dump : \n" + msg.printMessage());
            }
            sendPduPacket(encoding, encodingLength) ;
            return true ;
        } catch (SnmpTooBigException ar) {
    
            if (isDebugOn()) {
                debug("sendPdu", ar);
            }
      
            //handle too-big error while generating message.
    
            setErrorStatusAndIndex(snmpReqPacketOverflow, ar.getVarBindCount()) ;
            requestPdu = null ;
            reason = ar.getMessage() ;
            if (isDebugOn()) {
                debug("sendPdu", "Packet Overflow while building Request");
            }
            return handleTooBigError() ;
        } catch (java.io.IOException ioe) {
            setErrorStatusAndIndex(snmpReqSocketIOError, 0) ;
            reason = ioe.getMessage() ;
        } catch (Exception e) {
            if (isDebugOn()) {
                debug("sendPdu", e);
            }
            setErrorStatusAndIndex(snmpReqUnknownError, 0) ;
            reason = e.getMessage() ;
        }
        return false ;
    }

    final private void invokeOnReady() {
        if (requestPdu == null) {
            if (varBindList.size() > getPeer().getVarBindLimit()) {
                if (isDebugOn()) {
                    debug("invokeOnReady", "Attempt too big. varBindList exceeds max allowed for peer. Actual = " + varBindList.size() +
                          " Allowed = " + getPeer().getVarBindLimit());
                }	
                setErrorStatusAndIndex(snmpReqHandleTooBig, 0) ;
                if (!handleTooBigError())
                    queueResponse() ;
                return ;  // go no further.
            }
            requestPdu = constructPduPacket() ;
        }
        if (requestPdu != null) {
            if (sendPdu() == false)
                queueResponse() ;
        }
    }

    /**
     * A copy of request PDU is used.
     */
    final private void invokeOnRetry() {  
        invokeOnReady() ;
    }
    
    final private void invokeOnTimeout() {
        errorStatus = snmpReqTimeout ; 
        queueResponse() ;
    }
    
    final private void queueResponse() {
        /*
          if (isInternalRequest()) {
          try {
          processResponse() ;
          } catch (Exception e) {
          if (isDebugOn()) {
          debug("queueResponse", "Exception processing response to an internal request");
          }
          }
          return ;
          }
        */
        getSnmpSession().addResponse(this) ;
    }

    private void updateInternalVarBindWithResult(int cmd, SnmpVarBind[] list, int version) {
     
        if ((list == null) || (list.length == 0))
            return;
    
        int idx = 0 ;
        switch (cmd) 
            {
            case pduGetRequestPdu :
            case pduSetRequestPdu :
            case pduInformRequestPdu :   
                for(int i=0; i < internalVarBind.length && idx < list.length; i++) {
                    SnmpVarBind avar= internalVarBind[i];
                    if (avar == null)
                        continue;

                    SnmpVarBind res = list[idx];
                    avar.setSnmpValue(res.value);
                    // Update the varBind status.
                    // Fix bug jaw.00608
                    //
                    if (version == SnmpDefinitions.snmpVersionTwo) {
                        if (avar.value.isNoSuchObjectValue())
                            avar.status = SnmpVarBind.stValueNoSuchObject;
                        else if (avar.value.isNoSuchInstanceValue())
                            avar.status = SnmpVarBind.stValueNoSuchInstance;
                        else if (avar.value.isEndOfMibViewValue())
                            avar.status = SnmpVarBind.stValueEndOfMibView;
                    }
                    idx++;
                }
                break;

            case pduGetNextRequestPdu :   
                for(int i=0; i < internalVarBind.length && idx < list.length; i++) {
                    SnmpVarBind avar= internalVarBind[i];
                    if (avar == null)
                        continue;

                    SnmpVarBind res = list[idx];
                    avar.setSnmpValue(res.value);
                    avar.oid= res.oid;
                    // Update the varBind status.
                    // Fix bug jaw.00608
                    //
                    if (version == SnmpDefinitions.snmpVersionTwo) {
                        if (avar.value.isNoSuchObjectValue())
                            avar.status = SnmpVarBind.stValueNoSuchObject;
                        else if (avar.value.isNoSuchInstanceValue())
                            avar.status = SnmpVarBind.stValueNoSuchInstance;
                        else if (avar.value.isEndOfMibViewValue())
                            avar.status = SnmpVarBind.stValueEndOfMibView;
                    }
                    idx++;
                }
                break;

            case pduGetBulkRequestPdu : // Bulk processing does not call this method.
            default :
                break;
            }                
    }
  
    private void updateVarBindList(SnmpVarBind[] list) {
    
        varBindList = new SnmpVarBindList() ;
        for(int i=0; i < list.length; i++) {
            SnmpVarBind bind= list[i];
            SnmpVarBind var = new SnmpVarBind(bind.oid, bind.value) ;
            varBindList.addVarBind(var) ;
        }
    }
  
    /**
     * This method submits the request for polling and marks the request active.
     * It does nothing if the request is already active.
     * The poll will be scheduled to happen immediately.
     * @param starttime The start time for polling.
     */
    private synchronized void startRequest(long starttime) {
        nextPollTime = starttime ;
        prevPollTime = 0 ;
        schedulePoll() ;
    }
    
    /**
     * This method creates a new request ID. The ID is submitted to the poll server for scheduling.
     */
    private void schedulePoll() {
        numTries = 0 ;
        initNewRequest() ;
        setRequestStatus(stWaitingToSend) ;
        SnmpQManager.getTheInstance().addRequest(this) ;
    }

    // PACKAGE METHODS
    //----------------
    
    /**
     * Indicates that the request has been created internally be the package. 
     */
    final void setInternalRequest() {
        // In this situation the callback automatically also is of type
        // ReqRedirectSrvIf.
        mode = CREATED_INTERNALLY ;  
    }
    
    /**
     * Marks this request as being created by an internal handler or as being handled 
     * by an internal handler. 
     * For example, handling <CODE>SnmpTooBig</CODE> errors or multiplexing.
     * Note that if the snmp callback is equal to the proxy, it means that this request 
     * was created by the internal mechanism.
     * @param obj The proxy object.
     */
    final void setProxyObject(ReqRedirectSrv obj) {
        proxy = obj ;
    }
    
    /**
     * This method cancels an active request and removes it from the polling list.
     */
    void stopRequest() {
        
        // Remove the clause synchronized of the stopRequest method.
        // Synchronization is isolated as possible to avoid thread lock.
        // Note: the method removeRequest from SendQ is synchronized.
        // fix bug jaw.00392.B
        //
        synchronized(this) {
            setRequestStatus(stAborted) ;
        }
        SnmpQManager.getTheInstance().removeRequest(this) ;
        synchronized(this) {
            requestId = 0 ;
        }
    }
    
    /**
     * This method determines whether the request is to be retried. This is used if the peer 
     * did not respond to a previous request. If the request exceeds the retry limit, a timeout is signaled.
     */
    void action() {
        if (inProgress() == false)
            return ;
        while (true) {
            try {
                if (numTries == 0) {
                    invokeOnReady() ;
                } else if (numTries < getMaxTries()) {
                    invokeOnRetry() ;
                } else {
                    invokeOnTimeout() ;
                }
                return ;
            } catch (OutOfMemoryError omerr) {
                // Consider it as a try !
                //
                numTries++;
                if (isDebugOn()) {
                    debug("action", "Request hit out of memory situation...");
                }
                Thread.currentThread().yield();
            }
        }
    }

    /**
     * Starts a request in asynchronous mode. The callback interface
     * is used to notify the user upon request completion. 
     * @param vblst The <CODE>SnmpVarBindList</CODE> to be used.
     * @param when The absolute time in milliseconds when this request should start.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    synchronized void start(SnmpVarBindList vblst, long when) throws SnmpStatusException {
        start(vblst, true, when) ;
    }
    
    /**
     * For Java DMK internal use only.
     * Starts a request in asynchronous mode. The callback interface
     * is used to notify the user upon request completion.  
     * For most users this should be set to <CODE>true</CODE>.
     * @param vblst The <CODE>SnmpVarBindList</CODE> to be used.
     * @param copyvb Whether the <CODE>vblst</CODE> should be copied. 
     * @param when The absolute time in milliseconds when this request should start.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    synchronized void start(SnmpVarBindList vblst, boolean copyvb, long when) throws SnmpStatusException {
        if (inProgress())
            throw  new SnmpStatusException("Request already in progress.") ;
        switch (getCommand()) {
        case pduSetRequestPdu :  
            if (getParam().getProtocolVersion() == snmpVersionTwo)
                options = 0 ;  // no options for an atomic set operation.
            if (copyvb)
                setVarBindList((SnmpVarBindList) vblst.clone()) ;
            else
                setVarBindList(vblst) ;
            break ;

        case pduGetRequestPdu : 
        case pduGetNextRequestPdu :  
        case pduGetBulkRequestPdu :  
            if (copyvb)
                setVarBindList((SnmpVarBindList)vblst.cloneWithoutValue());
            else
                setVarBindList(vblst) ;
            break ;
            
        case pduInformRequestPdu :  
            // The clone operation has been performed in SnmpSession.snmpInform() method.
            setVarBindList(vblst) ;
            break ;
            
        default :
            throw  new SnmpStatusException("Unsupported Command." +
                                           "  Should never happen. ??!! " + getCommand()) ;
        }

        initializeAndFire(when) ;
    }

    /**
     * Calls the user implementation of the <CODE>SnmpRequestHandler</CODE> interface.
     */
    void handleSuccess() {
        setRequestStatus(stResultsAvailable) ;

        if (isTraceOn()) {
            trace("handleSuccess", "Invoking user defined callback...");
        }

        if (!(this instanceof SnmpPollRequest))
            deleteRequest() ;  // delete only non-poll request.
        notifyClient() ;
		
        requestPdu = responsePdu = null ;
        internalVarBind = null ;

        try {  // catch all user exception which may happen in callback.
            if (callback != null)
                callback.processSnmpPollData(this, errorStatus, errorIndex, getVarBindList()) ;
        } catch (Exception e) {
            if (isDebugOn()) {
                debug("handleSuccess", "Exception generated by user callback");
                debug("handleSuccess", e);
            }
        } catch (OutOfMemoryError ome) {
            if (isDebugOn()) {
                debug("handleSuccess", "OutOfMemory Error generated by user callback");
                debug("handleSuccess", ome);
            }
            Thread.currentThread().yield();
        }
        return ;
    }

    /**
     * Calls the user implementation of the <CODE>SnmpRequestHandler</CODE> interface.
     */
    void handleInternalError(String msg) {
        setRequestStatus(stInternalError) ;
        if (reason == null)
            reason = msg ;

        if (isDebugOn()) {
            debug("handleInternalError", "Snmp error/index = " + snmpErrorToString(errorStatus) + "/" +
                  errorIndex + ". Invoking internal error user defined callback...\n" + getVarBindList().varBindListToString());
        }

        deleteRequest() ;
        notifyClient() ;

        requestPdu = responsePdu = null ;
        internalVarBind = null ;
		
        try {
            if (callback != null) 
                callback.processSnmpInternalError(this, reason) ;
        } catch (Exception e) {  // catch any exception a user might not handle.
            if (isDebugOn()) {
                debug("handleInternalError", "Exception generated by user callback");
                debug("handleInternalError", e);
            }
        } catch (OutOfMemoryError ome) {
            if (isDebugOn()) {
                debug("handleInternalError", "OutOfMemory Error generated by user callback");
                debug("handleInternalError", ome);
            }
            Thread.currentThread().yield();
        }
    }

    /**
     * For Java DMK internal use only.
     * You should specify the varBindList at SnmpRequest creation time.
     * You cannot modify it during the life-time of the object.
     */
    final synchronized void setVarBindList(SnmpVarBindList newvblst) {
        varBindList = newvblst ;
        if (internalVarBind == null || internalVarBind.length != varBindList.size()) {
            internalVarBind = new SnmpVarBind[varBindList.size()] ;
        }
        varBindList.copyInto(internalVarBind) ;
    }

    final synchronized boolean fixVarBindList(SnmpVarBind[] vararray, int idx) {

        int i ;
        int saveIdx = idx ;
        SnmpVarBind avar = null ;

        if (isTraceOn()) {
            trace("fixVarBindList", "Fix Pdu at index " + idx);
        }
        
        for (i = 0; i < vararray.length ; i++) {
            if (vararray[i] != null) {
                --idx ;
                if (idx < 0) break ;
            }
        }
        if (i >= vararray.length) {
            reason = "Can't find variable binding with idx = " + saveIdx ;
            if (isDebugOn()) {
                debug("fixVarBindList", reason);
            }
            return false ;
        }

        avar = vararray[i] ;
        vararray[i] = null ;

        switch (getCommand()) {
        case pduGetRequestPdu :  
            avar.status = SnmpVarBind.stValueNoSuchInstance ;
            break ;
        case pduGetNextRequestPdu :  
            avar.status = SnmpVarBind.stValueEndOfMibView ;
            break ;
        }
        return true ;
    }

    /**
     * For Java DMK internal use only.
     * @exception SnmpStatusException An error occurred while accessing a MIB node.
     */
    final void processResponse() throws SnmpStatusException {
    
        if (isTraceOn()) {
            trace("processResponse", "errstatus = " + errorStatus);
        }

        if (inProgress() == false) {  // check if this request is still alive.
            responsePdu = null ;
            return ;  // the request may have  cancelled.
        }

        if (errorStatus >= snmpReqInternalError) {
            handleInternalError("Internal Error...") ;
            return ;
        }

        try {
            parsePduPacket(responsePdu) ;
            responsePdu = null ;

            // At this point the errorIndex is rationalized to start with 0.
            switch (errorStatus) {
            case snmpRspNoError : 
                handleSuccess() ;
                return ;
            case snmpReqTimeout : 
                handleTimeout() ;
                return ;
            case snmpReqInternalError : 
                handleInternalError("Unknown internal error.  deal with it later!") ;
                return ;
            case snmpReqHandleTooBig :
                setErrorStatusAndIndex(snmpRspTooBig, 0) ;
                if (handleTooBigError())
                    return ;
                handleError("Cannot handle too-big situation...") ;
                return ;
            case snmpReqRefireAfterVbFix :
                // Refire request after fixing varBindList.
                initializeAndFire(0) ;
                return ;
            default :
                if ((command != pduSetRequestPdu) && (command != pduGetBulkRequestPdu))
                    translateErrorIndex();
                handleError("Error status set in packet...!!") ;
                return ;
            }
        } catch (SnmpStatusException se) {
            errorStatus = snmpRspDecodingError ;
            reason = se.getMessage() ;
        } catch (Exception e) {
            if (isDebugOn()) {
                debug("processResponse", e);
            }
            reason = e.getMessage() ;
        } 
        handleInternalError(reason) ;
    }

    /**
     * For Java DMK internal use only.
     */
    final synchronized void setErrorStatusAndIndex(int stat, int idx) {
        errorStatus = stat ;
        errorIndex = idx ;
    }
    
    /**
     * For Java DMK internal use only.
     */
    final void invokeOnResponse(Object resp) {
        if (resp != null) {
            if (resp instanceof SnmpPduRequest) 
                responsePdu = (SnmpPduRequest) resp ;
            else 
                return ;
        }
        setRequestStatus(stReceivedReply) ;
        queueResponse() ;
    }

    final synchronized void deleteRequest() {
        session.deleteRequest(this) ;
    }

    /**
     * For Java DMK internal use only.
     * To access the <CODE>SnmpVarBindList</CODE> associated with this request,
     * use SnmpRequest#getRequestVarBindList or SnmpRequest#getResponseVarBindList.
     * @return The <CODE>SnmpVarBindList</CODE>.
     */
    final synchronized SnmpVarBindList getVarBindList() {
        return varBindList ;
    }

    /**
     * Sends the prepared PDU packet to the peer and updates the data structure
     * to expect a response. It acquires a lock on the socket to prevent a case
     * where a response arrives before this thread could insert the request into the wait queue.
     * @exception IOException Signals that an I/O exception of some sort has occurred.
     */
    final void sendPduPacket(byte[] buffer, int length) throws java.io.IOException {
    
        if (isTraceOn()) {
            trace("sendPduPacket", "Send to peer. " + thePeer.toString() + "Length = " +  length +
                  "\nDump : \n" + SnmpMessage.dumpHexBuffer(buffer,0, length));
        }
        SnmpSocket theSocket = getSnmpSession().getSocket() ;
        synchronized (theSocket) {
            theSocket.sendPacket(buffer, length, thePeer.getDestAddr(), thePeer.getDestPort()) ;
            setRequestSentTime(System.currentTimeMillis()) ;
        }
    }

    /**
     * Gets a short description of how and who created the request.
     * You can use this to check the history of how a request is processed.
     * @return A brief info about how this request came into existence.
     */
    final synchronized String getRequestCreationType() {
        switch (mode) {
        case CREATED_BY_USER :	
            return "UserCreated." ;
        case CREATED_INTERNALLY :
            /* callback points to the object who created this request. */
            return "Internal_" + callback.getClass().getName() ;
        default :
            return "UnknownRequest????" ;
        }
    }
  
    /**
     * Translates an error index to the correct index, after taking care
     * of SNMP exceptions that may have occurred and fixing problems with the PDU.
     */
    final void translateErrorIndex() {
        if (!fixPduOnError())
            return ;                // do nothing if the pdu fix was disabled.

        SnmpVarBindList vblst = getVarBindList() ;
        int max= vblst.size();
        for (int i = 0; i < max ; i++) {
            if (!vblst.getVarBindAt(i).hasVarBindException()) {
                --errorIndex ;
                if (errorIndex == 0) {
                    errorIndex = i ;
                    return ;
                }
            }
        }
    }

    /**
     * For Java DMK internal use only.
     */
    final synchronized void setPreviousPollTimestamp(long prev) {
        prevPollTime = prev ;
    }

    final  void setRequestSentTime(long sendtime) {
        numTries++;
        setPreviousPollTimestamp(sendtime) ;
        waitTimeForResponse = prevPollTime + timeout ;
        setRequestStatus(stWaitingForReply) ;
    
        if (isTraceOn()) {
            trace("setRequestSentTime", "Request Successfully sent");
        }
        SnmpQManager.getTheInstance().addWaiting(this) ;
    }

    /**
     * Initializes the request id from the request counter.
     */
    final synchronized void initNewRequest() {
        requestId = requestCounter.getNewId() ;
    }

    /**
     * For Java DMK internal use only.
     */
    long timeRemainingForAction(long currtime) {
        switch (reqState) {
        case stWaitingToSend :
            return nextPollTime - currtime ;
        case stWaitingForReply :
            return waitTimeForResponse - currtime ;
        default :
            return -1 ;
        }
    }

    /**
     * Returns the string state corresponding to the specified integer state.
     * @param state The integer state.
     * @return The string state.
     */
    final static String statusDescription(int state) {
        switch (state) {
        case stWaitingToSend :
            return "Waiting to send." ;
        case stWaitingForReply :
            return "Waiting for reply." ;
        case stReceivedReply :
            return "Response arrived." ;
        case stAborted  :
            return "Aborted by user." ;
        case stTimeout :
            return "Timeout Occured." ;
        case stInternalError :
            return "Internal error." ;
        case stResultsAvailable :
            return "Results available" ;
        case stNeverUsed :
            return "Request in createAndWait state" ;
        }
        return "Unknown Request state." ;
    }

    /**
     * Sets the request status to the specified value.
     * @param reqst The new status request.
     */
    final synchronized void setRequestStatus(int reqst) {
        reqState = reqst ;
    }
}
