/*
 * @(#)file      SnmpQManager.java
 * @(#)author    Sun Microsystems, Inc.
 * @(#)version   4.15
 * @(#)date      01/08/24
 *
 * Copyright 2000 Sun Microsystems, Inc. All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2000 Sun Microsystems, Inc. Tous droits rservs.
 * Ce logiciel est propriet de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation. 
 *
 */
// Copyright (c) 1995-96 by Cisco Systems, Inc.

package javax.management.snmp.manager;



import java.util.Vector;
import java.net.InetAddress;
import java.io.Serializable;

import javax.management.snmp.SnmpPduFactory;

// import debug stuff
//
import com.sun.jdmk.Trace;

/**
 * This class implements a server queue manager.  This class is for
 * internal use.
 *
 * @version     4.15     08/23/01
 * @author      Sun Microsystems, Inc
 * @author      Cisco Systems, Inc.
 */


final class SnmpQManager implements Serializable {

    final static long pollMargin = 900 ; /* some milli seconds */

    public static synchronized SnmpQManager getTheInstance() {
        if (theSnmpq == null)
            theSnmpq = new SnmpQManager();
        return theSnmpq ;
    }

    private SnmpQManager () {
        nbSessions = 0;
        newq = new SendQ (20, 5) ;
        waitq = new WaitQ (20, 5) ;

        queueThreadGroup = new ThreadGroup("Qmanager Thread Group") ;
    
        // TIME BOMB HERE
        startQThreads() ;
    }

    public void startQThreads() {
        if (timerQThread == null || timerQThread.isAlive() == false) {
            timerQThread   = new SnmpTimerServer(queueThreadGroup, this) ;
        }
        if (requestQThread == null || requestQThread.isAlive() == false) {
            requestQThread = new SnmpSendServer(queueThreadGroup, this) ;
        }
    }

    /* NPCTE fix for bug 4468685, esc 530909 MR 13-June-2001 */
    public void finalize() {
	queueThreadGroup.destroy();
    }
    /* end of NPCTE fix for bugid 4468685 */

    public void stopQThreads() {
        if (timerQThread != null && timerQThread.isAlive() == true) {   
            ((SnmpTimerServer)timerQThread).stopTimerServer();
        }
        waitq = null;
        timerQThread = null;
        
        if (requestQThread != null && requestQThread.isAlive() == true) {            
            ((SnmpSendServer)requestQThread).stopSendServer();
        }
        newq = null;
        requestQThread = null;
        
        theSnmpq = null;
    }
    
    public String statusReport() {
        return new String("Requests outstanding/waiting : " + newq.size() + 
                          "/" + waitq.size() + " Send/timer servers : " +
                          requestQThread.isAlive() + "/" + timerQThread.isAlive()) ;
    }


    public void addRequest(SnmpRequest reqc) {
        newq.addRequest(reqc) ;
        return ;
    }

    public void addWaiting (SnmpRequest reqc) {
        waitq.addWaiting(reqc) ;
        return ;
    }

    public Vector getAllOutstandingRequest(long range) {
        return newq.getAllOutstandingRequest(range) ;
    }

    public SnmpRequest getOutstandingRequest() {
        return newq.getOutstandingRequest() ;
    }

    public SnmpRequest getTimeoutRequests() {
        return waitq.getTimeoutRequests() ;
    }

    public void removeRequest(SnmpRequest reqc) {
        newq.removeElement(reqc) ;
        waitq.removeElement(reqc) ;
    }

    public SnmpRequest removeRequest(long reqid) {
        SnmpRequest reqc = null ;

        if ((reqc = newq.removeRequest(reqid)) == null) 
            reqc = waitq.removeRequest(reqid) ;
	
        return reqc ;
    }
  
    public SnmpPduFactory findPduFactory(InetAddress address, int port) {
        SnmpPduFactory factory = newq.findPduFactory(address, port);
        if (factory == null)
            factory = waitq.findPduFactory(address, port) ;
        return factory ;
    }

    public String listQContents() {
        try {
            long time = System.currentTimeMillis() ;
            StringBuffer s = new StringBuffer("******* Qdump begins ****\n" );
            s.append(newq.printAllRequest(time) + "\n") ;
            s.append(waitq.printAllRequest(time) + "\n") ;
            s.append("********* Qdump ends ***********\n");
            return s.toString() ;
        } catch (Exception e) {
            if (isDebugOn()) {
                debug("listQContents", e);
            }
        }
        return null ;
    }
    
    synchronized void incNbSessions() {
        nbSessions++;
    }

    synchronized void decNbSessions() {
        if (--nbSessions == 0) {
            ((SnmpTimerServer)timerQThread).isBeingDestroyed = true;
            waitq.isBeingDestroyed = true;
            ((SnmpSendServer)requestQThread).isBeingDestroyed = true;
            newq.isBeingDestroyed = true;
            stopQThreads();
        }
    }
    
    // TRACES & DEBUG
    //---------------
    
    static boolean isTraceOn() {
        return Trace.isSelected(Trace.LEVEL_TRACE, Trace.INFO_SNMP);
    }

    static void trace(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_TRACE, Trace.INFO_SNMP, clz, func, info);
    }

    static void trace(String func, String info) {
        SnmpQManager.trace(dbgTag, func, info);
    }
    
    static boolean isDebugOn() {
        return Trace.isSelected(Trace.LEVEL_DEBUG, Trace.INFO_SNMP);
    }

    static void debug(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, info);
    }

    static void debug(String clz, String func, Throwable exception) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_SNMP, clz, func, exception);
    }
    
    static void debug(String func, String info) {
        SnmpQManager.debug(dbgTag, func, info);
    }
    
    static void debug(String func, Throwable exception) {
        SnmpQManager.debug(dbgTag, func, exception);
    }
    
    static String dbgTag = "SnmpQManager";
    
    private SendQ  newq ;
    private WaitQ  waitq ;

    private static SnmpQManager theSnmpq = new SnmpQManager() ;

    private ThreadGroup queueThreadGroup = null ;
    private Thread requestQThread = null ;
    private Thread timerQThread = null ;

    private int nbSessions; // The number of sessions currently linked to this SnmpQManager
}

/**
 * This vector manages the requests to be sent to the agent.
 */
class SendQ extends Vector {
    
    final  long sendQ_pollMargin = 0 ; /* some milli seconds */

    SendQ(int initialCapacity, int capacityIncr) {
        super(initialCapacity , capacityIncr) ;
    }

    public synchronized String printAllRequest(long time) {
        if (isEmpty()) 
            return "------- SendQ empty......." ;
        int max= size();
        StringBuffer str = new StringBuffer("\n------------------------ Requests in SendQ -> " + max) ;
        for (int i = 0 ; i < max; i++) {
            SnmpRequest req = getRequestAt(i) ;
            str.append("\n" + i + ". (" + req.toString() + 
                       " RemainingTimeToSend = " + req.timeRemainingForAction(time) +
                       "  ) ................................\n") ;
        }
        return str.toString() ;
    }

    private synchronized void notifyClients() {
        this.notifyAll() ;
    }

    public synchronized void addRequest(SnmpRequest req) {
                
        long nextPoll = req.getAbsNextPollTime() ;

        int i ;
        for (i = size() ; i > 0 ; i--) {
            if (nextPoll < getRequestAt(i-1).getAbsNextPollTime())
                break ;
        }
        if (i == size()) {
            addElement(req) ;
            notifyClients() ;
        } else
            insertElementAt(req, i) ;
        return ;
    }

    public synchronized SnmpRequest getOutstandingRequest() {
        waitUntilReady() ;
        SnmpRequest req = (SnmpRequest) lastElement() ;
        elementCount-- ;
        return req ;
    }

    synchronized void introduceSleep(long tm) {
        long origtm = System.currentTimeMillis() ;
        while (true) {
            try {
                this.wait(tm) ;
                if ((System.currentTimeMillis() - origtm) < tm) 
                    continue ;
                return ;
            } catch(Exception e) {
            }
        }
    }

    public synchronized boolean waitUntilReady() {
        while (true) {
            
            if (isBeingDestroyed == true) {
                return false;
            }
            
            long tmp = 0 ;
            if (isEmpty() == false) {
                long currTime = System.currentTimeMillis() ;
                SnmpRequest req = (SnmpRequest) lastElement() ;
                tmp = req.getAbsNextPollTime() - currTime ;
                if (tmp <= sendQ_pollMargin) {
                    return true ;
                }
            }
            waitOnThisQueue(tmp) ;
        }
    }

    public synchronized Vector getAllOutstandingRequest(long margin) {
        int i ;
        Vector outreq = new Vector() ;
        while (true) {
            if (waitUntilReady() == true) {

                long refTime = System.currentTimeMillis() + margin ;

                for (i = size() ; i > 0 ; i--) {
                    SnmpRequest req = getRequestAt(i-1) ;
                    /*
                      System.err.println("nextpoll = " +
                      req.getAbsNextPollTime() + " ReferenceTime = " + refTime
                      + " diff = " + 
                      (req.getAbsNextPollTime() - refTime ));
                    */
                    if (req.getAbsNextPollTime() > refTime) 
                        break ;
                    outreq.addElement(req) ;
                }
			
                if (! outreq.isEmpty()) {
                    elementCount -= outreq.size() ;
                    return outreq ;
                }
                //introduceSleep(1000) ;
            }
            else
                return null;
        }
    }

    public synchronized void waitOnThisQueue(long time) {
        if (time == 0 && !isEmpty()) {
            if (SnmpQManager.isDebugOn()) {
                SnmpQManager.debug("waitOnThisQueue", "[" + Thread.currentThread().toString() + "]:" +
                                   "Fatal BUG :: Blocking on newq permenantly. But size = " + size());
            }
        }

        try {
            this.wait(time) ;
        } catch (InterruptedException e) {
        }
    }

    public SnmpRequest getRequestAt(int idx) {
        return (SnmpRequest)elementAt(idx) ;
    }
    
    public synchronized SnmpRequest removeRequest(long reqid) {
        int max= size() ;
        for (int i = 0 ; i < max ; i++) {
            SnmpRequest reqc = getRequestAt(i) ;
            if (reqid == reqc.getRequestId()) {
                removeElementAt(i) ;
                return reqc ;
            }
        }
        return null ;
    }

    public synchronized SnmpPduFactory findPduFactory(InetAddress address, int port) {
        SnmpPduFactory result = null ;
        int max = size() ;
        int i = 0 ;
        while ((i < max) && (result == null)) {
            SnmpRequest reqc = getRequestAt(i) ;
            SnmpPeer peer = ((SnmpRequest)reqc).getPeer() ;
            if ((peer.getDestAddr().equals(address)) && 
                (peer.getDestPort() == port)) {
                result = peer.getPduFactory() ;    
            }        
            i++ ;
        }
        
        return result ;
    }


    // This boolean is used to stop handling requests while the corresponding SnmpQManager
    // is being destroyed.
    //
    boolean isBeingDestroyed = false;
}

/**
 * This vector manages the requests to be retried to the agent.
 */
class WaitQ extends Vector {
    
    WaitQ(int initialCapacity, int capacityIncr) {
        super(initialCapacity , capacityIncr) ;
    }

    public synchronized String printAllRequest(long time) {
        if (isEmpty()) 
            return "------- WaitQ empty......." ;
        StringBuffer str = 
            new StringBuffer("\n------------------------ Requests in WaitQ -> " +
                             size()) ;
        int max= size();
        for (int i = 0 ; i < max ; i++) {
            SnmpRequest req = getRequestAt(i) ;
            str.append("\n" + i + ". (" + req.toString() + 
                       " RemainingTimeToSend = " + req.timeRemainingForAction(time) +
                       "  ) ................................\n") ;
        }
        return str.toString() ;
    }

    public synchronized void addWaiting(SnmpRequest req) {
        
        long waitTime = req.getAbsMaxTimeToWait() ;
        int i ;
        for (i = size() ; i > 0 ; i--) {
            if (waitTime < getRequestAt(i-1).getAbsMaxTimeToWait())
                break ;
        }
        if (i == size()) {
            addElement(req) ;
            notifyClients() ;
        } else
            insertElementAt(req, i) ;
        return ;
    }

    public synchronized boolean waitUntilReady() {
        while (true) {
            
            if (isBeingDestroyed == true)
                return false;
            
            long tmp = 0 ;
            if (isEmpty() == false) {
                long currTime = System.currentTimeMillis() ;
                SnmpRequest req = (SnmpRequest) lastElement() ;
                tmp = req.getAbsMaxTimeToWait() - currTime ;
                if (tmp <= 0) {
                    return true ;
                }
            }
            waitOnThisQueue(tmp) ;
        }
    }

    public synchronized SnmpRequest getTimeoutRequests() {
        if (waitUntilReady() == true) {
            SnmpRequest req = (SnmpRequest) lastElement() ;
            elementCount-- ;
            return req ;
        }
        else
            return null;
    }

    private synchronized void notifyClients() {
        this.notifyAll() ;
    }

    public synchronized void waitOnThisQueue(long time) {
        if (time == 0 && !isEmpty()) {
            if (SnmpQManager.isDebugOn()) {
                SnmpQManager.debug("waitOnThisQueue", "[" + Thread.currentThread().toString() + "]:" +
                                   "Fatal BUG :: Blocking on waitq permenantly. But size = " + size());
            }
        }
				
        try {
            this.wait(time) ;
        } catch (InterruptedException e) {
        }
    }

    public SnmpRequest getRequestAt(int idx) {
        return (SnmpRequest)elementAt(idx) ;
    }

    public synchronized SnmpRequest removeRequest(long reqid) {
        int max= size();
        for (int i = 0 ; i < max ; i++) {
            SnmpRequest reqc = getRequestAt(i) ;
            if (reqid == reqc.getRequestId()) {
                removeElementAt(i) ;
                return reqc ;
            }
        }
        return null ;
    }
    

    public synchronized SnmpPduFactory findPduFactory(InetAddress address, int port) {
        SnmpPduFactory result = null ;
        int max = size() ;
        int i = 0 ;
        while ((i < max) && (result == null)) {
            SnmpRequest reqc = getRequestAt(i) ;
            if (reqc instanceof SnmpRequest) {
                SnmpPeer peer = ((SnmpRequest)reqc).getPeer() ;
                if ((peer.getDestAddr().equals(address)) && 
                    (peer.getDestPort() == port)) {
                    result = peer.getPduFactory() ;    
                }
            }
            i++ ;
        }
      
        return result ;
    }

    // This boolean is used to stop handling requests while the corresponding SnmpQManager
    // is being destroyed.
    //
    boolean isBeingDestroyed = false;
    
}
