/*
 * @(#)file      DiscoveryMonitor.java
 * @(#)author    Sun Microsystems, Inc.
 * @(#)version   4.21
 * @(#)date      01/09/11
 *
 * Copyright 2000 Sun Microsystems, Inc. All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 * Copyright 2000 Sun Microsystems, Inc. Tous droits rservs.
 * Ce logiciel est propriet\351 de Sun Microsystems, Inc.
 * Distribu\351 par des licences qui en restreignent l'utilisation.
 *
 */

package com.sun.jdmk.discovery;

// ----------------------------------------------------------
// java import
// ----------------------------------------------------------
import java.util.*;
import java.io.*;
import java.lang.*;
import java.net.*;

// ----------------------------------------------------------
// jdmk import
// ----------------------------------------------------------
import javax.management.*;
import com.sun.jdmk.* ; 


 /**
 * Describe an MBean that listens for registering and deregistering information sent by
 * {@link com.sun.jdmk.discovery.DiscoveryResponder} objects on a given multicast group.
 * Any agent that is to use multicast discovery must have a
 * {@link com.sun.jdmk.discovery.DiscoveryResponder} registered in its MBean server.
 * When a {@link com.sun.jdmk.discovery.DiscoveryResponder} is registered in an MBean server and when its start or stop methods
 * are called, it informs the rest of the multicast group by sending
 * a multicast message. The format of this message is not exposed.
 * Whenever a <CODE>DiscoveryMonitor</CODE> receives a registration or
 * deregistration message, it sends a {@link com.sun.jdmk.discovery.DiscoveryResponderNotification}
 * to its notification listener.
 * <p>
 * A <CODE>DiscoveryMonitor</CODE> can be instantiated either in stand alone
 * mode (Client side) or added to an MBean Server. In the first case, the client should
 * call the appropriate constructor to initialize the <CODE>multicastGroup</CODE>
 * and <CODE>multicastPort</CODE> parameters.
 * The default values for the group and the port are 224.224.224.224 and
 * 9000. 
 *
 * A <CODE>DiscoveryMonitor</CODE> can be stopped by calling the <CODE>stop</CODE> method. When it is stopped, the
 * <CODE>DiscoveryMonitor</CODE> no longer listens for registering and
 * deregistering messages from {@link com.sun.jdmk.discovery.DiscoveryResponder} objects.
 * A <CODE>DiscoveryMonitor</CODE> can be restarted by invoking the <CODE>start</CODE> method.
 * <p>   
 * A <CODE>DiscoveryMonitor</CODE> has a <CODE>state</CODE> property which reflects its
 * activity.
 * <TABLE> 
 * <TR><TH>DiscoveryMonitor</TH>                 <TH>State</TH></TR>
 * <TR><TD><CODE>running</CODE></TD>          <TD><CODE>ONLINE</CODE></TD></TR>
 * <TR><TD><CODE>stopped</CODE></TD>          <TD><CODE>OFFLINE</CODE></TD></TR>
 * <TR><TD><CODE>stopping</CODE></TD>         <TD><CODE>STOPPING</CODE></TD></TR>
 * <TR><TD><CODE>starting</CODE></TD>         <TD><CODE>STARTING</CODE></TD></TR>
 * </TABLE>
 * <p>    
 * The transition between <CODE>ONLINE</CODE> and <CODE>OFFLINE</CODE> may not
 * be immediate. The <CODE>DiscoveryMonitor</CODE> may need some time to finish
 * or interrupt the active requests. During this time the state of the
 * <CODE>DiscoveryMonitor</CODE> is <CODE>STOPPING</CODE>.
 * When a <CODE>DiscoveryMonitor</CODE> is removed from a Java DMK agent, it is automatically stopped.
 *
 * @version     4.21     09/11/01
 * @author      Sun Microsystems, Inc
 */

public class DiscoveryMonitor extends NotificationBroadcasterSupport implements java.io.Serializable, DiscoveryMonitorMBean, MBeanRegistration {

    // ----------------------------------------------------------
    // States of an DiscoveryMonitor
    // ----------------------------------------------------------
    /** Marks the "state" property as running. */
    public static final int ONLINE = 0 ;
    /** Marks the "state" property as stopped. */
    public static final int OFFLINE = 1 ;
    /** Marks the "state" property as in-transition from ONLINE to OFFLINE. */
    public static final int STOPPING = 2 ;
    /** Marks the "state" property as in-transition from OFFLINE to ONLINE. */
    public static final int STARTING = 3 ;

    // ----------------------------------------------------------
    // Constructor
    // ----------------------------------------------------------

    /**
     * Constructs a <CODE>DiscoveryMonitor</CODE>.
     * <P>
     * This constructor creates and initializes a 
     * multicast socket used to listen for {@link com.sun.jdmk.discovery.DiscoveryResponder}
     * objects registering or deregistering.
     * The default group (224.224.224.224) and port (9000) are used.
     * No check is done on the default values: check will be performed by the start method.
     *
     */
    public DiscoveryMonitor () {
        // ------------------------
        // set Default multicastPort/multicastGroup
        // ------------------------
        this.multicastGroup = defaultMulticastGroup ;
        this.multicastPort  = defaultMulticastPort ;
	if (isTraceOn()) {
		trace("constructor " , "Set group to '" + multicastGroup + "'" );
		trace("constructor " , "Set port  to '" + multicastPort + "'" );
	}
    }

    /**
     * Constructs a <CODE>DiscoveryMonitor</CODE>.
     * <P>
     * This constructor creates and initializes a 
     * multicast socket used to listen for {@link com.sun.jdmk.discovery.DiscoveryResponder}
     * objects registering or deregistering.
     * No check is done on the parameter values: check will be performed by the start method.
     *
     * @param multicastGroup The multicast group name.
     * @param multicastPort The multicast port number.
     * 
     */
    public DiscoveryMonitor (String multicastGroup, int multicastPort) {
        // ------------------------
        // set Default multicastPort/multicastGroup
        // ------------------------
        this.multicastGroup = multicastGroup ;
        this.multicastPort  = multicastPort ;
	if (isTraceOn()) {
		trace("constructor " , "Set group to '" + multicastGroup + "'" );
		trace("constructor " , "Set port  to '" + multicastPort + "'" );
	}
    }

    // NPCTE fix for bugId 4499338, esc 0 , MR (from JSL), 04 Sept 2001
    public DiscoveryMonitor (String multicastGroup, int multicastPort, InetAddress inf) {
        // ------------------------
        // set Default multicastPort/multicastGroup
        // ------------------------
        this.multicastGroup = multicastGroup ;
        this.multicastPort  = multicastPort ;

	usrInet = inf;

	if (isTraceOn()) {
		trace("constructor " , "Set group to '" + multicastGroup + "'" );
		trace("constructor " , "Set port  to '" + multicastPort + "'" );
		trace("constructor " , "Set interface  to '" + inf + "'" );
	}
    }
    // end of NPCTE fix for bugId 4499338


    // ----------------------------------------------------------
    // MBeanServer interaction
    // ----------------------------------------------------------
    /**
     * Allows the MBean to perform any operations it needs before being registered
     * in the MBean server. If the name of the MBean is not specified, the
     * MBean can provide a name for its registration. If any exception is
     * raised, the MBean will not be registered in the MBean server.
     *   
     * @param server The MBean server in which the MBean will be registered.
     * @param name The object name of the MBean.
     *   
     * @return  The name of the MBean registered.
     *   
     * @exception java.lang.Exception This exception should be caught by the MBean server and re-thrown
     * as an {@link javax.management.MBeanRegistrationException}.
     */
    public ObjectName preRegister (MBeanServer server, ObjectName name) throws java.lang.Exception {
	if (isTraceOn()) {
		trace("preRegister " , "object name   = " + name );
	}
        // ----------------
        // Identify the host for multicast
        // ----------------
        String grp ;
        if ( (grp = (String)name.getKeyProperty(GROUP)) != null) {
                multicastGroup =  grp;
        }
	if (isTraceOn()) {
		trace("preRegister " , "Set group to '" + multicastGroup + "'" );
	}

        // ----------------
        // Identify the port for multicast
        // ----------------
        String port ;
        if ( (port = name.getKeyProperty(PORT)) != null) {
                multicastPort  = Integer.parseInt(port) ;
        }
	if (isTraceOn()) {
		trace("preRegister " , "Set Port  to '" + multicastPort + "'" );
	}
        monitorObjectName = name;

	// ----------------
	// Return part
	// ----------------
	return name ;
    }

    /**
     * Allows the MBean to perform any operations needed after having been
     * registered in the MBean server or after the registration has failed.
     *
     * @param registrationDone Indicates whether or not the MBean has been successfully registered in
     * the MBean server. The value false means that the registration phase has failed.
     */
    public void postRegister (Boolean registrationDone) {
	if ( registrationDone == Boolean.FALSE ) {
		return ;
	}
    }

    /**                                                   
     * Allows the MBean to perform any operations it needs before being de-registered
     * by the MBean server.
     *
     * @exception java.langException  This exception should be caught by the MBean server and re-thrown
     * as an {@link javax.management.MBeanRegistrationException}.
     */
    public void preDeregister() throws java.lang.Exception {
        // ------------------------
        // Stop corresponding thread
        // ------------------------
        stop () ;
     }

    /**
     * Allows the MBean to perform any operations needed after having been
     * de-registered in the MBean server.
     */
    public void postDeregister() {
        // ------------------------
        // free 'remained' allocated resource
        // ------------------------
        System.runFinalization() ;
     }



    // ----------------------------------------------------------
    // start method
    // ----------------------------------------------------------
    /**
     * Starts listening for {@link com.sun.jdmk.discovery.DiscoveryResponder} objects registering/deregistering.
     * <P>
     * This method has no effect if the <CODE>DiscoveryMonitor</CODE> is <CODE>ONLINE</CODE> or
     * <CODE>STOPPING</CODE> or <CODE>STARTING</CODE>.
     *
     * @exception IOException The creation of the Multicast socket failed.
     */

    public void start() throws IOException {
        // ----------------
        // start actual monitor
        // ----------------
        if (state == OFFLINE) {
                try {
		    changeState(STARTING);
                        // ----------------
                        // Create a new ActualMonitor 
                        // ----------------
                        monitor = new ActualMonitor(multicastGroup,multicastPort,this);
 		
			// NPCTE fix for bugId 4499338, esc 0, MR (from JSL), 04 Sept 2001
			if (usrInet != null) {
			    monitor.setInterface(usrInet);
			    if (isTraceOn()) {
                        	trace("start" , "set to the interface "+usrInet) ;
			    }
                	}
			// end of NPCTE fix for bugId 4499338
   
                        // ----------------
                        // Create a new thread to receive multicast msg ;
                        // ----------------
                        if ( cmf == null ){
                            monitorThread = new Thread(monitor);
                        } else {
                            // monitorThread = cmf.getThreadAllocatorSrvIf().obtainThread(monitorObjectName, monitor);
                            monitorThread = new Thread( monitor);
                        }
                        monitorThread.setName("Multicast monitor");
                } catch ( IOException e) {
                	if (isDebugOn()) {
                        	debug("start" , e) ;
                	}
                        throw e ;
               } catch (NullPointerException e) {
                        // ------------------------
                        // the set group did not worked
                        // ------------------------
                        if (isDebugOn()) {
                                debug("start " , e) ;
                        }
                        throw new IOException(e.getMessage() ) ;
                }   

                monitorThread.start() ;
   
            } else {
		if (isTraceOn()) {
			trace("start " , "not OFFLINE ") ;
		}
            }
    }

    // ----------------------------------------------------------
    // stop method
    // ----------------------------------------------------------
    /**
     * Stops this <CODE>DiscoveryMonitor</CODE>.
     * <P>
     * This method has no effect if the monitor is <CODE>OFFLINE</CODE> or
     * <CODE>STOPPING</CODE> or <CODE>STARTING</CODE>.
     */
 
    public void stop() {
 
        if (state == ONLINE)
            {
		changeState(STOPPING);
                // ------------------------
                // Stop the monitor thread
                // ------------------------
                monitor.stopMonitor() ;

		synchronized(monitor.interrupted) {
		    if (!monitor.interrupted.booleanValue()) {		    
			monitorThread.interrupt() ;
		    }
		}

		// Fix for cases when the interrupt does not work (Windows NT)
		try {
		    MulticastSocket ms = new MulticastSocket(multicastPort);

		    // NPCTE fix for bug id 4499338, esc 0, MR (from JSL), 04 Sept 2001
		    if (usrInet != null) {
			ms.setInterface(usrInet);

			if (isTraceOn()) {
			    trace("stop " , "use the interface "+usrInet) ;
			}
		    }
		    // end of NPCTE fix for bug id 4499338
		    InetAddress group = InetAddress.getByName(multicastGroup);
		    ms.joinGroup(group);
		    ms.send(new DatagramPacket(new byte[1], 1, group, multicastPort));
		    ms.leaveGroup(group);
		} catch (Exception e) {
		    if (isTraceOn()) {
			trace("stop " , "Unexpected exception occured trying to send empty message " + e.getMessage()) ;
		    }
		}

		monitor = null ;
            }   
        else
            {
		if (isTraceOn())
		{
			trace("stop " , "not ONLINE") ;
		}
            }
    }

    // ----------------------------------------------------------
    // getter / setter
    // ----------------------------------------------------------
    /**
     * Returns the state of this <CODE>DiscoveryMonitor</CODE>.
     *
     * @return <CODE>ONLINE</CODE>,<CODE>OFFLINE</CODE>, <CODE>STOPPING</CODE> or <CODE>STARTING</CODE>.
     */

    public Integer getState() {
        return new Integer(state) ;
    }

    /**
     * Returns the state of this <CODE>DiscoveryMonitor</CODE> in string form.
     *
     * @return One of the strings "ONLINE", "OFFLINE", "STOPPING" or "STARTING".
     */

    public String getStateString() {
        String result = "UNKNOWN" ;

        switch(state) {
        case ONLINE: result = "ONLINE" ; break ;
        case OFFLINE: result = "OFFLINE" ; break ;
        case STOPPING: result = "STOPPING" ; break ;
	case STARTING: result = "STARTING" ; break ;
        }

        return result ;
    }

    /**
     * Returns the multicast group.
     ** A multicast group is specified by a class D IP address, those in the range 224.0.0.1 to 239.255.255.255.
     *
     * @return A string containing the multicast group name.
     */
    public String getMulticastGroup() {
        return multicastGroup;
    }

    /**
     * Sets the multicast group name.
     ** A multicast group is specified by a class D IP address, those in the range 224.0.0.1 to 239.255.255.255.
     * <P>
     * Only available if state in OFFLINE
     *
     * @param multicastGroup The multicast group name.
     *
     * @exception java.lang.IllegalStateException This method has been invoked while
     *            the <CODE>DiscoveryMonitor</CODE> was ONLINE or STARTING.
     */
    public void setMulticastGroup(String multicastGroup) 
    throws java.lang.IllegalStateException {
        if ( state == OFFLINE ) {
            this.multicastGroup = multicastGroup ;
        }
	else {
		throw new java.lang.IllegalStateException() ;
	}
    }

    /**
     * Returns the multicast port.
     * It can be any standard UDP port number.
     *
     * @return The multicast port number.
     */
    public int getMulticastPort() {
        return multicastPort ;
    }

    /**
     * Sets the multicast port.
     * It can be any standard UDP port number.
     * <P>
     * Only available if state in OFFLINE
     *
     * @param multicastPort The multicast port.
     *
     * @exception java.lang.IllegalStateException This method has been invoked while
     *            the <CODE>DiscoveryMonitor</CODE> was ONLINE or STARTING.
     */
    public void setMulticastPort(int multicastPort)
    throws java.lang.IllegalStateException  {
        if ( state == OFFLINE ) {
            this.multicastPort = multicastPort ;
        } else {
		throw new java.lang.IllegalStateException() ;
	}
    }


   /**
     * Waits untill either the State attribute of this MBean equals the specified <VAR>state</VAR> parameter, 
     * or the specified  <VAR>timeOut</VAR> has elapsed. The method <CODE>waitState</CODE> returns with a boolean value indicating whether
     * the specified <VAR>state</VAR> parameter equals the value of this MBean's State attribute at the time the method terminates.
     *
     * Two special cases for the <VAR>timeOut</VAR> parameter value are:
     * <UL><LI> if <VAR>timeOut</VAR> is negative then <CODE>waitState</CODE> returns immediately (i.e. does not wait at all),</LI>
     * <LI> if <VAR>timeOut</VAR> equals zero then <CODE>waitState</CODE> waits untill the value of this MBean's State attribute 
     * is the same as the <VAR>state</VAR> parameter (i.e. will wait indefinitely if this condition is never met).</LI></UL>
     * 
     * @param <VAR>state</VAR> The value of this MBean's State attribute to wait for. <VAR>state</VAR> can be one of:
     * <CODE>DiscoveryMonitor.OFFLINE</CODE>, <CODE>DiscoveryMonitor.ONLINE</CODE>,
     * <CODE>DiscoveryMonitor.STARTING</CODE>, <CODE>DiscoveryMonitor.STOPPING</CODE>.
     * @param <VAR>timeOut</VAR> The maximum time to wait for, in milliseconds, if positive. 
     * Infinite time out if 0, or no waiting at all if negative.
     *
     * @return true if the value of this MBean's State attribute is the same as the <VAR>state</VAR> parameter; false otherwise.
     */
    public boolean waitState(int state, long timeOut) {
        if (isTraceOn()) {
            trace("waitState",state + "(0on,1off,2st) TO=" + timeOut  + " ; current state = " + getStateString()) ;
        }
        // ------------
        // Test timeout
        // ------------
        if (timeOut < 0) {
            return (this.state == state) ;
        }
        
        boolean done = (this.state == state) ;
        Date currentDate ;
        long currentTimeOut = -1;

        // -----------------
        // Date start 
        // -----------------
        Date endDate = new Date((new Date()).getTime() + timeOut) ;     
        while ( ! done ) {
            // -----------------
            // Find out timeout
            // ----------------
            if (timeOut != 0){
                currentTimeOut = endDate.getTime() - ( new Date()).getTime() ;
                if (currentTimeOut <= 0 ) {
                    done = true ;
                    break ;
                }
            }

            try {
                synchronized(this) {
                    if (timeOut == 0){
                        if (isTraceOn()) {
                            trace("waitState","Start waiting infinite, current state = "+this.state) ;
                        }
                        done = (this.state == state)  ;
                        while ( !done )
                            {
                                done = (this.state == state)  ;
                                try { wait(1000); } catch (Exception e) {} 
                            }
                    } else {
                        if (isTraceOn()) {
                            trace("waitState","Start waiting "+currentTimeOut+" current state = "+this.state) ;
                        }
                        wait (currentTimeOut) ;
                    }
                }
                done = (this.state == state) ;
            }
            catch (InterruptedException e) {
                done = (this.state == state) ;
            }
        }
        if (isTraceOn()) {
            trace("waitState","End, TO="+currentTimeOut ) ;
        }
        return (this.state == state) ;
    }


    /**
     * For JDMK internal use only.
     */
    synchronized void changeState(int s) {
        if ( state == s ) {
            return;
        }       
        int old = state;
        state = s ;
        notifyAll() ;
    }
    


// ----------------------------------------------------------
// Trace stuff
// ----------------------------------------------------------
    private int infoType = Trace.INFO_DISCOVERY;
    private String localClassName = "com.sun.jdmk.discovery.DiscoveryMonitor" ;

    private String dbgTag = localClassName ;                                 
 
    private boolean isTraceOn() {
        return Trace.isSelected(Trace.LEVEL_TRACE, infoType);
    }
 
    private void trace(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_TRACE, infoType, clz, func, info);
    }

    private void trace(String clz, String func, Exception e) {
        Trace.send(Trace.LEVEL_TRACE, infoType, clz, func, e);
    }

    private boolean isDebugOn() {
        return Trace.isSelected(Trace.LEVEL_DEBUG, infoType);
    }

    private void debug(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_DEBUG, infoType, clz, func, info);
    }

    private void debug(String clz, String func, Exception e) {
        Trace.send(Trace.LEVEL_DEBUG, infoType, clz, func, e);
    }

    private void trace(String func, String info) {
        trace(localClassName, func, info);
    }

    private void trace(String func, Exception e) {
        trace(localClassName, func, e);
    }

    private void debug(String func, String info) {
        debug(localClassName, func, info);
    }

    private void debug(String func, Exception e) {
        debug(localClassName, func, e);
    }


    // ----------------------------------------------------------
    // Private variables
    // ----------------------------------------------------------
    
    /**
     * The object name of the discovery monitor.
     *
     * @serial
     */
    private ObjectName              monitorObjectName       = null;

    /**
     * This field holds a reference to the core management framework.
     *
     * @serial
     */
    private MBeanServer               cmf                     = null;
    
    private static final String     sccs_id                 = "@(#)DiscoveryMonitor.java 4.21 01/09/11 SMI" ;

    private static int              defaultMulticastPort    = 9000  ;
    private static String           defaultMulticastGroup   = "224.224.224.224" ;

    /**
     * The multicast port number.
     *
     * @serial
     */
    private int                     multicastPort           ;

    /**
     * The multicast group name.
     *
     * @serial
     */
    private String                  multicastGroup          ;

    private transient ActualMonitor monitor         = null ;
    private transient Thread        monitorThread   = null ;
    /** Reflects the current state of the discovery monitor. */
    private transient volatile int           state  = OFFLINE ;

    private transient Vector        listeners       = new Vector();

    private static final String     GROUP           = "group";
    private static final String     PORT            = "port";

    // NPCTE fix for bugId 4499338, esc 0, MR (from JSL), 04 Sept 2001
    private InetAddress usrInet = null;
    // end of NPCTE fix for bugId 4499338

}
