/*
 * @(#)file      CascadingAgent.java
 * @(#)author    Sun Microsystems, Inc.
 * @(#)version   1.18
 * @(#)date      01/09/11
 *
 * Copyright 2000 Sun Microsystems, Inc. All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2000 Sun Microsystems, Inc. Tous droits rservs.
 * Ce logiciel est propriet de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation. 
 */


package com.sun.jdmk.cascading;

// java import
import java.io.Serializable;
import java.util.Vector;
import java.util.Iterator;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;


// jdmk import
import javax.management.ObjectName;
import javax.management.ObjectInstance;
import javax.management.JMRuntimeException;
import javax.management.MBeanServer;
import javax.management.MBeanRegistration;
import javax.management.MBeanServerNotification;
import javax.management.Notification;
import javax.management.NotificationFilter;
import javax.management.NotificationListener;
import javax.management.QueryExp;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.ServiceNotFoundException;
import javax.management.NotCompliantMBeanException;
import javax.management.MBeanRegistrationException;
import com.sun.jdmk.Trace;
import com.sun.jdmk.GenericProxy;
import com.sun.jdmk.Proxy;
import com.sun.jdmk.ServiceName;
import com.sun.jdmk.comm.RemoteMBeanServer;
import com.sun.jdmk.comm.ConnectorAddress;
import com.sun.jdmk.comm.RmiConnectorAddress;
import com.sun.jdmk.comm.CommunicationException;
import com.sun.jdmk.comm.HeartBeatNotification;
import com.sun.jdmk.comm.HeartBeatClientHandler;


/**
 * This class implements the cascading agent service.
 * <P>
 * Cascading agents enable you to implement a hierarchy of master agents and
 * subagents.The MBeans in a subagent are mirrored in the master agent.
 * A master agent can contain more than one cascading agent, 
 * but must contain one cascading agent for each subagent it 
 * communicates with.<P>
 *
 * When instantiating the cascading agent, the following parameters
 * should be specified to the CascadingAgent constructor:
 * <UL>
 * <LI><B>ConnectorAddress</B>: the address of the remote connector to contact.</LI>
 * For example, for an Rmi connection the <CODE>RmiConnectorAddress</CODE> and for an HTTP connection
 * the <CODE>HttpConnectorAddress</CODE>.
 * <LI><B>client connector name</B>: the name of the client connector to connect to the subagent 
 * (by default com.sun.jdmk.comm.RmiConnectorClient).</LI>
 * </UL>
 */
public class CascadingAgent 
    implements Serializable, NotificationListener, MBeanRegistration, CascadingAgentMBean, NotificationFilter {


    // PRIVATE VARIABLES
    //------------------

    /**
     * This field holds a reference to the core management mserver.
     *
     * @serial
     */
    private MBeanServer mserver = null;

    /**
     * The remte address to connect to.
     *
     * @serial
     */
    private ConnectorAddress address = new RmiConnectorAddress("localhost", ServiceName.RMI_CONNECTOR_PORT, ServiceName.RMI_CONNECTOR_SERVER);

    /**
     * The name of the client connector to connect to the cascading agent.
     *
     * @serial
     */
    private String conClName = null;

    /**
     * The active state of the cascading agent.
     */
    private transient boolean isActive = false;

    /**
     * The state of the cascading agent while being stopeed.
     */
    private transient boolean isStopping = false;

    /**
     * The client connector to connect to the cascading agent.
     */
    private transient RemoteMBeanServer conClient = null ;

    /**
     * A Vector of registered remote objects.
     */
    private transient Vector registeredRemoteObjs = new Vector();

    /**
     * The query expression to be applied for cascaded MBeans
     */
    private transient QueryExp query  = null;

    /**
     * The pattern to be applied for cascaded MBeans
     */
    private transient ObjectName pattern  = null;

    /** The name of this class to be used for tracing */
    private String dbgTag = "CascadingAgent";


  
    /**
     * Constructs a default <CODE>CascadingAgent</CODE>.
     * <P>
     * Initializes a newly created <CODE>CascadingAgent</CODE> with the following
     * default values:
     * <UL>
     * <LI><B>host</B>: the local machine name</LI>
     * <LI><B>port</B>: 1099</LI>
     * <LI><B>protocol</B>: rmi</LI>
     * <LI><B>RMI connector server ObjectName</B><CODE><I>defaultdomain</I>:name=RmiConnectorServer</CODE>.
     * <LI><B>client connector name</B>: com.sun.jdmk.comm.RmiConnectorClient</LI>
     * </UL>
     * By default, all the MBeans in the subagent will be represented in the master agent.
     */
    public CascadingAgent() {
    }


    /**
     * Constructs a <CODE>CascadingAgent</CODE>.
     * <P>
     * In order to initialize the cascading agent, implementation names of the
     * different services required by the MBean Server must be provided.
     * All the MBeans in the subagent will be represented in the master agent.
     *
     * @param addr address to connect to in the cascading agent
     * @param conClName class name of the client connector to connect to the cascading agent.
     */
    public CascadingAgent(ConnectorAddress addr, String conClName) {	
        this.address = addr;
	this.conClName = conClName;
    }


    /**
     * Constructs a <CODE>CascadingAgent</CODE>.
     * <P>
     * In order to initialize the cascading agent, implementation names of the
     * different services required by the MBean Server must be provided.
     *
     * @param addr address to connect to in the cascading agent
     * @param conClName class name of the client connector to connect to the cascading agent.
     * @param pattern the <CODE>ObjectName</CODE> pattern to apply to the MBeans of the subagent.
     * If the pattern is null or contains an empty domain and key properties,
     * all the MBeans in the subagent will be selected.
     * @param query  the <CODE>query</CODE> to apply to the MBeans of the subagent.
     * If null no query will be applied in the MBeans of the subagent.
     */
    public CascadingAgent(ConnectorAddress addr, String conClName, ObjectName pattern, QueryExp query) {	
        this.address = addr;
	this.conClName = conClName;
	this.pattern = pattern;
	this.query = query;
    }

    /**
     * Constructs a <CODE>CascadingAgent</CODE>.
     * <P>
     * In order to initialize the cascading agent, implementation names of the
     * different services required by the MBean Server must be provided.
     * All the MBeans in the subagent will be represented in the master agent.
     * The client connector that will be used is the JDMK connector client 
     * corresponding to the protocol of the <CODE>ConnectorAddress</CODE> parameter.
     *
     * @param addr address to connect to in the cascading agent. It can be an instance
     * of <CODE>com.sun.jdmk.comm.RmiConnectorClient</CODE>, <CODE>com.sun.jdmk.comm.HttpConnectorAddress</CODE> 
     * or <CODE>com.sun.jdmk.comm.HttpsConnectorAddress</CODE> classes.
     *
     * @exception java.lang.IllegalArgumentException The address parameter is not an instance of a valid class.
     */
    public CascadingAgent(ConnectorAddress addr) {	
        this.address = addr;
	this.conClName = findConnectorClientClassName(addr);
    }


    /**
     * Constructs a <CODE>CascadingAgent</CODE>.
     * <P>
     * In order to initialize the cascading agent, implementation names of the
     * different services required by the MBean Server must be provided.
     * The client connector that will be used is the JDMK connector client 
     * corresponding to the protocol of the <CODE>ConnectorAddress</CODE> parameter.
     *
     * @param addr address to connect to in the cascading agent. It can be an instance
     * of <CODE>com.sun.jdmk.comm.RmiConnectorClient</CODE>, <CODE>com.sun.jdmk.comm.HttpConnectorAddress</CODE> 
     * or <CODE>com.sun.jdmk.comm.HttpsConnectorAddress</CODE> classes.
     * @param pattern the <CODE>ObjectName</CODE> pattern to apply to the MBeans of the subagent.
     * If the pattern is null or contains an empty domain and key properties,
     * all the MBeans in the subagent will be selected.
     * @param query  the <CODE>query</CODE> to apply to the MBeans of the subagent.
     * If null no query will be applied in the MBeans of the subagent.
     *
     * @exception java.lang.IllegalArgumentException The address parameter is not an instance of a valid class.
     */
    public CascadingAgent(ConnectorAddress addr, ObjectName pattern, QueryExp query) {	
        this.address = addr;
	this.conClName = findConnectorClientClassName(addr);
	this.pattern = pattern;
	this.query = query;
    }


    /**
     * Notification filter for local MBean Server Notifications
     * Enables reception of MBean Server Notifications concerning "cascaded" MBeans.
     */
    public synchronized boolean isNotificationEnabled(Notification notification) {
	ObjectName name = ((MBeanServerNotification)notification).getMBeanName();
	if (registeredRemoteObjs.contains(name)) {
	    return true;
	} else {
	    return false;
	}
    }

    /**
     * Initializes the cascading agent.
     * <P>
     *
     * @param <VAR>agent</VAR> The MBean Server to register the service with.
     * @param <VAR>name</VAR> The object name of the MBean.
     *    
     */
    public ObjectName preRegister(MBeanServer server, ObjectName name) throws java.lang.Exception {
 
        this.mserver= server;
 
        if (conClient == null) {
        
            // Try to get a client connector ...
            //
            Class  conClientClass = null;
            try {
                conClientClass = Class.forName(getClientConnectorClassName());
                conClient = (RemoteMBeanServer)conClientClass.newInstance();
            } catch (Exception ee) {
                throw new ServiceNotFoundException(ee.getClass().getName() + ": " + ee.getMessage());
            }
        }
	return name;
    }

    public void postRegister(java.lang.Boolean registrationDone) {
    }

    public void preDeregister() throws java.lang.Exception {
    }
    

    // GETTERS AND SETTERS
    //--------------------

    /**
     * Getter for the <CODE>RemoteMBeanServer</CODE> of the <CODE>CascadingAgent</CODE>.
     *
     * @return the current value of the <CODE>RemoteMBeanServer</CODE> property.
     */
    public RemoteMBeanServer getRemoteMBeanServer() {
        return conClient;
    }

    /**
     * Getter for the <CODE>ConnectorAddress</CODE> to which the <CODE>CascadingAgent</CODE> is connected.
     *
     * @return the current value of the <CODE>ConnectorAddress</CODE> property.
     */
      public ConnectorAddress getAddress() {
        return address;
      }

    /**
     * Setter for the <CODE>ConnectorAddress</CODE> to which the <CODE>CascadingAgent</CODE> is connected.
     *
     * @exception java.lang.IllegalStateException The cascading agent is active.
     *
     */
    public void setAddress(ConnectorAddress adr) {
	if (isActive) throw new IllegalStateException("The cascading agent is active");
        this.address = adr;
    }

    /**
     * Getter for the class name of the <CODE>RemoteMBeanServer</CODE> of the <CODE>CascadingAgent</CODE>.
     *
     * @return the current value of the the class name of the <CODE>RemoteMBeanServer</CODE>.
     */
    public String getClientConnectorClassName() {
    
        // The connector name was given by the user
        //
        if (conClName != null) {
            return conClName;
        }

        // Otherwise build the connector name suited for the protocol
        //
        return "com.sun.jdmk.comm.RmiConnectorClient";
    }

    /**
     * Setter for the class name of the <CODE>RemoteMBeanServer</CODE> of the <CODE>CascadingAgent</CODE>.
     *          
     * @exception java.lang.IllegalStateException The cascading agent is active.
     */
    public void setClientConnectorClassName(String name) {
	if (isActive) throw new IllegalStateException("The cascading agent is active");    
	conClName = name; 
    }


    /**
     * Getter for the number of MBeans "cascaded" by the <CODE>CascadingAgent</CODE>.     
     */
    public Integer getNbOfRemoteMBeans() {
	return (new Integer(registeredRemoteObjs.size()));
    }

    /**
     * Getter for the MBeans "cascaded" by the <CODE>CascadingAgent</CODE>.
     *
     * @return a <CODE>java.util.Set</CODE> containig all the "cascaded" MBeans.
     */
    public Set getRemoteMBeans() {

	Set remObjs = new HashSet();
	ObjectInstance oi = null;

	for (Enumeration i=registeredRemoteObjs.elements();i.hasMoreElements();) {
	    try {
		oi = conClient.getObjectInstance((ObjectName)i.nextElement());
	    } catch (Exception e) {}
	    remObjs.add(oi);
	}
	
        return remObjs;
    }

    /**
     * Getter for the <CODE>ObjectName</CODE> pattern to apply to the MBeans of the subagent.
     * If no <CODE>ObjectName</CODE> pattern was specified in the <CODE>CascadingAgent</CODE>
     * constructor the method returns null.
     */
    public ObjectName getPattern() {	
        return pattern;
    }

    /**
     * Getter for the <CODE>QueryExp</CODE> query to apply to the MBeans of the subagent.
     * If no <CODE>QueryExp</CODE>  was specified in the <CODE>CascadingAgent</CODE>
     * constructor the method returns null.
     */
    public QueryExp getQuery() {	
        return query;
    }


    /**
     * Processes a MBean Server Notification emitted by the MBean Server.<P>
     *
     * @param notification The MBean Server Notification to be processed.
     * @param handback  The handback object.
     */
   
    public void handleNotification(Notification notification, java.lang.Object handback) {
	
	if (handback == null) {
	    if (notification instanceof com.sun.jdmk.comm.HeartBeatNotification) {
		// Received Heartbeat notification
		handleHeartbeatNotification(notification);
	    } else {
		// Received notification from sub-agent
		handleRemoteNotification(notification);	    
	    }
	} else {
	    // Received local notification
	    handleLocalNotification(notification);
	}
    }

    /**
     * Unregisters all the objects cascaded by the cascading agent.
     */
    public void postDeregister() {
    
        // Become inactive : this will remove all objects and disconnect
        //
        try {
            stop();
        } catch(Exception e) {
	    if (isDebugOn()) {
		debugex("postDeregister", e);
	    } 
        }
    }


    // connector setup stuff
    //------------------------------------------
    
    /**
     * Activates the <CODE>CascadingAgent</CODE> MBean.
     */
    public synchronized void start() {
        // Activate the CascadingAgent.
        //
        if (!isActive) {

	    if ((conClient == null) || !(conClient.getClass().getName().equals(getClientConnectorClassName()))) {
		
		// Try to get a client connector ...
                Class  conClientClass = null;
                try {
                    conClientClass = Class.forName(getClientConnectorClassName());
                    conClient = (RemoteMBeanServer) conClientClass.newInstance();
                } catch (Exception ee) {
                    throw new JMRuntimeException(ee.getClass().getName() + ": " + ee.getMessage());		    
                }
	    }
	    
            try {
                conClient.connect(address);
            } catch(CommunicationException e) {		
                throw e;
            }

	    // Listen to heartbeat notifications to detect communication problems	
	    // As the Heartbeat interface is not part of the RemoteMBeanServer interface
	    // we need to test if the connector client implements the HeartBeatClientHandler interface
	    
	    if (conClient instanceof  com.sun.jdmk.comm.HeartBeatClientHandler) {	    	
		try {
		    // set the heartbeat period to 2ms
		    ((HeartBeatClientHandler)conClient).setHeartBeatPeriod(2000);
		    ((HeartBeatClientHandler)conClient).addHeartBeatNotificationListener(this, null, null);
		} catch(Exception ex) {
		    if (isDebugOn()) {
			debug("start", "Failed to become listener of heartbeat notifications on the remote MBean Server " + ex.getMessage());
		    }    
		}	    
	    }

            // Load the Proxy MBeans from the cascading agent into
            // the local MBean Server
            //
            try {	
                loadRemoteObjects(pattern, query);
            } catch(Exception ex) {
		if (isDebugOn()) {
		    debug("start", ex.getMessage());
		}     
            }

            // Get the managed object for the MBean Server
            //
            try {
		conClient.addNotificationListener(new ObjectName(ServiceName.DELEGATE), this, null, null);
            } catch(Exception ex) {
		if (isDebugOn()) {
		    debug("start", "Failed to become listener of the remote MBean Server " + ex.getMessage());
		}    
            }
      
            // Set the active state to active.
            //
            isActive = true;
	    isStopping = false;       

	} else {
	    if (isTraceOn()) {
		trace("start", "The CascadingAgent is already activated.");
	    }   
        }
	    // Listen to local MBeanServer notifications
	    try {
		mserver.addNotificationListener(new ObjectName(ServiceName.DELEGATE), this, this, new String("local"));
		
	    } catch(Exception ex) {
		if (isDebugOn()) {
		    debug("start", "Failed to become listener of the local MBean Server " + ex.getMessage());
		}    
	    }	   

    }
  
    /**
     * Deactivates the <CODE>CascadingAgent</CODE> MBean.
     */
    public synchronized void stop() {
        // Deactivate the CascadingAgent.        
        if (isActive) {
        
            // Stop listening to remote mserver events
            try {
		conClient.removeNotificationListener(new ObjectName(ServiceName.DELEGATE), this);      
            } catch(Exception e) {
            }

	    // Stop listening to heartbeat notifications
	    if (conClient instanceof  com.sun.jdmk.comm.HeartBeatClientHandler) {
		try {	
		    ((HeartBeatClientHandler)conClient).removeHeartBeatNotificationListener(this);
		} catch(Exception ex) {	
		}
	    }

            try {
		isStopping = true;
                for (;!registeredRemoteObjs.isEmpty();) {
                    Object o = registeredRemoteObjs.firstElement();
                    try {
                        unregister((ObjectName)o);
                    } catch(InstanceNotFoundException ex) {
                        // Try to remove the object anyway !!
                        //
                        registeredRemoteObjs.removeElement(o);
                    } catch( MBeanRegistrationException ex) {
                        // Try to remove the object anyway !!
                        //
                        registeredRemoteObjs.removeElement(o);
                    }
                }
		// Disconnect from the cascading agent
                if (conClient.isConnected()) {
		    conClient.disconnect();
		}
	    } catch(CommunicationException e) {
                throw e;
            }

            // Set the active state to active.
            //
	    // NPCTE fix for bugId 4499327, esc 0, MR , 04 sept 2001
            //isActive = false;
	    //isStopping = false;
	    finally { isActive = false; isStopping = false; }
	    // end of NPCTE fix for bugId 4499327
        } else {
	    if (isTraceOn()) {
		trace("stop", "The CascadingAgent is already deactivated.");
	    }             
        }
    }


    /**
     * Tests if the <CODE>CascadingAgent</CODE> is active.
     */
    public boolean isActive() {
        return isActive;
   
    }

    // NPCTE fix for bugId 4500398, esc 0, MR 07 Sept 2001
    /*
     * Tests whether string s is matched by pattern p.
     * Supports "?", "*" each of which may be escaped with "\";
     * Not yet supported: internationalization; "\" inside brackets.<P>
     * Wildcard matching routine by Karl Heuer.  Public Domain.<P>
     */  
    private boolean wildmatch(String s, String p) {
        char c;
        int si = 0, pi = 0;
        int slen = s.length();
        int plen = p.length();

        while (pi < plen) {            // While still string
            c = p.charAt(pi++);
            if (c == '?') {
                if (++si > slen)
                    return false;
            } else if (c == '*') {        // Wildcard
                if (pi >= plen)
                    return true;
                do {
                    if (wildmatch(s.substring(si), p.substring(pi)))
                        return true;
                } while (++si < slen);
                return false;
            } else {
                if (si >= slen || c != s.charAt(si++))
                    return false;
            }
        }
        return (si == slen);
    }
    // end of NPCTE fix for bugId 4500398, esc 0, MR 07 Sept 2001

    /**
     * Processes an MBeanServer Notification emitted by the sub-agent. 
     */
    private void handleRemoteNotification(Notification notification) {

	// Received notification from sub-agent
	// NPCTE fix for bugId 4500398, esc 0, MR 07 September 2001
	//HashSet names = (HashSet)conClient.queryNames(pattern, query);
	ObjectName on = (ObjectName)((MBeanServerNotification)notification).getMBeanName();     
	// if ObjectName doesn't match the pattern supplied at construction of CascadingAgent -> do nothing	

	if ((pattern != null) && !(wildmatch(on.getCanonicalName(),pattern.getCanonicalName())))
		return;
	// end of NPCTE fix for bugId 4500398
 
	if (notification.getType().equals(MBeanServerNotification.REGISTRATION_NOTIFICATION)) {
	    // NPCTE fix for bugId 4500398, esc 0, MR 07 September 2001
	    try {
	    	if ((query == null) || query.apply(on)) {	    
	    	//if (names.contains(on)) {
	   // end of NPCTE fix for bugId 4500398	
			// Deal with the creation of an object in the cascading agent            	   
			Proxy remObjs = null;
			ObjectInstance remObjsInst = null;		    
			GenericProxy gproxy = null;
		
			try {
		    		remObjsInst = conClient.getObjectInstance(on);	       
		    		gproxy = new GenericProxy(remObjsInst, conClient);
		    		register(new CascadeGenericProxy(gproxy), on);
			} catch (Exception e) {
		    		if (isDebugOn()) {
					debug("handleRemoteNotification", "Could not get GenericProxy for object " + on + " " + e.getMessage());
		    		}	
			}		    		        
		}
	    } catch (Exception e) {
		if (isDebugOn()) {
                                        debug("handleRemoteNotification", e.getMessage() );
		}
	    }
	} else {
	    if (notification.getType().equals(MBeanServerNotification.UNREGISTRATION_NOTIFICATION)) {
		
		// Deal with the deletion of an object in the cascading agent		          
		try {
		    if (isTraceOn()) {
			trace("handleRemoteNotification", "Remove deleted MBean " + on);
		    }                  
		    unregister(on);
		} catch(Exception ex) {
		    if (isDebugOn()) {
			debug("handleRemoteNotification", "Could not unregister object  " + on + " " + ex.getMessage());
		    }                     
		}
	    }
	}	    
    }


   /**
    * Processes a local MBeanServer Notification.
    */
    private void handleLocalNotification(Notification notification) {

	// received local notification
	if (!isStopping) {
	    if (notification.getType().equals(MBeanServerNotification.UNREGISTRATION_NOTIFICATION)) {
		ObjectName on = (ObjectName)((MBeanServerNotification)notification).getMBeanName();     
		if (registeredRemoteObjs.contains(on)) {
		    try {
			conClient.unregisterMBean(on);
		    } catch (Exception e) {
			if (isDebugOn()) {
			    debug("handleLocalNotification", "Could not unregister object  " + on + " " + e.getMessage());
			}  
		    }
		}
	    }
	}
    }


    /**
    * Processes a Heartbeat Notification.
    */
    private void handleHeartbeatNotification(Notification notification) {
	if (!isStopping) {
	    // if the communication id lost
	    if (notification.getType().equals(HeartBeatNotification.CONNECTION_LOST)) {
		stop();
	    }
	}
    }


   /**
     * Getter for the "Connected" property.
     *
     * @return the current value of the "Connected" property.
     */
    private Boolean getConnected() {
        boolean isConn;
        if (conClient == null) {
            isConn = false;
        } else {
            isConn = conClient.isConnected();
        }
        return new Boolean(isConn);
    }
  
    /**
     * Register a remote object in the local mserver.
     */
    private void register(Object obj, ObjectName objName) throws InstanceAlreadyExistsException, 
	MBeanRegistrationException, NotCompliantMBeanException  {
        mserver.registerMBean(obj, objName);
        registeredRemoteObjs.addElement(objName);
    }


    /**
     * Unregister a remote object in the local mserver.
     */
    private void unregister(ObjectName objName)
        throws InstanceNotFoundException, MBeanRegistrationException {

        mserver.unregisterMBean(objName);
	registeredRemoteObjs.removeElement(objName);
    }


    /**
     * Load the objects from the remote agent into the local mserver.
     */
    private void loadRemoteObjects(ObjectName pattern, QueryExp query)
        throws ServiceNotFoundException, CommunicationException, 
        IllegalAccessException {
	
        // Get the names of remote objects
        //
	HashSet remObjNames = null;
	
	remObjNames = (HashSet)conClient.queryMBeans(pattern, query);	       
        
	if (isDebugOn()) {
	    debug("loadRemoteObjects", "Found "+ remObjNames.size()+" names");
	} 
    
        // Get the objects themselves
        //

        for (Iterator i = remObjNames.iterator(); i.hasNext(); ) {
            ObjectInstance objInst = (ObjectInstance) i.next();

	    GenericProxy gproxy = null;
            Object remObjs = null;

           try {	                	
	       gproxy = new GenericProxy(objInst, conClient);
	   } catch (Exception e) {
	       if (isDebugOn()) {
		   debug("loadRemoteObjects", "Could not get GenericProxy for object " + objInst + " " + e.getMessage());
	       }	
	   }	   
                  
            // Object could not be "retrieved" : forget it !
            //
            if (remObjs == null) {
                remObjs = new CascadeGenericProxy(gproxy);
            }

            // Put the Proxy MBeans in the MBean Server with the same name
	    if (!objInst.getObjectName().getDomain().equals("JMImplementation")) {
		try {
		    register(remObjs, objInst.getObjectName());
		} catch(InstanceAlreadyExistsException ex) {
		    if (isDebugOn()) {
			debugex("loadRemoteObjects", ex);
		    }           
		} catch(MBeanRegistrationException e) {
		    if (isDebugOn()) {
			debugex("loadRemoteObjects", e);
		    }    
		} catch(NotCompliantMBeanException n) {
		    if (isDebugOn()) {
			debugex("loadRemoteObjects", n);
		    }   
		}
		if (isTraceOn()) {
		    trace("loadRemoteObjects", "Proxy MBean "+ objInst.getObjectName() +" has been registered");
		}     
	    }
	}	  
    }

    /**
     * Returns the class name of the JDMK ConnectorClient implemetation corresponding to the 
     * specified ConnectorAddress.
     */
    private String findConnectorClientClassName(ConnectorAddress addr){
	if (addr instanceof com.sun.jdmk.comm.RmiConnectorAddress) {	    
	    return "com.sun.jdmk.comm.RmiConnectorClient";
	} 
	if (addr instanceof com.sun.jdmk.comm.HttpConnectorAddress) {	    
		return "com.sun.jdmk.comm.HttpConnectorClient";
	} 
	if (addr instanceof com.sun.jdmk.comm.HttpsConnectorAddress) {	    
	    return "com.sun.jdmk.comm.HttpsConnectorClient";
	}	
	throw new IllegalArgumentException("The ConnectorAddress passed in parameter does not correspond to a JDMK ConnectorAddress implementation");
    	
    }

 // TRACES & DEBUG
    //---------------
    
    private boolean isTraceOn() {
	return Trace.isSelected(Trace.LEVEL_TRACE, Trace.INFO_MISC);
    }
     
    private void trace(String clz, String func, String info) {
	Trace.send(Trace.LEVEL_TRACE, Trace.INFO_MISC, clz, func, info);
    }
    
    private void trace(String func, String info) {
        trace(dbgTag, func, info);
    }
    
    private boolean isDebugOn() {
        return Trace.isSelected(Trace.LEVEL_DEBUG, Trace.INFO_MISC);
    }

    private void debug(String clz, String func, String info) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_MISC, clz, func, info);
    }

    private void debug(String func, String info) {
        debug(dbgTag, func, info);
    }

    private void debugex(String clz, String func, Throwable ex) {
        Trace.send(Trace.LEVEL_DEBUG, Trace.INFO_MISC, clz, func, ex);
    }
    private void debugex(String func, Throwable ex) {
        debugex(dbgTag, func, ex);
    }


}
