#!/bin/sh
# Copyright 11/20/01 Sun Microsystems, Inc. All Rights Reserved.
# pragma ident  "@(#)es-common.sh	1.100 01/11/20 Sun Microsystems"
#
#  Common shell functions used by various scripts
#
# Note:  This script is used internally by Sun Enterprise SyMON,
# and is not intended to be called directly by users.
# It should not be modified.
#
set_inst_base_dir() {

    if [ "$RELOCATIONDIR" = "" ]; then
       RELOCATIONDIR=`/usr/bin/pkgparam SUNWescom BASEDIR`
    fi

    BASEDIR=$RELOCATIONDIR/SUNWsymon
    ESROOT="$BASEDIR"
    SYMONBASE=$RELOCATIONDIR
    export ESROOT   #  Needed by other scripts later

}

# $1 - Relocated root directory (optional)
set_commondirs() {
    if [ "$1" = "" ]; then
	RELOC_DIR=""
    else
	RELOC_DIR="$1"
    fi

    CONFDIR="${1}/etc/opt/SUNWsymon"
    VAROPTDIR="${1}/var/opt/SUNWsymon"
    export VAROPTDIR
    PLATFORM_FILE=$VAROPTDIR/platform/platform.prop
    export PLATFORM_FILE
    VARDIR=$VAROPTDIR/cfg
    ME=`/usr/ucb/whoami`

    [ ! -d "$VARDIR" ] && [ "$ME" = "root" ] && /usr/bin/mkdir -p "$VARDIR"
}

set_db_env() {
    inst_path=`/usr/bin/pkginfo -r SUNWescom`/SUNWsymon
#    ORACLE_BASE=$ESROOT/SunMCDB01/app/oracle; export ORACLE_BASE
    ORACLE_BASE=$inst_path/SunMCDB01/app/oracle; export ORACLE_BASE
    ORACLE_HOME=$ORACLE_BASE/product/8.1.6; export ORACLE_HOME
    LD_LIBRARY_PATH=$ORACLE_HOME/lib:$LD_LIBRARY_PATH; export LD_LIBRARY_PATH
    ORA_NLS33=$ORACLE_HOME/ocommon/nls/admin/data; export ORA_NLS33
    TNS_ADMIN=$ORACLE_HOME/network/admin; export TNS_ADMIN

    # ensure that the two_task var is unset/unused for here.
    if [ -n "$TWO_TASK" ]; then
	unset TWO_TASK
    fi

    SQLPATH=$ORACLE_HOME/rdbms/admin:$BASEDIR/db/build:$BASEDIR/db/util; export SQLPATH
    ORACLE_SID=SunMC30; export ORACLE_SID
    ORACLE_OWNER=root; export ORACLE_OWNER
    SUNMCLISTENER=smcdblistener; export SUNMCLISTENER
    SMCORAU=smcorau; export SMCORAU    
    SMCORAG=smcorag; export SMCORAG    
    unset OPT
}

check_development_env() {
#checks the system. If the system is installed as development
#system then return 0 otherwise returns 1. If a valid SDK kay 
#is installed in the SDK license file then it is a development
#system

    /usr/bin/pkginfo -qi SUNWessdk > /dev/null 2>&1
    if [ $? -eq 0 ] ; then
        return 0
    else
        return 1
    fi
}

echolog() {
    eval echo "`/usr/bin/gettext "$1"`"
    if [ "$LOGFILE" != "" ] ; then
	eval echo "`/usr/bin/gettext "$1"`" >> "$LOGFILE"
    fi
}

get_input_from_user() {
    question="$1" 
    if [ "$LOGFILE" != "" ]; then
	eval echo `/usr/bin/gettext "$question"` " \\\c" >> "$LOGFILE"
    fi
    eval echo $GUIQSTART `/usr/bin/gettext "$question"` $GUIQEND " \\\c" 

#    if [ "$AUTOCONFIG_RUNFILE" != "" ]; then # Run automated configuration
#	answer=`/usr/bin/cat $AUTOCONFIG_RUNFILE | /usr/bin/head -1`
#	echo $answer
#	/usr/bin/cat $AUTOCONFIG_RUNFILE | /usr/bin/tail +2 > ${AUTOCONFIG_RUNFILE}.1
#	/usr/bin/mv ${AUTOCONFIG_RUNFILE}.1 $AUTOCONFIG_RUNFILE
#    else
#	read answer
#    fi

    read answer

    if [ "$LOGFILE" != "" ]; then
       /usr/bin/stty | /usr/bin/grep "\-echo" > /dev/null 2>&1 
       [ $? -ne 0 ] && /usr/bin/echo "$answer" >> "$LOGFILE"
    fi

#    if [ "$AUTOCONFIG_SAVEFILE" != "" ]; then # Save automated configuration details
#	echo "$answer" >> "$AUTOCONFIG_SAVEFILE"
#    fi
}


ask_user() {
    question="$1"
    answer=""
    while [ "$answer" = "" ]
    do
	if [ "$LOGFILE" != "" ] ; then
	    eval echo `/usr/bin/gettext "$question"` " \\\c" >> "$LOGFILE"
	    eval echo `/usr/bin/gettext '[y\\\|n\\\|q]'` " \\\c" >> "$LOGFILE"
	fi	
	eval echo $GUIQSTART `/usr/bin/gettext "$question"` " \\\c" 
	eval echo `/usr/bin/gettext '[y\\\|n\\\|q]'` $GUIQEND " \\\c" 

	read user_answer

	case "$user_answer" in
		y*) answer=1
		    break
		    ;;
		n*) answer=0
		    break
		    ;;
		q*) answer=2
		    break
		    ;;
		Y*) answer=1
		    break
		    ;;
		N*) answer=0
		    break
		    ;;
		Q*) answer=2
		    break
		    ;;
		*)  echolog ''
		    echo "  \c"
		    echolog 'Enter \\\"y\\\" or \\\"n\\\" or \\\"q\\\"'
		    user_answer=""
		    ;;
	esac
    done

    if [ "$LOGFILE" != "" ] ; then
	echo "$user_answer" >> "$LOGFILE"
    fi


    if [ $answer -eq 2 ]; then
	echolog ''
	echolog 'Exiting at user request'
	echolog ''
	exit 2
    fi
}

# Check OS version
# $1 - supplied OSVERS (for -R option)
check_osver() {
    if [ -z "$1" ]; then
	OSVERS=`/usr/bin/uname -r`
    else
	OSVERS="$1"
    fi

    case $OSVERS in
	5.5.1*) OSVERS=2.5  ;;
	5.6*)   OSVERS=2.6  ;;
	5.7*)   OSVERS=2.7  ;;
	5.8*)   OSVERS=2.8  ;;
	5.9*)   OSVERS=2.9  ;;
	*)      echolog 'Unsupported OS version: $2' "$OSVERS"
	        exit 1	    ;;
    esac
}

check_root() {
    ME=`/usr/ucb/whoami`

    [ "$ME" = "root" ] && return 0

    echo ""
    /usr/bin/gettext "This action is limited to superuser only.\n"
    eval echo `/usr/bin/gettext 'Exiting $PROGNAME.'`
    echo ""
    /usr/bin/sleep 2
    exit 1

}

check_logfile() {
#
#  $1 = logfile type (install or setup)
#
    if [ -z "$LOGFILE" ] ; then
	LOGDIR=$VAROPTDIR/install
	[ ! -d "$LOGDIR" ] && /usr/bin/mkdir -p "$LOGDIR"
	DATE_FORMAT=`eval echo "+\\\%y\\\%m\\\%d\\\%H\\\%M\\\%S"`
	TIMESTAMP=`/usr/bin/date $DATE_FORMAT`
	LOGFILE="$LOGDIR/$1.$TIMESTAMP.$$"
    fi

    if [ ! -f "$LOGFILE" ] ; then
	/usr/bin/touch "$LOGFILE" >/dev/null 2>&1
	/usr/bin/chmod 640 "$LOGFILE"
	if [ $? -ne 0 ] ; then
	    echo ""
	    eval echo `/usr/bin/gettext 'Unable to create logfile $LOGFILE.'`
	    echo ""
	    exit 1
	fi
    fi

    eval echo `/usr/bin/gettext "Started $0 at"` '`/usr/bin/date`'. >> "$LOGFILE"
    eval echo `/usr/bin/gettext "Running on"` `/usr/bin/uname -a`. >> "$LOGFILE"
}

check_for_l10n() {

    L10N_REQUIRED=0
    if [ -n "$LANG" ] && [ "$LANG" != "C" ]; then
        if [ "$LANG" != "zh_TW" ]; then
            L10N_CODE=`/usr/bin/expr "$LANG" : '\(..\).*'`
    	    LC_MES_CODE=`/usr/bin/expr "$LC_MESSAGES" : '\(..\).*'`
    	    if [ "$LC_MES_CODE" = "$L10N_CODE" ]; then
              LC_MESSAGES=$LANG
              export LC_MESSAGES
            fi
        else
            L10N_CODE=$LANG
        fi
        [ "$L10N_CODE" != "en" ] && L10N_REQUIRED=1
    else
        L10N_CODE=C
    fi
}

setup_textdomain () {
#
#  PKG_DIR is the trigger here - if it is set, then we must
#  be running the install.  In that case we use /tmp/SyMON for
#  location of the link and the package dir as the TEXTDOMAIN.
#  Otherwise, everything goes in /var/opt/SUNWsymon.
#

    [ "$L10N_REQUIRED" -eq 0 ] && return
    if [ -n "$locale_directory" -a -d "$locale_directory" ]; then
	#It is core SunMC installation
	locale_dir="$locale_directory"
	link_dir=/tmp/SyMON
	[ ! -d "$link_dir" ] && /usr/bin/mkdir "$link_dir"
           
    else
        #either Addon only installation or some other script like es-setup
	locale_dir=$VAROPTDIR/locale
	link_dir=$VAROPTDIR/locale
    fi

    if [ $1 ]; then
        TEXTDOMAIN=$1
    else
        TEXTDOMAIN=ES_SCRIPTS
    fi
    export TEXTDOMAIN

    if [ -d "$locale_dir/$LANG" ]; then
	    TEXTDOMAINDIR="$locale_dir"
	    export TEXTDOMAINDIR
	    return
    fi

    if [ -z "$L10N_CODE" ] ; then
	    echolog 'L10N_CODE not set!, LANG=$2' "$LANG"
	    return
    fi

    if [ -d "$locale_dir/$L10N_CODE" ]; then
	    [ -h "$link_dir/$LANG" ] && /usr/bin/rm "$link_dir/$LANG"
	    /usr/bin/ln -s "$locale_dir/$L10N_CODE" "$link_dir/$LANG"
	    TEXTDOMAINDIR="$link_dir"
	    export TEXTDOMAINDIR
    else
	    echolog 'Cannot find locale directory for LANG: $2' "$LANG"
	    echolog 'Expected to find it in: $2' "$locale_dir/$L10N_CODE"
    fi

}  # setup_textdomain


set_basedir() {

    /usr/bin/pkginfo -qic symon
    if [ $? -eq 1 ] ; then
	echolog "No Sun Management Center Packages are installed. Exiting."
	exit 1
    fi

    BASEDIR=`/usr/bin/pkginfo -r SUNWescom`/SUNWsymon

    CONFDIR=/etc/opt/SUNWsymon
    SYMONHOME=$BASEDIR
    export SYMONHOME
    VAROPTDIR=/var/opt/SUNWsymon
    VARDIR=$VAROPTDIR/cfg
    PLATFORM_FILE="$VAROPTDIR/platform/platform.prop"
    export PLATFORM_FILE

    check_for_l10n
    [ "$L10N_REQUIRED" -eq 1 ] && setup_textdomain 

    ESROOT="$BASEDIR"
    export ESROOT   #  Needed by other scripts later
    #
    #  This might be called by a non-root user (such as someone
    #  starting the console with es-start -c).  Only create VARDIR
    #  if running as root.
    #

    ME=`/usr/ucb/whoami`

    [ ! -d "$VARDIR" ] && [ "$ME" = "root" ] && /usr/bin/mkdir -p "$VARDIR"

    ESPRIVATEID=esmaster; export ESPRIVATEID
    ESPUBLICID=espublic; export ESPUBLICID
}

set_xput() {

    [ -z "$OSVERS" ] && check_osver
    XPUT="$BASEDIR/base/sbin/sparc-sun-solaris${OSVERS}/xput"
    if [ ! -x "$XPUT" ] ; then
	echolog ""
	echolog 'Could not find xput executable: $2' $XPUT
	echolog ""
	exit 1
    fi
    XGET=`echo "$XPUT" | /usr/bin/sed s/xput/xget/`
}

set_inst_flags() {
    /usr/bin/pkginfo -qi SUNWesmod
    no_agent=$?

    /usr/bin/pkginfo -qi SUNWessrv
    no_server=$?
    no_metadata=$no_server

    /usr/bin/pkginfo -qi SUNWescon
    no_console=$?

    /usr/bin/pkginfo -qi SUNWescli
    no_cli=$?

    /usr/bin/pkginfo -qi SUNWessvc
    no_service=$?
}

set_paths() {

    
    if [ ! -f $BASEDIR/startup.profile ] ; then
	echolog 'Cannot find file $2' "$BASEDIR/startup.profile"
	exit 1
    fi
    
    User_Env_Path=`echo $PATH`

    . "$BASEDIR"/startup.profile
    
    PATH=$PATH:/usr/ccs/bin:$User_Env_Path
    export PATH
 
    ESROOT="$BASEDIR"

    if [ ! -f "$BASEDIR"/es.profile ] ; then
	echolog 'Cannot find file $2' "$BASEDIR/es.profile"


	exit 1
    fi
#
#   The following if condition is here for the reason that
#   the console installations do not have the base-setup.sh
#   It is okay not to source this file in case of console.
#
    if [ -f "$BASEDIR"/base/sbin/base-setup.sh ] ; then
	[ -n "$ESENV" ] && unset ESENV
	. "$BASEDIR"/es.profile
    fi

}

unset_env_variables() {

    [ -n "$JAVA_HOME" ] && unset JAVA_HOME
    #
    #  If we are running the Korn shell AND ENV is set then unset it.
    #  This prevents executing the ENV file, which runs when
    #  jre is executed.  It is common to put a setting for JAVA_HOME in that file.
    #

    #
    # Make sure SHELL is set . SHELL may not be set when Invoked from 
    # rc3.d.
    #

    if [ -n "$SHELL" ]; then

	[ `/usr/bin/expr "$SHELL" : '.*\(...\)'` = "ksh" ] && [ -n "$ENV" ] && unset ENV
    fi	

    if [ -n "$ESENV" ] ; then
	#echo "ESENV already set. Unsetting."
	unset ESENV
    fi
}

move_datfile() {
#
# $1=Dat file name
#
    if [ -f "$1" ] ; then
	echologverbose 'Moving $2 to $3' "$1" "$1.old"
	/usr/bin/mv "$1" "$1.old"
    fi
}

kill_processes() {
#
#  $1 = unique string used to identify process in ps output
#  $2 = signal to send to the process
#
    [ $1 -ne 0 ] && return 0

    pids=`/usr/ucb/ps -ww -a -g -x | /usr/bin/fgrep SUNWsymon | \
	/usr/bin/fgrep -v grep | /usr/bin/fgrep "$1" | /usr/bin/awk '{print $1}'`
    for each_pid in $pids
    do
        if [ -z "$2" ] ; then
	    /usr/bin/kill "$each_pid" >/dev/null 2>&1
        else
            /usr/bin/kill -s "$2" "$each_pid" >/dev/null 2>&1
        fi
    done

}

port_busy() {
#
#  $1 = X object value checked
#  $2 = component
#
    #  Save the original port number for use later
    port1="$port"

    echolog ""
    echolog ""
    echolog "--------------  WARNING  -------------------"
    echolog 'It appears that $2 $3 is already in use.' "$1" "$port"
    echolog 'Sun Management Center $2 may not be able to run due to this conflict.' "$2"
    echolog "There are two ways to correct this conflict:"
    echolog "  1. Reconfigure the port that Sun Management Center uses."
    echolog "  2. Stop the process that is using the port."
    proc_count=`/usr/bin/ps -aef | /usr/bin/fgrep -v grep | \
	/usr/bin/grep snmpdx | /usr/bin/wc -l`
    if [ "$proc_count" -ne 0 ] ; then
	echolog "    You are currently running snmpdx, which may be causing the conflict."
	echolog ""
    fi
    echolog ""
    ask_user 'Do you want to use a different port number for $2?' $2
    if [ "$answer" -eq 1 ]; then
	#get_input_from_user "What port number would you like to use?"
	get_input_from_user "Please enter any port [ 1100 to 65535 ] : "
	port="$answer"
	while :
	do
            #checking for the validity of the port number

            /usr/bin/expr $port + 1  1> /dev/null  2>&1
            if [ $? -ne 0 ] ; then
                get_input_from_user 'Port $2 is not a valid port number, try another number : ' "$port"
                port="$answer"
                continue
            fi
            if [ $port -lt 1100 ] ; then
                get_input_from_user "Please enter any port greater or equal to 1100 : "
                port="$answer"
                continue
            fi
            if [ $port -gt 65535 ] ; then
                get_input_from_user "Please enter any port less or equal to 65535 : "
                port="$answer"
                continue
            fi


            #checking for the validity of the port number done.


	    check_port_number $1 $2
	    [ $? -eq 0 ] && break
	    get_input_from_user 'Port $2 is also busy, try another - use s to skip setting the port number:' "$port"
	    if [ "$answer" = "s" ] ; then
		echolog ""
		echolog 'Skipping the setting of port number for $2' $1
		echolog 'NOTE: Prior to starting Sun Management Center $2, stop the process using port $3.' "$2" "$port1"
		port="$answer"
		break
	    fi
	    port="$answer"
	done
	if [ "$port" != "s" ] ; then
	    echolog 'Updating $2 with new port number.' "$VARDIR/domain-config.x"
	    [ -x "$XPUT" ] && set_xput
	    $XPUT $VARDIR/domain-config.x "$1" "$answer"
	fi
    else 
	echolog ""
	echolog 'NOTE: Prior to starting Sun Management Center $2, stop the process using port $3.' "$2" "$port1"
    fi

}

check_port_number() {
#
#  $1 = domain-config.x parameter to check
#  $2 = process to check
#

    /usr/bin/netstat -an -P udp | /usr/bin/grep "\*\.$port " >/dev/null
    [ $? -ne 0 ] && return 0
    #
    #  Make sure it isn't in use by one of the SyMON processes!
    #
    numproc=`/usr/bin/ps -aef | /usr/bin/grep 'esd - init $2' | \
	/usr/bin/grep -v /usr/bin/grep | /usr/bin/wc -l`
    numproc=`echo $numproc`
    [ "$numproc" -eq 0 ] && return 1

    return 0
}

check_port() {
#
#  $1 = domain-config.x parameter to check
#  $2 = process to check
#

    port=""
    port=`$XGET "$VARDIR/domain-config.x" $1`
#    [ -z "$port" ] && port_busy $1 $2

    check_port_number $1 $2
    [ $? -ne 0 ] && port_busy $1 $2
}



#this defines few easy functions to deal with the configuration
#file. The configuration file for these functions has very simple
#format. They can have comments and comments must start with '#' at
#the first charcter position.
#Every noncomment line in the file will represent a name value pair.
#The format of the name value pair is 'name=value'
#the name must start at the first character. The name is case sensitive.


#gets the value of variable from cfgfile. If file not present
#or not writable then returns 2.If variable is not in file then 
#returns 1 else returns 0 and prints out the value on stdout

get_value_from_cfgfile()
{
    cfgfile=$1
    variable=$2
    separator=$3
    if [ ! -r $cfgfile ] ; then
        return 2
    fi
    varline=""
    varline=`/usr/bin/cat $cfgfile | /usr/bin/grep "^$variable" 2> /dev/null`
    if [ -z "$varline" ] ; then
         return 1
    fi
    value=`echo $varline | /usr/bin/cut -f2 -d $separator `
    echo $value
    return 0
}

#puts the variable and value pair in the cfgfile. If file not present
#or not writable then returns 2 else returns 0. If the variable is 
#already defined in the file then replaces the old by new one.

put_var_in_cfgfile()
{
    cfgfile=$1
    variable=$2
    value=$3
    separator=$4
    if [ ! -w $cfgfile ] ; then
        return 2
    fi
    varline=""
    varline=`/usr/bin/cat $cfgfile | /usr/bin/grep "^$variable$separator" 2> /dev/null`
    if [ -z "$varline" ] ; then
         echo "$variable$separator$value" >> $cfgfile
         return 0
    fi
    /usr/bin/mv $cfgfile $cfgfile.old
    while read LINE
    do
        echo $LINE | /usr/bin/grep "^$variable$separator" > /dev/null 2>&1 
        if [ $? -eq 0 ] ; then
              echo "$variable$separator$value" >> $cfgfile
              continue
        fi
        echo $LINE >> $cfgfile
     
    done  < $cfgfile.old
    return 0
    
}

#puts the variable and value pair in the cfgfile. If file not present
#or not writable then returns 2 else returns 0. If the variable is 
#already defined in the file then appends the new value in the old value.

append_var_in_cfgfile()
{
    cfgfile=$1
    variable=$2
    value=$3
    separator=$4
    if [ ! -w $cfgfile ] ; then
        return 2
    fi
    varline=""
    varline=`/usr/bin/cat $cfgfile | /usr/bin/grep "^$variable$separator" 2> /dev/null`
    if [ -z "$varline" ] ; then
         echo "$variable$separator$value" >> $cfgfile
         return 0
    fi
    /usr/bin/mv $cfgfile $cfgfile.old
    while read LINE
    do
        echo $LINE | /usr/bin/grep "^$variable$separator"  > /dev/null 2>&1
        if [ $? -eq 0 ] ; then
            currentvalue=`echo $LINE | /usr/bin/cut -f2 -d$separator`
            echo "$variable$separator$currentvalue $value" >> $cfgfile
            continue
        fi
        echo $LINE >> $cfgfile
     
    done  < $cfgfile.old
    return 0
}

#removes one entry from the file. If the file is not present or not 
#writable then returns 2 else returns 0.

remove_var_from_cfgfile()
{
    cfgfile=$1
    variable=$2
    separator=$3
    if [ ! -w $cfgfile ] ; then
        return 2
    fi
    varline=""
    varline=`/usr/bin/cat $cfgfile | /usr/bin/grep "^$variable$separator" 2> /dev/null`
    if [ -z "$varline" ] ; then
         return 0
    fi
    /usr/bin/mv $cfgfile $cfgfile.old
    while read LINE
    do
        echo $LINE | /usr/bin/grep "^$variable$separator"  > /dev/null 2>&1
        if [ $? -eq 0 ] ; then
            continue
        fi
        echo $LINE >> $cfgfile
     
    done  < $cfgfile.old
    return 0
}

is_port_open()
{
    /usr/bin/netstat -an -P tcp | /usr/bin/grep "\*\.$1 " >/dev/null
    [ $? -ne 0 ] && return 0
    return 1
}

echologverbose() {
    if [ "$VERBOSE" = "ON" ] ; then
	eval echo "`/usr/bin/gettext "$1"`"
    fi
    if [ "$LOGFILE" != "" ] ; then
	eval echo "`/usr/bin/gettext "$1"`" >> "$LOGFILE"
    fi
}

get_security_key () {

#
#  $1 is do_default flag
#

    echolog "This part of setup generates security keys used for communications"
    echolog "between processes. A seed must be provided to initialize the keys."
    echolog "Please make sure you use the same seed for all the machines you install."
    echolog "You may like to keep record of this seed for future use."
    echolog ""

    if [ "$1" = "1" ]; then
       seed=maplesyr
    else
      while [ 1 ]; do
         key_entered=0
         while [ $key_entered -eq 0 ]; do
            /usr/bin/stty -echo
            get_input_from_user "Please enter the seed to generate keys:"  
            /usr/bin/stty echo
            echolog ''
            key_entered=1
            [ -z "$answer" ] && key_entered=0
         done
    
         temp_seed=$answer
   
         /usr/bin/stty -echo
         get_input_from_user "Please re-enter the seed to confirm:"  
         /usr/bin/stty echo
         echolog ''
         [ "$answer" = "$temp_seed" ] && seed=$temp_seed && break
         echolog 'This does not match the seed you entered the first time.'
         echolog ''
      done
    fi
    
    echolog ''
    key_char_count=`/usr/bin/echo $seed | /usr/bin/wc -c`
    [ $key_char_count -gt 9 ] && seed=`/usr/bin/echo $seed  | /usr/bin/cut -c1-8`
    
    unset key_char_count temp_seed
    
    return 0
}



# params:  groupname

verifyCreateOracleGroup ()
{
  if [ -z "$1" ]
  then
    echolog 'verifyCreateOracleGroup - the groupname parm is null, returning failure code'
    return 99
  fi



  if [ ! -r /etc/group ]
  then
     echolog 'verifyCreateOracleGroup - /etc/group file not found'
     echolog
     echolog
     return 1
  fi

  gname=$1



  gline=`/usr/bin/grep -i "^[ ]*$gname:" /etc/group`


  if [ "$gline" = "" ]
  then
    gncnt=0
  else
    gncnt=`echo $gline | wc -l`
  fi


  if [ $gncnt -ne 1 ]
  then
    if [ $gncnt -eq 0 ]
    then
       echologverbose 'verifyCreateOracleGroup - group named $2 does not exist in the /etc/group file' "$1"

       grstring=`$ESROOT/base/bin/sparc-sun-solaris${OSVERS}/smcgrchk $gname 2>/dev/null`

       retcode=$?

       if [ $retcode = 0 ]
       then
          echologverbose 'the group $2 was found by call to getgrnam system call ,probably nis, nis+, etc..' "$1"
       else
         if [ $retcode = 2 ]
         then

            echologverbose 'creating new group $2' "$1"

            /usr/sbin/groupadd $1

            retcode=$?

            if [ $retcode != 0 ]
            then
               echolog 'verifyCreateOracleGroup - failed to create new group $2' "$1"
               return $retcode
            fi

            gline=`/usr/bin/grep -i "^[ ]*$gname:" /etc/group`
  
            if [ "$gline" = "" ]
            then
              gncnt=0
            else
              gncnt=`echo $gline | wc -l`
            fi

            if [ $gncnt -ne 1 ]
            then
               echolog 'verifyCreateOracleGroup - inconsistent group details in /etc/group file for new group $2' "$1"
               return 1
            fi

         fi
       fi

    fi 
    if [ $gncnt -gt 1 ]
    then
       echolog 'verifyCreateOracleGroup - more than one entry for the group named $2 exists in the /etc/group file' "$1"
       return 1
    fi 
  fi 



  grstring=`$ESROOT/base/bin/sparc-sun-solaris${OSVERS}/smcgrchk $gname 2>/dev/null`

  retcode=$?

  if [ $retcode = 0 ]
  then
     gvalue=`echo $grstring | awk -F':' '{print $3}'`

     retcode=$?

     if [ $retcode != 0 ]
     then
        echolog 'verifyCreateOracleGroup - failed to obtain group id count info from awk command'
        return $retcode
     fi

  else
     if [ $retcode = 2 ]
     then

        gvalue=`echo $gline | awk -F':' '{print $3}'`

        retcode=$?

        if [ $retcode != 0 ]
        then
           echolog 'verifyCreateOracleGroup - failed to obtain group id count info from awk command'
           return $retcode
        fi

        gvcnt=`cat /etc/group | awk -F':' '{print $3}' | /usr/bin/grep -i $gvalue | wc -l`

        if [ $gncnt -ne 1 ]
        then
          if [ $gncnt -eq 0 ]
          then
             echolog 'verifyCreateOracleGroup - group id for group $2 does not exist in the /etc/group file' "$1"
             return 1
          fi 
          if [ $gncnt -gt 1 ]
          then
             echolog 'verifyCreateOracleGroup - more than one entry for the group id for group $2 exists in the /etc/group file' "$1"
             return 1
          fi 
        fi 

     fi
  fi


 groupid=$gvalue

  return 0
}




# params:  username, groupname, groupvalue, homedir

verifyCreateOracleUser ()
{
  if [ -z "$1" ]
  then
    echolog 'verifyCreateOracleUser - the username parm is null, returning failure code'
    return 99
  fi

  if [ -z "$2" ]
  then
    echolog 'verifyCreateOracleUser - the groupname parm is null, returning failure code'
    return 99
  fi

  if [ -z "$3" ]
  then
    echolog 'verifyCreateOracleUser - the groupid parm is null, returning failure code'
    return 99
  fi

  if [ -z "$4" ]
  then
    echolog 'verifyCreateOracleUser - the homedir parm is null, returning failure code'
    return 99
  fi



  if [ ! -r /etc/passwd ]
  then
     echolog 'verifyCreateOracleUser - /etc/passwd file not found'
     echolog
     echolog
     return 1
  fi

  uname=$1
  gname=$2
  gvalue=$3
  homedir=$4



  trailslash=`echo $homedir | /usr/bin/grep -i "/$" | wc -l`

  if [ $trailslash = 0 ]
  then
     homedir="$homedir/"
  fi



  uline=`/usr/bin/grep -i "^[ ]*$uname:" /etc/passwd`



  if [ "$uline" = "" ]
  then
    uncnt=0

    if [ -d $homedir$uname ]; then
      /usr/bin/rm -rf $homedir$uname
    fi

  else
    uncnt=`echo $uline | wc -l`

    if [ $uncnt -eq 1 ]
    then

      if [ ! -d $homedir$uname ]; then
         echolog "verifyCreateOracleUser - the existing $uname user home directory - $homedir$uname does not exist"
         echolog "verifyCreateOracleUser - will perform a userdel on the current user, before re-creating"

         hdir=`echo $uline | awk -F':' '{print $6}'`
   
         hdircnt=`echo $hdir | wc -l`

         if [ $hdircnt -eq 1 ]
         then
            if [ -d $hdir ]
            then
               /usr/sbin/userdel -r $uname
               retcode=$?
            else
              /usr/sbin/userdel $uname
              retcode=$?
            fi
         else
              /usr/sbin/userdel $uname
              retcode=$?
         fi


         if [ $retcode != 0 ]
         then
            echolog 'verifyCreateOracleUser - failed to delete user $2' "$1"
            return $retcode
         fi

         # critical line to ensure that subsequent logic creates the user
         uncnt=0

      fi

    else

       if [ -d $homedir$uname ]; then
         /usr/bin/rm -rf $homedir$uname
       fi

    fi


  fi


  if [ $uncnt -ne 1 ]
  then
    if [ $uncnt -eq 0 ]
    then
       echologverbose 'verifyCreateOracleUser - user named $2 does not exist in the /etc/passwd file' "$1"

       grstring=`$ESROOT/base/bin/sparc-sun-solaris${OSVERS}/smcpwchk $uname 2>/dev/null`

       retcode=$?

       if [ $retcode = 0 ]
       then
          echologverbose 'the group $2 was found by call to getpwnam system call ,probably nis, nis+, etc..' "$1"

         if [ ! -d "${homedir}${uname}" ]; then
            /usr/bin/mkdir -p ${homedir}${uname}
         fi

       else
         if [ $retcode = 2 ]
         then

            echologverbose 'creating new user $2' "$1"

            /usr/sbin/useradd -d $homedir$uname -m -g $gvalue -s /bin/sh $uname > /dev/null 2>&1

            retcode=$?
 
            if [ $retcode != 0 ]
            then
               echolog 'verifyCreateOracleUser - failed to create new user $2' "$1"
               return $retcode
            fi

            uline=`/usr/bin/grep -i "^[ ]*$uname:" /etc/passwd`
 
            if [ "$uline" = "" ]
            then
              uncnt=0
            else
              uncnt=`echo $uline | wc -l`
            fi

            if [ $uncnt -ne 1 ]
            then
               echolog 'verifyCreateOracleUser - inconsistent user details in /etc/passwd file for new user $2' "$1"
               return 1
            fi

            if [ ! -d $homedir$uname ]; then
               echolog "verifyCreateOracleUser - home directory - $homedir$uname does not exist"
               return 1
            fi

         fi
       fi

    fi 

    if [ $uncnt -gt 1 ]
    then
       echolog 'verifyCreateOracleUser - more than one entry for the user named $2 exists in the /etc/passwd file' "$1"
       return 1
    fi 

  fi 



  pwstring=`$ESROOT/base/bin/sparc-sun-solaris${OSVERS}/smcpwchk $uname 2>/dev/null`

  retcode=$?

  if [ $retcode = 0 ]
  then
     gvalue=`echo $pwstring | awk -F':' '{print $3}'`

     retcode=$?

     if [ $retcode != 0 ]
     then
        echolog 'verifyCreateOracleUser - failed to obtain user id value from awk command'
        return $retcode
     fi

  else
     if [ $retcode = 2 ]
     then

        uvalue=`echo $uline | awk -F':' '{print $3}'`

        retcode=$?

        if [ $retcode != 0 ]
        then
           echolog 'verifyCreateOracleUser - failed to obtain user id value from awk command'
           return $retcode
        fi

        uvcnt=`cat /etc/passwd | awk -F':' '{print $3}' | /usr/bin/grep -i $uvalue | wc -l`

        if [ $uvcnt -ne 1 ]
        then
          if [ $uvcnt -eq 0 ]
          then
             echolog 'verifyCreateOracleUser - user id for user $2 does not exist in the /etc/passwd file' "$1"
             return 1
          fi 
          if [ $uvcnt -gt 1 ]
          then
             echolog 'verifyCreateOracleUser - more than one entry for the user id for user $2 exists in the /etc/passwd file' "$1"
             return 1
          fi 
        fi 

     fi
  fi



  userid=$uvalue

 
  #
  #Now the user is created successfully, so add .hushlogin file in the
  #home directory to supress the login messages.
  #

 if [ -d "${homedir}${uname}" ]; then
    /usr/bin/touch "${homedir}${uname}/.hushlogin"
    if [ $? -ne 0 ] ; then
       echolog 'Could not touch the file .hushlogin, login messages will be displayed.'
    fi
 fi





  return 0
}




# params:  groupname

verifyDeleteOracleGroup ()
{

  if [ -z "$1" ]
  then
    echolog 'verifyDeleteOracleGroup - the groupname parm is null, returning failure code'
    return 99
  fi



  if [ ! -r /etc/group ]
  then
     echolog 'verifyDeleteOracleGroup - /etc/group file not found'
     echolog
     echolog
     return 1
  fi

  gname=$1



  gline=`/usr/bin/grep -i "^[ ]*$gname:" /etc/group`


  if [ "$gline" = "" ]
  then
    gncnt=0
  else
    gncnt=`echo $gline | wc -l`
  fi


    if [ $gncnt -gt 0 ]
    then

       #
       #  May have to check for the execution of ypbind? and then
       #  we may also need to check for nis+ process nisbind? or something
       #

       if [  -r /etc/nsswitch.conf ]
       then
         grfirst=`/usr/bin/grep -i "^[ ]*group[ ]*:" /etc/nsswitch.conf | cut -f2 -d':' | awk '{print $1}'`
         nsfile_exists="true"
       else
         # if file does not exist, then set grfirst to equal files
         # so remaining logic flows as predicted
         grfirst="files"
         nsfile_exists="false"
       fi 

         #
         #  If the the nsswitch.conf files exists and the first entry for group 
         #  is not files, then inform the user we can't try a groupdel.  The user 
         #  will have to deal with this manually.
         #

         if [ "$grfirst" != "files" ]
         then
            echolog 'verifyDeleteOracleGroup - the nsswitch.conf files first entry for group is not files'
            echolog 'verifyDeleteOracleGroup - you must manually delete the local group $2' "$SMCORAG" 
         else

            echologverbose 'verifyDeleteOracleGroup - group named $2 does exist in the /etc/group file' "$1"
            echologverbose 'deleting existing group $2' "$1"
            /usr/sbin/groupdel $1
 
            retcode=$?

            if [ $retcode != 0 ]
            then
               echolog 'verifyDeleteOracleGroup - failed to delete group $2' "$1"
            
               if [ $retcode = 6 ]
               then
                   echolog 'verifyDeleteOracleGroup -  **** Please manually delete group  ****'
               fi
               return $retcode
            fi

            gline=`/usr/bin/grep -i "^[ ]*$gname:" /etc/group`

            if [ "$gline" = "" ]
            then
              gncnt=0
            else
              gncnt=`echo $gline | wc -l`
            fi
 
            if [ $gncnt -ne 0 ]
            then
               echolog 'verifyDeleteOracleGroup - inconsistent group details in /etc/group file for delete group $2' "$1"
               return 1
            fi

         fi

    fi 


   return 0

}




# params:  username

verifyDeleteOracleUser ()
{
  if [ -z "$1" ]
  then
    echolog 'verifyDeleteOracleUser - the username parm is null, returning failure code'
    return 99
  fi


  if [ ! -r /etc/passwd ]
  then
     echolog 'verifyDeleteOracleUser - /etc/passwd file not found'
     echolog
     echolog
     return 1
  fi

  uname=$1

  uline=`/usr/bin/grep -i "^[ ]*$uname:" /etc/passwd`



  if [ "$uline" = "" ]
  then
    uncnt=0
  else
    uncnt=`echo $uline | wc -l`
  fi


    if [ $uncnt -gt 0 ]
    then

       #
       #  May have to check for the execution of ypbind? and then
       #  we may also need to check for nis+ process nisbind? or something
       #

       if [  -r /etc/nsswitch.conf ]
       then
         pwfirst=`/usr/bin/grep -i "^[ ]*passwd[ ]*:" /etc/nsswitch.conf | cut -f2 -d':' | awk '{print $1}'`
         nsfile_exists="true"
       else
         # if the file does not exist, then set pwfirst to files
         # so the remaining logic flows as predicted
         pwfirst="files"
         nsfile_exists="false"
       fi 

         #
         #  If the the nsswitch.conf files exists and the first entry for passwd 
         #  is not files, then inform the user we can't try a userdel.  The user 
         #  will have to deal with this manually.
         #

         if [ "$pwfirst" != "files" ]
         then
            echolog 'verifyDeleteOracleUser - the nsswitch.conf files first entry for passwd is not files'
            echolog 'verifyDeleteOracleUser - you must manually delete the local user $2' "$SMCORAU" 
         else
            echologverbose 'verifyDeleteOracleUser - user named $2 does exist in the /etc/passwd file' "$1"
            echologverbose 'deleting user $2' "$1"

            hdir=`echo $uline | awk -F':' '{print $6}'`
   
            hdircnt=`echo $hdir | wc -l`

            if [ $hdircnt -eq 1 ]
            then
               if [ -d $hdir ]
               then
                  /usr/sbin/userdel -r $uname
                  retcode=$?
               else
                 /usr/sbin/userdel $uname
                 retcode=$?
               fi
            else
                 /usr/sbin/userdel $uname
                 retcode=$?
            fi


            if [ $retcode != 0 ]
            then
               echolog 'verifyDeleteOracleUser - failed to delete user $2' "$1"
               return $retcode
            fi

            uline=`/usr/bin/grep -i "^[ ]*$uname:" /etc/passwd`
 
            if [ "$uline" = "" ]
            then
              uncnt=0
            else
              uncnt=`echo $uline | wc -l`
            fi

            if [ $uncnt -ne 0 ]
            then
               echolog 'verifyDeleteOracleUser - inconsistent user details in /etc/passwd file for deleted user $2' "$1"
               return 1
            fi

         fi

    fi 


   return 0

}

kill_all_processes() {
#
#  $1 = unique string used to identify process in ps output
#  $2 = signal to send to the process
#

    pids=`/usr/ucb/ps -ww -a -g -x | /usr/bin/fgrep -v grep | \
	/usr/bin/fgrep "$1" | /usr/bin/awk '{print $1}'`
    for each_pid in $pids
    do
        if [ -z "$2" ] ; then
	    /usr/bin/kill "$each_pid" >/dev/null 2>&1
        else
            /usr/bin/kill -s "$2" "$each_pid" >/dev/null 2>&1
        fi
    done

}

# Gets a value from a specified automated configuration file
# $1 = key
# Return: the value of the key
getAutoConfig() {
    if [ "$AUTOCONFIG_RUNFILE" = "" ] ; then
	echo ""
    else
	tmp_getautoconfig=`$CMD_GREP "$1" "$AUTOCONFIG_RUNFILE"`
	if [ "$tmp_getautoconfig" = "" ] ; then
	    echo ""
	else
	    echo `$CMD_EXPR $tmp_getautoconfig : "${1}\(.*\)" | $CMD_CUT -d: -f2`
	fi
    fi
}

# Saves a value to a specified automated configuration file
# $1 = key
# $2 = value
saveAutoConfig() {
    [ "$AUTOCONFIG_SAVEFILE" = "" ] && return
    echo "${1}:${2}" >> "$AUTOCONFIG_SAVEFILE"
}



#
#  2.1 to 3.0 cfg related functionality
#
#

copy_alarm_files() {
    if [ $# -ne 1  ] 
    then
        echolog 'USAGE: copy_alarm_files <modules-file>'
        return
    fi

    MODULES_FILE=$1

    cat ${MODULES_FILE} | cut -d '=' -f1 | cut -d ' ' -f1 | while read INSTANCE
    do 
        INSTANCE_FILE=${EXPORT_CFG_DIRECTORY}/${INSTANCE}.dat
        
        if [ -f ${INSTANCE_FILE} ] 
        then
            echologverbose 'cp ${INSTANCE_FILE}       ${SYMON_VAR}/cfg'
            cp ${INSTANCE_FILE}       ${SYMON_VAR}/cfg
        fi
    done
}


fix_ent_vers() {
#
#  Fixes the enterprise and version fields
#
#    echo "\nBefore:\n$LINE"

    LINE=`echo $LINE | /usr/bin/sed s/halcyon/sun/g`
    LINE=`echo $LINE | /usr/bin/sed s/'version *= *\"1\.0\"'/'version = \"2.0\"'/`

}

fix_topology_modules() {

	[ -f $VARDIR/topology-modules-d.dat.old ] && /usr/bin/rm $VARDIR/topology-modules-d.dat.old
	if [ -f $VARDIR/topology-modules-d.dat ]; then
		/usr/bin/mv $VARDIR/topology-modules-d.dat $VARDIR/topology-modules-d.dat.old
		SED=/usr/bin/sed

		while read LINE
		do
			module=`echo $LINE | /usr/bin/awk '{print $1}'`
			case $module in
				domain-control)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.serverSupport\.domain\.control/prod.sunsymon.agent.modules.domainControl/g`
					LINE=`echo $LINE | $SED s/'location = \"\"'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.domainControl\"'/`
					;;
				topology-license)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.localApplication\.topology\.license/prod.sunsymon.agent.modules.topologyLicense/g`
					LINE=`echo $LINE | $SED s/'location = \"\"'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.topologyLicense\"'/`
					;;
				topology*)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.serverSupport\.topology/prod.sunsymon.agent.modules.topology/g`
					LINE=`echo $LINE | $SED s/'location = \"\"'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.topology\"'/`
					;;
				discovery)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.serverSupport\.discovery/prod.sunsymon.agent.modules.discovery/g`
					LINE=`echo $LINE | $SED s/'location = \"\"'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.discovery\"'/`
					;;
				*)
					;;
			esac
			LINE=`echo $LINE | /usr/bin/sed s/'"\; '/'\\\"\; '/g`
			LINE=`echo $LINE | /usr/bin/sed s/'= "'/'= \\\"'/g`
			#  Remove the first \
			LINE=`echo $LINE | /usr/bin/sed s:'\\\'::`
			echo "$LINE" >> $VARDIR/topology-modules-d.dat

		done < "$VARDIR/topology-modules-d.dat.old"

	fi

}

fix_cfgserver_modules() {

	[ -f $VARDIR/cfgserver-modules-d.dat.old ] && /usr/bin/rm $VARDIR/cfgserver-modules-d.dat.old
	if [ -f $VARDIR/cfgserver-modules-d.dat ] ; then
		/usr/bin/mv $VARDIR/cfgserver-modules-d.dat $VARDIR/cfgserver-modules-d.dat.old
		SED=/usr/bin/sed

		while read LINE
		do
			module=`echo $LINE | /usr/bin/awk '{print $1}'`
			case $module in
				cfgserver)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.serverSupport\.cfgserver/prod.sunsymon.agent.modules.cfgserver/g`
					LINE=`echo $LINE | $SED s/'location = \"\"'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.cfgserver\"'/`
					;;
				*)
					;;
			esac
			LINE=`echo $LINE | /usr/bin/sed s/'"\; '/'\\\"\; '/g`
			LINE=`echo $LINE | /usr/bin/sed s/'= "'/'= \\\"'/g`
			#  Remove the first \
			LINE=`echo $LINE | /usr/bin/sed s:'\\\'::`
			echo "$LINE" >> $VARDIR/cfgserver-modules-d.dat

		done < "$VARDIR/cfgserver-modules-d.dat.old"

	fi
}

fix_trap_modules() {

	return 0

}

fix_event_modules() {

	[ -f $VARDIR/event-modules-d.dat.old ] && /usr/bin/rm $VARDIR/event-modules-d.dat.old
	if [ -f $VARDIR/event-modules-d.dat ] ; then 
		/usr/bin/mv $VARDIR/event-modules-d.dat $VARDIR/event-modules-d.dat.old
		SED=/usr/bin/sed

		while read LINE
		do
			module=`echo $LINE | /usr/bin/awk '{print $1}'`
			case $module in
				eventmgr)
					fix_ent_vers
					LINE=`echo $LINE | $SED s/primealert\.modules\.localApplication\.eventmgr/prod.sunsymon.agent.modules.eventmgr/g`
					LINE=`echo $LINE | $SED s/'location = ""'/'location = \".iso.org.dod.internet.private.enterprises.sun.prod.sunsymon.agent.modules.eventmgr\"'/`
					;;
				*)
					;;
			esac
			LINE=`echo $LINE | /usr/bin/sed s/'"\; '/'\\\"\; '/g`
			LINE=`echo $LINE | /usr/bin/sed s/'= "'/'= \\\"'/g`
			#  Remove the first \
			LINE=`echo $LINE | /usr/bin/sed s:'\\\'::`
			echo "$LINE" >> $VARDIR/event-modules-d.dat

		done < "$VARDIR/event-modules-d.dat.old"

	fi
}

fix_cstservice_modules() {

	return 0

}

import_2x_data_files()
{
  #location of Symon 2.1 installation
  OPT=`/usr/bin/pkginfo -r SUNWescom`

  # Base location of exported data
  EXPORT_BASE_DIRECTORY=/var/opt/SUNWsymon/2x

  # Location of exported cfg files
  EXPORT_CFG_DIRECTORY=${EXPORT_BASE_DIRECTORY}/cfg

  # Root of Symon 2.1
  SYMON_OPT=${OPT}/SUNWsymon

  # Root of Symon 2.1 configuration
  SYMON_VAR=/var/opt/SUNWsymon
  VARDIR=${SYMON_VAR}/cfg

  # CFG import log file
  IMPORT_CFG_LOG_TEMP=${EXPORT_BASE_DIRECTORY}/db-cfg-import.$$
  IMPORT_CFG_LOG=${EXPORT_BASE_DIRECTORY}/db-cfg-import.log

  export OPT
  export EXPORT_BASE_DIRECTORY
  export EXPORT_CFG_DIRECTORY
  export SYMON_VAR

  UPGRADE_DONE="YES"
  export UPGRADE_DONE
(
    # ====================== RESTORE TOPOLOGY DATA =======================
    echolog 'Merging topology files'
    cp  ${EXPORT_CFG_DIRECTORY}/topology+view-*.dat     ${SYMON_VAR}/cfg
    cp  ${SYMON_OPT}/base/cfg/topology-modules-d.dat    ${SYMON_VAR}/cfg
    chmod 755 ${SYMON_VAR}/cfg/topology-modules-d.dat
    if [ -f ${EXPORT_CFG_DIRECTORY}/topology-modules-d.dat ]
    then
        /usr/bin/grep 'topology+view-' ${EXPORT_CFG_DIRECTORY}/topology-modules-d.dat >>  ${SYMON_VAR}/cfg/topology-modules-d.dat
    fi

    # ====================== RESTORE CONFIG DATA =======================
    echolog 'Restoring config files'
	if [ -f ${EXPORT_CFG_DIRECTORY}/config-diffs.dat ]
	then
        cp  ${EXPORT_CFG_DIRECTORY}/config-diffs.dat ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/config-id.dat ]
	then
        cp  ${EXPORT_CFG_DIRECTORY}/config-id.dat ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/cfgserver-engines.dat ]
	then
        cp  ${EXPORT_CFG_DIRECTORY}/cfgserver-engines.dat ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/cfgserver-users.dat ]
	then
        cp  ${EXPORT_CFG_DIRECTORY}/cfgserver-users.dat ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/auth-suppl.dat ]
	then
		cp  ${EXPORT_CFG_DIRECTORY}/auth-suppl.dat ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/server-global-keys.x ]
	then
		cp  ${EXPORT_CFG_DIRECTORY}/server-global-keys.x ${SYMON_VAR}/cfg
	fi

	if [ -f ${EXPORT_CFG_DIRECTORY}/domain-config.x ]
	then
		cp  ${EXPORT_CFG_DIRECTORY}/domain-config.x ${SYMON_VAR}/cfg
	fi

    # Restore usm files

    cp ${EXPORT_CFG_DIRECTORY}/*usm*.dat ${SYMON_VAR}/cfg

    # Restore engine files

    cp ${EXPORT_CFG_DIRECTORY}/*engine*.dat ${SYMON_VAR}/cfg

    # ====================== UPDATE base-modules-d.dat ===================
    echolog 'Merging base-modules-d.dat'
    if [ -f ${EXPORT_CFG_DIRECTORY}/base-modules-d.dat ]
    then
        cp ${EXPORT_CFG_DIRECTORY}/base-modules-d.dat     ${SYMON_VAR}/cfg

        /usr/bin/grep 'hdreg =' ${SYMON_OPT}/base/cfg/base-modules-d.dat >> ${SYMON_VAR}/cfg/base-modules-d.dat
        /usr/bin/grep 'mcp =' ${SYMON_OPT}/base/cfg/base-modules-d.dat >> ${SYMON_VAR}/cfg/base-modules-d.dat

        copy_alarm_files ${EXPORT_CFG_DIRECTORY}/base-modules-d.dat
    fi

    fix_topology_modules
    fix_cfgserver_modules
    fix_trap_modules
    fix_event_modules
    fix_cstservice_modules
) > ${IMPORT_CFG_LOG_TEMP} 2>&1

mv ${IMPORT_CFG_LOG_TEMP} ${IMPORT_CFG_LOG}

}

# Java version checking
# This function expects JAVA_HOME to be pointing to the base of the 
# directory containing a full JDK.
#
#
# Return: 0 - successful setting .java_home file and java dir in Registry
#         1 - java not found in path
#         2 - java found but not JAR
# Sets: SYMON_JAVAHOME JAVA_VERSION JAVA_FULLVERSION JAVA_MINIMUM_REQUIRED
#
checkJavaOK() {

    #FOR_INSTALL=0
    #[ "$1" = "FOR_INSTALL" ] && FOR_INSTALL=1
    
    # Check for OS version
    #
    [ -z "$OSVERS" ] && check_osver # Must in case of relocated root.
    JAVA_MINIMUM_REQUIRED="Solaris_JDK_1.2.2_06"
    [ "$OSVERS" = "2.9" -o "$OSVERS" = "9" -o "$OSVERS" = "5.9" ] && JAVA_MINIMUM_REQUIRED="Solaris_JDK-1.4"
    
    java_status=1
    JAVA_HOME_FILE="$VAROPTDIR/.java_home"    
    
    #check if JAVA_HOME is set
    #
    if [ ! -z "$JAVA_HOME" -a -x ${JAVA_HOME}/bin/java ] ; then            
        SYMON_JAVAHOME=$JAVA_HOME
        checkJavaVersion #"$FOR_INSTALL"
        java_status=$?
        [ $java_status -eq 0 ] && updateJavaHomeFile && return $?
        [ $java_status -eq 2 ] && return 2
    fi    

    #check .java_home file
    #   
    if [ -f "$JAVA_HOME_FILE" ] ;then
       if [ -x "`/usr/bin/cat $JAVA_HOME_FILE`/bin/java" ] ; then           
          SYMON_JAVAHOME=`/usr/bin/cat $JAVA_HOME_FILE`
          checkJavaVersion #"$FOR_INSTALL"
          java_status=$?
          [ $java_status -eq 0 -o $java_status -eq 2 ] && return $java_status
       fi    
    fi
    
    # Last try - default location
    #
    if [ -x "/usr/java1.2/bin/java" ] ; then
       SYMON_JAVAHOME=/usr/java1.2
       checkJavaVersion #"$FOR_INSTALL"
       java_status=$?
       [ $java_status -eq 0 ] && updateJavaHomeFile && return $?
       [ $java_status -eq 2 ] && return 2
    fi

    return 1
   
}   

# returns 0 if valid version
#         1 if unsupported
#         2 if JAVA is present but not JAR
# Input $1 = 1 => FOR_INSTALL else for other scripts
#
checkJavaVersion () {      

    retval=1
    JAVA=${SYMON_JAVAHOME}/bin/java
    JAVA_VERSION=`$JAVA -version 2>&1 | /usr/bin/head -1 | /usr/bin/cut -d' ' -f3`
    JAVA_FULLVERSION=`$JAVA -fullversion 2>&1 | /usr/bin/head -1 | /usr/bin/cut -d' ' -f4`
    OUT=`echo $JAVA_VERSION  | /usr/bin/tr -d "\"" | /usr/bin/tr '.' ' '`
    set $OUT    
    
    MINOR=""
    case $1 in   
        1*)  case $2 in
                0*)  retval=1
                     ;;
                1*)  retval=1
                     ;;        
                2*)  MINOR=`/usr/bin/echo $JAVA_FULLVERSION  | /usr/bin/sed 's/^.*\(1\.2\.[0-9_]*\).*$/\1/' | /usr/bin/cut -d_ -f2`
                     if [ "$OSVERS" = "2.9" -o "$OSVERS" = "9" -o "$OSVERS" = "5.9" ]; then
                        retval=1
                     else
                        if [ -z "$MINOR" ] ; then
                           retval=1
                        else
                           # Following logic is required for minor version check of java
                           # e.g. 1.2.2_007 should not treated latter than 1.2.2_06
                           #
                           cmp_string="06" # For s9 , the min version is 1.2.2_06
                           cmp_length=3
                           minor_length=`/usr/bin/echo $MINOR | /usr/bin/wc -c `
                           while [ $minor_length -gt $cmp_length ]; do
                              cmp_string="${cmp_string}0" # Attach 0 at back - equivalant of multiply by 10
                              cmp_length=`/usr/bin/expr $cmp_length + 1`
                           done
                           [ $MINOR -ge $cmp_string ] && retval=0
                        fi
                     fi
                     ;;
                    
                3*)  # Supported till solaris 8 only
                     if [ "$OSVERS" = "2.9" -o "$OSVERS" = "9" -o "$OSVERS" = "5.9" ]; then
                        retval=1
                     else
                        retval=0
                     fi
                     ;; 
            
                *)   retval=0
                     ;;
            
            esac
            ;;               
        *)
            ;;
    
    esac
   
    # Find if JAR is present or not 
    #
    if [ $retval -eq 0 ] ; then
       [ ! -x "${SYMON_JAVAHOME}/bin/jar" ] && retval=2
       if [ "$MINOR" -ne 06 -a $retval -eq 2 ] ; then
             echolog ''
             [ -n "$JAVA_FULLVERSION" ] && echolog 'Found incomplete installation of $2' "$JAVA_FULLVERSION"
             echolog 'Sun Management Center requires complete installation of $2 or above.' "$JAVA_MINIMUM_REQUIRED"
             echolog 'Please follow the following steps before using this script.'
             echolog '1. Install supported JDK version.'
             echolog '2. set JAVA_HOME to directory where JDK is installed.'
             echolog '3. export JAVA_HOME'
             echolog ''
             exit 1
       fi
    fi
   
    unset JAVA OUT MINOR
    [ $retval -eq 0 ] && export SYMON_JAVAHOME JAVA_VERSION JAVA_FULLVERSION
    return $retval

}

# expects: SYMON_JAVAHOME
#          JAVA_HOME_FILE
#
updateJavaHomeFile() {    
    if [ "`/usr/ucb/whoami`" = "root" ] ; then # install/setup or server-agent start stop
        echo $SYMON_JAVAHOME > $JAVA_HOME_FILE
        /usr/bin/chmod 644 $JAVA_HOME_FILE
        ret=0
    else
        setup_required=0
        if [ -r "$JAVA_HOME_FILE" ] ; then
            temp_string=`/usr/bin/cat $JAVA_HOME_FILE`
            [ "$temp_string" != "$SYMON_JAVAHOME" ] && setup_required=1
        else
            setup_required=1
        fi

        if [ $setup_required -eq 1 ]; then
           [ -n "$JAVA_FULLVERSION" ] && echolog 'Found installation of $2' "$JAVA_FULLVERSION"
           echolog 'Sun Management Center requires complete installation of $2 or above.' "$JAVA_MINIMUM_REQUIRED"
           echolog 'Please follow the following steps before using this script.'
           echolog '1. Install supported JDK version.'
           echolog '2. set JAVA_HOME to directory where JDK is installed.'
           echolog '3. export JAVA_HOME'
           echolog '4. Run $2' "$BASEDIR/sbin/es-setup"
           echolog ''
           /usr/bin/rm -f $JAVA_HOME_FILE
           exit 1
        fi
    fi
    unset setup_required
}

