#!/bin/ksh
#
# Copyright (c) 2000 by Sun Microsystems, Inc.
# All rights reserved.
#
#ident        "@(#)stesocal.ksh 1.4     00/09/13 SMI"
#
# Create the appropriate /kernel/drv/socal.conf file from
# the /etc/opt/SUNWte/ste.cf file
#

USAGE="usage: stesocal.ksh [input_file output_file]"  # create socal.conf file from ste.cf

temp_file="/var/tmp/socal.conf$$"
trap 'rm -f $temp_file' EXIT

# Check for command-line arguments
if (($# != 0)) && (($# != 2))
then
  print -u2 $USAGE
  print -u2 "Error: there must be 0 or 2 command-line arguments."
  exit 1
else
  input_file=${1:-/etc/opt/SUNWte/ste.cf}
  output_file=${2:-/kernel/drv/socal.conf}
fi

# Files must be named ste.cf and socal.conf respectively
if [[ ${input_file##*/} != "ste.cf" ]]
then
  print -u2 $USAGE
  print -u2 "Error: input-file must be named 'ste.cf'."
  exit 1
fi

if [[ ${output_file##*/} != "socal.conf" ]]
then
  print -u2 $USAGE
  print -u2 "Error: output-file must be named 'socal.conf'."
  exit 1
fi


# Now do a variety of checks on the specified files...
#  Make sure they exist, correct permissions, etc.

if [[ ! -f $input_file || ! -r $input_file ]]
then
  print -u2 $USAGE
  print -u2 "Error: input-file ($input_file) does not exist."
  exit 1
fi


# Create a structure to hold the hba information
#  Note: We're using ksh88, which doesn't support 
#   structures.  Instead, we'll create individual
#   variables and reference them through arrays
#   as a workaround
typeset hba_parent
integer hba_slot
integer hba_port0_loopid
integer hba_port1_loopid


integer size=0
integer line_number=1

print "Parsing STE configuration file '$input_file'."

# Read in and parse ste.cf file
while read -r line
do
  if [[ $line == tm* ]]
  then
    parent=${line#*devices/}
    parent=${parent%/SUNW*}
    slot=${line#*al@}
    slot=${slot%,*}
    port=${line#*:}
    port=${port%%[[:blank:]]*}
    loopid=${line#*:}
    loopid=${loopid##[0-9]?([0-9]|[0-9][0-9])[[:blank:]]}
    loopid=${loopid##* }

    integer i=0
    found=FALSE
    while ((i < $size))
    do
      if [[ $parent == ${hba_parent[$i]} && $slot == ${hba_slot[$i]} ]]
      then
	# Here, add new (hopefully) port loop id to entry
	found=TRUE
	if ((port == 0))
	then
          if [[ -n ${hba_port0_loopid[$i]} ]]
          then
            print -u2 "\nError: invalid port number($port) on line $line_number in file '$input_file'.  Already set."
            exit 1
          fi

	  hba_port0_loopid[$i]=$loopid
	elif ((port == 1))
	then
          if [[ -n ${hba_port1_loopid[$i]} ]]
          then
            print -u2 "\nError: invalid port number($port) on line $line_number in file '$input_file'.  Already set."
            exit 1
          fi

	  hba_port1_loopid[$i]=$loopid
	else
  	  print -u2 "\nError: invalid port number($port) on line $line_number in file '$input_file'."
	  exit 1
        fi
      fi
      ((i+=1))
    done

    # If we didn't find an entry above, make a new entry in the array
    if [[ $found == FALSE ]]
    then
      hba_parent[$size]=$parent
      hba_slot[$size]=$slot
      if ((port == 0))
      then
	hba_port0_loopid[$size]=$loopid
      elif ((port == 1))
      then
        hba_port1_loopid[$size]=$loopid
      else
  	print -u2 "\nError: invalid port number($port) on line $line_number in file '$input_file'."
        exit 1
      fi
      ((size+=1))
    fi
  fi

  ((line_number+=1))
done < $input_file

print "Parsing done."
print


i=0
while (($i < $size))
do
  # Loop thru each hba entry, and output the socal line.

  print "name=\"SUNW,socal\" parent = \"${hba_parent[$i]}\""
  print -n "\treg=0x${hba_slot[$i]}, 0x0, 0x10000, "
  print -n "0x${hba_slot[$i]}, 0x10000, 0x10000, "
  print "0x${hba_slot[$i]}, 0x20000, 0x18"
  print -n "\t"
  if [[ -n ${hba_port0_loopid[$i]} ]]
  then
    print -n "port0-loop-id=${hba_port0_loopid[$i]}"
    if [[ -n ${hba_port1_loopid[$i]} ]]
    then
      print -n " "
    fi
  fi
  if [[ -n ${hba_port1_loopid[$i]} ]]
  then
    print -n "port1-loop-id=${hba_port1_loopid[$i]}"
  fi
  print ";"
  print

  ((i+=1))
done > $temp_file

# Compare out temp file to the current 'socal.conf' file (if it
#  exists).  If they're the same, exit, else write the new file
#  if desired by the user.

if [[ -f $output_file && -r $output_file ]] 
then
  file1=$(< $temp_file)
  file2=$(< $output_file)

  if [[ "$file1" == "$file2" ]]
  then
    print "The current $output_file is up to date."
    print "No action needs to be taken."
    exit 0
  fi
fi

# Now that we're done parsing the ste.cf file, output
#  the appropriate socal.conf file.

print "SOCAL configuration file needs to be changed."
read ans?"Write the SOCAL configuration file '$output_file'? (y/n)"
if [[ $ans != Y* && $ans != y* ]]
then
  exit 1
fi

cp $temp_file $output_file

print "Wrote SOCAL configuration file '$output_file'."
print "\nMust reboot for changes to take effect!"
