/*
 * Copyright (c) 1999, 2001 by Sun Microsystems, Inc.
 * All rights reserved.
 *
 * Open Card Framework (OCF) Core Services header file
 */

#ifndef	_OCF_CORE_H
#define	_OCF_CORE_H

#pragma ident	"@(#)ocf_core.h	1.43	01/08/20 SMI"

#ifdef	__cplusplus
extern "C" {
#endif

/*
 * OCF_CurrentVersion can be used by clients in their call to
 *	OCF_RegisterClient to specify this version of the OCF
 *	interfaces.
 */
#define	OCF_CurrentVersion	"1.0"

/*
 * General typedefs and constants used by multiple functions.
 */
typedef uint64_t OCF_ClientHandle_t;
typedef uint64_t OCF_CardHandle_t;
typedef uint8_t * OCF_CardServiceClass_t;
typedef uint8_t * OCF_CardServiceMethod_t;
typedef void * OCF_CardServiceParams_t;
typedef void * OCF_CardServiceReturn_t;
typedef uint32_t OCF_Boolean_t;

typedef struct OCF_Any_t {
	int32_t		length;
	void		*data;
} OCF_Any_t;

typedef struct OCF_Timeout_t {
	uint32_t		flags;		/* wait flags */
	uint32_t	timeout;	/* timeout in mS to wait */
} OCF_Timeout_t;

/*
 * OCF_Timeout_t->uint32_t values.
 */
#define	OCF_Timeout_NoBlock	0x00000001	/* don't block */


/* general status */
#define	OCF_Success		0	/* operation OK */
#define	OCF_NoCard		1	/* no card present */
#define	OCF_CardError		2	/* error on card */
#define	OCF_ReaderError		3	/* error on reader */
#define	OCF_ParameterError	4	/* parameter error */
#define	OCF_OperationTimeout	5	/* operation timed out */
#define	OCF_ResourceUnavailable	6	/* resource unavailable */
#define	OCF_CardLocked		7	/* card is locked */
#define	OCF_DuplicateClient	8	/* duplicate client */
#define	OCF_DuplicateHandler	9	/* duplicate handler */
#define	OCF_DuplicateEvent	10	/* duplicate event */
#define	OCF_VerifyError		11	/* object did not verify */
#define	OCF_PINAlreadySet	12	/* PIN already set */
#define	OCF_NoPermission	13	/* no permisison for operation */
#define	OCF_UnknownResource	14	/* requested resource unknown */
#define	OCF_EncodingError	15	/* encode/decode error */
#define	OCF_ServerError		16	/* error in server communication */
#define	OCF_VersionMismatch	17	/* client/server version mismatch */
#define	OCF_DoubleLock		18	/* already have a lock */
#define	OCF_Exception		20	/* exception from server */
#define	OCF_NoSuchProperty	21	/* no such property from server */
/* handle errors */
#define	OCF_InvalidClientHandle	30	/* invalid client handle */
#define	OCF_InvalidCardHandle	31	/* invalid card handle */
#define	OCF_InvalidEventHandle	32	/* invalid event handle */
#define	OCF_NoReadersConfigured	33	/* no readers configured */
#define	OCF_NoCardsConfigured	34	/* no cards configured */

/* OCF_Authenticate */
#define	OCF_AuthenticationError	70	/* authentication error */
#define	OCF_AuthenticationFailed 71	/* authentication failed */

/* OCF_Authenticate internal status codes */
#define	OCF_AuthOp		80	/* auth operation */
#define	OCF_InvalidPin		81	/* invalid PIN */
#define	OCF_InvalidPassword	82	/* invalid Password */
#define	OCF_UnsupportedAuthMech	83	/* unsupported auth mechanism */

/* unspecified error */
#define	OCF_UnknownError	90	/* unknown error */

/*
 * OCF_RegisterClient - Client registration function.
 * OCF_DeregisterClient - Client deregistration function.
 *
 */

/* Prototypes */
uint32_t OCF_RegisterClient(int8_t *, int8_t *,
						OCF_ClientHandle_t *);
uint32_t OCF_DeregisterClient(OCF_ClientHandle_t);

/*
 * OCF_RegisterEventHandler - Client event handler registration function.
 * OCF_DeregisterEventHandler - Client event handler deregistration function.
 *
 */
typedef uint32_t OCF_Event_t;

typedef struct OCF_EventData_t {
	OCF_Event_t		event;		/* event type */
	OCF_ClientHandle_t	clienthandle;	/* client handle */
	OCF_CardHandle_t	cardhandle;	/* card handle */
	void*	eventprivate;	/* event private data */
	void*	clientdata;	/* client private data */
} OCF_EventData_t;

/*
 * Prototype of Client Event Handler
 */
typedef uint32_t (OCF_EventHandler_t)(OCF_Event_t, OCF_EventData_t);

/* Prototypes */
uint32_t OCF_RegisterEventHandler(OCF_ClientHandle_t, void*,
							OCF_EventHandler_t);
uint32_t OCF_DeregisterEventHandler(OCF_ClientHandle_t);

/*
 * OCF_EventData_t->void* event-specific structures
 *
 * OCF_Event_Exception
 */
typedef struct OCF_Event_Exception_t {
	/* XXX */
	uint32_t		flags;
} OCF_Event_Exception_t;

/*
 * OCF_RegisterForEvent - Register for an OCF event.
 * OCF_DeregisterForEvent - Deregister for an OCF event.
 *
 */
typedef void * OCF_EventHandle_t;
typedef uint32_t OCF_EventRegistration_t;	/* same values as OCF_Event_t */
typedef void * OCF_EventRegistrationData_t;	/* event specific */

/* Prototypes */
uint32_t OCF_RegisterForEvent(OCF_ClientHandle_t,
					OCF_Event_t,
					OCF_EventRegistrationData_t,
					void*,
					OCF_EventHandle_t *);
uint32_t OCF_DeregisterForEvent(OCF_EventHandle_t);

/*
 * OCF_CardSpec_t - A structure that describes the specifications
 *	of a reader, card and card service.
 */

typedef struct OCF_CardSpec_t {
	uint32_t			flags;
	OCF_CardHandle_t		cardhandle;	/* card handle */
	int8_t *		readername;	/* card reader */
	int8_t *			cardname;	/* card */
	int8_t *			aid;		/* AID number */
	OCF_Timeout_t			timeout;	/* timeout parameters */
} OCF_CardSpec_t;

/*
 * OCF_CardSpec_t->uint32_t flags
 */
#define	OCF_CardSpec_UseHandle	0x00000001	/* use card handle */

/*
 * OCF_WaitForCardInserted - Waits for the specified card to be
 *	inserted in the specified reader.
 * OCF_WaitForCardRemoved - Waits for the specified card to be
 *	removed from the reader.
 * OCF_CardPresent - Check if the specified card is present in the
 *	specified reader.
 */

/* Prototypes */
uint32_t OCF_WaitForCardInserted(OCF_ClientHandle_t,
					OCF_CardSpec_t,
					OCF_CardHandle_t *);
uint32_t OCF_WaitForCardRemoved(OCF_ClientHandle_t, OCF_CardHandle_t,
					OCF_Timeout_t);
uint32_t OCF_CardPresent(OCF_ClientHandle_t, OCF_CardSpec_t);

/*
 * OCF_CardSpec_t int8_t * and int8_t * values.
 *
 * int8_t * - Name of reader to wait on.
 *	OCF_WaitForCardSpec_AnyReader - Wait for card on any reader.
 *	OCF_WaitForCardSpec_DefaultReader - Wait for card on default reader.
 *
 * int8_t * - Name of card to wait on.
 *	OCF_WaitForCardSpec_AnyCard - Wait for any card.
 *	OCF_WaitForCardSpec_DefaultCard - Wait for default card.
 *
 * int8_t * - Name of Card Service to wait on.
 *	OCF_WaitForCardSpec_AnyService - Wait for any Card Service.
 *	OCF_WaitForCardSpec_DefaultService - Wait for default Card Service.
 *
 * The default reader and default card are ones that are specified in OCF
 *	for this client.
 */
#define	OCF_WaitForCardSpec_AnyReader		"WaitForCardSpec_AnyReader"
#define	OCF_WaitForCardSpec_DefaultReader	"WaitForCardSpec_DefaultReader"
#define	OCF_WaitForCardSpec_AnyCard		"WaitForCardSpec_AnyCard"
#define	OCF_WaitForCardSpec_DefaultCard		"WaitForCardSpec_DefaultCard"
#define	OCF_WaitForCardSpec_DefaultAID		"WaitForCardSpec_DefaultAID"
#define	OCF_WaitForCardSpec_AnyAID		"WaitForCardSpec_AnyAID"

/*
 * OCF_LockCard - Locks card with client.
 * OCF_UnlockCard - Unlocks card with client.
 *
 */

/* Prototypes */
uint32_t OCF_LockCard(OCF_ClientHandle_t, OCF_CardHandle_t, OCF_Timeout_t);
uint32_t OCF_UnlockCard(OCF_ClientHandle_t, OCF_CardHandle_t);

/*
 * Enumerated values for OCF_Event_t event types.
 */
#define	OCF_Event_CardInserted	0x00000001	/* card inserted */
#define	OCF_Event_CardRemoved	0x00000002	/* card removed */
#define	OCF_Event_CardReset	0x00000004	/* card was reset */
#define	OCF_Event_CardError	0x00000008	/* internal card error */
#define	OCF_Event_Exception	0x00000010	/* OCF exception */

/*
 * OCF_EventRegistrationData_t structures for each OCF_EventRegistration_t
 *	event type.
 *
 * OCF_Event_CardInserted
 */
typedef struct OCF_Event_CardInserted_RegistrationData_t {
	OCF_CardSpec_t	cardspec;	/* card spec to look for */
} OCF_Event_CardInserted_RegistrationData_t;

/*
 * OCF_Event_CardRemoved
 */
typedef struct OCF_Event_CardRemoved_RegistrationData_t {
	OCF_CardSpec_t	cardspec;	/* card spec to look for */
} OCF_Event_CardRemoved_RegistrationData_t;

/*
 * OCF_Event_CardReset
 */
typedef struct OCF_Event_CardReset_RegistrationData_t {
	OCF_CardHandle_t	cardhandle;	/* card handle */
} OCF_Event_CardReset_RegistrationData_t;

/*
 * OCF_Event_CardError
 */
typedef struct OCF_Event_CardError_RegistrationData_t {
	OCF_CardHandle_t	cardhandle;	/* card handle */
} OCF_Event_CardError_RegistrationData_t;

/*
 * OCF Property Interface - provides a basic get/set property
 *	interface.
 */

/* Prototypes for properties */
typedef struct OCF_OCF_PropertyList_t *OCF_PropertyList_t;
int8_t * OCF_ListProperty_First(OCF_PropertyList_t);
int8_t * OCF_ListProperty_Next(OCF_PropertyList_t);
uint32_t OCF_DestroyPropertiesList(OCF_PropertyList_t);

/* set/get OCF Server properties */
uint32_t OCF_SetProperty(int8_t *, int8_t *);
uint32_t OCF_GetProperty(int8_t *, int8_t * *);
uint32_t OCF_DeleteProperty(int8_t *);
uint32_t OCF_ListProperties(OCF_PropertyList_t *);

/* set/get Client Handle properties */
uint32_t OCF_SetClientProperty(OCF_ClientHandle_t, int8_t *,
					int8_t *);
uint32_t OCF_GetClientProperty(OCF_ClientHandle_t, int8_t *,
					int8_t * *);
uint32_t OCF_DeleteClientProperty(OCF_ClientHandle_t, int8_t *);
uint32_t OCF_ListClientProperties(OCF_ClientHandle_t, OCF_PropertyList_t *);

/* set/get Card properties */
uint32_t OCF_SetCardProperty(OCF_ClientHandle_t, OCF_CardHandle_t,
	int8_t *, int8_t *);
uint32_t OCF_GetCardProperty(OCF_ClientHandle_t, OCF_CardHandle_t,
	int8_t *, int8_t * *);
uint32_t OCF_DeleteCardProperty(OCF_ClientHandle_t, OCF_CardHandle_t,
	int8_t *);
uint32_t OCF_ListCardProperties(OCF_ClientHandle_t, OCF_CardHandle_t,
	OCF_PropertyList_t *);

/* set/get Card Handle properties */
uint32_t OCF_SetCardInfoProperty(OCF_ClientHandle_t, OCF_CardHandle_t,
					int8_t *,
					int8_t *);
uint32_t OCF_GetCardInfoProperty(OCF_ClientHandle_t, OCF_CardHandle_t,
					int8_t *,
					int8_t * *);

/*
 * Card Handle properties.
 */
#define	OCF_ArtificialEvent		"card.OCF_ArtificialEvent"
#define	OCF_ValidCard			"card.OCF_ValidCard"
#define	OCF_WaitForCard_AutoLock_Card	"card.OCF_WaitForCard_AutoLock_Card"
#define	OCF_WaitForCard_UnlockedOnly	"card.OCF_WaitForCard_UnlockedOnly"

/*
 * Lists returned from OCF.
 *
 * OCF_CardServicesList_t - a list of card services
 * OCF_CardsList_t - a list of cards
 * OCF_ReadersList_t - a list of readers
 */
typedef struct OCF_OCF_CardServicesList_t *OCF_CardServicesList_t;
typedef struct OCF_OCF_CardsList_t *OCF_CardsList_t;
typedef struct OCF_OCF_ReadersList_t *OCF_ReadersList_t;

/* Prototypes */
uint32_t OCF_ListCardServices(int8_t *, OCF_CardServicesList_t *);

/* Present list */
uint32_t OCF_ListCardsPresent(OCF_CardsList_t *);
uint32_t OCF_ListReadersPresent(OCF_ReadersList_t *);

/* Configured list */
uint32_t OCF_ListCardsConfigured(OCF_CardsList_t *);
uint32_t OCF_ListReadersConfigured(OCF_ReadersList_t *);

/* Prototype to iterate the list */
int8_t * OCF_ListCardServices_First(OCF_CardServicesList_t);
int8_t * OCF_ListCardServices_Next(OCF_CardServicesList_t);
int8_t * OCF_ListCards_First(OCF_CardsList_t);
int8_t * OCF_ListCards_Next(OCF_CardsList_t);
int8_t * OCF_ListReaders_First(OCF_ReadersList_t);
int8_t * OCF_ListReaders_Next(OCF_ReadersList_t);

/* Routines to destroy lists */
uint32_t OCF_DestroyCardServicesList(OCF_CardServicesList_t);
uint32_t OCF_DestroyCardsList(OCF_CardsList_t);
uint32_t OCF_DestroyReadersList(OCF_ReadersList_t);

/* Routine to get time_t since a card has been validated */
uint32_t OCF_TimeSinceValidated(OCF_ClientHandle_t, OCF_CardHandle_t,
    uint64_t *);

/* Routine to check if the card is a valid card for the application */
uint32_t OCF_CardValid(OCF_ClientHandle_t, OCF_CardHandle_t);

#ifdef	__cplusplus
}
#endif

#endif	/* _OCF_CORE_H */
