#   File:    cfgserver-engine-d.prc
#   Author:  Phil Matthews, Halcyon Inc.
#   Version: 1.25 01/05/01 12:38:18
#
#   Copyright (c) 1993-1998 Halcyon Inc.
#
#   PrimeAlert Config/Security Module Procedures (engine-specific)
#

##############################################################################
# Proc:		getEngineCfg
#
# Abstract:	Method to retrieve the USM user config and timestamp from the
#		specified engine ID.
#
# Arguments:	engineID - identifying agent in agent table
#
# Returns:	Nothing.
##############################################################################
proc getEngineCfg { engineid } {

    ddl print trace "getEngineCfg: engine ID \[$engineid\]\n"

#   Get the agent table information corresponding to this engine ID.
#   ----------------------------------------------------------------
    set agentloc [ getRowValue netloc $engineid ]
    lextract $agentloc 0 host 1 port

#   Remember that we have an outstanding engine cfg 'get' to this agent,
#   out there.
#   --------------------------------------------------------------------
    set enginecfgtid [ idnext tid ]
    define enginecfgwait $enginecfgtid $engineid

#   Register a re-send of this command; increment the number of retries and
#   turn off the ack'ed flag.
#   -----------------------------------------------------------------------
    set agentretries [ getRowValue getcfgidretries $engineid ]

    set interval [ retries2Interval $agentretries ]

    setRowValue getcfgidretries $engineid [ incr agentretries ]
    setRowValue getcfgidack $engineid 0

    ddl print trace "register getEngineCfg in \[$interval\] secs; retries \[$agentretries\]\n"
    set resendid [ registerOneShot $interval [ list getEngineCfg $engineid ] ]
    ddl print trace "enginecfgresend:\[$engineid\] = \[$resendid\]\n"
    define enginecfgresend $engineid $resendid

#   Send the command.
#   Use public noauth for snmp since superuser key changes may not
#   have been successfully propagated to all components, in
#   which case authenticated operations might fail
#   --------------------------------------------------------------
    set vb [ makeEngineCfgVB ]
    set cb [ list [ toe_self ] "handleEngineCfgResponse %result" $enginecfgtid ]
    set cmd "get $host $port -1 $vb -securityLevel noauth -community [ lookup value generaluser ] -retries 0 -timeout 3600"

    setParameters command $cmd ecm $cb

    set target [ lookup value snmpService ]
    set cmd [ lookup value snmpCommand ]
    set result [ evalECM [ list $target $cmd ] ]

    clearParameters command ecm

#   Send the immediate results to our callback.
#   -------------------------------------------
    handleEngineCfgResponse $result
}

##############################################################################
# Proc:		retries2Interval
#
# Abstract:	Method to compute an interval based on the specified number
#		of retries.  This interval is how long to wait before
#		re-sending a enginecfg 'get' request.
#
# Arguments:	retries		The number of retries already attempted.
#
# Returns:	The computed interval.
##############################################################################
proc retries2Interval { retries } {
    return [ expr 150 * ( $retries + 1 ) ]
}

##############################################################################
# Proc:		handleEngineCfgResponse
#
# Abstract:	Method to handle responses to a enginecfg 'get' request.
#		Triggers an agent user update, upon receipt of a 'data'
#		response.
#
# Arguments:	arc		The asynch return code:
#				* type - wait, error or data
#				* tid - the request's transaction ID
#				* data - the latest TID, or an error msg,
#				         or actual return data
#
# Returns:	Nothing.
##############################################################################
proc handleEngineCfgResponse { arc } {

    ddl print trace "handleEngineCfgResponse: arc \[$arc\]\n"

    lextract $arc 0 type 1 tid 2 data

#   Figure out what this is in response to.
#   ---------------------------------------
    if { [ catch { ilookup enginecfgwait $tid } result ] } {
	ddl print info "ignoring unexpected enginecfg response - tid \[$tid\]\n"
	return
    }
    undefine enginecfgwait $tid

    switch $type \
	{wait} {

#	    'data' is the new tid to wait for.
#	    ----------------------------------
	    define enginecfgwait $data $result
	} \
	{error} {

	    ddl print warning "enginecfg response ERROR: \[$result\]\n"
	} \
	{data} {

#	    Validate the data.  (Should have a timestamp, cfg ID, and server context)
#	    ---------------------------------------------------------
	    if { [ llength $data ] != 3 } {
		ddl print warning "Invalid engine cfg info from engine \[$engineid\]: \[$data\]\n"
		return
	    }

#	    Now we know this agent's current config.  Update its user table
#	    accordingly.  Reset the enginecfg get resend timer, and the # of
#	    retries, now that we have "heard" from the agent.
#	    ---------------------------------------------------------------
	    set engineid $result
	    ddl print trace "...data: cancelling enginecfgresend:\[$engineid\]\n"
	    if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
		cancelOneShot $resendid
	    }

	    setRowValue getcfgidretries $engineid 0
	    setRowValue getcfgidack $engineid 1

	    set interval [ retries2Interval 0 ]
	    ddl print trace "register getEngineCfg in \[$interval\] secs; retries \[0\]\n"
	    set resendid [ registerOneShot $interval \
					[ list getEngineCfg $engineid ] ]
	    ddl print trace "enginecfgresend:\[$engineid\] = \[$resendid\]\n"
	    define enginecfgresend $engineid $resendid

	    updateAgent $engineid $data
	}
}


##############################################################################
# Proc:		handleTblUpdateResponse
#
# Abstract:	Method to handle responses to a User Table update request.
#
# Arguments:	arc		The asynch return code:
#				* type - wait, error or data
#				* tid - the request's transaction ID
#				* data - the latest TID, or an error msg,
#				         or actual return data
#
# Returns:	Nothing.
##############################################################################
proc handleTblUpdateResponse { arc } {

    ddl print trace "handleTblUpdateResponse: arc \[$arc\]\n"

    lextract $arc 0 type 1 tid 2 data

#   Figure out what this is in response to.
#   ---------------------------------------
    if { [ catch { ilookup tblupdatewait $tid } result ] } {
	ddl print warning \
		"ignoring unexpected table update response - tid \[$tid\]\n"
	return
    }
    undefine tblupdatewait $tid

    switch $type \
	{wait} {

#	    'data' is the new tid to wait for.
#	    ----------------------------------
	    define tblupdatewait $data $result
	} \
	{error} {

	    undefine tblupdatewait $tid

#           Check whether the set failed due to 'inconsistentValue'; if so,
#           the agent has wound up in a bad state, and retrying won't help.
#	    ----------------------------------------------------------------
            if { [ regexp "inconsistentValue" $data ] } {
	        set engineid $result
                set addr [ getRowValue netloc $engineid ]
                set msg "Agent at $addr \[id $engineid\] is in an inconsistent state, and needs to be reseeded.\n"
                # Don't enunciate to syslog -- yet.
                #ddl print syslog $msg
                ddl print error $msg
	        if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
		    cancelOneShot $resendid
	        }
            } else {
	        ddl print warning "tblupdate response ERROR: \[$result $data\]\n"
            }
	} \
	{data} {

#	    The agent is now officially updated with new config information.
#	    ----------------------------------------------------------------
	    set engineid $result

#	    Reset the enginecfg information:  ACK = 1; RETRIES = 0; cancel
#	    re-send timer.
#	    -------------------------------------------------------------
	    ddl print trace "...data: cancelling enginecfgresend:\[$engineid\]\n"
	    if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
		cancelOneShot $resendid
	    }
	    setRowValue getcfgidretries $engineid 0
	    setRowValue getcfgidack $engineid 1

#	    Reset the tbl update information:  ACK = 1; TIME = NOW.
#	    -------------------------------------------------------
	    setRowValue setcfgidtime $engineid [ clock seconds ]
	    setRowValue setcfgidack $engineid 1

#	    We used to do a setRowValue authkey at this point,
#	    which was failing and preventing the rest of this function
#	    from running. We do not use the authkey column and it is
#	    no longer accessible.
#	    ----------------------------------------------------------

#	    Save the current superuser key for this engine so we can
#	    update the cfgserver's usm user table and the snmp stack
#	    thereby enabling the cfgserver to talk to this engine in
#	    future with the right (localized) superuser key.
#	    Of course this doesn't have to be done for the cfgserver
#	    itself since it has already updated itself via updateAgent.
#	    ----------------------------------------------------------

            if { [ isCfgServer $engineid ] == 0 } {
                undefine authupdateretries $engineid-new
	        setAuthKey $engineid new
            } else {
                ddl print debug "handleTblUpdateResponse: skipping setAuthKey for cfgserver\n"
            }


#	    If the current config has changed in the meantime, start the
#	    whole process over.
#	    ---------------------------------------------------------------
	    set cfgid [ getRowValue setcfgid $engineid ]
	    if { $cfgid != [ evaluateMethod user "" getConfigID "" ] } {
		getEngineCfg $engineid
	    }
	}
}

proc toeObj2Sym { obj } {

    set symbolicname [ toe_name -full $obj ]
    regsub ^\. $symbolicname / symbolicname

    return $symbolicname
}

##############################################################################
# Proc:		makeEngineCfgVB
#
# Abstract:	Method to create the USM user enginecfg 'get' varbind.
#
# Arguments:	None
#
# Returns:	The one-component varbind.
##############################################################################
proc makeEngineCfgVB {} {

    set usmUser .iso.org.dod.internet.snmpV2.snmpModules.snmpUsmMIB.usmMIBObjects.usmUser
    set baseInfoSys .iso*base.info.system

    return "{{oid//[
	lookup internal $usmUser.usmUserSpinLock.fulloid
    ]#0} {oid//[
	lookup internal $usmUser.usmUserTimeStamp.fulloid
    ]#0} {oid//[
        lookup internal $baseInfoSys.trapaddr.fulloid
    ]#0}}"
}

##############################################################################
# Proc:		makeTblResetVB
#
# Abstract:	Method to create the USM user table reset 'set' varbind.
#
# Arguments:	host		The destination host.
#		port		The destination port.
#		engineid	The engine id.
#		cfgid		The config id.
#
# Returns:	The multi-component varbind.
##############################################################################
proc makeTblResetVB { host port engineid cfgid } {

    ddl print trace "makeTblResetVB: host \[$host\]; port \[$port\]; engineid \[$engineid\]; cfgid \[$cfgid\]\n"

    # Make sure that the right private key for this engine exists
    undefine authupdateretries $engineid-new
    setAuthKey $engineid new

    set usmusertbl [ lookup value usmUserTable ]
    set usmuserobj [ locate $usmusertbl ]

    set spinlockobj [ toe_send $usmuserobj locate usmUserSpinLock ]
    set spinlockname [ toeObj2Sym $spinlockobj ]
    set timestampobj [ toe_send $usmuserobj locate usmUserTimeStamp ]
    set timestampname [ toeObj2Sym $timestampobj ]

    set spinlockobj [ toe_send $usmuserobj locate usmUserSpinLock ]
    set spinlockname [ toeObj2Sym $spinlockobj ]

    set vb ""

    append vb "{sym/${spinlockname}?testandmod#0 \
				{OCTET STRING} {$cfgid 0}} "
    append vb "{sym/${timestampname}#0 \
				{INTEGER} 0} "
    return [ list $vb ]
}

##############################################################################
# Proc:		makeTblUpdateVB
#
# Abstract:	Method to create the USM user table update 'set' varbind.
#
# Arguments:	host		The destination host.
#		port		The destination port.
#		engineid	The destination engine ID.
#		cfgid		The destination current cfg ID.
#		currcfgid	The current cfg ID.
#		timestamp	The current cfg ID's timestamp.
#
# Returns:	The multi-component varbind.
##############################################################################
proc makeTblUpdateVB { host port engineid cfgid currcfgid timestamp } {

    ddl print trace "makeTblUpdateVB: host \[$host\]; port \[$port\]; engineid \[$engineid\]; cfgid \[$cfgid\]; currcfgid \[$currcfgid\]; timestamp \[$timestamp\]\n"

#   Get security and authentication info.
#   -------------------------------------
    set superuser [ evaluateMethod user "" getSuperUser "" ]
    set superkey [ evaluateMethod user "" getSuperuserAuthKey "" ]
    set sulocalize [ evaluateMethod user "" "isUserKeyLocalized $superuser" "" ]
    set superusmrow [ evaluateMethod user "" \
				"makeUsmUserRowName $superuser $engineid" "" ]

    set usmusertbl [ lookup value usmUserTable ]
    set usmuserobj [ locate $usmusertbl ]

    set authProtocol [ getRowValue $usmusertbl.usmUserAuthProtocol $superusmrow ]
    set privProtocol [ getRowValue $usmusertbl.usmUserPrivProtocol $superusmrow ]
    if { $privProtocol != "0.0" && $privProtocol != "1.3.6.1.6.3.10.1.2.1" } {
	set hasPrivProtocol true
    } else {
	set hasPrivProtocol false
    }
    set maxlen [ lookup value maxAuthKeyLen ]

    set secnameoid [ toe_send $usmuserobj fullOidOf usmUserSecurityName ]

#   OID-type localized superuser index.
#   -----------------------------------
    set superidx [ join [ list [
	indexStrToOid [ lindex [ split $superusmrow "," ] 0 ] +1x:
    ] [
	indexStrToOid $superuser +
    ] ] "." ]

    regsub ^\. $usmusertbl / usmusertbl
    set vbpfx "sym/${usmusertbl}"
    set vb ""


#   Get the diffs between this agent's config ID and the current config ID.
#   diffs:	A list of user changes of the form
#		- {{username {prevRowData} {newRowData}} ...}
#		  prevRowData BLANK for a new user
#		  newRowData BLANK for a gone user
#		  both non-BLANK for a changed user
#   -----------------------------------------------------------------------
    set diffs [ evaluateMethod user "" "getCfgDiff $cfgid $currcfgid" "" ]
    ddl print debug "makeTblUpdateVB: diffs are $diffs\n"
    set timestamp [ evaluateMethod user "" "getCfgIDTimeStamp $currcfgid" "" ]

#   Go through the difference list.
#   -------------------------------

    set suflag 0
    set newidx 0
    set goneidx 1
    set changedidx 2

    set dosend 0

    foreach diff $diffs {

	set prevkey ""
	set prevgrps ""
	set newkey ""
	set newgrps ""

	set engidkey ""

	lextract $diff 0 user 1 prevdata 2 newdata
	catch { lextract $prevdata 0 prevkey 1 prevgrps }
	catch { lextract $newdata 0 newkey 1 newgrps }

	if { $prevgrps == "" } { set prevgrps [ list {} ] }
	if { $newgrps == "" } { set newgrps [ list {} ] }

#	If there is no new or old auth key, this user has never logged in
#	(and is only in our diff list because its group list has changed).
#	Do not send an update for this user.
#	------------------------------------------------------------------
	if { $newkey == "" && $prevkey == "" } {
	    ddl print trace "...$user never logged in...skipping...\n"
	    continue
	}
	set dosend 1

#	If the row is gone, we won't find this (but we won't need it)!
#	--------------------------------------------------------------
	if { [ catch { evaluateMethod user "" \
			"getRowValue userTable.userEntry.localize $user" "" } \
				localize ] } {
	    ddl print info "\[$user\]:localize not found: defaulting to YES\n"
	    set localize 1
	}

	if { $prevkey == "" } {

#	    CREATE user.  Use the superuser auth key as a "prev key".
#	    ---------------------------------------------------------
	    ddl print trace "...$user to be created...\n"

	    if { $localize == 1 } {
		ddl print trace "...to be localized...\n"
		set newkey [ snmp key $authProtocol localize $newkey $engineid ]
		set engidkey [ strtourl $engineid "1x:" ]
	    }
	    if { $sulocalize == 1 } {
		set lclsuperkey [ snmp key $authProtocol localize \
							$superkey $engineid ]
	    } else {
		set lclsuperkey $superkey
	    }
	    set keychange [ snmp key $authProtocol createchange \
						$lclsuperkey $newkey $maxlen ]


	    append vb "{${vbpfx}.usmUserCloneFrom#${engidkey},${user} \
				{OBJECT IDENTIFIER} ${secnameoid}.${superidx}} "
	    append vb "{${vbpfx}.usmUserAuthKeyChange#${engidkey},${user} \
				{OCTET STRING} ${keychange} 1x:} "


	    if { $hasPrivProtocol } {
		append vb "{${vbpfx}.usmUserPrivKeyChange#${engidkey},${user} \
				    {OCTET STRING} ${keychange} 1x:} "
	    }
	    append vb "{${vbpfx}?setusergrp#${engidkey},${user} \
				{OCTET STRING} ${newgrps}} "
	    append vb "{${vbpfx}.usmUserStatus#${engidkey},${user} \
				INTEGER 4} "

	} elseif { $newkey != "" } {

	    if { $newkey != $prevkey } {

#		MODIFY user.
#		------------
#
		ddl print trace "... $user to be modified...\n"

		if { $localize == 1 } {
		    set newkey [
			snmp key $authProtocol localize $newkey $engineid ]
		    set prevkey [
			snmp key $authProtocol localize $prevkey $engineid ]
		    set engidkey [ strtourl $engineid "1x:" ]
		}
		set keychange \
		[ snmp key $authProtocol createchange $prevkey $newkey $maxlen ]

		ddl print debug "makeTblUpdateVB: prevkey is $prevkey, newkey is $newkey, keychange is $keychange\n"
		set applychange \
               [ snmp key $authProtocol applychange $prevkey $keychange $maxlen ]
		ddl print debug "makeTblUpdateVB: applychange to get newkey yields $applychange\n"

#		If user is superuser, then save the previous key
#		so the new key can be written to the engine.
#		Call setAuthKey to make sure the
#	        prevkey is in the cfgserver's snmp stack, since this is
#		the last one known to the engine we're
#	        about to talk to.
#		-------------------------------------------
		if { $user == $superuser } {
		    define oldauthkey $engineid $prevkey 
		    define newauthkey $engineid $newkey 
		    define authkeychange $engineid $keychange
		    set suflag 1
                    undefine authupdateretries $engineid-old
		    setAuthKey $engineid old
                }

		append vb "{${vbpfx}.usmUserAuthKeyChange#${engidkey},${user} \
				{OCTET STRING} ${keychange} 1x:} "

		if { $hasPrivProtocol } {
		    append vb "{${vbpfx}.usmUserPrivKeyChange#${engidkey},${user} \
				    {OCTET STRING} ${keychange} 1x:} "
		}
		append vb "{${vbpfx}.usmUserStatus#${engidkey},${user} \
				INTEGER 1} "
	    }
	    if { $newgrps != $prevgrps } {
		append vb "{${vbpfx}?setusergrp#${engidkey},${user} \
				{OCTET STRING} ${newgrps}} "
	    }

	} else {

#	    DELETE user.
#	    ------------
	    ddl print trace "... $user to be deleted...\n"

	    if { $localize == 1 } {
                set engidkey [ strtourl $engineid "1x:" ]
	    }
	    append vb "{${vbpfx}.usmUserStatus#${engidkey},${user} INTEGER 6} "
	    append vb "{${vbpfx}?rmusergrp#${engidkey},${user} \
				{OCTET STRING} {}} "

	}
    }

    if { $dosend == 1 } {

	# Make sure cfgserver has the right superuser key for this
	# engine in the usm user table
	if { $suflag == 0 } {
            undefine authupdateretries $engineid-new
            setAuthKey $engineid new
        }


	set spinlockobj [ toe_send $usmuserobj locate usmUserSpinLock ]
	set spinlockname [ toeObj2Sym $spinlockobj ]
	set timestampobj [ toe_send $usmuserobj locate usmUserTimeStamp ]
	set timestampname [ toeObj2Sym $timestampobj ]

	set spinlockobj [ toe_send $usmuserobj locate usmUserSpinLock ]
	set spinlockname [ toeObj2Sym $spinlockobj ]

	append vb "{sym/${spinlockname}?testandmod#0 \
					{OCTET STRING} {$cfgid $currcfgid}} "
	append vb "{sym/${timestampname}#0 \
					{INTEGER} $timestamp} "
	return [ list $vb ]
    } else {
	ddl print trace "makeTblUpdateVB: nothing to send.\n"
	return ""
    }
}

##############################################################################
# Proc:		updateAgent
#
# Abstract:	Method to update the specified agent from the specified
#		configuration to the current config.  This method is invoked
#		upon successful	receipt of the agent's config ID/timestamp,
#		via a get on the usmUserSpinLock/usmUserTimeStamp nodes.
#
# Arguments:	engineid	Agent's engine ID.
#		cfginfo		Agent's last known config ID/timestamp.
#
# Returns:	Nothing.
##############################################################################
proc updateAgent { engineid cfginfo } {

    ddl print trace "updateAgent: engineid \[$engineid\]; cfginfo \[$cfginfo\]\n"

    lextract $cfginfo 0 cfgid 1 timestamp 2 context

#   Check to see if the agent is in a different server context.
#   If so, do not configure!
    if { $context != [ toe_send [ locate .services.io.snmp ] getTrapServerAddress ] } {
        ddl print warning "Engine $engineid is in a different server context ($context). Unregistering.\n"
        unregisterAgent $engineid
        return
    } 

    set currcfgid [ evaluateMethod user "" getConfigID "" ]
    set currtimestamp [ evaluateMethod user "" \
					"getCfgIDTimeStamp $currcfgid" "" ]

    set reset 0

#   Handle special cases first.
#   ---------------------------
    if { $cfgid == 0 } {
	ddl print info "Engine \[$engineid\] cfgID zero - probably NEW!\n"
    } elseif { [ catch \
	{ evaluateMethod user "" "getCfgIDTimeStamp $cfgid" "" } \
							ourtimestamp ] } {
	ddl print info "Engine \[$engineid\] has unknown cfgID \[$cfgid\] - resetting!\n"
	set reset 1
    } elseif { $cfgid > $currcfgid } {
	ddl print info "Engine \[$engineid\] cfgID \[$cfgid\] > current cfgid \[$currcfgid\] - resetting!\n"
	set reset 1
    } elseif { $cfgid > 0 && $ourtimestamp != $timestamp } {
	ddl print info "Engine \[$engineid\] timestamp \[$timestamp\] != our timestamp \[$ourtimestamp\] for cfgID \[$cfgid\] - resetting!\n"
	set reset 1
    } elseif { $cfgid == $currcfgid } {
	ddl print debug "Engine \[$engineid\] cfgID \[$cfgid\] timestamp \[$timestamp\] is current - nothing to do!\n"

	setRowValue setcfgidtime $engineid [ clock seconds ]
	setRowValue setcfgidack $engineid 1
	if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
	    cancelOneShot $resendid
	}
	return
    }

#   Update the agent's "current" config ID to the real current configID.
#   --------------------------------------------------------------------
    setRowValue setcfgidtime $engineid 0
    setRowValue setcfgidack $engineid 0

    set agentloc [ getRowValue netloc $engineid ]
    lextract $agentloc 0 host 1 port

#   Construct the proper command and update the engine's "current" config
#   ID entry properly.
#   ---------------------------------------------------------------------

    if { $reset == 0 } {
	setRowValue setcfgid $engineid $currcfgid
	set vb [ makeTblUpdateVB $host $port $engineid $cfgid $currcfgid \
								$timestamp ]
    } else {
	setRowValue setcfgid $engineid 0
	set vb [ makeTblResetVB $host $port $engineid $cfgid ]
    }
    if { $vb == "" } {

#	There was nothing to send (no update necessary)!  Just update the
#	agent table info and quit.
#	-----------------------------------------------------------------
	setRowValue setcfgidtime $engineid [ clock seconds ]
	setRowValue setcfgidack $engineid 1
	ddl print trace "...nothing to send: cancelling enginecfgresend:\[$engineid\]\n"
	if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
	    cancelOneShot $resendid
	}
	return
    }

#   Remember that we have an outstanding table update 'set' to this
#   agent, out there.
#   ---------------------------------------------------------------
    set tblupdatetid [ idnext tid ]
    define tblupdatewait $tblupdatetid $engineid

    set cb [ list [ toe_self ] "handleTblUpdateResponse %result" $tblupdatetid ]

#   Set retries to 0, otherwise the snmp set could go through but return an
#   error in the set response, which would trigger a retry that could then
#   apply a key change a 2nd time, which then makes the keys incorrect
#   -----------------------------------------------------------------------
    set cmd "set $host $port -1 $vb -community [ lookup value superuser ] -retries 0 -timeout 3600"

    ddl print debug "updateAgent: snmp cmd is $cmd\n"

    setParameters command $cmd ecm $cb

    set target [ lookup value snmpService ]
    set cmd [ lookup value snmpCommand ]
    set result [ evalECM [ list $target $cmd ] ]

    clearParameters command ecm

#   Send the immediate results to our callback.
#   -------------------------------------------
    handleTblUpdateResponse $result
}

##############################################################################
# Proc:		updateAgents
#
# Abstract:	Method to update agents with new user data.
#
# Arguments:	skipbusy	1 to SKIP agents with unack'ed sends; 0 to
#				SKIP agents without unack'ed sends (the
#				latter is useful at startup).
#
# Returns:	Nothing.
##############################################################################
proc updateAgents { skipbusy } {

    ddl print trace "updateAgents: skipbusy \[$skipbusy\]\n"
    ddl print debug "updateAgents { $skipbusy }\n"

    #
    # Cancel any pending timer
    #
    if { ! [ catch { ilookup updateAgentTimer $skipbusy } timer ] } {
        cancelOneShot $timer
        undefine updateAgentTimer $skipbusy
    }


    #
    # Create list of agents to update
    #
    set agents {}
    foreach engineid [ getRowNames ] {
        set agentack [ getRowValue getcfgidack $engineid ]
        if { ( $agentack == 0 && $skipbusy == 1 )
             || ( $agentack == 1 && $skipbusy == 0 ) } {
            continue
	}
        lappend agents $engineid
    } 

    _updateAgents $agents $skipbusy
}

proc _updateAgents { agents skipbusy } {

    ddl print trace "_updateAgents skipbusy \[$skipbusy\]\n"
    ddl print debug "_updateAgents {[ list $agents $skipbusy ]}\n"

    set N [ lookup -d 10 value updateAgentBatchSize ]
    set T [ lookup -d 30 value updateAgentBatchInterval ]

    #
    # If list is more than N long, keep N and reschedule the rest in T seconds.
    #
    if { [ llength $agents ] > $N } {
        ddl print trace "_updateAgents: will only do $N; rest deferred.\n"
        set deferAgents [ lrange $agents $N end ]
        set agents [ lrange $agents 0 [ expr $N - 1 ] ]
        define updateAgentTimer $skipbusy [ registerOneShot $T [ list _updateAgents $deferAgents $skipbusy ] ]
    } else {
        undefine updateAgentTimer $skipbusy
    }

    foreach engineid $agents {
	getEngineCfg $engineid
    }
}

##############################################################################
# Proc:		registerAgent
#
# Abstract:	Method to handle an incoming userConfig trap.
#
# Arguments:	pdu		The incoming trap.
#
# Returns:	Nothing.
##############################################################################
proc registerAgent { pdu } {
    ddl print trace "registerAgent:  Got userConfig trap!\n"
    toe_send [ locate .services.io.snmp ] dumpPDU $pdu

    setParameters pdu $pdu

    #----------------------------------------------------------------------
    # Get data out of the varbind.
    #----------------------------------------------------------------------

    # <from class-event.toe:>
    # The varbind data looks like this for a SNMPv2 Trap:
    #
    #              <sysUpTime>
    #              <snmpTrapOID>        <-- the userConfig oid
    #		   <customOidSpec>      <-- { userCfgOID {<engineID> <cfgID>} }
    #		   <snmpTrapDomain>     <-- snmp UDP domain OID
    #		   <snmpTrapNetLoc>     <-- host netloc
    #		   <snmpTrapCommunity>  <-- only for SNMPv2u
    #
    # This info is compiled and sent upon instantiation of the
    # usmUserEngineCfg node in the usm-user table (after a short delay,
    # to make sending to self possible).

    set varbind [ getTrapParam varbind ]
    set trapinfo [ lindex $varbind 2 ]

    set netlocinfo [ lindex $varbind 4 ]
    lextract [ split [ lindex $netlocinfo 1 ] ":" ] 0 ipaddr 1 port

#   Verify the contents of the trap information.
#   --------------------------------------------
    if { [ llength [ lindex $trapinfo 1 ] ] != 2 } {
	ddl print warning "userConfig trap received from entity \[$netlocinfo\] with bad info \[$trapinfo\]\n"
	return
    }
    lextract [ lindex $trapinfo 1 ] 0 engineid 1 cfgid
    if { ![ validateEngineID $engineid ] } {
	ddl print warning "userConfig trap received from entity \[$netlocinfo\] with bad engine ID \[$engineid\]\n"
	return
    }

#   Get the agent table row, creating if necessary.
#   -----------------------------------------------
    if { [ catch { getRowIndex $engineid } idx ] } {
	ddl print trace "new engine - adding!\n"
	set idx [ getRowIndex $engineid -create ]
	updateInstances
	promoteDefVals $engineid
	setRowValue netloc $engineid [ list [ list $ipaddr $port ] ]
    }

#   Force the spin lock parameters to their "default" values in the engine
#   table.
#   ----------------------------------------------------------------------
    setRowValue getcfgidretries $engineid 0

#   Create new row in cfgserver's usm user table if necessary
#   ---------------------------------------------------------
    if { ![ isCfgServer $engineid ] } {
        set usmusertbl [ lookup value usmUserTable ]
        set usmuserobj [ locate $usmusertbl ]
        toe_send $usmuserobj "createUsmRow $engineid"
    }

#   Initiate a enginecfg request, unless one is already OUTSTANDING.  (If one
#   is already scheduled, cancel it.)
#   ------------------------------------------------------------------------
    sliceforeach tidkey engval enginecfgwait {
	if { $engval == $engineid } {
	    ddl print info "already have outstanding request for newly registered engine \[$engineid\] - skipping new request\n"
	    return
	}
    }
    if { ![ catch { lookup enginecfgresend $engineid } resendid ] } {
	cancelOneShot $resendid
    }

    getEngineCfg $engineid
}

##############################################################################
# Proc:		unregisterAgent
#
# Abstract:	Method to remove an agent from the cfgserver.
#
# Arguments:	engineid	Engine id of agent to be removed.
#
# Returns:	Nothing.
##############################################################################
proc unregisterAgent { engineid } {
    ddl print trace "unregisterAgent:  Removing engineid $engineid\n"

#   Remove the agent table row.
#   ---------------------------
    set engentryobj [ locate .iso*engineEntry ]
    if { [ catch { toe_send $engentryobj "getRowIndex $engineid" } idx ] } {
	ddl print trace "unregisterAgent: engineid $engineid already unregistered\n"
    } else {
        clearTableRow .iso*engineEntry $idx
	toe_send $engentryobj "syncSlice data"
    }


#   Remove row from cfgserver's snmp stack
#   and from usm user table.
#   --------------------------------------
    if { ![ isCfgServer $engineid ] } {
	set usmstatusobj [ locate .iso*usmUserEntry.usmUserStatus ]
	set usmentryobj [ locate .iso*usmUserEntry ]
        set superuser [ getSuperUser ]
        set sulocalize [ isUserKeyLocalized $superuser ]
        set superusmrow [ makeUsmUserRowName $superuser $engineid ] 
        toe_send $usmstatusobj "updateSnmpStackByRow $superusmrow 6"

        if { [ catch { toe_send $usmentryobj "getRowIndex $superusmrow" } idx ] } {
    	    ddl print trace "unregisterAgent: usm user $superusmrow already removed\n"
        } else {
            clearTableRow .iso*usmUserEntry $idx
	    toe_send $usmentryobj "syncSlice data"
        }
    }

#   Remove any enginecfg requests.
#   ------------------------------
    sliceforeach tidkey engval enginecfgwait {
	if { $engval == $engineid } {
	    undefine enginecfgwait $tidkey
	    return
	}
    }
    if { ![ catch { ilookup enginecfgresend $engineid } resendid ] } {
	cancelOneShot $resendid
	undefine enginecfgresend $engineid
    }
}


##############################################################################
# Proc:		setAuthKey
#
# Abstract:	Sets superuser's authentication key
#		for the specified engine into the cfgserver's
#	        usm user table (and snmp stack) so that the
#	        cfgserver can talk to the engine with the
#		correct superuser auth key.
#
#		If superuser's key is global (instead of
#		localized, as governed by localizeSuperuser in domain-config.x),
#		setAuthKey ALWAYS uses the <specified engine id>,<superuser_name>
#		as the instance name (except for cfgserver itself),
#		otherwise there is no way to
#		distinguish one superuser key from another for the
#		various engines.
#
# Arguments:	engineid	Engine ID whose superuser auth key
#				is being set in the cfgserver
#				usm user table.
#		keytype		Which key to set, i.e. old or new
#
# Returns:	Nothing.
##############################################################################
proc setAuthKey { engineid keytype } {

    ddl print trace "setAuthKey: engineid \[$engineid\], keytype \[$keytype\]\n"

    set superuser [ lookup value superuser ]
    append keyslice $keytype authkey
    ddl print debug "setAuthKey: keyslice is $keyslice\n"

#   Get last saved superuser auth key for this engine
#   -------------------------------------------------
    set savedAuthKey [ lookup -d "" $keyslice $engineid ]

    if { $savedAuthKey == "" } {
        ddl print info "setAuthKey: no value in $keyslice:$engineid\n"
	return
    }


#   Get last superuser auth key from cfgserver's usm user table 
#   -----------------------------------------------------------
    set usmAuthKey [ getAuthKey $engineid $superuser ]


#   If keys differ, set key into cfgserver's usm user table
#   -------------------------------------------------------
    ddl print debug "setAuthKey: engineid is $engineid, savedAuthKey is $savedAuthKey, usmAuthKey is $usmAuthKey\n"
    if { $savedAuthKey == $usmAuthKey } {
        ddl print debug "setAuthKey: no key change\n"
	return
    }


#   Get security and authentication info.
#   -------------------------------------
    set superusmrow [ evaluateMethod user "" \
				"makeUsmUserRowName $superuser $engineid" "" ]

    set usmusertbl [ lookup value usmUserTable ]
    set usmuserobj [ locate $usmusertbl ]

    set authProtocol [ getRowValue $usmusertbl.usmUserAuthProtocol $superusmrow ]
    set privProtocol [ getRowValue $usmusertbl.usmUserPrivProtocol $superusmrow ]
    if { $privProtocol != "0.0" && $privProtocol != "1.3.6.1.6.3.10.1.2.1" } {
	set hasPrivProtocol true
    } else {
	set hasPrivProtocol false
    }
    set maxlen [ lookup value maxAuthKeyLen ]

    set secnameoid [ toe_send $usmuserobj fullOidOf usmUserSecurityName ]

#   OID-type localized superuser index.
#   -----------------------------------
    set superidx [ join [ list [
	indexStrToOid [ lindex [ split $superusmrow "," ] 0 ] +1x:
    ] [
	indexStrToOid $superuser +
    ] ] "." ]

    regsub ^\. $usmusertbl / usmusertbl
    set vbpfx "sym/${usmusertbl}"
    set vb ""

    set userobj [ locate .iso*users ]
    if { [ catch { toe_send $userobj \ 
                   "getRowValue userTable.userEntry.localize $superuser" } \
                                localize ] } {
        set localize 1
    }

    # If we are using global superuser keys, then the cfgserver
    # (and only the cfgserver) instance names in the usm user table
    # will have a blank engineid.  The engineid WILL be present
    # for all other engines, whether we are using global or localized
    # superuser keys.
    if { [ isCfgServer $engineid ] && $localize == 0 } {
        set engidkey ""
    } else {
        set engidkey [ strtourl $engineid "1x:" ]
    }

    set keychange [ lookup -d "" authkeychange $engineid ]
    if { $keychange == "" } {
        ddl print warning "setAuthKey: no keychange found, generating one now ...\n"

        set oldAuthKey [ lookup -d "" oldauthkey $engineid ]
        set newAuthKey [ lookup -d "" newauthkey $engineid ]

        if { $oldAuthKey == "" || $newAuthKey == "" } {
            ddl print error "setAuthKey: unable to generate keychange\n"
            return
        }

        set keychange \
		[ snmp key $authProtocol createchange $oldAuthKey $newAuthKey $maxlen ]
    }


    append vb "{${vbpfx}.usmUserAuthKeyChange#${engidkey},${superuser} \
				{OCTET STRING} ${keychange} 1x:} "

    if { $hasPrivProtocol } {
        append vb "{${vbpfx}.usmUserPrivKeyChange#${engidkey},${superuser} \
 			    {OCTET STRING} ${keychange} 1x:} "
    }

    append vb "{${vbpfx}.usmUserStatus#${engidkey},${superuser} \
				INTEGER 1} "

#   Remember that we have an outstanding auth update 'set' to this
#   agent, out there.
#   ---------------------------------------------------------------
    set authupdatetid [ idnext tid ]
    define authupdatewait $authupdatetid [ list $engineid $keytype ]

    if { [ catch { lookup authupdateretries $engineid-$keytype } retries ] } {
        define authupdateretries $engineid-$keytype 1
    } else {
        define authupdateretries $engineid-$keytype [ incr retries -1 ]
    }

    set host [ lookup value .config.cfgserver.cfgserverServer ] 
    set port [ lookup value .config.cfgserver.snmpPort ] 

    set cb [ list [ toe_self ] "handleAuthUpdateResponse %result" $authupdatetid ]
    set cmd "set $host $port -1 {$vb} -community $superuser -retries 0 -timeout 3600"

    ddl print debug "setAuthKey: snmp cmd is $cmd\n"

    setParameters command $cmd ecm $cb

    set target [ lookup value snmpService ]
    set cmd [ lookup value snmpCommand ]
    set result [ evalECM [ list $target $cmd ] ]

    clearParameters command ecm

#   Send the immediate results to our callback.
#   -------------------------------------------
    handleAuthUpdateResponse $result
}

##############################################################################
# Proc:		getAuthKey
#
# Abstract:	Gets superuser's authentication key
#		for the specified engine from the cfgserver's
#	        usm user table.  As is true for setAuthKey, so it is also
#		with getAuthKey that while global keys may be supported,
#		the instance name will always be localized, i.e. of
#		the form <engine id>,<username>, except for the cfgserver
#		itself, where if we use global keys, the engineid WILL be
#		blank.
#		
#
# Arguments:	engineid	Engine ID whose user's auth key
#				is being retrieved from the cfgserver
#				usm user table.
#		user		User name.
#
# Returns:	Nothing.
##############################################################################
proc getAuthKey { engineid user } {

    ddl print trace "getAuthKey: engineid \[$engineid\], user \[$user\]\n"
    
    # If we are using global superuser keys, then the cfgserver
    # (and only the cfgserver) instance names in the usm user table
    # will have a blank engineid.  The engineid WILL be present
    # for all other engines, whether we are using global or localized
    # superuser keys.

    set userobj [ locate .iso*users ]
    if { [ catch { toe_send $userobj "getRowValue userTable.userEntry.localize $user" } localize ] } { 
	ddl print debug "getAuthKey: result from getRowValue of localize is $localize"
        set localize 1
    }

    ddl print debug "getAuthKey: localize is $localize\n"

    if { [ isCfgServer $engineid ] && $localize == 0 } {
        set engidkey ""
    } else {
# No strtourl conversion required since we are not doing any
# snmp operation here
#        set engidkey [ strtourl $engineid "1x:" ]
	 set engidkey $engineid
    }
    set usmusertbl [ lookup value usmUserTable ]
    set usmrow $engidkey,$user

    return [ getRowValue $usmusertbl.usmUserAuthKeyChange $usmrow ]
}


##############################################################################
# Proc:		handleAuthUpdateResponse
#
# Abstract:	Handles responses to a setAuthKey request.
#
# Arguments:    arc             The async return code:
#                               * type - wait, error or data
#                               * tid - the request's transaction ID
#                               * data - the latest TID, or an error msg,
#                                        or actual return data
#
# Returns:	Nothing.
##############################################################################
proc handleAuthUpdateResponse { arc } {

    ddl print trace "handleAuthUpdateResponse: arc \[$arc\]\n"
    lextract $arc 0 type 1 tid 2 data

#   Figure out what this is in response to.
#   ---------------------------------------
    if { [ catch { ilookup authupdatewait $tid } result ] } {
	ddl print warning \
		"ignoring unexpected auth update response - tid \[$tid\]\n"
	return
    }

    lextract [ lookup authupdatewait $tid ] 0 engineid 1 keytype

    undefine authupdatewait $tid

    switch $type \
	{wait} {

#	    'data' is the new tid to wait for.
#	    ----------------------------------
	    define authupdatewait $data $result
	} \
	{error} {

	    ddl print warning "authupdate response ERROR: \[$result\]\n"

            if { ![ catch { lookup authupdateretries $engineid-$keytype } retries ] } {

                if { $retries > 0 } {
                    ddl print info "retrying setAuthKey ...\n"
                    setAuthKey $engineid $keytype
                } else {
                    undefine authupdateretries $engineid-$keytype 
                }
            }
	} \
	{data} {

#	    The cfgserver is now officially updated with new superuser
#	    auth key for this engine.
#	    ----------------------------------------------------------
            undefine authupdateretries $engineid-$keytype 

	}
}


##############################################################################
# Proc:		isCfgServer
#
# Abstract:	Determines if cfgserver is the target host for an snmp packet
#
# Arguments:    engineid being checked
#
# Returns:	1 if cfgserver is the target, 0 otherwise
##############################################################################
proc isCfgServer { engineid } {

    ddl print trace "isCfgServer engineid \[$engineid\]\n"

    if { $engineid == [ snmp get snmpEngineID ] } {
        return 1
    } else {
        return 0
    }
}
