#! /bin/sh
#
# Component : ChorusOS patch
# Module    : unpatchit.sh
# Sypnosis  : script to unpatch a patched ChorusOS
#
# Copyright (c) 1998 by Sun Microsystems, Inc.
# All rights reserved.
#
# ident	"@(#)unpatchit.sh	1.0	98/09/12 SMI"
#



######################################################################
#
# FUNCTIONS
#
######################################################################

    #
    # ERROR
    #     Displays a message and exit 1
    #
ERROR() {
  echo " ** ERROR : $1"
  exit 1
}

    #
    # flush
    #    Black magic, to work around a bug in cygwin32.
    #
flush() {
    if [ "${OS}" = "win32" -a "${DISABLE_CYGWIN32_READ_WORKAROUND}" = "" ]; then
	exec <&2
    fi
}

    # 
    # usage
    #    Displays the script usage and exit 1
    #
usage () {
	PROGNAME=`basename $0`
	echo "USAGE:\n\t${PROGNAME}\n\n" \
	"\t\t\tStart this script from the patch saving directory\n" >&2
	exit 1
}


    #
    # check_path
    #      Check the value of a path by verifying it's contents 
    #      point to a readable file.
    #      Exit from program if the readable file is not present.
    #
check_path ()
{
  VARIABLE="$1"
  DESCRIPTION="$2"
  if [ -z "${DESCRIPTION}" ]; then
    DESCRIPTION = ""
  else
    DESCRIPTION="as ${2}"
  fi
  XFILE="$3"

  if [ -z "${XFILE}" ]; then
	if [ -d "${VARIABLE}" ] ; then
		echo "Using ${VARIABLE} ${DESCRIPTION}"
	else
		ERROR "${VARIABLE} is not a directory."
	fi
  else
	if [ -r "${VARIABLE}/${XFILE}" ] ; then
		echo "Using ${VARIABLE} ${DESCRIPTION}"
	else
		ERROR "${VARIABLE}/${XFILE} is not readable."
	fi
  fi
}


    #
    # check_file
    #      Check if the file is readable.
    #      Exit from program if the readable file is not present.
    #
check_file ()
{
  XFILE="$1"
  DESCRIPTION="$2"

  if [ ! -r "${XFILE}" ] ; then
    if [ -z "${DESCRIPTION}" ]; then
      ERROR "${XFILE} does not exist."
    else
      ERROR "${DESCRIPTION}" 
    fi
  fi
}

    #
    # getpath
    #      Ask a path
    #      If no answer is given, return the default path
    #
getpath () {
    flush
    echo "$1 [$2] " >&2
    read answer
    if [ -z "${answer}" ]; then
        echo "$2"
    else
        echo "${answer}"
    fi
}

    #
    # make_patch_profile
    #      Create the profile data
    #      Save in it all _CHORUSOS_XXXX variables
    #
make_patch_profile () {
	
chmod u+w "${PATCH_DIR}/etc/patch.env"
if [ $? != 0 ]; then
  ERROR "Can not edit the environment file."
fi

cat <<EOF > "${PATCH_DIR}/etc/patch.env"
#      Patch information

_CHORUSOS_PATCH_ID="${_CHORUSOS_PATCH_ID}"
_CHORUSOS_PARTNUMBER="${_CHORUSOS_PARTNUMBER}"
_CHORUSOS_HOST="${_CHORUSOS_HOST}"
_CHORUSOS_HOST_NICKNAME="${_CHORUSOS_HOST_NICKNAME}"
_CHORUSOS_RELEASE="${_CHORUSOS_RELEASE}"
_CHORUSOS_TARGET="${_CHORUSOS_TARGET}"
_CHORUSOS_BANNER="${_CHORUSOS_BANNER}"
_CHORUSOS_PATCHES_DIR="${_CHORUSOS_PATCHES_DIR}"
_CHORUSOS_PATCH_DATE="${_CHORUSOS_PATCH_DATE}"
_CHORUSOS_PATCH_LASTOP="`date`"

EOF
}

    #
    # default_profile_values
    #      Set the profile variables with the default values
    #
default_profile_values () {
_CHORUSOS_PATCH_ID="${_CHORUSOS_PATCH_ID:-123456}"
_CHORUSOS_PARTNUMBER="${_CHORUSOS_PARTNUMBER:-CLX320-AAAA}"
_CHORUSOS_HOST="${_CHORUSOS_HOST:-SPARC/Solaris}"
_CHORUSOS_HOST_NICKNAME="${_CHORUSOS_HOST_NICKNAME:-'solaris'}"
_CHORUSOS_RELEASE="${_CHORUSOS_RELEASE:-'3.2'}"
_CHORUSOS_TARGET="${_CHORUSOS_TARGET:-Intel x86}"
_CHORUSOS_BANNER="${_CHORUSOS_BANNER:-ChorusOS r3.2 Binary for Intel x86 on SPARC/Solaris host}"
_CHORUSOS_PATCHES_DIR="${_CHORUSOS_PATCHES_DIR:-'/ChorusOS/patches/'}"
_CHORUSOS_PATCH_DATE="${_CHORUSOS_PATCH_DATE:-`date`}"
_CHORUSOS_PATCH_LASTOP="${_CHORUSOS_PATCH_LASTOP:-`date`}"
}

    #
    # host_specific
    #      Find the OS
    #
host_specific () {
  # Find the host type
  UNAME=`uname`
  if [ "${UNAME}" = "Linux" ]; then
    OS=linux
  elif [ "${UNAME}" = "SunOS" ]; then
    case `uname -r` in
	4.*) OS=sunos;;
	5.*) OS=solaris;;
    esac;
  elif [ "${UNAME}" = "UNIX_SV" ]; then
    OS=unixware
  elif [ "${UNAME}" = "HP-UX" ]; then
    OS=hpux
  elif [ "${UNAME}" = "CYGWIN32/NT" ]; then
    OS=win32
  elif [ "${UNAME}" = "CYGWIN32_NT" ]; then
    OS=win32
  fi

  if [ -z "${OS}" ]; then
    ERROR "I have not been able to determine your host type. Sorry..."
  fi

  # Host specifics
  case "${OS}" in
    linux)	ECHO_N="echo -n";
		;;
    sunos)	ECHO_N="echo -n";
		;;
    solaris)	ECHO_N="printf %s";
		;;
    unixware)	ECHO_N="echo -n";
		;;
    svr4)	ECHO_N="echo -n";
		;;
    hpux)      	ECHO_N="printf %s";
		;;
    win32)	ECHO_N="printf %s";
		;;
  esac
}

    #
    # check_os
    #      Verify if the subpatch can be applied on this OS
    #
check_os() {
  # Check the host is the expected one
  if [ "${OS}" != "${_CHORUSOS_HOST_NICKNAME}" ]; then
    ERROR "This install script must be run on a ${_CHORUSOS_HOST} OS. Sorry"
  fi
}

    #
    # hereify
    #      Make all specified paths absolute
    #
HERE=`pwd`
hereify() {
  if [ -z "$1" ]; then
    return
  fi
  if echo "$1" | grep -v "^/" >/dev/null 2>&1; then
    if [ "$1" = "." ]; then
      echo "${HERE}"
    else
      echo "${HERE}/$1"
    fi
  else
    echo "$1"
  fi
}


##############################################################################
#
# BANNERS
#
##############################################################################

BANNER_UNPATCHIT() {
cat <<EOF

+---------------------------- ChorusOS Unpatchit -----------------------------+

EOF
}

BANNER_ENV() {
cat <<EOF

----- Finding patch environment -----

EOF
}

BANNER_COPY() {
cat <<EOF

----- Copy the patch files -----

EOF
}

BANNER_UNARCHIVE() {
cat <<EOF

----- Unarchive the patch files -----

EOF
}

BANNER_PREVIOUS_PATCH() {
cat <<EOF

----- Uninstall previous patch -----

EOF
}

BANNER_PATCHING() {
cat <<EOF

----- Applying patches -----

EOF
}

BANNER_UNPATCH() {
cat <<EOF

----- Unpatching -----

EOF
}

BANNER_END() {
cat <<EOF

----- The patch ${_CHORUSOS_PATCH_ID} has been successfully removed. -----

+----------------------------- ChorusOS Unpatchit -----------------------------+
EOF
}

##############################################################################
#
# main, Main, MAIN
#
##############################################################################

host_specific

# retrieve patch dir
PATCH_DIR=`dirname $0`
PATCH_DIR=`hereify "${PATCH_DIR}"`

# Test the current directory
$ECHO_N $$ > ${PATCH_DIR}/test.$$
if [ `cat ${PATCH_DIR}/test.$$` != $$ ]; then
  $ECHO_N "The directory ${PATCH_DIR} is not writable"
  $ECHO_N "Run unpatchit.sh from a writable directory"
  exit 1
fi
rm -f ${PATCH_DIR}/test.$$

# Log output and errors here
PATCH_OUTFILE="${PATCH_DIR}/unpatchit.out"	# Log output and errors here

# Rename the old log
[ -f "${PATCH_OUTFILE}" ] && mv "${PATCH_OUTFILE}" "${PATCH_OUTFILE}.old"

(
DISABLE_CYGWIN32_READ_WORKAROUND=TRUE
export DISABLE_CYGWIN32_READ_WORKAROUND

BANNER_UNPATCHIT

CIX="${PATCH_DIR}/bin/cix"
check_file "${CIX}" "Unable to find ${CIX}"
UNSUBPATCHIT="${PATCH_DIR}/bin/unsubpatchit.sh"
check_file "${CIX}" "Unable to find ${UNSUBPATCHIT}"

BANNER_ENV

# Load environments if any
if [ -f "${PATCH_DIR}/etc/patch.env" ]; then
	. "${PATCH_DIR}/etc/patch.env"
else
  ERROR "Unable to find the patch info\n"\
        "Launch the script from the patch saving directory."
fi



# Create default values if needed
default_profile_values

# Verify the OS
check_os

BANNER_UNPATCH

# Find subscripts environment files
LIST=`ls ${PATCH_DIR}/P*/subpatch.env`
if [ ! $? = 0 ]; then
  ERROR "WARNING: Nothing to apply in this patch"
fi

# Launch subscripts
for p in "${LIST}"
do
    # Launch of subscript
    if  sh "${UNSUBPATCHIT}" -E "${p}" ; then
	echo "${UNSUBPATCHIT} on ${p} applied."
    else
	ERROR "Unable to apply ${UNSUBPATCHIT} on ${p}\nThe  patch can not be removed"
    fi
done

make_patch_profile

PATCHES_DIR=`dirname "${PATCH_DIR}"`
if [ -d "${PATCHES_DIR}/current" ]; then
	rm -rf "${PATCHES_DIR}/current"
fi

BANNER_END "${_CHORUSOS_PATCH_ID}"

) 2>&1 | tee /dev/tty > "${PATCH_OUTFILE}" 2>&1


