###################################################################
## This is the subroutines package.
## created 9/23/98   Mike Shapiro
###################################################################

=head1 NAME 
   
st_globals - define Globals Constants

=head1 SYNOPSIS

B<&st_globals>

=head1 DESCRIPTION

B<st_globals> define global constants ( like a .h file in "c" )

=cut

sub st_globals {
    &check_language();
    if ( opendir( CDROM_FD, "/.tmp_proto" ) ) {
        $CDROM = "CDROM";
        close( CDROM_FD );
        if ( open( LITE_FD, "< .lite" ) ) {
           $LITE = "Lite";
           close( LITE_FD );
        }
    } 
    $PRODUCT    = "StorTools";
    $PKGNAME    = "STORtools";
    $BASEDIR    = "STORtools";
    $VARDIR     = "/var/opt";
    $PKGDIR     = "${VARDIR}/${BASEDIR}";
    $LOGDIR     = "${PKGDIR}/logs";
    $BINDIR     = "${PKGDIR}/bin";
    $LOGDIAGS   = "${LOGDIR}/diags";
    $LOGIVM     = "${LOGDIR}/ivm";
    $LOGSEDR    = "${LOGDIR}/sedr";
    $SYSLOG     = "/var/adm/messages";
    $MANDIR     = "${PKGDIR}/man";
    $TMPDIR     = "/tmp";
    $TMP_RAW    = "${TMPDIR}/current_${STORAGE}_raw";
    $STORAGE    = "A5";  # make A5K the default 
    $STOREDGE   = "StorEdge";
    $GOLD_PATH  = "${LOGDIR}/${STORAGE}";

    $T300_MESSAGES = "/var/adm/messages.t300";

    if ( $DEBUG ) {
        print "PRODUCT       $PRODUCT\n";
        print "PKGNAME       $PKGNAME\n";
        print "BASEDIR       $BASEDIR\n";
        print "VARDIR        $VARDIR\n";
        print "PKGDIR        $PKGDIR\n";
        print "LOGDIR        $LOGDIR\n";
        print "BINDIR        $BINDIR\n";
        print "LOGDIAGS      $LOGDIAGS\n";
        print "LOGIVM        $LOGIVM\n";
        print "LOGSEDR       $LOGSEDR\n";
        print "SYSLOG        $SYSLOG\n";
        print "TMPDIR        $TMPDIR\n";
        print "A5_TMP_RAW    $A5_TMP_RAW\n";
        print "T3_TMP_RAW    $T3_TMP_RAW\n";
    }

    $PRIMARY   = "primary";
    $ALTERNATE = "alternate";
    $NONE      = "none";
    $BOTH_AP   = "both";
    $LUN_DISK  = "Drive";
}

=head1 NAME 
   
st_sync - sync the filesystem before starting diagnostic testing

=head1 SYNOPSIS

B<&st_sync>

=head1 DESCRIPTION

B<st_sync> sync the filesystem before starting diagnostic testing

=cut

sub st_sync {

    # do 3, we always do 3. Thinks its a old unix bug probably fixed.
    system("/bin/sync");
    system("/bin/sync");
    system("/bin/sync");
}

=head1 NAME 
   
st_header - Displays a Standard header message

=head1 SYNOPSIS

B<&st_header>

=head1 DESCRIPTION

B<st_header> Displays a Standard header message

=cut

sub st_header {
    my $mode = $_[0];
    # defaults
    my $storage_type = "A5K";
    my $bus_type = "SBUS";

    my $host_name = `/usr/bin/hostname`;
    chomp( $host_name );
    ($VERSION) = grep { s/VERSION:\s+(\S.*)\s$/$1/ } `/usr/bin/pkginfo -l ${PKGNAME}`;
    if ( $STORAGE eq "T3" ) {
       $storage_type = "T300";
    } elsif ( defined @STORAGE_DEVICES ) {
       # for special case from mess_alert, get_storage()
       $storage_type = "";
       foreach $device ( @STORAGE_DEVICES ) {
           $storage_type .= $device;
       }
    }
    if ( $HBA eq "P" ) {
        $bus_type = "PCI";
    }
    if ( $mode eq "version" ) {
        print "${PRODUCT} Storage Management ${LITE}  Version ${VERSION}\n";
    } elsif ( $mode eq "log" ) {
        push @message, sprintf "${PRODUCT} Storage Management ${LITE}   Version ${VERSION}\n";
        push @message, sprintf "$bus_type $storage_type Series Storage on host $host_name.\n";
        return @message;
    } else {
        print "${PRODUCT} Storage Management  ${LITE}  Version ${VERSION}\n";
        print "$bus_type $storage_type Series Storage on host $host_name.\n";
    }
}

sub st_matrix_msg {
    print "\nBy default, revisions are verified using the most recent \"${PRODUCT}-StorEdge\n";
    print "Software/Firmware/Hardware Revision Matrix\" available.\n";
    print "Always run with the official version of this matrix!\n";
    print "\n";
    print "Reference: http://sunsolve.Sun.COM\n";
    print "Select \"site map\"\n";
    print "Select \"SSA/A5K/T300 Matrix\"\n";
    print "\n";
    print "If the current ${PRODUCT} Matrix is not up-to-date, load\n";
    print "the latest ${PRODUCT} patch or version.\n";
    print "Reference the Release Notes for more information.\n";
    print "\n";
}


sub system_clear {
        printf("\n\n\n");
}

=head1 NAME 
   
st_sum_test - Display test results

=head1 SYNOPSIS

B<&st_sum_test>

=head1 DESCRIPTION

B<st_sum_test> Display test results

=cut

sub st_sum_test {

    my $which_test = $_[0];
    my $test_name = $_[1];

    my %RC;
    my @summary;
    my $lp; 
    my $grlp;
    $RETCODE = 0;

    @summary = ("\n","$test_name Summary\n\n");

    foreach $port (@TEST_LP_LIST) {
        $grlp = `/usr/bin/ksh -c '/bin/egrep -i fail $LOGFILE.str.${which_test}.$CNUM{$port} 2> /dev/null'`;
        if ($grlp eq "") {
            $RC{$port} = "PASS";
        } else {
            $RC{$port} = "++++ FAIL ++++";
            $RETCODE = -1;
        }
        @summary = (@summary, "LOOP: ($CNUM{$port}) $port \t $RC{$port} \n");
    }

    print @summary;
    if ($RETCODE == 0) {
        print "${GREEN} ****** Test Passed ****** ${RC}\n";
    } else {
        print "${RED} ****** Test FAILED ****** ${RC}\n";
    }
    return $RETCODE;

}

=head1 NAME 
   
select_dual_port_mode - Get the user selected Dual Port Mode

=head1 SYNOPSIS

B<&select_dual_port_mode>

=head1 DESCRIPTION

B<select_dual_port_mode> Get the user selected Dual Port Mode

=cut

sub select_dual_port_mode {
    my $quiet = $_[0];
    my $dual_port = $NONE;
    while ( $dual_port eq $NONE ) {
        if ( $quiet ne "q" ) {
            print "WARNING: Do NOT run both Primary and Alternate!\n";
            print "         Executing I/O on both will result in a failover.\n";
        }
        print "Enter Dual Port Mode: [<p>, a]: ";
        $dp_choice = &get_response( "p" );
        if ( $dp_choice eq "p" ) {
            $dual_port = $PRIMARY;
        } elsif ( $dp_choice eq "a" ) {
            $dual_port = $ALTERNATE;
        } else {
            print "\n";
            print "Invalid choice: $dp_choice\n";
            print "p - primary path\n";
            print "a - alternate path\n";
            print "\n";
        }
     }
     return $dual_port;

}

=head1 NAME 
   
get_ddcnt - Get the correct disk drive count

=head1 SYNOPSIS

B<&get_ddcnt>

=head1 DESCRIPTION

B<get_ddcnt> Get the correct disk drive count

=cut

sub get_ddcnt
{
    my $port = $_[0];
    my $ddcnt;
    if ( $DUAL_PORT eq $PRIMARY ) {
       $ddcnt = $P_DDCNT{ $port }; 
    } elsif ( $DUAL_PORT eq $ALTERNATE ) {
       $ddcnt = $A_DDCNT{ $port }; 
    } else {
       $ddcnt = $DDCNT{ $port }; 
    }
    return $ddcnt;
}
=head1 NAME 
   
get_dual_port_disks - Get the correct disk list per CNUM based on the 
                      dual port mode

=head1 SYNOPSIS

B<&get_dual_port_disks>

=head1 DESCRIPTION

B<get_dual_port_disks> Get the correct disk list per CNUM based on the 
                       dual port mode

=cut

sub get_dual_port_disks {
    my $dual_port_mode = $_[0];
    my $port = $_[1];
    if ( $port !~ /^c/i ) {
        $port = $CNUM{ $port };
    }
    if ( ( $dual_port_mode ne $PRIMARY ) && 
         ( $dual_port_mode ne $ALTERNATE ) ) {
       print "Program error!\n";
       print "Invalid parameter (dual_port_mode: $dual_port_mode)\n";
       print "forcing parameter to -${PRIMARY}-\n";
       $dual_port_mode = $PRIMARY;
    }
    my @drives = ();
    $dl = `${BINDIR}/disk_inquiry -q -d 2>&1`;
    @disklist = split ('\n', $dl);
    foreach $line (@disklist) {
       ($drive, $vendor, $product, $revision, $serial_no, $dual_port) = split( " ", $line );
       #if ( ( $drive =~ /$CNUM{$port}/ ) && ( $dual_port eq $dual_port_mode ) ) {
       if ( ( $drive =~ /$port/ ) && ( $dual_port eq $dual_port_mode ) ) {
           push @drives, $drive;
           print "drive: $drive port: $dual_port\n" if ( $DEBUG );
       }
    }
    return @drives;
}


=head1 NAME 
   
sig_handler - catch signals and execute appropriate actions

=head1 SYNOPSIS

B<&sig_handler>

=head1 DESCRIPTION

B<sig_handler> is installed as the signal catcher in the hash
$SIG{}.  The user may define a C<&clean_up> routine it will be
called before B<sig_handler> calls exit.  A standard perl C<die>
message will be emitted if C<$VERBOSE> is defined.

=cut

sub sig_handler {
    my $signame = shift;
    if(defined &clean_up) {
        clean_up($signame);
    } 
    die "$PROGNAME caught SIG$signame!" if ($VERBOSE);
    exit($signame);
}

=head1 NAME

kill_local - send a signal to programs controlled by this programs tty

=head1 SYNOPSIS

B<&kill_local> I<$progs>, I<optional $signal>

=head1 DESCRIPTION

B<&kill_local> sends the I<$signal> to the specified I<progs>.  When no
I<$signal> is specified the C<TERM> signal is sent.  The work is actually
done by b<kill_proc>

=cut

sub kill_local {
    my ($progs, $signal) = @_;
    my $tty = `/bin/tty`;
    chomp($tty);
    $tty =~ s!/dev/!!; # clean off the leading /dev/
    &kill_proc ("$tty.*$progs", $signal);
}

=head1 NAME

kill_proc - send a signal to all programs specified

=head1 SYNOPSIS

B<&kill_proc> I<$string>, I<optional $signal>

=head1 DESCRIPTION

B<kill_proc> builds a list of processes to send signals, based on the parameter
I<$string>.  The optional parameter I<$signal> may be used to specify the signal
to use.  I<$signal> may be any of the values defined by C<kill -l>.

=cut

sub kill_proc {
    my ($string, $signal) = @_;
    my @kills;
    print "kill_proc: $string $signal\n" if ($DEBUG);
    my @candidates = grep { /$string/ } `/bin/ps -eo pid,tty,args`;
    foreach  (@candidates) {
        /^\s*(\d+).*/;
        push @kills, $1;
    }
    if ($#kills >= 0) {
        kill defined($signal) ? $signal : 'TERM', @kills;
    } else {
        print STDERR "No processes to kill\n";
    }
}

=head1 NAME

get_response - read I<STDIN> for user input

=head1 SYNOPSIS

B<&get_response> I<optional $default_value>

=head1 DESCRIPTION

B<get_response> reads I<STDIN> and sets the return value to
I<$default_value> when whitespace is the user response.

=cut

sub get_response {
    my ($default_value) = @_;
    chomp ($default_value);
    $response = <STDIN>;
    chomp($response);
    $response = $default_value if ($response eq '');
    $response =~ s/^[\s]+//;
    $response =~ s/\s+$//;
    return $response;
}

=head1 NAME

get_golden_snapshot - get the golden snapshot filename

=head1 SYNOPSIS

B<&get_golden_snapshot>

=head1 DESCRIPTION

B<get_golden_snapshot> reads the log directory and returns the
file name of the golden snapshot file.

=cut

sub get_golden_snapshot {
    use File::Basename;
    my ($file);
    open(LS, "/usr/bin/ksh -c 'ls -1t $GOLD_PATH/Golden* 2>/dev/null' |");
    chomp($file=<LS>);
    close LS;
    return basename($file);
}

=head1 NAME
logger - send messages to the system logger

=head1 SYNOPSIS

B<&logger> I<$status>, I<$error_number>, I<message>

=head1 DESCRIPTION

B<logger> passes off status messages to the system logger program.

=cut

sub logger {
    my ($status, $error_number, $message) = @_;
    if ($status eq 'ERROR') {
        $level = 'crit';
    } elsif ($status eq 'WARNING') {  
        # send these both as alert status since
        # warning and info may be disabled on many systems.
        $level = 'crit';
        $message = "warning: " . $message;
    } else {
        $level = 'crit';
        $message = "info: " . $message;
    }
	if ( $STORAGE eq "A5" ) {
		$STORAGE_DEVICE = "A5000";
	} elsif ( $STORAGE eq "T3" ) {
		$STORAGE_DEVICE = "T300";
	}
    $exec_str = 
    "/usr/bin/logger -p mail.$level -i -t \"[$STORAGE_DEVICE:DIAG:$status:$error_number]:\" \"$message\"";
    my $return_value = `/usr/bin/ksh -c '$exec_str 2>&1'`;

}

=head1 NAME

do_menu - print a menu and get a response

=head1 SYNOPSIS

B<&do_menu> I<\@menu_array>, [ I<$default_response> ]

=head1 DESCRIPTION

B<do_menu> prints the referenced menu array along with a standard
trailer and waits for user input.

=cut

sub do_menu {
    my $menu = shift @_;
    my $default = shift @_;
    my @trailer = 
      (
       "\n [q] Quit\n",
       "\nSelect an operation to perform: "
      );
    &system_clear();
	&st_header();
    print @$menu, @trailer;
    my $resp = get_response($default);
    return $resp;
}



=head1 NAME

kill_pid_files - will processes ids kept in <controller>.pids files in  logs dir

=head1 SYNOPSIS

B<&kill_pid_files> I<logdir> I< c | k > I<"TRUE" | "FALSE">

=head1 DESCRIPTION

This routine will first check to see if the each pid in a pid file is running.
If it is it is killed

arg1 = log directory.


There are two options for the second arg
if arg2 = k:   kill all pids
if arg2 = c:   kill pids by controller group


arg3 = if TRUE function silently

if arg3 is TRUE and arg2 is c ( kill by controller ) arg3 overides and 
kills all processes silently

=cut

sub kill_pid_files {

    my $logdir = $_[0];
    my $switch = $_[1];
    my $silent = $_[2];
    my (@pids, $i, $p, @delete_list, $fname);      
    my $header = 1;
    my $dont_check_dex = 0;
    my $first = 1;
    my $some = 0;

# The -t ( in the below /bin/ls line )
# is because it is important 
# to kill rw_dex before
# dex32 processes.  Otherwise it will start a new dex32.
# See rw_dex.


    $allfiles = `/usr/bin/ksh -c '/bin/ls -t $logdir/pids/c*.[0-9]* 2> /dev/null'`;
    $allfiles2 = `/usr/bin/ksh -c '/bin/ls -t $logdir/pids/f*.[0-9]* 2> /dev/null'`;
    $len = length($allfiles) + length($allfiles2);
    if( $len ) {
        %pids = ();
        @files = split(/\n/, $allfiles);
        for($i = 0 ; $i <= $#files; $i++ ) {
            @b = split(/\//, $files[$i]);
            @c = split(/\./, $b[$#b]);  # get controller
            $pids{$c[0]} = $pids{$c[0]} . " " . $c[1];
        }
        if ($allfiles2 ne '' and $switch !~ /^c/i) {
            @delete_list = ();
            @files = split(/\n/, $allfiles2);
            for($i = 0 ; $i <= $#files; $i++ ) {
                @b = split(/\//, $files[$i]);
                @c = split(/\./, $b[$#b]);  # get controller
                $pids{$c[0]} .= " $c[1]"; # append all pids for a filename
                push @delete_list, $testfile if ($writemode eq '1');
            }
        }

        foreach $i ( sort keys(%pids)) {
            if(  ($switch =~ /^c/i) && $silent ne "TRUE") {
                $rc = `/bin/ps -lf -p \"$pids{$i}\"`;
		@t = split(/\n/, $rc);
		next if(! ($#t > 0) );
		
		if($header) {

	        	if($silent ne "TRUE") {
       	     		  if(  ($switch =~ /^c/i) ) {
       	        	   print "\nKilling processes by controller\n";
       	        	   print "--------------------------------------\n";
       		     	  } else {
       	        	   print "\nKilling all processes\n";
       	        	   print "--------------------------------------\n";
       		     	}

			$header = 0;
		}

		$some++;
        }
                printf("\n$rc\n");
		$badinput = 1;
		while($badinput) {
                	printf("\nKill processes running on this Controller $i (y, <n>, q)?: ");
                	$in = <STDIN>;
			chomp($in);

			if( $in =~ /^y/i || $in =~ /^n/i || $in =~ /^q/i || $in eq "" ) {
				$badinput = 0;
			} else {
				printf("Invalid entry.  Please select again.\n");
			}
	
		}
            } else { 
                $in = "Y";
            }
                        
            if($in =~ /^Y/i ) {
                @pid_array = split(/ /, $pids{$i});
                foreach $p (@pid_array) {
                    if(!($p =~ /^\d+/)  ) { # check for vaid pid in array
                         next;   
                    }
                    $res = `/usr/bin/ksh -c '/bin/ps -p $p 2>/dev/null'`;

# mpat_lbf cannot be killed with INT
# must check to see if pid is a mpat_lbf or not

                    printf("Terminating $p and associated child processes.\n");

                    if($res =~ /dex32/ ) {
                        $res = `ps -g $p`;
                        @x = $res =~ /\w*\d{3,5}\w*/g; 
                        $res = system("/usr/bin/ksh -c 'kill $p 2>/dev/null'");  
                        sleep 2;
                        foreach $cpid (@x) {
                            if(wait2comp($cpid)) {
                                if($first) {
                                    printf("Note: Processes may continue to run for severval minutes \nbefore exiting while completing outstanding I/Os\n");
                                    $first = 0;
                                }
                                printf("Dex32 child process [$cpid] on controller [$i] has not terminated \nafter termination of parent process.\n");
                                $dont_check_dex = 1;
                            }

                        }

                    } else {
                        if( ($res =~ /mpat/) || ($res =~ /rw_dex/) ) {
                            $res = system("/usr/bin/ksh -c 'kill $p 2>/dev/null'");  
                        } else {
                            $res = system("/usr/bin/ksh -c 'kill -INT $p 2>/dev/null'"); 
                        }
                    }       
                    `/usr/bin/ksh -c '/bin/rm "$logdir/pids/$i.$p" 2>/dev/null'`;
                }
            } elsif ( $in =~ /^Q/i  ) { 
                return;
            } elsif ( $in =~ /^N/i || $in eq "") {
		
		$dont_check_dex = 1;
		$some = 1;
		next;
	    }
        }
    } else { 
        $some = 0;
    }

    printf("\n");
    if(!procs_a_runnin()) { # kill the tail files if no more processes are running
        $files = `/usr/bin/ksh -c '/bin/ls $logdir/pids/tail.* 2> /dev/null'`;
        if ( length($files) ) { 
            @files = split(/\n/,$files);    
            for($i = 0 ; $i <= $#files; $i++ ) {
                @b = split(/\//, $files[$i]);
                @c = split(/\./, $b[$#b]);
                `/usr/bin/ksh -c 'kill $c[1] 2>/dev/null'`;
                `/usr/bin/ksh -c 'rm $files[$i] 2>/dev/null'`;
            }
        }

        foreach $fname (@delete_list) {
            ###################################################
            # remove all files that dex with write were using #
            # since dex leaves them laying around             #
            ###################################################
            `/usr/bin/ksh -c '/bin/rm $fname 2>/dev/null`;
        }

        # also just for the heck of it check to see if any dex32s are running.
        if(!$dont_check_dex) {
            $some += check_dex32(); 
        }
    }

    if(!$some) {

        if($silent ne "TRUE" ) { 
            printf("No processes running.\n");
        }
    }

}


=head1 NAME

 show_pid_files()

=head1 SYNOPSIS

 B<show_pid_files> I<logdir>
 
=head1 DESCRIPTION
 
 Show processes that are running by controller
 for each pid in the pid files

=cut

sub show_pid_files {

    my $logdir = $_[0];
    my %pids;
    my $totalcount = 0;

    $files = `/usr/bin/ksh -c '/bin/ls $logdir/pids/c*.[0-9]*  2> /dev/null' `;
    $files2 = `/usr/bin/ksh -c '/bin/ls $logdir/pids/f*.[0-9]*  2> /dev/null' `;

    open(OUTPUT, "> /tmp/output$$");

    if ( length($files) ) {
        print  OUTPUT "\nCurrently running processes by controller (Parent processes only):\n\n";
        $first = 1;
        @files = split(/\n/,$files);
        for($i = 0 ; $i <= $#files; $i++ ) {
            #######################################
            # group all pids by controller number #
            # in hash $pids{c1}, $pids{c2} etc..  #
            #######################################
            @b = split(/\//, $files[$i]);
            @c = split(/\./, $b[$#b]);
            $pids{$c[0]} = $pids{$c[0]} . " " . $c[1];
        }
        foreach $i ( sort keys(%pids)) {
            ####################################################
            # perform a ps to see if each pid is still running #
            ####################################################
            $rc = `/bin/ps -lf -p \"$pids{$i}\"`;
            @a = split(/\n/, $rc);
            if($#a > 0 )  {  # check to see that it's not just the header line from ps
                print OUTPUT "Controller [$i]\n";
                print OUTPUT "------------------------------\n";
                print OUTPUT "$rc\n";
                $totalcount = $totalcount + $#a;
            }
        }
    }
    foreach $i (keys(%pids)) {
        ##############
        # clear hash #
        ##############
        delete $pids{$i};
    }

    if ( length($files2) ) {
        my ($pidnum, $file_contents, $fname, $writeflag);
        print OUTPUT "\nCurrently running processes by file (Parent processes only):\n\n";
        $first = 1;
        @files = split(/\n/,$files2);
        for($i = 0 ; $i <= $#files; $i++ ) {
            ########################################
            # group all pids by filename           #
            # in hash $pids{/tmp/bigfile}, etc.    #
            ########################################
            $pidnum = $1 if ($files[$i] =~ /.*file\.(\d+)$/);
            $file_contents = `cat $files[$i]`;
            ($fname, $writeflag) = split /\n/, $file_contents;
            $pids{$fname} = $pids{$fname} . " " . $pidnum;
        }
        foreach $i ( sort keys(%pids)) {
            ####################################################
            # perform a ps to see if each pid is still running #
            ####################################################
            $rc = `/bin/ps -lf -p \"$pids{$i}\"`;
            @a = split(/\n/, $rc);
            if($#a > 0 )  {  # check to see that it's not just the header line from ps
                print OUTPUT "File Name [$i]\n";
                print OUTPUT "------------------------------\n";
                print OUTPUT "$rc\n";
                $totalcount = $totalcount + $#a;
            }
        }
    }
    print OUTPUT "\nTotal: $totalcount test(s) running\n\n";
    close(OUTPUT);

    if ($totalcount) {
        system("/usr/bin/more /tmp/output$$");
        `/usr/bin/ksh -c '/bin/rm /tmp/output$$ 2>/dev/null'`;
    } else {
        print "No processes to show.\n";

    }
}



=head1 NAME 
   
get_fcal_hbas - find the existing HBA hba(s)

=head1 SYNOPSIS

B<&get_fcal_hbas>

=head1 DESCRIPTION

B<get_fcal_hbas> is called to detect all HBA(s), The
caller must check the global varibles to determine the detected HBA(s).

    $SBUS_SOCAL_HBAS TRUE detected, FALSE not found
    $PCI_QL2100_HBAS TRUE detected, FALSE not found
    $PCI_QL2200_HBAS TRUE detected, FALSE not found

=cut

sub get_fcal_hbas {

    # type of Host Bus Adapters(s)
    $SBUS_SOCAL_HBAS = "FALSE";
    $PCI_QL2100_HBAS = "FALSE";
    $PCI_QL2200_HBAS = "FALSE";
    my $pci_platform = "FALSE";

    chomp( $sbus_socal_platform = `/usr/sbin/prtconf -D | grep "SUNW,socal"` ); 
    if ( $sbus_socal_platform ) {
        $SBUS_SOCAL_HBAS = "TRUE";
    }

    chomp( $pci_platform = `/usr/sbin/prtconf -pv | grep "vendor-id:  00001077"` );
    if ( $pci_platform ) {
        chomp( $pci_ql2100_platform = `/usr/sbin/prtconf -pv | grep "device-id:  00002100"` );
        if ( $pci_ql2100_platform ) {
            $PCI_QL2100_HBAS = "TRUE";
        }
        chomp( $pci_ql2200_platform = `/usr/sbin/prtconf -pv | grep "device-id:  00002200"` );
        if ( $pci_ql2200_platform ) {
            $PCI_QL2200_HBAS = "TRUE";
        }
        # make sure there is an ifp driver for PCI HBA.
        # test for QLOGIC board with SUNW,ifp name property
        chomp( $ifp_driver = `/usr/sbin/prtconf -D | grep "SUNW,ifp"` );
        if ( ! $ifp_driver ) {
            # test for QLOGIC board with scsi name property
            chomp( $ifp_driver = `/usr/sbin/prtconf -D | grep "scsi" | grep "ifp"` );
        }
        if ( ! $ifp_driver ) {
            if ( $PCI_QL2100_HBAS eq "TRUE" ) {
                $PCI_QL2100_HBAS = "FALSE";
            }
            if ( $PCI_QL2200_HBAS eq "TRUE" ) {
                $PCI_QL2200_HBAS = "FALSE";
            }
        }
    }
}

=head1 NAME 
   
select_fcal_hbas - select the appropriate HBA 

=head1 SYNOPSIS

B<&select_fcal_hbas>

=head1 DESCRIPTION

B<select_fcal_hbas> is called to detect all HBA(s), The
caller must check the global varibles to determine the selected HBA(s).
$HBA is used for command line passing.

    $SBUS_SOCAL_HBAS TRUE selected, FALSE not selected
    $PCI_QL2100_HBAS TRUE selected, FALSE not selected
    $PCI_QL2200_HBAS TRUE selected, FALSE not selected
    $HBA = "S"  - for SBUS
    $HBA = "P"  - for Qlogic 2100 PCI
    $HBA = "Q"  - for Qlogic 2200 PCI

=cut

sub select_fcal_hbas {

    if ( $SBUS_SOCAL_HBAS eq "FALSE" && 
         $PCI_QL2100_HBAS eq "FALSE" &&
         $PCI_QL2200_HBAS eq "FALSE" ) {
        $HBA = "";
        return;
    }
    if  ( ( $SBUS_SOCAL_HBAS eq "TRUE" && $PCI_QL2100_HBAS eq "TRUE" ) || 
          ( $SBUS_SOCAL_HBAS eq "TRUE" && $PCI_QL2200_HBAS eq "TRUE" ) ||
          ( $PCI_QL2100_HBAS eq "TRUE" && $PCI_QL2200_HBAS eq "TRUE" ) ) {
        $muliple_hba_type = "TRUE";
    }
    while  ( $muliple_hba_type eq "TRUE" ) {
        print "\n";
        print "Multiple types of HBA(s) Detected.\n";
        print "Only one type of controller may be selected.\n";
        print "[s] SUN SBUS FC100\n" if ( $SBUS_SOCAL_HBAS eq "TRUE" );
        print "[p] QLogic 2100 PCI FC100\n" if ( $PCI_QL2100_HBAS eq "TRUE" );
        print "[q] QLogic 2200 PCI FC100 (unsupported)\n" if ( $PCI_QL2200_HBAS eq "TRUE" );
        print "\n";
        print "Select an HBA: ";
        $HBA = &get_response;
        print "\n";
        if ( ( $HBA eq "s" || $HBA eq "S" ) && ( $SBUS_SOCAL_HBAS eq "TRUE" ) ){
            $PCI_QL2100_HBAS = "FALSE";
            $PCI_QL2200_HBAS = "FALSE";
            $muliple_hba_type = "FALSE";
            last;
        }
        if ( ( $HBA eq "p" || $HBA eq "P" ) && ( $PCI_QL2100_HBAS eq "TRUE" ) ){
            $SBUS_SOCAL_HBAS = "FALSE";
            $PCI_QL2200_HBAS = "FALSE";
            $muliple_hba_type = "FALSE";
            last;
        }
        if ( ( $HBA eq "q" || $HBA eq "Q" ) && ( $PCI_QL2200_HBAS eq "TRUE" ) ){
            $PCI_QL2100_HBAS = "FALSE";
            $SBUS_SOCAL_HBAS = "FALSE";
            $muliple_hba_type = "FALSE";
            last;
        }
        print "$HBA is an invalid response, try again.\n";
    }

    print "SBUS_SOCAL_HBAS: $SBUS_SOCAL_HBAS \n" if($DEBUG);
    print "PCI_QL2100_HBAS: $PCI_QL2100_HBAS \n" if($DEBUG);
    print "PCI_QL2200_HBAS: $PCI_QL2200_HBAS \n" if($DEBUG);

    if ( $SBUS_SOCAL_HBAS eq "TRUE" ) {
        $HBA = "S";
    } elsif ( $PCI_QL2100_HBAS eq "TRUE" ) {
        $HBA = "P";
    } elsif ( $PCI_QL2200_HBAS eq "TRUE" ) {
        $HBA = "P";  # for the time being force to P, when supported use Q
    }
}

=head1 NAME 
   
get_storage - get all of the storage types

=head1 SYNOPSIS

B<&get_storage>

=head1 DESCRIPTION

B<get_storage> 

=cut

sub get_storage {
    my $mode = $_[0];
    my $storage;
    my $storage_device;
    undef @STORAGE_DEVICES;

    if ( $#LD_A5_PORTS >= 0 ) {
        push @STORAGE_DEVICES, "A5K";
    } 
    if ( $#LD_T3_PORTS >= 0 ) {
        if ( defined @STORAGE_DEVICES ) {
            push @STORAGE_DEVICES, " and ";
        }
        push @STORAGE_DEVICES, "T300";
    } 
    if ( $#LD_A5_PORTS < 0 && $#LD_T3_PORTS < 0 ) {
        undef @STORAGE_DEVICES;
    }
}

=head1 NAME 
   
select_storage - select the appropriate storage type 

=head1 SYNOPSIS

B<&select_storage>

=head1 DESCRIPTION

B<select_storage> 

    $STORAGE A5 = A5X00 selected; T3 = T3X0 selected

=cut

sub select_storage {
    my $mode = $_[0];

    if ( ( $#LD_A5_PORTS >= 0 ) && !( $#LD_T3_PORTS >= 0 ) ) {
        $STORAGE = "A5";
        $STORAGE_DEVICE = "A5000";
        $LUN_DISK = "Drive";
    } elsif ( ( $#LD_T3_PORTS >= 0 ) && !( $#LD_A5_PORTS >= 0 ) ) {
        $STORAGE = "T3";
        $STORAGE_DEVICE = "T300";
        $LUN_DISK = "LUN";
    } elsif ( $#LD_A5_PORTS < 0 && $#LD_T3_PORTS < 0 ) {
        print "WARNING: select_storage(): Storage Devices not Found\n";
        print "WARNING: select_storage(): defaulting to A5\n";
        $STORAGE = "A5";
        $STORAGE_DEVICE = "A5000";
        $LUN_DISK = "Drive";
    }

    while  ( $#LD_A5_PORTS >= 0 && $#LD_T3_PORTS >= 0 ) {
        print "\n";
        print "A5K Series and T300 Series Storage Detected.\n";
        print "Only one type of Storage may be selected.\n";
        print "[a] A5K series\n";
        print "[t] T300 series\n";
        print "\n";
        print "Select Storage: ";
        $STORAGE = &get_response;
        print "\n";
        if ( $STORAGE eq "a" || $STORAGE eq "A" ) {
            $STORAGE = "A5";
            $STORAGE_DEVICE = "A5000";
            $LUN_DISK = "Drive";
            last;
        }
        if ( $STORAGE eq "t" || $STORAGE eq "T" ) {
            $STORAGE = "T3";
            $STORAGE_DEVICE = "T300";
            $LUN_DISK = "LUN";
            last;
        }
        print "$STORAGE is an invalid response, try again.\n";
    }

    $GOLD_PATH  = "${LOGDIR}/${STORAGE}";
    print "STORAGE: $STORAGE \n" if($DEBUG);
}

=head1 NAME 
   
check_bus_option - check for valid CLI bus option 

=head1 SYNOPSIS

B<&check_bus_option>

=head1 DESCRIPTION

B<check_bus_option> will validate the CLI bus option against
the detected controllers.

=cut

sub check_bus_option {
    &get_fcal_hbas();
    if ( $_[0] eq "S" ) {
        if ( $SBUS_SOCAL_HBAS ne "TRUE" ) {
        die "-S (SBUS FC100) selected, HBA not detected!\n";
    }
        return "S";
    } elsif ( $_[0] eq "P" ) {
        if ( $PCI_QL2100_HBAS ne "TRUE" ) {
            die "-P (PCI FC100) selected, HBA not detected!\n";
        }
        return "P";
    }
}

=head1 NAME 
   
check_lite - check for lite version access

=head1 SYNOPSIS

B<&check_lite( $progname )>

=head1 DESCRIPTION

B<check_lite> is called to validate lite access.

=cut

sub check_lite {

    if ( defined $LITE ) {
        die "\t$_[0] is not execuable from $PRODUCT $LITE!\n";
    }
}

=head1 NAME 
   
check_language - check for LANG "C" 

=head1 SYNOPSIS

B<&check_language()>

=head1 DESCRIPTION

B<check_language> is called to validate environment varible LANG 

=cut

sub check_language {

    if (! $ENV{'LANG'}) {
        # Default case not defined.
        # This is OK
    } elsif ( $ENV{'LANG'} ne "C") {
        $ENV{'LANG'} = "C";
    }
}
 
=head1 NAME 
   
check_root_access - check for root access

=head1 SYNOPSIS

B<&check_root_access( $progname )>

=head1 DESCRIPTION

B<check_root_access> is called to validate user as root.
If the user is not root, terminate the perl script.

=cut

sub check_root_access {

    $user_id = $>;
    if ( $user_id != 0 ) {
        die "\t$_[0] requires root access!\n";
    }
}

sub accept_license {

    my $licensefile = "$BINDIR/NOTICE.txt";
    my $acceptancefile = "$BINDIR/accept";
    my $response;

    ##########################################
    # Display license and ask for acceptance #
    ##########################################

    if (-e $acceptancefile ) {
        ###################################################################
        # acceptance file exists, so user answered question "yes" already #
        ###################################################################
        return 1
    } else {
        $response = "view";
        if (! -e $licensefile ) {
            die "\n${PRODUCT} improperly installed. Please reinstall.\n\n";
        }
        while ($response =~ /^v/i) {
            system("/usr/bin/more $licensefile");
            print "\nDo you accept the terms? [yes, no, <view>]: ";
            $response = &get_response("view");
            if ($response =~ /^y/i) {
                system("/usr/bin/touch $acceptancefile");
                return 1;
            }
            return 0 if ($response =~ /^n/i);
            $response = "view";
        }
    }
}


=head1 NAME

wait2comp - check to see if specific pids are running

=head1 SYNOPSIS

B<wait2comp(pid list )>

=head1 DESCRIPTION

B<wait2comp> will return 1 if any of list is still running
             else return 0
=cut


####################################################################
# wait2comp 
#  wait for a specific jobs to finish
#       INPUT:  string of pids(s) to complete
#       OUTPUT: "TRUE" if job is currently working (in process table)
#               "" if job is done
#
#
#  ie-  wait2comp(pid1 , pid2 , pid3 .... )
#
####################################################################

sub wait2comp {

    my @pids = @_;  
    my $pid;
    my $p;

    $pCount = $#pids;

    foreach( $p = 0 ; $p <= $pCount ; $p++ ) {
        $pid = $pids[$p];
        if ($DEBUG) {
            printf("wait2comp: checking pid = $pid\n");
        }
        $ps = `ps -ef | fgrep -v fgrep | fgrep $pid`;
        @a = split(/\n/, $ps);
        foreach $i (@a) {
            $i =~ tr/ //s;
            @b = split(/ /, $i);
            if($DEBUG) {
                printf("wait2comp: b[2] = [$b[2]] pid = [$pid]\n");
            }
            chomp $pid;
            if( $b[2] eq $pid ) {
                return(1);
            } 
        }

    }

    return(0);

}


# make a pid file in $logdir/pids directory wth name controller.pid
# arg1 = log directory 
# arg2 = controller name 
# arg3 = pid

# 
=head1 NAME

make_pid_file - make a pid file in 

=head1 SYNOPSIS

B<make_pid_file> - make a pid file in $logdir/pids directory with name controller.pid 

=head1 DESCRIPTION

B<make_pid_file> - Arg1 = log directory, Arg2 = controller name arg3 = pid
                   Touch a file in subdirectory <pids> in logdir with the 
                   name controller.pid   

=cut



sub make_pid_file {

    my $logdir = $_[0];
    my $cont = $_[1];
    my $pid = $_[2];

    $logdir .= "/pids";
    if( ! -e "$logdir" ) {

        $res = system("/bin/mkdir $logdir");
        if($res) {
            printf("cannot make $logdir directory\n");
        }
    }

    if ( $pid ) {
        system("touch $logdir/$cont.$pid");
    } else {
        print"make_pid_file() Program error: Missing pid!\n";
    }

}


=head1 NAME

 procs_a_runnin()

=head1 SYNOPSIS

 B<procs_a_runnin> I<logdir>
 
=head1 DESCRIPTION
 
 returns(1) if any processes are runnin
 returns(0) if no processes are runnin

=cut

sub procs_a_runnin {

    my $logdir = $_[0];
    my (%pids, $files, $files2, @a, @b, @c);

    $files = `/usr/bin/ksh -c '/bin/ls $logdir/pids/c*.[0-9]*  2> /dev/null' `;
    $files2 = `/usr/bin/ksh -c '/bin/ls $logdir/pids/f*.[0-9]*  2> /dev/null' `;
       
    if ( length($files) ) {
        @files = split(/\n/,$files);
        for($i = 0 ; $i <= $#files; $i++ ) {
            @b = split(/\//, $files[$i]);
            @c = split(/\./, $b[$#b]);
            chomp $c[1];
            $pids{$c[0]} = $pids{$c[0]} . " " . $c[1];
        }
    }
    if ( length($files2) ) {
        @files = split(/\n/,$files2);
        for ($i = 0 ; $i <= $#files; $i++ ) {
            chomp $files[$i];
            if ($files[$i] =~ /.*file\.(\d+)$/) {
                $pids{'file'} = $pids{'file'} . " " . $1;
            }
        }
    }

    foreach $i (sort(keys(%pids))) {
        $rc = `/bin/ps -lf -p \"$pids{$i}\"`;
        @a = split(/\n/, $rc);
        return 1 if($#a > 0 );
    }
    return 0;
}


=head1 NAME

 check_dex32()

=head1 SYNOPSIS

 B<check_dex32> 
 
=head1 DESCRIPTION
 
 check specifically for dex32 processes
 returns(1) if any dex32 processes are runnin
 returns(0) if no dex32 processes are runnin

=cut


sub check_dex32 {

    $res = `ps -ef | fgrep dex32 | fgrep -v fgrep`; 

    @a = split(/\n/, $res);

    if($#a > 1) {

        printf("These dex32 processes are running.  They have previously been sent a kill signal.\n"); 

        $pcount = 0;
        foreach $i (@a) {
            printf("$i\n");
            ($pd) = $i =~ /\s*\w*\s*(\d+)/;
            $procs[$pcount++] = $pd;
        }

        printf("Note: These processes may continue to run for severval minutes \nbefore exiting while completing outstanding I/Os.\n");
        printf("Would you like to try to kill them again(<n> , y): ");
        $ans = <STDIN>;
        if($ans =~ /^Y/i ) {
            for($i = 0; $i < $pcount; $i++) {
                printf("Sending kill -9 signal to $procs[$i]\n");
                `/usr/bin/ksh -c 'kill -9 $procs[$i] 2>/dev/null'`;
            }
        }
        return(1);

    }

    return(0);

}

=head1 NAME

 make_port_list()

=head1 SYNOPSIS

 B<make_port_list> 
 
=head1 DESCRIPTION
 
 make a list of all of the online ports and arrange as:

 Offline port info
 LD_OFF_PORTS = List of Offline FC100 ports, available for HBA testing only

 Online port info
 LD_A5_PORTS   = List of Online FC100 ports with A5X00 Units Attached
                 ses and drives
                 available for all testing ( exlcude sedr )
 LD_ND_PORTS   = List of Online FC100 ports with ses
                 ses and no drives
                 HBA test, lbf tests (Loop Integrity)
 LD_T3_PORTS   = List of Online FC100 ports with T3XX Units attached
                 no ses, drives/LUNs, t300 in scsi inquiry
                 available for all testing ( exclude filtr )
 LD_MB_PORTS   = List of FCAL Onboard Motherboard port(s) 
                 no ses, drives
                 available for all testing ( exclude filtr )
 LD_STE_PORTS  = List of FC100 ports w/o loops
                 no ses, no drives
                 HBA test, lbf tests (Loop Integrity)
 LD_WARN_PORTS = List of FC100 ports with Warning messages from luxdiag
                 No testing

 SFNUM = # of Socal Fiber ports
 PORTNUM = # of Socal Fiber ports
 SOCNUM = # of FC100 HBA(s) (both ifp and socal)
 CNUM = c# per c#t#d#
 DDCNT = Disk Drive Count
 P_DDCNT = Disk Drive Count (Primary per Dual ported)
 A_DDCNT = Disk Drive Count (Alternate per Dual Ported)

 0 = normal return
 1 = no paths found

=cut

sub make_port_list {

    my $hba = $_[0];

    my $ldprt; 
    my $line;
    my $luxdiag_on_ports;
    my @luxdiag_on_ports;
    my $luxdiag_off_ports;
    my @luxdiag_off_ports;
    my @temporary_ports;
    my $portnum;

    # globals need to be cleared or run the risk of multiple calls appending

    undef @LD_OFF_PORTS;
    undef @LD_A5_PORTS;
    undef @LD_ND_PORTS;
    undef @LD_T3_PORTS;
    undef @LD_MB_PORTS;
    undef @LD_STE_PORTS;
    undef @LD_WARN_PORTS;

    undef $SFNUM;
    undef $PORTNUM;
    undef $SOCNUM;
    undef $CNUM;
    undef $DDCNT;
    undef $P_DDCNT;
    undef $A_DDCNT;

    &check_bus_option( $hba );

    # generate offline port info for HBA testing only!

    $luxdiag_off_ports = `/usr/bin/ksh -c '${BINDIR}/luxdiag -$hba port -f 2>/dev/null'`;
    @luxdiag_off_ports = split('\n', $luxdiag_off_ports);

    # Check the array for valid entries
    foreach $path ( @luxdiag_off_ports ) {
        # detect HBA paths that are in an error state and save them for
        # info only 
        # Note: will be detected by luxdiang -n and -f, report only once 
        if ( $path =~ /Warning/ ) {
            push @LD_OFF_WARN_PORTS, $path;
            printf( "LD_OFF_WARN_PORTS: $path \n" ) if ( $DEBUG );

        # we cannot detect an OFFLINE socal that is used for internal drives
        # only, because there would be no sf anyway, 
        # just stick them in offline for ha testing

        } else {
            push @LD_OFF_PORTS, $path;
            printf( "LD_OFF_PORTS: $path \n" ) if ( $DEBUG );
        }
    }

    # generate Online port info 

    $luxdiag_on_ports = `/usr/bin/ksh -c '${BINDIR}/luxdiag -$hba port -n 2>/dev/null'`;
    if ($luxdiag_on_ports eq "") {
        return 1;
    }
    @luxdiag_on_ports = split('\n', $luxdiag_on_ports);

    # Check the array for valid entries
    foreach $path ( @luxdiag_on_ports ) {
        # detect HBA paths that are in an error state and save them for
        # the corrective action, no testing can be performed on a bad path.
        if ( $path =~ /Warning/ ) {
            push @LD_WARN_PORTS, $path;
            printf( "LD_WARN_PORTS: $path \n" ) if ( $DEBUG );
        } else {
            # all the invalid HBA(s) should be eliminated at this point
            # now temporarily save for the domain
            push @temporary_ports, $path;
            printf( "temporary_ports: $path \n" ) if ( $DEBUG );
        }
    }

    $dl = `${BINDIR}/disk_inquiry -d -P -q 2>/dev/null`;
    @disklist = split ('\n', $dl);

    # init the disk drive count
    foreach $ldprt (@temporary_ports) {
        $DDCNT{$ldprt} = 0;
    }

    # convert luxdiag output to match disklist and path-to-inst
    foreach $ldprt (@temporary_ports) {
        ($socpp{$ldprt}) = $ldprt =~ /\/devices(\/\S+):(\d+)/;
        $portnum{$ldprt} = $2;
        if ( $hba eq "S" ) {
            $phys_path{$ldprt} = "$socpp{$ldprt}/sf\@$portnum{$ldprt},0";
        } elsif ( $hba eq "P" ) { 
            $phys_path{$ldprt} = $socpp{$ldprt};
            if ( $1 =~ /ifp/ ) {
                $phys_path{$ldprt} =~ s/(ifp@\d+),0/$1/;
            } elsif ( $1 =~ /scsi/ ) {
                $phys_path{$ldprt} =~ s/(scsi@\d+),0/$1/;
            }
        }
    }

    # count the disks and get the cnum for this path
    foreach $line (@disklist) {
        ($drive, $vendor, $product, $revision, $serial_no, $dual_port) = split( " ", $line );
        foreach $ldprt (@temporary_ports) {
            if ($line =~ /$phys_path{$ldprt}/) {
                ($CNUM{$ldprt}) = $line =~ /(c\d+)t\d+d\d+/;
                if ( $DEBUG ) {
                    print "phys_path $phys_path{$ldprt}\n";
                    print "line $line\n";
                    print "CNUM $CNUM{$ldprt}\n";
                    print "ldprt $ldprt\n";
                }
                if ( !defined $P_DDCNT{$ldprt} ) {
                    $P_DDCNT{$ldprt} = 0;
                }
                if ( !defined $A_DDCNT{$ldprt} ) {
                    $A_DDCNT{$ldprt} = 0;
                }
                if ( $dual_port eq $PRIMARY ) { 
                    $P_DDCNT{$ldprt}++;
                } elsif ( $dual_port eq $ALTERNATE ) { 
                    $A_DDCNT{$ldprt}++;
                }
                $DDCNT{$ldprt}++;
            }
        }
    }

    # Create Global PORTNUM
    foreach $ldprt (@temporary_ports) {
        ($socpp{$ldprt}) = $ldprt =~ /\/devices(\/\S+):(\d+)/;
        $PORTNUM{$ldprt} = $2;
    }

    my $path_to_inst = $ENV{PATH_TO_INST} || "/etc/path_to_inst";
    my @P2INST;
    if (open(P2INST, "<${path_to_inst}")) {
        @P2INST = <P2INST>; # read entire file
    } else {
        # could not open file
        die "\nCould not locate $path_to_inst file\nPlease set ENV variable \$PATH_TO_INST to proper location of /etc/path_to_inst\n\n";
    }
    if ($P2INST[0] =~ /#path_to_inst_bootstrap 1/) {
        ##########################################
        # Did not find correct path_to_inst file #
        ##########################################
        die "\nCould not locate proper /etc/path_to_inst file\nPlease set ENV variable \$PATH_TO_INST to proper location of this file\n\n";
    }

    # get the driver instance for this port
    foreach $line (@P2INST) {
        foreach $ldprt (@temporary_ports) {

            if ( $hba eq "S" ) {
                if ($line =~ /\"sf\"/) {
                    if ($line =~ /$phys_path{$ldprt}/) {
                        @line = split(' ', $line);
                        $SFNUM{$ldprt} = "sf$line[1]";
                    }
                } elsif ($line =~ /\"socal\"/) {
                    if ($line =~ /$socpp{$ldprt}/) {    
                        @line = split(' ', $line);
                        $SOCNUM{$ldprt} = "socal$line[1]";
                    }
                }
            } elsif ( $hba eq "P" ) {
                if ( ( $line =~ /\"ifp\"/ ) || ( $line =~ /\"scsi\"/ ) ) {
                    if ($line =~ /$phys_path{$ldprt}/) {
                        @line = split(' ', $line);
                        $SOCNUM{$ldprt} = "ifp$line[1]";
                        $SFNUM{$ldprt} = "ifp$line[1]";
                    }
                }
            }
        }
    }

    # get all the disk and paths on the system and parse for purple(s)
    $di = `${BINDIR}/disk_inquiry -p 2>/dev/null`;
    @di_items = split ( '\n', $di );
    $pp = 0;
    for ( $i = 0; $i < $#di_items; $i++ ) {
        $item = $di_items[ $i ];
        if ( $item =~ /T300/ ) {
           print "item: $item\n" if ( $DEBUG );
           ++$i;
           if ( ( $hba eq "S" ) && ( $di_items[ $i ] !~ /pci/ ) ) {
               ($purple_path) = $di_items[ $i ] =~ /(\/\S+)\/sf/;
               $purple_path .= ":";
               ($portnum) = $di_items[ $i ] =~ /sf@(\d),/;
               $purple_path .= $portnum;
           } elsif ( ( $hba eq "P" ) && ( $di_items[ $i ] !~ /sbus/ ) ) {
               ($purple_path) = $di_items [ $i ] =~ /(\/\S+)\/ssd/;
		   }
          
           print "purple_path: $purple_path\n" if ( $DEBUG );
           $offset = 0;
           foreach $path ( @temporary_ports ) {
               print "path: $path\n" if ( $DEBUG );
               if ( $path =~ $purple_path ) {
                   $duplicate_port = grep { /$path/ } @LD_T3_PORTS;
                   if ( ! $duplicate_port ) {
                       $LD_T3_PORTS[ $pp++ ] = $path;
                       print "T300: $path\n" if ( $DEBUG );
                   }
               }
            }
        }
    }

    # remove purple ports from temporary_ports

    foreach $t3x0_path ( @LD_T3_PORTS ) {
        $offset = 0;
        foreach $path ( @temporary_ports ) {
            $pp = grep { /$path/ } @LD_T3_PORTS;
            if ( $pp ) {
                splice ( @temporary_ports, $offset, 1 );
            }
            $offset++;
        }
    }

    foreach $path (@temporary_ports) {
        if ( $DDCNT{$path} != 0 ) {
            # detect onboard or MotherBoard socal(s) and save them for
            # hba, lbf and drive testing.
            # these ports should have a drive count w/o any ses devices
            if ( $path =~ /,10000:/ ) {
                # get all ses devices
                print "path: $path\n" if ( $DEBUG );
                $sf_path = $path;
                $sf_path =~ s/:\d/\/sf*/;
                print "sf_path: $sf_path\n" if ( $DEBUG );
                $ses_devices = `/usr/bin/ksh -c '/usr/bin/ls -l $sf_path 2>&1' | /usr/bin/grep ses`;
                print "ses_devices: $ses_devices\n" if ( $DEBUG );
                if ( !$ses_devices ) {
                    # no ses and a disk drive count, assume interal soc.
                    push @LD_MB_PORTS, $path;
                    printf( "LD_MB_PORTS: $path \n" ) if ( $DEBUG );
                } else {
                    # ses and a disk drive count, A5K
                    push @LD_A5_PORTS, $path;
                    printf( "LD_A5_PORTS: $path \n" ) if ( $DEBUG );
                }
            } else {
                # not 1000 and a disk drive count, A5K
                # watch out here for FC3500 may need to check for ses!
                push @LD_A5_PORTS, $path;
                printf( "LD_A5_PORTS: $path \n" ) if ( $DEBUG );
            }
        } 

        if ( $DDCNT{$path} == 0 ) {
            # get all ses devices
            print "path: $path\n" if ( $DEBUG );
            $port_path = $path;
            # adjust path for ls of ses devices
            if ( $path =~ /socal/ ) {
                $port_path =~ s/:\d/\/sf*/;
            } elsif ( $path =~ /ifp/ ) {
                $port_path =~ s/,0:0/\//;
            } elsif ( $path =~ /scsi/ ) {
                $port_path =~ s/,0:0/\//;
            }
            print "port_path: $port_path\n" if ( $DEBUG );
            $ses_devices = `/usr/bin/ksh -c '/usr/bin/ls -l $port_path 2>&1' | /usr/bin/grep ses`;
            print "ses_devices: $ses_devices\n" if ( $DEBUG );
            if ( !$ses_devices ) {
                # no drives or ses, assume STE
                push @LD_STE_PORTS, $path;
                $CNUM{$path} = "s";
                $CNUM{$path} .= "$#LD_STE_PORTS";
                printf( "LD_STE_PORTS: $path \n" ) if ( $DEBUG );
            } else {
                # no drives but ses, assume A5K w/o drives
                push @LD_ND_PORTS, $path;
                $CNUM{$path} = "n";
                $CNUM{$path} .= "$#LD_ND_PORTS";
                printf( "LD_ND_PORTS: $path \n" ) if ( $DEBUG );
            }
        }

    }

    if ( @LD_T3_PORTS ) {
        chomp($uname_out = `uname -a`);
        @uname_out = split(' ', $uname_out);
        $os_ver = $uname_out[2];
        print "DEBUG: OS VERSION = $os_ver \n" if ($DEBUG);
        if ($os_ver =~ "5.5" ) {
            undef @LD_T3_PORTS;
            print "\tRemoving T300 ports from set of available test ports.\n";
            print "\tThis is not a supported OS release (2.5.x). [$os_ver] for T300.\n";
            print "\nPress ENTER to continue or q to quit: ";
            my $ans = &get_response;
            if ( $ans =~ /^q/i ) {
                exit 0;
            }
        }

    }
    &print_ports() if ( $DEBUG );
    close(P2INST);
    return 0;
}

sub print_ports {

    foreach $line ( @LD_A5_PORTS ) {
        print "A5 $line\n";
    }
    foreach $line ( @LD_T3_PORTS ) {
        print "T3 $line\n";
    }
    foreach $line ( @LD_ND_PORTS ) {
        print "ND $line\n";
    }
    foreach $line ( @LD_MB_PORTS ) {
        print "MB $line\n";
    }
    foreach $line ( @LD_STE_PORTS ) {
        print "STE $line\n";
    }
    foreach $line ( @LD_OFF_PORTS ) {
        print "OFF $line\n";
    }

}

=head1 NAME 
   
set_color - turn color printing on/off

=head1 SYNOPSIS

B<&set_color( $state )>

=head1 DESCRIPTION

B<set_color> is called to turn color printing of warnings/errors ON or OFF

=cut

sub set_color {

    my $state = $_[0];

    if ( $state eq "ON" ) {
        $RED="\033[31m";         # set to red character. (errors)
        $BLUE="\033[1;34m";      # set to blue bold character. (quiet info)
        $GREEN="\033[32m";       # set to green character. (pass, good, go)
        $YELLOW="\033[1;33m";    # set to bold yellow (warninings)
        $RC="\033[0m";           # reset color to default (black and white)
    } elsif ( $state eq "OFF" ) {
        undef $RED;
        undef $BLUE;
        undef $GREEN;
        undef $YELLOW;
        undef $RC;
    } else {
        die "\tProgram error: invalid argument to set_color( $_[0] )\n";
    }


}


=head1 NAME 
   
check_dma_errors - Check /var/adm/messages for DMA errors #

=head1 SYNOPSIS

B<&check_dma_errors()>

=head1 DESCRIPTION

B<check_dma_errors>  Check /var/adm/messages for DMA errors #

=cut
# Jul  6 11:06:51 tester unix: WARNING: /sbus@2,0/SUNW,socal@d,10000/sf@1,0 (sf1):
# Jul  6 11:06:51 tester unix:  Transport error on target=0x4:  Fiber Channel: INCOMPLETE DMA XFER on sbus 

sub check_dma_errors {

    my $stale_dma_errors = $_[0];
    my $verbose = $_[1];

    my ($dateinfo, $socalinfo, $realerror);
    my $status = 0;
    my $dma_errors = 0;

    if ( ! $SYSLOG ) {
	$SYSLOG = "/var/adm/messages";
    }

    open FILE, $SYSLOG or return;
    while (<FILE>) {
        if (/^(\w+\s+.*\d+:\d+:\d+)\s+\w+.*WARNING:\s+(.*)/) {
            $dateinfo = $1;
            $socalinfo = $2;
            $realerror = <FILE>;
            if ($realerror =~ /INCOMPLETE DMA XFER/) {
		$status = -1;
		$dma_errors++;
		if ( ( $verbose == 1 ) && ( $dma_errors > $stale_dma_errors ) ) {
                    print "\n${RED}WARNING: Detected an error on $dateinfo on $socalinfo ${RC}\n";
                    print "${RED}If this host adaptor has not already been replaced, it should be replaced immediately. ${RC}\n";
		}
            }
        }
    }
    
    $dma_errors = $dma_errors - $stale_dma_errors;
    if ( ( $status == -1 )  && ( $dma_errors > 0 )  && ( $verbose == 1 ) ) {
        print "\n${YELLOW}$dma_errors DMA error(s). ${RC}\n";
    }
    return $dma_errors;
}

=head1 NAME

 in_cluster()

=head1 SYNOPSIS

 B<in_cluster>

=head1 DESCRIPTION

 See if this host is in a cluster. If it is, return the name of the cluster
 otherwise return 0.

=cut

sub in_cluster {

  # detect if cluster daemon is running and ask it if we are in the cluster right now.
  # return 1 if we are in the cluster
  my $clusternamefile = "/etc/opt/SUNWcluster/conf/default_clustername";
  my ($psinfo, $clustername, $state);

  $psinfo = `/usr/bin/ps -ef | /usr/bin/grep clustd | /usr/bin/fgrep -v grep`;
  chomp $psinfo;
  if ($psinfo) {
      # clustd daemon is running ask it about the cluster state
      # "end" = running
      # find cluster name from file
      return 0 if (! -e $clusternamefile);
      $clustername = `/usr/bin/ksh -c '/usr/bin/cat $clusternamefile 2>&1'`;
      chomp $clustername;
      return 0 if (! $clustername);
      $state = `/usr/bin/ksh -c '/opt/SUNWcluster/bin/clustm getstate $clustername 2>&1'`;
      chomp $state;
      return $clustername if ($state eq 'end');
      return 0;
  }
  return 0;
}

=head1 NAME

 get_volume_manager_data()

=head1 SYNOPSIS

 B<get_volume_manager_data>

=head1 DESCRIPTION

 Get all known Volume Manager Disk Information.
 Systems could have Veritas and SDS on same host. This info is useful
 for Snapshot comparison as well as to detect deportation of volume groups.
 Also, the disks can then be checked for VM control and we can refuse to
 write on them.

=cut

sub get_volume_manager_data {

  # Get all known Volume Manager Disk Information.
  # Systems could have Veritas and SDS on same host. This info is useful
  # for Snapshot comparison as well as to detect deportation of volume groups.
  # Also, the disks can then be checked for VM control and we can refuse to
  # write on them.

  my ($mode, $psinfo, $return);

  # Veritas - check for daemon running
  $psinfo = `/usr/bin/ps -ef | /usr/bin/grep vxconfigd | /usr/bin/fgrep -v grep`;
  chomp $psinfo;

  $mode = `/usr/bin/ksh -c '/usr/sbin/vxdctl mode 2>&1'` if ($psinfo);
  if ($mode =~ /enabled/) {
      $VERITAS_DISK_LIST = `/usr/bin/ksh -c '/usr/sbin/vxdisk list 2>&1' | /usr/bin/grep -v ERROR`;
      chomp $VERITAS_DISK_LIST;
      $VERITAS_PRINT_LIST = `/usr/bin/ksh -c '/usr/sbin/vxprint -htv 2>&1' | /usr/bin/grep -v ERROR`;
      chomp $VERITAS_PRINT_LIST;
  }
  # Solstice Disk Suite - check for daemon
  $pkginfo = `/usr/bin/ksh -c '/bin/pkginfo SUNWmd 2>&1'`;
  if ($pkginfo !~ /was not found/ ) {
      $SDS_DISK_LIST = `/usr/bin/ksh -c '/usr/opt/SUNWmd/sbin/metastat 2>&1' | /usr/bin/grep -v 'no existing databases'`;
      chomp $SDS_DISK_LIST;
  }
  $results = "";
  if ($VERITAS_DISK_LIST) {
      $results .= "VERITAS\n";
      $results .= "-------\n\n";
      $results .= "$VERITAS_DISK_LIST\n\n$VERITAS_PRINT_LIST\n";
  }
  if ($SDS_DISK_LIST) {
      $results .= "SOLSTICE DISK SUITE\n";
      $results .= "-------------------\n\n";
      $results .= "$SDS_DISK_LIST\n";
  }
  return $results;
}


=head1 NAME

 get_volume_manager_hash()

=head1 SYNOPSIS

 B<get_volume_manager_hash>

=head1 DESCRIPTION

 Get Volume Manager Disk statuses into a Perl hash.
 This info is useful for determining if a disk is being managed by Veritas or SDS.
 Any disk not in the hash is not managed by Veritas or SDS.

=cut

sub get_volume_manager_hash {

  my %vmhash;
  my $vmdata = &get_volume_manager_data;

  if ($vmdata) {
      @vm = split /\n/, $vmdata;
      my $inrange = 0;
      foreach $line (@vm) {
          chomp $line;
          if ($line =~ /^DEVICE/) {
              $inrange = 1;
              next;
          }
          if ($line eq '') {
              $inrange = 0;
              next;
          }
          next if (! $inrange);
          @fields = split /\s+/, $line;
          if ($fields[5] ne '') {
              $vmhash{$fields[0]} = "$fields[4] $fields[5]";
          } else {
              $vmhash{$fields[0]} = $fields[4];
          }
      }
      foreach $line (@vm) {
          chomp $line;
          if ($line =~ /^\s+Device\s+Start\s+Block/) {
              $inrange = 1;
              next;
          }
          if ($line eq '') {
              $inrange = 0;
              next;
          }
          next if (! $inrange);
          @fields = split /\s+/, $line;
          $vmhash{$fields[1]} = $fields[4];
      }
  }

  return %vmhash;
}


sub choose_patterns {

  # Allow user to choose a set of patterns to use for testing loops
  # this subroutine returns a file path to the patterns except if all
  # patterns are desired, then 'all' is returned.
  # '' if quit was selected

  my $regex = shift;
  my (@regex, $i, $j, $p_count, %seen, @used_patterns, @temp, $ans, $ans2);
  my $pattern_choice = "";
PAT_MENU:
  print "\nSelect Patterns\n";
  print "---------------\n\n";
  print " [1] Test With All Patterns\n";
  print " [2] Test With Failed Patterns From Logs\n";
  print " [3] Test With Patterns From A File\n";
  print " [4] Enter A List Of Patterns\n";
  print "\n\n";
  print " [q] = Quit \n";
  print "\n\n";
  print "Please select patterns to test (1 - 4, <q>): ";
  $pattern_choice = &get_response;

  if ($pattern_choice =~ /^q/i) {
      return '';
  } elsif (($pattern_choice eq '1') or ($pattern_choice eq 'a')) {
      return 'all';
  } elsif ($pattern_choice eq '2') {
      # Failed patterns
      return '' if (! $regex);
      # see if multiple regex's separated by '|' symbol
      @regex = split /\|/, $regex;
      foreach $expression (@regex) {
          if (open PATTERNS, "grep FAIL $LOGDIAGS/$expression* |") {
              # pull out all the failed patterns
              while (<PATTERNS>) {
                  chomp;
                  if (/Pat=0x(([a-fA-F0-9])+)\s+/) {
                      push @temp, $1;
                  }
              }
              close PATTERNS;
          }
      }
      print "\nPatterns\n";
      print "--------\n\n";
      $p_count = 0;
      %seen = ();
      @used_patterns = ();
      @used_patterns = grep { ! $seen{$_} ++ } @temp; #de-dupe
      foreach $i (@used_patterns) {
          $p_count ++;
          print "$i\n";
      }
      print "\n$p_count failed patterns found.\n";
      goto PAT_MENU if (! $p_count);
      print "\nUse These Patterns [<y>, n, q]: ";
      $ans = &get_response;
      return '' if ($ans =~ /^q/i);
      if (($ans eq '') or ($ans =~ /^y/i)) {
PAT_FILENAME:
          print "Enter filename to save patterns to [<$LOGDIR/patterns>]: ";
          $ans = &get_response;
          chomp $ans;
          $ans = "$LOGDIR/patterns" if ($ans eq '');
          if (-e $ans) {
PAT_FILENAME2:
              print "\nFile already exists. Overwrite, Append, Change Name or Quit [<o>, a, c, q]: ";
              $ans2 = &get_response;
              goto PAT_FILENAME if ($ans =~ /^c/i);
              goto PAT_NEWFILE if (($ans2 =~ /^o/i) or ($ans2 eq ''));
              if ($ans2 =~ /^a/i) {
                  # Read existing file first and import all the valid patterns
                  # Then write as a new file
                  if (open PATTERNS, "<$ans") {
                      $p_count = 0;
                      while ($i = <PATTERNS>) {
                          $p_count ++;
                          chomp $i;
                          $j = &input_pattern($i); # verify valid pattern
                          if ($j) {
                              push @temp, $j;
                          } else {
                              print "Skipping illegal pattern: $j\n";
                          }
                      }
                      close PATTERNS;
                      goto PAT_NEWFILE;
                  } else {
                      print "\nFailed to open file $ans for reading\n";
                      goto PAT_MENU;
                  }
              } elsif ($ans2 =~ /^q/i) {
                  return '';
              } else { goto PAT_FILENAME2; }

          } else {
PAT_NEWFILE:
              # This is a new file, save the patterns
              %seen = ();
              @used_patterns = ();
              @used_patterns = grep { ! $seen{$_} ++ } @temp; #de-dupe
              
              if (open PATTERNS, ">$ans") {
                  $p_count = 0;
                  foreach $i (@used_patterns) {
                      $p_count ++;
                      print PATTERNS "$i\n";
                  }
                  print "\n$p_count patterns total.\n";
                  close PATTERNS;
                  return $ans;
              } else {
                  print "\nFailed to open file $ans for writing\n";
                  goto PAT_MENU;
              }
          }
      } else {
          goto PAT_MENU;
      }
      return $ans;
  } elsif ($pattern_choice eq '3') {
      # Read patterns from file
      # no de-duping
      print "Enter filename to read patterns from [<$LOGDIR/patterns>]: ";
      $ans = &get_response;
      chomp $ans;
      $ans = "$LOGDIR/patterns" if ($ans eq '');

      if (open PATTERNS, "<$ans") {
          $p_count = 0;
          print "\nPatterns\n";
          print "--------\n\n";
          while ($i = <PATTERNS>) {
              $p_count ++;
              chomp $i;
              $j = &input_pattern($i); # verify valid pattern
              if ($j eq '') {
                  # found an illegal pattern, reject file
                  print "\nIllegal pattern \'$i\' detected on line $p_count.\n";
                  goto PAT_MENU;
              } else {
                  print "$j\n";
              }
          }
          # file is good, patterns are valid
          close PATTERNS;
          print "\n$p_count patterns total.\n";
          return $ans;
      } else { 
          print "\nFile $ans Not Found\n";
          goto PAT_MENU;
      }
  } elsif ($pattern_choice eq '4') {
      # Input patterns directly by user
      print "Enter Patterns\n";
      print "--------------\n";
      print "\nAll patterns are hexadecimal and up to 8 characters in length\n";
      print "Example: 7effdd32\n";
      print "Enter a hexadecimal value or q for 'quit'.\n";
      $p_count = 1;
      while (1) {
          print "$p_count) ";
          $ans = &get_response;
          chomp $ans;
          last if (($ans =~ /^q/i) or ($ans eq ''));
          $i = input_pattern($ans);
          if ($i) {
              push @temp,$i;
              $p_count ++;
          } else {
              print "Illegal pattern.\n";
          }
      }
      print "\n";
      %seen = ();
      @used_patterns = ();
      @used_patterns = grep { ! $seen{$_} ++ } @temp; #de-dupe
      $p_count = 0;
      foreach $i (@used_patterns) {
          $p_count ++;
          print "$i\n";
      }
      print "\n$p_count patterns entered.\n";
      goto PAT_MENU if (! $p_count);
      print "\nUse These Patterns [<y>, n, q]: ";
      $ans = &get_response;
      return '' if ($ans =~ /^q/i);
      goto PAT_FILENAME if (($ans eq '') or ($ans =~ /^y/i));
      goto PAT_MENU;
  } else {
      printf("$pattern_choice is an invalid choice, Please try again\n");
      $pattern_choice = '';
      goto PAT_MENU;
  }
}

sub input_pattern {
  # This subroutine prompts for a hexadecimal test pattern, if none was passed
  # into it, and verifies it.
  # Even numbers of hex characters are allowed up to 8. Rejects return null.

  my $inp = shift;
  $inp = &get_response if ($inp eq '');
  chomp $inp;
  if ($inp =~ /^([a-fA-F0-9])+$/) {
      # User entered all hex characters
      if (length($inp)%2) {
          # odd number of hex characters entered is illegal
          return '';
      } else {
          # even number of hex characters entered is ok, pad to 8 characters
          $inp = $inp x 4;
          $inp = substr $inp, 0, 8;
          return $inp;
      }
  } else {
      return '';
  }
}


=head1 NAME

 get_test_list()

=head1 SYNOPSIS

 B<get_test_list>

=head1 DESCRIPTION

Get a list of devices to test.
list is a comma or space seperated list of numbers that have
valid indexes in @LD_XX_PORTS. 

=cut

sub get_test_list {

    my $test = $_[0];
    # test = "dex", "lbf" or null ("dex" is the default and is equ to null) 

    my $lut;
    my @test_lp_list;
    my $error;
    my $num_ports;
    my @test_ports;

    if ( $STORAGE eq "A5" ) {
        $num_ports = $#LD_A5_PORTS + 1;
        @test_ports = @LD_A5_PORTS;
    } elsif ( $STORAGE eq "T3" ) {
        $num_ports = $#LD_T3_PORTS + 1;
        @test_ports = @LD_T3_PORTS;
    } 
    if ( @LD_MB_PORTS ) {
        $num_ports += $#LD_MB_PORTS + 1;
        push @test_ports, @LD_MB_PORTS;
    }
    if ( $test eq "lbf" ) {
        if ( @LD_STE_PORTS ) {
            $num_ports += $#LD_STE_PORTS + 1;
            push @test_ports, @LD_STE_PORTS;
        }
        if ( @LD_ND_PORTS ) {
            $num_ports += $#LD_ND_PORTS + 1;
            push @test_ports, @LD_ND_PORTS;
        }
    }

    if ($num_ports < 1) {
        print "get_test_list: Programming Error. Invalid Loop Count.\n";
        exit;
    }

    while(1) {
        $lut = "";
        $error = 0;
        &print_test_loops( @test_ports );
        print "\n";
        printf("Enter comma or space delimited list of loops, [a,#,<q>]: ");
        $choice = get_response();

        if($choice eq "" || $choice =~ /^q/i)  {
            return;
        }

        if ($choice eq "a") {
            @test_lp_list = @test_ports;
            $lut = "All";
            last;
         }
                
         if (! ($choice =~ /^\s*\d+/) ) {
             printf("$choice is an Invalid list, Please try again\n");
             next;	
         }
         if (grep {/,/} $choice) {                       # delimiter is , or
             @lpchoice = split(',', $choice);            #  blank
         } else {
             @lpchoice = split(' ', $choice);
         }
         foreach $tmp (@lpchoice) {
             ($t1) = $tmp =~ /\s*(\d+)/;
             if (($t1 >=1) && ($t1 <= $num_ports)) {
                 $t2 = $t1 - 1;
                 if(! defined($test_ports[${t2}]) ) {
                     printf("Bad device number $t1\n");
                     next;
                 } else {
                     if($lut eq "" ) {
                         $lut = "$t1";
                     } else {
                         $lut .= ", $t1";
                     }
                     push @test_lp_list, $test_ports[($t2)];
                 }
             } else {
                 print "$t1 is an Invalid entry.  Please try again.  \n";
                 undef @test_lp_list;
                 $error = 1;
             }
         }
         if ($error) {
             next;
         } else {
             last;
         }
    }

    @TEST_LP_LIST = @test_lp_list;
    $LUT = $lut;
}

=head1 NAME

 print_test_loop()

=head1 SYNOPSIS

 B<print_test_loop>

=head1 DESCRIPTION

Print out the list of loops available for diagnostic testing.

=cut
####################################################################
# print test loop menu
####################################################################
sub print_test_loops {

    my @test_ports = @_;

    my $num_ports;

    if ( $num_ports < 0 ) {
        print "\n\nNo Online ports.\n\n";
        print "\npress <return> to continue ... ";
        &get_response;
        exit(1);
    }
    $port_cnt = 1;
    print "\n";
    print ("Loops Available for Testing\n");
    print ("---------------------------\n");
    foreach $port (@test_ports) {
        #$ddcnt = &get_ddcnt( $port );
        print "$port_cnt. ($CNUM{$port}) ($SFNUM{$port}) $port ";
        if ( $CNUM{$port} =~  /c\d+/ ) {
            print "($DDCNT{ $port} ${LUN_DISK}(s))";
            if ( $STORAGE eq "T3" ) {
                print ": $P_DDCNT{ $port } Pri, $A_DDCNT{ $port } Alt)";
            }
        } elsif ( $CNUM{$port} =~  /s\d+/ ) {
            print "STE or Loopback\n";
        } elsif ( $CNUM{$port} =~  /n\d+/ ) {
            print "A5K ses w/o drives  or Loopback\n";
        }
        print "\n";
        $port_cnt++;
    }
}

=head1 NAME

 mod_lbf_params()

=head1 SYNOPSIS

 B<mod_lbf_params>

=head1 DESCRIPTION

Menu for modifying lbf parameters.

=cut
####################################################################
# modify lbf parameters menu
####################################################################
sub mod_lbfparms {
    my ($exit_loop, $choice);
    my $found;
    print "\n\n";
    while (! $exit_loop) {
        print "\nLBF Parameter Selection Menu\n";
        print "----------------------------\n\n";
        print "Options                                Value\n";
        print "===========================            ====================\n";
        print "[c] Compare Flag                       $LBF_COMPARE_F\n";
        print "[k] Transfer Size                      ${LBF_XFR_SZ}\n";
        print "[n] Number of Passes                   ${LBF_PASSES}\n";
        #print "[4] lbf or plbf                        ${WHICH_LBF}\n";
        #print "[5] plbf Number of Processes           $PLBF_PROCS\n";
        #print "[l] Logfile Directory                  ${LBF_LOGDIR}\n";
        #print "[7] lbf Pattern Flag                   ${LBF_PATTERN_F}\n";
        print "\n";
        print "[d] Restore Default Values\n";
        print "[q] Quit \n";
        print "\nSelect an item to edit: ";
        $choice = &get_response;
        return if $choice =~ /^q/i;
        if ($choice =~ /^q/i) {
#	    print "****** get out of here******** \n";
	    return;
	} elsif ($choice eq "d") {
            $LBF_PASSES = $DEFAULT_LBF_PASSES;
            $LBF_XFR_SZ = $DEFAULT_LBF_XFR_SZ;
            $LBF_COMPARE_F = $DEFAULT_LBF_COMPARE_F;
	} elsif ($choice eq "n") {
	    print "Change lbf passes: (${LBF_PASSES})\n";
	    print "Enter a new value (1-20000): ";
	    $choice = &get_response;
	    if (($choice > 0) && ($choice <= 20000)) {
		$LBF_PASSES = $choice;
	    } else {
		print "\tIllegal value.  Please try again.\n";
	    }
	} elsif ($choice eq "k") {
	    print "Change lbf transfer size:  (${LBF_XFR_SZ})\n";
	    print "Enter a new value (1-64) (in kilobytes):";
	    $choice = &get_response;
	    if (($choice > 0) && ($choice <= 64)) {
		$LBF_XFR_SZ = $choice;
	    } else {
		print "\tIllegal value.  Please try again. \n";
	    }
	} elsif ($choice eq "c") {
	    if ($LBF_COMPARE_F eq "ON")
               {
               $LBF_COMPARE_F = "OFF";
               }
            else
               {
               $LBF_COMPARE_F = "ON";
               }
	} elsif ($choice eq "4") {
	    print "Choose lbf or plbf:    (${WHICH_LBF}) \n";
	    print "Enter choice (<p> or l); ";
	    $choice = &get_response;
	    if ($choice =~ /^[pl]/i) {
		if ($choice =~ /^p/i) {
		    if ($found = `/bin/which plbf`) {
			chomp($found);
			$WHICH_LBF = $found;
		    } else {
			if (-e ".\/plbf") {
			    $WHICH_LBF = ".\/plbf";
			    next;
			} elsif (-e "\/var\/opt\/${BASEDIR}\/bin\/plbf") {
			    $WHICH_LBF = "\/var\/opt\/${BASEDIR}\/bin\/plbf";
			    next;
 			} elsif (-e "/home/tel/tasks/plbf")  {
			    $WHICH_LBF = "\/home\/tel\/tasks\/plbf";
			    next;
			} else {
			    print "Could not find plbf.  Good bye!!  \n";
			    return;
			    print "\n\tplbf not found.  Enter a directory or quit:  ";
			    $choice = &get_response;
			    if (-e "${choice}/plbf") {
				$WHICH_LBF = $choice;
			    } else {
				print "Not a valid choice\n";
			    }
			    next;
			}
		    }
		} else {
		    if ($found = `/bin/which lbf`) {
			chomp($found);
			$WHICH_LBF = $found;
		    } else {
			if (-e "/var/opt/${BASEDIR}/bin/lbf") {
			    $WHICH_LBF = "/var/opt/${BASEDIR}/bin/lbf";
			    next;
			} elsif (-e "/home/tel/tasks/lbf") {
			    $WHICH_LBF = "/home/tel/tasks/lbf";
			    next;
			} else {
			    print "Could not find lbf.  Good bye!! \n";
			    return;
			}
		    }
		}
	    } else {
		print "Illegal value.  Please try again. \n";
	    }
	} elsif ($choice eq "5") {
	    print "Choose number of plbf processes:  ($PLBF_PROCS) \n";
	    print "  (plbf only) \n";
	    print "Enter new value (1-8):  ";
	    $choice = &get_response;
	    if ($choice =~ /^[1-8]/) {
		$PLBF_PROCS = $choice;
	    } else {
		print "Illegal value.  Please try again. \n";
	    }
	} elsif ($choice eq "6") {
	    print "Change lbf logfile directory:  ($LBF_LOGDIR) \n";
	    print "Enter new directory:  ";
	    $choice = &get_response;
	    if ((-d $choice) && (-w $choice)) {
		$LBF_LOGDIR = $choice;
	    } else {
		print "Illegal value.  Please try again. \n";
	    }
	} 
#elsif ($choice eq "7") {
#	    print "Change lbf pattern flag:  ($LBF_PATTERN_F) \n";
#	    print "Enter 'USER' or 'DEFAULT': ";
#	    $choice = &get_response;
#	    if (($choice eq "USER") | ($choice eq "DEFAULT")) {
#		$LBF_PATTERN_F = $choice;
#	    } else {
#	print "Illegal value.  Please try again. \n";
#    }
#} 
        else {
	    print "Illegal value.  Please try again. \n";
	}
    }
}

=head1 NAME

 dex_input_pattern()

=head1 SYNOPSIS

 B<dex_input_pattern>

=head1 DESCRIPTION

Prompts for a hexadecimal pattern. If a pattern was passed to it then do not
prompt for a pattern, just verify it and expand or truncate it to 8 characters.

=cut

sub dex_input_pattern {
############################################################################
# Purpose: Prompts for a hexadecimal test pattern, if a pattern was passed #
#          to it then do not prompt for a pattern.                         #
#          This pattern is then verified and expanded to 8 characters if   #
#          necessary (or truncated to 8 characters).                       #
# Input:   Optional pattern to verify or "P{num}" P0 - P8 for dex's        #
#          internal list of standard patterns.                             #
# Output:  Returns three possibilities: a good pattern prepended by "0x",  #
#          or P0 - P8 or returns an empty string if user enters nothing.   # 
############################################################################

  my $inp = shift;
  $inp = &get_response if ($inp eq '');
  chomp $inp;
  $inp =~ s/0x//;
  if ($inp =~ /^([a-fA-F0-9])+$/) {
      # User entered all hex characters
      $inp = "0" . $inp if (length($inp)%2); # fixing odd number of hex chars
      # pad to 8 characters
      $inp = $inp x 4;
      $inp = substr $inp, 0, 8;
      return "0x$inp";
  } elsif ($inp =~ /^P(\d+)$/) {
      # user asked for standard Dex internal pattern. P0 - P8
      return $inp;
  } else {
      return '';
  }
}

sub dexoptions {
##################################################################
# Version: 1.0 by Herb Rubin, May 1999                           #
# Purpose: To generate the dex command line arguments through a  #
#          menu system.                                          #
# Input:   4 strings: Two strings and 2 optional y/n parameters. #
#          1) a string of dex options to be the defaults.        #
#          2) the current options string manipulated by the user #
#          3) y/n to turn on/off changing the test type          #
#          4) y/n to turn on/off changing the test mode (rw)     #
# Output:  The final string of dex options.                      #
#                                                                #
##################################################################

my $default_cli = shift; # default options
my $current_cli = shift; # current options
my $no_change_rw = shift || "0"; # optional parameter
my $no_change_test = shift || "0"; # optional parameter

my ($Verbose, $Test, $Per_Pass, $Range, $Iterations, $Time, $Xfer, $RW, $IOdelay);
my ($PassDelay, $StartDelay, $ProcPerDevice, $dev, $answer, $i, $j);
my ($OpenClose, $InitFirst, $ErrorLimit, $CompareErrorLimit, $TargetDevice);
my ($TestPattern, $cli);
my @DEX_TESTS = (
        "S",  "Sequential",
        "R",  "Random",
        "SR", "Sequential Reverse",
        "M",  "Memory Map Sequential",
        "MR", "Memory Map Sequential Reverse",
        "B",  "Burst Random",
        "L",  "Low-Power Seek",
        "LB", "Low-Power Butterfly",
        "LR", "Low-Power Reverse Butterfly",
        "A",  "Actuator",
        "AB", "Actuator Butterfly",
        "AR", "Actuator Reverse Butterfly",
        "LS", "Low-Power Sequential",
        "C",  "Cluster (database activity)",
        );
my @MODE = (
	"r",  "Read Only",
	"w",  "Write Only",
	"wr", "Write and Read",
	"w/r", "Write then Read"
	);
$no_change_rw = 0 if ($no_change_rw =~ /^n/i);
$no_change_test = 0 if ($no_change_test =~ /^n/i);

RESTORE_DEFAULTS:
 if ($current_cli ne '') {
     # use current dex parameters if they exist
     $cli = $current_cli;  
 } else {
     $cli = $default_cli;
 }
 $Test = '';
 $Per_Pass = '';
 $Iterations = '';
 $Range = '';
 $Xfer = '';
 $ErrorLimit = '';
 $CompareErrorLimit = '';
 $RW = '';
 $ProcPerDevice = '';
 $OpenClose = '';
 $InitFirst = '';
 $TestPattern = '';

 my @cli_args = split /\s+/, $cli;
 my ($arg1, $arg2, $arg3, $arg4);
 ############################################
 # parse input string and setup all default #
 # values                                   #
 ############################################
 for ($i = 0; $i <= $#cli_args; $i++) { 
     $arg1 = '';
     $arg2 = '';
     $arg3 = '';
     $arg4 = '';
     if ($cli_args[$i] =~ /^-(.*)/) {
         # found next switch, gather its values after it
         $arg1 = $1;
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg2 = $cli_args[$i];
             if ($arg2 =~ /^\//) {
                 $TargetDevice = $arg2;
                 $arg2 = '';
             }
         }
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg3 = $cli_args[$i];
             if ($arg3 =~ /^\//) {
                 $TargetDevice = $arg3;
                 $arg3 = '';
             }
         }
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg4 = $cli_args[$i];
             if ($arg4 =~ /^\//) {
                 $TargetDevice = $arg4;
                 $arg4 = '';
             }
         }
     } elsif ($cli_args[$i] =~ /^\//) {
        ######################################
        # Found device list, save for later. #
        # Return list unmodified.            #
        ######################################
        $TargetDevice .= " $cli_args[$i]";
        next;
     }
     if ($arg1 eq "e") {
         # setting default error limits
         $ErrorLimit = $arg2 if $arg2;
         $CompareErrorLimit = $arg3 if $arg3;
     } elsif ($arg1 =~ /^P$/) {
         # Test pattern
         $TestPattern = $arg2 if $arg2;
     } elsif ($arg1 =~ /^P(\d+)$/) {
         # Standard Test Pattern chosen #
         $TestPattern = $arg1;
     } elsif ($arg1 eq uc($arg1)) {
         # Test type selection
         if ($arg1 =~ /^(\w+)(\d+)$/) {
             $Test = $1; 
             $Per_Pass = $2;
         } elsif ($arg1 =~ /^(\w+)$/) {
             $Test = $1;
         }
         if ($Test =~ /I/) {
             $InitFirst = "yes";
             $Test =~ s/I//;
         }
         if (length($Test) > 1) {
             # check for C flag
             if ($Test =~ /(.*)C$/) {
                 $OpenClose = "yes";
                 $Test = $1;
             }
         }
         # figure out what number is associated with $Test
         for ($j = 0; $j < $#DEX_TESTS; $j = $j +2) {
            $Test = $j if ($Test eq $DEX_TESTS[$j]);
         }
         $Range = $arg2;
         $Iterations = $arg3;
         $Time = $arg4;
     } elsif ($arg1 eq "p") {
         #setting default processes per device
         $ProcPerDevice = $arg2 if $arg2;
     } elsif ($arg1 eq "r") {
         # setting default mode to Read Only
         $RW = "0";
     } elsif (($arg1 eq "rw") or ($arg1 eq "wr")) {
         # Write And Read mode #
         $RW = "4";
     } elsif (($arg1 eq "r\/w") or ($arg1 eq "w\/r")) {
         # Write Then Read mode #
         $RW = "6";
     } elsif ($arg1 eq "s") {
         # setting default delays
         $IOdelay = $arg2 if $arg2;
         $PassDelay = $arg3 if $arg3;
         $StartDelay = $arg4 if $arg4;
     } elsif ($arg1 eq "v") {
         $Verbose = "-v";
     } elsif ($arg1 eq "w") {
         $RW = "2";
     } elsif ($arg1 eq "x") {
         # setting default for transfer size
         $Xfer = $arg2 if $arg2;
     } else {
         #################################################
         # Found arguments we don't modify, return them  #
         # later intact.                                 #
         #################################################
         $Extra_args .= "-$arg1"; 
         $Extra_args .= " $arg2" if $arg2;
         $Extra_args .= " $arg3" if $arg3;
         $Extra_args .= " $arg4" if $arg4;
     }
 }
 #################################################
 # setup remaining options to resonable defaults #
 #################################################
 $Test = 0 if ($Test eq '');
 $Per_Pass = "8" if ($Per_Pass eq '');
 $Iterations = "0" if ($Iterations eq '');
 $Range = "8g" if ($Range eq '');
 $Xfer = "128k" if ($Xfer eq '');
 $ErrorLimit = "0" if ($ErrorLimit eq '');
 $CompareErrorLimit = "0" if ($CompareErrorLimit eq '');
 $RW = "0" if ($RW eq '');
 $ProcPerDevice = "1" if ($ProcPerDevice eq '');
 $OpenClose = "no" if ($OpenClose eq '');
 $InitFirst = "no" if ($InitFirst eq '');

 #############
 # main loop #
 #############

 while (1) {
    print "\n\n";
    print "Disk Exerciser Options                 Value\n";
    print "============================           ====================\n";
    if ($no_change_test) {
        print "    Test Type (not selectable)         $DEX_TESTS[$Test +1]\n";
    } else {
        print "[t] Select Test Type                   $DEX_TESTS[$Test +1]\n";
    }
    if ($no_change_rw) {
        print "    Mode of Test (not selectable)      $MODE[$RW + 1]\n";
    } else {
        print "[m] Select Mode of Test                $MODE[$RW + 1]\n";
    }
    print  "[a] Select Range Across Media          $Range\n";
    print  "[s] Select Data Transfer Size          $Xfer\n";
    print  "[l] Select Time Limit                  ";
    if ($Time eq '') {
        print "none\n";
    } else {
        print "$Time\n";
    }

    print "\n[n] Select Number of I/O's per Pass    $Per_Pass\n";
    print "[o] Select Open/Close On Every Pass    $OpenClose\n";
    print "[i] Select Iterations (Passes)         ";
    if ($Iterations ne "0") {
        print "$Iterations\n";
    } else {
        print "unlimited\n";
    }
    print  "[p] Select Processes per Device        $ProcPerDevice\n";
    print  "[d] Select IO, Pass & Start Delays     ";
    if ($IOdelay) {
        print "I/O: $IOdelay, ";
    } else {
        print "I/O: 0, ";
    }
    if ($PassDelay) {
        print "Pass: $PassDelay, ";
    } else {
        print "Pass: 0, ";
    }
    if ($StartDelay) {
        print "Start: $StartDelay\n";
    } else {
        print "Start: 0\n";
    }
    print  "[e] Select Error Limit                 ";
    if ($ErrorLimit eq '0' ) {
        print "none\n";
    } else {
        print "$ErrorLimit\n";
    }
    print  "[c] Select Compare Error Limit         ";
    if ($CompareErrorLimit eq '0') {
        print "none\n";
    } else {
        print "$CompareErrorLimit\n";
    }
    print  "[f] Select Initialize Device First     $InitFirst\n";
    print  "[b] Select Buffer Test Pattern         ";
    if ($TestPattern) {
        print "$TestPattern\n";
    } else {
        print "default\n";
    }
    print  "[v] Verbose Mode                       ";
  
    if ($Verbose) {
        print "On\n";
    } else {
        print "Off\n";
    }

    print "\n[u] Undo, Restore Default Settings\n";
    print "[?] Help\n";
    print "\n[q] Quit\n\n";

    print "Enter Selection: ";
    $choice = &get_response;                            # user response
    $choice =~ s/\s+//g;    # remove leading and trailing white space
   
    if ($choice =~ /^t/i) {
        next if ($no_change_test);
        if (length($choice) > 1) {
            # User specified a specific test
            # Find its numerical index number in list 
            $choice =~ s/t//;
            for ($i = 0; $i < $#DEX_TESTS; $i = $i +2) {
                $Test = $i if ($choice eq $DEX_TESTS[$i]);
            }
        } else {
            # select the next test in the list
            $Test = $Test + 2;
            if ($Test > $#DEX_TESTS) { $Test = 0; }
        }
    } elsif ($choice =~ /^\?/) {
        system("/usr/bin/ksh -c '/bin/man -M $MANDIR dex 2>&1 | more ");
        print "\nPlease press <return> to continue ... ";
        &get_response;

    } elsif ($choice =~ /^b/i) {
        #######################
        # Select Test Pattern #
        #######################
        if (length($choice) > 1) {
            $choice =~ s/b//;
            $answer =~ s/\s+//g;
            $answer =~ s/0x//;
            $answer = &dex_input_pattern($choice);
        } else {
            print "Enter Hexadecimal Test Pattern (8 characters): ";
            $answer = &dex_input_pattern;
        }
        if ($answer) {
            $TestPattern = "$answer";
        } else {
            print "Test Pattern Unchanged.\n";
        }
    } elsif ($choice =~ /^c/i) {
      
        if (length($choice) > 1) {
            $choice =~ s/c//;
            $answer = $choice;
        } else {
            print "How many compare errors will equal one error (0=no limit):";
            $answer = &get_response;
        }
        $answer =~ s/\s+//g;
        if (($answer =~ /\d+/) && ($answer >= 0) ) {
            $CompareErrorLimit = $answer;
        } else {
            print "Compare Error Limit must be a positive number. Compare Error Limit Unchanged\n";
        }
    } elsif ($choice =~ /^e/i) {
        ######################################
        # Error limit before we stop testing #
        ######################################
        if (length($choice) > 1) {
            $choice =~ s/e//;
            $answer = $choice;
        } else {
            print "How many errors maximum until we stop the testing (0=no limit):";
            $answer = &get_response;
        }
        $answer =~ s/\s+//g;
        if (($answer =~ /\d+/) && ($answer >= 0) ) {
            $ErrorLimit = $answer;
        } else {
            print "Error Limit must be a positive number. Error Limit Unchanged\n";
        }
    } elsif ($choice =~ /^f/i) {
        ############################
        # Initialize First, toggle #
        ############################
        if ($InitFirst eq "no") {
            $InitFirst = "yes";
        } else {
            $InitFirst = "no";
        }
    } elsif ($choice =~ /^q/i) {
        ######################################################################
        # Run Test. Build the final command line argument list and return it.#
        # This subroutine doesn't run the test directly, the caller does.    #
        ######################################################################
 
        my $arglist;
        $arglist  = "$Verbose " if $Verbose;
        $arglist .= "-$DEX_TESTS[$Test]";
        $arglist .= "C" if ($OpenClose eq "yes");
        $arglist .= "I" if ($InitFirst eq "yes");
        $arglist .= "$Per_Pass $Range $Iterations";
        $arglist .= " $Time" if $Time;
        $arglist .= " -x $Xfer";
        $arglist .= " -$MODE[$RW]";
        $arglist .= " -e $ErrorLimit";
        $arglist .= " $CompareErrorLimit" if ($CompareErrorLimit);
        if (($IOdelay) or ($PassDelay) or ($StartDelay)) {
            $arglist .= " -s ";
            if ($IOdelay) {
               $arglist .= "$IOdelay ";
            } else {
               $arglist .= "0 ";
            }
            if ($PassDelay) {
               $arglist .= "$PassDelay ";
            } else {
               $arglist .= "0 ";
            }
            if ($StartDelay) {
               $arglist .= "$StartDelay";
            } else {
               $arglist .= "0";
            }
        }
        $arglist .= " -p $ProcPerDevice" if ($ProcPerDevice ne "1");
        $arglist .= " $Extra_args" if $Extra_args;
        if ($TestPattern) {
            if ($TestPattern !~ /^P/) {
                # User specified test pattern
                $arglist .= " -P $TestPattern";
            } else {
                # User chose standard pattern
                $arglist .= " -$TestPattern";
            }
        }
        $arglist .= " $TargetDevice" if $TargetDevice;
        return $arglist;

    } elsif ($choice =~ /^h/i) {
        ########
        # help #
        ########
        print "\nDex Help\n";
        print "This program issues I/O's to a device in passes (iterations). It can exercise\n";
        print "disk drives, tape drives and file systems.\n";
        print "\nTest Type List\n\n";
        for ($i = 0; $i < $#DEX_TESTS; $i = $i +2) {
            print "$DEX_TESTS[$i]\t$DEX_TESTS[$i + 1]\n";
        }
        print "\nModes of Operation\n\n";
        for ($i = 0; $i < $#MODE; $i = $i +2) {
            print "$MODE[$i]\t$MODE[$i + 1]\n";
        }
        print "\nYou may enter letters at the prompt such as 't SR' or 'm w' \n";
        print "otherwise the next choice is selected from the list.\n";
        print "\nPress ENTER to continue ";
        $answer = &get_response;
    } elsif ($choice =~ /^i/i) {
        #######################################
        # Select Iterations, zero is infinite #
        #######################################
        print "Enter number of iterations (0=unlimited): ";
        $answer = &get_response;
        if ($answer =~ /^\d+$/) {
            $Iterations = $answer;
        } else {
            print 'Only positive integers are allowed, Iterations Unchanged.';
        }
    } elsif ($choice =~ /^l/i) {
        #####################
        # Select Time Limit #
        #####################
        if (length($choice) > 1) {
            $choice =~ s/l//;
            $answer = $choice;
        } else {
            print "Enter Time Limit (h=hours, m=minutes, s=seconds): ";
            $answer = &get_response;
        }
        $answer =~ s/\s+//g;

        if ($answer =~ /^(\d+)[hms]$/) {
            if ($1 eq '0') {
                $Time = '';
            } else {
                $Time = $answer;
            }
        } elsif ($answer eq '0') {
            $Time = '';
        } else {
            print "Time Limit Unchanged\n";
        }
    } elsif ($choice =~ /^m/i) {
        next if ($no_change_rw);
        # select the next mode in the circular list
        if (length($choice) > 1) {
            # User specified a specific mode
            # Find its numerical index number in list
            $choice =~ s/m//;
            for ($i = 0; $i < $#MODE; $i = $i +2) {
                $RW = $i if ($choice eq $MODE[$i]);
            }
        } else {
            # increment to the next mode (circularly)
            $RW = $RW + 2;
            $RW = 0 if ($RW > $#MODE);
        }
    } elsif ($choice =~ /^n/i) {
        ###################################
        # Select number of I/O's per Pass #
        ###################################
        if (length($choice) > 1) {
            $choice =~ s/n//;
            $answer = $choice;
        } else {
            print "Enter Number of I/O's per Pass: ";
            $answer = &get_response;
        }
        $answer =~ s/\s+//g;
        if ($answer !~ /^\d+$/) {
            print "Number of I/O's must be a positive number. I/O's per Pass Unchanged.\n";
        } else {
            $Per_Pass = $answer;
        }
    } elsif ($choice =~ /^o/i) {
        # toggle Open/Close on every pass y/n #
        if ($OpenClose eq "no") {
            $OpenClose = "yes";
        } else {
            $OpenClose = "yes";
        }
    } elsif ($choice =~ /^p/i) {
        #############################################
        # Select number of Dex processes per device #
        #############################################
        if (length($choice) > 1) {
            $choice =~ s/p//;
            $answer = $choice;
        } else {
            print "Enter Number of processes per device (1=minumum):";
            $answer = &get_response;
        }
        $answer =~ s/\s+//g;
        if (($answer =~ /^\d+$/) && ($answer >= 1) ) {
            $ProcPerDevice = $answer;
        } else {
            print "Number of processes must be a positive number. Processes Per Device Unchanged\n";
        }
    } elsif ($choice =~ /^d/i) {
        #############################
        # Select 3 Different Delays #
        #############################
        print "Select IO Delay (h=hours, m=minutes, s=seconds, u=microseconds): ";
        $answer = &get_response;
        if ($answer =~ /^(\d+)[hmsu]$/) {
            if ($1 > 0) {
                $IOdelay = $answer;
            } elsif ($1 eq '0') {
                $IOdelay = '';
            } else {
                print "Delay must be a positive number. I/O Delay Unchanged.\n";
            }
        } else {
            if ($answer eq '0') {
                $IOdelay = '';
            } else {
                print "Incorrect format. " if ($answer);
                print "I/O Delay Unchanged.\n";
            }
        }
        print "Select Pass Delay (h=hours, m=minutes, s=seconds, u=microseconds): ";
        $answer = &get_response;
        if ($answer =~ /^(\d+)[hmsu]$/) {
            if ($1 >= 0) {
                $PassDelay = $answer;
            } elsif ($1 eq '0') {
                $PassDelay = '';
            } else {
                print "Delay must be a positive number. Pass Delay Unchanged.\n";
            }
        } else {
            if ($answer eq '0') {
                $PassDelay = '';
            } else {
                print "Incorrect format. " if ($answer);
                print "Pass Delay Unchanged.\n";
            }
        }

        print "Select Process Start Delay (h=hours, m=minutes, s=seconds, u=microseconds): ";
        $answer = &get_response;
        if ($answer =~ /^(\d+)[hmsu]$/) {
            if ($1 >= 0) {
                $StartDelay = $answer;
            } elsif ($1 eq '0') {
                $StartDelay = '';
            } else {
                print "Delay must be a positive number. Start Delay Unchanged.\n";
            }
        } else {
            if ($answer eq '0') {
                $StartDelay = '';
            } else {
                print "Incorrect format. " if ($answer);
                print "Start Delay Unchanged.\n";
            }
        }

    } elsif ($choice =~ /^a/i) {
        ############################
        # Select Range Across Disk #
        ############################
        if (length($choice) > 1) {
            $choice =~ s/r//;
            $answer = $choice;
        } else {
            print "Enter Range (g=gigabytes, m=megabytes, k=kilobytes, b=bytes, %=percent): ";
            $answer = &get_response;
            $answer =~ s/\s+//g;    # remove leading & trailing white space
        }
        if ($answer =~ /^(\d+)([gmkb\%])$/) {
            if (($2 eq "%") and ($1 > 100)) {
                print "Percent cannot be greater than 100. Range Unchanged\n";
            } else {
                $Range = $answer;
            }
        } else {
            print "Range Unchanged\n";
        }
    } elsif ($choice =~ /^s/i) {
        ############################
        # Select I/O Transfer Size #
        ############################
        if (length($choice) > 1) {
            $choice =~ s/s//;
            $answer = $choice;
        } else {
            print "Enter Transfer Size (g=giga, m=mega, k=kilo, b=bytes): ";
            $answer = &get_response;
            $answer =~ s/\s+//g;
        }
        if ($answer =~ /^\d+[gmkb]$/) {
            $Xfer = $answer;
        } else {
            print "Transfer Size Unchanged\n";
        }
    } elsif ($choice =~ /^v/i) {
        #######################
        # Toggle Verbose Mode #
        #######################
        if ($Verbose) {
            $Verbose = '';
        } else {
            $Verbose = "-v";
        }
    } elsif ($choice =~ /^u/i) {
        ############################
        # Restore Default Settings #
        ############################
        $current_cli = $default_cli if $current_cli;
        goto RESTORE_DEFAULTS;
    }
 }
}

sub storstat_options {
#######################################################################
# Version: 1.0 by Herb Rubin, May 1999                                #
# Purpose: To generate the storstat command line arguments            #
#          through a menu system.                                     #
# Input:   A string of storstat options to be used as the defaults.   #
#          You may pass in 2 sets of options, defaults & current      #
#          options.                                                   #
# Output:  The final string of storstat options.                      #
#                                                                     #
#######################################################################

my $default_cli = shift; # default options
my $current_cli = shift; # current options
my ($args, $Disk_Test, $Fan_Test, $PS_Test, $FRU_Test, $Driver_Test, $FW_Test, $Min_Test, $Patch_Test, $Verbose, $All_Test, $Warn_Flag, $Log_Flag, $Mailto, $Show_Dots);
my ($VRTS_Test, $others);
my $test_choices;
my $wait4return;

my $restore_default = "yes";

   if ($STORAGE eq "A5") {
      $test_choices = "'a', 'd', 'f', 'p', 'u', 'w', 'm', 'c', 'e'";
   } elsif ($STORAGE eq "T3") {
      $test_choices = "'a','c','e','w'";
      # Unsupported options
      $Disk_Test = 'no'; 
      $Fan_Test = 'no'; 
      $PS_Test = 'no'; 
      $FRU_Test = 'no'; 
      $Min_Test = 'no'; 
   } else {
      print "ERROR: Programming Error: Unknown Storage Unit.\n";
   }

chomp $default_cli;
chomp $current_cli;

######################################
# process cli and set internal flags #
######################################

 if ($current_cli ne '') {
     ###########################################################
     # Two sets of storstat cli options are available          #
     # so, use 2nd set as default options until user chooses   #
     # restore defaults.                                       #
     ###########################################################

     $cli = $current_cli;
 } else {
     ###########################################################
     # Only a single set of storstat cli options are available #
     # so, use them as default and current options             #
     ###########################################################

     $cli = $default_cli;
 }
RESTORE_DEFAULTS:

 my @cli_args = split /\s+/, $cli;
 $All_Test = '';
 $Disk_Test = '';
 $Fan_Test = '';
 $PS_Test = '';
 $FRU_Test = '';
 $Driver_Test = '';
 $FW_Test = '';
 $Min_Test = '';
 $Patch_Test = '';
 $VRTS_Test = '';
 $Verbose = '';
 $Warn_Flag = '';
 $Log_Flag = '';
 $Show_Dots = '';
 $Mailto = '';
 $others = '';
 
 for ($i = 0; $i <= $#cli_args; $i++) {
     $arg1 = '';
     $arg2 = '';
     $arg3 = '';
     $arg4 = '';
     if ($cli_args[$i] =~ /^(-.*)/) {
         # found next switch, gather its values after it
         $arg1 = $1;
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg2 = $cli_args[$i];
         }
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg3 = $cli_args[$i];
         }
         if ($cli_args[$i + 1] !~ /^-/) {
             $i++;
             $arg4 = $cli_args[$i];
         }
     }
     if ($arg1 =~ /^-warn$/i) {
         $Warn_Flag = 'yes';
     } elsif ($arg1 =~ /^-all$/i) {
         if ($STORAGE eq "A5") {
            $All_Test = "yes";
            $Disk_Test = "yes";
            $Fan_Test = "yes";
            $PS_Test = "yes";
            $FRU_Test = "yes";
            $Driver_Test = "yes";
            $FW_Test = "yes";
            $Min_Test = "yes";
            $Patch_Test = "yes";
            $VRTS_Test = "yes";
         } else {
            $All_Test = "yes";
            $Driver_Test = "yes";
            $FW_Test = "yes";
            $Patch_Test = "yes";
            $VRTS_Test = "yes";
         }

     } elsif ($arg1 =~ /^-disk$/i) {
         $Disk_Test = 'yes';
         $All_Test = "no";
     } elsif ($arg1 =~ /^-drvr$/i) {
         $Driver_Test = 'yes';
         $All_Test = "no";
     } elsif ($arg1 =~ /^-fan$/i) {
         $Fan_Test = 'yes';
         $All_Test = "no";
     } elsif ($arg1 =~ /^-ps$/i) {
         $PS_Test = 'yes';
         $All_Test = "no";
     } elsif ($arg1 =~ /^-fru$/i) {
         $FRU_Test = "yes";
         $All_Test = "no";
     } elsif ($arg1 =~ /^-fw$/i) {
         $FW_Test = "yes";
     } elsif ($arg1 =~ /^-mail$/i) {
         $Mailto = $arg2;
     } elsif ($arg1 =~ /^-min$/i) {
         $Min_Test = "yes";
     } elsif ($arg1 =~ /^-pc$/i) {
         $Patch_Test = "yes";
     } elsif ($arg1 =~ /^-dots$/i) {
         $Show_Dots = "yes";
     } elsif ($arg1 =~ /^-log$/i) {
         $Log_Flag = "yes";
     } elsif ($arg1 =~ /^-vrts$/i) {
         $VRTS_Test = "yes";
         $All_Test = "no";
     } elsif ($arg1 =~ /^-v$/i) {
         $Verbose = "yes";
     } else {
         ############################################
         # catch all others for appending to output #
         ############################################
         $others .= "$arg1 " if $arg1;
         $others .= "$arg2 " if $arg2;
         $others .= "$arg3 " if $arg3;
         $others .= "$arg4 " if $arg4;
     }
 }

##############################################
# set any defaults that were not already set #
##############################################

$All_Test = "no" if ($All_Test eq '');
$Log_Flag = "no" if ($Log_Flag eq '');
$Warn_Flag = "no" if ($Warn_Flag eq '');
$Verbose = "no" if ($Verbose eq '');
$Disk_Test = 'no' if ($Disk_Test eq '');
$Fan_Test = 'no' if ($Fan_Test eq '');
$PS_Test = 'no' if ($PS_Test eq '');
$FRU_Test = 'no' if ($FRU_Test eq '');
$Driver_Test = 'no' if ($Driver_Test eq '');
$FW_Test = 'no' if ($FW_Test eq '');
$Min_Test = 'no' if ($Min_Test eq '');
$VRTS_Test = 'no' if ($VRTS_Test eq '');
$Patch_Test = 'no' if ($Patch_Test eq '');
$Show_Dots = 'no' if ($Show_Dots eq '');


 while (1) {
    if ($STORAGE eq "A5") {
         if ( ($Disk_Test eq "yes") and
             ($Fan_Test eq "yes") and
             ($PS_Test eq "yes") and
             ($FRU_Test eq "yes") and
             ($FW_Test eq "yes") and
             ($Driver_Test eq "yes") and
             ($Min_Test eq "yes") and
             ($VRTS_Test eq "yes") and
             ($Patch_Test eq "yes") ) {

             $All_Test = "yes";
         }
    } else {
         # T300 support
             if ( ($Patch_Test eq "yes") and
                  ($FW_Test eq "yes")) {

             $All_Test = "yes";
        }
    }

    if ( ($Disk_Test eq "yes") and
        ($Fan_Test eq "yes") and
        ($PS_Test eq "yes") ) {

        $FRU_Test = "yes";
    }

    if ($All_Test eq "yes") {
        if ($STORAGE eq "A5") {
           $Disk_Test = "yes";
           $Fan_Test = "yes";
           $PS_Test = "yes";
           $FRU_Test = "yes";
           $Min_Test = "yes";
        }
        $Driver_Test = "yes";
        $FW_Test = "yes";
        $Patch_Test = "yes";
        $VRTS_Test = "yes";
    }
    if ($FRU_Test eq "yes") {
        $Disk_Test = "yes";
        $Fan_Test = "yes";
        $PS_Test = "yes";
    }
    print "\n\n";
    print "Storstat Options                       Value\n";
    print "=================                      ==================\n";
    print " [a] All Checks                        $All_Test\n";
    
    if ($STORAGE eq "A5") {
        print " [d] Check Disks                       $Disk_Test\n";
        print " [f] Check Fans                        $Fan_Test\n";
        print " [m] Check Minimum Configuration       $Min_Test\n";
        print " [p] Check Power Supplies              $PS_Test\n";
        print " [u] FRU Checks (disk, fans, power)    $FRU_Test\n";
    }

    print " [c] Check Patch Levels                $Patch_Test\n";
    print " [e] Check Veritas Patch Levels        $VRTS_Test\n";
    print " [n] Check Drivers                     $Driver_Test\n";
    print " [w] Check Firmware                    $FW_Test\n\n";

    print " [v] Verbose Mode                      $Verbose\n";
    print " [s] Show Warnings                     $Warn_Flag\n";
    print " [l] Log To /var/adm/messages          $Log_Flag\n";
    print " [o] Show Activity Dots                $Show_Dots\n";
    if ( $Mailto ) {
        print " [t] Email Results To                  $Mailto\n\n";
    } else {
        print " [t] Email Results To                  <none>\n\n";
    }

    print " [r] Restore Default Settings\n";
    print " [q] Quit\n\n";

    print "Enter Selection: ";
    $choice = &get_response;  
    $choice =~ s/\s+//g;    # remove leading and trailing white space
  
    if ($choice =~ /^q/i) {
        $args = "";
        $args .= "-v " if ($Verbose eq 'yes');
        if ($All_Test eq 'yes') {
            $args .= "-all ";
        } else {
            if ($FRU_Test eq 'yes') {
                $args = "-fru ";
            } else {
                $args .= "-disk " if ($Disk_Test eq 'yes');
                $args .= "-fan " if ($Fan_Test eq 'yes');
                $args .= "-ps " if ($PS_Test eq 'yes');
            }
            $args .= "-fw " if ($FW_Test eq 'yes');
            $args .= "-min " if ($Min_Test eq 'yes');
            $args .= "-pc " if ($Patch_Test eq 'yes');
        }
        $args .= "-vrts " if (($Patch_Test eq 'no') and ($VRTS_Test eq 'yes'));
        $args .= "-drvr " if (($Driver_Test eq 'yes'));
        $args .= "-log " if ($Log_Flag eq 'yes');
        $args .= "-warn " if ($Warn_Flag eq 'yes');
        $args .= "-dots " if ($Show_Dots eq 'yes');
        $args .= "-mail $Mailto " if ($Mailto ne '');
        $args .= "$others" if ($others ne '');
        chop $args if ($args =~ / $/);

        # Check if we've got something to do
        if (($All_Test eq "yes") || ($Disk_Test eq "yes") ||
            ($Fan_Test eq "yes") || ($Min_Test eq "yes") ||
            ($Patch_Test eq "yes") || ($VRTS_Test eq "yes") ||
            ($PS_Test eq "yes") || ($FRU_Test eq "yes") ||
            ($FW_Test eq "yes") || ($Driver_Test eq "yes")) {
            if ( $restore_default eq "yes" ) {
                return $default_cli;
            } else {
                return $args;
            }
        } else {
              print "   Select an option to check.\n   Enable at least one of the following Options:\n      $test_choices. \n\n";
              print "Press <return> to continue ...";
              $wait4return = &get_response;
        }
    } elsif ($choice =~ /^a/i) {
        $restore_default = "no";
        if ($All_Test eq "yes") {
            $All_Test = "no";
            $Disk_Test = "no";
            $Driver_Test = "no";
            $Fan_Test = "no";
            $Min_Test = "no";
            $Patch_Test = "no";
            $VRTS_Test = "no";
            $PS_Test = "no";
            $FRU_Test = "no";
            $FW_Test = "no";
        } else {
            $All_Test = "yes";
        }
    } elsif ($choice =~ /^c/i) {
        $restore_default = "no";
        if ($Patch_Test eq 'no') {
            $Patch_Test = "yes";
            $VRTS_Test = "yes";
        } else {
            $Patch_Test = "no";
            $VRTS_Test = "no";
            $All_Test = "no";
        }
    } elsif ($choice =~ /^d/i) {
        $restore_default = "no";
        if ($Disk_Test eq 'no') {
            $Disk_Test = "yes";
        } else {
            $Disk_Test = "no";
            $All_Test = "no";
            $FRU_Test = "no";
        }
    } elsif ($choice =~ /^f/i) {
        $restore_default = "no";
        if ($Fan_Test eq "no") {
            $Fan_Test = "yes";
        } else {
            $Fan_Test = "no";
            $All_Test = "no";
            $FRU_Test = "no";
        }
    } elsif ($choice =~ /^l/i) {
        $restore_default = "no";
        if ($Log_Flag eq "no") {
            $Log_Flag = "yes";
        } else {
            $Log_Flag = "no";
        }
    } elsif ($choice =~ /^m/i) {
        $restore_default = "no";
        if ($Min_Test eq "no") {
            $Min_Test = "yes";
        } else {
            $Min_Test = "no";
            $All_Test = "no";
        }
    } elsif ($choice =~ /^n/i) {
        $restore_default = "no";
        if ($Driver_Test eq "no") {
            $Driver_Test = "yes";
        } else {
            $Driver_Test = "no";
            $All_Test = "no";
        }
    } elsif ($choice =~ /^o/i) {
        $restore_default = "no";
        if ($Show_Dots eq "no") {
            $Show_Dots = "yes";
        } else {
            $Show_Dots = "no";
        }
    } elsif ($choice =~ /^p/i) {
        $restore_default = "no";
       if ($PS_Test eq "no") {
           $PS_Test = "yes";
       } else {
           $PS_Test = "no";
           $All_Test = "no";
            $FRU_Test = "no";
       }
    } elsif ($choice =~ /^e/i) {
        $restore_default = "no";
       if ($VRTS_Test eq "no") {
           $VRTS_Test = "yes";
       } else {
           $VRTS_Test = "no";
#           $Patch_Test = "no";
           $All_Test = "no";
       }
    } elsif ($choice =~ /^r/i) {
        $restore_default = "yes";
        $cli = $default_cli;
        goto RESTORE_DEFAULTS;
    } elsif ($choice =~ /^s/i) {
        $restore_default = "no";
        if ($Warn_Flag eq "no") {
            $Warn_Flag = "yes";
        } else {
            $Warn_Flag = "no";
        }
    } elsif ($choice =~ /^t/i) {
        $restore_default = "no";
        ###########
        # Mail To #
        ###########
        $choice =~ s/t//;
        $choice =~ s/\s+//g;
        if (length($choice) > 1) {
            $Mailto = $choice;
        } else {
            print "Mail To: ";
            $Mailto = <STDIN>;
            chomp $Mailto;
        }
    } elsif ($choice =~ /^u/i) {
        $restore_default = "no";
        if ($FRU_Test eq "no") {
            $Disk_Test = "yes";
            $Fan_Test = "yes";
            $PS_Test = "yes";
            $FRU_Test = "yes";
        } else {
            $Disk_Test = "no";
            $Fan_Test = "no";
            $PS_Test = "no";
            $FRU_Test = "no";
            $All_Test = "no";
        }
    } elsif ($choice =~ /^w/i) {
        $restore_default = "no";
        if ($FW_Test eq "no") {
            $FW_Test = "yes";
        } else {
            $FW_Test = "no";
            $All_Test = "no";
        }
    } elsif ($choice =~ /^v/i) {
        $restore_default = "no";
        if ($Verbose eq "no") {
            $Verbose = "yes";
        } else {
            $Verbose = "no";
        }
    } else {
        print "\n$choice is an invalid choice!\n";
    }
 }
}


# T3 Support

# Add Subs for T3X0 Support

sub initcheck {

    $SIG{"INT"} = 'interrupt';
    
    $LOGFILE = "$PROGNAME.log";
    $ERROR_LOG  = "${LOGDIR}/${PROGNAME}_error_log";

    $ALLPORTS = "ALLPORTS:";
    $ONLPORTS = "ONLINE PORTS:";
    $LPMAPS = "LOOP MAPS:";
    $LONGMAPS = "LONG MAPS:";
    $LBFTEST = "LBF TEST:";
    $DISKINQ = "DISK INQUIRY:";
    $SESINQ = "SES INQUIRY:";

    $RETCODE = 0;       # init to good result

    if ( ! $STORAGE ) {
        &make_port_list( $HBA );
        &select_storage();
    }
    if ( $STORAGE eq "A5" ) {
        @LD_TEST_PORTS = @LD_A5_PORTS;
    } elsif ( $STORAGE eq "T3" ) {
        @LD_TEST_PORTS = @LD_T3_PORTS;
    }
    &display_stortools_domain();

    if ( $HBA eq "S" ) {
        my $dma_errors = &check_dma_errors( 0, 0 );
        if ( $dma_errors ) {
           printf("${YELLOW}A scan of /var/adm/messages detected $dma_errors DMA errors. ${RC}\n");
           printf("${YELLOW}These are serious errors, however they may be stale. ${RC}\n");
           printf("${YELLOW}If not stale, replace offending FC100 HBA immediately. ${RC}\n");
           printf("${YELLOW}Use ha_port_test to diagnose the problem. ${RC}\n");
           printf("${YELLOW}Otherwise ignore or remove DMA errors from /var/adm/messages. ${RC}\n");
           print "Would you like to see a DMA error summary? [<y>, n]: ";
           $yorn = &get_response("y");
           if ($yorn =~ /^y/i) {
               $dma_errors = &check_dma_errors( 0, 1 );
           }
        }
    }

    &open_logfile($LOGFILE);
    open(ERRORS, ">${ERROR_LOG}") or die "Unable to open error log : $ERROR_LOG\n";

    $date = `/bin/date`;
    print LOG $date;
    &print_field_seperator;

    if ( !@LD_TEST_PORTS && !@LD_STE_PORTS && !@LD_OFF_PORTS && !@LD_ND_PORTS ) {
        print "\n${YELLOW} No FC100 HBAs within ${PRODUCT} domain! ${RC}\n";
        print LOG "\n No FC100 HBAs within ${PRODUCT} domain!. \n";
        $RETCODE = 4;
        return $RETCODE;
    }

    if ( ! @LD_TEST_PORTS ) {
        print "\n${YELLOW} No Loops detected. ${RC}\n";
        print LOG "\n No Loops detected. \n";
        $RETCODE = 2;
        return $RETCODE;
    }

    ####################################################################
    # do a luxdiag port and port -n and save into log
    ####################################################################

    $ld_portall = `/usr/bin/ksh -c '$BINDIR/luxdiag -$HBA port 2>/dev/null'`;
    $ld_porton = `/usr/bin/ksh -c '$BINDIR/luxdiag -$HBA port -n 2>/dev/null'`;
    print LOG $ALLPORTS, "\n\n";
    print LOG $ld_portall;
    &print_field_seperator;
    print LOG $ONLPORTS, "\n\n";
    print LOG $ld_porton;
    
    &print_field_seperator;
    
    print "\n\nALL PORTS - luxdiag -$HBA port : \n" if ($DEBUG);
    print $ld_portall if ($DEBUG);
    print "\n\nONLINE PORTS - luxdiag -$HBA port -n : \n" if ($DEBUG);
    print $ld_porton if ($DEBUG);
    print "\n\n" if ($DEBUG);


    ####################################################################
    # generate report of components
    ####################################################################
    
    foreach $path (@LD_TEST_PORTS) {
        if (defined $FAILURES{$path}) {
            print ERRORS "$path  $FAILURES{$path} \n";
            print "${RED} Path: $path  $FAILURES{$path} ${RC}\n";
        }
        if (defined $WARNINGS{$path}) {
            print "${YELLOW}Path: $path  $WARNINGS{$path} ${RC}\n";
        }
    }
    if ($RETCODE == 0) {
        print "${GREEN}PASS ${RC}\n";
    } else {
        print "${RED}FAIL status = $RETCODE ${RC}\n";
    }

    &close_logfile;
    close(ERRORS);
    return $RETCODE;

}

sub cmd_exists {
##################################################################
# Make sure that Veritas has been completely installed,          #
# Is on the right OS version and has all the patches up to date? #
# Then verify that the vxdmpadm file is there and executable.    #
# Only then return a 1 signifying ok.                            #
##################################################################
        my $line;
        $VXDMPADM_PATH = "";
        $VX3_0_2 = "";
        if ( -e "/usr/lib/vxvm/bin/vxdmpadm" ) {
            $VX3_0_2 = "FALSE";
            $VXDMPADM_PATH = "/usr/lib/vxvm/bin/vxdmpadm";
        } elsif ( -e "/usr/sbin/vxdmpadm" ) {
            $VX3_0_2 = "TRUE";
            $VXDMPADM_PATH = "/usr/sbin/vxdmpadm";
        } else {
            return 0;
        }

        if ($VERITAS eq '') {

            # check for compatible Veritas version on this OS and
            # that the required patches have been loaded

            $line = `$BINDIR/storstat -$STORAGE -$HBA -vrts`;
            $VERITAS = "OK" if ($line =~ /PASS/);
        }
        return 0 if ($VERITAS ne 'OK'); # storstat FAILED

        if ((-e $VXDMPADM_PATH)&&(-x $VXDMPADM_PATH)) {
            # See if Veritas Daemon is running now
            $line = `/usr/sbin/vxdctl mode `;
            return 0  if ($line =~ /not-running/);
            return 1 ;
        }

        return 0;
}

####################################################################
# catch the control c signal
####################################################################
sub interrupt {
    print "Interrupted\n";
    $RETCODE = 3;
    exit($RETCODE);
}
####################################################################
# luxadm inq to each individual drive (timeout at 10 seconds)
#   OUTPUTS:    0   drives ok or not tested
#           !0  drive failed
####################################################################
sub eachdisk {
    ($cnum, $my_path) = @_;
    my $rc = 0;
    my $line;
    if (defined $FAILURES{$my_path}) {
        return $rc;
    }
    print LOG $DISKINQ, "\n\n";
    if ( $HBA eq "S" ) {
        $disksocal = `/bin/ls -l /dev/rdsk/${cnum}t*s2 | grep socal`;
    } elsif ( $HBA eq "P" ) {
        # since we are more rigid about getting the cnum
        # we can open this up get all pci fibre disk
        #$disksocal = `/bin/ls -l /dev/rdsk/${cnum}t*s2 | grep ifp`;
        $disksocal = `/bin/ls -l /dev/rdsk/${cnum}t*s2 | grep pci | grep ssd`;
    }
    @disksocal = split('\n', $disksocal);
    foreach $line (@disksocal) {
        @line = split(' ', $line);
        ($lgcdisk) = $line[8] =~ /\/(c\d+\S*s2)/;
        print $lgcdisk, "\n" if ($DEBUG);

#       $diskinq{$lgcdisk} = `/usr/sbin/luxadm inq $line[8]`;
        $diskinq{$lgcdisk} = `/usr/bin/ksh -c '$BINDIR/sindinq $line[8] 2>&1'`;

        if ($? != 0) {
            $rc++;
                if ($RETCODE == 1) {
                    $FAILURES{$my_path} = "Drive Inquiry failed $rc $line[8]";
                } else {
                    $FAILURES{$my_path} .= "$rc $line[8]";
                }    
                $RETCODE = 1;
        }
        print ".";
        print LOG "This is the return code: $?\n";
        print "This is the return code: $?\n" if ($DEBUG);
        print LOG $diskinq{$lgcdisk};
        if ($rc > 2) {          # if 3 drives are bad, exit
            return $rc;     
        }
        if (defined $SHORT_TEST) {  # if short test, return after 1 drive
            return $rc;
        }
    }
    if ( $STORAGE eq "T3" ) {
        print "($DDCNT{ $my_path } LUN(s) Primary: $P_DDCNT{ $my_path }  Alternate: $A_DDCNT{ $my_path })\n";
    } else {
        print "($DDCNT{ $my_path } drive(s))\n";
    }
    &print_field_seperator;
    return $rc;
}

####################################################################
# do luxdiag getmap 
#  this should be the first step after getting the ports
#  no failure checking prior to this step
####################################################################
sub getmaps {
    my ($my_path) = @_;
    if ( $HBA eq "P" ) {
       if ( $my_path =~ /ifp/ ) {
           $my_path =~ s/ifp\@(\d),0:0/ifp\@$1:devctl/;
       } elsif ( $my_path =~ /scsi/ ) {
           $my_path =~ s/scsi\@(\d),0:0/scsi\@$1:devctl/;
       }
    }
    $LD_GETMAP{$my_path} = `$BINDIR/luxdiag -$HBA getmap $my_path`;
    $LD_GML{$my_path} = `/usr/bin/ksh -c '$BINDIR/luxdiag -$HBA getmap -l $my_path 2>/dev/null'`;
    print $LD_GETMAP{$my_path} if ($DEBUG);

    print LOG "\t $LPMAPS \n\n";

    print LOG $my_path, "\n";
    print LOG $LD_GETMAP{$my_path}, "\n\n";   

    if (&length_of_one($LD_GETMAP{$my_path})) {
        $WARNINGS{$my_path} = "WARNING:  Loop Map has length of 1.   
        May be in internal loopback mode.  
        Run 'luxdiag nol $my_path' ";
    }

    &print_field_seperator;

    # now get the long format maps with WWN

    print LOG "\t $LONGMAPS \n\n";
    print LOG $my_path, ":\n";
    print LOG $LD_GML{$my_path}, "\n\n";   
    if (&lookwwn0($LD_GML{$my_path})) {
        $FAILURES{$my_path} = "WWN failure";
    }
    &print_field_seperator;
}
####################################################################
# run a couple of frames of lbf to each online port
####################################################################
sub lbftest {
    my ($my_path) = @_;
    my $rc = 0;
    if (defined $FAILURES{$my_path}) {
        return $rc;
    }
    print LOG "\t $LBFTEST \n\n";

    $lbfres = `$BINDIR/lbf -n 5 -k 64 -t 7e7e7e7e $my_path`;

    if ($lbfres =~ /fail/i) {
        print "${RED} LBF failed:  $lbfres ${RC}\n";
        print LOG "LBF FAIL :  $lbfres \n\n";
        $FAILURES{$my_path} = "LBF failure";
        $RETCODE = 1;
        $rc = 1;
    }
    print LOG $my_path, ":\n";
    print LOG $lbfres, "\n\n";

    &print_field_seperator;
    return $rc;
}
####################################################################
# check the getmap length of 1
#  this is not necessarily an error.  however, if the ha is in 
#  internal loopback mode, the user may not know it.
#  the map may be 1 if:
#       there is a loopback cable attached
#   ha is only port in hub
#   internal loopback mode
#  
#  INPUT: string input of map
#  OUTPUT: 0    okay - map length greater than 1
#      1    bad - length = 1
####################################################################
sub length_of_one {
    ($shortmap) = @_;
    my $rc = 0;
    my $line;
    @shortmap = split('\n', $shortmap);
    foreach $line (@shortmap) {
        if ($line =~ /^length/) {
            ($maplength) = $line =~ /length\:\s+(\d+)/;
            if ($maplength == 1) {
                $rc = 1;
            }
        }
    }
    return $rc;
}
####################################################################
# check the getmap for valid WWN (non zero)
#  INPUT: string input of map
#  OUTPUT: 0    wwn okay
#      1    bad wwn found
####################################################################
sub lookwwn0 {
    ($longmap) = @_;
    my $line;
    my $wwn_count = 0;
    $wwnstate = 0;
    @longmap = split('\n', $longmap);
    foreach $line (@longmap) {
        @line = split(' ', $line);
        if ($line[5] eq "0000000000000000") {
            $wwn_count++;
            # if the type is 0 (drive) or d (ses), then invalid
            if (($line[4] =~ /^0/) | ($line[4] =~ /^d/)) {
                print "This is the WWN = $line[5] \n" if ($DEBUG);
                print "${RED} Found bad WWN:  $line ${RC}\n";
                $wwnstate = 1;
                print LOG "FAIL : Bad WWN found: $line \n";
                $RETCODE = 1;
                # should never have more than 1 WWN of 0
            } elsif ($wwn_count > 1) {
                print "${RED}Found bad WWN:  $line ${RC}\n";
                $wwnstate = 1;
                print LOG "FAIL : Bad WWN found: $line \n";
                $RETCODE = 1;
            }
        }
    }
    return $wwnstate;
}
####################################################################
# open the logfile for writing
####################################################################
sub open_logfile {
    my ($logfile, $logdir) = @_;
    if ( !defined $logdir ) {
        $logdir = $LOGDIR;
    }
    if (! -d ${logdir}) {
        $return_value = `/usr/bin/ksh -c '/usr/bin/mkdir -p ${logdir} 2>&1'`;
        if ( $return_value ) {
           die "/usr/bin/mkdir -p ${logdir} failed, status $return_value\n";
        }
    }
    if (! -d ${logdir}) {
        die "Unable to create ${logdir}\n";
    }
    open(LOG, ">${logdir}/${logfile}") or die "Unable to open ${logdir}/${logfile}\n";
}

####################################################################
####################################################################

sub close_logfile {

    close LOG;

}

####################################################################
####################################################################

sub print_field_seperator {
    print LOG "\n";
    print LOG "=" x 60;
    print LOG "\n";
}


####################################################################
####################################################################

sub header {
    print LOG "HEADER:\n\n";
    $header = `/usr/bin/uname -a`;
    print LOG $header;
    $date = `/usr/bin/date`;
    print LOG $date;
    &print_field_seperator;
}

sub display_stortools_domain {

    # ports we cannot test do to a bad path

    if ( @LD_WARN_PORTS ) {
        print " ${RED}---- System configuration problem: Probable cause(s) are ----\n";
        print " ---- System was reconfigured, perform: luxadm remove, boot -r, luxadm insert\n";
        foreach $path (@LD_WARN_PORTS) {
            &old_qlogic_fcode( $path );
            print " ---- Malfunctioning HBA: ${path} ----\n";
        }
        print "${RC}\n";
    }

    # offline ports, can only be tested by ha_port_test (SBUS)
    # will need to add hba_test to ha_test for PCI  

    if ( @LD_OFF_PORTS ) {
        print " ${BLUE}---- Offline port(s) (HBA tests only) ----\n";
        foreach $path (@LD_OFF_PORTS) {
            &old_qlogic_fcode( $path );
            print " ---- Offline ports: ${path} ----\n";
        }
        print "${RC}\n";
    }

    print "\n";

    # online ports without associated ses drivers and disk drives
    # these must be either SCSI Target emulation ports of ports configured
    # with a loopback cable.

    if ( @LD_STE_PORTS ) {
        print " ${BLUE}---- Online FCAL port with no ses and no drives(s) (limited testing)\n";
        if ( $HBA eq "S" ) {
            print " ---- Suspected STE or Loopback: Test with the appropriate HBA or lbf diagnostic(s) ----\n";
        } elsif ( $HBA eq "P" ) {
            print " ---- Suspected STE or Loopback: Test with the appropriate HBA diagnostic(s) ----\n";
        }
        foreach $path (@LD_STE_PORTS) {
            &old_qlogic_fcode( $path );
            print " ---- STE/LB: ${path} ----\n";
        }
        print "${RC}\n";
    }

    # online ports that are enterprise server Motherboard ports.
    # these are FC-AL disk loops.

    if ( @LD_MB_PORTS ) {
        print " ${BLUE}---- Enterprise System onboard port(s)\n";
        foreach $path (@LD_MB_PORTS) {
            &old_qlogic_fcode( $path );
            print " ---- OB onlne ports: ${path} ----\n";
            if ($CNUM{$path} ne "") {
               &eachdisk($CNUM{$path}, $path);   # luxadm inq to each individual drive w/ timeout
               print "\n";
            } else {
               print"${YELLOW}Program Error! Invalid CNUM{path} path: $path\n\n";
            }
        }
        print "${RC}\n";
    }
    
    # online ports with associated ses drivers, but no disk drives
    # either A5Ks w/o drives inserted or a Loopback cable attached
    # after configuring an A5K

    if ( @LD_ND_PORTS ) {
        print " ${YELLOW}---- Online FCAL port with ses and no drives(s) (limited testing)\n";
        if ( $HBA eq "S" ) {
            print " ---- Suspected A5K w/o drives or Loopback: Test with the appropriate HBA or lbf diagnostic(s) ----\n";
        } elsif ( $HBA eq "P" ) {
            print " ---- Suspected A5K w/o drives or Loopback: Test with the appropriate HBA diagnostic(s) ----\n";
        }
        foreach $path (@LD_ND_PORTS) {
            &old_qlogic_fcode( $path );
            print " ---- Suspect A5K port: ${path} ----\n";
        }
        print "${RC}\n";
    }

    # valid A5K or T300 ports

    print "\n";
    if ( @LD_TEST_PORTS ) {
        print " ${GREEN}---- FCAL HBA(s) within the ${PRODUCT} domain ----\n";
        foreach $path (@LD_TEST_PORTS) {
            &old_qlogic_fcode( $path );
            &display_paths();
        }
        print "${RC}\n";
    }
}

sub display_paths() {
    print " ---- Online: ${path} ----\n";
    if ( $HBA eq "S" ) {
      print " \t($CNUM{$path}) ($SFNUM{$path})  ($SOCNUM{$path}:Port $PORTNUM{$path}) \n";
      &getmaps($path);   # get map and long map and check for bad wwn
      &lbftest($path);   # short lbf test for hard errors
    } elsif ( $HBA eq "P" ) {
      &getmaps($path);   # get map and long map and check for bad wwn
    }
    print "DEBUG:  This is the cnum:  $path = $CNUM{$path} \n" if ($DEBUG);
    if ($CNUM{$path} ne "") {
       &eachdisk($CNUM{$path}, $path);   # luxadm inq to each individual drive w/ timeout
       print "\n";
    } else {
       print"${YELLOW}Program Error! Invalid CNUM{path} path: $path ${RC}\n\n";
    }
}

sub old_qlogic_fcode() {
    my $path = $_[0];
    if ( ( $path =~ /scsi/ ) && ( $HBA eq "P" ) ) {
       print "${YELLOW} WARNING: Invalid PCI Qlogic FCODE detected\n";
       print "${YELLOW} $CNUM{$path} path: $path\n";
       print "${YELLOW} FCAL fcode compatible property incorrectly reports -scsi-\n";
       print"${YELLOW} Obtain the correct FCODE from SUN.\n\n";
       print "${RC}\n";
    }
}

sub free_to_grow {
    ################################################################
    # passes 2 parameters: file path & number of gigs, megs, or kb #
    ################################################################
    my ($filepath, $sizerequest) = @_;
    my ($diskspace, $ksize, $device, $freespace, $perm, $temp, $size, @filesystem, $mod, $units, $dirname);
    ####################################################################
    # Check to see if there is enough free disk space to grow the file #
    ####################################################################
    if ($sizerequest =~ /(\d+)([gmk])/) {
        ################################################################################
        # figure out how much total disk space is requested for this test in kilobytes #
        ################################################################################
        $diskspace = $1;
        $units = $2;
        $diskspace = $diskspace * 1024 if ($units =~ /^m$/i);
        $diskspace = $diskspace * 1024 * 1024 if ($units =~ /^g$/i);
    } else {
        return "Illegal disk size request $sizerequest"; # illegal size request
    }
    if (-f $filepath) {
        ########################################################
        # Regular file exists. Subtract out current file size, #
        # if positive result then more space required.         #
        ########################################################
        $size = `/usr/bin/ksh -c 'ls -goL $filepath 2> /dev/null'`; # current file size
        chomp $size;
        ($perm, $temp, $size) = split /\s+/, $size;
        $ksize = $size / 1024; # convert to kilobytes from bytes
        $ksize = int $ksize;
        $mod = $size - ($ksize * 1024);
        $ksize += 1 if ($mod);
        $diskspace = $diskspace - $ksize; # if positive then need to have free space
        $dirname = $1 if ($filepath =~ /(.*)\/.*/);
    } elsif ( -d $filepath ) {
        $dirname = $filepath;
    } else {
        ##############################################
        # file does not exist, keep directory portion#
        ##############################################
        $dirname = $1 if ($filepath =~ /(.*)\/.*/);
    }
    $dirname = "/" if ($dirname eq '');
    if ($diskspace > 0) {
        ##############################################################
        # File not big enough, check for enough free space available #
        ##############################################################
        $filesystem = `/usr/bin/ksh -c '/usr/sbin/df -e $dirname 2> /dev/null'`;
        @filesystem = split /\n/, $filesystem;
        ($device, $freespace) = split /\s+/, $filesystem[1];
        $freespace = $freespace - $diskspace;
        if ($freespace < 0) {
            $freespace = abs $freespace;
            return "$freespace kb space needed in filesystem $device.\n";
        }
    }
}


# Perform a standardized mail using sendmail that can be called
# from different perl modules.
# args:  1) recipient 
#	 2) subject 
#	 3) header message 
#	 4) name of file to mail.

sub mail_message {

	my ($to, $subject,  $fileToMail) = @_;
	my $TEMPMAIL = "/tmp/mailsub$$";
	my $host_info=`/usr/bin/uname -a`;
	my $host_id=`/bin/hostid`;
	chomp($host=`/usr/bin/uname -n`);
	
	if (!open(ORIGFILE, "< $fileToMail")) {
		printf("mail_message():  Cannot open $fileToMail\n");
		return(0);
	}

	if(!open(NEWMAIL, "> $TEMPMAIL") ) {
		printf("mail_message():  Cannot open $TEMPMAIL\n");
		 return(0);
	}

	print NEWMAIL "To: $to\n";
	print NEWMAIL "From: ${PRODUCT}\@$host\n";
	print NEWMAIL "Subject: $subject\n";
	print NEWMAIL "\n";
	print NEWMAIL "HostID: $host_id\n";
	print NEWMAIL "$host_info\n";

  	$text = "The following indicates FC-AL/Storage related problems.  \nTake appropriate action.";

	print NEWMAIL "$text\n\n";

  	($a, $b, $c, $d, $e, $f, $g, $size, $i, $j, $k, $l, $m) = stat($fileToMail);

 	if (  $size  ) { 
		while ( <ORIGFILE> ) {
			print NEWMAIL "$_";
		}
	} else {
		print NEWMAIL "\nmail_message() No e-mail message:  Program error.\n\n";

	}
	close(NEWMAIL);
	close(ORIGFILE);

	system("cat $TEMPMAIL | /usr/lib/sendmail -t");
        `/usr/bin/rm $TEMPMAIL`;

	return(1);
}

sub match_hostname {

    my $line = shift;
    my $hostname;
    ($hostname) = $line =~ /\w+\s+\d+\s+\d+:\d+:\d+\s(\S+)/;
    if ( $hostname ) {
        return $hostname;
    }
    ($hostname) = $line =~ /\w+\s+\d+\s+\d+:\d+:\d+\s(\[\d+.\d+.\d+.\d+.\d+.\d+\])/;
    if ( $hostname ) {
        return $hostname;
    }
    return "INVALID HOSTNAME";
}

#perlism according to David, a require file must return a value
1;
