/*
 * Copyright (c) 1990-2001 by Sun Microsystems, Inc.
 * All rights reserved.
 */

#ifndef	_SYS_SUNDDI_H
#define	_SYS_SUNDDI_H

#pragma ident	"@(#)sunddi.h	1.118	01/08/28 SMI"

/*
 * Sun Specific DDI definitions
 */

#include <sys/isa_defs.h>
#include <sys/dditypes.h>
#include <sys/ddipropdefs.h>
#include <sys/devops.h>
#include <sys/time.h>
#include <sys/cmn_err.h>
#include <sys/ddidevmap.h>
#include <sys/ddi_implfuncs.h>
#include <sys/ddi_isa.h>
#include <sys/model.h>
#include <sys/devctl.h>
#if defined(i386) || defined(__i386)
#include <sys/dma_engine.h>
#endif
#include <sys/sunpm.h>
#include <sys/nvpair.h>
#include <sys/sysevent.h>

#ifdef	__cplusplus
extern "C" {
#endif

/*
 * Generic Sun DDI definitions.
 */

#define	DDI_SUCCESS	(0)	/* successful return */
#define	DDI_FAILURE	(-1)	/* unsuccessful return */
#define	DDI_NOT_WELL_FORMED (-2)  /* A dev_info node is not valid */

/*
 * General-purpose DDI error return value definitions
 */
#define	DDI_ENOMEM		1	/* memory not available */
#define	DDI_EBUSY		2	/* busy */
#define	DDI_ETRANSPORT		3	/* transport down */
#define	DDI_ECONTEXT		4	/* context error */


/*
 * General DDI sleep/nosleep allocation flags
 */
#define	DDI_SLEEP	0
#define	DDI_NOSLEEP	1

/*
 * The following special nodeid values are reserved for use when creating nodes
 * ONLY.  They specify properties about the node being created, specifically,
 * DEVI_PSEUDO_NODEID specificies a pseudo-node without the property of
 * persistence and DEVI_SID_NODEID specifies a pseudo-node with the property
 * of persistence. A node with the 'persistent' property will not be
 * automatically removed by the framework in the current implementation.
 * Typically, .conf file nodes created by the framework do not have the
 * 'perisistent' property.
 *
 * The actual nodeid value may be assigned by the framework and may be
 * different than these special values. Drivers may not make assumptions
 * about the nodeid value that is actually assigned to the node.
 */

#define	DEVI_PSEUDO_NODEID	((int)-1)
#define	DEVI_SID_NODEID		((int)-2)

#define	DEVI_PSEUDO_NEXNAME	"pseudo"
#define	DEVI_ISA_NEXNAME	"isa"
#define	DEVI_EISA_NEXNAME	"eisa"
#define	CLONE_DEV		1

/*
 * The idea behind this madness is simple. We are or'ing the value
 * of class with is_clone flag in ddi_creat_minor_node. The is_clone
 * flag is (to borrow a term from C++) over-loaded to be able
 * to define if the device node has one of the qualities listed below
 */

#define	GLOBAL_DEV		0x2	/* global device (/dev/le /dev/ip) */
#define	NODEBOUND_DEV		0x4	/* bound to one node  (/dev/tiocts) */
#define	NODESPECIFIC_DEV	0x6	/* specific to node only (/dev/kmem) */
#define	ENUMERATED_DEV		0x8	/* each node distinguished (c0t0d0s0) */
#define	DEVCLASS_MASK		0xe
#define	CLASS_DEV(flag)		((flag) & DEVCLASS_MASK)

/*
 * Device type defines which are used by the 'node_type' element of the
 * ddi_minor_data structure
 */
#define	DDI_NT_SERIAL	"ddi_serial"		/* Serial port */
#define	DDI_NT_SERIAL_MB "ddi_serial:mb"	/* the 'built-in' serial */
						/* ports (the old ttya, b */
						/* (,c ,d)) */
#define	DDI_NT_SERIAL_DO "ddi_serial:dialout"	/* dialout ports */
#define	DDI_NT_SERIAL_MB_DO "ddi_serial:dialout,mb" /* dialout for onboard */
						/* ports */
/*
 * *_CHAN disk type devices have channel numbers or target numbers.
 * (i.e. ipi and scsi devices)
 */
#define	DDI_NT_BLOCK	"ddi_block"		/* hard disks */
/*
 * The next define is for block type devices that can possible exist on
 * a sub-bus like the scsi bus or the ipi channel.  The 'disks' program
 * will pick up on this and create logical names like c0t0d0s0 instead of
 * c0d0s0
 */
#define	DDI_NT_BLOCK_CHAN	"ddi_block:channel"
#define	DDI_NT_BLOCK_WWN	"ddi_block:wwn"
#define	DDI_NT_CD	"ddi_block:cdrom"	/* rom drives (cd-rom) */
#define	DDI_NT_CD_CHAN	"ddi_block:cdrom:channel" /* rom drives (scsi type) */
#define	DDI_NT_FD	"ddi_block:diskette"	/* floppy disks */

#define	DDI_NT_TAPE	"ddi_byte:tape"		/* tape drives */

#define	DDI_NT_NET	"ddi_network"		/* DLPI network devices */

#define	DDI_NT_DISPLAY	"ddi_display"		/* display devices */

#define	DDI_PSEUDO	"ddi_pseudo"		/* general pseudo devices */

#define	DDI_NT_AUDIO	"ddi_audio"		/* audio device */

#define	DDI_NT_MOUSE	"ddi_mouse"		/* mouse device */

#define	DDI_NT_KEYBOARD	"ddi_keyboard"		/* keyboard device */

#define	DDI_NT_PARALLEL	"ddi_parallel"		/* parallel port */

#define	DDI_NT_PRINTER	"ddi_printer"		/* printer device */

#define	DDI_NT_NEXUS	"ddi_ctl:devctl"	/* nexus drivers */

#define	DDI_NT_SCSI_NEXUS	"ddi_ctl:devctl:scsi"	/* nexus drivers */

#define	DDI_NT_SBD_ATTACHMENT_POINT	"ddi_ctl:attachment_point:sbd"
						/* generic bd attachment pt */
#define	DDI_NT_FC_ATTACHMENT_POINT	"ddi_ctl:attachment_point:fc"
						/* FC attachment point */
#define	DDI_NT_ATTACHMENT_POINT	"ddi_ctl:attachment_point" /* attachment pt */

#define	DDI_NT_SCSI_ATTACHMENT_POINT	"ddi_ctl:attachment_point:scsi"
						/* scsi attachment pt */
#define	DDI_NT_PCI_ATTACHMENT_POINT	"ddi_ctl:attachment_point:pci"
						/* PCI attachment pt */
#define	DDI_NT_BLOCK_FABRIC	"ddi_block:fabric"	/* Fabric Devices */

#define	DDI_NT_SMARTCARD_READER	"ddi_smartcard_reader" /* Smartcard reader */

/*
 * DDI event definitions
 */
#define	EC_DEVFS	"EC_devfs"	/* Event class devfs */

/* Class devfs subclasses */
#define	ESC_DEVFS_MINOR_CREATE	"ESC_devfs_minor_create"
#define	ESC_DEVFS_MINOR_REMOVE	"ESC_devfs_minor_remove"
#define	ESC_DEVFS_DEVI_ADD	"ESC_devfs_devi_add"
#define	ESC_DEVFS_DEVI_REMOVE	"ESC_devfs_devi_remove"
#define	ESC_DEVFS_INSTANCE_MOD	"ESC_devfs_instance_mod"

/* DDI/NDI event publisher */
#define	EP_DDI	SUNW_KERN_PUB"ddi"

/*
 * devfs event class attributes
 *
 * The following attributes are private to EC_DEVFS
 * event data.
 */
#define	DEVFS_DRIVER_NAME	"di.driver"
#define	DEVFS_PATHNAME		"di.path"
#define	DEVFS_MINOR_NAME	"mi.name"
#define	DEVFS_MINOR_NODETYPE	"mi.nodetype"
#define	DEVFS_MINOR_ISCLONE	"mi.isclone"
#define	DEVFS_MINOR_MAJNUM	"mi.majorno"
#define	DEVFS_MINOR_MINORNUM	"mi.minorno"

/*
 * Fault-related definitions
 *
 * The specific numeric valus have been chosen to be ordered, but
 * not consecutive, to allow for future interpolation if required.
 */
typedef enum {
    DDI_SERVICE_LOST = -32,
    DDI_SERVICE_DEGRADED = -16,
    DDI_SERVICE_UNAFFECTED = 0,
    DDI_SERVICE_RESTORED = 16
} ddi_fault_impact_t;

typedef enum {
    DDI_DATAPATH_FAULT = -32,
    DDI_DEVICE_FAULT = -16,
    DDI_EXTERNAL_FAULT = 0
} ddi_fault_location_t;

typedef enum {
    DDI_DEVSTATE_OFFLINE = -32,
    DDI_DEVSTATE_DOWN = -16,
    DDI_DEVSTATE_QUIESCED = 0,
    DDI_DEVSTATE_DEGRADED = 16,
    DDI_DEVSTATE_UP = 32
} ddi_devstate_t;


#ifdef	_KERNEL

/*
 * Function class definitions.
 */

/*
 * Interrupt related definitions.
 */

/*
 * Returned by ddi_add_intr or ddi_add_fastintr in order to signal
 * the the caller requested interrupt number to be added does not
 * exist.
 */

#define	DDI_INTR_NOTFOUND	1	/* interrupt not found error */

/*
 * For use by ddi_add_softintr in order to specify a priority preference.
 */
#define	DDI_SOFTINT_FIXED	0	/* Fixed priority soft interrupt */
#define	DDI_SOFTINT_LOW		8	/* Low priority soft interrupt */
#define	DDI_SOFTINT_MED		128	/* Medium priority soft interrupt */
#define	DDI_SOFTINT_HIGH	256	/* High priority soft interrupt */

/*
 * For use by driver interrupt service routines to return to the
 * system whether an interrupt was for the driver or not.
 */
#define	DDI_INTR_CLAIMED	1	/* returned when driver claims intr */
#define	DDI_INTR_UNCLAIMED	0	/* returned when driver doesn't "" */


/*
 * Values that the function supplied to the dev_info
 * tree traversal functions defined below must return.
 */

/*
 * Continue search, if appropriate.
 */
#define	DDI_WALK_CONTINUE	0

/*
 * Terminate current depth of traversal. That is, terminate
 * the current traversal of children nodes, but continue
 * traversing sibling nodes and their children (if any).
 */

#define	DDI_WALK_PRUNECHILD	-1

/*
 * Terminate current width of traversal. That is, terminate
 * the current traversal of sibling nodes, but continue with
 * traversing children nodes and their siblings (if appropriate).
 */

#define	DDI_WALK_PRUNESIB	-2

/*
 * Terminate the entire search.
 */

#define	DDI_WALK_TERMINATE	-3

/*
 * Drivers that are prepared to support full driver layering
 * should create and export a null-valued property of the following
 * name.
 *
 * Such drivers should be prepared to be called with FKLYR in
 * the 'flag' argument of their open(9E), close(9E) routines, and
 * with FKIOCTL in the 'mode' argument of their ioctl(9E) routines.
 *
 * See ioctl(9E) and ddi_copyin(9F) for details.
 */
#define	DDI_KERNEL_IOCTL	"ddi-kernel-ioctl"

/*
 * Model definitions for ddi_mmap_get_model(9F) and ddi_model_convert_from(9F).
 */
#define	DDI_MODEL_MASK		DATAMODEL_MASK	/* Note: 0x0FF00000 */
#define	DDI_MODEL_ILP32		DATAMODEL_ILP32
#define	DDI_MODEL_LP64		DATAMODEL_LP64
#define	DDI_MODEL_NATIVE	DATAMODEL_NATIVE
#define	DDI_MODEL_NONE		DATAMODEL_NONE

/*
 * If you add or remove a function or data reference, please
 * remember to duplicate the action below the #else clause for
 * __STDC__.
 */

#ifdef	__STDC__
/*
 * Functions and data references which really should be in <sys/ddi.h>
 */

extern int maxphys;
extern void minphys(struct buf *);
extern int physio(int (*)(struct buf *), struct buf *, dev_t,
	int, void (*)(struct buf *), struct uio *);
extern void disksort(struct diskhd *, struct buf *);

extern size_t strlen(const char *);
extern char *strcpy(char *, const char *);
extern char *strncpy(char *, const char *, size_t);
extern char *strchr(const char *, int);
extern int strcmp(const char *, const char *);
extern int strncmp(const char *, const char *, size_t);
extern int bcmp(const void *, const void *, size_t);
extern int stoi(char **);
extern void numtos(ulong_t, char *);
extern void bcopy(const void *, void *, size_t);
extern void bzero(void *, size_t);
/* XXX: Should we put kcopy/kzero in too? */

/*
 * ddi_map_regs
 *
 *	Map in the register set given by rnumber.
 *	The register number determine which register
 *	set will be mapped if more than one exists.
 *	The parent driver gets the information
 *	from parent private data and sets up the
 *	appropriate mappings and returns the kernel
 *	virtual address of the register set in *kaddrp.
 *	The offset specifies an offset into the register
 *	space to start from and len indicates the size
 *	of the area to map. If len and offset are 0 then
 *	the entire space is mapped.  It returns DDI_SUCCESS on
 *	success or DDI_FAILURE otherwise.
 *
 */
int
ddi_map_regs(dev_info_t *dip, uint_t rnumber, caddr_t *kaddrp,
	off_t offset, off_t len);

/*
 * ddi_unmap_regs
 *
 *	Undo mappings set up by ddi_map_regs.
 *	The register number determines which register
 *	set will be unmapped if more than one exists.
 *	This is provided for drivers preparing
 *	to detach themselves from the system to
 *	allow them to release allocated mappings.
 *
 *	The kaddrp and len specify the area to be
 *	unmapped. *kaddrp was returned from ddi_map_regs
 *	and len should match what ddi_map_regs was called
 *	with.
 */

void
ddi_unmap_regs(dev_info_t *dip, uint_t rnumber, caddr_t *kaddrp,
	off_t offset, off_t len);

int
ddi_map(dev_info_t *dp, ddi_map_req_t *mp, off_t offset, off_t len,
	caddr_t *addrp);

int
ddi_apply_range(dev_info_t *dip, dev_info_t *rdip, struct regspec *rp);

/*
 * ddi_rnumber_to_regspec: Not for use by leaf drivers.
 */
struct regspec *
ddi_rnumber_to_regspec(dev_info_t *dip, int rnumber);

int
ddi_bus_map(dev_info_t *dip, dev_info_t *rdip, ddi_map_req_t *mp, off_t offset,
	off_t len, caddr_t *vaddrp);

int
nullbusmap(dev_info_t *dip, dev_info_t *rdip, ddi_map_req_t *mp, off_t offset,
	off_t len, caddr_t *vaddrp);

/*
 * Peek and poke by asking the parent nexi
 */
int common_ddi_peek(dev_info_t *devi, size_t size, void *addr, void *value_p);
int common_ddi_poke(dev_info_t *devi, size_t size, void *addr, void *value_p);

#ifdef _LP64

int ddi_peek8(dev_info_t *dip, int8_t *addr, int8_t *val_p);
int ddi_peek16(dev_info_t *dip, int16_t *addr, int16_t *val_p);
int ddi_peek32(dev_info_t *dip, int32_t *addr, int32_t *val_p);
int ddi_peek64(dev_info_t *dip, int64_t *addr, int64_t *val_p);

int ddi_poke8(dev_info_t *dip, int8_t *addr, int8_t val);
int ddi_poke16(dev_info_t *dip, int16_t *addr, int16_t val);
int ddi_poke32(dev_info_t *dip, int32_t *addr, int32_t val);
int ddi_poke64(dev_info_t *dip, int64_t *addr, int64_t val);

#else /* _ILP32 */

int ddi_peekc(dev_info_t *dip, int8_t *addr, int8_t *val_p);
#define	ddi_peek8	ddi_peekc

int ddi_peeks(dev_info_t *dip, int16_t *addr, int16_t *val_p);
#define	ddi_peek16	ddi_peeks

int ddi_peekl(dev_info_t *dip, int32_t *addr, int32_t *val_p);
#define	ddi_peek32	ddi_peekl

int ddi_peekd(dev_info_t *dip, int64_t *addr, int64_t *val_p);
#define	ddi_peek64	ddi_peekd

int ddi_pokec(dev_info_t *dip, int8_t *addr, int8_t val);
#define	ddi_poke8	ddi_pokec

int ddi_pokes(dev_info_t *dip, int16_t *addr, int16_t val);
#define	ddi_poke16	ddi_pokes

int ddi_pokel(dev_info_t *dip, int32_t *addr, int32_t val);
#define	ddi_poke32	ddi_pokel

int ddi_poked(dev_info_t *dip, int64_t *addr, int64_t val);
#define	ddi_poke64	ddi_poked

#endif /* _LP64 */

/*
 * Peek and poke to and from a uio structure in xfersize pieces,
 * using the parent nexi.
 */
int ddi_peekpokeio(dev_info_t *devi, struct uio *uio, enum uio_rw rw,
	caddr_t addr, size_t len, uint_t xfersize);

/*
 * Pagesize conversions using the parent nexi
 */
unsigned long ddi_btop(dev_info_t *dip, unsigned long bytes);
unsigned long ddi_btopr(dev_info_t *dip, unsigned long bytes);
unsigned long ddi_ptob(dev_info_t *dip, unsigned long pages);

/*
 * ddi_add_intr: Add an interrupt to the system.
 *
 *	The interrupt number "inumber" determines which interrupt will
 *	be added. The interrupt number is associated with interrupt
 *	information provided from self identifying devices or configuration
 *	information for non-self identifying devices. If only one interrupt
 *	is associated with the device then the interrupt number should be 0.
 *
 *	If successful, "*iblock_cookiep" will contain information necessary
 *	for initializing locks (mutex_init, cv_init, etc.) as well as for
 *	possible later removal of the interrupt from the system.
 *
 *	If successful, "*idevice_cookiep" will contain the correct programmable
 *	device interrupt value (see <sys/dditypes.h> in the form of the
 *	type ddi_idevice_cookie_t).
 *
 *	Either cookie pointer may be specified as a NULL pointer
 *	in which case no value will be returned.
 *
 *	The interrupt handler "int_handler" is the address of the routine
 *	to be called upon receipt of an appropriate interrupt. The
 *	interrupt handler should return DDI_INTR_CLAIMED if the
 *	interrupt was claimed, else DDI_INTR_UNCLAIMED. The argument
 *	"int_handler_arg" will be passed to the "int_handler"
 *	upon receipt of an appropriate interrupt.
 *
 *	If successful ddi_add_intr will return DDI_SUCCESS.
 *	If the interrupt information cannot be found it will
 *	return DDI_INTR_NOTFOUND.
 *
 */
int
ddi_add_intr(dev_info_t *dip, uint_t inumber,
	ddi_iblock_cookie_t *iblock_cookiep,
	ddi_idevice_cookie_t *idevice_cookiep,
	uint_t (*int_handler)(caddr_t int_handler_arg),
	caddr_t int_handler_arg);

/*
 * Return non-zero if the specified interrupt exists and the handler
 * will be restricted to using only certain functions because the
 * interrupt level is not blocked by the scheduler.  I.e., it cannot
 * signal other threads.
 */
int
ddi_intr_hilevel(dev_info_t *dip, uint_t inumber);

/*
 * The following function is for Sun's internal use only at present
 */
int
ddi_add_fastintr(dev_info_t *dip, uint_t inumber,
	ddi_iblock_cookie_t *iblock_cookiep,
	ddi_idevice_cookie_t *idevice_cookiep,
	uint_t (*hi_int_handler)(void));

/*
 * ddi_add_softintr:	Add a "soft" interrupt to the system.
 *
 *	Like ddi_add_intr, only for system interrupts that you can trigger
 *	yourself. You specify a preference (see above) for the level you
 *	want. You get an identifier back which you can use to either trigger
 *	a soft interrupt or, later, remove it.
 */
int
ddi_add_softintr(dev_info_t *dip, int preference, ddi_softintr_t *idp,
	ddi_iblock_cookie_t *iblock_cookiep,
	ddi_idevice_cookie_t *idevice_cookiep,
	uint_t (*int_handler)(caddr_t int_handler_arg),
	caddr_t int_handler_arg);

void
ddi_remove_softintr(ddi_softintr_t id);

void
ddi_trigger_softintr(ddi_softintr_t id);

/*
 * ddi_remove_intr:	Remove interrupt set up by ddi_add_intr.
 *
 *	This routine is intended to be used by drivers that are
 *	preparing to unload themselves "detach" from the system.
 */
void
ddi_remove_intr(dev_info_t *dip, uint_t inum,
	ddi_iblock_cookie_t iblock_cookie);

int
ddi_get_iblock_cookie(dev_info_t *dip, uint_t inumber,
    ddi_iblock_cookie_t *iblock_cookiep);

int
ddi_get_soft_iblock_cookie(dev_info_t *dip, int preference,
    ddi_iblock_cookie_t *iblock_cookiep);


/*
 * There are no more "block" interrupt functions, per se.
 * All thread of control should be done with MP/MT lockings.
 *
 * However, there are certain times in which a driver needs
 * absolutely a critical guaranteed non-preemptable time
 * in which to execute a few instructions.
 *
 * The following pair of functions attempt to guarantee this,
 * but they are dangerous to use. That is, use them with
 * extreme care. They do not guarantee to stop other processors
 * from executing, but they do guarantee that the caller
 * of ddi_enter_critical will continue to run until the
 * caller calls ddi_exit_critical. No intervening DDI functions
 * may be called between an entry and an exit from a critical
 * region.
 *
 * ddi_enter_critical returns an integer identifier which must
 * be passed to ddi_exit_critical.
 *
 * Be very sparing in the use of these functions since it is
 * likely that absolutely nothing else can occur in the system
 * whilst in the critical region.
 */

unsigned int
ddi_enter_critical(void);

void
ddi_exit_critical(unsigned int);

/*
 * devmap functions
 */
int
devmap_setup(dev_t dev, offset_t off, ddi_as_handle_t as, caddr_t *addrp,
	size_t len, uint_t prot, uint_t maxprot, uint_t flags,
	struct cred *cred);

int
ddi_devmap_segmap(dev_t dev, off_t off, ddi_as_handle_t as, caddr_t *addrp,
	off_t len, uint_t prot, uint_t maxprot, uint_t flags,
	struct cred *cred);

int
devmap_load(devmap_cookie_t dhp, offset_t offset, size_t len, uint_t type,
	uint_t rw);

int
devmap_unload(devmap_cookie_t dhp, offset_t offset, size_t len);

int
devmap_devmem_setup(devmap_cookie_t dhp, dev_info_t *dip,
	struct devmap_callback_ctl *callback_ops,
	uint_t rnumber, offset_t roff, size_t len, uint_t maxprot,
	uint_t flags, ddi_device_acc_attr_t *accattrp);

int
devmap_umem_setup(devmap_cookie_t dhp, dev_info_t *dip,
	struct devmap_callback_ctl *callback_ops,
	ddi_umem_cookie_t cookie, offset_t off, size_t len, uint_t maxprot,
	uint_t flags, ddi_device_acc_attr_t *accattrp);

int
devmap_devmem_remap(devmap_cookie_t dhp, dev_info_t *dip,
	uint_t rnumber, offset_t roff, size_t len, uint_t maxprot,
	uint_t flags, ddi_device_acc_attr_t *accattrp);

int
devmap_umem_remap(devmap_cookie_t dhp, dev_info_t *dip,
	ddi_umem_cookie_t cookie, offset_t off, size_t len, uint_t maxprot,
	uint_t flags, ddi_device_acc_attr_t *accattrp);

void
devmap_set_ctx_timeout(devmap_cookie_t dhp, clock_t ticks);

int
devmap_default_access(devmap_cookie_t dhp, void *pvtp, offset_t off,
	size_t len, uint_t type, uint_t rw);

int
devmap_do_ctxmgt(devmap_cookie_t dhp, void *pvtp, offset_t off, size_t len,
	uint_t type, uint_t rw, int (*ctxmgt)(devmap_cookie_t, void *, offset_t,
	size_t, uint_t, uint_t));


void *ddi_umem_alloc(size_t size, int flag, ddi_umem_cookie_t *cookiep);

void ddi_umem_free(ddi_umem_cookie_t cookie);

/*
 * Functions to lock user memory and do repeated I/O or do devmap_umem_setup
 */
int
ddi_umem_lock(caddr_t addr, size_t size, int flags, ddi_umem_cookie_t *cookie);

void
ddi_umem_unlock(ddi_umem_cookie_t cookie);

struct buf *
ddi_umem_iosetup(ddi_umem_cookie_t cookie, off_t off, size_t len, int direction,
    dev_t dev, daddr_t blkno, int (*iodone)(struct buf *), int sleepflag);

/*
 * Mapping functions
 */
int
ddi_segmap(dev_t dev, off_t offset, struct as *asp, caddr_t *addrp, off_t len,
	uint_t prot, uint_t maxprot, uint_t flags, cred_t *credp);

int
ddi_segmap_setup(dev_t dev, off_t offset, struct as *as, caddr_t *addrp,
	off_t len, uint_t prot, uint_t maxprot, uint_t flags, cred_t *cred,
	ddi_device_acc_attr_t *accattrp, uint_t rnumber);

int
ddi_map_fault(dev_info_t *dip, struct hat *hat, struct seg *seg, caddr_t addr,
	struct devpage *dp, pfn_t pfn, uint_t prot, uint_t lock);

int
ddi_device_mapping_check(dev_t dev, ddi_device_acc_attr_t *accattrp,
	uint_t rnumber, uint_t *hat_flags);

/*
 * The following routines are used to create and manage mapdev segments.
 */
int
ddi_mapdev(dev_t dev, off_t offset, struct as *asp, caddr_t *addrp, off_t len,
    uint_t prot, uint_t maxprot, uint_t flags, cred_t *cred,
    struct ddi_mapdev_ctl *m_ops, ddi_mapdev_handle_t *handlep,
    void *private_data);

int
ddi_mapdev_intercept(ddi_mapdev_handle_t handle, off_t offset, off_t len);

int
ddi_mapdev_nointercept(ddi_mapdev_handle_t handle, off_t offset, off_t len);

/*
 * Property functions:   See also, ddipropdefs.h.
 *			In general, the underlying driver MUST be held
 *			to call it's property functions.
 */

/*
 * Used to create, modify, and lookup integer properties
 */
int ddi_prop_get_int(dev_t match_dev, dev_info_t *dip, uint_t flags,
    char *name, int defvalue);
int ddi_prop_lookup_int_array(dev_t match_dev, dev_info_t *dip, uint_t flags,
    char *name, int **data, uint_t *nelements);
int ddi_prop_update_int(dev_t match_dev, dev_info_t *dip,
    char *name, int data);
int ddi_prop_update_int_array(dev_t match_dev, dev_info_t *dip,
    char *name, int *data, uint_t nelements);

/*
 * Used to create, modify, and lookup string properties
 */
int ddi_prop_lookup_string(dev_t match_dev, dev_info_t *dip, uint_t flags,
    char *name, char **data);
int ddi_prop_lookup_string_array(dev_t match_dev, dev_info_t *dip, uint_t flags,
    char *name, char ***data, uint_t *nelements);
int ddi_prop_update_string(dev_t match_dev, dev_info_t *dip,
    char *name, char *data);
int ddi_prop_update_string_array(dev_t match_dev, dev_info_t *dip,
    char *name, char **data, uint_t nelements);

/*
 * Used to create, modify, and lookup byte properties
 */
int ddi_prop_lookup_byte_array(dev_t match_dev, dev_info_t *dip, uint_t flags,
    char *name, uchar_t **data, uint_t *nelements);
int ddi_prop_update_byte_array(dev_t match_dev, dev_info_t *dip,
    char *name, uchar_t *data, uint_t nelements);

/*
 * Used to verify the existance of a property or to see if a boolean
 * property exists.
 */
int ddi_prop_exists(dev_t match_dev, dev_info_t *dip, uint_t flags, char *name);

/*
 * Used to free the data returned by the above property routines.
 */
void ddi_prop_free(void *data);

/*
 * nopropop: For internal use in `dummy' cb_prop_op functions only
 */

int
nopropop(dev_t dev, dev_info_t *dip, ddi_prop_op_t prop_op, int mod_flags,
	char *name, caddr_t valuep, int *lengthp);

/*
 * ddi_prop_op: The basic property operator for drivers.
 *
 * In ddi_prop_op, the type of valuep is interpreted based on prop_op:
 *
 *	prop_op			valuep
 *	------			------
 *
 *	PROP_LEN		<unused>
 *
 *	PROP_LEN_AND_VAL_BUF	Pointer to callers buffer
 *
 *	PROP_LEN_AND_VAL_ALLOC	Address of callers pointer (will be set to
 *				address of allocated buffer, if successful)
 */

int
ddi_prop_op(dev_t dev, dev_info_t *dip, ddi_prop_op_t prop_op, int mod_flags,
	char *name, caddr_t valuep, int *lengthp);


/*
 * Variable length props...
 */

/*
 * ddi_getlongprop:	Get variable length property len+val into a buffer
 *		allocated by property provider via kmem_alloc. Requestor
 *		is responsible for freeing returned property via kmem_free.
 *
 * 	Arguments:
 *
 *	dev:	Input:	dev_t of property.
 *	dip:	Input:	dev_info_t pointer of child.
 *	flags:	Input:	Possible flag modifiers are:
 *		DDI_PROP_DONTPASS:	Don't pass to parent if prop not found.
 *		DDI_PROP_CANSLEEP:	Memory allocation may sleep.
 *	name:	Input:	name of property.
 *	valuep:	Output:	Addr of callers buffer pointer.
 *	lengthp:Output:	*lengthp will contain prop length on exit.
 *
 * 	Possible Returns:
 *
 *		DDI_PROP_SUCCESS:	Prop found and returned.
 *		DDI_PROP_NOT_FOUND:	Prop not found
 *		DDI_PROP_UNDEFINED:	Prop explicitly undefined.
 *		DDI_PROP_NO_MEMORY:	Prop found, but unable to alloc mem.
 */

int
ddi_getlongprop(dev_t dev, dev_info_t *dip, int flags,
	char *name, caddr_t valuep, int *lengthp);

/*
 *
 * ddi_getlongprop_buf:		Get long prop into pre-allocated callers
 *				buffer. (no memory allocation by provider).
 *
 *	dev:	Input:	dev_t of property.
 *	dip:	Input:	dev_info_t pointer of child.
 *	flags:	Input:	DDI_PROP_DONTPASS or NULL
 *	name:	Input:	name of property
 *	valuep:	Input:	ptr to callers buffer.
 *	lengthp:I/O:	ptr to length of callers buffer on entry,
 *			actual length of property on exit.
 *
 *	Possible returns:
 *
 *		DDI_PROP_SUCCESS	Prop found and returned
 *		DDI_PROP_NOT_FOUND	Prop not found
 *		DDI_PROP_UNDEFINED	Prop explicitly undefined.
 *		DDI_PROP_BUF_TOO_SMALL	Prop found, callers buf too small,
 *					no value returned, but actual prop
 *					length returned in *lengthp
 *
 */

int
ddi_getlongprop_buf(dev_t dev, dev_info_t *dip, int flags,
	char *name, caddr_t valuep, int *lengthp);

/*
 * Integer/boolean sized props.
 *
 * Call is value only... returns found boolean or int sized prop value or
 * defvalue if prop not found or is wrong length or is explicitly undefined.
 * Only flag is DDI_PROP_DONTPASS...
 *
 * By convention, this interface returns boolean (0) sized properties
 * as value (int)1.
 */

int
ddi_getprop(dev_t dev, dev_info_t *dip, int flags, char *name, int defvalue);

/*
 * Get prop length interface: flags are 0 or DDI_PROP_DONTPASS
 * if returns DDI_PROP_SUCCESS, length returned in *lengthp.
 */

int
ddi_getproplen(dev_t dev, dev_info_t *dip, int flags, char *name, int *lengthp);


/*
 * Interface to create/modify a managed property on child's behalf...
 * Only flag is DDI_PROP_CANSLEEP to allow memory allocation to sleep
 * if no memory available for internal prop structure.  Long property
 * (non integer sized) value references are not copied.
 *
 * Define property with DDI_DEV_T_NONE dev_t for properties not associated
 * with any particular dev_t. Use the same dev_t when modifying or undefining
 * a property.
 *
 * No guarantee on order of property search, so don't mix the same
 * property name with wildcard and non-wildcard dev_t's.
 */

/*
 * ddi_prop_create:	Define a managed property:
 */

int
ddi_prop_create(dev_t dev, dev_info_t *dip, int flag,
	char *name, caddr_t value, int length);

/*
 * ddi_prop_modify:	Modify a managed property value
 */

int
ddi_prop_modify(dev_t dev, dev_info_t *dip, int flag,
	char *name, caddr_t value, int length);

/*
 * ddi_prop_remove:	Undefine a managed property:
 */

int
ddi_prop_remove(dev_t dev, dev_info_t *dip, char *name);

/*
 * ddi_prop_remove_all:		Used before unloading a driver to remove
 *				all properties. (undefines all dev_t's props.)
 *				Also removes `undefined' prop defs.
 */

void
ddi_prop_remove_all(dev_info_t *dip);


/*
 * ddi_prop_undefine:	Explicitly undefine a property.  Property
 *			searches which match this property return
 *			the error code DDI_PROP_UNDEFINED.
 *
 *			Use ddi_prop_remove to negate effect of
 *			ddi_prop_undefine
 */

int
ddi_prop_undefine(dev_t dev, dev_info_t *dip, int flag, char *name);


/*
 * The default ddi_bus_prop_op wrapper...
 */

int
ddi_bus_prop_op(dev_t dev, dev_info_t *dip, dev_info_t *ch_dip,
	ddi_prop_op_t prop_op, int mod_flags,
	char *name, caddr_t valuep, int *lengthp);


/*
 * Routines to traverse the tree of dev_info nodes.
 * The general idea of these functions is to provide
 * various tree traversal utilities. For each node
 * that the tree traversal function finds, a caller
 * supplied function is called with arguments of
 * the current node and a caller supplied argument.
 * The caller supplied function should return one
 * of the integer values defined below which will
 * indicate to the tree traversal function whether
 * the traversal should be continued, and if so, how,
 * or whether the traversal should terminate.
 */

/*
 * This general-purpose routine traverses the tree of dev_info nodes,
 * starting from the given node, and calls the given function for each
 * node that it finds with the current node and the pointer arg (which
 * can point to a structure of information that the function
 * needs) as arguments.
 *
 * It does the walk a layer at a time, not depth-first.
 *
 * The given function must return one of the values defined above.
 *
 */

void
ddi_walk_devs(dev_info_t *, int (*)(dev_info_t *, void *), void *);

/*
 * Routines to get at elements of the dev_info structure
 */

/*
 * ddi_node_name gets the device's 'name' from the device node.
 *
 * ddi_binding_name gets the string the OS used to bind the node to a driver,
 * in certain cases, the binding name may be different from the node name,
 * if the node name does not name a specific device driver.
 *
 * ddi_get_name is a synonym for ddi_binding_name().
 */
char *
ddi_get_name(dev_info_t *dip);

char *
ddi_binding_name(dev_info_t *dip);

const char *
ddi_driver_name(dev_info_t *dip);

char *
ddi_node_name(dev_info_t *dip);

int
ddi_get_nodeid(dev_info_t *dip);

int
ddi_get_instance(dev_info_t *dip);

struct dev_ops *
ddi_get_driver(dev_info_t *dip);

void
ddi_set_driver(dev_info_t *dip, struct dev_ops *devo);

void
ddi_set_driver_private(dev_info_t *dip, caddr_t data);

caddr_t
ddi_get_driver_private(dev_info_t *dip);

/*
 * ddi_dev_is_needed tells system that a device is about to use a
 * component. Returns when component is ready.
 */
int
ddi_dev_is_needed(dev_info_t *dip, int cmpt, int level);

/*
 * check if DDI_SUSPEND may result in power being removed from a device.
 */
int
ddi_removing_power(dev_info_t *dip);

/*
 *  (Obsolete) power entry point
 */
int
ddi_power(dev_info_t *dip, int cmpt, int level);

/*
 * ddi_get_parent requires that the branch of the tree with the
 * node be held (ddi_hold_installed_driver) or that the devinfo tree
 * lock be held
 */
dev_info_t *
ddi_get_parent(dev_info_t *dip);

/*
 * ddi_get_child and ddi_get_next_sibling require that the devinfo
 * tree lock be held
 */
dev_info_t *
ddi_get_child(dev_info_t *dip);

dev_info_t *
ddi_get_next_sibling(dev_info_t *dip);

dev_info_t *
ddi_get_next(dev_info_t *dip);

void
ddi_set_next(dev_info_t *dip, dev_info_t *nextdip);

/*
 * dev_info manipulation functions
 */

/*
 * Add and remove child devices. These are part of the system framework.
 *
 * ddi_add_child creates a dev_info structure with the passed name,
 * nodeid and instance arguments and makes it a child of pdip. Devices
 * that are known directly by the hardware have real nodeids; devices
 * that are software constructs use the defined DEVI_PSEUDO_NODEID
 * for the node id.
 *
 * ddi_remove_node removes the node from the tree. This fails if this
 * child has children. Parent and driver private data should already
 * be released (freed) prior to calling this function.  If flag is
 * non-zero, the child is removed from it's linked list of instances.
 *
 * ddi_append_dev takes an existing dev_info and makes it
 * a child of another dev_info. ddi_append_dev is an internal function
 * that may be only be called via ddi_add_child.
 */

dev_info_t *
ddi_add_child(dev_info_t *pdip, char *name, uint_t nodeid, uint_t instance);

int
ddi_remove_child(dev_info_t *dip, int flag);

void
ddi_append_dev(dev_info_t *pdip, dev_info_t *cdip);

/*
 * Given the major number for a driver, make sure that dev_info nodes
 * are created form the driver's hwconf file, the driver for the named
 * device is loaded and attached, as well as any drivers for parent devices.
 * Return a pointer to the driver's dev_ops struct with the dev_ops held.
 * Note - Callers must release the dev_ops.
 *
 * When a driver is held, the branch of the devinfo tree from any of the
 * drivers devinfos to the root node are automatically held.  This only
 * applies to tree traversals up (and back down) the tree following the
 * parent pointers.
 */

struct dev_ops *
ddi_hold_installed_driver(major_t major);

void
ddi_rele_driver(major_t);

struct dev_ops *
ddi_hold_devi(dev_info_t *);

void
ddi_rele_devi(dev_info_t *);

/*
 * Given the identifier string passed, make sure that dev_info nodes
 * are created form the driver's hwconf file, the driver for the named
 * device is loaded and attached, as well as any drivers for parent devices.
 *
 * Note that the driver is not held and is subject to being removed the instant
 * this call completes.  You probably really want ddi_hold_installed_driver.
 */

int
ddi_install_driver(char *idstring);

/*
 * Routines that return specific nodes
 */

dev_info_t *
ddi_root_node(void);

/*
 * Given a name and an instance number, find and return the
 * dev_info from the current state of the device tree.
 *
 * If instance number is -1, return the first named instance.
 *
 * The flag hasdriver, if 1, means exclude all
 * instances of this device that don't have an
 * attached driver.
 *
 * Requires that the devinfo tree be locked.
 * If hasdriver is non-zero, the driver must be held.
 */

dev_info_t *
ddi_find_devinfo(char *name, int instance, int hasdriver);

/*
 * DMA Mapping Setup
 *
 * The basic interface function is ddi_dma_setup(). This function
 * is to designed to allow a DMA mapping to be established to a
 * memory object. This function returns DDI_DMA_MAPPED if the
 * request was successfully filled. If this occurs, then the
 * argument handlep is filled in. This value is the DMA handle
 * for the mapping, and is used in a variety of other functions.
 * The handle is an opaque handle on the mapping, and no further
 * information may be inferred from it by the caller.
 *
 * Specifics of arguments to ddi_dma_setup:
 *
 * dip - devinfo pointer, which identifies the base device that wishes
 * to establish a dma mapping. The device may either be a leaf device,
 * or a device which is both a leaf and a nexus (e.g., a device which
 * has a dma engine but no children devices).
 *
 * dmareqp - pointer to a dma request structure. This structure contains
 * all the info necessary to establish the mapping (see <sys/ddidmareq.h>).
 * This structure may be impermanent, as its information is copied and
 * saved, if necessary, by implementation specific functions. The caller
 * is responsible for filling in the dmar_flags, dmar_length, dmar_type,
 * dmar_addr_un, dmar_fp and dmar_arg fields. Any other elements of the
 * ddi_dma_req structure should neither be examined or modified by the
 * caller.
 *
 * handlep - this is a pointer to a ddi_dma_handle_t. It is the callers
 * responsibility to hang on to this handle, because it becomes the token
 * used in all other DDI dma functions. If the handle pointer is NULL,
 * then no mapping is made, and the call is being used by the caller
 * to simply determine whether such a mapping *could* be made.
 *
 * Discussion of DMA resouce callback functions:
 *
 * If a request could not be filled, it was because either there were
 * not enough mapping resources available to satisfy the request, and the
 * dmar_fp field was not set to DDI_DMA_SLEEP, or the mapping could not
 * be established at all (DDI_DMA_NOMAPPING) due to a basic inability of
 * available hardware to map the object. Callers should be prepared to deal
 * with all possible returns. It is suggested that the appropriate system
 * error number for the DDI_DMA_NOMAPPING returns is EFAULT.
 *
 * If the caller does not care whether a DMA mapping can be set up now,
 * the caller should set the field dmar_fp to DDI_DMA_DONTWAIT. This
 * implies that the caller will appropriately deal with resource
 * exhaustion.
 *
 * If the caller either cannot or does not wish to sleep awaiting mapping
 * resources, the caller may specify, via the field dmar_fp, a function to
 * call with the argument specified in dmar_arg, when resources might have
 * become available. The callback function will be called from interrupt
 * context, but in such a fashion to guarantee that spl blocking (in systems
 * that use this method of data protection) by the caller will not be
 * bypassed.
 *
 *
 * When function specified via dmar_fp is called, it may attempt to try and get
 * the mapping again. If it succeeds in getting the mapping, or does not need
 * to get the mapping any more, it must return 1. If it tries to get the
 * mapping but fails to do so, and it wants to be called back later, it
 * must return 0.
 *
 * Failure to observe this protocol will have unpredictable results.
 *
 * The callback function must provide its own data structure integrity
 * when it is invoked.
 */

int
ddi_dma_setup(dev_info_t *dip, struct ddi_dma_req *dmareqp,
	ddi_dma_handle_t *handlep);

/*
 * The following three functions are convenience wrappers for ddi_dma_setup().
 */

int
ddi_dma_addr_setup(dev_info_t *dip, struct as *as, caddr_t addr, size_t len,
	uint_t flags, int (*waitfp)(), caddr_t arg,
	ddi_dma_lim_t *limits, ddi_dma_handle_t *handlep);

int
ddi_dma_buf_setup(dev_info_t *dip, struct buf *bp, uint_t flags,
	int (*waitfp)(), caddr_t arg, ddi_dma_lim_t *limits,
	ddi_dma_handle_t *handlep);

/*
 * Kernel addressability of the DMA object
 *
 * It might often be very useful to be able to get an IU mapping
 * to the object which has DMA active to/from it. In fact, it might
 * even really be a requirement.
 *
 * The cacheability of the object with respect to I/O and I/U caches
 * is affected by this function as follows:
 *
 *	If a kernel virtual mapping to the object owned by the handle
 *	existed already, and is IU cacheable, then the extant mapping
 *	is locked and returned in kaddrp. By inference, kaddrp will
 *	be an IU cacheable reference.
 *
 *	If a kernel virtual mapping to the object owned by the handle
 *	existed already, and is not IU cacheable, then the extant mapping
 *	is locked and returned in kaddrp. By inference, kaddrp will
 *	*not* be an IU cacheable reference.
 *
 *	If a kernel virtual mapping to the object owned by the handle
 *	does not exist already, a mapping will be created that will
 *	*not* be an IU cacheable reference.
 *
 *	The IO cacheability of the object owned by the handle is ignored
 *	and unaffected.
 *
 * This function returns the mapping values as describe above.
 *
 * When the DMA object owned by handle is freed (by ddi_dma_free()- see
 * below), any mappings created by ddi_dma_kvaddrp() cease to be valid.
 * This will be the convention that drivers must follow, as it will be
 * impossible to enforce this programmatically.
 */

int
ddi_dma_kvaddrp(ddi_dma_handle_t, off_t, size_t, caddr_t *);


/*
 * Device addressibility of the DMA object
 *
 * The handle that identifies an object mapped for DMA is an opaque entity.
 * When a device driver wishes to load its dma engine with the appropriate
 * values for transferring data to the mapped object, it has to get the
 * value. Since the exact shape and form of this address is device specific,
 * the value returned is a 'cookie' that each device may then interpret
 * as it needs to. See <sys/dditypes.h> for the form of what the DMA cookie
 * looks like.
 *
 * Returns DDI_SUCCESS for successful cookie generation,
 * or DDI_FAILURE if it cannot generate the DMA cookie.
 */

int
ddi_dma_htoc(ddi_dma_handle_t handle, off_t off, ddi_dma_cookie_t *cookiep);

/*
 * Given a DMA cookie, return its offset within the object referred to
 * by the DMA handle. This is so at the end of a dma transfer, the device
 * may take its specific ending address and find out how far into the
 * memory object described by the handle the device got.
 */

int
ddi_dma_coff(ddi_dma_handle_t handle, ddi_dma_cookie_t *cookiep, off_t *offp);

/*
 * DMA mapping manipulation
 *
 * It may be desirable or convenient for some devices to allow partial
 * mapping of an object for dma. This allows the mapping for DMA of
 * arbitrarily large objects since only a portion of the object may
 * be mapped for DMA at any point in time.
 *
 * In order to support this as well as other operations, the paradigm
 * of a 'mapping window' is defined here. The object to be mapped has
 * attributes of location and length. A window can be established upon
 * this object. The window has attributes of offset (from the base mapping
 * of the object) and length. It is assumed that length and offset are
 * positive with respect to the base of the mapped object.
 *
 * In order to get support for such a window, the flag DDI_DMA_PARTIAL
 * must be set in the request flags when the object is mapped for DMA.
 * Each implementation may elect whether or not to support such an
 * operation. Each implementation may also choose to ignore the request
 * for a PARTIAL mapping and either reject the mapping of the object
 * for being too big (DDI_DMA_TOOBIG) or may map the entire object.
 * The caller who asks the object to be mapped for DMA will know
 * whether a partial mapping has been made by receiving the qualified
 * return value of DDI_DMA_PARTIAL_MAP instead of DDI_DMA_MAPPED.
 * All dma window functions will return DDI_FAILURE if the object
 * is not mapped partially.
 *
 * All other DDI dma functions (except ddi_dma_Free) operate *only* on
 * the mapped portion of the object. That is, functions such as ddi_dma_sync,
 * ddi_dma_segtocookie, and so on, only operate on the currently mapped
 * window.
 */

#if defined(sparc) || defined(__sparc)

/*
 * ddi_dma_movwin - Move window from current offset/length to new
 * offset/length. Returns DDI_SUCCESS if able to do so, else returns
 * DDI_FAILURE if unable to do so, or the new window would be out of bounds
 * or the object isn't set up for windows. If length is (off_t) -1, the
 * If the optional cp argument is specified, an implicit ddi_dma_htoc
 * is done to fill that in. The new offset and length will be returned
 * in the arguments *offp and *lenp (resp).
 *
 * In this implementation, only fixed width windows are used. It is
 * recommended that the windowsize should be retrieved via the function
 * ddi_dma_curwin (below) and that used to specify new offsets and lengths
 * since the window will be fixed at that size and will only move modulo
 * winsize.
 *
 * The caller must guarantee that their device's dma engine is quiescent
 * with respect to the current DMA window.
 *
 * The implementation will try to be rapid with respect to moving a window,
 * but since an appropriate ddi_dma_sync() is likely to be done, there
 * will be no guaranteed latency. In practice this should not be too
 * horrible, but don't depend upon any particular latency.
 */

int
ddi_dma_movwin(ddi_dma_handle_t, off_t *offp, size_t *lenp, ddi_dma_cookie_t *);

#endif

/*
 * ddi_dma_curwin - report the current offset/length of the window.
 *
 * Returns DDI_SUCCESS if offset and length
 * successfully established, else DDI_FAILURE.
 */

int
ddi_dma_curwin(ddi_dma_handle_t handle, off_t *offp, size_t *lenp);

/*
 * Get next dma window
 *
 * ddi_dma_nextwin takes a handle and a window, and fills in a pointer to
 * the next window within the object. If win is "NULL", a pointer to the
 * first window within the object is filled in.
 *
 * Returns	DDI_SUCCESS if successfully filled in the window pointer,
 *		DDI_DMA_STALE if win does not refer to the currently active
 *				 window,
 *		DDI_DMA_DONE else there is no next window.
 */

int
ddi_dma_nextwin(ddi_dma_handle_t, ddi_dma_win_t, ddi_dma_win_t *);

/*
 * Get next segment
 *
 * ddi_dma_nextseg takes a window and a segment and fills in a pointer to
 * the next segment within the window. If seg is "NULL", a pointer to the
 * first segment within the window is filled in.
 *
 * Returns	DDI_SUCCESS if successfully filled in the segment pointer,
 *		DDI_DMA_STALE if win does not refer to the currently active
 *				 window.
 *		DDI_DMA_DONE else there is no next segment.
 */

int
ddi_dma_nextseg(ddi_dma_win_t, ddi_dma_seg_t, ddi_dma_seg_t *);

/*
 * Segment to cookie
 *
 * ddi_dma_segtocookie takes a segment and fills in the cookie pointed
 * to by cookiep with the appropriate address, length and bus type to be
 * used to program the DMA engine. ddi_dma_segtocookie also fills in the
 * range within the object (specified by <off, len>) this particular
 * segment is mapping. <off, len> are filled in to give some control
 * where in the object the current dma transfer is active.
 *
 * Returns	DDI_SUCCESS if successfully filled in all values,
 * else		DDI_FAILURE
 */

int
ddi_dma_segtocookie(ddi_dma_seg_t, off_t *, off_t *, ddi_dma_cookie_t *);

/*
 * Synchronization of I/O with respect to various
 * caches and system write buffers.
 *
 * Done at varying points during an I/O transfer (including at the
 * removal of an I/O mapping).
 *
 * Due to the support of systems with write buffers which may
 * not be able to be turned off, this function *must* used at
 * any point in which data consistency might be required.
 *
 * Generally this means that if a memory object has multiple mappings
 * (both for I/O, as described by the handle, and the IU, via, e.g.
 * a call to ddi_dma_kvaddrp), and one mapping may have been
 * used to modify the memory object, this function must be called
 * to ensure that the modification of the memory object is
 * complete, as well as possibly to inform other mappings of
 * the object that any cached references to the object are
 * now stale (and flush or invalidate these stale cache references
 * as necessary).
 *
 * The function ddi_dma_sync() provides the general interface with
 * respect to this capability. Generally, ddi_dma_free() (below) may
 * be used in preference to ddi_dma_sync() as ddi_dma_free() calls
 * ddi_dma_sync().
 *
 * Returns 0 if all caches that exist and are specified by cache_flags
 * are succesfully operated on, else -1.
 *
 * The argument offset specifies an offset into the mapping of the mapped
 * object in which to perform the synchronization. It will be silently
 * truncated to the granularity of underlying cache line sizes as
 * appropriate.
 *
 * The argument len specifies a length starting from offset in which to
 * perform the synchronization. A value of (uint_t) -1 means that the length
 * proceeds from offset to the end of the mapping. The length argument
 * will silently rounded up to the granularity of underlying cache line
 * sizes  as appropriate.
 *
 * The argument flags specifies what to synchronize (the device's view of
 * the object or the cpu's view of the object).
 *
 * Inquiring minds want to know when ddi_dma_sync should be used:
 *
 * +	When an object is mapped for dma, assume that an
 *	implicit ddi_dma_sync() is done for you.
 *
 * +	When an object is unmapped (ddi_dma_free()), assume
 *	that an implicit ddi_dma_sync() is done for you.
 *
 * +	At any time between the two times above that the
 *	memory object may have been modified by either
 *	the DMA device or a processor and you wish that
 *	the change be noticed by the master that didn't
 *	do the modifying.
 *
 * Clearly, only the third case above requires the use of ddi_dma_sync.
 *
 * Inquiring minds also want to know which flag to use:
 *
 * +	If you *modify* with a cpu the object, you use
 *	ddi_dma_sync(...DDI_DMA_SYNC_FORDEV) (you are making sure
 *	that the DMA device sees the changes you made).
 *
 * +	If you are checking, with the processor, an area
 *	of the object that the DMA device *may* have modified,
 *	you use ddi_dma_sync(....DDI_DMA_SYNC_FORCPU) (you are
 *	making sure that the processor(s) will see the changes
 *	that the DMA device may have made).
 */

int
ddi_dma_sync(ddi_dma_handle_t handle, off_t offset, size_t len, uint_t flags);

/*
 * DMA mapping de-allocation
 *
 * When an I/O transfer completes, the resources required to map the
 * object for DMA should be completely released. As a side effect,
 * various cache synchronization might need to occur (see above).
 *
 * Returns DDI_SUCCESS if the all underlying caches are successfully
 * flushed, else DDI_FAILURE.
 *
 */

int
ddi_dma_free(ddi_dma_handle_t handle);

/*
 * Device constraint cognizant kernel memory allocation- consistent access.
 *
 * IOPB allocation and de-allocation
 *
 * An IOPB allocation allocates some primary memory such that both
 * the kernel and the specified DMA device might be able to access it in a
 * non-cacheable (otherwise known as byte-consistent or non-streaming mode)
 * fashion. The allocation will obey the beginning alignment and padding
 * constraints as specified in the intial limits argument and as subsequently
 * modified by intervening parents. The limits argument may be NULL, in
 * which case the system picks a reasonable beginning limits.
 *
 * A kernel virtual address to the allocated primary memory is returned,
 * but no DMA mapping to the object is established (drivers must use the
 * ddi_dma_map() routines for that).
 *
 * If no iopb space can be allocated, DDI_FAILURE is returned.
 */

int
ddi_iopb_alloc(dev_info_t *dip, ddi_dma_lim_t *limits, uint_t length,
	caddr_t *iopbp);

/*
 * Deallocate an IOPB kernel virtual mapping.
 */

void
ddi_iopb_free(caddr_t iopb);

/*
 * Device constraint cognizant kernel memory allocation- streaming access.
 *
 * Similar to ddi_iopb_alloc, but for priamry memory that is intended
 * to be accessed in a streaming fashion. The allocation will obey the
 * beginning alignment and padding constraints as specified in the intial
 * limits argument and as subsequently modified by intervening parents.
 * The limits argument may be NULL, in which case the system picks a
 * reasonable beginning limits.
 *
 * A flags value of 0x1 indicates whether the caller can wait for
 * memory to become available. Other bits in the flags argument
 * are reserved for future use and must be zero.
 *
 * Upon return from a successful call, the new real length of
 * the allocation is returned (for use in mapping the memory
 * later).
 */

int
ddi_mem_alloc(dev_info_t *dip, ddi_dma_lim_t *limits, uint_t length,
	uint_t flags, caddr_t *kaddrp, uint_t *real_length);

/*
 * Free the memory allocated via ddi_mem_alloc().
 *
 * Note that passing an address not allocated via ddi_mem_alloc()
 * will panic the system.
 */

void
ddi_mem_free(caddr_t kaddr);

/*
 * Dma alignment, minimum transfers sizes, and burst sizes allowed.
 * Some with tears, some without.
 */

/*
 * Return the allowable DMA burst size for the object mapped by handle.
 * The burst sizes will returned in an integer that encodes power
 * of two burst sizes that are allowed in bit encoded format. For
 * example, a transfer that could allow 1, 2, 4, 8 and 32 byte bursts
 * would be encoded as 0x2f. A transfer that could be allowed as solely
 * a halfword (2 byte) transfers would be returned as 0x2.
 */

int
ddi_dma_burstsizes(ddi_dma_handle_t handle);

/*
 * Return the required beginning alignment for a transfer and
 * the minimum sized effect a transfer would have. The beginning
 * alignment will be some power of two. The minimum sized effect
 * indicates, for writes, how much of the mapped object will be
 * affected by the minimum access and for reads how much of the
 * mapped object will accessed.
 */

int
ddi_dma_devalign(ddi_dma_handle_t handle, uint_t *alignment, uint_t *mineffect);

/*
 * Like ddi_dma_devalign, but without having to map the object.
 * The object is assumed to be primary memory, and it is assumed
 * a minimum effective transfer is also the appropriate alignment
 * to be using. The streaming flag, if non-zero, indicates that the
 * returned value should be modified to account for streaming mode
 * accesses (e.g., with I/O caches enabled). The initial value
 * is passed by the requestor if it has a dma engine that has
 * a minimum cycle constraint (or, for streaming mode, the most
 * efficient size).
 */

int
ddi_iomin(dev_info_t *dip, int initial, int streaming);

/*
 * Given two DMA limit structures, apply the limitations
 * of one to the other, following the rules of limits
 * and the wishes of the caller.
 *
 * The rules of dma limit structures are that you cannot
 * make things *less* restrictive as you apply one set
 * of limits to another.
 *
 */

void
ddi_dmalim_merge(ddi_dma_lim_t *limit, ddi_dma_lim_t *modifier);

/*
 * Merge DMA attributes
 */

void
ddi_dma_attr_merge(ddi_dma_attr_t *attr, ddi_dma_attr_t *mod);

/*
 * Allocate a DMA handle
 */

int
ddi_dma_alloc_handle(dev_info_t *dip, ddi_dma_attr_t *attr,
	int (*waitfp)(caddr_t), caddr_t arg,
	ddi_dma_handle_t *handlep);

/*
 * Free DMA handle
 */

void
ddi_dma_free_handle(ddi_dma_handle_t *handlep);

/*
 * Allocate memory for DMA transfers
 */

int
ddi_dma_mem_alloc(ddi_dma_handle_t handle, size_t length,
	ddi_device_acc_attr_t *accattrp, uint_t xfermodes,
	int (*waitfp)(caddr_t), caddr_t arg, caddr_t *kaddrp,
	size_t *real_length, ddi_acc_handle_t *handlep);

/*
 * Free DMA memory
 */

void
ddi_dma_mem_free(ddi_acc_handle_t *hp);

/*
 * bind address to a DMA handle
 */

int
ddi_dma_addr_bind_handle(ddi_dma_handle_t handle, struct as *as,
	caddr_t addr, size_t len, uint_t flags,
	int (*waitfp)(caddr_t), caddr_t arg,
	ddi_dma_cookie_t *cookiep, uint_t *ccountp);

/*
 * bind buffer to DMA handle
 */

int
ddi_dma_buf_bind_handle(ddi_dma_handle_t handle, struct buf *bp,
	uint_t flags, int (*waitfp)(caddr_t), caddr_t arg,
	ddi_dma_cookie_t *cookiep, uint_t *ccountp);

/*
 * unbind mapping object to handle
 */

int
ddi_dma_unbind_handle(ddi_dma_handle_t handle);

/*
 * get next DMA cookie
 */

void
ddi_dma_nextcookie(ddi_dma_handle_t handle, ddi_dma_cookie_t *cookiep);

/*
 * get number of DMA windows
 */

int
ddi_dma_numwin(ddi_dma_handle_t handle, uint_t *nwinp);

/*
 * get specific DMA window
 */

int
ddi_dma_getwin(ddi_dma_handle_t handle, uint_t win, off_t *offp,
	size_t *lenp, ddi_dma_cookie_t *cookiep, uint_t *ccountp);

/*
 * activate 64 bit SBus support
 */

int
ddi_dma_set_sbus64(ddi_dma_handle_t handle, ulong_t burstsizes);

/*
 * Miscellaneous functions
 */

/*
 * ddi_report_dev:	Report a successful attach.
 */

void
ddi_report_dev(dev_info_t *dev);

/*
 * ddi_dev_regsize
 *
 *	If the device has h/w register(s), report
 *	the size, in bytes, of the specified one into *resultp.
 *
 *	Returns DDI_FAILURE if there are not registers,
 *	or the specified register doesn't exist.
 */

int
ddi_dev_regsize(dev_info_t *dev, uint_t rnumber, off_t *resultp);

/*
 * ddi_dev_nregs
 *
 *	If the device has h/w register(s), report
 *	how many of them that there are into resultp.
 *	Return DDI_FAILURE if the device has no registers.
 */

int
ddi_dev_nregs(dev_info_t *dev, int *resultp);

/*
 * ddi_dev_nintrs
 *
 *	If the device has h/w interrupt(s), report
 *	how many of them that there are into resultp.
 *	Return DDI_FAILURE if the device has no interrupts.
 */

int
ddi_dev_nintrs(dev_info_t *dev, int *resultp);

/*
 * ddi_dev_is_sid
 *
 *	If the device is self-identifying, i.e.,
 *	has already been probed by a smart PROM
 *	(and thus registers are known to be valid)
 *	return DDI_SUCCESS, else DDI_FAILURE.
 */


int
ddi_dev_is_sid(dev_info_t *dev);

/*
 * ddi_slaveonly
 *
 *	If the device is on a bus that precludes
 *	the device from being either a dma master or
 *	a dma slave, return DDI_SUCCESS.
 */

int
ddi_slaveonly(dev_info_t *);


/*
 * ddi_dev_affinity
 *
 *	Report, via DDI_SUCCESS, whether there exists
 *	an 'affinity' between two dev_info_t's. An
 *	affinity is defined to be either a parent-child,
 *	or a sibling relationship such that the siblings
 *	or in the same part of the bus they happen to be
 *	on.
 */

int
ddi_dev_affinity(dev_info_t *deva, dev_info_t *devb);


/*
 * ddi_set_callback
 *
 *	Set a function/arg pair into the callback list identified
 *	by listid. *listid must always initially start out as zero.
 */

void
ddi_set_callback(int (*funcp)(caddr_t), caddr_t arg, uintptr_t *listid);

/*
 * ddi_run_callback
 *
 *	Run the callback list identified by listid.
 */

void
ddi_run_callback(uintptr_t *listid);

/*
 * More miscellaneous
 */

int
nochpoll(dev_t dev, short events, int anyyet, short *reventsp,
	struct pollhead **phpp);

dev_info_t *
nodevinfo(dev_t dev, int otyp);

int
ddi_no_info(dev_info_t *dip, ddi_info_cmd_t infocmd, void *arg, void **result);

int
ddifail(dev_info_t *devi, ddi_attach_cmd_t cmd);

int
ddi_no_dma_map(dev_info_t *dip, dev_info_t *rdip,
    struct ddi_dma_req *dmareqp, ddi_dma_handle_t *handlep);

int
ddi_no_dma_allochdl(dev_info_t *dip, dev_info_t *rdip, ddi_dma_attr_t *attr,
    int (*waitfp)(caddr_t), caddr_t arg, ddi_dma_handle_t *handlep);

int
ddi_no_dma_freehdl(dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle);

int
ddi_no_dma_bindhdl(dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle, struct ddi_dma_req *dmareq,
    ddi_dma_cookie_t *cp, uint_t *ccountp);

int
ddi_no_dma_unbindhdl(dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle);

int
ddi_no_dma_flush(dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle, off_t off, size_t len,
    uint_t cache_flags);

int
ddi_no_dma_win(dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle, uint_t win, off_t *offp,
    size_t *lenp, ddi_dma_cookie_t *cookiep, uint_t *ccountp);

int
ddi_no_dma_mctl(register dev_info_t *dip, dev_info_t *rdip,
    ddi_dma_handle_t handle, enum ddi_dma_ctlops request,
    off_t *offp, size_t *lenp, caddr_t *objp, uint_t flags);

void
ddivoid();

cred_t *
ddi_get_cred(void);

clock_t
ddi_get_lbolt(void);

time_t
ddi_get_time(void);

pid_t
ddi_get_pid(void);

void
swab(void *src, void *dst, size_t nbytes);

minor_t
ddi_getiminor(dev_t dev);

int
ddi_create_minor_node(dev_info_t *dip, char *name, int spec_type,
    minor_t minor_num, char *node_type, int flag);

void
ddi_remove_minor_node(dev_info_t *dip, char *name);

int
ddi_in_panic(void);

int
ddi_streams_driver(dev_info_t *dip);

/*
 * DDI wrappers for ffs and fls
 */
int
ddi_ffs(long mask);

int
ddi_fls(long mask);

/*
 * The next five routines comprise generic storage management utilities
 * for driver soft state structures.
 */

/*
 * Allocate a set of pointers to 'n_items' objects of size 'size'
 * bytes.  Each pointer is initialized to nil. 'n_items' is a hint i.e.
 * zero is allowed.
 */
int
ddi_soft_state_init(void **state_p, size_t size, size_t n_items);

/*
 * Allocate a state structure of size 'size' to be associated
 * with item 'item'.
 */
int
ddi_soft_state_zalloc(void *state, int item);

/*
 * Fetch a pointer to the allocated soft state structure
 * corresponding to 'item.'
 */
void *
ddi_get_soft_state(void *state, int item);

/*
 * Free the state structure corresponding to 'item.'
 */
void
ddi_soft_state_free(void *state, int item);

/*
 * Free the handle, and any associated soft state structures.
 */
void
ddi_soft_state_fini(void **state_p);

/*
 * Set the addr field of the name in dip to name
 */
void
ddi_set_name_addr(dev_info_t *dip, char *name);

/*
 * Get the address part of the name.
 */
char *
ddi_get_name_addr(dev_info_t *dip);

void
ddi_set_parent_data(dev_info_t *dip, caddr_t pd);

caddr_t
ddi_get_parent_data(dev_info_t *dip);

int
ddi_initchild(dev_info_t *parent, dev_info_t *proto);

int
ddi_uninitchild(dev_info_t *dip);

void
ddi_optimize_dtree(dev_info_t *);

major_t
ddi_name_to_major(char *name);

char *
ddi_major_to_name(major_t major);

char *
ddi_deviname(dev_info_t *dip, char *name);

char *
ddi_pathname(dev_info_t *dip, char *path);

int
ddi_dev_pathname(dev_t devt, char *name);

dev_t
ddi_pathname_to_dev_t(char *pathname);

dev_info_t *
ddi_findchild(dev_info_t *p, char *cname, char *caddr);

/*
 * High resolution system timer functions.
 *
 * These functions are already in the kernel (see sys/time.h).
 * The ddi supports the notion of a hrtime_t type and the
 * functions gethrtime, hrtadd, hrtsub and hrtcmp.
 */


/*
 * Nexus wrapper functions
 *
 * These functions are for entries in a bus nexus driver's bus_ops
 * structure for when the driver doesn't have such a function and
 * doesn't wish to prohibit such a function from existing. They
 * may also be called to start passing a request up the dev_info
 * tree.
 */

/*
 * bus_ctl wrapper
 */

int
ddi_ctlops(dev_info_t *d, dev_info_t *r, ddi_ctl_enum_t o, void *a, void *v);

/*
 * bus_dma_map wrapper
 */

int
ddi_dma_map(dev_info_t *dip, dev_info_t *rdip,
	struct ddi_dma_req *dmareqp, ddi_dma_handle_t *handlep);

int
ddi_dma_allochdl(dev_info_t *dip, dev_info_t *rdip, ddi_dma_attr_t *attr,
	int (*waitfp)(caddr_t), caddr_t arg, ddi_dma_handle_t *handlep);

int
ddi_dma_freehdl(dev_info_t *dip, dev_info_t *rdip,
	ddi_dma_handle_t handle);

int
ddi_dma_bindhdl(dev_info_t *dip, dev_info_t *rdip,
	ddi_dma_handle_t handle, struct ddi_dma_req *dmareq,
	ddi_dma_cookie_t *cp, uint_t *ccountp);

int
ddi_dma_unbindhdl(dev_info_t *dip, dev_info_t *rdip,
	ddi_dma_handle_t handle);

int
ddi_dma_flush(dev_info_t *dip, dev_info_t *rdip,
	ddi_dma_handle_t handle, off_t off, size_t len,
	uint_t cache_flags);

int
ddi_dma_win(dev_info_t *dip, dev_info_t *rdip,
	ddi_dma_handle_t handle, uint_t win, off_t *offp,
	size_t *lenp, ddi_dma_cookie_t *cookiep, uint_t *ccountp);

/*
 * bus_dma_ctl wrapper
 */

int
ddi_dma_mctl(dev_info_t *dip, dev_info_t *rdip, ddi_dma_handle_t handle,
	enum ddi_dma_ctlops request, off_t *offp, size_t *lenp,
	caddr_t *objp, uint_t flags);

/*
 * dvma support for networking drivers
 */

unsigned long
dvma_pagesize(dev_info_t *dip);

int
dvma_reserve(dev_info_t *dip,  ddi_dma_lim_t *limp, uint_t pages,
	ddi_dma_handle_t *handlep);

void
dvma_release(ddi_dma_handle_t h);

void
dvma_kaddr_load(ddi_dma_handle_t h, caddr_t a, uint_t len, uint_t index,
	ddi_dma_cookie_t *cp);

void
dvma_unload(ddi_dma_handle_t h, uint_t objindex, uint_t type);

void
dvma_sync(ddi_dma_handle_t h, uint_t objindex, uint_t type);

/*
 * Layered driver support
 */

extern int ddi_copyin(const void *, void *, size_t, int);
extern int ddi_copyout(const void *, void *, size_t, int);

/*
 * Send signals to processes
 */
extern void *proc_ref(void);
extern void proc_unref(void *pref);
extern int proc_signal(void *pref, int sig);

/* I/O port access routines */
extern unsigned char inb(int port);
extern unsigned short inw(int port);
extern unsigned long inl(int port);
extern void repinsb(int port, unsigned char *addr, int count);
extern void repinsw(int port, unsigned short *addr, int count);
extern void repinsd(int port, unsigned long *addr, int count);
extern void outb(int port, unsigned char value);
extern void outw(int port, unsigned short value);
extern void outl(int port, unsigned long value);
extern void repoutsb(int port, unsigned char *addr, int count);
extern void repoutsw(int port, unsigned short *addr, int count);
extern void repoutsd(int port, unsigned long *addr, int count);

/*
 * Console bell routines
 */
extern void ddi_ring_console_bell(clock_t duration);
extern void ddi_set_console_bell(void (*bellfunc)(clock_t duration));

/*
 * Fault-related functions
 */
extern int ddi_check_acc_handle(ddi_acc_handle_t);
extern int ddi_check_dma_handle(ddi_dma_handle_t);
extern void ddi_dev_report_fault(dev_info_t *, ddi_fault_impact_t,
	ddi_fault_location_t, const char *);
extern ddi_devstate_t ddi_get_devstate(dev_info_t *);

#else	/* __STDC__ */
extern int maxphys;
extern void minphys();
extern int physio();
extern void disksort();
extern size_t strlen();
extern char *strcpy();
extern char *strncpy();
extern char *strchr();
extern int strcmp();
extern int strncmp();
extern int bcmp();
extern int stoi();
extern void numtos();
extern void bcopy();
extern void bzero();
extern int ddi_map_regs();
extern void ddi_unmap_regs();
extern int ddi_map();
extern int ddi_apply_range();
extern struct regspec *ddi_rnumber_to_regspec();
extern int ddi_bus_map();
extern int nullbusmap();
extern int common_ddi_peek();
extern int ddi_peekc();
extern int ddi_peek();
extern int ddi_peekl();
extern int ddi_peekd();
extern int common_ddi_poke();
extern int ddi_pokec();
extern int ddi_poke();
extern int ddi_pokel();
extern int ddi_poked();
extern unsigned long ddi_btop();
extern unsigned long ddi_btopr();
extern unsigned long ddi_ptob();
extern int ddi_add_intr();
extern int ddi_add_fastintr();
extern int ddi_add_softintr();
extern void ddi_remove_softintr();
extern void ddi_trigger_softintr();
extern void ddi_remove_intr();
extern int ddi_get_iblock_cookie();
extern int ddi_get_soft_iblock_cookie();
extern unsigned int ddi_enter_critical();
extern void ddi_exit_critical();
extern int devmap_setup();
extern int ddi_devmap_segmap();
extern int devmap_load();
extern int devmap_unload();
extern int devmap_devmem_setup();
extern int devmap_umem_setup();
extern void devmap_set_ctx_timeout();
extern int devmap_default_access();
extern int devmap_umem_remap();
extern int devmap_devmem_remap();
extern int devmap_do_ctxmgt();
extern void *ddi_umem_alloc();
extern void ddi_umem_free();
extern int ddi_umem_lock();
extern void ddi_umem_unlock();
extern struct buf *ddi_umem_iosetup();
extern int ddi_segmap();
extern int ddi_segmap_setup();
extern int ddi_map_fault();
extern int ddi_prop_exists();
extern int ddi_prop_get_int();
extern int ddi_prop_lookup_int_array();
extern int ddi_prop_lookup_string();
extern int ddi_prop_lookup_string_array();
extern int ddi_prop_lookup_byte_array();
extern int ddi_prop_free();
extern int ddi_prop_update_int();
extern int ddi_prop_update_int_array();
extern int ddi_prop_update_string();
extern int ddi_prop_update_string_array();
extern int ddi_prop_update_byte_array();
extern int nopropop();
extern int ddi_prop_op();
extern int ddi_getlongprop();
extern int ddi_getlongprop_buf();
extern int ddi_getprop();
extern int ddi_getproplen();
extern int ddi_prop_create();
extern int ddi_prop_modify();
extern int ddi_prop_remove();
extern void ddi_prop_remove_all();
extern int ddi_prop_undefine();
extern int ddi_bus_prop_op();
extern void ddi_walk_devs();
extern char *ddi_get_name();
extern char *ddi_node_name();
extern char *ddi_binding_name();
extern int ddi_get_nodeid();
extern int ddi_get_instance();
extern struct dev_ops *ddi_get_driver();
extern void ddi_set_driver();
extern void ddi_set_driver_private();
extern int ddi_dev_is_needed();
extern int ddi_removing_power();
extern int ddi_power();
extern caddr_t ddi_get_driver_private();
extern dev_info_t *ddi_get_parent();
extern dev_info_t *ddi_get_child();
extern dev_info_t *ddi_get_next_sibling();
extern dev_info_t *ddi_add_child();
extern int ddi_remove_child();
extern void ddi_append_dev();
extern int ddi_install_driver();
extern dev_info_t *ddi_root_node();
extern dev_info_t *ddi_find_devinfo();
extern int ddi_dma_setup();
extern int ddi_dma_addr_bind_handle();
extern int ddi_dma_buf_bind_handle();
extern void ddi_dma_nextcookie();
extern int ddi_dma_numwin();
extern void ddi_dma_getwin();
extern int ddi_dma_alloc_handle();
extern void ddi_dma_free_handle();
extern int ddi_dma_mem_alloc();
extern void ddi_dma_mem_free();
extern int ddi_dma_addr_setup();
extern int ddi_dma_buf_setup();
extern int ddi_dma_kvaddrp();
extern int ddi_dma_htoc();
extern int ddi_dma_coff();
extern int ddi_dma_movwin();
extern int ddi_dma_curwin();
extern int ddi_dma_nextwin();
extern int ddi_dma_nextseg();
extern int ddi_dma_segtocookie();
extern int ddi_dma_sync();
extern int ddi_dma_free();
extern int ddi_iopb_alloc();
extern void ddi_iopb_free();
extern int ddi_mem_alloc();
extern void ddi_mem_free();
extern int ddi_dma_burstsizes();
extern int ddi_dma_devalign();
extern int ddi_iomin();
extern void ddi_dmalim_merge();
extern int ddi_dma_alloc_handle();
extern void ddi_dma_free_handle();
extern int ddi_dma_mem_alloc();
extern void ddi_dma_mem_free();
extern void ddi_report_dev();
extern int ddi_dev_regsize();
extern int ddi_dev_nregs();
extern int ddi_dev_nintrs();
extern int ddi_dev_is_sid();
extern int ddi_slaveonly();
extern int ddi_dev_affinity();
extern void ddi_set_callback();
extern void ddi_run_callback();
extern int nochpoll();
extern dev_info_t *nodevinfo();
extern int ddifail();
extern int ddi_no_dma_map();
extern int ddi_no_dma_allochdl();
extern int ddi_no_dma_freehdl();
extern int ddi_no_dma_bindhdl();
extern int ddi_no_dma_unbindhdl();
extern int ddi_no_dma_flush();
extern int ddi_no_dma_win();
extern int ddi_no_dma_mctl();
extern void ddivoid();
extern cred_t *ddi_get_cred();
extern clock_t ddi_get_lbolt();
extern time_t ddi_get_time();
extern pid_t ddi_get_pid();
extern void swab();
extern int ddi_in_panic();
extern int ddi_ffs();
extern int ddi_fls();
extern int ddi_soft_state_init();
extern int ddi_soft_state_zalloc();
extern void *ddi_get_soft_state();
extern void ddi_soft_state_free();
extern void ddi_soft_state_fini();
extern int ddi_ctlops();
extern int ddi_dma_map();
extern int ddi_dma_mctl();
extern int ddi_copyin();
extern int ddi_copyout();
extern void *proc_ref();
extern void proc_unref();
extern int proc_signal();
extern unsigned char inb();
extern unsigned short inw();
extern unsigned long inl();
extern void repinsb();
extern void repinsw();
extern void repinsd();
extern void outb();
extern void outw();
extern void outl();
extern void repoutsb();
extern void repoutsw();
extern void repoutsd();
extern unsigned long dvma_pagesize();
extern int dvma_reserve();
extern void dvma_release();
extern void dvma_kaddr_load();
extern void dvma_unload();
extern void dvma_sync();
extern void ddi_ring_console_bell();
extern void ddi_set_console_bell();
extern int ddi_check_acc_handle();
extern int ddi_check_dma_handle();
extern void ddi_dev_report_fault();
extern ddi_devstate_t ddi_get_devstate();
#endif	/* __STDC__ */

/*
 * Miscellaneous redefines
 */
#define	uiophysio	physio

/*
 * utilities - "reg" mapping and all common portable data access functions
 */

/*
 * error code from ddi_regs_map_setup
 */

#define	DDI_REGS_ACC_CONFLICT	(-10)

/*
 * Device address advance flags
 */

#define	 DDI_DEV_NO_AUTOINCR	0x0000
#define	 DDI_DEV_AUTOINCR	0x0001

#ifdef	__STDC__
int
ddi_regs_map_setup(dev_info_t *dip, uint_t rnumber, caddr_t *addrp,
	offset_t offset, offset_t len, ddi_device_acc_attr_t *accattrp,
	ddi_acc_handle_t *handle);

void
ddi_regs_map_free(ddi_acc_handle_t *handle);

/*
 * these are the prototypes for the common portable data access functions
 */

#ifdef _LP64

uint8_t
ddi_get8(ddi_acc_handle_t handle, uint8_t *addr);

uint16_t
ddi_get16(ddi_acc_handle_t handle, uint16_t *addr);

uint32_t
ddi_get32(ddi_acc_handle_t handle, uint32_t *addr);

uint64_t
ddi_get64(ddi_acc_handle_t handle, uint64_t *addr);

void
ddi_rep_get8(ddi_acc_handle_t handle, uint8_t *host_addr, uint8_t *dev_addr,
	size_t repcount, uint_t flags);

void
ddi_rep_get16(ddi_acc_handle_t handle, uint16_t *host_addr, uint16_t *dev_addr,
	size_t repcount, uint_t flags);

void
ddi_rep_get32(ddi_acc_handle_t handle, uint32_t *host_addr, uint32_t *dev_addr,
	size_t repcount, uint_t flags);

void
ddi_rep_get64(ddi_acc_handle_t handle, uint64_t *host_addr, uint64_t *dev_addr,
	size_t repcount, uint_t flags);

void
ddi_put8(ddi_acc_handle_t handle, uint8_t *addr, uint8_t value);

void
ddi_put16(ddi_acc_handle_t handle, uint16_t *addr, uint16_t value);

void
ddi_put32(ddi_acc_handle_t handle, uint32_t *addr, uint32_t value);

void
ddi_put64(ddi_acc_handle_t handle, uint64_t *addr, uint64_t value);

void
ddi_rep_put8(ddi_acc_handle_t handle, uint8_t *host_addr, uint8_t *dev_addr,
	size_t repcount, uint_t flags);
void
ddi_rep_put16(ddi_acc_handle_t handle, uint16_t *host_addr, uint16_t *dev_addr,
	size_t repcount, uint_t flags);
void
ddi_rep_put32(ddi_acc_handle_t handle, uint32_t *host_addr, uint32_t *dev_addr,
	size_t repcount, uint_t flags);

void
ddi_rep_put64(ddi_acc_handle_t handle, uint64_t *host_addr, uint64_t *dev_addr,
	size_t repcount, uint_t flags);

#else /* _ILP32 */

uint8_t
ddi_getb(ddi_acc_handle_t handle, uint8_t *addr);
#define	ddi_get8	ddi_getb

uint16_t
ddi_getw(ddi_acc_handle_t handle, uint16_t *addr);
#define	ddi_get16	ddi_getw

uint32_t
ddi_getl(ddi_acc_handle_t handle, uint32_t *addr);
#define	ddi_get32	ddi_getl

uint64_t
ddi_getll(ddi_acc_handle_t handle, uint64_t *addr);
#define	ddi_get64	ddi_getll

void
ddi_rep_getb(ddi_acc_handle_t handle, uint8_t *host_addr, uint8_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_get8	ddi_rep_getb

void
ddi_rep_getw(ddi_acc_handle_t handle, uint16_t *host_addr, uint16_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_get16	ddi_rep_getw

void
ddi_rep_getl(ddi_acc_handle_t handle, uint32_t *host_addr, uint32_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_get32	ddi_rep_getl

void
ddi_rep_getll(ddi_acc_handle_t handle, uint64_t *host_addr, uint64_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_get64	ddi_rep_getll

void
ddi_putb(ddi_acc_handle_t handle, uint8_t *addr, uint8_t value);
#define	ddi_put8	ddi_putb

void
ddi_putw(ddi_acc_handle_t handle, uint16_t *addr, uint16_t value);
#define	ddi_put16	ddi_putw

void
ddi_putl(ddi_acc_handle_t handle, uint32_t *addr, uint32_t value);
#define	ddi_put32	ddi_putl

void
ddi_putll(ddi_acc_handle_t handle, uint64_t *addr, uint64_t value);
#define	ddi_put64	ddi_putll

void
ddi_rep_putb(ddi_acc_handle_t handle, uint8_t *host_addr, uint8_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_put8	ddi_rep_putb

void
ddi_rep_putw(ddi_acc_handle_t handle, uint16_t *host_addr, uint16_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_put16	ddi_rep_putw

void
ddi_rep_putl(ddi_acc_handle_t handle, uint32_t *host_addr, uint32_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_put32	ddi_rep_putl

void
ddi_rep_putll(ddi_acc_handle_t handle, uint64_t *host_addr, uint64_t *dev_addr,
	size_t repcount, uint_t flags);
#define	ddi_rep_put64	ddi_rep_putll

#endif /* _LP64 */

/*
 * these are special device handling functions
 */
int
ddi_device_zero(ddi_acc_handle_t handle, caddr_t dev_addr,
	size_t bytecount, ssize_t dev_advcnt, uint_t dev_datasz);

int
ddi_device_copy(
	ddi_acc_handle_t src_handle, caddr_t src_addr, ssize_t src_advcnt,
	ddi_acc_handle_t dest_handle, caddr_t dest_addr, ssize_t dest_advcnt,
	size_t bytecount, uint_t dev_datasz);

/*
 * these are software byte swapping functions
 */
uint16_t
ddi_swap16(uint16_t value);

uint32_t
ddi_swap32(uint32_t value);

uint64_t
ddi_swap64(uint64_t value);

/*
 * these are the prototypes for PCI local bus functions
 */
/*
 * PCI power management capabilities reporting in addition to those
 * provided by the PCI Power Management Specification.
 */
#define	PCI_PM_IDLESPEED	0x1		/* clock for idle dev - cap  */
#define	PCI_PM_IDLESPEED_ANY	(void *)-1	/* any clock for idle dev */
#define	PCI_PM_IDLESPEED_NONE	(void *)-2	/* regular clock for idle dev */

int
pci_config_setup(dev_info_t *dip, ddi_acc_handle_t *handle);

void
pci_config_teardown(ddi_acc_handle_t *handle);

#ifdef _LP64

uint8_t
pci_config_get8(ddi_acc_handle_t handle, off_t offset);

uint16_t
pci_config_get16(ddi_acc_handle_t handle, off_t offset);

uint32_t
pci_config_get32(ddi_acc_handle_t handle, off_t offset);

uint64_t
pci_config_get64(ddi_acc_handle_t handle, off_t offset);

void
pci_config_put8(ddi_acc_handle_t handle, off_t offset, uint8_t value);

void
pci_config_put16(ddi_acc_handle_t handle, off_t offset, uint16_t value);

void
pci_config_put32(ddi_acc_handle_t handle, off_t offset, uint32_t value);

void
pci_config_put64(ddi_acc_handle_t handle, off_t offset, uint64_t value);

#else /* _ILP32 */

uint8_t
pci_config_getb(ddi_acc_handle_t handle, off_t offset);
#define	pci_config_get8		pci_config_getb

uint16_t
pci_config_getw(ddi_acc_handle_t handle, off_t offset);
#define	pci_config_get16	pci_config_getw

uint32_t
pci_config_getl(ddi_acc_handle_t handle, off_t offset);
#define	pci_config_get32	pci_config_getl

uint64_t
pci_config_getll(ddi_acc_handle_t handle, off_t offset);
#define	pci_config_get64	pci_config_getll

void
pci_config_putb(ddi_acc_handle_t handle, off_t offset, uint8_t value);
#define	pci_config_put8		pci_config_putb

void
pci_config_putw(ddi_acc_handle_t handle, off_t offset, uint16_t value);
#define	pci_config_put16	pci_config_putw

void
pci_config_putl(ddi_acc_handle_t handle, off_t offset, uint32_t value);
#define	pci_config_put32	pci_config_putl

void
pci_config_putll(ddi_acc_handle_t handle, off_t offset, uint64_t value);
#define	pci_config_put64	pci_config_putll

#endif /* _LP64 */

int
pci_report_pmcap(dev_info_t *dip, int cap, void *arg);

int
pci_restore_config_regs(dev_info_t *dip);

int
pci_save_config_regs(dev_info_t *dip);

/*
 * the prototype for the C Language Type Model inquiry.
 */
model_t	ddi_mmap_get_model(void);
model_t	ddi_model_convert_from(model_t);

/*
 * these are the prototypes for device id functions.
 */
int
ddi_devid_valid(ddi_devid_t devid);

int
ddi_devid_register(dev_info_t *dip, ddi_devid_t devid);

void
ddi_devid_unregister(dev_info_t *dip);

int
ddi_devid_init(dev_info_t *dip, ushort_t devid_type, ushort_t nbytes,
    void *id, ddi_devid_t *ret_devid);

size_t
ddi_devid_sizeof(ddi_devid_t devid);

void
ddi_devid_free(ddi_devid_t devid);

int
ddi_devid_compare(ddi_devid_t id1, ddi_devid_t id2);

int
ddi_lyr_get_devid(dev_t dev, ddi_devid_t *ret_devid);

int
ddi_lyr_get_minor_name(dev_t dev, int spec_type, char **minor_name);

int
ddi_lyr_devid_to_devlist(ddi_devid_t devid, char *minor_name, int *retndevs,
    dev_t **retdevs);

void
ddi_lyr_free_devlist(dev_t *devlist, int ndevs);

/*
 * Event notification interfaces which enable device drivers to be notified
 * of bus events, managing storage and retrieval of eventcookies.
 */
#define	DDI_EVENT_CLAIMED		0
#define	DDI_EVENT_UNCLAIMED		1
#define	DDI_EVENT_CLAIMED_UNREGISTER	2

/*
 * Event to post to when a devinfo node is removed.
 */
#define	DDI_DEVI_REMOVE_EVENT		"DDI:DEVI_REMOVE"
#define	DDI_DEVI_INSERT_EVENT		"DDI:DEVI_INSERT"
#define	DDI_DEVI_BUS_RESET_EVENT	"DDI:DEVI_BUS_RESET"
#define	DDI_DEVI_DEVICE_RESET_EVENT	"DDI:DEVI_DEVICE_RESET"

/*
 * Invoke bus nexus driver's implementation of the
 * (*bus_remove_eventcall)() interface to remove a registered
 * callback handler for "event".
 */
int
ddi_remove_eventcall(dev_info_t *dip, ddi_eventcookie_t event);

/*
 * Invoke bus nexus driver's implementation of the
 * (*bus_add_eventcall)() interface to register a callback handler
 * for "event".
 */
int
ddi_add_eventcall(dev_info_t *dip, ddi_eventcookie_t event,
	int (*handler)(dev_info_t *, ddi_eventcookie_t, void *, void *),
	void *arg);

/*
 * Return a handle for event "name" by calling up the device tree
 * hierarchy via  (*bus_get_eventcookie)() interface until claimed
 * by a bus nexus or top of dev_info tree is reached.
 */
int
ddi_get_eventcookie(dev_info_t *dip, char *name,
	ddi_eventcookie_t *event_cookiep, ddi_plevel_t *plevelp,
	ddi_iblock_cookie_t *iblock_cookiep);

/*
 * log a system event
 */
int
ddi_log_sysevent(dev_info_t *dip, char *vendor, char *class, char *subclass,
	nvlist_t *attr_list, sysevent_id_t *eidp, int sleep_flag);

/*
 * ddi_log_sysevent() vendors
 */
#define	DDI_VENDOR_SUNW		"SUNW"


#endif	/* __STDC__ */

#endif	/* _KERNEL */

#ifdef	__cplusplus
}
#endif

#endif	/* _SYS_SUNDDI_H */
