/*
 * Copyright 1999-2002 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#ifndef _SYS_CHEETAHREGS_H
#define	_SYS_CHEETAHREGS_H

#pragma ident	"@(#)cheetahregs.h	1.10	02/08/16 SMI"

#include <sys/spitasi.h>

#ifdef	__cplusplus
extern "C" {
#endif

/*
 * This file is cpu dependent.
 */

#ifdef _KERNEL

#ifdef _ASM
/*
 * assembler doesn't understand the 'ull' suffix for C constants so
 * use the inttypes.h macros and undefine them here for assembly code
 */
#undef INT64_C
#define	INT64_C(x)	(x)
#define	UINT64_C(x)	(x)
#endif

/*
 * DCU Control Register
 *
 * +------+----+----+----+----+----+-----+-----+----+----+----+
 * | Resv | CP | CV | ME | RE | PE | HPE | SPE | SL | WE | PM |
 * +------+----+----+----+----+----+-----+-----+----+----+----+
 *  63:50   49   48   47   46	45    44    43   42   41 40:33
 *
 * +----+----+----+----+----+------------+----+----+----+---+
 * | VM | PR | PW | VR | VW | Reserved   | DM | IM | DC | IC|
 * +----+----+----+----+----+------------+----+----+----+---+
 * 32:25  24   23   22   21      20:4       3    2    1   0
 */

#define	ASI_DCU	ASI_LSU			    /* same as spitfire ASI_LSU 0x45 */
#define	DCU_IC	INT64_C(0x0000000000000001) /* icache enable		*/
#define	DCU_DC	INT64_C(0x0000000000000002) /* dcache enable		*/
#define	DCU_IM	INT64_C(0x0000000000000004) /* immu enable		*/
#define	DCU_DM	INT64_C(0x0000000000000008) /* dmmu enable		*/
#define	DCU_VW	INT64_C(0x0000000000200000) /* virt watchpoint write enable  */
#define	DCU_VR	INT64_C(0x0000000000400000) /* virt watchpoint read enable   */
#define	DCU_PW	INT64_C(0x0000000000800000) /* phys watchpoint write enable  */
#define	DCU_PR	INT64_C(0x0000000001000000) /* phys watchpoint read enable   */
#define	DCU_VM	INT64_C(0x00000001FE000000) /* virtual watchpoint write mask */
#define	DCU_PM	INT64_C(0x000001FE00000000) /* phys watchpoint write mask    */
#define	DCU_WE	INT64_C(0x0000020000000000) /* write cache enable	*/
#define	DCU_SL	INT64_C(0x0000040000000000) /* second load control	*/
#define	DCU_SPE	INT64_C(0x0000080000000000) /* software prefetch enable */
#define	DCU_HPE	INT64_C(0x0000100000000000) /* hardware prefetch enable */
#define	DCU_PE	INT64_C(0x0000200000000000) /* prefetch enable		*/
#define	DCU_RE	INT64_C(0x0000400000000000) /* RAW bypass enable	*/
#define	DCU_ME	INT64_C(0x0000800000000000) /* noncache store merging enable */
#define	DCU_CV	INT64_C(0x0001000000000000) /* virt cacheability when DM=0   */
#define	DCU_CP	INT64_C(0x0002000000000000) /* phys cacheable when DM,IM=0   */
#define	DCU_CACHE (DCU_IC|DCU_DC|DCU_WE|DCU_SPE|DCU_HPE|DCU_PE)

/*
 * Safari Configuration Register
 */
#define	ASI_SAFARI_CONFIG	ASI_UPA_CONFIG /* Safari Config Reg, 0x4A */
#define	SAFARI_CONFIG_ECLK_1	INT64_C(0x0000000000000000) /* 1/1 clock */
#define	SAFARI_CONFIG_ECLK_1_DIV	1	/* clock divisor: 1 */
#define	SAFARI_CONFIG_ECLK_2	INT64_C(0x0000000040000000) /* 1/2 clock */
#define	SAFARI_CONFIG_ECLK_2_DIV	2	/* clock divisor: 2 */
#define	SAFARI_CONFIG_ECLK_32	INT64_C(0x0000000080000000) /* 1/32 clock */
#define	SAFARI_CONFIG_ECLK_32_DIV	32	/* clock divisor: 32 */
#define	SAFARI_CONFIG_ECLK_MASK	(SAFARI_CONFIG_ECLK_32 | SAFARI_CONFIG_ECLK_2)

/*
 * Miscellaneous ASI definitions not in spitasi.h
 */
#define	ASI_DC_UTAG	0x43	/* Dcache Microtag Fields */
#define	ASI_DC_SNP_TAG	0x44	/* Dcache Snoop Tag Fields */
#define	ASI_IC_SNP_TAG	0x68	/* Icache Snoop Tag Fields */
#define	ASI_MC_DECODE	0x72	/* Memory Address Decoding Registers */
#define	ASI_EC_DATA	0x74	/* Ecache Data Staging Registers */
#define	ASI_EC_CTRL	0x75	/* Ecache Control Register */

/*
 * Defines for the different types of dcache_flush
 * it is stored in dflush_type
 */
#define	FLUSHALL_TYPE	0x0		/* blasts all cache lines */
#define	FLUSHMATCH_TYPE	0x1		/* flush entire cache but check each */
					/* each line for a match */
#define	FLUSHPAGE_TYPE	0x2		/* flush only one page and check */
					/* each line for a match */

/*
 * D-Cache Tag Data Register
 *
 * +----------+--------+----------+
 * | Reserved | DC_Tag | DC_Valid |
 * +----------+--------+----------+
 *  63	    31 30     1		 0
 *
 */
#define	ICACHE_FLUSHSZ	0x20	/* one line in i$ */
#define	CHEETAH_DC_VBIT_SHIFT	1
#define	CHEETAH_DC_VBIT_MASK	0x1

/*
 * Definitions of UltraSparc III cpu implementations as specified
 * in version register
 */
#define	CHEETAH_IMPL			0x14
#define	IS_CHEETAH(impl)		((impl) == CHEETAH_IMPL)
#define	CHEETAH_MAJOR_VERSION(rev)	(((rev) >> 4) & 0xf)
#define	CHEETAH_MINOR_VERSION(rev)	((rev) & 0xf)

/*
 * Definitions of UltraSPARC III+ cpu implementation as specified
 * in version register
 */
#define	CHEETAHPLUS_IMPL		0x15
#define	IS_CHEETAHPLUS(impl)		((impl) == CHEETAHPLUS_IMPL)
#define	CHEETAHPLUS_MAJOR_VERSION(rev)	CHEETAH_MAJOR_VERSION(rev)
#define	CHEETAHPLUS_MINOR_VERSION(rev)	CHEETAH_MINOR_VERSION(rev)

/*
 * Cheetah includes the process info in its mask to make things
 * more difficult.  The process is the low bit of the major mask,
 * so to convert to the netlist major:
 * netlist_major = ((mask_major >> 1) + 1)
 */
#define	REMAP_CHEETAH_MASK(x)	(((((x) >> 1) + 0x10) & 0xf0) | ((x) & 0xf))

/*
 * Bits of Cheetah Asynchronous Fault Status Register
 *
 * +---+--+----+----+----+----+---+---+---+---+--+----
 * |rsv|ME|PRIV|PERR|IERR|ISAP|EMC|EMU|IVC|IVU|TO|BERR
 * +---+--+----+----+----+----+---+---+---+---+--+----
 * 63:54 53 52   51   50   49   48  47  46  45 44  43
 * +---+---+---+---+---+---+---+---+--+--+---+------+---+-------+
 * |UCC|UCU|CPC|CPU|WDC|WDU|EDC|EDU|UE|CE|rsv|M_SYND|rsv||E_SYND|
 * +---+---+---+---+---+---+---+---+--+--+---+------+---+-------+
 *   42  41  40  39  38  37  36  35 34 33 32:20 19:16 15:9 8:0
 *
 */
#if defined(CHEETAH_PLUS)
/*
 * Bits of Cheetah+ Asynchronous Fault Status Register
 *
 * +---+----------------------------
 * |rsv|DTO|DBERR|THCE|TSCE|TUE|DUE|
 * +---+---------------------------- . . .
 * 63:60 59    58   57   56  55  54
 *
 */
#define	C_AFSR_DTO  INT64_C(0x0800000000000000)	/* disrupting TO error */
#define	C_AFSR_DBERR INT64_C(0x0400000000000000) /* disrupting BERR error */
#define	C_AFSR_THCE INT64_C(0x0200000000000000)	/* h/w correctable E$ tag err */
#define	C_AFSR_TSCE INT64_C(0x0100000000000000)	/* s/w correctable E$ tag err */
#define	C_AFSR_TUE  INT64_C(0x0080000000000000)	/* uncorrectable E$ tag error */
#define	C_AFSR_DUE  INT64_C(0x0040000000000000)	/* disrupting UE error */
#endif	/* CHEETAH_PLUS */
#define	C_AFSR_ME   INT64_C(0x0020000000000000)	/* errors > 1, same type!=CE */
#define	C_AFSR_PRIV INT64_C(0x0010000000000000)	/* priv code access error    */
#define	C_AFSR_PERR INT64_C(0x0008000000000000)	/* system interface protocol */
#define	C_AFSR_IERR INT64_C(0x0004000000000000)	/* internal system interface */
#define	C_AFSR_ISAP INT64_C(0x0002000000000000)	/* system request parity err */
#define	C_AFSR_EMC  INT64_C(0x0001000000000000)	/* mtag   with   CE   error  */
#define	C_AFSR_EMU  INT64_C(0x0000800000000000)	/* mtag   with   UE   error  */
#define	C_AFSR_IVC  INT64_C(0x0000400000000000)	/* intr vector with CE error */
#define	C_AFSR_IVU  INT64_C(0x0000200000000000)	/* intr vector with UE error */
#define	C_AFSR_TO   INT64_C(0x0000100000000000)	/* bus timeout from sys bus  */
#define	C_AFSR_BERR INT64_C(0x0000080000000000)	/* bus error from system bus */
#define	C_AFSR_UCC  INT64_C(0x0000040000000000)	/* E$ with software CE error */
#define	C_AFSR_UCU  INT64_C(0x0000020000000000)	/* E$ with software UE error */
#define	C_AFSR_CPC  INT64_C(0x0000010000000000) /* copyout  with  CE  error  */
#define	C_AFSR_CPU  INT64_C(0x0000008000000000) /* copyout  with  UE  error  */
#define	C_AFSR_WDC  INT64_C(0x0000004000000000) /* writeback ecache CE error */
#define	C_AFSR_WDU  INT64_C(0x0000002000000000) /* writeback ecache UE error */
#define	C_AFSR_EDC  INT64_C(0x0000001000000000) /* ecache  CE  ECC  error    */
#define	C_AFSR_EDU  INT64_C(0x0000000800000000) /* ecache  UE  ECC  error    */
#define	C_AFSR_UE   INT64_C(0x0000000400000000) /* uncorrectable ECC error   */
#define	C_AFSR_CE   INT64_C(0x0000000200000000) /* correctable   ECC error   */
#define	C_AFSR_M_SYND	INT64_C(0x00000000000f0000) /* mtag  ECC  syndrome   */
#define	C_AFSR_E_SYND	INT64_C(0x00000000000001ff) /* data  ECC  syndrome   */

/* bit shifts for selected errors */
#define	C_AFSR_WDU_SHIFT	37
#define	C_AFSR_UCU_SHIFT	41
#define	C_AFSR_UCC_SHIFT	42

/*
 * Overloaded AFSR fields. During error processing, some of the reserved fields
 * within the saved AFSR are overwritten with extra information.
 */
#define	C_AFSR_TTYPE_SHIFT	20
#define	C_AFSR_TL_SHIFT		29
#define	C_AFSR_PANIC_SHIFT	30
#define	C_AFSR_DPE_SHIFT	60
#define	C_AFSR_IPE_SHIFT	61
#define	C_AFSR_CEEN_SHIFT	62
#define	C_AFSR_FIRSTFLT_SHIFT	63

#if defined(CHEETAH_PLUS)

/* Ch+ AFSR bits from Disrupting (Corrected) ECC error Trap (Trap 0x63) */
#define	C_AFSR_CECC_ERRS	(C_AFSR_CE | C_AFSR_EMC | C_AFSR_EDU | \
			C_AFSR_EDC | C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | \
			C_AFSR_CPC | C_AFSR_IVU | C_AFSR_IVC | C_AFSR_DUE | \
			C_AFSR_THCE | C_AFSR_DBERR | C_AFSR_DTO)

/* Ch+ AFSR bits from {Instruction,Data}_access_error traps (Traps 0xa, 0x32) */
#define	C_AFSR_ASYNC_ERRS	(C_AFSR_UE | C_AFSR_EMU | C_AFSR_EDU | \
			C_AFSR_TO | C_AFSR_BERR)

/* Ch+ AFSR bits from Fast_ECC_error trap (Trap 0x70) */
#define	C_AFSR_FECC_ERRS	(C_AFSR_UCU | C_AFSR_UCC | C_AFSR_TSCE)

/* Ch+ AFSR bits from Fatal errors (processor asserts ERROR pin) */
#define	C_AFSR_FATAL_ERRS	(C_AFSR_PERR | C_AFSR_IERR | C_AFSR_ISAP | \
			C_AFSR_TUE)

/* Ch+ AFSR all valid error status bits */
#define	C_AFSR_ALL_ERRS	(C_AFSR_FATAL_ERRS | C_AFSR_FECC_ERRS | \
			C_AFSR_CECC_ERRS | C_AFSR_ASYNC_ERRS | C_AFSR_ME)

/* Ch+ AFSR bits due to an Ecache error */
#define	C_AFSR_ECACHE	(C_AFSR_UCU | C_AFSR_UCC | C_AFSR_EDU | C_AFSR_EDC | \
			C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | C_AFSR_CPC | \
			C_AFSR_TUE | C_AFSR_TSCE | C_AFSR_THCE)

/* Ch+ AFSR bits due to a Memory error */
#define	C_AFSR_MEMORY	(C_AFSR_UE | C_AFSR_CE | C_AFSR_EMU | C_AFSR_EMC | \
			C_AFSR_DUE)

/* Ch+ AFSR bits due to an Mtag error and have a valid MSYND */
#define	C_AFSR_MSYND_ERRS	(C_AFSR_EMU | C_AFSR_EMC)

/* Ch+ AFSR bits with a valid ESYND field */
#define	C_AFSR_ESYND_ERRS	(C_AFSR_UE | C_AFSR_CE | \
			C_AFSR_UCU | C_AFSR_UCC | C_AFSR_EDU | C_AFSR_EDC | \
			C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | C_AFSR_CPC | \
			C_AFSR_IVU | C_AFSR_IVC | C_AFSR_DUE)

/* Ch+ AFSR error bits for AFT Level 1 messages (uncorrected + TO + BERR) */
#define	C_AFSR_LEVEL1	(C_AFSR_UE | C_AFSR_UCU | C_AFSR_EMU | C_AFSR_EDU | \
			C_AFSR_WDU | C_AFSR_CPU | C_AFSR_IVU | C_AFSR_TO | \
			C_AFSR_BERR | C_AFSR_DUE | C_AFSR_TUE | C_AFSR_DTO | \
			C_AFSR_DBERR)

#else	/* CHEETAH_PLUS */

/* AFSR bits from Disrupting (Corrected) ECC error Trap (Trap 0x63) */
#define	C_AFSR_CECC_ERRS	(C_AFSR_CE | C_AFSR_EMC | C_AFSR_EDU | \
			C_AFSR_EDC | C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | \
			C_AFSR_CPC | C_AFSR_IVU | C_AFSR_IVC)

/* AFSR bits from {Instruction,Data}_access_error traps (Traps 0xa, 0x32) */
#define	C_AFSR_ASYNC_ERRS	(C_AFSR_UE | C_AFSR_EMU | C_AFSR_EDU | \
			C_AFSR_TO | C_AFSR_BERR)

/* AFSR bits from Fast_ECC_error trap (Trap 0x70) */
#define	C_AFSR_FECC_ERRS	(C_AFSR_UCU | C_AFSR_UCC)

/* AFSR bits from Fatal errors (processor asserts ERROR pin) */
#define	C_AFSR_FATAL_ERRS	(C_AFSR_PERR | C_AFSR_IERR | C_AFSR_ISAP)

/* AFSR all valid error status bits */
#define	C_AFSR_ALL_ERRS	(C_AFSR_FATAL_ERRS | C_AFSR_FECC_ERRS | \
			C_AFSR_CECC_ERRS | C_AFSR_ASYNC_ERRS | C_AFSR_ME)

/* AFSR bits due to an Ecache error */
#define	C_AFSR_ECACHE	(C_AFSR_UCU | C_AFSR_UCC | C_AFSR_EDU | C_AFSR_EDC | \
			C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | C_AFSR_CPC)

/* AFSR bits due to a Memory error */
#define	C_AFSR_MEMORY	(C_AFSR_UE | C_AFSR_CE | C_AFSR_EMU | C_AFSR_EMC)

/* AFSR bits due to an Mtag error and have a valid MSYND */
#define	C_AFSR_MSYND_ERRS	(C_AFSR_EMU | C_AFSR_EMC)

/* AFSR bits with a valid ESYND field */
#define	C_AFSR_ESYND_ERRS	(C_AFSR_UE | C_AFSR_CE | \
			C_AFSR_UCU | C_AFSR_UCC | C_AFSR_EDU | C_AFSR_EDC | \
			C_AFSR_WDU | C_AFSR_WDC | C_AFSR_CPU | C_AFSR_CPC | \
			C_AFSR_IVU | C_AFSR_IVC)

/* AFSR error bits for AFT Level 1 messages (uncorrected + TO + BERR) */
#define	C_AFSR_LEVEL1	(C_AFSR_UE | C_AFSR_UCU | C_AFSR_EMU | C_AFSR_EDU | \
			C_AFSR_WDU | C_AFSR_CPU | C_AFSR_IVU | C_AFSR_TO | \
			C_AFSR_BERR)

#endif	/* CHEETAH_PLUS */

/* AFSR all valid bits */
#define	C_AFSR_MASK	(C_AFSR_ALL_ERRS | C_AFSR_PRIV | C_AFSR_M_SYND | \
			C_AFSR_E_SYND)

#define	C_M_SYND_SHIFT	16
#define	GET_M_SYND(afsr)	(((afsr) & C_AFSR_M_SYND) >> C_M_SYND_SHIFT)
#define	GET_E_SYND(afsr)	((afsr) & C_AFSR_E_SYND)

/*
 * Bits of Cheetah Asynchronous Fault Address Register
 */
#define	C_AFAR_PA INT64_C(0x000007fffffffff0) /* PA<42:4> physical address */

#if defined(CHEETAH_PLUS)
/*
 * Cheetah+ External Cache Control Register Bits.
 */
#define	ECCR_ASSOC	INT64_C(0x0000000001000000) /* Ecache Assoc. */
#define	ECCR_ASSOC_SHIFT	24
#endif	/* CHEETAH_PLUS */

/*
 * Bits of Cheetah External Cache Error Enable Register
 *
 * +-----+-----+-------+-----+-------+-------+--------+-------+------+
 * | rsv | FMT | FMECC | FMD | FDECC | UCEEN | ISAPEN | NCEEN | CEEN |
 * +-----+-----+-------+-----+-------+-------+--------+-------+------+
 *  63:19   18  17  14    13    12:4     3        2       1       0
 *
 */
#define	EN_REG_FMT	INT64_C(0x0000000000040000) /* force system mtag ECC */
#define	EN_REG_FMECC	INT64_C(0x000000000003C000) /* forced mtag ECC vector */
#define	EN_REG_FMD	INT64_C(0x0000000000002000) /* force system data ECC */
#define	EN_REG_FDECC	INT64_C(0x0000000000001ff0) /* forced data ECC vector */
#define	EN_REG_UCEEN	INT64_C(0x0000000000000008) /* enable UCC,UCU */
#define	EN_REG_ISAPEN	INT64_C(0x0000000000000004) /* enable ISAP */
#define	EN_REG_NCEEN INT64_C(0x0000000000000002) /* UE,EDU,WDU,BERR,IVU,EMU */
#define	EN_REG_CEEN INT64_C(0x0000000000000001) /* enable CE,EDC,WDC,IVC,EMC */
#define	EN_REG_DISABLE	INT64_C(0x0000000000000000) /* no errors enabled */
#define	EN_REG_ECC_DISABLE (EN_REG_UCEEN | EN_REG_ISAPEN)
#define	EN_REG_CE_DISABLE (EN_REG_UCEEN | EN_REG_ISAPEN | EN_REG_NCEEN)
#define	EN_REG_ENABLE \
	(EN_REG_UCEEN | EN_REG_ISAPEN | EN_REG_NCEEN | EN_REG_CEEN)

/*
 * bit shifts for selected bits
 */
#define	EN_REG_CEEN_SHIFT	0

/*
 * To help understand the following definitions, this block of comments
 * provides information on Cheetah's E$.
 *
 * Cheetah supports three different E$ sizes (1MB, 4MB, and 8MB). The
 * number of E$ lines remains constant regardless of the size of the E$
 * as does the subblock size, however the number of 64-byte subblocks per
 * line varies depending on the E$ size.
 *
 * An E$ tag (for an E$ line) contains an EC_tag field, corresponding to the
 * high order physical address bits of that E$ line's contents, and 1 to 8
 * EC_state fields, indicating the state of each subblock. Due to the E$ line
 * size variance depending on the total size of the E$, the number of bits in
 * the EC_tag field varies as does the number of subblocks (and EC_state
 * fields) per E$ line.
 *
 * A 1MB E$ has a line size of 64 bytes, containing 1 subblock per line.
 * A 4MB E$ has a line size of 256 bytes, containing 4 subblocks per line.
 * An 8MB E$ has a line size of 512 bytes, containing 8 subblocks per line.
 *
 * An E$ tag for a particular E$ line can be read via a diagnostic ASI
 * as a 64-bit value.
 * Within the E$ tag 64-bit value, the EC_tag field is interpreted as follows:
 *	- for a 1MB E$, the EC_tag is in bits <43:21> and corresponds
 *		to physical address bits <42:20> (bits <41:19> for Cheetah+)
 *	- for a 4MB E$, the EC_tag is in bits <43:23> and corresponds
 *		to physical address bits <42:22> (bits <41:21> for Cheetah+)
 *	- for an 8MB E$, the EC_tag is in bits <43:24> and corresponds
 *		to physical address bits <42:23> (bits <41:22> for Cheetah+)
 * Within the E$ tag 64-bit value, the EC_state field(s) is(are) interpreted
 * as follows:
 *	- for a 1MB E$, EC_state0 is in bits <2:0>
 *	- for a 4MB E$, EC_state0 is in bits <2:0>, EC_state1 is in
 *		bits <5:3>, EC_state2 is in bits <8:6>, EC_state3 is
 *		in bits <11:9>
 *	- for an 8MB E$, EC_state0 is in bits <2:0>, EC_state1 is in
 *		bits <5:3>, EC_state2 is in bits <8:6>, EC_state3 is
 *		in bits <11:9>, EC_state4 is in bits <14:12>, EC_state5
 *		is in bits <17:15>, EC_state6 is in bits <20:18>,
 *		EC_state7 is in bits <23:21>
 * Note that each EC_state field contains a value representing the state
 * of its corresponding subblock.
 *
 */
/* Ecache sizes */
#define	CH_ECACHE_8M_SIZE	0x800000
#define	CH_ECACHE_4M_SIZE	0x400000
#define	CH_ECACHE_1M_SIZE	0x100000

/* Cheetah Ecache is direct-mapped, Cheetah+ can be 2-way or direct-mapped */
#define	CH_ECACHE_NWAY		1
#if defined(CHEETAH_PLUS)
#define	CHP_ECACHE_NWAY		2
#endif	/* CHEETAH_PLUS */

/* Minimum Ecache line size */
#define	CH_ECACHE_MIN_LSIZE	64

/* Maximum Ecache line size - 8Mb Ecache has 512 byte linesize */
#define	CH_ECACHE_MAX_LSIZE	512

/* Size of Ecache data staging register size (see Cheetah PRM 10.7.2) */
#define	CH_ECACHE_STGREG_SIZE	32
#define	CH_ECACHE_STGREG_TOTALSIZE	40	/* data regs + ecc */

/* The number of staging registers containing data, for ASI_EC_DATA */
#define	CH_ECACHE_STGREG_NUM	(CH_ECACHE_STGREG_SIZE / sizeof (uint64_t))

/* Size of Ecache data subblock which has state field in Ecache tag */
#define	CH_ECACHE_SUBBLK_SIZE	64
#define	CH_ECACHE_SUBBLK_SHIFT	6

/*
 * Bits to shift EC_tag field of E$ tag to form PA
 * (See Cheetah PRM 10.7.4, Cheetah+ Delta PRM 10.7)
 */
#if defined(CHEETAH_PLUS)
#define	CH_ECTAG_PA_SHIFT	2
#else	/* CHEETAH_PLUS */
#define	CH_ECTAG_PA_SHIFT	1
#endif	/* CHEETAH_PLUS */

/*
 * Constants representing the complete Cheetah Ecache tag state:
 */
#define	CH_ECSTATE_SIZE		3		/* three bits per subblock */
#define	CH_ECSTATE_MASK		0x7		/* three bit field */
#define	CH_ECSTATE_INV		0x0		/* invalid */
#define	CH_ECSTATE_SHR		0x1		/* shared */
#define	CH_ECSTATE_EXL		0x2		/* exclusive */
#define	CH_ECSTATE_OWN		0x3		/* owner */
#define	CH_ECSTATE_MOD		0x4		/* modified */
#define	CH_ECSTATE_RES1		0x5		/* reserved */
#define	CH_ECSTATE_OWS		0x6		/* owner/shared */
#define	CH_ECSTATE_RES2		0x7		/* reserved */

/*
 * Macros for Cheetah Ecache tags
 */

/* Number of subblock states per Ecache line. */
#define	CH_ECTAG_NSUBBLKS(totalsize)	((totalsize) / CH_ECACHE_1M_SIZE)

/* Mask for Tag state(s) field, 3 bits per subblock state. */
#define	CH_ECTAG_STATE_MASK(totalsize)			\
	((uint64_t)					\
	((1 << (CH_ECTAG_NSUBBLKS(totalsize) * CH_ECSTATE_SIZE)) - 1))

/* For a line to be invalid, all of its subblock states must be invalid. */
#define	CH_ECTAG_LINE_INVALID(totalsize, tag)		\
	(((tag) & CH_ECTAG_STATE_MASK(totalsize)) == 0)

/* Build address mask for tag physical address bits. */
#define	CH_ECTAG_PA_MASK(setsize)	P2ALIGN(C_AFAR_PA, (int)(setsize))

/* Get physical address bits from the EC_tag field of an E$ tag */
#define	CH_ECTAG_TO_PA(setsize, tag)	(((tag) >> CH_ECTAG_PA_SHIFT) &	\
	CH_ECTAG_PA_MASK(setsize))

/* Given a physical address, compute index for subblock tag state. */
#define	CH_ECTAG_PA_TO_SUBBLK(totalsize, pa)		\
	(((pa) >> CH_ECACHE_SUBBLK_SHIFT) & (CH_ECTAG_NSUBBLKS(totalsize) - 1))

/* Given a physical address and assoc. tag, get the subblock state. */
#define	CH_ECTAG_PA_TO_SUBBLK_STATE(totalsize, pa, tag)			\
	(((tag) >>							\
	(CH_ECTAG_PA_TO_SUBBLK(totalsize, pa) * CH_ECSTATE_SIZE)) &	\
	CH_ECSTATE_MASK)

#if defined(CHEETAH_PLUS)
/*
 * Cheetah+ Tag ECC Bit and Displacement Flush Bit in Ecache Tag Access.
 * See Cheetah+ Delta PRM 10.7
 */
#define	CHP_ECACHE_IDX_TAG_ECC		INT64_C(0x0000000000800000)
#define	CHP_ECACHE_IDX_DISP_FLUSH	INT64_C(0x0000000001000000)
#endif	/* CHEETAH_PLUS */

/*
 * Macros for Cheetah Dcache diagnostic accesses.
 */

/*
 * Dcache Index Mask for bits from *AFAR*.  Note that Dcache is virtually
 * indexed, so only bits [12:5] are valid from the AFAR.  This
 * means we have to search through the 4 ways + bit 13 (i.e. we have
 * to try 8 indexes).
 */
#define	CH_DCACHE_IDX_MASK		0x01fe0
#define	CH_DCACHE_IDX_INCR		0x02000
#define	CH_DCACHE_IDX_LIMIT		0x10000
#define	CH_DCACHE_NWAY			4
#define	CH_DCTAG_PA_MASK		INT64_C(0x000007ffffffe000)
#define	CH_DCTAG_PA_SHIFT		12
#define	CH_DCTAG_VALID_BIT		INT64_C(0x0000000000000001)
#define	CH_DCTAG_LINE_INVALID(tag)	(((tag) & CH_DCTAG_VALID_BIT) == 0)
#define	CH_DCIDX_TO_ADDR(idx)		((idx) & CH_DCACHE_IDX_MASK)
#define	CH_DCTAG_TO_PA(tag)		(((tag) << CH_DCTAG_PA_SHIFT) & \
					    CH_DCTAG_PA_MASK)
#define	CH_DCTAG_MATCH(tag, pa)		(!CH_DCTAG_LINE_INVALID(tag) && \
			    ((pa) & CH_DCTAG_PA_MASK) == CH_DCTAG_TO_PA(tag))
#define	CH_DCSNTAG_MASK			INT64_C(0x000007ffffffe000)
#define	CH_DCSNTAG_TO_PA(tag)		((tag << CH_DCTAG_PA_SHIFT) \
							& CH_DCSNTAG_MASK)
#define	CH_DCUTAG_TO_UTAG(tag)		((tag) & 0xff)
#define	CH_DCUTAG_TO_VA(tag)		((tag & 0xff) << 14)
#define	CH_DCUTAG_IDX_MASK		0x03fe0
#define	CH_DC_DATA_REG_SIZE		32
#if defined(CHEETAH_PLUS)
#define	CHP_DCTAG_PARMASK		INT64_C(0x000000007ffffffe)
#define	CHP_DCSNTAG_PARMASK		INT64_C(0x000000007ffffffe)
#define	CHP_DCTAG_MASK			INT64_C(0x000003ffffffe000)
#define	CHP_DCSNTAG_MASK		INT64_C(0x000003ffffffe000)
#define	CHP_DCWAY_MASK			INT64_C(0x0000000000003fe0)
#define	CHP_DCUTAG_TO_UTAG(tag)		((tag) & 0xffff)
#define	CHP_DCPATAG_TO_PA(tag)		((tag << CH_DCTAG_PA_SHIFT) \
							& CHP_DCTAG_MASK)
#define	CHP_DCSNTAG_TO_PA(tag)		((tag << CH_DCTAG_PA_SHIFT) \
							& CHP_DCSNTAG_MASK)
#define	CHP_DC_IDX(dcp)			((dcp->dc_idx & 0x1fc0) >> 5)
#define	CHP_DCTAG_PARITY(tag)		(tag & CHP_DC_TAG)
#define	CHP_DCSNTAG_PARITY(tag)		(tag & CHP_DC_SNTAG)
#define	CHP_DC_TAG			0x1
#define	CHP_DC_SNTAG			0x2
#endif	/* CHEETAH_PLUS */

/*
 * Macros for Cheetah Icache diagnostic accesses.
 */

/*
 * Icache Index Mask for bits from *AFAR*.  Note that Icache is virtually
 * indexed, so only bits [12:5] are valid from the AFAR.  But since the
 * Icache is indexed only by the virtual bits [12:5], we only need to look
 * through the 4 ways.  The Icache index is also confusing because we need
 * to shift the virtual address bits [12:5] left by one for the index.
 */
#define	CH_ICACHE_IDX_MASK		0x01fe0
#define	CH_ICACHE_IDX_SHIFT		1
#define	CH_ICACHE_IDX_INCR		0x04000
#define	CH_ICACHE_IDX_LIMIT		0x10000
#define	CH_ICACHE_NWAY			4
#define	CH_ICTAG_PA			0x00
#define	CH_ICTAG_UTAG			0x08
#define	CH_ICTAG_UPPER			0x10
#define	CH_ICTAG_LOWER			0x30
#define	CH_ICTAG_TMASK			0x3f
#define	CH_ICPATAG_MASK			INT64_C(0x000007ffffffe000)
#define	CH_ICPATAG_LBITS		0xff	/* lower 8 bits undefined */
#define	CH_ICPATAG_SHIFT		5
#define	CH_ICIDX_TO_ADDR(idx)		(((idx) >> CH_ICACHE_IDX_SHIFT) & \
						CH_ICACHE_IDX_MASK)
#define	CH_ICPATAG_TO_PA(tag)		(((tag) << CH_ICPATAG_SHIFT) & \
						CH_ICPATAG_MASK)
#define	CH_ICPATAG_MATCH(tag, pa)	(CH_ICPATAG_TO_PA(tag) == \
						((pa) & CH_ICPATAG_MASK))
#define	CH_ICUTAG_MASK			INT64_C(0x00000000001fe000)
#define	CH_ICUTAG_TO_UTAG(tag)		(((tag) >> 38) & 0xff)
#define	CH_ICUTAG_TO_VA(tag)		(((tag) >> 25) & CH_ICUTAG_MASK)
#define	CH_ICSNTAG_MASK			INT64_C(0x000007ffffffe000)
#define	CH_ICSNTAG_TO_PA(tag)		(((tag) << 5) & CH_ICSNTAG_MASK)
#define	CH_ICLOWER_VALID		INT64_C(0x0004000000000000)
#define	CH_ICUPPER_VALID		INT64_C(0x0004000000000000)
#define	CH_ICLOWER_TO_VPRED(lower)	(((lower) >> 46) & 0xf)
#define	CH_ICUPPER_TO_VPRED(upper)	(((upper) >> 46) & 0xf)
#define	CH_ICTAG_MATCH(icp, pa)		((icp->ic_lower & CH_ICLOWER_VALID) &&\
					    CH_ICPATAG_MATCH(icp->ic_patag, pa))
#define	CH_IC_DATA_REG_SIZE		64
#if defined(CHEETAH_PLUS)
#define	CHP_IC_IDX(icp)			((icp->ic_idx & 0x3fc0) >> 6)
#define	CHP_ICPATAG_MASK		INT64_C(0x000003ffffffe000)
#define	CHP_ICSNTAG_MASK		INT64_C(0x000003ffffffe000)
#define	CHP_ICUTAG_MASK			INT64_C(0x00000000001fe000)
#define	CHP_ICWAY_MASK			INT64_C(0x0000000000003fe0)
#define	CHP_ICPATAG_TO_PA(tag)		((tag << 5) & CHP_ICPATAG_MASK)
#define	CHP_ICSNTAG_TO_PA(tag)		((tag << 5) & CHP_ICSNTAG_MASK)
#define	CHP_ICUTAG_TO_VA(tag)		((tag >> 25) & CHP_ICUTAG_MASK)
#define	CHP_ICPATAG_PARMASK		INT64_C(0x0000003fffffff00)
#define	CHP_ICSNTAG_PARMASK		INT64_C(0x0000003fffffff00)
/*
 * Cheetah+ Icache data parity masks, see Cheetah+ Delta PRM 7.3
 * PC-relative instructions have different bits protected by parity.
 * Predecode bit 7 is not parity protected and indicates if the instruction
 * is PC-relative or not.
 */
#define	CH_ICDATA_PRED_ISPCREL		INT64_C(0x0000008000000000)
#define	CHP_ICDATA_PCREL_PARMASK	INT64_C(0x0000039ffffff800)
#define	CHP_ICDATA_NPCREL_PARMASK	INT64_C(0x000003bfffffffff)
#define	CHP_ICTAG_PARITY(tag)		(tag & CHP_IC_TAG)
#define	CHP_ICSNTAG_PARITY(tag)		(tag & CHP_IC_SNTAG)
#define	CHP_IC_TAG			0x1
#define	CHP_IC_SNTAG			0x2
#endif	/* CHEETAH_PLUS */

/*
 * CPU Log Out Structure parameters.
 * This structure is filled in by the Error Trap handlers and captures the
 * Ecache/Dcache/Icache line(s) associated with the AFAR.
 * For Cheetah Phase II, this structure is filled in at the TL=0 code.  For
 * Cheetah Phase III, this will be filled in at the trap handlers.
 */

/*
 * We use this to mark the LOGOUT structure as invalid.  Note that
 * this cannot be a valid AFAR, as AFAR bits outside of [41:5] should always
 * be zero.
 */
#define	LOGOUT_INVALID_U32	0xecc1ecc1
#define	LOGOUT_INVALID_L32	0xecc1ecc1
#define	LOGOUT_INVALID		UINT64_C(0xecc1ecc1ecc1ecc1)

/*
 * Used to mark the Error at TL>0 rudimentary CPU logout data as invalid.
 */
#define	CH_ERR_TL1_LOGOUT_INVAL -1

/*
 * Flags for the I$/D$ parity error handling routine.
 */
#define	CH_PARITY_ERR_TL	1	/* error occurred at TL > 0 */
#define	CH_PARITY_ERR_PANIC	2	/* error caused panic */
#define	CH_PARITY_ERR_ICACHE	4	/* error is in I$ parity */

/*
 * Interval for deferred CEEN reenable
 */
#define	CPU_CEEN_DELAY_SECS		6

/*
 * flags for calling cpu_check_ce
 */
#define	SCRUBBER_CEEN_CHECK		0
#define	TIMEOUT_CEEN_CHECK		1

/*
 * flags for flt_trapped_ce variable
 */
#define	CE_CEEN_DEFER		0x1	/* no CEEN reenable in trap handler */
#define	CE_CEEN_NODEFER		0x2	/* reenable CEEN in handler */
#define	CE_CEEN_TIMEOUT		0x4	/* CE caught by timeout */
#define	CE_CEEN_TRAPPED		0x8	/* CE caught by trap */

/*
 * default value for cpu_ce_not_deferred
 */
#define	CPU_CE_NOT_DEFERRED		C_AFSR_CECC_ERRS & \
					    ~(C_AFSR_CE | C_AFSR_EMC)

#if defined(CHEETAH_PLUS)
/*
 * VA for shadow AFSR/AFAR registers
 */
#define	ASI_SHADOW_REG_VA	0x8
#endif

#ifndef	_ASM

#include <sys/kstat.h>

/*
 * One Ecache data element, 32 bytes of data, 8 bytes of ECC.
 * See Cheetah PRM 10.7.2.
 */
typedef struct ec_data_elm {
	uint64_t ec_d8[CH_ECACHE_STGREG_NUM];
	uint64_t ec_eccd;	/* EC_data_ECC field */
} ec_data_elm_t;

/*
 * Ecache data captured by cpu log out code and get_ecache_dtag.
 * See Cheetah PRM 10.7.4.
 */
typedef struct ch_ec_data {
	uint64_t ec_idx;	/* Ecache index */
	uint64_t ec_tag;	/* Ecache Tag */
	ec_data_elm_t ec_data[CH_ECACHE_SUBBLK_SIZE/CH_ECACHE_STGREG_SIZE];
} ch_ec_data_t;

/*
 * Dcache data captured by cpu log out code and get_dcache_dtag.
 * See Cheetah PRM 10.6.[1-4].
 */
typedef struct ch_dc_data {
	uint64_t dc_idx;	/* Dcache index */
	uint64_t dc_tag;	/* Tag/Valid Fields */
	uint64_t dc_utag;	/* Microtag */
	uint64_t dc_sntag;	/* Snoop Tag */
	uint64_t dc_data[CH_DC_DATA_REG_SIZE/sizeof (uint64_t)]; /* Data */
} ch_dc_data_t;

/*
 * Icache data captured by cpu log out code and get_icache_dtag.
 * See Cheetah PRM 10.4.[1-3].
 */
typedef struct ch_ic_data {
	uint64_t ic_idx;	/* Icache index */
	uint64_t ic_patag;	/* Physical address tag */
	uint64_t ic_utag;	/* Microtag */
	uint64_t ic_upper;	/* Upper valid/predict tag */
	uint64_t ic_lower;	/* Lower valid/predict tag */
	uint64_t ic_sntag;	/* Snoop Tag */
	uint64_t ic_data[CH_IC_DATA_REG_SIZE/sizeof (uint64_t)]; /* Data */
} ch_ic_data_t;

/*
 * CPU Error State
 */
typedef struct ch_cpu_errors {
	uint64_t afsr;		/* AFSR */
	uint64_t afar;		/* AFAR */
	/*
	 * The following registers don't exist on cheetah
	 */
	uint64_t shadow_afsr;	/* Shadow AFSR */
	uint64_t shadow_afar;	/* Shadow AFAR */
} ch_cpu_errors_t;

typedef struct ch_diag_data {
	uint64_t chd_afar;		/* AFAR */
	uint64_t chd_afsr;		/* AFSR */
	ch_ec_data_t chd_ec_data;	/* Ecache data */
	ch_ec_data_t chd_ec_data1;	/* Ecache data for 2-way Ecache */
	ch_dc_data_t chd_dc_data;	/* Dcache data */
	ch_ic_data_t chd_ic_data;	/* Icache data */
} ch_diag_data_t;

/*
 * CPU log out structure.
 * NOTE: This structure should be the same for Cheetah and Cheetah+
 *	 since assembler code relies on one set of offsets.
 */
typedef struct ch_cpu_logout {
	ch_diag_data_t clo_data;	/* Diag data for primary AFAR */
	ch_diag_data_t clo_sdw_data;	/* Diag data for shadow AFAR */
} ch_cpu_logout_t;

#if defined(CHEETAH_PLUS)
/*
 * Parity error logging structure.
 */
typedef union ch_parity_log {
	struct {
		int cpl_way;				/* Cache line way */
		int cpl_off;				/* Cache line offset */
		int cpl_tag;				/* Faulty tags list */
		int cpl_lcnt;				/* Faulty cache lines */
		ch_dc_data_t cpl_dc[CH_DCACHE_NWAY];	/* D$ data nWays */
	} dpe;	/* D$ parity error */
	struct {
		int cpl_way;				/* Cache line way */
		int cpl_off;				/* Cache line offset */
		int cpl_tag;				/* Faulty tags list */
		int cpl_lcnt;				/* Faulty cache lines */
		ch_ic_data_t cpl_ic[CH_ICACHE_NWAY];	/* I$ data nWays */
	} ipe;	/* I$ parity error */
} ch_parity_log_t;
#endif	/* CHEETAH_PLUS */

/*
 * Error at TL>0 rudimentary CPU logout data.
 */
typedef struct ch_err_tl1_data {
	uint64_t err_tl1_afar;		/* Primary AFAR */
	uint64_t err_tl1_afsr;		/* Primary AFSR */
	uint64_t err_tl1_sdw_afar;	/* Shadow AFAR */
	uint64_t err_tl1_sdw_afsr;	/* Shadow AFSR */
	uintptr_t err_tl1_tpc;		/* Trap PC */
} ch_err_tl1_data_t;

/*
 * The scrub_misc structure contains miscellaneous bookeepping
 * items for scrubbing the I$, D$, and E$.
 *
 * The first 3 items in the array (*_scrub_outstanding) MUST
 * remain in the order they appear below and must be at the
 * beginning of the structure.  These 3 items are counters of outstanding
 * scrub requests (modeled after what is done for the USI,II scrubber).
 * A counter for a given CPU id is atomically incremented and
 * decremented _only_ on that CPU.
 */
typedef struct scrub_misc {
	uint32_t		ec_scrub_outstanding;	/* outstanding reqs */
	uint32_t		dc_scrub_outstanding;	/* outstanding reqs */
	uint32_t		ic_scrub_outstanding;	/* outstanding reqs */
	int			ecache_flush_index;	/* E$ line to flush */
	int			dcache_flush_index;	/* D$ line to flush */
	int			icache_flush_index;	/* I$ line to flush */
	int			ecache_nlines;		/* no. of E$ lines */
	int			ecache_busy;	/* keeps track if cpu busy */
	kstat_t			*ecache_ksp;    /* ptr to the kstat */
} scrub_misc_t;

/*
 * Cheetah module private data structure.  One of these is allocated for
 * each valid cpu at setup time and is pointed to by the machcpu
 * "cpu_private" pointer.  For Cheetah, we have the miscellaneous scrubber
 * variables and cpu log out structures for Fast ECC traps at TL=0,
 * Disrupting (correctable) traps and Deferred (asynchronous) traps.  For
 * Disrupting traps only one log out structure is needed because we cannot
 * get a TL>0 disrupting trap since it obeys IE.  For Deferred traps we
 * cannot get a TL>0 because we turn off NCEEN during log out capture.  E$
 * set size (E$ size / nways) is saved here to avoid repeated calculations.
 */
typedef struct cheetah_private {
	scrub_misc_t		chpr_scrub_misc;
	ch_cpu_logout_t		chpr_fecctl0_logout;
	ch_cpu_logout_t		chpr_cecc_logout;
	ch_cpu_logout_t		chpr_async_logout;
	int			chpr_ec_set_size;
} cheetah_private_t;

#endif /* _ASM */

#endif /* _KERNEL */

#ifdef	__cplusplus
}
#endif

#endif	/* _SYS_CHEETAHREGS_H */
