// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// Copyright (C) 1993-1995  Microsoft Corporation.  All Rights Reserved.
//
//  MODULE:		treeview.c   
//
//  PURPOSE:    Contains functions that perform the directory traversal
//
//  PLATFORMS: 	Windows 95, Windows NT
//
//  FUNCTIONS:
//
//  COMMENTS:
//
//

#include <windows.h>
#include <commctrl.h>
#include "globals.h"

//
//  FUNCTION:   GetDirectoryContents(HWND, LPTSTR, HTREEITEM) 
//
//  PURPOSE:    Enumerates the contents of the specified directory and adds
//              them to the TreeView control
//
//  PARAMETERS:
//      hwndTV          - TreeView to add the contents to
//      pszDirectory    - Path of the directory to list the contents for
//      htiParent       - TreeView item to add the contents as children of
//
//  RETURN VALUE:
//      (BOOL) Returns TRUE if the items are added sucessfully, 
//             FALSE otherwise.
//
//  COMMENTS:
//

BOOL GetDirectoryContents(HWND hwndTV, LPTSTR pszDirectory, 
                          HTREEITEM htiParent) 
{
    WIN32_FIND_DATA findData;
    HANDLE fileHandle;
    BOOL fInserted = FALSE;
    
    __try
    {
        // Get the first file in the directory
        fileHandle = FindFirstFile(pszDirectory, &findData);    
        if (fileHandle != INVALID_HANDLE_VALUE)
        {
            if (!AddItemToTree(hwndTV, findData.cFileName, htiParent, 
                    findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
                __leave;

            // Loop on all remaining entries in the directory
            while (FindNextFile(fileHandle, &findData))
            {
                if (!AddItemToTree(hwndTV, findData.cFileName, htiParent, 
                        findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
                    __leave;
            }
        }
		else
		{
			OutputDebugString("FindFirstFile returned invalid handle");
			ErrorHandler();
		}

        // All done, everything worked.
        fInserted = TRUE;
    }
    __finally
    {
        FindClose(fileHandle);
    }

    return (fInserted);
}    


//
//  FUNCTION:   GetDrives(HWND)
//
//  PURPOSE:    This function will retrieve a list of all the drives connected
//				the system and adds them to the tree.
//
//  PARAMETERS:
//      hwndTV  - Handle of the treeview that these image lists are being added
//                to.
//
//  RETURN VALUE:
//      (BOOL) Returns TRUE if the drives were added successfully, 
// 			   FALSE otherwise
//
//  COMMENTS:
//

BOOL GetDrives(HWND hwndTV)
{
	LPTSTR lpszDrives, lpsz;
	DWORD  cchMax;
	DWORD  cch = 12; //128 * sizeof(TCHAR);

	// Allocate some memory for the drive string.  If there isn't enough space
	// then the call to GetLogicalDriveStrings() will tell us how much we need
	// and we can realloc as necessary

	lpszDrives = (LPTSTR) HeapAlloc(GetProcessHeap(), 
									HEAP_ZERO_MEMORY, 
									cch);
	if (!lpszDrives)
	{
		ErrorHandler();
		return (FALSE);
	}

	cchMax = GetLogicalDriveStrings(cch, lpszDrives);
	if (cchMax > cch)
	{
		// The original string size was not long enough, so realloc the
		// string and try again.  Note, the value cchMax returned from
		// GetLogicalDriveStrings() returns the length needed NOT INCLUDING
		// the NULL terminator.  So, when we realloc, we add the size of
		// the null terminator.
		lpszDrives = (LPTSTR) HeapReAlloc(GetProcessHeap(),
		       					 		  HEAP_ZERO_MEMORY,
								 		  lpszDrives,
										  cchMax + sizeof(TCHAR));
		if (!lpszDrives)
		{
			ErrorHandler();
			return (FALSE);
		}
		
		GetLogicalDriveStrings(cchMax, lpszDrives);
	}

	// Now we have the string, let's save the pointer to free later and
	// add the drives to the tree
	lpsz = lpszDrives;
	while (*lpsz)
	{
		// Remove the backslash, it will be easier to parse later in 
		// BuildDirectory()
		lpsz[lstrlen(lpsz) - 1] = 0;
		AddItemToTree(hwndTV, lpsz, NULL, TRUE);

		// Advance the pointer to the next drive		
		lpsz += lstrlen(lpsz) + sizeof(TCHAR) + sizeof(TCHAR);
	}

	// All done, now clean up after ourselves
	HeapFree(GetProcessHeap(), 0, lpszDrives);

	return (TRUE);
}


			
