// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// Copyright (C) 1993-1995  Microsoft Corporation.  All Rights Reserved.
//
//  MODULE:   mdichild.c
//
//  PURPOSE:
//    To implement the basic mdi child commands.
//
//  FUNCTIONS:
//    InitMDIChild      - To register the MDI child window class.
//    MDIChildWndProc   - Processes messages for MDI child windows.
//    MsgCommand        - Handle the WM_COMMAND messages for mdi children.
//    MsgMenuSelect     - Handle the WM_MENUSELECT message for the mdi child
//    MsgCreate         - Handle the WM_CREATE message for the MDI child
//    MsgSize           - Handle the WM_SIZE message for the MDI child
//    MsgSetFocus       - Handle the WM_SETFOCUS message for the MDI child
//    MsgClose          - Handle the WM_CLOSE message for the MDI child
//    MsgDestroy        - Handle the WM_DESTROY message for the MDI child
//    CmdEdit           - Handle the notification messages for the edit control
//                        contained within the MDI child
//
//  COMMENTS:
//
//
//  SPECIAL INSTRUCTIONS: N/A
//

#include <windows.h>            // required for all Windows applications
#include <windowsx.h>
#include "globals.h"            // prototypes specific to this application
#include "resource.h"
#include <oledlg.h>
#include <richedit.h>
// Shouldn't need to #include <initguid.h> here, but IID_IRichEditOleCallback
// (defined in richole.h) has gone AWOL from riched32.dll, so just do this
// for now.
#include <initguid.h>
#include <richole.h>
#include "olehelp.h"
#include "rtf.h"
#include "ruler.h"
#include "IFrame.h"
#include "REDoc.h"


//Module specific globals

static char szChildName[12];


//
//  FUNCTION: CreateMDIChild(LPSTR)
//
//  PURPOSE: Create an MDI child window, setting caption to pszFileName
//
//  PARAMETERS:
//    LPTSTR - Window caption (filename or "untitled")
//
//  RETURN VALUE:
//    TRUE  - If creation was successful.
//    FALSE - If creation failed.
//
//  COMMENTS:
//

BOOL CreateMDIChild(LPTSTR pszFileName)
{
    CRichEditDoc* pDoc = new CRichEditDoc(pszFileName);

    if (!pDoc)
        return FALSE;
        
    if (!pDoc->Create())    // create windows (can fail)
    {
        delete pDoc; 
        pDoc = NULL;
    }
    else
        pDoc->AddRef();     // Bump up ref count to keep object alive

    return (pDoc != NULL);
}


//
//  FUNCTION: InitMDIChild(HINSTANCE)
//
//  PURPOSE: To register the MDI child window class.
//
//  PARAMETERS:
//    hinst - The instance of the application used to register the class.
//
//  RETURN VALUE:
//    TRUE - Succeeded in registering the class.
//    FALSE - Failed to register the class.
//
//  COMMENTS:
//
//

BOOL InitMDIChild(HINSTANCE hinst)
{
    WNDCLASSEX wc;

    LoadString(hinst, IDS_CHILDNAME, szChildName, DIM(szChildName));

    wc.cbSize        = sizeof(WNDCLASSEX);
    wc.style         = 0;
    wc.lpfnWndProc   = CRichEditDoc::RichEditDocWndProc;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;
    wc.hInstance     = hinst;
    wc.hIcon         = LoadIcon(hinst, MAKEINTRESOURCE(IDI_CHILDICON));
    wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
    wc.lpszMenuName  = NULL;
    wc.lpszClassName = szChildName;
	wc.hIconSm       = LoadImage(hinst,		// Load small icon image
	                             MAKEINTRESOURCE(IDI_CHILDICON),
	                             IMAGE_ICON,
	                             16, 16,
	                             0);

    // Register the window class and return FALSE if unsuccesful.

    if (!RegisterClassEx(&wc))
    {
        //Assume we are running on NT where RegisterClassEx() is
        //not implemented, so let's try calling RegisterClass().

        if (!RegisterClass((LPWNDCLASS)&wc.style))
        	return FALSE;
    }

    return TRUE;
}



//
//  FUNCTION: CRichEditDoc::CRichEditDoc(LPTSTR)
//
//  PURPOSE:  CRichEditDoc constructor
//
//  PARAMETERS:
//    LPTSTR    - initial window title (filename or "untitled")
//
//  RETURN VALUE:
//    None
//
//  COMMENTS:
//

CRichEditDoc::CRichEditDoc(LPTSTR pszFileName)
    : m_cRef(0), m_hwndDoc(NULL), m_hwndEdit(NULL), m_hwndRuler(NULL),
      m_hdcTarget(NULL), m_fInitialized(FALSE), m_fFileNameSet(FALSE),
      m_fCSHMode(FALSE), m_pRECallback(NULL), m_pReObj(NULL), m_pActiveObject(NULL)
{
    if (pszFileName)
        lstrcpy(m_szFileName, pszFileName);
    else
        m_szFileName[0] = 0;
}


//
//  FUNCTION: CRichEditDoc::!CRichEditDoc()
//
//  PURPOSE:  CRichEditDoc destructor
//
//  PARAMETERS:
//    None
//
//  RETURN VALUE:
//    None
//
//  COMMENTS:
//

CRichEditDoc::~CRichEditDoc()
{
    if (m_hdcTarget)
        DeleteDC(m_hdcTarget);

    if (m_pActiveObject)
        m_pActiveObject->Release();

    if (m_pRECallback)
        delete m_pRECallback;

    if (m_pReObj)
        m_pReObj->Release();
}


//
//  FUNCTION: CRichEditDoc::Create()
//
//  PURPOSE: Create an MDI child window associated with the CRichEditDoc object
//
//  PARAMETERS:
//    None
//
//  RETURN VALUE:
//    BOOLEAN indicating success/failure
//
//  COMMENTS:
//

BOOL CRichEditDoc::Create()
{
    // First make sure we haven't already been called for this object
    if (m_hwndDoc)
        return FALSE;

    // Create the MDI child window

    // Windows NT and Windows 95 present different options for creating
    // an MDI child window.  While using the WM_MDICREATE message will
    // work on both Windows versions, Windows 95 presents a new window
    // style which simplifies the process.  Here the function uses the
    // method apropriate for the system it's running on to illustrate
    // both methods.

    DWORD dwVersion = GetVersion();
    if ((dwVersion < 0x80000000) || (LOBYTE(LOWORD(dwVersion)) < 4))
    {
        // This is Windows NT or Win32s, so use the WM_MDICREATE message
        MDICREATESTRUCT mcs;

        mcs.szClass = szChildName;      // window class name
        mcs.szTitle = m_szFileName;     // window title
        mcs.hOwner  = hInst;            // owner
        mcs.x       = CW_USEDEFAULT;    // x position
        mcs.y       = CW_USEDEFAULT;    // y position
        mcs.cx      = CW_USEDEFAULT;    // width
        mcs.cy      = CW_USEDEFAULT;    // height
        mcs.style   = 0;                // window style
        mcs.lParam  = (LPARAM)(CRichEditDoc*)this;     // lparam

        SendMessage(hwndMDIClient,
                    WM_MDICREATE,
                    0,
                    (LPARAM)(LPMDICREATESTRUCT)&mcs);
    }
    else
    {
        // This method will only work with Windows 95, not Windows NT or Win32s
        CreateWindowEx(WS_EX_MDICHILD,  // EX window style
                       szChildName,     // window class name
                       m_szFileName,    // window title
                       0,               // window style
                       CW_USEDEFAULT,   // x position
                       CW_USEDEFAULT,   // y position
                       CW_USEDEFAULT,   // width
                       CW_USEDEFAULT,   // height
                       hwndMDIClient,   // parent
                       NULL,            // menu (child ID)
                       hInst,           // owner
                       (LPVOID)(CRichEditDoc*)this);   // lparam
    }

    if (m_hwndDoc)      // set during WM_CREATE
    {
        ShowWindow(m_hwndDoc, SW_SHOW);
        SetFocus(m_hwndDoc);

        // increment global document counts
        cUntitled++;
        cOpen++;

        m_fInitialized = TRUE;      // Tell WM_DESTROY handler to clean up
        return TRUE;
    }
    else
        return FALSE;
}


/*
 *	CRichEditDoc::QueryInterface
 *
 *	Purpose:
 *		Returns a pointer to the specified interface.
 *
 *	Arguments:
 *		REFIID			Interface desired.
 *		LPUNKNOWN *		Interface we return.
 *
 *	Returns:
 *		HRESULT		 Error status.
 */

STDMETHODIMP CRichEditDoc::QueryInterface(REFIID riid, LPVOID* ppvObj)
{
    *ppvObj = NULL;

    if (IsEqualIID(riid, IID_IUnknown) ||
        IsEqualIID(riid, IID_IOleWindow) ||
		IsEqualIID(riid, IID_IOleInPlaceUIWindow))
	{
        *ppvObj = this;
	}
    else if (IsEqualIID(riid, IID_IRichEditOleCallback))
    {
        *ppvObj = m_pRECallback;
    }
	else
		return E_NOINTERFACE;

    if (*ppvObj)
        ((LPUNKNOWN)*ppvObj)->AddRef();

	return S_OK;
}


/*
 *	CRichEditDoc::AddRef
 *
 *	Purpose:
 *		Increments reference count on the specified object.
 *
 *	Arguments:
 *		None
 *
 *	Returns:
 *		ULONG			New value of reference count.
 */

STDMETHODIMP_(ULONG) CRichEditDoc::AddRef()
{
	return ++m_cRef;
}


/*
 *	CRichEditDoc::Release
 *
 *	Purpose:
 *		Decrements reference count on the specified object.  If count is
 *		decremented to zero, the object is freed.
 *
 *	Arguments:
 *		None
 *
 *	Returns:
 *		ULONG			New value of reference count.
 */

STDMETHODIMP_(ULONG) CRichEditDoc::Release()
{
    ULONG cRef = --m_cRef;

//    if(!cRef)         // CRichEditDoc destruction happens during WM_DESTROY
//        delete this;

    return cRef;
}


/*
 *	CRichEditDoc::GetWindow
 *
 *	Purpose:
 *		Return the window handle of the doc-level window for in place use.
 *
 *	Arguments:
 *		HWND*               Out param for window handle.
 *
 *	Returns:
 *		HRESULT				Error status.
 */
STDMETHODIMP CRichEditDoc::GetWindow(HWND* phwnd)
{
    *phwnd = m_hwndDoc;
	return S_OK;
}


/*
 *	CRichEditDoc::ContextSensitiveHelp
 *
 *	Purpose:
 *		Notifies the frame that the object has entered Context Help mode.
 *
 *	Arguments:
 *		BOOL                CSH mode.
 *
 *	Returns:
 *		HRESULT				Error status.
 */
STDMETHODIMP CRichEditDoc::ContextSensitiveHelp(BOOL fEnterMode)
{
    if (m_fCSHMode != fEnterMode)
    {
        m_fCSHMode = fEnterMode;

        // this code "trickles" the context sensitive help via shift+f1
        // to the inplace active object.  See the technotes for implementation
        // details.
        if (m_pActiveObject)
        {
            LPOLEINPLACEOBJECT pipo;
            m_pActiveObject->QueryInterface(IID_IOleInPlaceObject, (LPVOID*)&pipo);
            if (pipo)
            {
                pipo->ContextSensitiveHelp(fEnterMode);
                pipo->Release();
            }
        }
    }
    return S_OK;
}


/*
 *	CRichEditDoc::GetBorder
 *
 *	Purpose:
 *		Returns a RECT structure in which the object can put frame
 *		adornments while an object is active in place.
 *
 *	Arguments:
 *      LPRECT lprectBorder - out param to contain the outermost 
 *                            rect for frame adornments
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP CRichEditDoc::GetBorder(LPRECT prcBorder)
{
    if (!prcBorder)
        return E_INVALIDARG;

    GetClientRect(m_hwndDoc, prcBorder);

    // save room for ruler window
    if (m_hwndRuler)
    {
        RECT rect;
        GetWindowRect(m_hwndRuler, &rect);
        prcBorder->top += (rect.bottom - rect.top);
    }

	return S_OK;
}


/*
 *	CRichEditDoc::RequestBorderSpace
 *
 *	Purpose:
 *		Determines whether tools can be installed around the objects 
 *		window frame while the object is active in place.
 *
 *	Arguments:
 *		LPCBORDERWIDTHS     Requested border space
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP CRichEditDoc::RequestBorderSpace(LPCBORDERWIDTHS pbw)
{
    // Deny all requests
    return (pbw ? INPLACE_E_NOTOOLSPACE : S_OK);
}


/*
 *	CRichEditDoc::SetBorderSpace
 *
 *	Purpose:
 *		Allocates space for the border requested in the call to the 
 *		RequestBorderSpace member function.
 *
 *	Arguments:
 *		LPCBORDERWIDTHS     Requested border space
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP CRichEditDoc::SetBorderSpace(LPCBORDERWIDTHS pbw)
{
    // Deny all requests
    return (pbw ? INPLACE_E_NOTOOLSPACE : S_OK);
}


/*
 *	CRichEditDoc::SetActiveObject
 *
 *	Purpose:
 *		Called by the object to provide the document window a direct channel
 *		of communication with the active in-place object.
 *
 *	Arguments:
 *		LPOLEINPLACEACTIVEOBJECT    New active object (or NULL, if none)
 *      LPCOLESTR                   Object name
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP CRichEditDoc::SetActiveObject(LPOLEINPLACEACTIVEOBJECT pipaobj,
                                           LPCOLESTR szObjName)
{
    if (szObjName)
    {
        TCHAR szNewTitle[MAX_PATH];
#ifdef UNICODE
        wsprintf(szNewTitle, TEXT("%s in %s"), szObjName, m_szFileName);
#else
        wsprintf(szNewTitle, TEXT("%s in %s"), WtoA(szObjName), m_szFileName);
#endif
        SetWindowText(m_hwndDoc, szNewTitle);
    }
    else
        SetWindowText(m_hwndDoc, m_szFileName);

	// Free any existing active object
	if(m_pActiveObject)
		m_pActiveObject->Release();

	// If we're given an object, AddRef it; update our remembered ipaobj
	if(pipaobj)
		pipaobj->AddRef();

	m_pActiveObject = pipaobj;

	return S_OK;
}


//
//  FUNCTION: CRichEditDoc::RichEditDocWndProc(HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE:  Processes messages for MDI child windows.
//
//  PARAMETERS:
//    hwnd     - window handle
//    uMessage - message number
//    wparam   - additional information (dependant of message number)
//    lparam   - additional information (dependant of message number)
//
//  RETURN VALUE:
//    Depends on the message number.
//
//  COMMENTS:
//    Dispatches messages to the appropriate method of the associated
//    CRichEditDoc object.
//

LRESULT CALLBACK CRichEditDoc::RichEditDocWndProc(HWND hwnd,
                                                  UINT uMessage,
                                                  WPARAM wparam,
                                                  LPARAM lparam)
{
    // not valid until WM_CREATE
    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwnd, GWL_USERDATA);

    if (uMessage == WM_CREATE)
    {
        pDoc = (CRichEditDoc*)((LPMDICREATESTRUCT)((LPCREATESTRUCT)lparam)->lpCreateParams)->lParam;
        if (pDoc)
        {
            pDoc->m_hwndDoc = hwnd;
            SetWindowLong(hwnd, GWL_USERDATA, (LONG)pDoc);
        }
    }
    else if (!pDoc)
        return DefMDIChildProc(hwnd, uMessage, wparam, lparam);

    switch (uMessage)
    {
        case WM_MENUSELECT:
            return pDoc->MsgMenuSelect(wparam, lparam);

        case WM_COMMAND:
            return pDoc->MsgCommand(wparam, lparam);

        case WM_NOTIFY:
            return pDoc->MsgNotify(wparam, lparam);

        case WM_MDIACTIVATE:
            return pDoc->MsgMDIActivate(wparam, lparam);

        case WM_SETFOCUS:
            return pDoc->MsgSetFocus(wparam, lparam);

        case WM_SIZE:
            return pDoc->MsgSize(wparam, lparam);

        case WM_CREATE:
            return pDoc->MsgCreate(wparam, lparam);

        case WM_CLOSE:
            return pDoc->MsgClose(wparam, lparam);

        case WM_DESTROY:
            return pDoc->MsgDestroy(wparam, lparam);

        default:
            return DefMDIChildProc(hwnd, uMessage, wparam, lparam);
    }

    return 0;
}


//
//  FUNCTION: CRichEditDoc::CreateEditControl(HWND)
//
//  PURPOSE: Create the RichEdit control with the MDI child as the parent
//
//  PARAMETERS:
//    None
//
//  RETURN VALUE:
//    TRUE - If initialization was successful.
//    FALSE - If initialization failed.
//
//  COMMENTS:
//

BOOL CRichEditDoc::CreateEditControl()
{
    // First make sure we have a doc window but _not_ an edit control
    if (!m_hwndDoc || m_hwndEdit)
        return FALSE;

    RECT rect;
    GetBorder(&rect);

    m_hwndEdit = CreateWindow("RichEdit",
                              NULL,
                              WS_VISIBLE |
                              WS_CHILD   | WS_CLIPCHILDREN | WS_CLIPSIBLINGS |
				              WS_HSCROLL | WS_VSCROLL |
                              ES_AUTOHSCROLL | ES_AUTOVSCROLL |
                              ES_MULTILINE   |
                              ES_SAVESEL | ES_NOHIDESEL |
                              ES_SELECTIONBAR | ES_SUNKEN,
                              rect.left,
                              rect.top,
                              rect.right - rect.left,
                              rect.bottom - rect.top,
                              m_hwndDoc,
                              (HMENU)IDC_EDIT,           // Child control i.d.
                              hInst,
                              NULL);

    if (!m_hwndEdit)
        return FALSE;

    // Set the RichEdit control to send the EN_SELCHANGE notification
    // via the WM_NOTIFY message.
    SendMessage(m_hwndEdit, EM_SETEVENTMASK, 0, ENM_SELCHANGE | EN_DROPFILES );

    // Set up the RichEdit control to act as WSYWIG as possible,
    // here we force a 7" wide page size.
    m_hdcTarget = CreateCompatibleDC(NULL);
//    m_hdcTarget = GetCurrentPrinterDC();
    if (!SendMessage(m_hwndEdit, EM_SETTARGETDEVICE, (WPARAM)m_hdcTarget, (LPARAM)(1440*7)))
    {
        DeleteDC(m_hdcTarget);
        m_hdcTarget = NULL;
    }

    // Set default character format...  The point here is to avoid having
    // the system font as default since it generally doesn't scale well.

    // Set up CHARFORMAT for default 10 point Times New Roman with no
    // extra effects (bold, italic, etc.)
    CHARFORMAT cf;
    cf.cbSize = sizeof(cf);
    cf.dwMask = CFM_FACE | CFM_SIZE | CFM_BOLD | CFM_ITALIC | CFM_STRIKEOUT
                | CFM_UNDERLINE | CFM_COLOR | CFM_OFFSET | CFM_PROTECTED;
    cf.dwEffects = CFE_AUTOCOLOR;   // use COLOR_WINDOWTEXT
    cf.yHeight = 200; // 200 twips == 10 points
    cf.yOffset = 0;
    cf.crTextColor = 0;
    cf.bPitchAndFamily = FF_SWISS | VARIABLE_PITCH;
    lstrcpy(cf.szFaceName, TEXT("Arial"));

    // Set the default character format.
    SendMessage(m_hwndEdit, EM_SETCHARFORMAT, SCF_SELECTION, (LPARAM)&cf);

    // Get the control's IRichEditOle interface
    SendMessage(m_hwndEdit, EM_GETOLEINTERFACE, 0, (LPARAM)&m_pReObj);
    if (m_pReObj)
    {
        m_pReObj->AddRef();
        m_pReObj->SetHostNames(szAppName, m_szFileName);
    }

    // Create IRichEditOleCallback interface and give to RichEdit control
    m_pRECallback = new CRichEditOleCallback(this);
    if (m_pRECallback)
        SendMessage(m_hwndEdit, EM_SETOLECALLBACK, 0, (LPARAM)m_pRECallback);
        // the control should AddRef this

    // Accept dropped files
    DragAcceptFiles(m_hwndEdit, TRUE);

    return TRUE;
}


//
//  FUNCTION: CRichEditDoc::MsgCommand(WPARAM, LPARAM)
//
//  PURPOSE: Handle the WM_COMMAND messages for mdi children.
//
//  PARAMETERS:
//    GET_WM_COMMAND_ID(wparam,lparam)   - Command identifier
//    GET_WM_COMMAND_HWND(wparam,lparam) - Control handle
//
//  RETURN VALUE:
//
//  COMMENTS:
//
//

LRESULT CRichEditDoc::MsgCommand(WPARAM wparam, LPARAM lparam)
{
    // look for notifications from edit control
    if (GET_WM_COMMAND_ID(wparam, lparam) == IDC_EDIT)
        return CmdEdit(GET_WM_COMMAND_CMD(wparam, lparam),
                       GET_WM_COMMAND_HWND(wparam, lparam));

    // pass other commands (e.g. from context menus) on to main frame window
    return SendMessage(hwndMain, WM_COMMAND, wparam, lparam);
}


//
//  FUNCTION: CRichEditDoc::MsgCreate(WPARAM, LPARAM)
//
//  PURPOSE:
//
//  PARAMETERS:
//    wparam   - Message specific data (unused).
//    lparam   - Message specific data (unused).
//
//  RETURN VALUE:
//    
//
//  COMMENTS:
//
//

LRESULT CRichEditDoc::MsgCreate(WPARAM wparam, LPARAM lparam)
{
    m_hwndRuler = CreateRuler(m_hwndDoc, ID_RULER, 1024, 32);
    if (!m_hwndRuler)
        return -1;

    if (!CreateEditControl())
        return -1;

    return 0;
}


//
//  FUNCTION: CRichEditDoc::MsgSize(WPARAM, LPARAM)
//
//  PURPOSE:  Resize the edit control window within the MDI Child
//
//  PARAMETERS:
//    wparam   - Message specific data (unused).
//    lparam   - Message specific data (unused).
//
//  RETURN VALUE:
//    
//
//  COMMENTS:
//    We must call DefMDIChildProc so that if the child is being maximized
//    we'll get the system menu, min, max and close buttons painted properly.
//

LRESULT CRichEditDoc::MsgSize(WPARAM wparam, LPARAM lparam)
{
    RECT rect;

    // reposition RichEdit control
    GetBorder(&rect);
    MoveWindow(m_hwndEdit,
               rect.left,
               rect.top,
               rect.right - rect.left,
               rect.bottom - rect.top,
               TRUE);

    if (m_pActiveObject)
        m_pActiveObject->ResizeBorder(&rect, this, FALSE);

    return DefMDIChildProc(m_hwndDoc, WM_SIZE, wparam, lparam);
}


//
//  FUNCTION: CRichEditDoc::MsgMenuSelect(WPARAM, LPARAM)
//
//  PURPOSE:  
//
//  PARAMETERS:
//    wparam   - Message specific data (unused).
//    lparam   - Message specific data (unused).
//
//  RETURN VALUE:
//    
//
//  COMMENTS:
//

LRESULT CRichEditDoc::MsgMenuSelect(WPARAM wparam, LPARAM lparam)
{
    return DefMDIChildProc(m_hwndDoc, WM_MENUSELECT, wparam, lparam);
}


//
//  FUNCTION: CRichEditDoc::MsgSetFocus(WPARAM, LPARAM)
//
//  PURPOSE:
//    Sets the focus to the edit control window contained within this MDI
//    child window.
//
//  PARAMETERS:
//    wparam   - Message specific data (unused).
//    lparam   - Message specific data (unused).
//
//  RETURN VALUE:
//    
//
//  COMMENTS:
//

LRESULT CRichEditDoc::MsgSetFocus(WPARAM wparam, LPARAM lparam)
{
    // Pass focus to the RichEdit control
    SetFocus(m_hwndEdit);

    // Update the ruler to reflect the margins
    RTF_ShowMargins(m_hwndRuler);

    // Update the toolbar to show the current text attributes
    RTF_ShowCharAttributes();

    return 0;
}


//
//  FUNCTION: CRichEditDoc::MsgClose(WPARAM, LPARAM)
//
//  PURPOSE:  Query user to save file before closing window
//
//  PARAMETERS:
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Return 0 if user hits "Cancel" from the query save dialog.  This
//    prevents the MDI child from closing.
//
//  COMMENTS:
//    Call DefMDIChildProc so that the window gets destroyed.
//

LRESULT CRichEditDoc::MsgClose(WPARAM wparam, LPARAM lparam)
{
    if (QuerySaveFile(m_hwndDoc))
    {
        if (m_pReObj)
        {
            // Deactivate any inplace object
            m_pReObj->InPlaceDeactivate();

            // Make sure all objects are shut down
            REOBJECT reobj = {0};
            reobj.cbStruct = sizeof(reobj);

            int cObj = m_pReObj->GetObjectCount();
            for (int i = 0; i < cObj; i++)
            {
                m_pReObj->GetObject(i, &reobj, REO_GETOBJ_POLEOBJ);
                if (reobj.poleobj)
                {
                    reobj.poleobj->Close(OLECLOSE_NOSAVE);
                    reobj.poleobj->Release();
                }
            }
        }

        return DefMDIChildProc(m_hwndDoc, WM_CLOSE, wparam, lparam);
    }
    else
        return 0;
}


//
//  FUNCTION: CRichEditDoc::MsgDestroy(WPARAM, LPARAM)
//
//  PURPOSE:
//
//  PARAMETERS:
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//
//

LRESULT CRichEditDoc::MsgDestroy(WPARAM wparam, LPARAM lparam)
{
    if (m_fInitialized)
    {
        // decrement global document count
        cOpen -= 1;

        // destroy CRichEditDoc object
        delete this;
    }

    return 0;
}

//
//  FUNCTION: CRichEditDoc::MsgNotify(WPARAM, LPARAM)
//
//  PURPOSE: Handles the WM_NOTIFY message
//
//  PARAMETERS:
//    wparam - ID of control sending notification
//    lparam - LPNOTIFY pointer to the NOTIFY structure
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//
//

LRESULT CRichEditDoc::MsgNotify(WPARAM wparam, LPARAM lparam)
{
    #define pnmHdr ((LPNMHDR)lparam)

    switch (pnmHdr->code)
    {
        // User changed the ruler's "knobbies" or tabs- See RULER.H
        case RN_CHANGEDSETTINGS: 
            #define pnmRuler ((LPNMRULER)lparam)
            RTF_ChangeMargins(pnmHdr->hwndFrom,         // m_hwndRuler
                              pnmRuler->iFirstLineIndent,
                              pnmRuler->iRight,
                              pnmRuler->iLeft - pnmRuler->iFirstLineIndent);
            #undef pnmRuler
            break;

        // User changed text in RichEdit control- See COMMCTRL.H
        case EN_SELCHANGE:
            RTF_ShowMargins(m_hwndRuler);
            RTF_ShowCharAttributes();
            break;
    }
    return 0;

    #undef pnmHdr
}


//
//  FUNCTION: CRichEditDoc::MsgMDIActivate(WPARAM, LPARAM)
//
//  PURPOSE: Passes doc window activation notification on to the current
//      in-place active object, if any.
//
//  PARAMETERS:
//    wparam    - Extra data     (Unused)
//    lparam    - Extra data     (Unused)
//
//  RETURN VALUE:
//
//    Always returns 0 - Message handled
//
//  COMMENTS:
//
//

LRESULT CRichEditDoc::MsgMDIActivate(WPARAM wparam, LPARAM lparam)
{
    BOOL bActivate = ((HWND)lparam == m_hwndDoc);
    COleInPlaceFrame* pfr = (COleInPlaceFrame*)pFrame;

    // If we are activating, make sure the frame is in sync with
    // us regarding active objects.  (This ensures the frame isn't
    // using an active object from an inactive document.)
    if (pfr)
        pfr->SetActiveObject(m_pActiveObject, NULL);
            // should probably remember object name instead of passing NULL here

    // Pass activation notification to active object, if any
    if (m_pActiveObject)
        m_pActiveObject->OnDocWindowActivate(bActivate);

    else if (bActivate) // else we are activating without an active object, show our UI
    {
        if (pfr) // should never be NULL, but just in case...
            pfr->ReinstateUI();
    }

    return 0;
}

//
//  FUNCTION: CRichEditDoc::CmdEdit(HWND, WORD, WORD, HWND)
//
//  PURPOSE: Handle edit control notifications.
//
//  PARAMETERS:
//    wNotify  - EN_*
//    hwndCtrl - NULL (unused)
//
//  RETURN VALUE:
//    Always returns 0 - command handled.
//
//  COMMENTS:
//    Handle the edit control's out of space error by putting up an
//    "Out of Memory" warning dialog.
//

LRESULT CRichEditDoc::CmdEdit(WORD wNotify, HWND hwndCtrl)
{
    switch (wNotify)
    {
        case EN_ERRSPACE:
            MessageBox(m_hwndDoc, 
                       "Out of memory.", 
                       szTitle, 
                       MB_ICONHAND | MB_OK);
            break;
    }
    return 0;
}


//
//	CRichEditDoc::GetNewStorage
//
//	Purpose:
//		Gets storage for a new object.
//
//	Arguments:
//		LPSTORAGE*          Where to return storage
//
//	Returns:
//		HRESULT	            Error status.
//

HRESULT CRichEditDoc::GetNewStorage(LPSTORAGE* ppstg)
{
    if (!ppstg)
        return E_INVALIDARG;

    // first, NULL the out pointer
    *ppstg = NULL;

    // Create a temporary storage object on HGLOBAL.  First create an
    // ILockBytes object on HGLOBAL with delete-on-release, then create
    // storage on it, then release the ILockBytes so its only remaining
    // reference is from the storage.  Releasing the storage will free
    // everything.

    LPLOCKBYTES pLockBytes;
    HRESULT hr = CreateILockBytesOnHGlobal(NULL, TRUE, &pLockBytes);

    if (FAILED(hr))
        return hr;

    hr = StgCreateDocfileOnILockBytes(pLockBytes,
                                      STGM_SHARE_EXCLUSIVE | STGM_CREATE
                                       | STGM_READWRITE,
                                      0,
                                      ppstg);
    pLockBytes->Release();

    return hr;
}


//
//	CRichEditDoc::DoVerb
//
//	Purpose:
//		Executes OLE verb
//
//	Arguments:
//		LONG                Verb #
//
//	Returns:
//		HRESULT	            Error status.
//

HRESULT CRichEditDoc::DoVerb(LONG iVerb)
{
    HRESULT hr = S_FALSE;

    if (!m_hwndEdit || !m_pReObj)
        return E_FAIL;

    REOBJECT reobj = {0};
    reobj.cbStruct = sizeof(reobj);

    // Get IOleObject and IOleClientSite interfaces for the selected object
    if (S_OK == m_pReObj->GetObject(REO_IOB_SELECTION,
                                    &reobj,
                                    REO_GETOBJ_POLEOBJ | REO_GETOBJ_POLESITE))
    {
        // Get object position
        POINT pt;
        SendMessage(m_hwndEdit, EM_POSFROMCHAR, (WPARAM)&pt, REO_IOB_SELECTION);

        // Calculate object rect in screen units for DoVerb
        RECT rect = {0};
        HDC hdc = GetDC(NULL);  // screen DC
        rect.right  = MulDiv(reobj.sizel.cx,
                             GetDeviceCaps(hdc, LOGPIXELSX),
                             2540);
        rect.bottom = MulDiv(reobj.sizel.cy,
                             GetDeviceCaps(hdc, LOGPIXELSY),
                             2540);
        ReleaseDC(NULL, hdc);
        OffsetRect(&rect, pt.x, pt.y);
 
        // Call the object
        hr = reobj.poleobj->DoVerb(iVerb,
                                   NULL,
                                   reobj.polesite,
                                   0,
                                   m_hwndEdit,
                                   &rect);

        reobj.poleobj->Release();
        reobj.polesite->Release();
    }

    return hr;
}


//
//	CRichEditDoc::InsertObject
//
//	Purpose:
//		Insert new object into RichEdit control
//
//	Arguments:
//		None
//
//	Returns:
//		BOOL                Success/failure
//

BOOL CRichEditDoc::InsertObject()
{
    OLEUIINSERTOBJECT   io = {0};
    LPOLEOBJECT         poleobj = NULL;
    LPOLECLIENTSITE     polesite = NULL;
    LPSTORAGE           pstgItem = NULL;
    TCHAR               szFile[MAX_PATH];
    DWORD               dwRet;
    REOBJECT            reobj;
    BOOL                bResult = FALSE;

    if (!m_hwndEdit || !m_pReObj)
        return FALSE;

    if (m_pReObj->GetClientSite(&polesite))
    {
        MessageBox(m_hwndDoc, TEXT("No client site!"), NULL, MB_OK);
        goto error;
    }

    if (FAILED(GetNewStorage(&pstgItem)))
    {
        MessageBox(m_hwndDoc, TEXT("Can't create item storage!"), NULL, MB_OK);
        goto error;
    }

    szFile[0] = 0;
    io.cbStruct         = sizeof(io);
    io.dwFlags          = IOF_SHOWHELP | IOF_CREATENEWOBJECT | IOF_CREATEFILEOBJECT |
                            IOF_CREATELINKOBJECT | IOF_SELECTCREATENEW;
//                            IOF_VERIFYSERVERSEXIST;
    io.hWndOwner        = m_hwndDoc;
    io.lpszCaption      = NULL;
    io.lpszFile         = szFile;
    io.cchFile          = MAX_PATH;
    io.iid              = IID_IOleObject;
    io.oleRender        = OLERENDER_DRAW;
    io.lpIOleClientSite = polesite;
    io.lpIStorage       = pstgItem;
    io.ppvObj           = (LPVOID*)&poleobj;
    io.clsid            = CLSID_NULL;

    if ((dwRet = OleUIInsertObject(&io)) != OLEUI_SUCCESS)
    {
        if (dwRet == OLEUI_CANCEL)
            goto error;

        else if (dwRet == OLEUI_IOERR_SCODEHASERROR)
            wsprintf(szFile, TEXT("OleUIInsertObject scode is %lx"), io.sc);

        else
            wsprintf(szFile, TEXT("OleUIInsertObject returned %ld"), dwRet);

        MessageBox(m_hwndDoc, szFile, szAppName, MB_ICONEXCLAMATION | MB_OK);
        goto error;
    }

    reobj.cbStruct  = sizeof(REOBJECT);
    reobj.cp        = REO_CP_SELECTION;
    reobj.clsid     = io.clsid;
    reobj.poleobj   = poleobj;
    reobj.pstg      = pstgItem;
    reobj.polesite  = polesite;
    reobj.sizel.cx  = 0;
    reobj.sizel.cy  = 0;
    reobj.dvaspect  = DVASPECT_CONTENT;
    reobj.dwFlags   = REO_RESIZABLE;
    reobj.dwUser    = 0;

    if (io.dwFlags & IOF_SELECTCREATENEW)
        reobj.dwFlags |= REO_BLANK;

    // RichEdit doesn't setup advises if reobj.clsid == CLSID_NULL
    // Try our darndest to get a CLSID

    if (IsEqualCLSID(reobj.clsid, CLSID_NULL))
    {
#ifdef UNICODE
        if (!SUCCEEDED(GetClassFile(szFile, &reobj.clsid)))
#else
        if (!SUCCEEDED(GetClassFile(AtoW(szFile), &reobj.clsid)))
#endif
            MessageBox(m_hwndDoc,
                       TEXT("No CLSID, but forging on"),
                       szAppName,
                       MB_ICONINFORMATION | MB_OK);
    }

    // Do we want an iconized version ?
    if (io.dwFlags & IOF_CHECKDISPLAYASICON)
    {
        BOOL fUpdate;               // Can't pass in NULL instead of &this

        // This will update dvaspect on success
        if(OleStdSwitchDisplayAspect(poleobj,
                                     &reobj.dvaspect,
                                     DVASPECT_ICON,
                                     io.hMetaPict,
                                     TRUE,
                                     FALSE,
                                     NULL,
                                     &fUpdate))
        {
            MessageBox(m_hwndDoc,
                       TEXT("Object couldn't be displayed as an icon."),
                       szAppName,
                       MB_ICONEXCLAMATION | MB_OK);
        }
    }

    // Put the thing in the edit control
    if (m_pReObj->InsertObject(&reobj))
    {
        MessageBox(m_hwndDoc,
                   TEXT("Object couldn't be inserted"),
                   szAppName,
                   MB_ICONEXCLAMATION | MB_OK);
        goto error;
    }

    bResult = TRUE;     // Success!

    // Do show verb only on new objects
    if (io.dwFlags & IOF_SELECTCREATENEW)
    {
        // Get object position
        POINT pt;
        SendMessage(m_hwndEdit, EM_POSFROMCHAR, (WPARAM)&pt, REO_IOB_SELECTION);

        RECT rect = {0, 0, 50, 50};
        OffsetRect(&rect, pt.x, pt.y);

        poleobj->DoVerb(OLEIVERB_SHOW,
                        NULL,
                        polesite,
                        0,
                        m_hwndEdit,
                        &rect);
    }

error:
    if (io.hMetaPict)
        OleUIMetafilePictIconFree(io.hMetaPict);
    if (polesite)
        polesite->Release();
    if (pstgItem)
        pstgItem->Release();
    if (poleobj)
        poleobj->Release();

    return bResult;
}


//
//	CRichEditDoc::PasteSpecial
//
//	Purpose:
//		Paste object from clipboard into RichEdit control
//
//	Arguments:
//		None
//
//	Returns:
//		BOOL                Success/failure
//

BOOL CRichEditDoc::PasteSpecial()
{
    MessageBox(m_hwndDoc, "Not yet implemented.", szAppName, MB_ICONSTOP | MB_OK);
    return FALSE;
}


//
//	CRichEditDoc::ConvertObject
//
//	Purpose:
//		Convert object to new type
//
//	Arguments:
//		None
//
//	Returns:
//		BOOL                Success/failure
//

BOOL CRichEditDoc::ConvertObject()
{
    MessageBox(m_hwndDoc, "Not yet implemented.", szAppName, MB_ICONSTOP | MB_OK);
    return FALSE;
}


//
//	CRichEditDoc::SetFileName
//
//	Purpose:
//		Set new Document name
//
//	Arguments:
//		LPTSTR              New name
//
//	Returns:
//		BOOL                Success/failure
//

BOOL CRichEditDoc::SetFileName(LPTSTR pszFileName)
{
    if (pszFileName)
    {
        lstrcpyn(m_szFileName, pszFileName, DIM(m_szFileName));
        m_fFileNameSet = TRUE;

        if (m_hwndDoc)
            SetWindowText(m_hwndDoc, m_szFileName);

        if (m_pReObj)
            m_pReObj->SetHostNames(szAppName, m_szFileName);

        return TRUE;
    }

    return FALSE;
}


//
//	CRichEditDoc::GetFileName
//
//	Purpose:
//		Return Document name
//
//	Arguments:
//		LPTSTR              Buffer to return name
//      UINT                Size of buffer (chars for Unicode, bytes for MBCS)
//
//	Returns:
//		BOOL                Success/failure
//

BOOL CRichEditDoc::GetFileName(LPTSTR pszFileName, UINT cbFileName)
{
    lstrcpyn(pszFileName, m_szFileName, cbFileName);
    return m_fFileNameSet;
}


//
//  CRichEditOleCallback implementation
//

//
//  CRichEditOleCallback IUnknown implementation
//
//  CRichEditOleCallback::QueryInterface
//  CRichEditOleCallback::AddRef
//  CRichEditOleCallback::Release
//
//  COMMENTS:
//    Delegates back to the Document object
//

STDMETHODIMP CRichEditOleCallback::QueryInterface(REFIID riid, LPVOID* ppvObj)
{
    return m_pDoc->QueryInterface(riid, ppvObj);
}

STDMETHODIMP_(ULONG) CRichEditOleCallback::AddRef()
{
    return m_pDoc->AddRef();
}

STDMETHODIMP_(ULONG) CRichEditOleCallback::Release()
{
    return m_pDoc->Release();
}


//
//  CRichEditOleCallback::GetNewStorage
//
//  Purpose:
//      Gets storage for a new object.
//
//  Arguments:
//      LPSTORAGE*          Where to return storage.
//
//  Returns:
//      HRESULT	            Error status.
//
//  Comments:
//    Implemented in the Document object
//

STDMETHODIMP CRichEditOleCallback::GetNewStorage(LPSTORAGE* ppstg)
{
    return m_pDoc->GetNewStorage(ppstg);
}


//
//  CRichEditOleCallback::GetInPlaceContext
//
//  Purpose:
//      Gets context information for an in place object.
//
//  Arguments:
//      LPOLEINPLACEFRAME*      Frame window object.
//      LPOLEINPLACEUIWINDOW*   Document window object.
//      LPOLEINPLACEFRAMEINFO   Frame window information.
//
//  Returns:
//      HRESULT                 Error status.
//

STDMETHODIMP CRichEditOleCallback::GetInPlaceContext(LPOLEINPLACEFRAME* ppFrame,
                                                     LPOLEINPLACEUIWINDOW* ppDoc,
                                                     LPOLEINPLACEFRAMEINFO pFrameInfo)
{
    if (!ppFrame || !ppDoc || !pFrameInfo)
        return E_INVALIDARG;

    // Return interface pointers
    *ppDoc = (LPOLEINPLACEUIWINDOW)m_pDoc;
    m_pDoc->AddRef();

    *ppFrame = pFrame;
    if (pFrame)
        pFrame->AddRef();

    // Fill in default frame window information
    pFrameInfo->fMDIApp = TRUE;
    pFrameInfo->hwndFrame = hwndMain;
    pFrameInfo->haccel = hAccelInPlace;
    pFrameInfo->cAccelEntries = CopyAcceleratorTable(hAccelInPlace, NULL, 0);
                                // 'NULL' ==> return # of entries in table
    return S_OK;
}


//
//  CRichEditOleCallback::ShowContainerUI
//
//  Purpose:
//      Displays or hides container UI.
//
//  Arguments:
//      BOOL			
//
//  Returns:
//      HRESULT                 Error status.
//

STDMETHODIMP CRichEditOleCallback::ShowContainerUI(BOOL fShowContainerUI)
{
    COleInPlaceFrame* pfr = (COleInPlaceFrame*)pFrame;

    if (!pfr) // should never be NULL, but just in case...
        return E_FAIL;

	if(fShowContainerUI)
    {
        pfr->ReinstateUI();
        SetFocus(m_pDoc->m_hwndEdit);
    }
    else
        pfr->ShowUIAndTools(FALSE, FALSE);

	return S_OK;
}


STDMETHODIMP CRichEditOleCallback::QueryInsertObject(LPCLSID /* pclsid */,
                                                     LPSTORAGE /* pstg */,
                                                     LONG /* cp */)
{
    // Let RichEdit handle this
	return S_OK;
}


STDMETHODIMP CRichEditOleCallback::DeleteObject(LPOLEOBJECT /* poleobj */)
{
    // This call is a notification only
	return S_OK;
}


STDMETHODIMP CRichEditOleCallback::QueryAcceptData(LPDATAOBJECT /* pdataobj */,
                                                   CLIPFORMAT* /* pcfFormat */,
                                                   DWORD /* reco */,
                                                   BOOL /* fReally */,
                                                   HGLOBAL /* hMetaPict */)
{
    // accept everything
	return S_OK;
}


STDMETHODIMP CRichEditOleCallback::ContextSensitiveHelp(BOOL fEnterMode)
{
    return E_NOTIMPL;
}


STDMETHODIMP CRichEditOleCallback::GetClipboardData(CHARRANGE* /* pchrg */,
                                                    DWORD /* reco */,
                                                    LPDATAOBJECT* ppdataobj)
{
    *ppdataobj = NULL;  // NULL out pointer

	// delegate back to RichEdit
	return E_NOTIMPL;
}


STDMETHODIMP CRichEditOleCallback::GetDragDropEffect(BOOL fDrag,
                                                     DWORD grfKeyState,
                                                     LPDWORD pdwEffect)
{
	// delegate back to RichEdit
	return E_NOTIMPL;
}


//
//  FUNCTION: CRichEditOleCallback::GetContextMenu(WORD, LPOLEOBJECT, CHARRANGE*, HMENU*)
//
//  PURPOSE: Creates context menu for alternate mouse clicks in RichEdit
//
//  PARAMETERS:
//    seltype   - Selection type
//    poleobj   - IOleObject interface of selected object, if any
//    pchrg     - Selection range
//    phmenu    - Out param for returning menu
//
//  RETURN VALUE:
//
//  HRESULT     - Error status
//
//  COMMENTS:
//
//

STDMETHODIMP CRichEditOleCallback::GetContextMenu(WORD seltype,
                                                  LPOLEOBJECT poleobj,
                                                  CHARRANGE* pchrg,
                                                  HMENU* phmenu)
{
	HMENU hmenuVerbs = NULL;

    *phmenu = CreatePopupMenu();
    if(!*phmenu)
        return E_FAIL;

	// Put the verbs on the menu if we have an object
	if(poleobj)
	{
		OleUIAddVerbMenu(poleobj,
		                 NULL,
		                 *phmenu,
		                 0,
		                 IDM_OBJECTMIN,
		                 IDM_OBJECTMAX,
		                 TRUE,
		                 IDM_EDITCONVERTOBJECT,
		                 &hmenuVerbs);
        AppendMenu(*phmenu, MF_SEPARATOR, 0, NULL);
	}

    // Add the cut, copy, paste verbs

    UINT mf = MF_STRING | MF_BYCOMMAND;
    mf |= (pchrg->cpMin == pchrg->cpMax ? MF_GRAYED : MF_ENABLED);
    AppendMenu(*phmenu, mf, IDM_EDITCUT, TEXT("Cu&t"));
    AppendMenu(*phmenu, mf, IDM_EDITCOPY, TEXT("&Copy"));

    mf = MF_STRING | MF_BYCOMMAND;
    mf |= (SendMessage(m_pDoc->m_hwndEdit, EM_CANPASTE, 0, 0) ? MF_ENABLED : MF_GRAYED);
    AppendMenu(*phmenu, mf, IDM_EDITPASTE, TEXT("&Paste"));

    return S_OK;
}
