// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// Copyright (C) 1993-1995  Microsoft Corporation.  All Rights Reserved.
//
//  MODULE:   OleHelp.cpp
//
//  PURPOSE:  Random OLE helper functions
//
//  FUNCTIONS:
//    OleUIMetafilePictIconFree - (OLESTD) Delete metafile and METAFILEPICT
//    OleStdSwitchDisplayAspect - (OLESTD) Switch between DVASPECT_ICON and
//                                  DVASPECT_CONTENT
//    UpdateObjectMenu          - Add verbs to Edit menu
//
//  COMMENTS:
//

 
#include <windows.h>     // required for all Windows applications
#include <windowsx.h>
#include "globals.h"     // prototypes specific to this application
#include "resource.h"
#include <oledlg.h>
#include <richedit.h>
#include <richole.h>
#include "olehelp.h"
#include "IFrame.h"
#include "ReDoc.h"

// Used by Ansi/Unicode conversion macros in writepad.inl
int _temp;


/*
 * OleUIMetafilePictIconFree    [stolen from OLE2UI]
 *
 * Purpose:
 *  Deletes the metafile contained in a METAFILEPICT structure and
 *  frees the memory for the structure itself.
 *
 * Parameters:
 *  hMetaPict       HGLOBAL metafilepict structure created in
 *                  OleUIMetafilePictFromIconAndLabel
 *
 * Return Value:
 *  None
 */

STDAPI_(void) OleUIMetafilePictIconFree(HGLOBAL hMetaPict)
{
    LPMETAFILEPICT pMF;
    
    if (NULL == hMetaPict)
        return;
    
    pMF = (LPMETAFILEPICT)GlobalLock(hMetaPict);
    
    if (NULL != pMF)
    {
        if (NULL != pMF->hMF)
            DeleteMetaFile(pMF->hMF);
    }
    
    GlobalUnlock(hMetaPict);
    GlobalFree(hMetaPict);
    return;
}

/* OleStdSwitchDisplayAspect    [stolen from OLE2UI]
** -------------------------
**    Switch the currently cached display aspect between DVASPECT_ICON
**    and DVASPECT_CONTENT.
**
**    NOTE: when setting up icon aspect, any currently cached content
**    cache is discarded and any advise connections for content aspect
**    are broken.
**
**    RETURNS:
**      S_OK -- new display aspect setup successfully
**      E_INVALIDARG -- IOleCache interface is NOT supported (this is
**                  required).
**      <other SCODE> -- any SCODE that can be returned by
**                  IOleCache::Cache method.
**      NOTE: if an error occurs then the current display aspect and
**            cache contents unchanged.
*/

STDAPI OleStdSwitchDisplayAspect(LPOLEOBJECT  lpOleObj,
                                 LPDWORD      lpdwCurAspect,
                                 DWORD        dwNewAspect,
                                 HGLOBAL      hMetaPict,
                                 BOOL         fDeleteOldAspect,
                                 BOOL         fSetupViewAdvise,
                                 LPADVISESINK lpAdviseSink,
                                 LPBOOL       lpfMustUpdate)
{
    LPOLECACHE      lpOleCache = NULL;
    LPVIEWOBJECT    lpViewObj = NULL;
    LPENUMSTATDATA  lpEnumStatData = NULL;
    STATDATA        StatData;
    FORMATETC       FmtEtc;
    STGMEDIUM       Medium;
    DWORD           dwAdvf;
    DWORD           dwNewConnection;
    DWORD           dwOldAspect = *lpdwCurAspect;
    HRESULT         hrErr;

    if (lpfMustUpdate)
        *lpfMustUpdate = FALSE;

    lpOleObj->QueryInterface(IID_IOleCache, (LPVOID*)&lpOleCache);

    // if IOleCache* is NOT available, do nothing
    if (!lpOleCache)
        return E_INVALIDARG;

    // Setup new cache with the new aspect
    FmtEtc.cfFormat = 0;     // whatever is needed to draw
    FmtEtc.ptd      = NULL;
    FmtEtc.dwAspect = dwNewAspect;
    FmtEtc.lindex   = -1;
    FmtEtc.tymed    = TYMED_NULL;

    /* OLE2NOTE: if we are setting up Icon aspect with a custom icon
    **    then we do not want DataAdvise notifications to ever change
    **    the contents of the data cache. thus we set up a NODATA
    **    advise connection. otherwise we set up a standard DataAdvise
    **    connection.
    */
    if (dwNewAspect == DVASPECT_ICON && hMetaPict)
        dwAdvf = ADVF_NODATA;
    else
        dwAdvf = ADVF_PRIMEFIRST;

    hrErr = lpOleCache->Cache(&FmtEtc, dwAdvf, &dwNewConnection);
    if (!SUCCEEDED(hrErr))
    {
        lpOleCache->Release();
        return hrErr;
    }

    *lpdwCurAspect = dwNewAspect;

    /* OLE2NOTE: if we are setting up Icon aspect with a custom icon,
    **    then stuff the icon into the cache. otherwise the cache must
    **    be forced to be updated. set the *lpfMustUpdate flag to tell
    **    caller to force the object to Run so that the cache will be
    **    updated.
    */
    if (dwNewAspect == DVASPECT_ICON && hMetaPict)
    {
       FmtEtc.cfFormat = CF_METAFILEPICT;
       FmtEtc.ptd      = NULL;
       FmtEtc.dwAspect = DVASPECT_ICON;
       FmtEtc.lindex   = -1;
       FmtEtc.tymed    = TYMED_MFPICT;

       Medium.tymed          = TYMED_MFPICT;
       Medium.hGlobal        = hMetaPict;
       Medium.pUnkForRelease = NULL;

       hrErr = lpOleCache->SetData(&FmtEtc, &Medium, FALSE /* fRelease */);
    }
    else
    {
        if (lpfMustUpdate)
            *lpfMustUpdate = TRUE;
    }

    if (fSetupViewAdvise && lpAdviseSink)
    {
        /* OLE2NOTE: re-establish the ViewAdvise connection */
        lpOleObj->QueryInterface(IID_IViewObject, (LPVOID*)&lpViewObj);

        if (lpViewObj)
        {
            lpViewObj->SetAdvise(dwNewAspect, 0, lpAdviseSink);
            lpViewObj->Release();
        }
    }

    /* OLE2NOTE: remove any existing caches that are set up for the old
    **    display aspect. It WOULD be possible to retain the caches set
    **    up for the old aspect, but this would increase the storage
    **    space required for the object and possibly require additional
    **    overhead to maintain the unused cachaes. For these reasons the
    **    strategy to delete the previous caches is prefered. If it is a
    **    requirement to quickly switch between Icon and Content
    **    display, then it would be better to keep both aspect caches.
    */

    if (fDeleteOldAspect)
    {
        hrErr = lpOleCache->EnumCache(&lpEnumStatData);

        while (hrErr == NOERROR)
        {
            hrErr = lpEnumStatData->Next(1, &StatData, NULL);
            if (hrErr != NOERROR)
                break;              // DONE! no more caches.

            if (StatData.formatetc.dwAspect == dwOldAspect)
            {
                // Remove previous cache with old aspect
                lpOleCache->Uncache(StatData. dwConnection);
            }
        }

        if (lpEnumStatData)
        {
            if (lpEnumStatData->Release() != 0)
                MessageBox(NULL,
                           TEXT("OleStdSwitchDisplayAspect: Cache enumerator NOT released"),
                           NULL,
                           MB_ICONEXCLAMATION | MB_OK);
        }
    }

    if (lpOleCache)
        lpOleCache->Release();

    return NOERROR;
}


/* 
 *	COleInPlaceFrame_New
 *
 *	Purpose:
 *		'C' function to create a new COleInPlaceFrame interface object.
 *
 *	Arguments:
 *		HWND                    Application frame window
 *		LPOLEINPLACEFRAME*      Out param to return the newly created interface
 *
 *	Returns:
 *		HRESULT		 Error status.
 *
 */

STDMETHODIMP COleInPlaceFrame_New(HWND hwndFrame, LPOLEINPLACEFRAME* ppFrame)
{
    SCODE sc = E_FAIL;

    if (!ppFrame)
        return sc;

    *ppFrame = NULL;

    if (!hwndFrame)
        return sc;

    *ppFrame = new COleInPlaceFrame(hwndFrame);

    if (*ppFrame)
    {
        (*ppFrame)->AddRef();     // Start with one reference
        sc = S_OK;
    }

    return sc;
}


BOOL UpdateObjectMenu(HWND hwndEdit, HMENU hEditMenu, int iVerbMenuPos)
{
    if (!hEditMenu)
        return FALSE;

    HMENU hMenuVerb = NULL;
    REOBJECT reobj = {0};
    reobj.cbStruct = sizeof(reobj);

    if (hwndEdit)
    {
        // single object selected?
        if (SEL_OBJECT == SendMessage(hwndEdit, EM_SELECTIONTYPE, 0, 0))
        {
            LPRICHEDITOLE preole = NULL;
            SendMessage(hwndEdit, EM_GETOLEINTERFACE, 0, (LPARAM)&preole);
            if (preole)
            {
                preole->GetObject(REO_IOB_SELECTION, &reobj, REO_GETOBJ_POLEOBJ);
                preole->Release();
            }
        }
    }

    HRESULT hr = OleUIAddVerbMenu(reobj.poleobj,        // can be NULL
                                  NULL,
                                  hEditMenu,
                                  iVerbMenuPos,
                                  IDM_OBJECTMIN,
                                  IDM_OBJECTMAX,
                                  TRUE,
                                  IDM_EDITCONVERTOBJECT,
                                  &hMenuVerb);

    if (reobj.poleobj)
        reobj.poleobj->Release();

    return SUCCEEDED(hr);
}


// C wrappers for CRichEditDoc functions

HRESULT DoVerb(HWND hwndDoc, LONG iVerb)
{
    if (!hwndDoc)
        return E_INVALIDARG;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->DoVerb(iVerb);

    return E_FAIL;
}


BOOL InsertObject(HWND hwndDoc)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->InsertObject();

    return FALSE;
}


BOOL PasteSpecial(HWND hwndDoc)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->PasteSpecial();

    return FALSE;
}


BOOL ConvertObject(HWND hwndDoc)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->ConvertObject();

    return FALSE;
}


BOOL IsFileNameSet(HWND hwndDoc)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->IsFileNameSet();

    return FALSE;
}


BOOL SetFileName(HWND hwndDoc, LPTSTR pszFileName)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->SetFileName(pszFileName);

    return FALSE;
}


BOOL GetFileName(HWND hwndDoc, LPTSTR pszFileName, UINT cbFileName)
{
    if (!hwndDoc)
        return FALSE;

    CRichEditDoc* pDoc = (CRichEditDoc*)GetWindowLong(hwndDoc, GWL_USERDATA);
    if (pDoc)
        return pDoc->GetFileName(pszFileName, cbFileName);

    return FALSE;
}
