/*
 *	IFrame.cpp -- IOleInPlaceFrame interface implementation
 */

#include <windows.h>     // required for all Windows applications
#include <windowsx.h>
#include "globals.h"     // prototypes specific to this application
#include "resource.h"
#include "olehelp.h"
#include "statbar.h"
#include "toolbar.h"
#include "ruler.h"
#include "IFrame.h"


/*
 *	COleInPlaceFrame Implementation
 */

/*
 *	COleInPlaceFrame::QueryInterface
 *
 *	Purpose:
 *		Returns a pointer to the specified interface.
 *
 *	Arguments:
 *		REFIID			Interface requested.
 *		LPUNKNOWN *		Interface we return.
 *
 *	Returns:
 *		HRESULT			Error status.
 */

STDMETHODIMP COleInPlaceFrame::QueryInterface(REFIID riid, LPVOID* ppvObj)
{
    if (!ppvObj)
        return E_INVALIDARG;

    *ppvObj = NULL;

	if (IsEqualIID(riid, IID_IUnknown) ||
		IsEqualIID(riid, IID_IOleWindow) ||
		IsEqualIID(riid, IID_IOleInPlaceUIWindow) ||
		IsEqualIID(riid, IID_IOleInPlaceFrame))
	{
        *ppvObj = this;
	}
	else
		return E_NOINTERFACE;

    AddRef();
	return S_OK;
}


/*
 *	COleInPlaceFrame::AddRef
 *
 *	Purpose:
 *		Increments reference count on the specified object.
 *
 *	Arguments:
 *		None
 *
 *	Returns:
 *		ULONG			New value of reference count.
 */

STDMETHODIMP_(ULONG) COleInPlaceFrame::AddRef()
{
	return ++m_cRef;
}


/*
 *	COleInPlaceFrame::Release
 *
 *	Purpose:
 *		Decrements reference count on the specified object.  If count is
 *		decremented to zero, the object is freed.
 *
 *	Arguments:
 *		None
 *
 *	Returns:
 *		ULONG			New value of reference count.
 */

STDMETHODIMP_(ULONG) COleInPlaceFrame::Release()
{
	ULONG cRef = --m_cRef;

	if(!cRef)
        delete this;

	return cRef;
}


/*
 *	COleInPlaceFrame::GetWindow
 *
 *	Purpose:
 *		Return the window handle of the app-level window for in place use.
 *
 *	Arguments:
 *		HWND*               Out param for window handle
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::GetWindow(HWND* phwnd)
{
    if (!phwnd)
        return E_INVALIDARG;

    *phwnd = m_hwndFrame;
	return S_OK;
}


/*
 *	COleInPlaceFrame::ContextSensitiveHelp
 *
 *	Purpose:
 *		Notifies the frame that the object has entered Context Help mode.
 *
 *	Arguments:
 *		BOOL                CSH mode
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::ContextSensitiveHelp(BOOL fEnterMode)
{
    if (m_fCSHMode != fEnterMode)
    {
        m_fCSHMode = fEnterMode;

        // this code "trickles" the context sensitive help via shift+f1
        // to the inplace active object.  See the technotes for implementation
        // details.
        if (pActiveObject)
        {
            LPOLEINPLACEOBJECT pipo;
            pActiveObject->QueryInterface(IID_IOleInPlaceObject, (LPVOID*)&pipo);
            if (pipo)
            {
                pipo->ContextSensitiveHelp(fEnterMode);
                pipo->Release();
            }
        }
    }
    return S_OK;
}


/*
 *	COleInPlaceFrame::GetBorder
 *
 *	Purpose:
 *		Returns a RECT structure in which the object can put toolbars and 
 *		similar controls while an object is active in place.
 *
 *	Arguments:
 *      LPRECT lprectBorder - out param to contain the outermost 
 *                            rect for frame adornments
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::GetBorder(LPRECT prcBorder)
{
    if (!prcBorder)
        return E_INVALIDARG;

    GetClientRect(m_hwndFrame, prcBorder);

    // save room for status window
    RECT rect;
    GetWindowRect(hWndStatusbar, &rect);
    prcBorder->bottom -= (rect.bottom - rect.top);

	return S_OK;
}


/*
 *	COleInPlaceFrame::RequestBorderSpace
 *
 *	Purpose:
 *		Determines whether tools can be installed around the objects 
 *		window frame while the object is active in place.
 *
 *	Arguments:
 *		LPCBORDERWIDTHS     Requested border space
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::RequestBorderSpace(LPCBORDERWIDTHS /* pbw */)
{
    // Accept all requests
	return S_OK;
}


/*
 *	COleInPlaceFrame::SetBorderSpace
 *
 *	Purpose:
 *		Allocates space for the border requested in the call to the 
 *		RequestBorderSpace member function.
 *
 *	Arguments:
 *		LPCBORDERWIDTHS     Requested border space
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::SetBorderSpace(LPCBORDERWIDTHS pbw)
{
    RECT rcClient;          // New client window position
    RECT rcClientOld;       // Previous client window position
    RECT rcStatbar;
    int  cyStatbar;

    // Remember previous position of client window
    GetWindowRect(hwndMDIClient, &rcClientOld);
    MapWindowPoints(NULL, m_hwndFrame, (LPPOINT)&rcClientOld, 2);

    // Get client area minus space for status bar
    GetClientRect(m_hwndFrame, &rcClient);
    GetWindowRect(hWndStatusbar, &rcStatbar);
    cyStatbar = rcStatbar.bottom - rcStatbar.top;
    rcClient.bottom -= cyStatbar;

    if (pbw)
    {
        // Set space for object tools
        rcClient.left   += pbw->left;
        rcClient.top    += pbw->top;
        rcClient.right  -= pbw->right;
        rcClient.bottom -= pbw->bottom;

        // Save new border widths
        CopyRect(&rcBorderSpace, pbw);

        // Add room for Status bar in saved border widths
        rcBorderSpace.bottom += cyStatbar;
    }
    else
    {
        // No tools (except status bar)
        SetRect(&rcBorderSpace, 0, 0, 0, cyStatbar);
    }

    // Only move the client window (& children) if necessary
    if (memcmp(&rcClient, &rcClientOld, sizeof(RECT)))
        MoveClient(&rcClient,                           // new position
                   rcClient.left - rcClientOld.left,    // horizontal shift
                   rcClient.top - rcClientOld.top);     // vertical shift

    return S_OK;
}


/*
 *	COleInPlaceFrame::SetActiveObject
 *
 *	Purpose:
 *		Called by the object to provide the frame window a direct channel 
 *		of communication with the active in-place object.
 *
 *	Arguments:
 *		LPOLEINPLACEACTIVEOBJECT    New active object (or NULL, if none)
 *      LPCOLESTR                   Object name
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::SetActiveObject(LPOLEINPLACEACTIVEOBJECT pipaobj,
                                               LPCOLESTR /* szObjName */)
{
	// Free any existing active object
	if(pActiveObject)
		pActiveObject->Release();

	// If we're given an object, AddRef it; update our remembered ipaobj
	if(pipaobj)
		pipaobj->AddRef();

	pActiveObject = pipaobj;

	return S_OK;
}


/*
 *	COleInPlaceFrame::InsertMenus
 *
 *	Purpose:
 *		Called by the object server to allow the container to insert 
 *		its menu groups in the composite menu that will be used during
 *		the in-place session.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::InsertMenus(HMENU hmenuShared,
                                           LPOLEMENUGROUPWIDTHS pmgw)
{
    if (!(hmenuShared && pmgw))
        return E_INVALIDARG;

    TCHAR szMenuString[32];

    // add file menu
    GetMenuString(hmenuMain, 0, szMenuString, 32, MF_BYPOSITION);
    AppendMenu(hmenuShared,
               MF_POPUP,
               (UINT)GetSubMenu(hmenuMain, 0),
               szMenuString);
	pmgw->width[0] = 1;


    // no 'container' menus
	pmgw->width[2] = 0;


    // add window menu
    GetMenuString(hmenuMain, 3, szMenuString, 32, MF_BYPOSITION);
    AppendMenu(hmenuShared,
               MF_POPUP,
               (UINT)GetSubMenu(hmenuMain, 3),
               szMenuString);
	pmgw->width[4] = 1;


	return S_OK;
}


/*
 *	COleInPlaceFrame::SetMenu
 *
 *	Purpose:
 *		Installs the composite menu into the window frame containing
 *		the object that is being activated in place.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::SetMenu(HMENU hmenuShared,
                                       HOLEMENU holemenu,
                                       HWND hwndActiveObject)
{
	// Update the menu on the MDI frame window
    ::SetMenu(m_hwndFrame, hmenuShared && holemenu ? hmenuShared : hmenuMain);
	DrawMenuBar(m_hwndFrame);

	// Pass the menu descriptor to OLE
	return OleSetMenuDescriptor(holemenu,
	                            m_hwndFrame,
	                            hwndActiveObject,
	                            (LPOLEINPLACEFRAME)this,
	                            pActiveObject);
}


/*
 *	COleInPlaceFrame::RemoveMenus
 *
 *	Purpose:
 *		Called by the object server to give the container a chance to 
 *		remove its menu elements from the in-place composite menu.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::RemoveMenus(HMENU hmenuShared)
{
    if (!hmenuShared)
        return E_INVALIDARG;

    int nResult;

    while ((nResult = GetMenuItemCount(hmenuShared)) && (nResult != -1))
        RemoveMenu(hmenuShared, 0, MF_BYPOSITION);

	return S_OK;
}


/*
 *	COleInPlaceFrame::SetStatusText
 *
 *	Purpose:
 *		Sets and displays status text about the in-place object in the
 *		containers frame window status line.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::SetStatusText(LPCOLESTR szStatusText)
{
    if (!szStatusText)
        szStatusText = AtoW(szTitle);

#ifdef UNICODE
    UpdateStatusBar(szStatusText, 0, 0);
#else
    UpdateStatusBar(WtoA(szStatusText), 0, 0);
#endif
	return S_OK;
}


/*
 *	COleInPlaceFrame::EnableModeless
 *
 *	Purpose:
 *		Enables or disables modeless dialogs of the frame.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::EnableModeless(BOOL /* fEnable */)
{
    // No modeless dialogs here
	return S_OK;
}


/*
 *	COleInPlaceFrame::TranslateAccelerator
 *
 *	Purpose:
 *		Translates keystrokes intended for the container frame while
 *		an object is active in place.
 *
 *	Arguments:
 *		ITPOLEINPLACESITE *	Client site.
 *
 *	Returns:
 *		HRESULT				Error status.
 */

STDMETHODIMP COleInPlaceFrame::TranslateAccelerator(LPMSG pmsg, WORD wID)
{
    if (!pmsg)
        return E_INVALIDARG;

    if (!TranslateMDISysAccel(hwndMDIClient, pmsg))
        if (!::TranslateAccelerator(m_hwndFrame, hAccelInPlace, pmsg))
            return S_FALSE;         // keystroke not used

	return S_OK;                    // keystroke used
}


/*
 *	COleInPlaceFrame::MoveClient
 *
 *	Purpose:
 *		Moves MDI client window while moving all children the same amount
 *      in the opposite direction so they appear to stay in the same place.
 *
 *	Arguments:
 *		LPRECT              New position of MDI client window
 *      int                 Horizontal shift from previous position
 *      int                 Vertical shift from previous position
 *
 *	Returns:
 *		none
 */

void COleInPlaceFrame::MoveClient(LPRECT prc, int dx, int dy)
{
    HWND hwnd;
    RECT rect;

    // Suppress repainting while moving windows
    SendMessage(m_hwndFrame, WM_SETREDRAW, FALSE, 0);

    // Move MDI client window to new position
    SetWindowPos(hwndMDIClient,
                 NULL,
                 prc->left,
                 prc->top,
                 prc->right - prc->left,
                 prc->bottom - prc->top,
                 SWP_NOZORDER | SWP_NOACTIVATE);

    // Move all client children in the opposite direction as the MDI client
    // so that they maintain the same position on the screen.
    if (!(dx ==0 && dy ==0))
    {
        for (hwnd = ::GetWindow(hwndMDIClient, GW_CHILD);
             hwnd != NULL;
             hwnd = ::GetWindow(hwnd, GW_HWNDNEXT))
        {
            if (!IsZoomed(hwnd))
            {
                GetWindowRect(hwnd, &rect);
                MapWindowPoints(NULL, hwndMDIClient, (LPPOINT)&rect, 2);

                SetWindowPos(hwnd,
                             NULL,
                             rect.left - dx,
                             rect.top - dy,
                             rect.right - rect.left,
                             rect.bottom - rect.top,
                             SWP_NOZORDER | SWP_NOACTIVATE | SWP_NOSIZE);
            }
        }
    }

    // Turn frame window painting back on
    SendMessage(m_hwndFrame, WM_SETREDRAW, TRUE, 0);

    // Force a full repaint
    RedrawWindow(m_hwndFrame,
                 NULL,
                 NULL,
                 RDW_ERASE | RDW_INVALIDATE | RDW_UPDATENOW | RDW_ALLCHILDREN);
}


/*
 *	COleInPlaceFrame::ShowUIAndTools
 *
 *	Purpose:
 *		Shows or hides the application's menu and/or tools.
 *
 *	Arguments:
 *		BOOL                TRUE ==> show container's tools
 *		BOOL                TRUE ==> show container's menu
 *
 *	Returns:
 *		none
 */

void COleInPlaceFrame::ShowUIAndTools(BOOL fShow, BOOL fMenu)
{
    if (fShow)
    {
        if (fMenu)
            SetMenu(NULL, NULL, NULL);

        ShowWindow(hWndToolbar, SW_SHOW);
        RedrawWindow(hWndToolbar,
                     NULL,
                     NULL,
                     RDW_INVALIDATE | RDW_UPDATENOW | RDW_ALLCHILDREN);
    }
    else
        ShowWindow(hWndToolbar, SW_HIDE);

    // Enable or disable the Ruler window
    EnableWindow(GetDlgItem(GetActiveMDIChild(), ID_RULER), fShow);
}


/*
 *	COleInPlaceFrame::ReinstateUI
 *
 *	Purpose:
 *		Restores the application's menu and tools.
 *
 *	Arguments:
 *		none
 *
 *	Returns:
 *		none
 */

void COleInPlaceFrame::ReinstateUI(void)
{
    RECT rect;

    // Set default border widths (statbar is automatic)
    GetWindowRect(hWndToolbar, &rect);
    SetRect(&rect, 0, rect.bottom - rect.top, 0, 0);
    SetBorderSpace(&rect);

    // Show our menu and tools
    ShowUIAndTools(TRUE, TRUE);
}
