// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//
// Copyright (C) 1993-1995  Microsoft Corporation.  All Rights Reserved.
//
//  MODULE:   about.c
//
//  PURPOSE:   Displays the "FileOpenDialog" dialog box
//
//  FUNCTIONS:
//    CmdDialog        - Displays the "FileOpenDialog" dialog box
//    FOpenDialog      - Processes messages for "FileOpenDialog" dialog box.
//    MsgDialogInit    - To initialize the Dialog box with version info
//                      from resources.
//    MsgDialogCommand - Process WM_COMMAND message sent to the Dialog box.
//    CmdDialogDone    - Free the Dialog box and related data.
//
//  COMMENTS:
//
//

#include <windows.h>            // required for all Windows applications
#include "globals.h"            // prototypes specific to this application
#include "resource.h"

LRESULT CALLBACK FOpenDialog(HWND, UINT, WPARAM, LPARAM);
LRESULT MsgDialogInit(HWND, UINT, WPARAM, LPARAM);
LRESULT MsgDialogHelp(HWND, UINT, WPARAM, LPARAM);
LRESULT MsgDialogContextMenu(HWND, UINT, WPARAM, LPARAM);
LRESULT MsgDialogCommand(HWND, UINT, WPARAM, LPARAM);
LRESULT CmdDialogDone(HWND, WORD, WORD, HWND);

// Used for processing WinHelp (HELP_WM_HELP and HELP_CONTEXTMENU)
static const DWORD aSampleMenuHelpIDs[] =
{
	edt1, 	IDH_EDT1	,
	lst1,		IDH_LST1	,
	lst2,		IDH_LST2	,
	cmb1,		IDH_CMB1	,
	cmb2,		IDH_CMB2	,
	pshHelp,	IDH_HELP	,
	chx1,		IDH_CHX1	,
	psh14,	IDH_PSH14,
   0,    0
}  ;

// Dialog dialog message table definition.
MSD rgmsdDialog[] =
{
    {WM_COMMAND,     MsgDialogCommand},
    {WM_HELP,        MsgDialogHelp},
    {WM_CONTEXTMENU, MsgDialogContextMenu},
    {WM_INITDIALOG,  MsgDialogInit}
};

MSDI msdiDialog =
{
    sizeof(rgmsdDialog) / sizeof(MSD),
    rgmsdDialog,
    edwpNone
};

// Dialog dialog command table definition.
CMD rgcmdDialog[] =
{
    {IDOK,     CmdDialogDone},
    {IDCANCEL, CmdDialogDone}
};

CMDI cmdiDialog =
{
    sizeof(rgcmdDialog) / sizeof(CMD),
    rgcmdDialog,
    edwpNone
};

// Module specific "globals"  Used when a variable needs to be
// accessed in more than on handler function.

HFONT hFontCopyright;

//
//  FUNCTION: CmdDialog(HWND, WORD, WORD, HWND)
//
//  PURPOSE: Displays the "Dialog" dialog box
//
//  PARAMETERS:
//    hwnd      - Window handle
//    wCommand  - IDM_Dialog (unused)
//    wNotify   - Notification number (unused)
//    hwndCtrl  - NULL (unused)
//
//  RETURN VALUE:
//
//    Always returns 0 - Message handled
//
//  COMMENTS:
//    To process the IDM_Dialog message, call DialogBox() to display the
//    Dialog dialog box.

LRESULT CmdDialog(HWND hwnd, WORD wCommand, WORD wNotify, HWND hwndCtrl)
{
    DialogBox(hInst, "FILEOPENDLG", hwnd, (DLGPROC)FOpenDialog);
    return 0;
}


//
//  FUNCTION: FOpenDialog(HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE:  Processes messages for "Dialog" dialog box.
//
//  PARAMETERS:
//    hdlg - window handle of the dialog box
//    wMessage - type of message
//    wparam - message-specific information
//    lparam - message-specific information
//
//  RETURN VALUE:
//    TRUE - message handled
//    FALSE - message not handled
//
//  COMMENTS:
//
//     Display version information from the version section of the
//     application resource.
//
//     Wait for user to click on "Ok" button, then close the dialog box.
//

LRESULT CALLBACK FOpenDialog(HWND hdlg, UINT uMessage, WPARAM wparam,
                             LPARAM lparam)
{
    return DispMessage(&msdiDialog, hdlg, uMessage, wparam, lparam);
}


//
//  FUNCTION: MsgDialogInit(HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE: To initialize the Dialog box with version info from resources.
//
//  PARAMETERS:
//    hwnd - The window handing the message.
//    uMessage - The message number. (unused).
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//    Uses the version apis to retrieve version information for
//    each of the static text boxes in the Dialog box.
//

LRESULT MsgDialogInit(HWND hdlg, UINT uMessage, WPARAM wparam, LPARAM lparam)
{
    // Center the dialog over the application window
    CenterWindow(hdlg, GetWindow(hdlg, GW_OWNER));
   return TRUE;
}

//
//  FUNCTION: MsgDialogHelp (HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE: To process the WM_HELP message
//
//  PARAMETERS:
//    hwnd - The window handing the message.
//    uMessage - The message number. (unused).
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//    Uses the version apis to retrieve version information for
//    each of the static text boxes in the Dialog box.
//

LRESULT MsgDialogHelp (HWND hdlg, UINT uMessage, WPARAM wparam, LPARAM lparam)
{
   LPHELPINFO lphi;

   lphi = (LPHELPINFO)lparam;
	if (lphi->iContextType == HELPINFO_WINDOW)   // must be for a control/window
   {
		WinHelp (lphi->hItemHandle,
               "GEN32.HLP",
               HELP_WM_HELP,
               (DWORD)(LPVOID)aSampleMenuHelpIDs);
   }
		
	return TRUE; 
}

//
//  FUNCTION: MsgDialogContextMenu (HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE: To process the WM_CONTEXTMENU message
//           which is sent when the Right Mouse button is clicked.
//
//  PARAMETERS:
//    hwnd - The window handing the message.
//    uMessage - The message number. (unused).
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//    Uses the version apis to retrieve version information for
//    each of the static text boxes in the Dialog box.
//

LRESULT MsgDialogContextMenu (HWND hdlg, 
  				  	               UINT uMessage, 
										WPARAM wparam, 
                              LPARAM lparam)
{
	WinHelp ((HWND)wparam,
            "GEN32.HLP",
             HELP_CONTEXTMENU,
            (DWORD)(LPVOID)aSampleMenuHelpIDs);

	return TRUE; 
}
//
//  FUNCTION: MsgDialogCommand(HWND, UINT, WPARAM, LPARAM)
//
//  PURPOSE: Process WM_COMMAND message sent to the Dialog box.
//
//  PARAMETERS:
//    hwnd - The window handing the message.
//    uMessage - The message number. (unused).
//    wparam - Message specific data (unused).
//    lparam - Message specific data (unused).
//
//  RETURN VALUE:
//    Always returns 0 - message handled.
//
//  COMMENTS:
//    Uses this DipsCommand function defined in wndproc.c combined
//    with the cmdiDialog structure defined in this file to handle
//    the command messages for the Dialog dialog box.
//

LRESULT MsgDialogCommand(HWND   hwnd, 
                        UINT   uMessage, 
                        WPARAM wparam, 
                        LPARAM lparam)
{
    return DispCommand(&cmdiDialog, hwnd, wparam, lparam);
}

//
//  FUNCTION: CmdDialogDone(HWND, WORD, HWND)
//
//  PURPOSE: Free the Dialog box and related data.
//
//  PARAMETERS:
//    hwnd - The window handling the command.
//    wCommand - The command to be handled (unused).
//    wNotify   - Notification number (unused)
//    hwndCtrl - NULL (unused).
//
//  RETURN VALUE:
//    Always returns TRUE.
//
//  COMMENTS:
//    Calls EndDialog to finish the dialog session.
//

LRESULT CmdDialogDone(HWND hdlg, WORD wCommand, WORD wNotify, HWND hwndCtrl)
{
    EndDialog(hdlg, TRUE);          // Exit the dialog
    return TRUE;
}

