/*-
 * Copyright (c) 1990, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Mike Olson.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
static char sccsid[] = "@(#)bt_open.c	8.3 (Berkeley) 9/16/93";
#endif /* LIBC_SCCS and not lint */

/*
 * Implementation of btree access method for 4.4BSD.
 *
 * The design here was originally based on that of the btree access method
 * used in the Postgres database system at UC Berkeley.  This implementation
 * is wholly independent of the Postgres code.
 */

#include <sys/param.h>
#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define	__DBINTERFACE_PRIVATE
#include <db.h>
#include "btree.h"

static int byteorder __P((void));
static int nroot __P((BTREE *));
#ifndef WIN32
static int tmp __P((void));
#else
HANDLE tmp __P((void));
#endif
/*
 * __BT_OPEN -- Open a btree.
 *
 * Creates and fills a DB struct, and calls the routine that actually
 * opens the btree.
 *
 * Parameters:
 *	fname:	filename (NULL for in-memory trees)
 *	flags:	open flag bits
 *	mode:	open permission bits
 *	b:	BTREEINFO pointer
 *
 * Returns:
 *	NULL on failure, pointer to DB on success.
 *
 */
DB *
__bt_open(fname, flags, mode, openinfo, dflags)
	const char *fname;
	int flags, mode, dflags;
	const BTREEINFO *openinfo;
{
	BTMETA m;
	BTREE *t;
	BTREEINFO b;
	DB *dbp;
	pgno_t ncache;
	struct stat sb;
	int machine_lorder, nr;
#ifdef WIN32
        int    ntoflags,ntaflags;
        BY_HANDLE_FILE_INFORMATION    FIBH;
#endif        
	t = NULL;

	/*
	 * Intention is to make sure all of the user's selections are okay
	 * here and then use them without checking.  Can't be complete, since
	 * we don't know the right page size, lorder or flags until the backing
	 * file is opened.  Also, the file's page size can cause the cachesize
	 * to change.
	 */
	machine_lorder = byteorder();

        if (openinfo) {
		b = *openinfo;

		/* Flags: R_DUP. */
		if (b.flags & ~(R_DUP))
			goto einval;

		/*
		 * Page size must be indx_t aligned and >= MINPSIZE.  Default
		 * page size is set farther on, based on the underlying file
		 * transfer size.
		 */
		if (b.psize &&
		    (b.psize < MINPSIZE || b.psize > MAX_PAGE_OFFSET + 1 ||
		    b.psize & sizeof(indx_t) - 1))
			goto einval;

		/* Minimum number of keys per page; absolute minimum is 2. */
		if (b.minkeypage) {
			if (b.minkeypage < 2)
				goto einval;
		} else
			b.minkeypage = DEFMINKEYPAGE;

		/* If no comparison, use default comparison and prefix. */
		if (b.compare == NULL) {
			b.compare = __bt_defcmp;
			if (b.prefix == NULL)
				b.prefix = __bt_defpfx;
		}

		if (b.lorder == 0)
			b.lorder = machine_lorder;
	} else {
		b.compare = __bt_defcmp;
		b.cachesize = 0;
		b.flags = 0;
		b.lorder = machine_lorder;
		b.minkeypage = DEFMINKEYPAGE;
		b.prefix = __bt_defpfx;
		b.psize = 0;
	}

	/* Check for the ubiquitous PDP-11. */
	if (b.lorder != BIG_ENDIAN && b.lorder != LITTLE_ENDIAN)
		goto einval;

	/* Allocate and initialize DB and BTREE structures. */
	if ((t = malloc(sizeof(BTREE))) == NULL)
		goto err;
	memset(t, 0, sizeof(BTREE));
	t->bt_bcursor.pgno = P_INVALID;
#ifndef WIN32
	t->bt_fd = -1;			/* Don't close unopened fd on error. */
#else
        t->bt_Hfd = INVALID_HANDLE_VALUE;/* Don't close unopened fd on error. */
#endif
	t->bt_lorder = b.lorder;
	t->bt_order = NOT;
	t->bt_cmp = b.compare;
	t->bt_pfx = b.prefix;
#ifndef WIN32
	t->bt_rfd = -1;
#else
        t->bt_Hrfd = INVALID_HANDLE_VALUE;
#endif

	if ((t->bt_dbp = dbp = malloc(sizeof(DB))) == NULL)
		goto err;
	t->bt_flags = 0;
	if (t->bt_lorder != machine_lorder)
		SET(t, B_NEEDSWAP);

	dbp->type = DB_BTREE;
	dbp->internal = t;
	dbp->close = __bt_close;
	dbp->del = __bt_delete;
	dbp->fd = __bt_fd;
	dbp->get = __bt_get;
	dbp->put = __bt_put;
	dbp->seq = __bt_seq;
	dbp->sync = __bt_sync;

	/*
	 * If no file name was supplied, this is an in-memory btree and we
	 * open a backing temporary file.  Otherwise, it's a disk-based tree.
	 */
	if (fname) {
		switch(flags & O_ACCMODE) {
		case O_RDONLY:
			SET(t, B_RDONLY);
			break;
		case O_RDWR:
			break;
		case O_WRONLY:
		default:
			goto einval;
		}
		
#ifndef WIN32
		if ((t->bt_fd = open(fname, flags, mode)) < 0)
			goto err;
#else
        ntoflags = OPEN_ALWAYS;
        ntaflags = 0;
        ntaflags |= (flags & O_WRONLY) ? GENERIC_WRITE : 0;
        ntaflags |= (flags & O_RDONLY) ? GENERIC_READ : 0;
        ntaflags |= (flags & O_RDWR) ? (GENERIC_WRITE|GENERIC_READ) : 0;
        ntoflags |= (flags & O_TRUNC) ? CREATE_NEW : 0;

        printf("bt_open: Createfile aflags %x oflags %x\n",
                ntaflags,ntoflags);

	if ((t->bt_Hfd =  CreateFile(fname,ntaflags,FILE_SHARE_READ | FILE_SHARE_WRITE,
          NULL, ntoflags, FILE_ATTRIBUTE_NORMAL,NULL)) ==
                    INVALID_HANDLE_VALUE) {
                printf("bt_open - failed CreateFile %d\n",GetLastError());
                printf("file name %s\nntaflags %x\nntoflags %x\n,",
                            fname,ntaflags,ntoflags);
			goto err;
                }
#endif

	} else {
		if ((flags & O_ACCMODE) != O_RDWR)
			goto einval;
#ifndef WIN32
		if ((t->bt_fd = tmp()) == -1)
			goto err;
#else


		if ((t->bt_Hfd = tmp()) == INVALID_HANDLE_VALUE)
			goto err;
#endif
		SET(t, B_INMEM);
	}

#ifndef WIN32
	if (fcntl(t->bt_fd, F_SETFD, 1) == -1)
		goto err;
	if (fstat(t->bt_fd, &sb))
		goto err;
#else


        /* have to delete file later somehow- fix this */
        /* also, don't accomodate a file larger than 2GB */
        GetFileInformationByHandle(t->bt_Hfd,&FIBH);
#endif

#ifndef WIN32
	if (sb.st_size) {
		nr = read(t->bt_fd, &m, sizeof(BTMETA));
		if (nr < 0)
			goto err;
		if (nr != sizeof(BTMETA))
			goto eftype;
#else
        printf("bt_open: FileSize %x\n",FIBH.nFileSizeLow);
        if (FIBH.nFileSizeHigh | FIBH.nFileSizeLow) {
         if (ReadFile(t->bt_Hfd, &m,sizeof(BTMETA), &nr,NULL) != TRUE) 
                    goto err;
    	 if (nr != sizeof(BTMETA))
			goto eftype;
#endif
		/*
		 * Read in the meta-data.  This can change the notion of what
		 * the lorder, page size and flags are, and, when the page size
		 * changes, the cachesize value can change too.  If the user
		 * specified the wrong byte order for an existing database, we
		 * don't bother to return an error, we just clear the NEEDSWAP
		 * bit.
		 */
		if (m.m_magic == BTREEMAGIC)
			CLR(t, B_NEEDSWAP);
		else {
			SET(t, B_NEEDSWAP);
			BLSWAP(m.m_magic);
			BLSWAP(m.m_version);
			BLSWAP(m.m_psize);
			BLSWAP(m.m_free);
			BLSWAP(m.m_nrecs);
			BLSWAP(m.m_flags);
		}
		if (m.m_magic != BTREEMAGIC || m.m_version != BTREEVERSION)
			goto eftype;
		if (m.m_psize < MINPSIZE || m.m_psize > MAX_PAGE_OFFSET + 1 ||
		    m.m_psize & sizeof(indx_t) - 1)
			goto eftype;
		if (m.m_flags & ~SAVEMETA)
			goto eftype;
		b.psize = m.m_psize;
		t->bt_flags |= m.m_flags;
		t->bt_free = m.m_free;
		t->bt_nrecs = m.m_nrecs;
	} else {
		/*
		 * Set the page size to the best value for I/O to this file.
		 * Don't overflow the page offset type.
		 */
		if (b.psize == 0) {
#ifndef WIN32
			b.psize = sb.st_blksize;
#else
			b.psize = 4096; /* adjust this later */
#endif
			if (b.psize < MINPSIZE)
				b.psize = MINPSIZE;
			if (b.psize > MAX_PAGE_OFFSET + 1)
				b.psize = MAX_PAGE_OFFSET + 1;
		}

		/* Set flag if duplicates permitted. */
		if (!(b.flags & R_DUP))
			SET(t, B_NODUPS);

		t->bt_free = P_INVALID;
		t->bt_nrecs = 0;
		SET(t, B_METADIRTY);
	}

	t->bt_psize = b.psize;

	/* Set the cache size; must be a multiple of the page size. */
	if (b.cachesize && b.cachesize & b.psize - 1)
		b.cachesize += (~b.cachesize & b.psize - 1) + 1;
	if (b.cachesize < b.psize * MINCACHE)
		b.cachesize = b.psize * MINCACHE;

	/* Calculate number of pages to cache. */
	ncache = (b.cachesize + t->bt_psize - 1) / t->bt_psize;

	/*
	 * The btree data structure requires that at least two keys can fit on
	 * a page, but other than that there's no fixed requirement.  The user
	 * specified a minimum number per page, and we translated that into the
	 * number of bytes a key/data pair can use before being placed on an
	 * overflow page.  This calculation includes the page header, the size
	 * of the index referencing the leaf item and the size of the leaf item
	 * structure.  Also, don't let the user specify a minkeypage such that
	 * a key/data pair won't fit even if both key and data are on overflow
	 * pages.
	 */
	t->bt_ovflsize = (t->bt_psize - BTDATAOFF) / b.minkeypage -
	    (sizeof(indx_t) + NBLEAFDBT(0, 0));
	if (t->bt_ovflsize < NBLEAFDBT(NOVFLSIZE, NOVFLSIZE) + sizeof(indx_t))
		t->bt_ovflsize =
		    NBLEAFDBT(NOVFLSIZE, NOVFLSIZE) + sizeof(indx_t);

	/* Initialize the buffer pool. */
#ifndef WIN32
	if ((t->bt_mp =
	    mpool_open(NULL, t->bt_fd, t->bt_psize, ncache)) == NULL)
		goto err;
#else

	if ((t->bt_mp =
	    mpool_open(NULL, t->bt_Hfd, t->bt_psize, ncache)) == NULL)
		goto err;
#endif
	if (!ISSET(t, B_INMEM))
		mpool_filter(t->bt_mp, __bt_pgin, __bt_pgout, t);

	/* Create a root page if new tree. */
	if (nroot(t) == RET_ERROR)
		goto err;


	/* Global flags. */
	if (dflags & DB_LOCK)
		SET(t, B_DB_LOCK);
	if (dflags & DB_SHMEM)
		SET(t, B_DB_SHMEM);
	if (dflags & DB_TXN)
		SET(t, B_DB_TXN);

	return (dbp);

einval:	errno = EINVAL;
	goto err;

eftype:	errno = EFTYPE;
	goto err;

err:	if (t) {
		if (t->bt_dbp)
			free(t->bt_dbp);
#ifndef WIN32
		if (t->bt_fd != -1)
			(void)close(t->bt_fd);
#else
		if (t->bt_Hfd != INVALID_HANDLE_VALUE)
			(void)CloseHandle(t->bt_Hfd);
#endif
		free(t);
	}
	return (NULL);
}

/*
 * NROOT -- Create the root of a new tree.
 *
 * Parameters:
 *	t:	tree
 *
 * Returns:
 *	RET_ERROR, RET_SUCCESS
 */
static int
nroot(t)
	BTREE *t;
{
	PAGE *meta, *root;
	pgno_t npg;

	if ((meta = mpool_get(t->bt_mp, 0, 0)) != NULL) {
		mpool_put(t->bt_mp, meta, 0);
		return (RET_SUCCESS);
	}

	if (errno != EINVAL)
		return (RET_ERROR);

	if ((meta = mpool_new(t->bt_mp, &npg)) == NULL)
		return (RET_ERROR);


	if ((root = mpool_new(t->bt_mp, &npg)) == NULL)
		return (RET_ERROR);


	if (npg != P_ROOT)
		return (RET_ERROR);
	root->pgno = npg;
	root->prevpg = root->nextpg = P_INVALID;
	root->lower = BTDATAOFF;
	root->upper = t->bt_psize;
	root->flags = P_BLEAF;
	memset(meta, 0, t->bt_psize);
	mpool_put(t->bt_mp, meta, MPOOL_DIRTY);
	mpool_put(t->bt_mp, root, MPOOL_DIRTY);
	return (RET_SUCCESS);
}

#ifndef WIN32
static int
#else
HANDLE
#endif
tmp()
{
#ifndef WIN32
	sigset_t set, oset;
	int fd;
	char *envtmp;
	char path[MAXPATHLEN];

	envtmp = getenv("TMPDIR");
	(void)snprintf(path,
	    sizeof(path), "%s/bt.XXXXXX", envtmp ? envtmp : "/tmp");

	(void)sigfillset(&set);
	(void)sigprocmask(SIG_BLOCK, &set, &oset);
	if ((fd = mkstemp(path)) != -1)
		(void)unlink(path);
	(void)sigprocmask(SIG_SETMASK, &oset, NULL);
	return(fd);
#else
        char namestr[MAX_PATH];
        char pathstr[MAX_PATH];
        HANDLE Hfd;
    
        GetTempPath(MAX_PATH,(LPCTSTR)&pathstr);
        GetTempFileName((LPCTSTR)&pathstr,"bt",0,(LPCTSTR)&namestr);
        /* The file should be created. We just need to open
            and use it */
        Hfd = CreateFile((LPCTSTR)&namestr,GENERIC_READ|GENERIC_WRITE,
                    0,NULL,TRUNCATE_EXISTING,FILE_FLAG_DELETE_ON_CLOSE,NULL);
        if (Hfd  == INVALID_HANDLE_VALUE) {
            printf("Tmp: failed with %s %d\n",&namestr,GetLastError());
            }
	return (Hfd);
#endif

}

static int
byteorder()
{
	u_long x;			/* XXX: 32-bit assumption. */
	u_char *p;

	x = 0x01020304;
	p = (u_char *)&x;
	switch (*p) {
	case 1:
		return (BIG_ENDIAN);
	case 4:
		return (LITTLE_ENDIAN);
	default:
		return (0);
	}
}

#ifndef WIN32
int
#else
HANDLE
#endif
__bt_fd(dbp)
        const DB *dbp;
{
	BTREE *t;

	t = dbp->internal;

	/* Toss any page pinned across calls. */
	if (t->bt_pinned != NULL) {
		mpool_put(t->bt_mp, t->bt_pinned, 0);
		t->bt_pinned = NULL;
	}

	/* In-memory database can't have a file descriptor. */
	if (ISSET(t, B_INMEM)) {
		errno = ENOENT;
		return (-1);
	}
#ifndef WIN32
	return (t->bt_fd);
#else
	return (t->bt_Hfd);
#endif
}
