##########################################################################
#
# Copyright(c) 2011-2014 Broadcom Corporation, all rights reserved
# Copyright(c) 2014 QLogic Corporation, all rights reserved
# Proprietary and Confidential Information.
#
# This source file is the property of QLogic Corporation, and
# may not be copied or distributed in any isomorphic form without 
# the prior written consent of QLogic Corporation. 
#
# Name:        windiagtest.tcl
#
# Description: 
#    This is a regression test for the EDIAG tool.
#    Tests the functionallity of all the EDIAG commands.#
#  
###########################################################################

# Specify any dependency on other packages
package require Tcl

namespace eval _windiagtest {
        namespace export windiagtest
}

#source "l2test.tcl"


# Give a short-name for the regression tests for XDIAG.
interp alias {} windiagtest {} "_windiagtest::main_test"
interp alias {} bartest {}     "_windiagtest::bar_test"
interp alias {} bitstest {}    "_windiagtest::bits_test"
interp alias {} devtest {}     "_windiagtest::device_test"
interp alias {} delaytest {}   "_windiagtest::delay_test"
interp alias {} debugtest {}   "_windiagtest::debug_test"
interp alias {} drivertest {}  "_windiagtest::driver_test"
interp alias {} gpiotest {}    "_windiagtest::gpio_test"
interp alias {} hmemtest {}    "_windiagtest::hmem_test"
interp alias {} helptest {}    "_windiagtest::help_test"
#interp alias {} l2test {}      "_windiagtest::l2test_test"
interp alias {} ledtest {}     "_windiagtest::led_test"
interp alias {} logtest {}     "_windiagtest::log_test"
interp alias {} lmdevtest {}   "_windiagtest::lmdev_test"
interp alias {} nvmtest {}     "_windiagtest::nvm_test"
interp alias {} regtest {}     "_windiagtest::reg_test"
interp alias {} pcitest {}     "_windiagtest::pci_test"
interp alias {} pcicfgtest {}  "_windiagtest::pcicfg_test"
interp alias {} sbtest {}      "_windiagtest::sb_test"
interp alias {} spiotest {}    "_windiagtest::spio_test"
interp alias {} vertest {}     "_windiagtest::version_test"

# -----------------------------------------------------------------
# The main XDIAG regression test
# -----------------------------------------------------------------
proc _windiagtest::main_test {} {

    set result 0
    
    log open diag.log

    # set to batch mode to avoid prints
    set ::sys(BATCH) 1

    
    # test the "bar" command    
    #if {[bartest] != 0} {        
    #    set result 1
    #}
    
    # test the "bits" command
    if {[bitstest] != 0} {         
        set result 1
    }
    
    # test the "device" command - not ready yet !!
    if {[devtest] != 0} {
        set result 1
    }

    # test the "delay" command
    #if {[delaytest] != 0} {
    #    set result 1
    #}
    
    # test the "driver" command
    #if {[drivertest] != 0} {
    #    set result 1
    #}
    
    # test the "debug" command
    if {[debugtest] != 0} {
        set result 1
    }

    # Command not implemented yet in WinDiag
    # test the "gpio" command
    #if {[gpiotest] != 0} { 
    #    set result 1
    #}
    
    # test the "add_help" and "help" command
    if {[helptest] != 0} { 
        set result 1
    }
    
    # test the "hmem" command
    #if {[hmemtest] != 0} { 
    #    set result 1
    #}

    # Command not implemented yet in WinDiag
    # test the "led" command
    #if {[ledtest] != 0} {
    #    set result 1
    #}
    
    # Command not implemented yet in WinDiag
    # test the "log" command
    #if {[logtest] != 0} {
    #    set result 1
    #}     
    
    # Command not implemented yet in WinDiag
    # test the "lmdev" command
    #if {[lmdevtest] != 0} {
    #    set result 1
    #}    
    
    # Command not implemented yet in WinDiag
    # L2pkt will be tested by the short_regress
    # Use the l2test.tcl script to test the l2pkt commands
    #if {[l2test] != 0} {
    #    set result 1
    #}
    
    # test the "nvm" command
    if {[nvmtest] != 0} {
        set result 1
    }
     
    # test the "pci" command
    #if {[pcitest] != 0} {
    #    set result 1
    #}

    # test the "pcicfg" command
    #if {[pcicfgtest] != 0} {
    #    set result 1
    #}

    # test the "reg" command
    #if {[regtest] != 0} {
    #    set result 1
    #}
    
    # Command not implemented yet in WinDiag
    # test the "sb" command
    #if {[sbtest] != 0} {
    #    set result 1
    #}
    
    # Command not implemented yet in WinDiag
    # test the "spio" command
    #if {[spiotest] != 0} {
    #    set result 1
    #}
    
    # test the "version" command
    if {[vertest] != 0} {
        set result 1
    }

    # remove batch mode
    set ::sys(BATCH) 0

    
    if {$result !=  0} {
        puts "***********************************************************"
        puts "*******************    WinDiag test failed  !!!!!!!!!!!!!!!!!!"
        puts "***********************************************************"               
    } else {
        puts "***********************************************************"
        puts "*******************    WinDiag test passed  ******************"
        puts "***********************************************************"
    }
    log close
}

# -----------------------------------------------------------------
# The "bar" command test
# -----------------------------------------------------------------
proc _windiagtest::bar_test {} {
    
    puts "Starting BAR test ..."
    
    # Check the Vendor ID at address 0x2000    
    if {[bar 0 read 0x2000 4] != 0x164E14E4} {
        puts "***********************"
        puts "Testing BAR failed !!!"
        puts "***********************"
        return -1
    }
    
    puts "Testing BAR passed"
    return 0
}


# -----------------------------------------------------------------
# The "bit" command test 
# -----------------------------------------------------------------
proc _windiagtest::bits_test {} {
    
    puts "Starting BITS test ..."
    
    # Verify that the "bits clear" command works o.k.
    set value [bits clear 0 31]
    if {$value != 0x7ffffffe} {
        puts "************************"
        puts "Testing BITS failed  !!!"
        puts "***********************"
        return -1
    } 
    
    # Verify that the "bits set" command works o.k.
    set value [bits set 2 31]
    if {$value != 0x80000004} {
        puts "************************"
        puts "Testing BITS faield  !!!"
        puts "************************"
        return -1
    } else  {
        puts "Testing BITS passed"
        return 0    
    }    
}

# -----------------------------------------------------------------
# The "device" command test
# -----------------------------------------------------------------
proc _windiagtest::device_test {} {  
    
    puts "Starting DEVICE test ..."
    
    # Verify that the "device" command works o.k.
    set devices [device -l]
    set device0 [lindex $devices 0]
    set device1 [lindex $devices 1]

    # check that return value includes the BROADCOM_ID 0x14e4 =5348
    # and the DEVICE ID 0x164e =5710
    if {![string match {5348 5710} $device0]} {
        puts "*************************"
        puts "Testing DEVICE failed !!!"       
        puts "*************************"
        return -1
    }
    if {![string match {5348 5710} $device1]} {
        puts "*************************"
        puts "Testing DEVICE failed !!!"
        puts "*************************"
        return -1
    }
    puts "Testing DEVICE passed"
    return 0    
}

# -----------------------------------------------------------------
# The "delay" command test
# -----------------------------------------------------------------
proc _windiagtest::delay_test {} {
    
     puts "Starting DELAY test ..."
    
    # Verify that the "delay" command works o.k.
    # use the "time" command 10 times to get a more accurate result
    set value [time {delay ms 10} 10]
    if {($value <= 90000) && ($value >= 110000)} {
        puts "************************"
        puts "Testing DELAY failed !!!"
        puts "************************"
        return -1
    }
    
    set value [time {delay us 100} 10]
    if {($value <= 90) && ($value >= 110)} {
        puts "************************"
        puts "Testing DELAY failed !!!"
        puts "************************"
        return -1
    }
    puts "Testing DELAY passed"
    return 0
}

# -----------------------------------------------------------------
# The "driver" command test
# -----------------------------------------------------------------
proc _windiagtest::driver_test {} {
    
    puts "Starting DRIVER test ..."
    
    # Verify that the "driver" command works o.k.        
    
    set str [driver link_state]
    if {[llength $str] != 4} {
        puts "**********************************************"
        puts "Testing DRIVER (option: link_state) failed !!!"
        puts "**********************************************"
        return -1
    }    
    
    driver strip_crc 0
    driver strip_crc 1
    
    driver unload   
    if {$::current(DRV_STATE) != "SETUP"} {
        puts "******************************************"
        puts "Testing DRIVER (option: unload) failed !!!"
        puts "******************************************"
        return -1
    }
    
    puts "Testing DRIVER passed"
    return 0
}

# -----------------------------------------------------------------
# The "debug" command test
# -----------------------------------------------------------------
proc _windiagtest::debug_test {} {
    
    puts "Starting DEBUG test"
    
    # Verify that the "debug" command works o.k.
    debug set level FATAL   
    debug set path L4_SP
    puts "Testing DEBUG passed"
    return 0
}

# -----------------------------------------------------------------
# The "gpio" command test
# -----------------------------------------------------------------
proc _windiagtest::gpio_test {} {
    
    puts "Starting GPIO test"
    
    # Verify that the "GPIO" command works o.k.
    gpio write 0 { 0 1 0 1}
    set result [gpio read]
    if {[string match "0: 0 1 0 1" $result]} { 
        puts "Testing GPIO passed"
        return 0
    } else {
        puts "******************************************"
        puts "Testing GPIO failed !!!"
        puts "******************************************"
        return -1  
    }
}

# -----------------------------------------------------------------
# The "help" command test
# -----------------------------------------------------------------
proc _windiagtest::help_test {} {
    
    puts "Starting HELP test"
    
    # Verify that the "help" and "add_help" commands works o.k.
    set help_info     "This is a dummy proc\n"    
    add_help dummy_proc $help_info
    
    set result [help dummy_proc]
    if {![string match "This is a dummy proc*" $result]} {
        puts "******************************************"
        puts "Testing HELP failed !!!"
        puts "******************************************"
        return -1
    }
    
    puts "Testing HELP passed"
    return 0
}

# -----------------------------------------------------------------
# The "hmem" command test
# -----------------------------------------------------------------
proc _windiagtest::hmem_test {} {
    
    puts "Starting HMEM test"
    set memSize 100
    
    # Verify that the "hmem" command works o.k.
    # Allocate $memSize bytes of virtual memory
    set addr [hmem alloc $memSize MyMemory]
    
    # Fill the memory with the "0x12121212" pattern
    hmem fill $addr $memSize 0x12121212
    
    # Verify using the "hmem inuse" command that the memory was allocated
    set str [hmem inuse]
    if { ![ string match "*$memSize bytes(Virt) MyMemory*" $str ]} {
        puts "***********************"
        puts "Testing HMEM failed !!!"
        puts "Hmem alloc failed !! (Expected 100 bytes(Virt) got $str)"
        puts "***********************"
        return -1
    }
    
    # Verify that the memory was filled with the "0x12121212" pattern
    for {set i 0} {$i < $memSize} {incr i 4} {
        set addr [add_to_addr $addr $i]
        set readValue [hmem read $addr]
        if {$readValue != "0x12121212"} {
            puts "***********************"
            puts "Testing HMEM failed !!!"
            puts "Hmem fill failed !! (Expected 0x12121212 got $readValue)"
            puts "***********************"
            return -1
        }
    }  
    
    # Check the "hmem write"
    hmem write $addr 0x34343434
    set readValue [hmem read $addr]
    if {$readValue != "0x34343434"} {
            puts "***********************"
            puts "Testing HMEM failed !!!"
            puts "Hmem write failed !! (Expected 0x34343434 got $readValue)"
            puts "***********************"
            return -1
    }
    hmem free $addr
    
    # Verify that the "hmem palloc" command works o.k.
    # Allocate $memSize of bytes of physical memory
    set addr [hmem palloc $memSize MyBuf]
    
    # Fill the memory with the "0x55AA55AA" pattern
    hmem fill $addr $memSize 0x55AA55AA
    
    # Verify using the "hmem inuse" command that the memory was allocated
    set str [hmem inuse]
    if { ![ string match "*$memSize bytes(Phys) MyBuf*" $str ]} {
        puts "***********************"
        puts "Testing HMEM failed !!!"
        puts "Hmem palloc failed !! (Expected $memSize bytes(Phys) got $str"
        puts "***********************"
        return -1
    }
    
    # Verify that the memory was filled with the "0x55AA55AA" pattern
    for {set i 0} {$i < $memSize} {incr i 4} {
        set addr [add_to_addr $addr $i]
        set readValue [hmem read $addr]
        if {$readValue != "0x55AA55AA"} {
                puts "***********************"
                puts "Testing HMEM failed !!!"
                puts "Hmem fill failed !! expected 0x55AA55AA got $readValue"
                puts "***********************"
                return -1
        }
    }
    hmem free $addr
    puts "Testing HMEM passed"
    return 0
}

# -----------------------------------------------------------------
# The "log" command test
# -----------------------------------------------------------------
proc _windiagtest::log_test {} {
    
    puts "Starting LOG test ..."    
    
    log close
    # Clear the "tmp.txt" file
    set outfile [open "tmp.txt" w]
    close $outfile

    # Verify that the "log" command works o.k.
    # Write a testing string to the log
    log open tmp.txt
    puts "This is a testing string"
    log close
    
    log open diag.log
    # Open the file using the "open" command
    # and verify that the testing string exists
    set outfile [open "tmp.txt" r]
    while {[eof $outfile] != 1} {
            set str [gets $outfile]    
            if {![string match "This is a testing string" $str]} {
                puts "Testing LOG passed"
                return 0
            }
    }

    # if we got here, we did not find the testing string
    puts "***********************"
    puts "Testing LOG failed !!!"
    puts "***********************"
    return -1
}

# -----------------------------------------------------------------
# The "led" command test
# -----------------------------------------------------------------
proc _windiagtest::led_test {} {
    
    puts "Starting LED test ..."
    
    driver load    
    led override 0 10M 1
    if {![led status 0 10M]} {
        puts "***********************"
        puts "Testing LOG failed !!!"
        puts "***********************"
        return -1  
    }
    
    led override 0 2500M 1
    if {![led status 0 2500M]} {
        puts "***********************"
        puts "Testing LOG failed !!!"
        puts "***********************"
        return -1  
    }
    driver unload
    
    puts "Testing LED passed"
    return 0
}

# -----------------------------------------------------------------
# The "lmdev" command test
# -----------------------------------------------------------------
proc _windiagtest::lmdev_test {} {
    
    puts "Starting LMDEV test ..."    
     
    lmdev show 0
    lmdev set_drop 0
    lmdev vlan_remove 0
    
    # Verify the driver load works after the lmdev changes
    driver load    
    driver unload
    
    puts "Testing LMDEV passed"
    return 0
}

# -----------------------------------------------------------------
# The "l2spec" "l2pkt" commands test 
# -----------------------------------------------------------------
proc _windiagtest::l2test_test {{type "basic"}} {
   
    # Set up the batch mode
    set ::sys(BATCH)  1
    
    puts "Starting L2SPEC, L2PKT test ..."
    driver load
    
    # test1: test l2 raw packets 
    set returnValue [l2test_raw]
    if {$returnValue != 0} {
        puts "**********************************************"
        puts "Testing L2SPEC, L2PKT (raw packets) failed !!!"
        puts "**********************************************"
        return -1
    }
    
    # test2: test tcp packets 
    set returnValue [l2test_tcp]
    if {$returnValue != 0} {
        puts "**********************************************"
        puts "Testing L2SPEC, L2PKT (tcp packets) failed !!!"
        puts "**********************************************"
        return -1
    }

    # test3: use the l2pkt checker
    set returnValue [use_checker]
    if {$returnValue != 0} {
        puts "******************************************"
        puts "Testing L2SPEC, L2PKT (checker) failed !!!"
        puts "******************************************"
        return -1
    }
    
    driver unload
    puts "Testing L2SPEC, L2PKT passed"
    
    # Go back to the interactive mode
    set ::sys(BATCH)  0     
    return 0
}

# -----------------------------------------------------------------
# The "nvm" command test
# -----------------------------------------------------------------
proc _windiagtest::nvm_test {} {
        
    puts "Starting NVM test ..."    
   
    nvm cfg 1=00:10:18:11:22:33
    nvm write 0x10100 0x5555
    set result [nvm read 0x10100]
    
    # check that return value includes the BROADCOM DEVICE ID
    if {$result != 0x5555} {
        puts "***********************"
        puts "Testing NVM failed !!! "        
        puts "***********************"
        return -1
    }
    
    nvm write 0x10100 0xAAAA
    set result [nvm read 0x10100]
    
    # check that return value includes the BROADCOM DEVICE ID
    if {$result != 0xAAAA} {
        puts "***********************"
        puts "Testing NVM failed !!! "        
        puts "***********************"
        return -1
    }
    puts "Testing NVM passed"
    return 0    
}
    
# -----------------------------------------------------------------
# The "pci" command test
# -----------------------------------------------------------------
proc _windiagtest::pci_test {} {
        
    puts "Starting PCI test ..."
    
    set DEVICE_ID    5710

    # Verify that "pci search" with -did device_id 5710 works o.k.
    set retValue [pci search -did $DEVICE_ID]
    
    # check that return value includes the BROADCOM DEVICE ID
    if {$retValue == ""} {
        puts "***********************"
        puts "Testing PCI failed !!!"        
        puts "***********************"
        return -1
    }
    puts "Testing PCI passed"
    return 0    
}

# -----------------------------------------------------------------
# The "pcicfg" command test
# -----------------------------------------------------------------
proc _windiagtest::pcicfg_test {} {
    
    set DEVICE_ID    "164E"
    set VENDOR_ID    "14E4" 
    
    puts "Starting PCICFG test ..."
    
    # Verify that the "pcicfg" command.
    #for now test only the pcicfg read (as pcicfg write to addr 0x78 does'nt work yet)
    set retValue [pcicfg read 0]   

    # check that return value includes the BROADCOM DEVICE ID
    if {$retValue != "0x$DEVICE_ID$VENDOR_ID"} {
        puts "***********************"
        puts "Testing PCICFG failed !!!"
        puts "***********************"
        return -1
    }
    puts "Testing PCICFG passed"
    return 0    
}

# -----------------------------------------------------------------
# The "reg" command test
# -----------------------------------------------------------------
proc _windiagtest::reg_test {} {
    
    puts "Starting REG test ..."
    
    # Remove chip from reset    
    reg write 0xa580 0xFFFFFFFF    
    reg write 0xa590 0xFFFF
    
    # Verify that the "REG" command works o.k. using some sample registers         
        
    # Write to PRS_REG_CM_HDR_TYPE_0 0x102400
    set addr 0x040078
    if {[test_register $addr] != 0} {
        puts "***********************"
        puts "Testing REG failed !!!"
        puts "***********************"
        return -1
    }   

    puts "Testing REG passed"
    return 0

}

# -----------------------------------------------------------------
# The "sb" command test
# -----------------------------------------------------------------
proc _windiagtest::sb_test {} {
       
    puts "Starting SB test ..."
     
    log close
    
    # Verify that the "sb" command works o.k.
    log open sb.txt  
    sb cons_id 0 u  
    sb status_index 0 u
    sb block_id 0 c
    sb show 0
    sb default
    log close
    
    log open diag.log

    # Open the log file using the "open" command
    # and verify that there is no "failed" string
    set outfile [open "sb.txt" r]
    while {[eof $outfile] != 1} {
            set str [gets $outfile]    
            if {[string match "failed" $str] != 0} {
                puts "***********************"
                puts "Testing SB failed !!!"
                puts "***********************"
                return -1
            }
    }

    # if we got here, we did not find the "failed" string
    puts "Testing SB passed"
    return 0
}

# -----------------------------------------------------------------
# The "spio" command test
# -----------------------------------------------------------------
proc _windiagtest::spio_test {} {
    
    puts "Starting SPIO test ..."    
    
    spio write 0 {1 0 1 0 1 0 1 0}    
    set result [spio read]
    if {![string match "*0: 1 0 1 0 1 0 1 0*" $result]} {
        puts "**************************"
        puts "Testing SPIO failed !!!"
        puts "**************************"
        return -1
    }   

    puts "Testing SPIO passed"
    return 0

}

# -----------------------------------------------------------------
# The "version" command test
# -----------------------------------------------------------------
proc _windiagtest::version_test {} {
    
    puts "Starting VERSION test ..."    
    
    set result [version]
    if {![string match "Ediag version *.*.*" $result]} {
        puts "**************************"
        puts "Testing VERSION failed !!!"
        puts "**************************"
        return -1
    }   

    puts "Testing VERSION passed"
    return 0

}

# ---------------------- Internal procedures -----------------------------

proc test_register {addr} {
    
    set value 0x55AA55AA
    set orgValue [reg dread $addr]
    
    reg dwrite $addr $value
    set readValue [reg dread $addr]
    if {$readValue != $value} {    
        puts "reg dread from address $addr failed !! (expected $value got $readValue)"
        return -1
    }

    reg dwrite $addr $orgValue
    return 0
}

# ------------------------------------------------------------------------# 
# This procedure performs 2 sub-tests of packets with
# incremental payload and encapsulation 802.3raw (l3type none).
# ------------------------------------------------------------------------
proc l2test_raw {} {          
      
    # Create a default l2pkt specification packet
    set specPkt [l2spec gen]
    l2spec update $specPkt [list maxBDs 3 minpktlen 64 maxpktlen 9000]
    l2spec update $specPkt [list pktlenmode increment:1 crcgen hardware]
    l2spec update $specPkt [list pktcnt 10 encapmode 802.3raw l3type none]
    
    # Set the parameters of the test sub-cases 
    set minLenList  [list  250 1500]
    set subCaseList [list "A" "B"]

    # Perform the sub-cases tests
    for {set idx 0} {$idx < [llength $subCaseList]} {incr idx} {
         set minLen [lindex $minLenList $idx]
                                       
         # Update the minpktlen parameter  
         l2spec update $specPkt [list minpktlen $minLen]
            
         # Generate the pktsList according to the specification packet specPkt.   
         l2pkt gen $specPkt pktsList            
            
         # Check the packets generated   
         set result [check_raw $pktsList $minLen]
         if { $result != 0 } {
                puts "Raw packets tests failed (check_raw)!!!"
                return -1
            }
            
         # Send the pktList
         l2pkt send $pktsList
    }
    #puts "Raw packets tests ended successfully"
    return 0    
}

# -------------------------------------------------------------------
# This procedure verifies that the l2 packets generated are according
# to parameters set
# -------------------------------------------------------------------
proc check_raw {{pktsList} {minLen}} {
    global pktArray
        
    # Check the context of the first packet
    set pkt [lindex $pktsList 0]
    array set pktArray [l2pkt dump $pkt]
    
    if {$pktArray(macHdr) != "802.3 \"Raw\""} {
        puts "ERROR: Got a wrong macHdr ($pktArray(macHdr) instead of Raw)!!!\n"
        return -1
    }
    if {$pktArray(macDA) != "ff-ff-ff-ff-ff-ff"} {
        puts "ERROR: Got a wrong macDA ($pktArray(macDA) instead of ff-ff-ff-ff-ff-ff) !!!\n"
        return -1
    }
    if {$pktArray(macSA) != "00-10-18-00-00-00"} {
        puts "ERROR: Got a wrong macSA ($pktArray(macSA) instead of 00-10-18-00-00-00) !!!\n"
        return -1
    }
    if {$pktArray(pktType) != "tx"} {
        puts "ERROR: Got a wrong pktType ($pktArray(pktType) instead tx) !!!\n"
        return -1
    }
    if {$pktArray(payload002) != "0x06 0x07 0x08 0x09 0x0a 0x0b 0x0c 0x0d 0x0e 0x0f "} {
        puts "ERROR: Got a wrong payload002, not incremental $pktArray(payload002) !!!\n"
        return -1
    }   
            
    return 0
}

# ------------------------------------------------------------------------
# This procedure performs 2 sub-tests of packets with:
# fixed payload, l3type ip, iptype tcp, src IP addr 1.1.16.16, dst IP addr 1.1.16.17
# tcp dst port 64 tcp source port 96.
# -----------------------------------------------------------------------
proc l2test_tcp {} {
    
        
    # Create a default l2pkt specification packet
    set specPkt [l2spec gen]
    l2spec update $specPkt [list maxBDs 3 minpktlen 64 maxpktlen 9000]
    l2spec update $specPkt [list payloadpttn 55aa crcgen hardware]
    l2spec update $specPkt [list tcpchksumgen hardware]
    l2spec update $specPkt [list pktcnt 10 encapmode ethernetII]
    l2spec update $specPkt [list l3type ip iptype tcp]
    l2spec update $specPkt [list ipDA "1.1.16.16" ipSA "1.1.16.17"]
    l2spec update $specPkt [list tcpDP 64 tcpSP 96]
    
    
    # Set the parameters of the test sub-cases
    set minLenList  [list  250 1500]
    set subCaseList [list "A" "B" ]
    
    # Perform the sub-cases tests
    for {set idx 0} {$idx < [llength $subCaseList]} {incr idx} {
        set minLen [lindex $minLenList $idx]
        
        # Update the minpktlen parameter
        l2spec update $specPkt [list minpktlen $minLen]
        
        # Generate the pktsList according to the specification packet specPkt.
        l2pkt gen $specPkt pktsList
        
        # Check the packets generated
        set result [check_tcp $pktsList $minLen]
        if { $result != 0 } {
            puts "TCP packets tests failed!!! (check tcp)"
            return -1
        }
        
        # Send the packets 
        l2pkt send $pktsList
    }    
    return 0
}

# -------------------------------------------------------------------
# This procedure verifies that the l2 packets generated are according
# to parameters set
# -------------------------------------------------------------------
proc check_tcp {{pktsList} {minLen}} {
    global pktArray
    
    # Check the context of the first packet
    set pkt [lindex $pktsList 0]
    array set pktArray [l2pkt dump $pkt]
    
    if {$pktArray(macHdr) != "ETHERNET II"} {
        puts "ERROR: Got a wrong macHdr ($pktArray(macHdr) instead of ETHERNETII) !!!\n"
        return -1
    }
    if {$pktArray(l3Type) != "0x0800  \[IP\]"} {
        puts "ERROR: Got a wrong pktType ($pktArray(l3Type) instead of 0x0800  \[IP\]) !!!\n"
        return -1
    }
    if {$pktArray(ipSA) != "1.1.16.17"} {
        puts "ERROR: Got a wrong ipSA ($pktArray(ipSA) instead of 1.1.16.16) !!!\n"
        return -1
    }
    if {$pktArray(ipDA) != "1.1.16.16"} {
        puts "ERROR: Got a wrong ipDA ($pktArray(ipDA) instead of 1.1.16.17) !!!\n"
        return -1
    }
    if {$pktArray(tcpSP) != "0x0060"} {
        puts "ERROR: Got a wrong tcpSP ($pktArray(tcpSP) instead of 64)!!!\n"
        return -1
    }
    if {$pktArray(tcpDP) != "0x0040"} {
        puts "ERROR: Got a wrong tcpDP ($pktArray(tcpDP) instead of 96) !!!\n"
        return -1
    }
    if {$pktArray(payload006) != "0x55 0xaa 0x55 0xaa 0x55 0xaa 0x55 0xaa 0x55 0xaa "} {
        puts "ERROR: Got a wrong payload006, not fixed 0x55AA $pktArray(payload006) !!!\n"
        return -1
    } 
        
    return 0
}

# -------------------------------------------------------------------
# _l2test::use_checker
# This procedure generates a TCP packet specification.
# Then checks two dummy received packets against the specification packet:
# One the same as the specification, thus should pass the "l2pkt check".
# Second with wrong payload, thus should fail the "l2pkt check"
# ------------------------------------------------------------------
proc use_checker {} {            
          
    # Generate a default specification packet.
    set specPkt [l2spec gen]
        
    # change the CRC to be specific and not generated, so that l2pkt check
    # will not fail cause of CRC - does not work ???????
    l2spec update $specPkt [list crcgen specified crcval 0xdeadbeef]
            
    # Generate a dummy packet with wrong payload
    # (fixed instead of incremental) and with a wrong macDA.
    set dummySpecPkt [l2spec gen]
    l2spec update $dummySpecPkt [list payloadPttn 55aa]
    l2spec update $dummySpecPkt [list macDA "AA:AA:AA:AA:AA:AA"]
    l2pkt gen $dummySpecPkt pktList
    
    # check the dummy packet against the packet specification
    # It should fail ! (as the payload and macDA is wrong)
    set pkt [lindex $pktList 0]
    set result [l2pkt check $specPkt $pkt]
    
    # Verify that the l2pkt check found the differnece in macDA and payload       
    if {([string match "*macDA*" $result]) && ([string match "*payload*" $result])} {        
        return 0
    } else  {
        puts "ERROR: l2pkt check failed to find the difference !!! \n"
        return -1          
    }
}


