/****************************************************************************
 * Copyright(c) 2000 Broadcom Corporation, all rights reserved
 * Proprietary and Confidential Information.
 *
 * This source file is the property of Broadcom Corporation, and
 * may not be copied or distributed in any isomorphic form without 
 * the prior written consent of Broadcom Corporation. 
 *
 * Name:        pal.h
 *
 * Description: Platform Abstraction Layer implementation for bxbdrv(mpd)
 *
 * Author:      John Chen
 *
 ****************************************************************************/
#ifndef __pal_h__
#define __pal_h__

//
// define _MSC_VER here to get rid of the warning of undefined _MSC_VER
// when built on SuSE linux 10(kernel 2.6.16.14-6)
//
#define _MSC_VER 0

#include "bcmtype.h"

//
// macro to force unaligned data structure
//
#define __packed1__ __attribute__ ((packed, aligned(1)))

#define REGPARM(a) __attribute__((regparm (a)))
//#define REGPARM(a)
//
// Platform Abstracted Data Types
//
typedef U8* PAL_SPINLOCK;        // ptr to spinlock_t
typedef U8* PAL_ATOMIC;          // ptr to atomic_t
typedef U8* PAL_DEVICE;          // ptr to struct device
typedef U8* PAL_DEVICE_STAT;     // ptr to struct net_device_stats
typedef U8* PAL_PACKET_LIST;     // ptr to sk_buff_head
typedef U8* PAL_EVENT;           // ptr to struct wait_queue*
typedef U8* PAL_MUTEX;           // ptr to struct semaphore
typedef U8* PAL_TASKLET;         // ptr to struct tasklet_struct
typedef U8* PAL_TIMER;           // ptr to struct timer_list

//
typedef U8* PAL_PACKET;          // ptr to sk_buff*
typedef U8* PAL_HANDLE;          // opaque handle
typedef U8* PAL_PROC_DIR_ENTRY;  // ptr to struct proc_dir_entry

//
// Resource type definition for Platform Abstracted Data Types
//
typedef enum
{
    PRT_SPINLOCK = 0,
    PRT_ATOMIC = 1,
    PRT_TIMER = 2,
    PRT_DEVICE_STAT = 3,
    PRT_TASKLET = 4,
    PRT_MUTEX = 5,
    PRT_EVENT = 6,
    PRT_PACKET_LIST = 7,
    PRT_MAXIMUM = 8
} PAL_RESOURCE_TYPE;

// tx offload feature definitions
#define BLF_SG       0x01
#define BLF_CSUM     0x02
#define BLF_HDMA     0x20
#define BLF_TSO      0x800

/*****************************************************************************************
 *  Teton statistics_block definition(it is same define as statistics_block in 5706_reg.h)
 *****************************************************************************************/
typedef struct teton_stat_block
{
    u32_t stat_IfHCInOctets_hi;
    u32_t stat_IfHCInOctets_lo;                             
    u32_t stat_IfHCInBadOctets_hi;
    u32_t stat_IfHCInBadOctets_lo;                          
    u32_t stat_IfHCOutOctets_hi;
    u32_t stat_IfHCOutOctets_lo;                            
    u32_t stat_IfHCOutBadOctets_hi;
    u32_t stat_IfHCOutBadOctets_lo;                         
    u32_t stat_IfHCInUcastPkts_hi;
    u32_t stat_IfHCInUcastPkts_lo;                          
    u32_t stat_IfHCInMulticastPkts_hi;
    u32_t stat_IfHCInMulticastPkts_lo;                      
    u32_t stat_IfHCInBroadcastPkts_hi;
    u32_t stat_IfHCInBroadcastPkts_lo;                      
    u32_t stat_IfHCOutUcastPkts_hi;
    u32_t stat_IfHCOutUcastPkts_lo;                         
    u32_t stat_IfHCOutMulticastPkts_hi;
    u32_t stat_IfHCOutMulticastPkts_lo;                     
    u32_t stat_IfHCOutBroadcastPkts_hi;
    u32_t stat_IfHCOutBroadcastPkts_lo;                     
    u32_t stat_emac_tx_stat_dot3statsinternalmactransmiterrors;
    u32_t stat_Dot3StatsCarrierSenseErrors;                 
    u32_t stat_Dot3StatsFCSErrors;                          
    u32_t stat_Dot3StatsAlignmentErrors;                    
    u32_t stat_Dot3StatsSingleCollisionFrames;              
    u32_t stat_Dot3StatsMultipleCollisionFrames;            
    u32_t stat_Dot3StatsDeferredTransmissions;              
    u32_t stat_Dot3StatsExcessiveCollisions;                
    u32_t stat_Dot3StatsLateCollisions;                     
    u32_t stat_EtherStatsCollisions;                        
    u32_t stat_EtherStatsFragments;                         
    u32_t stat_EtherStatsJabbers;                           
    u32_t stat_EtherStatsUndersizePkts;                     
    u32_t stat_EtherStatsOverrsizePkts;                     
    u32_t stat_EtherStatsPktsRx64Octets;                    
    u32_t stat_EtherStatsPktsRx65Octetsto127Octets;         
    u32_t stat_EtherStatsPktsRx128Octetsto255Octets;        
    u32_t stat_EtherStatsPktsRx256Octetsto511Octets;        
    u32_t stat_EtherStatsPktsRx512Octetsto1023Octets;       
    u32_t stat_EtherStatsPktsRx1024Octetsto1522Octets;      
    u32_t stat_EtherStatsPktsRx1523Octetsto9022Octets;      
    u32_t stat_EtherStatsPktsTx64Octets;                    
    u32_t stat_EtherStatsPktsTx65Octetsto127Octets;         
    u32_t stat_EtherStatsPktsTx128Octetsto255Octets;        
    u32_t stat_EtherStatsPktsTx256Octetsto511Octets;        
    u32_t stat_EtherStatsPktsTx512Octetsto1023Octets;       
    u32_t stat_EtherStatsPktsTx1024Octetsto1522Octets;      
    u32_t stat_EtherStatsPktsTx1523Octetsto9022Octets;      
    u32_t stat_XonPauseFramesReceived;                      
    u32_t stat_XoffPauseFramesReceived;                     
    u32_t stat_OutXonSent;                                  
    u32_t stat_OutXoffSent;                                 
    u32_t stat_FlowControlDone;                             
    u32_t stat_MacControlFramesReceived;                    
    u32_t stat_XoffStateEntered;                            
    u32_t stat_IfInFramesL2FilterDiscards;                  
    u32_t stat_IfInRuleCheckerDiscards;                     
    u32_t stat_IfInFTQDiscards;                             
    u32_t stat_IfInMBUFDiscards;                            
    u32_t stat_IfInRuleCheckerP4Hit;                        
    u32_t stat_CatchupInRuleCheckerDiscards;                
    u32_t stat_CatchupInFTQDiscards;                        
    u32_t stat_CatchupInMBUFDiscards;                       
    u32_t stat_CatchupInRuleCheckerP4Hit;                   
    u32_t stat_GenStat00;                                   
    u32_t stat_GenStat01;                                   
    u32_t stat_GenStat02;                                   
    u32_t stat_GenStat03;                                   
    u32_t stat_GenStat04;                                   
    u32_t stat_GenStat05;                                   
    u32_t stat_GenStat06;                                   
    u32_t stat_GenStat07;                                   
    u32_t stat_GenStat08;                                   
    u32_t stat_GenStat09;                                   
    u32_t stat_GenStat10;                                   
    u32_t stat_GenStat11;                                   
    u32_t stat_GenStat12;                                   
    u32_t stat_GenStat13;                                   
    u32_t stat_GenStat14;                                   
    u32_t stat_GenStat15;                                   
} teton_stat_block_t;

#define GET_NET_STATS64(ctr)                    \
    (unsigned long) ((unsigned long) (ctr##_hi) << 32) +    \
    (unsigned long) (ctr##_lo)

#define GET_NET_STATS32(ctr)        \
    (ctr##_lo)

#if (BITS_PER_LONG == 64)
#define GET_NET_STATS   GET_NET_STATS64
#else
#define GET_NET_STATS   GET_NET_STATS32
#endif

/*******************************************************************************
 * Teton media type(same define in lm_defs.h).
 ******************************************************************************/

#define LM_MEDIUM_AUTO_DETECT                   0x0000

#define LM_MEDIUM_TYPE_UNKNOWN                  0x0000
#define LM_MEDIUM_TYPE_BNC                      0x0001
#define LM_MEDIUM_TYPE_UTP                      0x0002
#define LM_MEDIUM_TYPE_FIBER                    0x0003
#define LM_MEDIUM_TYPE_PHY_LOOPBACK             0x00fd
#define LM_MEDIUM_TYPE_MAC_LOOPBACK             0x00fe
#define LM_MEDIUM_TYPE_NULL                     0x00ff
#define LM_MEDIUM_TYPE_MASK                     0x00ff
#define GET_MEDIUM_TYPE(m)                      ((m) & LM_MEDIUM_TYPE_MASK)
#define SET_MEDIUM_TYPE(m, t) \
    (m) = ((m) & ~LM_MEDIUM_TYPE_MASK) | (t)

#define LM_MEDIUM_SPEED_AUTONEG                 0x0000

#define LM_MEDIUM_SPEED_UNKNOWN                 0x0000
#define LM_MEDIUM_SPEED_10MBPS                  0x0100
#define LM_MEDIUM_SPEED_100MBPS                 0x0200
#define LM_MEDIUM_SPEED_1000MBPS                0x0300
#define LM_MEDIUM_SPEED_2500MBPS                0x0400
#define LM_MEDIUM_SPEED_AUTONEG_1G_FALLBACK     0x8300  /* Serdes */
#define LM_MEDIUM_SPEED_AUTONEG_2_5G_FALLBACK   0x8400  /* Serdes */
#define LM_MEDIUM_SPEED_HARDWARE_DEFAULT        0xff00  /* Serdes nvram def. */
#define LM_MEDIUM_SPEED_MASK                    0xff00
#define GET_MEDIUM_SPEED(m)                     ((m) & LM_MEDIUM_SPEED_MASK)
#define SET_MEDIUM_SPEED(m, s) \
    (m) = ((m) & ~LM_MEDIUM_SPEED_MASK) | (s)

#define LM_MEDIUM_FULL_DUPLEX                   0x00000
#define LM_MEDIUM_HALF_DUPLEX                   0x10000
#define GET_MEDIUM_DUPLEX(m)                    ((m) & LM_MEDIUM_HALF_DUPLEX)
#define SET_MEDIUM_DUPLEX(m, d) \
    (m) = ((m) & ~LM_MEDIUM_HALF_DUPLEX) | (d)

#define LM_MEDIUM_SELECTIVE_AUTONEG             0x01000000
#define GET_MEDIUM_AUTONEG_MODE(m)              ((m) & 0xff000000)

//
// Function prototypes
//
void 
drv_init(
    int enable_l2, 
    int debug_mode, 
    void *lock
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_init_board(
    void *pcidev, 
    char *name
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_remove_one(
    void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
bcm06_get_dev(
    void **dev_arr,
    u32_t arr_size,
    int *dev_num
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
bcm06_get_interface(
    void *bcm_dev,
    u32_t if_type,
    void *interface,
    u32_t if_size,
    u32_t if_ver
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
um_bdrv_handle_isr(
    void *dev_instance
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_suspend(
    void *pcidev, 
    u32_t state
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_resume(
    void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8 *
pal_alloc_resource( 
    void 
    );

U8 *
pal_get_resource_addr(
    U8 *base,
    PAL_RESOURCE_TYPE type,
    u32_t idx
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_free_resource(
    U8* base
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void *
pal_ioremap(
    unsigned long phy_addr,
    unsigned long size
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_iounmap(
    void *virt_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_init_MUTEX(
    PAL_MUTEX mutex
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_request_irq(
    u32_t irq,
    char *name,
    void *data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Event functions
//
long 
pal_wait_event(
    PAL_EVENT event,
    signed long ms
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

/*void
pal_set_event(
    PAL_EVENT* event
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif*/

void
pal_init_event(
    PAL_EVENT event
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Time related functions
//
unsigned long
pal_get_current_time(
    void
    );

U32
pal_diff_ms(
    unsigned long tstart,
    unsigned long tend
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned long
pal_advance_second(
    unsigned long tstart,
    unsigned long second
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Memory copy functions
//
void*
pal_memcpy(
    void* dst,
    void* src,
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_memcmp(
    void* dst,
    void* src,
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void*
pal_memset(
    void* dst,
    int val,
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void*
pal_malloc(
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_mfree(
    void* p
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_copy_from_user(
    void* to,
    void* from,
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_copy_to_user(
    void* to,
    void* from,
    U32 len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_stricmp(
    char* str1,
    char* str2
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Atomic functions
//
int
pal_atomic_read(
    PAL_ATOMIC atom
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_atomic_set(
    PAL_ATOMIC atom,
    int val
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_atomic_inc(
    PAL_ATOMIC atom
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_atomic_dec(
    PAL_ATOMIC atom
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_atomic_dec_and_test(
    PAL_ATOMIC atom
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_atomic_add(
    PAL_ATOMIC atom,
    int val
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Packet functions
//
PAL_PACKET
pal_alloc_packet(
    U32 size,
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_packet_reserve(
    PAL_PACKET skb,
    unsigned int len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_free_packet(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_init_packet_list(
    PAL_PACKET_LIST list
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_update_dev_time(
    PAL_DEVICE dev,
    u32_t tx
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_enqueue_packet(
    PAL_PACKET_LIST list,
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

PAL_PACKET
pal_dequeue_packet(
    PAL_PACKET_LIST list
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_packet_device(
    PAL_DEVICE dev,
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

PAL_DEVICE
pal_get_packet_device(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_packet_protocol(
    PAL_DEVICE dev,
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_send_packet(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8*
pal_get_packet_reserved(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8*
pal_packet_put(
    PAL_PACKET pkt,
    unsigned int len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_indicate_receive_packet(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_indicate_receive_vlan_tag(
    PAL_PACKET pkt,
    void *vlgrp,
    unsigned short vlan_tag
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_packet_len(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8*
pal_get_packet_data(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8*
pal_get_packet_raw(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_packet_set_ip_summed(
    PAL_PACKET pkt,
    U32 hw_chksum
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_packet_do_hw_cksum(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_vlan_tx_tag_present(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U16
pal_vlan_tx_tag_get(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_packet_frags_num(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_packet_frag_len(
    PAL_PACKET pkt,
    U32 idx
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_packet_headlen(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_packet_mss(
    PAL_PACKET pkt
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_packet_lso(
    PAL_PACKET pkt,
    u8_t *ip_hdr_len,
    u8_t *tcp_hdr_len,
    u32_t mss
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_module_get(void);

void
pal_module_put(void);

//
// Spinlock functions
//
void
pal_init_spinlock(
    PAL_SPINLOCK lock
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_spin_lock_irqsave(
    PAL_SPINLOCK lock,
    unsigned long* flags
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_spin_lock(
    PAL_SPINLOCK lock
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_spin_unlock_irqrestore(
    PAL_SPINLOCK lock,
    unsigned long* flags
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_spin_unlock(
    PAL_SPINLOCK lock
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_local_irq_save(
    unsigned long *flags
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_local_irq_restore(
    unsigned long *flags
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_spin_is_locked(
    PAL_SPINLOCK lock
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8 *
pal_alloc_tcp_con_locks(
    u32_t *lock_size
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

typedef void (*fp_hook)(void);

//
// Debug functions
//
void 
pal_print(
    const char* fmt,
    ...
    );

void
pal_panic(
    const char* msg
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_sprintf(
    char* buf,
    const char* fmt,
    ...
    );

//
// Device functions
//
int
pal_is_device_up(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

char*
pal_get_device_name(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

char*
pal_get_if_name(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U32
pal_get_ip_addr(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_device_name(
    PAL_DEVICE dev,
    char* name
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

PAL_DEVICE
pal_get_device_by_name(
    char* name
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_accumulate_device_stat(
    volatile teton_stat_block_t *stats_blk,
    PAL_DEVICE_STAT stat
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void*
pal_get_device_private(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_device_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

U8*
pal_get_device_mac_address(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_device_handlers(
    PAL_DEVICE dev,
    int (*open)(PAL_DEVICE),
    int (*stop)(PAL_DEVICE),
    int (*hard_start_xmit)(PAL_PACKET, PAL_DEVICE),
    void (*set_multicast_list)(PAL_DEVICE),
    int (*do_ioctl)(PAL_DEVICE, void*, int),
    PAL_DEVICE_STAT (*get_stats)(PAL_DEVICE),
    int (*set_mac_address)(PAL_DEVICE, void*),
    void (*vlan_rx_register)(PAL_DEVICE, void*),
    void (*vlan_rx_kill_vid)(PAL_DEVICE, u16_t),
    unsigned int irq
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_device_poll_hdl(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_dec_device_reference(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_inc_device_reference(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_get_device_mtu(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_device_mtu(
    PAL_DEVICE dev,
    int mtu
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

//
// Linux module support
//
int
pal_inc_usage(
    void
    );

void
pal_dec_usage(
    void
    );

void
pal_free_irq(
    u32_t   irq,
    void    *dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_dev_features(
    PAL_DEVICE dev,
    int highdma
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_send(
    PAL_PACKET pkt,
    PAL_DEVICE dev
   );

int
mpd_start_xmit(
    PAL_PACKET pkt,
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_open(
    PAL_DEVICE dev 
    );

int
mpd_open(
    PAL_DEVICE dev 
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_close(
    PAL_DEVICE dev 
    );

int
mpd_close(
    PAL_DEVICE dev 
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_ioctl(
    PAL_DEVICE dev,
    void* ifr,
    int cmd
    );

int
mpd_get_phy_id(
    PAL_DEVICE dev,
    u32_t *phy_id
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int 
mpd_read_phy_reg(
    PAL_DEVICE dev,
    u32_t phy_addr,
    u32_t *phy_data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_write_phy_reg(
    PAL_DEVICE dev,
    u32_t phy_addr,
    u32_t phy_data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

PAL_DEVICE_STAT
pal_get_stats(
    PAL_DEVICE dev
    );

PAL_DEVICE_STAT
mpd_get_stats(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_multicast_list(
    PAL_DEVICE dev 
    );

void
mpd_set_multicast_list(
    PAL_DEVICE dev 
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void 
pal_vlan_rx_register(
    PAL_DEVICE dev,
    void *vlgrp
    );

void 
mpd_vlan_rx_register(
    PAL_DEVICE dev,
    void *vlgrp
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void 
pal_vlan_rx_kill_vid(
    PAL_DEVICE dev,
    u16_t vid
    );

void 
mpd_vlan_rx_kill_vid(
    PAL_DEVICE dev,
    u16_t vid
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_kill_vid(
    void *vlgrp,
    u16_t vid
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_settings(
    PAL_DEVICE dev,
    u32_t *setting,
    u32_t *advertising,
    u32_t *autoneg
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_adv_init(
    u32_t fiber, 
    u32_t *advertising
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_set_settings(
    PAL_DEVICE dev, 
    u32_t req_medium, 
    u32_t advertising, 
    u32_t autoneg,
    u32_t selective_autoneg
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_drvinfo(
    PAL_DEVICE dev, 
    char **bus_info, 
    u32_t *fw_ver
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_wake_magic(
    PAL_DEVICE dev, 
    u32_t *wake_magic, 
    u32_t *enable
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_set_wol(
    PAL_DEVICE dev, 
    u32_t enable
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_nway_reset(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_get_eeprom_len(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_get_eeprom(
    PAL_DEVICE dev, 
    u32_t offset, 
    u32_t len, 
    u8_t *eebuf
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_set_eeprom(
    PAL_DEVICE dev, 
    u32_t offset, 
    u32_t len, 
    u8_t *eebuf
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_get_coalesce(
    PAL_DEVICE dev,
    u32_t *rx_coalesce_usecs, u32_t *rx_max_coalesced_frames, 
    u32_t *rx_coalesce_usecs_irq, u32_t *rx_max_coalesced_frames_irq,
    u32_t *tx_coalesce_usecs, u32_t *tx_max_coalesced_frames, 
    u32_t *tx_coalesce_usecs_irq, u32_t *tx_max_coalesced_frames_irq,
    u32_t *stats_block_coalesce_usecs
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_ringparam(
    PAL_DEVICE dev, 
    u32_t* rx_max_pending, 
    u32_t *tx_max_pending
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_pauseparam(
    PAL_DEVICE dev, 
    u32_t *flow_autoneg, 
    u32_t *rx_pause, 
    u32_t *tx_pause
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_set_pauseparam(
    PAL_DEVICE dev,
    u32_t flow_autoneg, 
    u32_t rx_pause, 
    u32_t tx_pause
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

u32_t
mpd_get_rx_csum(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_set_rx_csum(
    PAL_DEVICE dev, 
    u32_t data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_strings(
    PAL_DEVICE dev, 
    u8_t *buf
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_phys_id(
    PAL_DEVICE dev,
    u32_t data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
mpd_get_stats_count(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
mpd_get_ethtool_stats(
    PAL_DEVICE dev, 
    u64_t *buf
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_change_mac_address(
    PAL_DEVICE dev,
    void *p
    );

int
mpd_change_mac_addr(
    PAL_DEVICE dev, 
    void *sa_data, 
    u32_t addr_len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
poll_mpd(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int 
pal_pci_enable_device(void *pcidev, int *irq, int *domain, int *bus_num, int *devfn
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned long
pal_pci_is_iomap(void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int 
pal_pci_request_regions(void *pcidev, char *name
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_set_master(void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_pci_set_dma_mask(void *pcidev, u64_t mask
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_pci_set_consistent_dma_mask(void *pcidev, u64_t mask
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned long
pal_pci_resource_start(void *pcidev, u32_t bar
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned long
pal_pci_resource_len(void *pcidev, u32_t bar
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void 
pal_pci_set_drvdata (void *pcidev, void *data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void *
pal_pci_get_drvdata(void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_release_regions(void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_disable_device(void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

u64_t
pal_pci_map_single(
    void *pcidev,
    void *addr,
    u32_t size,
    int from_dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_unmap_single(
    void *pcidev,
    u64_t addr,
    u32_t size,
    int from_dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

u64_t
pal_pci_map_page(
    void *pcidev,
    PAL_PACKET pkt,
    u32_t idx,
    u32_t *len,
    int from_dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_unmap_page(
    void *pcidev,
    u64_t addr,
    u32_t size,
    int from_dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void *
pal_pci_alloc_consistent(
    void *pcidev,
    u32_t mem_size,
    u64_t *phy_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_pci_free_consistent(
    void *pcidev,
    u32_t mem_size,
    void  *virt_mem,
    u64_t phy_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_pci_write_config_dword(
    void *pcidev,
    int cfg_addr,
    u32_t val
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_pci_read_config_dword(
    void *pcidev,
    int cfg_addr,
    u32_t *val
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

char *
pal_pci_get_slot_name(
    void *pcidev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void*
pal_vmalloc(
    u32_t len
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_vmfree(
    void* p
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

PAL_DEVICE
pal_alloc_dev(void *pcidev, int sizeof_priv
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_register_dev(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_unregister_dev(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_free_dev(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_carrier_off(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_netif_running(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_stop_queue(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_device_detach(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_device_attach(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_wake_queue(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int 
pal_netif_queue_stopped(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_carrier_on(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_netif_start_queue(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned short
pal_dev_promisc(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

unsigned short
pal_dev_allmulti(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int 
pal_get_dev_mc_count(
    PAL_DEVICE dev
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void *
pal_get_dev_mclist_head(
    PAL_DEVICE dev,
    u8_t **mac_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void *
pal_get_dev_mclist_next(
    void *mclist,
    u8_t **mac_addr
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void pal_driver_pulse(unsigned long FunctionContext);
void pal_poll_for_link(unsigned long FunctionContext);

void
driver_pulse(
    unsigned long FunctionContext
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
poll_for_link(
    unsigned long FunctionContext
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_init_timer(
    PAL_TIMER timer
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_set_timer(
    void *pdev,
    PAL_TIMER timer,
    unsigned long msec,
    void (*tcallback)(unsigned long)
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_cancel_timer(
    PAL_TIMER timer
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_tasklet_init(
    PAL_TASKLET bcm_tasklet,
    void (*func)(unsigned long),
    unsigned long data
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_tasklet_schedule(
    PAL_TASKLET bcm_tasklet
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_cancel_tasklet(
    PAL_TASKLET bcm_tasklet
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_down(
    PAL_MUTEX mutex
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_up(
    PAL_MUTEX mutex
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_wake_up_event(
    PAL_EVENT event
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int
pal_schedule_timeout(
    int interruptible, 
    u32_t div_hz
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

int pal_is_dump_mode(void);

void
pal_disable_irq(
    u32_t irq
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_enable_irq(
    u32_t irq
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

void
pal_udelay(
    unsigned long usecs
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
    ) REGPARM(3);
#else
    );
#endif

#ifdef CONFIG_PPC64
#define pal_break_if(_c)
#else
#define pal_break_if(_c) \
    if(_c) \
    { \
        __asm__("int $129"); \
    }
#endif

#endif  // __pal_h__

