/****************************************************************************
 * Copyright(c) 2005 Broadcom Corporation, all rights reserved
 * Proprietary and Confidential Information.
 *
 * This source file is the property of Broadcom Corporation, and
 * may not be copied or distributed in any isomorphic form without
 * the prior written consent of Broadcom Corporation.
 *
 * Name:        b06_diag.c
 *
 * Description: Linux device driver for xdiag
 *
 *
 * Author:      John Chen
 *
 *
 ****************************************************************************/

static const char *version = "Broadcom NetXtreme II Kdiag Driver v7.2.50 (April 5, 2012)\n";

#if defined(CONFIG_SMP) && ! defined(__SMP__)
#define __SMP__
#endif
#if defined(CONFIG_MODVERSIONS) && defined(MODULE) && ! defined(MODVERSIONS)
#define MODVERSIONS
#endif

#include <linux/version.h>
#ifdef MODULE
#if defined(MODVERSIONS) && (LINUX_VERSION_CODE < 0x020500)
#include <linux/modversions.h>
#endif
#include <linux/module.h>
#else
#define MODULE_DEVICE_TABLE(pci, pci_tbl)
#endif

#include <linux/types.h>
#include <linux/string.h>
#include <asm/io.h>
#include <linux/errno.h>
#include <linux/netdevice.h>

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,20))
#include <linux/fs.h>
#include <linux/mm.h>
#endif

MODULE_LICENSE("Proprietary");

#include "diag_pal.h"

#define DIAG_MAJOR 240

int diag_major = DIAG_MAJOR;

void *b06_dev[MAX_06DEV];
int dev_num = 0;

int diag_open(struct inode *inode, struct file *filp)
{
    return b06diag_open(&filp->private_data);
}

int diag_close(struct inode *inode, struct file *filp)
{
    return b06diag_close(filp->private_data);
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36))
int diag_ioctl(struct inode *inode, struct file *filp, uint cmd, ulong arg)
{
    return b06diag_ioctl(filp->private_data, cmd, arg);
}
#else
long diag_ioctl(struct file *filp, uint cmd, ulong arg)
{
    return b06diag_ioctl(filp->private_data, cmd, arg);
}
#endif

int diag_mmap(struct file *filep, struct vm_area_struct *vma)
{
    int rc;
    unsigned long hdl = vma->vm_pgoff;
    void *kv_addr;

    kv_addr = b06diag_get_map_addr(hdl);

    if(NULL == kv_addr)
    {
        panic("b06_mmap: can not get map data\n");
        return -EINVAL;
    }

//    vma->vm_flags |= VM_LOCKED;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))
//    printk("early lx version\n");
    if((rc = io_remap_page_range(
                            vma,
                            vma->vm_start,
                            virt_to_phys(kv_addr),
                            vma->vm_end - vma->vm_start,
                            vma->vm_page_prot))) {
        printk("b06_mmap: io_remap_page_range failed. rc=%d\n", rc);
        return -EAGAIN;
    }
#else
#if CONFIG_XEN
//    printk("Xen patch here");
    /* This API call is a patch for Xen. CQ#38689 */
    /* when io_remap_page_range is called, it returns error code -14. The failure 
     * happens when updates hypervisor.
     * io_remap_page_range is defined to direct_remap_page_range on Xen in pgtable.h
     * direct_remap_page_range will call remap_pfn_range if flag
     * XENFEAT_auto_translated_physmap is on. To me this means bypass hypervisor.
     * since xdiag is designed to verify HW only, it is reasonable to just let Linux
     * kernel handle the io map and skip hypervisor.
     * Will revisit this problem once the "right" approach is discovered 01/26/2009
     */
    
    if((rc = remap_pfn_range(
                            vma,
                            vma->vm_start,
                            (virt_to_phys(kv_addr) >> PAGE_SHIFT),
                            vma->vm_end - vma->vm_start,
                            vma->vm_page_prot)))
    {
        printk("b06_mmap: remap_pfn_range(XEN) failed. rc=%d\n", rc);
        return -EAGAIN;
    }
#else
//    printk("NOT Xen patch");
    if((rc = io_remap_pfn_range(
                            vma,
                            vma->vm_start,
                            (virt_to_phys(kv_addr) >> PAGE_SHIFT),
                            vma->vm_end - vma->vm_start,
                            vma->vm_page_prot)))
    {
        printk("b06_mmap: io_remap_page_range failed. rc=%d\n", rc);
        return -EAGAIN;
    }
#endif /*CONFIG_XEN*/
#endif
    return 0;
}

#ifdef CONFIG_COMPAT
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))
#include <linux/ioctl.h>
#include <linux/ioctl32.h>

/* Use 'B' as magic number */
#define B06_IOC_MAGIC  'B'
#define MAX_IOCTL_NUM 53        //Need to update if new IOCTL is added

static int
compat_diag_ioctl(unsigned int fd, unsigned int cmd,
            unsigned long arg, struct file *filp)
{
    if (!filp->f_op || filp->f_op->ioctl != diag_ioctl)
    {
        printk(KERN_CRIT "<B06_DIAGS> ioctl is not matched!\n");
        return -EFAULT;
    }
    return filp->f_op->ioctl(filp->f_dentry->d_inode, filp, cmd, arg);
}

static int
diag_ioctl32_register(void)
{
    int err, ret = 0;
    u32_t i, k;

    for (i = 0; i <= MAX_IOCTL_NUM; i++)
    {
        err = register_ioctl32_conversion(_IO(B06_IOC_MAGIC, i), compat_diag_ioctl);
        if (err < 0)
            break;
    }
    if (i  <= MAX_IOCTL_NUM)
    {
        //Can not register all IOCTL
        printk(KERN_CRIT "<B06_DIAGS> can not regidter all 32bit ioctl!\n");
        for (k = 0; k < i; k++)
            unregister_ioctl32_conversion(_IO(B06_IOC_MAGIC, k));

        ret = -1;
    }
    else
        ret = 0;

    return ret;
}

static void
diag_ioctl32_unregister(void)
{
    u32_t i;

    for (i = 0; i <= MAX_IOCTL_NUM; i++)
        unregister_ioctl32_conversion(_IO(B06_IOC_MAGIC, i));

    return;
}
#else /*#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))*/
static long
compat_diag_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    return b06diag_ioctl(filp->private_data, cmd, arg);
}
#endif /*#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))*/
#endif /*#ifdef CONFIG_COMPAT*/

struct file_operations diag_fops = {
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36))
    ioctl:diag_ioctl,
#else
    unlocked_ioctl:diag_ioctl,
#endif
#if defined(CONFIG_COMPAT) && (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,12))
    compat_ioctl:compat_diag_ioctl,
#endif
    mmap:diag_mmap,
    open:diag_open,
    release:diag_close,
};

int
init_module(void)
{
    int ret = -ENODEV;

    printk(KERN_INFO "%s", version);
//  __asm__("int $129");        //break point

    bcm_get_dev(b06_dev, MAX_06DEV, &dev_num);
    if (0 == dev_num)
    {
        printk(KERN_CRIT "<B06_DIAGS> No bcm5706 device found!\n");
        //Don't return failure even no device so that xdiag
        //can be used to read/write IO
        //return -ENODEV;
    }

    /* Resister the major and accept dynamic number */
    ret = register_chrdev(diag_major, "diag06", &diag_fops);

    if (diag_major == 0)
    {
        printk("diag_major = %d\n", diag_major);
        diag_major = ret;
    }
    if (ret < 0)
    {
        printk(KERN_CRIT "dpa: can't get major %d\n", diag_major);
    }
#if defined(CONFIG_COMPAT) && (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))
    else
    {
        ret = diag_ioctl32_register();
        if (ret)
            unregister_chrdev(diag_major, "diag06");
    }
#endif
    return ret;
}

void
cleanup_module(void)
{
    if (diag_major > 0)
        unregister_chrdev(diag_major, "diag06");

#if defined(CONFIG_COMPAT) && (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,12))
    diag_ioctl32_unregister();
#endif
    return;
}






