
/*
* Copyright 2014 QLogic Corporation
* The contents of this file are subject to the terms of the
* QLogic End User License (the "License").
* You may not use this file except in compliance with the License.
*
* You can obtain a copy of the License at
* http://www.qlogic.com/Resources/Documents/DriverDownloadHelp/
* QLogic_End_User_Software_License.txt
* See the License for the specific language governing permissions
* and limitations under the License.
*/

/*
 * Copyright (c) 2002, 2011, Oracle and/or its affiliates. All rights reserved.
 */

#include "bnx.h"
#ifdef __sparc
#include <v9/sys/mutex_impl.h>
#else
#include <sys/mutex_impl.h>
#endif
#include <sys/mdb_modapi.h>

char _depends_on[]= "misc/kmdbmod";

/* from mdb_ks.c */
extern int mdb_name_to_major(const char *name, major_t *major);


static int major_to_addr(major_t     major,
                         uintptr_t * vaddr)
{
    uint_t    devcnt;
    uintptr_t devnamesp;

    if (mdb_readvar(&devcnt, "devcnt") == -1)
    {
        mdb_warn("failed to read 'devcnt'");
        return -1;
    }

    if (mdb_readvar(&devnamesp, "devnamesp") == -1)
    {
        mdb_warn("failed to read 'devnamesp'");
        return -1;
    }

    if (major >= devcnt)
    {
        mdb_warn("%x is out of range [0x0-0x%x]\n", major, devcnt - 1);
        return -1;
    }

    *vaddr = devnamesp + (major * sizeof (struct devnames));
    return 0;
}


static void BnxMdbBasic(um_device_t * pUM)
{
    lm_device_t * pLM = (lm_device_t *)&pUM->lm_dev;

    mdb_printf("Version:            %-10s Firmware:              %s\n",
               pUM->version, pUM->versionFW);
    mdb_printf("ChipName:           %-10s\n",
               pUM->chipName);
    mdb_printf("\n");
    mdb_printf("L2 MacAddr:         %02x:%02x:%02x:%02x:%02x:%02x\n",
               pLM->params.mac_addr[0], pLM->params.mac_addr[1],
               pLM->params.mac_addr[2], pLM->params.mac_addr[3],
               pLM->params.mac_addr[4], pLM->params.mac_addr[5]);
    mdb_printf("\n");
    mdb_printf("intrAlloc:          %-10s intrFired:             %d\n",
               pUM->intrAlloc, pUM->intrFired);
}


void dword_dump(u32_t   offset,
                u32_t * buf,
                u32_t   buf_len)
{
    u32_t val;
    u32_t j;

    for(j = 0; j < buf_len/sizeof(u32_t); j ++)
    {
        if(!(j % 4))
        {
            mdb_printf("%08x:", offset);
        }

        if(!(j % 2))
        {
            mdb_printf(" ");
        }

        val = buf[j];

        if(!((j+1) % 4) || (j+1) >= buf_len/sizeof(u32_t))
        {
            mdb_printf("%08x\n", val);
        }
        else
        {
            mdb_printf("%08x ", val);
        }

        offset += 4;
    }
}

static u32_t mreg_rd(um_device_t * pUM,
                     u32_t         offset)
{
    u32_t   val = 0;
    ssize_t rc;

    if ((rc = mdb_vread(&val,
                        sizeof(u32_t),
                        (uintptr_t)((u8_t *)pUM->lm_dev.vars.regview +
                                    offset))) !=
        sizeof(u32_t))
    {
        mdb_printf("ERROR: mreg_rd vread failed 0x%p (%d)\n",
                   (uintptr_t)((u8_t*)pUM->lm_dev.vars.regview + offset),
                   rc);
    }

    return val;
}

static void mreg_wr(um_device_t * pUM,
                    u32_t         offset,
                    u32_t         val)
{
    ssize_t rc;

    if ((rc = mdb_vwrite(&val,
                         sizeof(u32_t),
                         (uintptr_t)((u8_t *)pUM->lm_dev.vars.regview +
                                     offset))) !=
        sizeof(u32_t))
    {
        mdb_printf("ERROR: mreg_wr vwrite failed 0x%p (%d)\n",
                   (uintptr_t)((u8_t*)pUM->lm_dev.vars.regview + offset),
                   rc);
    }
}

void reg_wr(um_device_t * pUM,
            u32_t         offset,
            u32_t         val)
{
    u32_t indirect_reg_start;

#if 0
    if (reg_type == REG_TYPE_5706)
    {
        indirect_reg_start = 0x8000;
    }
    else
    {
        indirect_reg_start = 0x10000;
    }
#else
    indirect_reg_start = 0x8000;
#endif

    if (offset < indirect_reg_start)
    {
        mreg_wr(pUM, offset, val);
    }
    else
    {
        mreg_wr(pUM, 0x78, offset);
        mreg_wr(pUM, 0x80, val);
    }
}

u32_t reg_rd(um_device_t * pUM,
             u32_t         offset)
{
    u32_t indirect_reg_start;
    u32_t val;

#if 0
    if (reg_type == REG_TYPE_5706)
    {
        indirect_reg_start = 0x8000;
    }
    else
    {
        indirect_reg_start = 0x10000;
    }
#else
    indirect_reg_start = 0x8000;
#endif

    if (offset < indirect_reg_start)
    {
        val = mreg_rd(pUM, offset);
    }
    else
    {
        mreg_wr(pUM, 0x78, offset);
        val = mreg_rd(pUM, 0x80);
    }

    return val;
}

static u32_t rd_ctx(um_device_t * pUM,
                    u32_t         cid,
                    u32_t         offset)
{
    u32_t page_offset;
    u64_t page_addr;
    u32_t page_idx;
    u32_t val;
    ssize_t rc;

    offset += (cid << CTX_SHIFT);

    if (CHIP_NUM(&pUM->lm_dev) == 0x57090000)
    {
#if 0
        page_idx = (offset / 0x1000);
        page_offset = (offset & 0xfff);

        reg_wr(pUM, 0x10c8, CTX_CACHE_CTRL_READ_REQ | page_idx);

        val = reg_rd(pUM, 0x10d0);
        page_addr = (val << 16);
        page_addr <<= 16;

        val = reg_rd(pUM, 0x10cc);

        if ((val & 0x1) == 0)
        {
            return 0xffffffff;
        }

        val &= 0xfffffff0;

        page_addr += val;
        page_addr += page_offset;

        if ((rc = mdb_pread(&val,
                            sizeof(u32_t),
                            (uint64_t)page_addr)) !=
            sizeof(u32_t))
        {
            mdb_printf("ERROR: rd_ctx pread failed 0x%llx (%d)\n",
                       (uint64_t)page_addr, rc);
        }
#else
        {
        int idx;
        int retry_cnt = 25;

        reg_wr(pUM, 0x101c, offset | CTX_CTX_CTRL_READ_REQ);

        for (idx = 0; idx < retry_cnt; idx++)
        {
            val = reg_rd(pUM, 0x101c);

            if ((val & CTX_CTX_CTRL_READ_REQ) == 0)
            {
                break;
            }

            //mm_wait(pUM, 5);
        }

        if (idx == retry_cnt)
        {
            mdb_printf("ERROR: failed to set context control address");
        }

        val = reg_rd(pUM, 0x1020);
        }
#endif
    }
    else
    {
        reg_wr(pUM, 0x1010, offset);
        val = reg_rd(pUM, 0x1014);
    }

    return val;
}

static void rd_ctx_in_buffer(um_device_t * pUM,
                             u32_t         cid,
                             u8_t *        ctx_buf,
                             u32_t         buf_len)
{
    u32_t offset;

    for (offset = 0; offset < buf_len; offset += 4)
    {
        *((u32_t *) ctx_buf) = 0;
        *((u32_t *) ctx_buf) = rd_ctx(pUM, cid, offset);

        ctx_buf += 4;
    }
}

static void dump_ctx_as_dword(um_device_t * pUM,
                              u32_t         cid_addr,
                              u32_t         cnt)
{
    u32_t ctx_buf[CTX_SIZE/sizeof(u32_t)];
    u32_t cid;

    cid = (cid_addr / CTX_SIZE);

    while (cnt)
    {
        rd_ctx_in_buffer(pUM, cid, (u8_t *)ctx_buf, CTX_SIZE);

        dword_dump(cid_addr, ctx_buf, CTX_SIZE);

        cid_addr += CTX_SIZE;

        cid++;

        cnt--;
    }
}

static void BnxMdbChain(um_device_t * pUM,
                        int           idx)
{
    lm_device_t *   pLM = (lm_device_t *)&pUM->lm_dev;
    lm_tx_chain_t * pTxq = &pLM->tx_info.chain[idx];
    lm_rx_chain_t * pRxq = &pLM->rx_info.chain[idx];
    u16_t hwTxqConIdx = 0;
    u16_t hwRxqConIdx = 0;

    if ((pTxq->hw_con_idx_ptr != NULL) ||
        (pRxq->hw_con_idx_ptr != NULL))
    {
        /* some might fail... */
        mdb_vread(&hwTxqConIdx, sizeof(u16_t), (uint64_t)pTxq->hw_con_idx_ptr);
        mdb_vread(&hwRxqConIdx, sizeof(u16_t), (uint64_t)pRxq->hw_con_idx_ptr);
    }

    mdb_printf("-- Chain %d --\n", idx);

    //BnxMdbSb(pUM, idx);

    mdb_printf(" < Rx CID=0x%x >\n", pRxq->cid_addr);
    mdb_printf("   hwRxqConIdx:    %-12d rxBDsAvail:     %d\n",
               hwRxqConIdx, pRxq->bd_left);
    mdb_printf("   rxqBdProdIdx:   %-12d rxqBdConsIdx:   %d\n",
               pRxq->prod_idx, pRxq->con_idx);
    mdb_printf("   rxFreeDesc:     %-12d rxActiveDesc:   %d\n",
               s_list_entry_cnt(&pRxq->free_descq),
               s_list_entry_cnt(&pRxq->active_descq));
    mdb_printf("   rxWaitingPkts:  %-12d\n",
               s_list_entry_cnt(&pUM->rxq[idx].waitq));

    mdb_printf(" < Tx CID=0x%x >\n", pTxq->cid_addr);
    mdb_printf("   hwTxqConIdx:    %-12d txBDsAvail      %d\n",
               hwTxqConIdx, pTxq->bd_left);
    mdb_printf("   txqBdProdIdx:   %-12d txqBdConsIdx:   %d\n",
               pTxq->prod_idx, pTxq->con_idx);
    mdb_printf("   txFreeDesc:     %-12d txWaitingPkts:  %d\n",
               s_list_entry_cnt(&pUM->txq[idx].free_tx_desc),
               s_list_entry_cnt(&pUM->txq[idx].tx_resc_que));
}


static void BnxMdbHelp(void)
{
    mdb_printf("addr::    dev_info address ('::devbindings -q bnx')\n");
    mdb_printf("-i #      driver instance # (instead of 'addr::')\n");
    mdb_printf("-a        show all (basic, all chains)\n");
    mdb_printf("-b        basic status (default)\n");
    mdb_printf("-c #      status for chain # (-1 for all chains)\n");
    mdb_printf("-x <cid>  context memory from CID addr (128 byte aligned)\n");
    mdb_printf("-n <cnt>  number of 128 byte chunks from context memory\n");
}


static int BnxMdb(uintptr_t         addr,
                  uint_t            flags,
                  int               argc,
                  const mdb_arg_t * argv)
{
    struct dev_info dip;
    um_device_t     um;
    struct devnames dn;
    uintptr_t       dn_addr;
    major_t         major;
    uint_t   opt_a = 0;
    uint_t   opt_b = 0;
#define OPT_NONE (uint64_t)0x0deadead
    uint64_t opt_i = OPT_NONE;
    uint64_t opt_c = OPT_NONE;
    uint64_t opt_x = OPT_NONE;
    uint64_t opt_n = OPT_NONE;
    int idx;

    if (!(flags & DCMD_ADDRSPEC))
    {
        if (mdb_name_to_major("bnx", &major) != 0)
        {
            mdb_printf("ERROR: Failed to get major number for bnx\n");
            return DCMD_ERR;
        }

        if (major_to_addr(major, &dn_addr) != 0)
        {
            mdb_printf("ERROR: Failed to get address for bnx\n");
            return DCMD_ERR;
        }

        if (mdb_vread(&dn, sizeof(struct devnames), dn_addr) == -1)
        {
            mdb_printf("ERROR: Couldn't read devnames array at %p\n",
                       dn_addr);
            return DCMD_ERR;
        }

        if (mdb_pwalk_dcmd("devi_next", "bnx", argc, argv,
                           (uintptr_t)dn.dn_head) != 0)
        {
            mdb_printf("ERROR: Couldn't walk the devinfo chain at %p\n",
                       dn.dn_head);
            return DCMD_ERR;
        }

        return DCMD_OK;
    }

    if (mdb_vread(&dip, sizeof(struct dev_info), addr) == -1)
    {
        mdb_printf("ERROR: Unable to read dev_info\n");
        return DCMD_ERR;
    }

    if (mdb_vread(&um, sizeof(um_device_t), (uint64_t)dip.devi_driver_data) == -1)
    {
        mdb_printf("ERROR: Unable to read um_device_t\n");
        return DCMD_ERR;
    }

    if (um.magic != BNX_MAGIC)
    {
        mdb_printf("ERROR: Invalid dip address for a bnx instance\n");
        return DCMD_ERR;
    }

    if (mdb_getopts(argc, argv,
                    'i', MDB_OPT_UINT64, &opt_i,
                    'a', MDB_OPT_SETBITS, 1, &opt_a,
                    'b', MDB_OPT_SETBITS, 1, &opt_b,
                    'c', MDB_OPT_UINT64, &opt_c,
                    'x', MDB_OPT_UINT64, &opt_x,
                    'n', MDB_OPT_UINT64, &opt_n,
                    NULL) != argc)
    {
        return DCMD_USAGE;
    }

    if ((opt_i != OPT_NONE) && (opt_i != um.instance))
    {
        return DCMD_OK;
    }

    if (opt_a && (opt_b ||
                  (opt_c != OPT_NONE) ||
                  (opt_x != OPT_NONE) ||
                  (opt_n != OPT_NONE)))
    {
        return DCMD_USAGE;
    }

    if (((opt_x != OPT_NONE) && (opt_n == OPT_NONE)) ||
        ((opt_x == OPT_NONE) && (opt_n != OPT_NONE)))
    {
        return DCMD_USAGE;
    }

    if (!(opt_a ||
          opt_b ||
          (opt_c != OPT_NONE) ||
          (opt_x != OPT_NONE) ||
          (opt_n != OPT_NONE)))
    {
        opt_b = 1;
    }

    mdb_printf("--------------------------------------------------\n");
    mdb_printf("\n");
    mdb_printf("%s -> %p (%p)\n", um.dev_name, addr, dip.devi_driver_data);
    mdb_printf("\n");

    if (opt_a || opt_b)
    {
        BnxMdbBasic(&um);
        mdb_printf("\n");

        //BnxMdbSb(&um, DEF_STATUS_BLOCK_INDEX);
        //mdb_printf("\n");
    }

    if (opt_a || (opt_c != OPT_NONE))
    {
        if (opt_a || (opt_c == (uint64_t)-1))
        {
            for (idx = 0; idx < um.lm_dev.tx_info.num_txq; idx++)
            {
                BnxMdbChain(&um, idx);
                mdb_printf("\n");
            }
        }
        else
        {
            if (opt_c < um.lm_dev.tx_info.num_txq)
            {
                BnxMdbChain(&um, opt_c);
            }
            else
            {
                mdb_printf("ERROR: Invalid chain index (%d)\n", opt_c);
            }

            mdb_printf("\n");
        }
    }

    if (opt_x != OPT_NONE)
    {
        if (opt_x & 0x7f)
        {
            mdb_printf("ERROR: Invalid CID address alignment\n");
        }
        else
        {
            dump_ctx_as_dword(&um, opt_x, opt_n);
        }

        mdb_printf("\n");
    }

    mdb_printf("--------------------------------------------------\n");

#if 0
{
  int       m_argc   = 2;
  mdb_arg_t m_argv[] =
    {
      { MDB_TYPE_STRING, "bnx`um_device_t" },
      { MDB_TYPE_STRING, "magic" }
    };
  mdb_call_dcmd("print",
                (uintptr_t)dip.devi_driver_data,
                DCMD_ADDRSPEC,
                m_argc,
                m_argv);
}
#endif

    return DCMD_OK;
}


static const mdb_dcmd_t bnx_dcmds[] =
{
    { "bnx", "?[ -i # ] [ -a | -b | -c # | -x # -n # ]", "bnx driver status", BnxMdb, BnxMdbHelp },
    { NULL }
};


static const mdb_walker_t bnx_walkers[] =
{
    NULL
};


static mdb_modinfo_t bnx_mdb =
{
    MDB_API_VERSION,
    bnx_dcmds,
    bnx_walkers
};


mdb_modinfo_t * _mdb_init(void)
{
    return &bnx_mdb;
}


void _mdb_fini(void)
{
    return;
}

