/****************************************************************************
 * Copyright(c) 2005 Broadcom Corporation, all rights reserved
 * Proprietary and Confidential Information.
 *
 * This source file is the property of Broadcom Corporation, and
 * may not be copied or distributed in any isomorphic form without
 * the prior written consent of Broadcom Corporation.
 *
 * Name:        pal.c
 *
 * Description: Platform Abstraction Layer implementation for bxbdrv(mpd)
 *
 * Author:      John Chen
 *
 *
 ****************************************************************************/
#if defined(CONFIG_SMP) && !defined(__SMP__)
#define __SMP__
#endif  // defined(CONFIG_SMP) && !defined(__SMP__)

#include <linux/version.h>

// only one C file can include module headers with __NO_VERSION__
#if defined(MODULE)
#define __NO_VERSION__
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#define MODVERSIONS
#endif  // defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#if defined(MODVERSIONS) && (LINUX_VERSION_CODE < 0x020500)
#include <linux/modversions.h>
#endif  // defined(CONFIG_MODVERSIONS)
#include <linux/module.h>
#endif  // defined(MODULE)

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>        // ???
#include <linux/ptrace.h>           // ???
#include <linux/ioport.h>           // ???
#include <linux/in.h>
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/slab.h>
#else
#include <linux/malloc.h>
#endif
#include <linux/string.h>
#include <linux/init.h>
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3,4,0))
#include <linux/bitops.h>
#include <linux/io.h>               // ???
#include <asm-generic/dma.h>        // ???
#include <linux/uaccess.h>          // copy_[from|to]_user
#include <linux/atomic.h>
#else
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>                 // ???
#include <asm/dma.h>                // ???
#include <asm/uaccess.h>            // copy_[from|to]_user
#include <asm/atomic.h>
#endif
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/inetdevice.h>
#include <linux/skbuff.h>
#include <linux/notifier.h>
#include <linux/string.h>
#include <linux/rtnetlink.h>
#include <linux/proc_fs.h>
#if (LINUX_VERSION_CODE < KERNEL_VERSION(3,0,0))
#include <linux/smp_lock.h>
#endif
#include <linux/vmalloc.h>

// version dependent includes
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#include <linux/spinlock.h>
#else
#include <asm/spinlock.h>
#endif

#include <linux/pci.h>
#include <linux/timer.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/if_vlan.h>

#ifdef NETIF_F_TSO
#include <net/ip.h>
#include <net/tcp.h>
#include <net/checksum.h>
#endif

#include "bdrv_ver.h"
#include "pal.h"
#include "pal2.h"

void
bcm_get_dev(
    void **dev_arr,
    u32_t arr_size,
    int *dev_num)
{
    bcm06_get_dev(dev_arr, arr_size, dev_num);
}

int
bcm_get_interface(
    void *bcm_dev,
    u32_t if_type,
    void *interface,
    u32_t if_size,
    u32_t if_ver)
{
    return bcm06_get_interface(bcm_dev, if_type, interface, if_size, if_ver);
}

EXPORT_SYMBOL(bcm_get_dev);
EXPORT_SYMBOL(bcm_get_interface);

static inline unsigned int pal_jiffies_to_msecs(const unsigned long j)
{
#if HZ <= 1000 && !(1000 % HZ)
        return (1000 / HZ) * j;
#elif HZ > 1000 && !(HZ % 1000)
         return (j + (HZ / 1000) - 1)/(HZ / 1000);
#else
         return (j * 1000) / HZ;
#endif
}

//
// linux kernel version check
//
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#undef KERNEL_2_2
//#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,2,0)
//#define KERNEL_2_2 1
#else
#error bxbdrv or related device driver requires at least Linux 2.4.0 kernel
#endif

//
// abstracted types
//
#if defined(KERNEL_2_2)
typedef struct device netdevice_t;
typedef struct wait_queue* event_t;
#else
typedef struct net_device netdevice_t;
typedef wait_queue_head_t event_t;
#endif // defined(KERNEL_2_2)

//
// compatibility macros
//
#if defined(KERNEL_2_2)
#define init_waitqueue_head(ev)     *(ev) = NULL
#define dev_get_by_name(dev)        dev_get(dev)
#endif // defined(KERNEL_2_2)

#ifndef IRQ_RETVAL
typedef void irqreturn_t;
#define IRQ_RETVAL(x)
#endif

//
// Resource Pool
//

#define PALI_NUM_LOCK                       30
#define PALI_NUM_ATOMIC                     8
#define PALI_NUM_TIMERS                     4
#define PALI_NUM_EVENTS                     6
#define PALI_NUM_SK_BUF_HEAD                2

typedef struct _PALI_RESOURCE_POOL
{
    U32 rp_signature;                       // id for debugging purpose
    struct _PALI_RESOURCE_POOL* rp_next;    // singly link-list for recycle
    spinlock_t              f_spinlock[PALI_NUM_LOCK];
    atomic_t                f_atomic[PALI_NUM_ATOMIC];
    struct timer_list       f_timers[PALI_NUM_TIMERS];
    event_t                 f_event[PALI_NUM_EVENTS];
    struct semaphore        f_mutex;
    struct tasklet_struct   f_tasklet;
    struct net_device_stats f_stat;
    struct sk_buff_head     f_head[PALI_NUM_SK_BUF_HEAD];
} PALI_RESOURCE_POOL;

//
// global variables
//
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39)) 
static spinlock_t print_lock = SPIN_LOCK_UNLOCKED;
#else
static DEFINE_SPINLOCK(print_lock);
#endif
static char msg[4096];

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
U8 *
pal_alloc_resource(
    void
    )
{
    PALI_RESOURCE_POOL* rp;

    rp = (PALI_RESOURCE_POOL*)kmalloc( sizeof(*rp), GFP_ATOMIC );
    if(rp)
    {
        memset( rp, 0, sizeof(*rp) );

        // initialize
        rp->rp_signature = 0x414C5000;
    }

    return (U8 *)rp;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8 *
pal_get_resource_addr(
    U8 *base,
    PAL_RESOURCE_TYPE type,
    U32 idx)
{
    U8 *addr = NULL;
    PALI_RESOURCE_POOL *rp = (PALI_RESOURCE_POOL *)base;

    switch (type)
    {
    case PRT_SPINLOCK:
        if (idx >= PALI_NUM_LOCK)
        {
            printk(KERN_ERR "Invalid spin lock index");
            break;
        }
        addr = (U8 *)&rp->f_spinlock[idx];
        break;
    case PRT_EVENT:
        if (idx >= PALI_NUM_EVENTS)
        {
            printk(KERN_ERR "Invalid event index");
            break;
        }
        addr = (U8 *)&rp->f_event[idx];
        break;
    case PRT_ATOMIC:
        if (idx >= PALI_NUM_ATOMIC)
        {
            printk(KERN_ERR "Invalid event index");
            break;
        }
        addr = (U8 *)&rp->f_atomic[idx];
        break;
    case PRT_TIMER:
        if (idx >= PALI_NUM_TIMERS)
        {
            printk(KERN_ERR "Invalid PRT_TIMER index");
            break;
        }
        addr = (U8 *)&rp->f_timers[idx];
        break;
    case PRT_MUTEX:
        addr = (U8 *)&rp->f_mutex;
        break;
    case PRT_TASKLET:
        addr = (U8 *)&rp->f_tasklet;
        break;
    case PRT_DEVICE_STAT:
        addr = (U8 *)&rp->f_stat;
        break;
    default:
        printk(KERN_ERR "Invalid pal resource type");
        break;
    }
    pal_break_if(0 == addr);
    return addr;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_free_resource(
    U8* base
    )
{
    PALI_RESOURCE_POOL* rp = (PALI_RESOURCE_POOL*)base;

    if( rp == NULL || rp->rp_signature != 0x414C5000 )
    {
        printk( KERN_ERR "Try to free invalid pal resource");
        return -EINVAL;
    }   // if( rp == NULL )

    kfree( rp );

    return 0;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void *
pal_ioremap(
    unsigned long phy_addr,
    unsigned long size
    )
{
    return ioremap(phy_addr, size);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_iounmap(
    void *virt_addr
    )
{
    iounmap(virt_addr);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_init_MUTEX(
    PAL_MUTEX mutex
    )
{
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,37))
    init_MUTEX((struct semaphore *)mutex);
#else
	sema_init((struct semaphore *)mutex, 1);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
irqreturn_t
mpd_isr_handle(
    int irq,
    void *dev_instance,
    struct pt_regs *regs
    )
{
    int irq_ret;

    irq_ret = um_bdrv_handle_isr(dev_instance);
    return IRQ_RETVAL(irq_ret);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_request_irq(
    u32_t irq,
    char *name,
    void *data
    )
{
#if (LINUX_VERSION_CODE < 0x020618) /* kernel 2.6.24 */
    return request_irq(irq, mpd_isr_handle, SA_SHIRQ, name, data);
#else
    return request_irq(irq, (irq_handler_t)mpd_isr_handle, IRQF_SHARED, name, data);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
long
pal_wait_event(
    PAL_EVENT event,
    signed long ms
    )
{
    if( interruptible_sleep_on_timeout( (event_t*)event, ms ) == 0 ) {
        return 0;
    }
    else {
        return 1;
    }
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
/*#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_event(
    PAL_EVENT* event
    )
{
    lock_kernel();
    wake_up( (event_t*)event );
    unlock_kernel();
}*/

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_init_event(
    PAL_EVENT event
    )
{
    init_waitqueue_head( (event_t*)event );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
unsigned long
pal_get_current_time(
    void
    )
{
    return jiffies;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_diff_ms(
    unsigned long tstart,
    unsigned long tend
    )
{
    if( tend > tstart )
    {
        return (pal_jiffies_to_msecs(tend) - pal_jiffies_to_msecs(tstart));
    }
    else
    {
        return ((U32)(-1) - pal_jiffies_to_msecs(tstart) + pal_jiffies_to_msecs(tend));
    }
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned long
pal_advance_second(
    unsigned long tstart,
    unsigned long second
    )
{
    return tstart + (unsigned long)(second) * HZ;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void*
pal_memcpy(
    void* dst,
    void* src,
    U32 len
    )
{
    return memcpy( dst, src, len );
}


/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_memcmp(
    void* dst,
    void* src,
    U32 len
    )
{
    return memcmp( dst, src, len );
}


/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void*
pal_memset(
    void* dst,
    int val,
    U32 len
    )
{
    return memset( dst, val, len );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void*
pal_malloc(
    U32 len
    )
{
    return kmalloc( len, GFP_ATOMIC );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_mfree(
    void* p
    )
{
    kfree( p );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_copy_from_user(
    void* to,
    void* from,
    U32 len
    )
{
	unsigned long ret;

    ret = copy_from_user( to, from, len ); /* On success, return zero */
	if (ret != 0)
		printk("copy_from_user failed\n");    
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_copy_to_user(
    void* to,
    void* from,
    U32 len
    )
{	
	unsigned long ret;

    ret = copy_to_user( to, from, len ); /* On success, return zero */
	if (ret != 0)
		printk("copy_to_user failed\n");    

}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_stricmp(
    char* str1,
    char* str2
    )
{
    int rc;

    rc = strlen( str1 ) - strlen( str2 );
    if( rc == 0 )
    {
        rc = strcmp( str1, str2 );
    }

    return rc;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_atomic_read(
    PAL_ATOMIC atom
    )
{
    return atomic_read( (atomic_t*)atom );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_atomic_set(
    PAL_ATOMIC atom,
    int val
    )
{
    atomic_set( (atomic_t*)atom, val );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_atomic_inc(
    PAL_ATOMIC atom
    )
{
    atomic_inc( (atomic_t*)atom );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_atomic_dec(
    PAL_ATOMIC atom
    )
{
    atomic_dec( (atomic_t*)atom );
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_atomic_dec_and_test(
    PAL_ATOMIC atom
    )
{
    return atomic_dec_and_test( (atomic_t*)atom );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_atomic_add(
    PAL_ATOMIC atom,
    int val
    )
{
    atomic_add( val, (atomic_t*)atom );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
PAL_PACKET
pal_alloc_packet(
    U32 size,
    PAL_DEVICE dev
    )
{
    struct sk_buff* skb;
    skb = dev_alloc_skb( size );
    if (skb == 0)
        return (PAL_PACKET) skb;
    skb->dev = (netdevice_t*)dev;
    return (PAL_PACKET)skb;
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_packet_reserve(
    PAL_PACKET skb,
    unsigned int len
    )
{
    skb_reserve((struct sk_buff*)skb, len);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_free_packet(
    PAL_PACKET pkt
    )
{
    dev_kfree_skb_any( (struct sk_buff*)pkt );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_init_packet_list(
    PAL_PACKET_LIST list
    )
{
    skb_queue_head_init( (struct sk_buff_head*)list );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_enqueue_packet(
    PAL_PACKET_LIST list,
    PAL_PACKET pkt
    )
{
    skb_queue_head( (struct sk_buff_head*)list, (struct sk_buff*)pkt );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
PAL_PACKET
pal_dequeue_packet(
    PAL_PACKET_LIST list
    )
{
    struct sk_buff* skb;

    skb = skb_dequeue( (struct sk_buff_head*)list );
    return (PAL_PACKET)skb;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_packet_device(
    PAL_DEVICE dev,
    PAL_PACKET pkt
    )
{
    ((struct sk_buff*)pkt)->dev = (netdevice_t*)dev;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
PAL_DEVICE
pal_get_packet_device(
    PAL_PACKET pkt
    )
{
    return (PAL_DEVICE)(((struct sk_buff*)pkt)->dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_packet_protocol(
    PAL_DEVICE dev,
    PAL_PACKET pkt
    )
{
    ((struct sk_buff*)pkt)->protocol = eth_type_trans((struct sk_buff*)pkt, (netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_update_dev_time(
    PAL_DEVICE dev,
    u32_t tx
    )
{
    if (tx)
        ((netdevice_t*)dev)->trans_start = jiffies;
    else
        ((netdevice_t*)dev)->last_rx = jiffies;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_send_packet(
    PAL_PACKET pkt
    )
{
    dev_queue_xmit( (struct sk_buff*)pkt );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8*
pal_get_packet_reserved(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->cb;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_indicate_receive_packet(
    PAL_PACKET pkt
    )
{
    netif_rx( (struct sk_buff*)pkt );
}


/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_indicate_receive_vlan_tag(
    PAL_PACKET pkt,
    void *vlgrp,
    unsigned short vlan_tag
    )
{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3,1,0))
    return netif_rx( (struct sk_buff*)pkt );
#else
    return vlan_hwaccel_rx((struct sk_buff*)pkt,
        (struct vlan_group *)vlgrp,
        vlan_tag);
#endif
}


/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_packet_len(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->len;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8*
pal_get_packet_data(
    PAL_PACKET pkt
    )
{
    return ((struct sk_buff*)pkt)->data;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8*
pal_get_packet_raw(
    PAL_PACKET pkt
    )
{
#if (LINUX_VERSION_CODE < 0x020616)
    return ((struct sk_buff*)pkt)->mac.raw;
#else
    return NULL; /*unused code in lx_diag */
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8*
pal_packet_put(
    PAL_PACKET pkt,
    unsigned int len
    )
{
    return skb_put( (struct sk_buff*)pkt, len );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_packet_set_ip_summed(
    PAL_PACKET pkt,
    U32 hw_chksum
    )
{
    ((struct sk_buff*)pkt)->ip_summed = hw_chksum ? CHECKSUM_UNNECESSARY : CHECKSUM_NONE;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_packet_do_hw_cksum(
    PAL_PACKET pkt
    )
{
    int rc = 0;
    if ((((struct sk_buff*)pkt)->ip_summed != CHECKSUM_NONE) &&
        (((struct sk_buff*)pkt)->ip_summed != CHECKSUM_UNNECESSARY))
        rc = 1;
    return rc;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_vlan_tx_tag_present(
    PAL_PACKET pkt
    )
{
    return vlan_tx_tag_present((struct sk_buff*)pkt);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U16
pal_vlan_tx_tag_get(
    PAL_PACKET pkt
    )
{
    return (U16)vlan_tx_tag_get((struct sk_buff*)pkt);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_packet_frags_num(
    PAL_PACKET pkt
    )
{
    return skb_shinfo((struct sk_buff*)pkt)->nr_frags;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_packet_frag_len(
    PAL_PACKET pkt,
    U32 idx
    )
{
    return skb_shinfo((struct sk_buff*)pkt)->frags[idx].size;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_packet_headlen(
    PAL_PACKET pkt
    )
{
    return skb_headlen((struct sk_buff*)pkt);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_packet_mss(
    PAL_PACKET pkt
    )
{
#ifdef NETIF_F_GSO
    return skb_shinfo((struct sk_buff*)pkt)->gso_size;
#else
    return skb_shinfo((struct sk_buff*)pkt)->tso_size;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_packet_lso(
    PAL_PACKET pkt,
    u8_t *ip_hdr_len,
    u8_t *tcp_hdr_len,
    u32_t mss
    )
{
#ifdef NETIF_F_TSO
#if (LINUX_VERSION_CODE < 0x020616)
    /*unused code in lx_diag */
    u32_t tcp_opt_len, ip_tcp_len;
    struct sk_buff *skb = (struct sk_buff*)pkt;

    tcp_opt_len = 0;
    if (skb->h.th->doff > 5) {
        tcp_opt_len = (skb->h.th->doff - 5) << 2;
    }
    ip_tcp_len = (skb->nh.iph->ihl << 2) + sizeof(struct tcphdr);

    skb->nh.iph->check = 0;
    skb->nh.iph->tot_len = ntohs(mss + ip_tcp_len + tcp_opt_len);
    skb->h.th->check =
        ~csum_tcpudp_magic(skb->nh.iph->saddr,
        skb->nh.iph->daddr,
        0, IPPROTO_TCP, 0);

    *ip_hdr_len = (u8_t)skb->nh.iph->ihl << 2;
    *tcp_hdr_len = (u8_t)skb->h.th->doff << 2;
#endif
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_module_get(void)
{
#if (LINUX_VERSION_CODE >= 0x20600)
    return try_module_get(THIS_MODULE);
#else
    return 1;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_module_put(void)
{
#if (LINUX_VERSION_CODE >= 0x20600)
    module_put(THIS_MODULE);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_init_spinlock(
    PAL_SPINLOCK lock
    )
{
    spin_lock_init( (spinlock_t*)lock );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_spin_lock_irqsave(
    PAL_SPINLOCK lock,
    unsigned long* flags
    )
{
    spin_lock_irqsave( (spinlock_t*)lock, *flags );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_spin_lock(
    PAL_SPINLOCK lock
    )
{
    spin_lock((spinlock_t*)lock);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_spin_unlock_irqrestore(
    PAL_SPINLOCK lock,
    unsigned long* flags
    )
{
    spin_unlock_irqrestore( (spinlock_t*)lock, *flags );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_spin_unlock(
    PAL_SPINLOCK lock
    )
{
    spin_unlock((spinlock_t*)lock);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_local_irq_save(
    unsigned long *flags
    )
{
    local_irq_save(*flags);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_local_irq_restore(
    unsigned long *flags
    )
{
    local_irq_restore(*flags);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_spin_is_locked(
    PAL_SPINLOCK lock
    )
{
    return spin_is_locked((spinlock_t*)lock);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8 *
pal_alloc_tcp_con_locks(
    u32_t *lock_size
    )
{
    U8 *locks;
    locks = kmalloc((2 * sizeof(spinlock_t)), GFP_ATOMIC );
    if (locks)
    {
        memset(locks, 0 , (2 * sizeof(spinlock_t)));
        *lock_size = 2 * sizeof(spinlock_t);
    }
    return locks;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_print(
    const char* fmt,
    ...
    )
{
    va_list ap;
    unsigned long flags;

    va_start( ap, fmt );
    spin_lock_irqsave( &print_lock, flags );
    vsprintf( msg, fmt, ap );
    printk( "%s", msg );
    spin_unlock_irqrestore( &print_lock, flags );
    va_end( ap );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_panic(
    const char* msg
    )
{
    panic( msg );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_sprintf(
    char* buf,
    const char* fmt,
    ...
    )
{
    int len;
    va_list ap;
    va_start( ap, fmt );
    len = vsprintf( buf, fmt, ap );
    va_end( ap );
    return len;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_is_device_up(
    PAL_DEVICE dev
    )
{
    if( ( ((netdevice_t*)dev)->flags & IFF_UP ) ==
        IFF_UP )
    {
        return 1;
    }
    else
    {
        return 0;
    }
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
char*
pal_get_device_name(
    PAL_DEVICE dev
    )
{
    return ((netdevice_t*)dev)->name;
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
char*
pal_get_if_name(
    PAL_DEVICE dev
    )
{
    return ((struct in_ifaddr*)dev)->ifa_label;
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U32
pal_get_ip_addr(
    PAL_DEVICE dev
    )
{
    return ((struct in_ifaddr*)dev)->ifa_address;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_device_name(
    PAL_DEVICE dev,
    char* name
    )
{
#if defined(KERNEL_2_2)
    ((netdevice_t*)dev)->name = name;
#else
    strncpy( ((netdevice_t*)dev)->name, name, IFNAMSIZ-1 );
#endif // defined(KERNEL_2_2)
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
PAL_DEVICE
pal_get_device_by_name(
    char* name
    )
{
#if (LINUX_VERSION_CODE < 0x020618)
    return (PAL_DEVICE)dev_get_by_name( name );
#else
    return NULL; /*unused code in lx_diag */
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_accumulate_device_stat(
    volatile teton_stat_block_t *stats_blk,
    PAL_DEVICE_STAT stat_in
    )
{
    struct net_device_stats* net_stats = (struct net_device_stats*)stat_in;

    if(stats_blk == NULL)
    {
        return;
    }
    net_stats->rx_packets =
        GET_NET_STATS(stats_blk->stat_IfHCInUcastPkts) +
        GET_NET_STATS(stats_blk->stat_IfHCInMulticastPkts) +
        GET_NET_STATS(stats_blk->stat_IfHCInBroadcastPkts);

    net_stats->tx_packets =
        GET_NET_STATS(stats_blk->stat_IfHCOutUcastPkts) +
        GET_NET_STATS(stats_blk->stat_IfHCOutMulticastPkts) +
        GET_NET_STATS(stats_blk->stat_IfHCOutBroadcastPkts);

    net_stats->rx_bytes =
        GET_NET_STATS(stats_blk->stat_IfHCInOctets);

    net_stats->tx_bytes =
        GET_NET_STATS(stats_blk->stat_IfHCOutOctets);

    net_stats->multicast =
        GET_NET_STATS(stats_blk->stat_IfHCOutMulticastPkts);

    net_stats->collisions =
        (unsigned long) stats_blk->stat_EtherStatsCollisions;

    net_stats->rx_length_errors =
        (unsigned long) (stats_blk->stat_EtherStatsUndersizePkts +
        stats_blk->stat_EtherStatsOverrsizePkts);

    net_stats->rx_over_errors =
        (unsigned long) stats_blk->stat_IfInMBUFDiscards;

    net_stats->rx_frame_errors =
        (unsigned long) stats_blk->stat_Dot3StatsAlignmentErrors;

    net_stats->rx_crc_errors =
        (unsigned long) stats_blk->stat_Dot3StatsFCSErrors;

    net_stats->rx_errors = net_stats->rx_length_errors +
        net_stats->rx_over_errors + net_stats->rx_frame_errors +
        net_stats->rx_crc_errors;

    net_stats->tx_aborted_errors =
            (unsigned long) (stats_blk->stat_Dot3StatsExcessiveCollisions +
        stats_blk->stat_Dot3StatsLateCollisions);

    net_stats->tx_carrier_errors =
        (unsigned long) stats_blk->stat_Dot3StatsCarrierSenseErrors;

    net_stats->tx_errors =
            (unsigned long)
        stats_blk->stat_emac_tx_stat_dot3statsinternalmactransmiterrors
        +
        net_stats->tx_aborted_errors +
        net_stats->tx_carrier_errors;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void*
pal_get_device_private(
    PAL_DEVICE dev
    )
{

#if (LINUX_VERSION_CODE >= 0x02061f) /*kernel 2.6.31*/

	unsigned short private = 0;
    	private = ((netdevice_t*)dev)->priv_flags;
	#if defined(CONFIG_X86_32)
		return (void *) ((u32_t)private);
	#else
		return (void *) ((u64_t)private);
	#endif

#else
    return ((netdevice_t*)dev)->priv;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_device_mac_address(
    PAL_DEVICE dev,
    U8* mac_addr
    )
{
    memcpy( ((netdevice_t*)dev)->dev_addr, mac_addr, 6 );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
U8*
pal_get_device_mac_address(
    PAL_DEVICE dev
    )
{
    return &((netdevice_t*)dev)->dev_addr[0];
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_get_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
    u32_t setting, advertising, autoneg;
    u32_t speed, duplex, meduim_type;

    mpd_get_settings((PAL_DEVICE)dev, &setting, &advertising, &autoneg);

    cmd->supported = SUPPORTED_Autoneg;
    meduim_type = GET_MEDIUM_TYPE(setting);
    if (LM_MEDIUM_TYPE_FIBER == meduim_type) {
        cmd->supported |= SUPPORTED_1000baseT_Full |
            SUPPORTED_FIBRE;

        cmd->port = PORT_FIBRE;
    }
    else {
        cmd->supported |= SUPPORTED_10baseT_Half |
            SUPPORTED_10baseT_Full |
            SUPPORTED_100baseT_Half |
            SUPPORTED_100baseT_Full |
            SUPPORTED_1000baseT_Full |
            SUPPORTED_TP;

        cmd->port = (LM_MEDIUM_TYPE_BNC == meduim_type) ? PORT_BNC : PORT_TP;
    }

    cmd->advertising = advertising;
    cmd->autoneg = autoneg ? AUTONEG_ENABLE : AUTONEG_DISABLE;

    if (netif_carrier_ok(dev)) {
        speed = GET_MEDIUM_SPEED(setting);
        switch (speed) {
        case LM_MEDIUM_SPEED_10MBPS:
            cmd->speed = SPEED_10;
            break;
        case LM_MEDIUM_SPEED_100MBPS:
            cmd->speed = SPEED_100;
            break;
        case LM_MEDIUM_SPEED_1000MBPS:
            cmd->speed = SPEED_1000;
            break;
        default:
            cmd->speed = 0;
            break;
        }
        duplex = GET_MEDIUM_DUPLEX(setting);
        cmd->duplex = (LM_MEDIUM_HALF_DUPLEX == duplex) ? DUPLEX_HALF : DUPLEX_FULL;
    }
    else {
        cmd->speed = -1;
        cmd->duplex = -1;
    }

    cmd->transceiver = XCVR_INTERNAL;
    cmd->phy_address = 1;

    return 0;
}

#define ETHTOOL_ALL_FIBRE_SPEED						\
	(ADVERTISED_1000baseT_Full)

#define ETHTOOL_ALL_COPPER_SPEED					\
	(ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full |		\
	ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full |		\
	ADVERTISED_1000baseT_Full)

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_adv_init(u32_t fiber, u32_t *advertising)
{
    if (fiber)
        *advertising = ETHTOOL_ALL_FIBRE_SPEED | ADVERTISED_Autoneg;
    else
        *advertising = ETHTOOL_ALL_COPPER_SPEED | ADVERTISED_Autoneg;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_settings(struct net_device *dev, struct ethtool_cmd *cmd)
{
    u32_t setting, advertising, autoneg;
    u32_t req_medium = LM_MEDIUM_AUTO_DETECT;
    u32_t meduim_type;
    u8_t selective_autoneg = 0;

    mpd_get_settings((PAL_DEVICE)dev, &setting, &advertising, &autoneg);
    meduim_type = GET_MEDIUM_TYPE(setting);

    if (cmd->autoneg == AUTONEG_ENABLE) {
        autoneg = 1;

        cmd->advertising &= ETHTOOL_ALL_COPPER_SPEED;

        /* allow advertising 1 speed */
        if ((cmd->advertising == ADVERTISED_10baseT_Half) ||
            (cmd->advertising == ADVERTISED_10baseT_Full) ||
            (cmd->advertising == ADVERTISED_100baseT_Half) ||
            (cmd->advertising == ADVERTISED_100baseT_Full)) {

            if (LM_MEDIUM_TYPE_FIBER == meduim_type)
                return -EINVAL;

            if (cmd->advertising == ADVERTISED_10baseT_Half) {
                SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_10MBPS);
                SET_MEDIUM_DUPLEX(req_medium, LM_MEDIUM_HALF_DUPLEX);
            }
            else if (cmd->advertising == ADVERTISED_10baseT_Full) {
                SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_10MBPS);
            }
            else if (cmd->advertising == ADVERTISED_100baseT_Half) {
                SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_100MBPS);
                SET_MEDIUM_DUPLEX(req_medium, LM_MEDIUM_HALF_DUPLEX);
            }
            else if (cmd->advertising == ADVERTISED_100baseT_Full) {
                SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_100MBPS);
            }
            selective_autoneg = 1;
            advertising = cmd->advertising;
        }
        else if (cmd->advertising == ADVERTISED_1000baseT_Full) {
            advertising = cmd->advertising;
        }
        else if (cmd->advertising == ADVERTISED_1000baseT_Half) {
            return -EINVAL;
        }
        else {
            if (LM_MEDIUM_TYPE_FIBER == meduim_type) {
                advertising = ETHTOOL_ALL_FIBRE_SPEED;
            }
            else {
                advertising = ETHTOOL_ALL_COPPER_SPEED;
            }
        }
        advertising |= ADVERTISED_Autoneg;
    }
    else {
        if ((meduim_type != LM_MEDIUM_TYPE_FIBER) &&
            (cmd->speed == SPEED_1000)) {
            return -EINVAL;
        }
        if ((cmd->speed == SPEED_1000) &&
            (DUPLEX_HALF == cmd->duplex))
        {
            return -EINVAL;
        }
        autoneg = 0;
        req_medium = meduim_type;
        switch (cmd->speed) {
        case SPEED_10:
            SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_10MBPS);
            break;
        case SPEED_100:
            SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_100MBPS);
            break;
        case SPEED_1000:
            SET_MEDIUM_SPEED(req_medium, LM_MEDIUM_SPEED_1000MBPS);
            break;
        }
        if (DUPLEX_HALF == cmd->duplex)
            SET_MEDIUM_DUPLEX(req_medium, LM_MEDIUM_HALF_DUPLEX);

        advertising = 0;
    }
    mpd_set_settings((PAL_DEVICE)dev, req_medium, advertising, autoneg, selective_autoneg);
    return 0;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_drvinfo(struct net_device *dev, struct ethtool_drvinfo *info)
{
    char *bus_info;
    u32_t fw_ver;

    strcpy(info->driver, DRV_MODULE_NAME);
    strcpy(info->version, DRV_MODULE_VERSION);
    mpd_get_drvinfo((PAL_DEVICE)dev, &bus_info, &fw_ver);

    strcpy(info->bus_info, bus_info);
    info->fw_version[0] = ((fw_ver & 0xff000000) >> 24) + '0';
    info->fw_version[2] = ((fw_ver & 0xff0000) >> 16) + '0';
    info->fw_version[4] = ((fw_ver & 0xff00) >> 8) + '0';
    info->fw_version[6] = (fw_ver & 0xff) + '0';
    info->fw_version[1] = info->fw_version[3] = info->fw_version[5] = '.';
    info->fw_version[7] = 0;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_wol(struct net_device *dev, struct ethtool_wolinfo *wol)
{
    u32_t wake_magic, enable;

    mpd_get_wake_magic((PAL_DEVICE)dev, &wake_magic, &enable);
    if (0 == wake_magic) {
        wol->supported = 0;
        wol->wolopts = 0;
    }
    else {
        wol->supported = WAKE_MAGIC;
        if (enable)
            wol->wolopts = WAKE_MAGIC;
        else
            wol->wolopts = 0;
    }
    memset(&wol->sopass, 0, sizeof(wol->sopass));
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_wol(struct net_device *dev, struct ethtool_wolinfo *wol)
{
    u32_t enable = 0;

    if (wol->wolopts & WAKE_MAGIC)
        //Set WOL mode
        enable = 1;

    return mpd_set_wol((PAL_DEVICE)dev, enable);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_nway_reset(struct net_device *dev)
{
    return mpd_nway_reset((PAL_DEVICE)dev);
}

#if (LINUX_VERSION_CODE >= 0x20418)
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_get_eeprom_len(struct net_device *dev)
{
    return mpd_get_eeprom_len((PAL_DEVICE)dev);
}
#endif

#ifdef ETHTOOL_GEEPROM
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_get_eeprom(struct net_device *dev, struct ethtool_eeprom *eeprom,
        u8_t *eebuf)
{
    return mpd_get_eeprom((PAL_DEVICE)dev,
                eeprom->offset,
                eeprom->len,
                eebuf);
}

#endif

#ifdef ETHTOOL_SEEPROM
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_eeprom(struct net_device *dev, struct ethtool_eeprom *eeprom,
        u8_t *eebuf)
{
    return mpd_set_eeprom((PAL_DEVICE)dev,
                eeprom->offset,
                eeprom->len,
                eebuf);

}
#endif

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_get_coalesce(struct net_device *dev, struct ethtool_coalesce *coal)
{
    memset(coal, 0, sizeof(struct ethtool_coalesce));

    return mpd_get_coalesce((PAL_DEVICE)dev,
                &coal->rx_coalesce_usecs, &coal->rx_max_coalesced_frames,
                &coal->rx_coalesce_usecs_irq, &coal->rx_max_coalesced_frames_irq,
                &coal->tx_coalesce_usecs, &coal->tx_max_coalesced_frames,
                &coal->tx_coalesce_usecs_irq, &coal->tx_max_coalesced_frames_irq,
                &coal->stats_block_coalesce_usecs);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_coalesce(struct net_device *dev, struct ethtool_coalesce *coal)
{
    return -EINVAL;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_ringparam(struct net_device *dev, struct ethtool_ringparam *ering)
{
    u32_t rx_max_pending, tx_max_pending;

    mpd_get_ringparam((PAL_DEVICE)dev, &rx_max_pending, &tx_max_pending);

    ((struct ethtool_ringparam *)ering)->rx_max_pending = rx_max_pending;
    ((struct ethtool_ringparam *)ering)->rx_mini_max_pending = 0;
    ((struct ethtool_ringparam *)ering)->rx_jumbo_max_pending = 0;

    ((struct ethtool_ringparam *)ering)->rx_pending = 0;
    ((struct ethtool_ringparam *)ering)->rx_mini_pending = 0;
    ((struct ethtool_ringparam *)ering)->rx_jumbo_pending = 0;

    ((struct ethtool_ringparam *)ering)->tx_max_pending = tx_max_pending;
    ((struct ethtool_ringparam *)ering)->tx_pending = 0;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_ringparam(struct net_device *dev, struct ethtool_ringparam *ering)
{
    return -EINVAL;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_pauseparam(struct net_device *dev, struct ethtool_pauseparam *epause)
{
    u32_t flow_autoneg, rx_pause, tx_pause;

    mpd_get_pauseparam((PAL_DEVICE)dev, &flow_autoneg, &rx_pause, &tx_pause);

    epause->autoneg = flow_autoneg;
    epause->rx_pause = rx_pause;
    epause->tx_pause = tx_pause;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_pauseparam(struct net_device *dev, struct ethtool_pauseparam *epause)
{
    return mpd_set_pauseparam((PAL_DEVICE)dev,
                epause->autoneg,
                epause->rx_pause,
                epause->tx_pause);
}

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39))
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static u32_t
pal_get_rx_csum(struct net_device *dev)
{
    return mpd_get_rx_csum((PAL_DEVICE)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_set_rx_csum(struct net_device *dev, u32_t data)
{
    return mpd_set_rx_csum((PAL_DEVICE)dev, data);
}
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39)) */

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_strings(struct net_device *dev, u32_t stringset, u8_t *buf)
{
    switch (stringset) {
    case ETH_SS_STATS:
        mpd_get_strings((PAL_DEVICE)dev, buf);
        break;
    }
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static int
pal_phys_id(struct net_device *dev, u32_t data)
{
    return mpd_phys_id((PAL_DEVICE)dev, data);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if (LINUX_VERSION_CODE < 0x020618) /* removed in kernel 2.6.24 */
static int
pal_get_stats_count(struct net_device *dev)
{
    return mpd_get_stats_count((PAL_DEVICE)dev);
}
#endif
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
static void
pal_get_ethtool_stats(struct net_device *dev,
        struct ethtool_stats *stats, u64_t *buf)
{
    mpd_get_ethtool_stats((PAL_DEVICE)dev, buf);
}

static struct ethtool_ops pal_ethtool_ops = {
    .get_settings       = pal_get_settings,
    .set_settings       = pal_set_settings,
    .get_drvinfo        = pal_get_drvinfo,
    .get_wol        = pal_get_wol,
    .set_wol        = pal_set_wol,
    .get_msglevel       = 0,
    .set_msglevel       = 0,
    .nway_reset     = pal_nway_reset,
    .get_link       = ethtool_op_get_link,
#if (LINUX_VERSION_CODE >= 0x20418)
    .get_eeprom_len     = pal_get_eeprom_len,
#endif
#ifdef ETHTOOL_GEEPROM
    .get_eeprom     = pal_get_eeprom,
#endif
#ifdef ETHTOOL_SEEPROM
    .set_eeprom     = pal_set_eeprom,
#endif
    .get_coalesce       = pal_get_coalesce,
    .set_coalesce       = pal_set_coalesce,
    .get_ringparam      = pal_get_ringparam,
    .set_ringparam      = pal_set_ringparam,
    .get_pauseparam     = pal_get_pauseparam,
    .set_pauseparam     = pal_set_pauseparam,
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39))
    .get_rx_csum        = pal_get_rx_csum,
    .set_rx_csum        = pal_set_rx_csum,
    .get_tx_csum        = ethtool_op_get_tx_csum,
#if (LINUX_VERSION_CODE >= 0x20418)
    .set_tx_csum        = ethtool_op_set_tx_csum,
#endif
    .get_sg         = ethtool_op_get_sg,
    .set_sg         = ethtool_op_set_sg,
#ifdef NETIF_F_TSO
    .get_tso        = ethtool_op_get_tso,
    .set_tso        = ethtool_op_set_tso,
#endif
#endif /* (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39)) */
    .get_strings    = pal_get_strings,
#if (LINUX_VERSION_CODE < 0x030000)
    .phys_id        = pal_phys_id,
#else
    .set_phys_id        = pal_phys_id,
#endif
#if (LINUX_VERSION_CODE < 0x020618) /* removed in kernel 2.6.24 */
    .get_stats_count    = pal_get_stats_count,
#endif
    .get_ethtool_stats  = pal_get_ethtool_stats,
};

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_device_handlers(
    PAL_DEVICE dev,
    int (*open)(PAL_DEVICE),
    int (*stop)(PAL_DEVICE),
    int (*hard_start_xmit)(PAL_PACKET, PAL_DEVICE),
    void (*set_multicast_list)(PAL_DEVICE),
    int (*do_ioctl)(PAL_DEVICE, void*, int),
    PAL_DEVICE_STAT (*get_stats)(PAL_DEVICE),
    int (*set_mac_address)(PAL_DEVICE, void*),
    void (*vlan_rx_register)(PAL_DEVICE, void*),
    void (*vlan_rx_kill_vid)(PAL_DEVICE, uint16_t),
    unsigned int irq
    )
{
#if (LINUX_VERSION_CODE >= 0x02061f) /*kernel 2.6.31*/
     struct net_device_ops mynetdev_ops;

     mynetdev_ops.ndo_open = (int (*)(netdevice_t*))open;
     mynetdev_ops.ndo_stop = (int (*)(netdevice_t*))stop;
     mynetdev_ops.ndo_start_xmit = (int (*)(struct sk_buff*, netdevice_t*))hard_start_xmit;
#if(LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0))
     mynetdev_ops.ndo_set_rx_mode = (void (*)(netdevice_t*))set_multicast_list;
#else
     mynetdev_ops.ndo_set_multicast_list = (void (*)(netdevice_t*))set_multicast_list;
#endif
     mynetdev_ops.ndo_set_mac_address = (int (*)(netdevice_t*, void*))set_mac_address;
     mynetdev_ops.ndo_do_ioctl = (int (*)(netdevice_t*, struct ifreq*, int))do_ioctl;
     mynetdev_ops.ndo_get_stats = (struct net_device_stats* (*)(netdevice_t*))get_stats;
#if(LINUX_VERSION_CODE < KERNEL_VERSION(3,1,0))
     mynetdev_ops.ndo_vlan_rx_register = (void (*)(netdevice_t*, struct vlan_group *))vlan_rx_register;
#endif
#if(LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0))
     mynetdev_ops.ndo_vlan_rx_kill_vid = (int (*)(netdevice_t*, __be16, u16))vlan_rx_kill_vid;
#elif(LINUX_VERSION_CODE >= KERNEL_VERSION(3,3,0))
     mynetdev_ops.ndo_vlan_rx_kill_vid = (int (*)(netdevice_t*,uint16_t))vlan_rx_kill_vid;
#else
     mynetdev_ops.ndo_vlan_rx_kill_vid = (void (*)(netdevice_t*,uint16_t))vlan_rx_kill_vid;
#endif

     ((netdevice_t*)dev)->netdev_ops = &mynetdev_ops;
#else /*0x020631*/
    ((netdevice_t*)dev)->open = (int (*)(netdevice_t*))open;
    ((netdevice_t*)dev)->stop = (int (*)(netdevice_t*))stop;
    ((netdevice_t*)dev)->hard_start_xmit = (int (*)(struct sk_buff*, netdevice_t*))hard_start_xmit;
    ((netdevice_t*)dev)->set_multicast_list = (void (*)(netdevice_t*))set_multicast_list;
    ((netdevice_t*)dev)->set_mac_address = (int (*)(netdevice_t*, void*))set_mac_address;
    ((netdevice_t*)dev)->do_ioctl = (int (*)(netdevice_t*, struct ifreq*, int))do_ioctl;
    ((netdevice_t*)dev)->get_stats = (struct net_device_stats* (*)(netdevice_t*))get_stats;
    ((netdevice_t*)dev)->vlan_rx_register = (void (*)(netdevice_t*, struct vlan_group *))vlan_rx_register;
    ((netdevice_t*)dev)->vlan_rx_kill_vid = (void (*)(netdevice_t*,uint16_t))vlan_rx_kill_vid;
    #if (LINUX_VERSION_CODE < 0x020616)
        ((netdevice_t*)dev)->weight = 64; /*unused code in lx_diag */
    #endif
    ((netdevice_t*)dev)->irq = irq;
#endif /*0x020631*/

    ((netdevice_t*)dev)->ethtool_ops = &pal_ethtool_ops;
}

#if defined(HAVE_POLL_CONTROLLER) || defined(CONFIG_NET_POLL_CONTROLLER)
/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_poll_controller(
    struct net_device *dev
    )
{
    poll_mpd((PAL_DEVICE)dev);
}
#endif

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_device_poll_hdl(
    PAL_DEVICE dev
    )
{
#if defined(HAVE_POLL_CONTROLLER) || defined(CONFIG_NET_POLL_CONTROLLER)
    #if (LINUX_VERSION_CODE >= 0x02061f) /*kernel 2.6.31*/
        struct net_device_ops mynetdev_ops;
        memcpy(&mynetdev_ops, ((netdevice_t*)dev)->netdev_ops, sizeof(struct net_device_ops));
        mynetdev_ops.ndo_poll_controller = pal_poll_controller;
        ((netdevice_t*)dev)->netdev_ops = &mynetdev_ops;
    #else
	    ((netdevice_t*)dev)->poll_controller = pal_poll_controller;
    #endif /* 0x020631 */
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_dec_device_reference(
    PAL_DEVICE dev
    )
{
#if !defined(KERNEL_2_2)
    dev_put( (netdevice_t*)dev );
#endif  // !defined(KERNEL_2_2)
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_inc_device_reference(
    PAL_DEVICE dev
    )
{
#if !defined(KERNEL_2_2)
    dev_hold( (netdevice_t*)dev );
#endif  // !defined(KERNEL_2_2)
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_get_device_mtu(
    PAL_DEVICE dev
    )
{
    return ((netdevice_t*)dev)->mtu;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_device_mtu(
    PAL_DEVICE dev,
    int mtu
    )
{
    ((netdevice_t*)dev)->mtu = mtu;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_dev_features(
    PAL_DEVICE dev,
    int highdma
    )
{
    int features = ((netdevice_t*)dev)->features;

#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,39))
    features |= NETIF_F_SG;
#endif
    features |= (NETIF_F_IP_CSUM |
#if(LINUX_VERSION_CODE >= KERNEL_VERSION(3,10,0))
		NETIF_F_HW_VLAN_CTAG_TX | NETIF_F_HW_VLAN_CTAG_RX);
#else
		NETIF_F_HW_VLAN_TX | NETIF_F_HW_VLAN_RX);
#endif
    if (highdma)
        features |= NETIF_F_HIGHDMA;

#ifdef NETIF_F_TSO
    features |= NETIF_F_TSO;
#endif

    ((netdevice_t*)dev)->features = features;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_send(
    PAL_PACKET pkt,
    PAL_DEVICE dev
    )
{
    return mpd_start_xmit(pkt, dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_open(
    PAL_DEVICE dev
    )
{
    return mpd_open(dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_close(
    PAL_DEVICE dev
    )
{
    return mpd_close(dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_ioctl(
    PAL_DEVICE dev,
    void* ifr,
    int cmd
    )
{
    struct mii_ioctl_data *data = (struct mii_ioctl_data *)&((struct ifreq *)ifr)->ifr_data;
    int rc;

    switch(cmd) {
    case SIOCGMIIPHY:
    {
        u32_t phy_id;
        if ((rc = mpd_get_phy_id(dev, &phy_id)))
            return rc;

        data->phy_id = phy_id;
    }
        /* fallthru */
    case SIOCGMIIREG:
    {
        u32_t phy_addr, phy_data;

        phy_addr = data->reg_num & 0x1f;
        rc = mpd_read_phy_reg(dev, phy_addr, &phy_data);
        data->val_out = phy_data;
        return rc;
    }
    case SIOCSMIIREG:
    {
        u32_t phy_addr;

        if (!capable(CAP_NET_ADMIN))
            return -EPERM;

        phy_addr = data->reg_num & 0x1f;
        rc = mpd_write_phy_reg(dev, phy_addr, data->val_in);
        return rc;
    }
    default:
        /* do nothing */
        break;
    }
    return -EOPNOTSUPP;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
PAL_DEVICE_STAT
pal_get_stats(
    PAL_DEVICE dev
    )
{
    return mpd_get_stats(dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_set_multicast_list(
    PAL_DEVICE dev
    )
{
    return mpd_set_multicast_list(dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_vlan_rx_register(
    PAL_DEVICE dev,
    void *vlgrp
    )
{
    mpd_vlan_rx_register(dev, vlgrp);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_vlan_rx_kill_vid(
    PAL_DEVICE dev,
    u16_t vid
    )
{
    mpd_vlan_rx_kill_vid(dev, vid);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_kill_vid(
    void *vlgrp,
    u16_t vid
    )
{
#if (LINUX_VERSION_CODE < 0x020615)
    /*unused code in lx_diag */
    ((struct vlan_group *)vlgrp)->vlan_devices[vid] = NULL;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_pci_enable_device(void *pcidev, int *irq, int *domain, int *bus_num, int *devfn)
{
    int rc;
    rc = pci_enable_device((struct pci_dev *)pcidev);
    *irq = ((struct pci_dev *)pcidev)->irq;
    *bus_num = ((struct pci_dev *)pcidev)->bus->number;
    *devfn = ((struct pci_dev *)pcidev)->devfn;
    *domain = pci_domain_nr( ((struct pci_dev *)pcidev)->bus );
    return rc;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned long
pal_pci_is_iomap(void *pcidev)
{
    return (pci_resource_flags((struct pci_dev *)pcidev, 0) & IORESOURCE_MEM);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_pci_request_regions(void *pcidev, char *name)
{
    return pci_request_regions((struct pci_dev *)pcidev, name);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_set_master(void *pcidev)
{
    pci_set_master((struct pci_dev *)pcidev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int 
pal_pci_set_dma_mask(void *pcidev, u64_t mask)
{
    return pci_set_dma_mask((struct pci_dev *)pcidev, mask);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if (LINUX_VERSION_CODE < 0x020547)
#define pci_set_consistent_dma_mask(pdev, mask) (0)
#endif

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int 
pal_pci_set_consistent_dma_mask(void *pcidev, u64_t mask)
{
    return pci_set_consistent_dma_mask((struct pci_dev *)pcidev, mask);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned long
pal_pci_resource_start(void *pcidev, u32_t bar)
{
    return pci_resource_start((struct pci_dev *)pcidev, bar);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned long
pal_pci_resource_len(void *pcidev, u32_t bar)
{
    return pci_resource_len((struct pci_dev *)pcidev, bar);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_set_drvdata(void *pcidev, void *data)
{
    pci_set_drvdata ((struct pci_dev *)pcidev, data);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void *
pal_pci_get_drvdata(void *pcidev)
{
    return pci_get_drvdata((struct pci_dev *)pcidev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_release_regions(void *pcidev)
{
    pci_release_regions((struct pci_dev *)pcidev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_disable_device(void *pcidev)
{
    pci_disable_device((struct pci_dev *)pcidev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
u64_t
pal_pci_map_single(
    void *pcidev,
    void *addr,
    u32_t size,
    int from_dev
    )
{
    int direction;

    direction = from_dev ? PCI_DMA_FROMDEVICE : PCI_DMA_TODEVICE;
    return (u64_t)pci_map_single((struct pci_dev *)pcidev,
                addr, size, direction);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_unmap_single(
    void *pcidev,
    u64_t addr,
    u32_t size,
    int from_dev
    )
{
    int direction;

    direction = from_dev ? PCI_DMA_FROMDEVICE : PCI_DMA_TODEVICE;
    pci_unmap_single((struct pci_dev *)pcidev,
        (dma_addr_t)addr, size, direction);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
u64_t
pal_pci_map_page(
    void *pcidev,
    PAL_PACKET pkt,
    u32_t idx,
    u32_t *len,
    int from_dev
    )
{
    int direction;
    skb_frag_t *frag = &skb_shinfo((struct sk_buff *)pkt)->frags[idx];

    direction = from_dev ? PCI_DMA_FROMDEVICE : PCI_DMA_TODEVICE;

    *len = frag->size;
#if(LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0))
    return (u64_t)pci_map_page((struct pci_dev *)pcidev, frag->page.p, frag->page_offset,
            frag->size, direction);
#else
    return (u64_t)pci_map_page((struct pci_dev *)pcidev, frag->page, frag->page_offset,
            frag->size, direction);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_unmap_page(
    void *pcidev,
    u64_t addr,
    u32_t size,
    int from_dev
    )
{
    int direction;

    direction = from_dev ? PCI_DMA_FROMDEVICE : PCI_DMA_TODEVICE;
    pci_unmap_page((struct pci_dev *)pcidev,
                (dma_addr_t)addr,
                size,
                direction);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
char *
pal_pci_get_slot_name(
    void *pcidev
    )
{
#if (LINUX_VERSION_CODE < 0x20428)
    return ((struct pci_dev *)pcidev)->slot_name;
#else
    return (char *)pci_name((struct pci_dev *)pcidev);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void *
pal_pci_alloc_consistent(
    void *pcidev,
    u32_t mem_size,
    u64_t *phy_addr
    )
{
    *phy_addr = 0;
	return pci_alloc_consistent((struct pci_dev *)pcidev, mem_size,
                        (dma_addr_t *)phy_addr);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_pci_free_consistent(
    void *pcidev,
    u32_t mem_size,
    void  *virt_mem,
    u64_t phy_addr
    )
{
	pci_free_consistent((struct pci_dev *)pcidev, mem_size,
		                virt_mem, phy_addr);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_pci_write_config_dword(
    void *pcidev,
    int cfg_addr,
    u32_t val
    )
{
    return pci_write_config_dword((struct pci_dev *)pcidev, cfg_addr, val);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_pci_read_config_dword(
    void *pcidev,
    int cfg_addr,
    u32_t *val
    )
{
    return pci_read_config_dword((struct pci_dev *)pcidev, cfg_addr, val);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void*
pal_vmalloc(
    u32_t len
    )
{
    return vmalloc(len);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_vmfree(
    void* p
    )
{
    vfree( p );
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
PAL_DEVICE
pal_alloc_dev(void *pcidev, int sizeof_priv)
{
    struct net_device *dev;

    /* dev zeroed in init_etherdev */
#if (LINUX_VERSION_CODE >= 0x20418)
    dev = alloc_etherdev(sizeof_priv);
#else
    dev = init_etherdev(NULL, sizeof_priv);
#endif

    if (dev)
    {
#if (LINUX_VERSION_CODE < 0x020618) 
        SET_MODULE_OWNER(dev); /*unused code in lx_diag */
#endif
#if (LINUX_VERSION_CODE >= 0x20419)
        SET_NETDEV_DEV(dev, &((struct pci_dev *)pcidev)->dev);
#endif
    }
    return (PAL_DEVICE)dev;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_register_dev(
    PAL_DEVICE dev
    )
{
    int rc = 0;
#if (LINUX_VERSION_CODE >= 0x20418)
    rc = register_netdev((netdevice_t*)dev);
#endif
    return rc;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_unregister_dev(
    PAL_DEVICE dev
    )
{
    unregister_netdev((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_free_dev(
    PAL_DEVICE dev
    )
{
#if (LINUX_VERSION_CODE >= 0x20418)
    free_netdev((netdevice_t*)dev);
#else
    kfree(dev);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_carrier_off(
    PAL_DEVICE dev
    )
{
    netif_carrier_off((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_netif_running(
    PAL_DEVICE dev
    )
{
    return netif_running((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_stop_queue(
    PAL_DEVICE dev
    )
{
    netif_stop_queue((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_device_detach(
    PAL_DEVICE dev
    )
{
    netif_device_detach((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_device_attach(
    PAL_DEVICE dev
    )
{
    netif_device_attach((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_wake_queue(
    PAL_DEVICE dev
    )
{
    netif_wake_queue((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_netif_queue_stopped(
    PAL_DEVICE dev
    )
{
    return netif_queue_stopped((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_carrier_on(
    PAL_DEVICE dev
    )
{
    return netif_carrier_on((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_netif_start_queue(
    PAL_DEVICE dev
    )
{
    netif_start_queue((netdevice_t*)dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned short
pal_dev_promisc(
    PAL_DEVICE dev
    )
{
    return (((netdevice_t*)dev)->flags & IFF_PROMISC);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
unsigned short
pal_dev_allmulti(
    PAL_DEVICE dev
    )
{
    return (((netdevice_t*)dev)->flags & IFF_ALLMULTI);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_get_dev_mc_count(
    PAL_DEVICE dev
    )
{
#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,35))
    return ((netdevice_t*)dev)->mc_count;
#else
	return netdev_mc_count((netdevice_t*)dev);
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void *
pal_get_dev_mclist_head(
    PAL_DEVICE dev,
    u8_t **mac_addr
    )
{
#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,35)) /*changed in kernel 2.6.35 */
    struct dev_mc_list *mclist = ((netdevice_t*)dev)->mc_list;

    *mac_addr = mclist ? (u8_t *)mclist->dmi_addr : NULL;
    return (void *)mclist;
#else
    struct netdev_hw_addr *ha;
    struct netdev_hw_addr_list mclist = ((netdevice_t *)dev)->mc;

    ha = list_first_entry(&(mclist.list), struct netdev_hw_addr, list);
    *mac_addr = ha ? (u8_t *)ha->addr : NULL;
    return (void *)ha;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void *
pal_get_dev_mclist_next(
    void *mclist,
    u8_t **mac_addr
    )
{
#if(LINUX_VERSION_CODE < KERNEL_VERSION(2,6,35))
    struct dev_mc_list *next_list  = ((struct dev_mc_list *)mclist)->next;

    *mac_addr = next_list ? (u8_t *)next_list->dmi_addr : NULL;
    return (void *)next_list;
#else
    struct netdev_hw_addr *next_ha;

    next_ha  = list_entry(((struct netdev_hw_addr *)mclist)->list.next, 
                            struct netdev_hw_addr, 
                            list);
    *mac_addr = next_ha ? (u8_t *)next_ha->addr : NULL;
    return (void *)next_ha;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_change_mac_address(
    PAL_DEVICE dev,
    void *p
    )
{
    struct sockaddr *addr = p;

    memcpy(((struct net_device *)dev)->dev_addr, addr->sa_data,
        ((struct net_device *)dev)->addr_len);
    return mpd_change_mac_addr(dev, addr->sa_data, ((struct net_device *)dev)->addr_len);
}

#define RUN_AT(x) (jiffies + (x))

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_driver_pulse(unsigned long FunctionContext)
{
    driver_pulse(FunctionContext);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_poll_for_link(unsigned long FunctionContext)
{
    poll_for_link(FunctionContext);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_init_timer(
    PAL_TIMER timer
    )
{
    init_timer((struct timer_list *)timer);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_set_timer(
    void *pdev,
    PAL_TIMER timer,
    unsigned long msec,
    void (*tcallback)(unsigned long)
    )
{
    ((struct timer_list *)timer)->expires = RUN_AT((msec*HZ)/1000);
    ((struct timer_list *)timer)->data = (unsigned long)pdev;
    ((struct timer_list *)timer)->function = tcallback;
    add_timer((struct timer_list *)timer);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_cancel_timer(
    PAL_TIMER timer
    )
{
    return del_timer_sync((struct timer_list *)timer);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_tasklet_init(
    PAL_TASKLET bcm_tasklet,
    void (*func)(unsigned long),
    unsigned long data
    )
{
    tasklet_init((struct tasklet_struct *)bcm_tasklet, func, data);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_tasklet_schedule(
    PAL_TASKLET bcm_tasklet
    )
{
    tasklet_schedule((struct tasklet_struct *)bcm_tasklet);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_cancel_tasklet(
    PAL_TASKLET bcm_tasklet
    )
{
    tasklet_disable((struct tasklet_struct *)bcm_tasklet);
    tasklet_kill((struct tasklet_struct *)bcm_tasklet);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_down(
    PAL_MUTEX mutex
    )
{
    down((struct semaphore *)mutex);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_up(
    PAL_MUTEX mutex
    )
{
    up((struct semaphore *)mutex);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_wake_up_event(
    PAL_EVENT event
    )
{
    wake_up_interruptible((event_t *)event);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
int
pal_schedule_timeout(int interruptible, u32_t div_hz)
{
    current->state = interruptible ? TASK_INTERRUPTIBLE : TASK_UNINTERRUPTIBLE;
    schedule_timeout(HZ / div_hz);
    return signal_pending(current);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_is_dump_mode(void)
{
#ifdef RED_HAT_LINUX_KERNEL
    return netdump_mode;
#else
    return 0;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_disable_irq(
    u32_t irq
    )
{
    disable_irq(irq);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_enable_irq(
    u32_t irq
    )
{
    enable_irq(irq);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_free_irq(
    u32_t   irq,
    void    *dev
    )
{
    free_irq(irq, dev);
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
int
pal_inc_usage(
    void
    )
{
    int rc = 0;
#if (LINUX_VERSION_CODE >= 0x20600)
    if (!try_module_get(THIS_MODULE))
    {
        rc = -1;
    }
#else
    MOD_INC_USE_COUNT;
#endif
    return rc;
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
void
pal_dec_usage(
    void
    )
{
#if (LINUX_VERSION_CODE >= 0x20600)
    module_put(THIS_MODULE);
#else
    MOD_DEC_USE_COUNT;
#endif
}

/****************************************************************************
 *
 *
 *
 *
 * Input;
 *
 *
 * Output:
 *
 *
 ****************************************************************************/
#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
pal_udelay(
    unsigned long usecs
    )
{
    udelay(usecs);
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
mm_read_barrier(
    void
    )
{
    rmb();
}

#if defined(CONFIG_X86) && ! defined(CONFIG_X86_64)
REGPARM(3)
#endif
void
mm_write_barrier(
    void
    )
{
    wmb();
}

#ifdef CONFIG_PPC64

u32_t
pal_readl(
    const volatile void *addr
    )
{
    return readl(addr);
}

void
pal_writel(
    u32_t b,
    volatile void *addr
    )
{
    writel(b, addr);
}

void
pal_writew(
    u16_t b,
    volatile void *addr
    )
{
    writew(b, addr);
}

void
pal_writeb(
    u8_t b,
    volatile void *addr
    )
{
    writeb(b, addr);
}
#endif //CONFIG_PPC64
