#ifndef _KERN_COMPAT_H
#define _KERN_COMPAT_H
/* kern_compat.h: Linux PCI network adapter backward compatibility code. */
/*
	$Revision: 1.3 $ $Date: 2000/04/08 01:07:02 $

	Kernel compatibility defines.
	This file provides macros to mask the difference between kernel versions.
	It is designed primarily for network device drivers.

	Written 1999-2000 Donald Becker, Scyld Computing Corporation
	This software may be used and distributed according to the terms
	of the GNU General Public License (GPL), incorporated herein by
	reference.  Drivers interacting with these functions are derivative
	works and thus are covered the GPL.  They must include an explicit
	GPL notice.

	This code provides inline scan and activate functions for PCI network
	interfaces.  It has an interface identical to pci-scan.c, but is
	intended as an include file to simplify using updated drivers with older
	kernel versions.
	This code version matches pci-scan.c:v0.05 9/16/99

	The author may be reached as becker@scyld.com, or
	Donald Becker
	Scyld Computing Corporation
	410 Severn Ave., Suite 210
	Annapolis MD 21403

	Other contributers:
	<none>
*/

#if ! defined(LINUX_VERSION_CODE)  ||  (LINUX_VERSION_CODE < 0x10000)
#include <linux/version.h>
#endif
#if LINUX_VERSION_CODE < 0x20300  &&  defined(MODVERSIONS)
#include <linux/modversions.h>
#endif

#if LINUX_VERSION_CODE < 0x20100  &&  ! defined(__alpha__)
#define ioremap(a,b)\
    (((unsigned long)(a) >= 0x100000) ? vremap(a,b) : (void*)(a))
#define iounmap(v)\
    do { if ((unsigned long)(v) >= 0x100000) vfree(v);} while (0)
#endif


#if LINUX_VERSION_CODE < 0x20115
#define MODULE_AUTHOR(name)  extern int nonesuch
#define MODULE_DESCRIPTION(string)  extern int nonesuch
#define MODULE_PARM(varname, typestring)  extern int nonesuch
#endif

#if LINUX_VERSION_CODE < 0x20123
#define hard_smp_processor_id() smp_processor_id()
#define test_and_set_bit(val, addr) set_bit(val, addr)
#define le16_to_cpu(val) (val)
#define cpu_to_le16(val) (val)
#define le16_to_cpus(val)
#define le32_to_cpu(val) (val)
#define cpu_to_le32(val) (val)
typedef long spinlock_t;
#define SPIN_LOCK_UNLOCKED 0
#define spin_lock(lock)
#define spin_unlock(lock)
#define spin_lock_irqsave(lock, flags)	save_flags(flags); cli();
#define spin_unlock_irqrestore(lock, flags) restore_flags(flags);
#endif

#if LINUX_VERSION_CODE <= 0x20139
#define	net_device_stats enet_statistics
#else
#define NETSTATS_VER2
#endif

#if LINUX_VERSION_CODE < 0x20155
#include <linux/bios32.h>
#define PCI_SUPPORT_VER1
/* A minimal version of the 2.2.* PCI support that handles configuration
   space access.
   Drivers that actually use pci_dev fields must do explicit compatibility.
   Note that the struct pci_dev * "pointer" is actually a byte mapped integer!
*/
#if LINUX_VERSION_CODE < 0x20020
struct pci_dev { int not_used; };
#endif

#define pci_find_slot(bus, devfn) (struct pci_dev*)((bus<<8) | devfn | 0xf0000)
#define bus_number(pci_dev) ((((int)(pci_dev))>>8) & 0xff)
#define devfn_number(pci_dev) (((int)(pci_dev)) & 0xff)

#ifndef CONFIG_PCI
extern inline int pci_present(void) { return 0; }
#else
#define pci_present pcibios_present
#endif

#define pci_read_config_byte(pdev, where, valp)\
	pcibios_read_config_byte(bus_number(pdev), devfn_number(pdev), where, valp)
#define pci_read_config_word(pdev, where, valp)\
	pcibios_read_config_word(bus_number(pdev), devfn_number(pdev), where, valp)
#define pci_read_config_dword(pdev, where, valp)\
	pcibios_read_config_dword(bus_number(pdev), devfn_number(pdev), where, valp)
#define pci_write_config_byte(pdev, where, val)\
	pcibios_write_config_byte(bus_number(pdev), devfn_number(pdev), where, val)
#define pci_write_config_word(pdev, where, val)\
	pcibios_write_config_word(bus_number(pdev), devfn_number(pdev), where, val)
#define pci_write_config_dword(pdev, where, val)\
	pcibios_write_config_dword(bus_number(pdev), devfn_number(pdev), where, val)
#else
#define PCI_SUPPORT_VER2
#endif

#if LINUX_VERSION_CODE < 0x20159
#define dev_free_skb(skb) dev_kfree_skb(skb, FREE_WRITE);
#else
#define dev_free_skb(skb) dev_kfree_skb(skb);
#endif

#if LINUX_VERSION_CODE < 0x2030d
#define net_device device
#endif

#if ! defined(CAP_NET_ADMIN)
#define capable(CAP_XXX) (suser())
#endif
#if ! defined(HAS_NETIF_QUEUE)
#define netif_wake_queue(dev)  mark_bh(NET_BH);
#endif

#ifndef _PCI_SCAN_H
/* A few user-configurable values that may be modified when a module. */
static int min_pci_latency = 32;
extern int debug;

/* Bhahh.  We must redeclare this to work with modversion.h */
extern unsigned long get_module_symbol(char *, char *);

#if (LINUX_VERSION_CODE < 0x20100)
#define PCI_CAPABILITY_LIST	0x34	/* Offset of first capability list entry */
#define PCI_STATUS_CAP_LIST	0x10	/* Support Capability List */
#define PCI_CAP_ID_PM		0x01	/* Power Management */
#endif

enum pci_id_flags_bits {
	/* Set PCI command register bits before calling probe1(). */
	PCI_USES_IO=1, PCI_USES_MEM=2, PCI_USES_MASTER=4,
	/* Read and map the single following PCI BAR. */
	PCI_ADDR0=0<<4, PCI_ADDR1=1<<4, PCI_ADDR2=2<<4, PCI_ADDR3=3<<4,
	PCI_ADDR_64BITS=0x100, PCI_NO_ACPI_WAKE=0x200, PCI_NO_MIN_LATENCY=0x400,
};

struct pci_id_info {
	const char *name;
	struct match_info {
		int	pci, pci_mask, subsystem, subsystem_mask;
		int revision, revision_mask; 				/* Only 8 bits. */
	} id;
	enum pci_id_flags_bits pci_flags;
	int io_size;				/* Needed for I/O region check or ioremap(). */
	int drv_flags;				/* Driver use, intended as capability flags. */
};

enum drv_id_flags {
	PCI_HOTSWAP=1, /* Leave module loaded for Cardbus-like chips. */
};
enum drv_pwr_action {
	DRV_NOOP,			/* No action. */
	DRV_ATTACH,			/* The driver may expect power ops. */
	DRV_SUSPEND,		/* Machine suspending, next event RESUME or DETACH. */
	DRV_RESUME,			/* Resume from previous SUSPEND  */
	DRV_DETACH,			/* Card will-be/is gone. Valid from SUSPEND! */
	DRV_PWR_WakeOn,		/* Put device in e.g. Wake-On-LAN mode. */
	DRV_PWR_DOWN,		/* Go to lowest power mode. */
	DRV_PWR_UP,			/* Go to normal power mode. */
};

struct drv_id_info {
	const char *name;			/* Single-word driver name. */
	int flags;
	int pci_class;				/* Typically PCI_CLASS_NETWORK_ETHERNET<<8. */
	struct pci_id_info *pci_dev_tbl;
	void *(*probe1)(struct pci_dev *pdev, void *dev_ptr,
					long ioaddr, int irq, int table_idx, int fnd_cnt);
	/* Optional, called for suspend, resume and detach. */
	int (*pwr_event)(void *dev, int event);
#if 0
	/* Internal values. */
	struct drv_id_info *next;
	void *cb_ops;
#endif
};

enum  acpi_pwr_state {ACPI_D0, ACPI_D1, ACPI_D2, ACPI_D3};
static inline int acpi_wake(struct pci_dev *pdev);
static inline int acpi_set_pwr_state(struct pci_dev *pdev, enum acpi_pwr_state new_state);

static int (*register_cb_hook)(struct drv_id_info *did);
static void (*unregister_cb_hook)(struct drv_id_info *did);

#if LINUX_VERSION_CODE > 0x20118  &&  defined(MODULE)
MODULE_PARM(min_pci_latency, "i");
#endif

/*
  This code is not intended to support every configuration.
  It is intended to minimize duplicated code by providing the functions
  needed in almost every PCI driver.

  The "no kitchen sink" policy:
  Additional features and code will be added to this module only if more
  than half of the drivers for common hardware would benefit from the feature.
*/

/*
  Ideally we would detect and number all cards of a type (e.g. network) in
  PCI slot order.
  But that does not work with hot-swap card, CardBus cards and added drivers.
  So instead we detect just the each chip table in slot order.

  This routine takes a PCI ID table, scans the PCI bus, and calls the
  associated attach/probe1 routine with the hardware already activated and
  single I/O or memory address already mapped.

  This routine will later be supplemented with CardBus and hot-swap PCI
  support using the same table.  Thus the pci_chip_tbl[] should not be
  marked as __initdata.
*/

#if LINUX_VERSION_CODE >= 0x20200
/* The PCI code in 2.2 is harder to use, and the extra complexity serves
   no real purpose.  The resource code in 2.3 is far worse.  It is a complex
   abstaction layer with negative benefit. */
static inline int pci_drv_register(struct drv_id_info *drv_id, void *initial_device)
{
	int chip_idx, cards_found = 0;
	struct pci_dev *pdev = NULL;
	struct pci_id_info *pci_tbl = drv_id->pci_dev_tbl;
	void *newdev;

	while ((pdev = pci_find_class(drv_id->pci_class, pdev)) != 0) {
		u32 pci_id, pci_subsys_id, pci_class_rev;
		u16 pci_command, new_command;
		int pci_flags, irq;
		long pciaddr;
		long ioaddr;

		pci_read_config_dword(pdev, PCI_VENDOR_ID, &pci_id);
		pci_read_config_dword(pdev, PCI_SUBSYSTEM_ID, &pci_subsys_id);
		pci_read_config_dword(pdev, PCI_REVISION_ID, &pci_class_rev);

		for (chip_idx = 0; pci_tbl[chip_idx].name; chip_idx++) {
			struct pci_id_info *chip = &pci_tbl[chip_idx];
			if ((pci_id & chip->id.pci_mask) == chip->id.pci
				&& (pci_subsys_id&chip->id.subsystem_mask) == chip->id.subsystem
				&& (pci_class_rev&chip->id.revision_mask) == chip->id.revision)
				break;
		}
		if (pci_tbl[chip_idx].name == 0) 		/* Compiled out! */
			continue;

		pci_flags = pci_tbl[chip_idx].pci_flags;
		irq = pdev->irq;
#if LINUX_VERSION_CODE >= 0x2030C
		/* Wow. A oversized, hard-to-use abstraction. Bogus.  Bad job Linus. */
		pciaddr = pdev->resource[(pci_flags >> 4) & 7].start;
#else
		pciaddr = pdev->base_address[(pci_flags >> 4) & 7];
#if defined(__alpha__)			/* Really any machine with 64 bit addressing. */
		if (pci_flags & PCI_ADDR_64BITS)
			pciaddr |= ((long)pdev->base_address[((pci_flags>>4)&7)+ 1]) << 32;
#endif
#endif
		if (debug > 2)
			printk(KERN_INFO "Found %s at PCI address %#lx, IRQ %d.\n",
				   pci_tbl[chip_idx].name, pciaddr, irq);

		if ((pciaddr & PCI_BASE_ADDRESS_SPACE_IO)) {
			ioaddr = pciaddr & PCI_BASE_ADDRESS_IO_MASK;
			if (check_region(ioaddr, pci_tbl[chip_idx].io_size))
				continue;
		} else if ((ioaddr = (long)ioremap(pciaddr & PCI_BASE_ADDRESS_MEM_MASK,
										   pci_tbl[chip_idx].io_size)) == 0) {
			printk(KERN_INFO "Failed to map PCI address %#lx for device "
				   "'%s'.\n", pciaddr, pci_tbl[chip_idx].name);
			continue;
		}

		if ( ! (pci_flags & PCI_NO_ACPI_WAKE))
			acpi_wake(pdev);
		pci_read_config_word(pdev, PCI_COMMAND, &pci_command);
		new_command = pci_command | (pci_flags & 7);
		if (pci_command != new_command) {
			printk(KERN_INFO "  The PCI BIOS has not enabled the"
				   " device at %d/%d!  Updating PCI command %4.4x->%4.4x.\n",
				   pdev->bus->number, pdev->devfn, pci_command, new_command);
			pci_write_config_word(pdev, PCI_COMMAND, new_command);
		}

		newdev = drv_id->probe1(pdev, initial_device,
								ioaddr, irq, chip_idx, cards_found);
		if (newdev  && (pci_flags & PCI_COMMAND_MASTER))
			pci_set_master(pdev);
		if (newdev  && (pci_flags & PCI_COMMAND_MASTER)  &&
			! (pci_flags & PCI_NO_MIN_LATENCY)) {
			u8 pci_latency;
			pci_read_config_byte(pdev, PCI_LATENCY_TIMER, &pci_latency);
			if (pci_latency < min_pci_latency) {
				printk(KERN_INFO "  PCI latency timer (CFLT) is "
					   "unreasonably low at %d.  Setting to %d clocks.\n",
					   pci_latency, min_pci_latency);
				pci_write_config_byte(pdev, PCI_LATENCY_TIMER, min_pci_latency);
			}
		}
		initial_device = 0;
		cards_found++;
	}

	if ((drv_id->flags & PCI_HOTSWAP)
		&& (register_cb_hook =
			(void *)get_module_symbol(NULL, "do_cb_register")) != 0
		&& (*register_cb_hook)(drv_id) == 0) {
		MOD_INC_USE_COUNT;
		return 0;
	} else
		return cards_found ? 0 : -ENODEV;
}
#else
static inline int pci_drv_register(struct drv_id_info *drv_id, void *initial_device)
{
	int pci_index, cards_found = 0;
	unsigned char pci_bus, pci_device_fn;
	struct pci_dev *pdev;
	struct pci_id_info *pci_tbl = drv_id->pci_dev_tbl;
	void *newdev;

	if ( ! pcibios_present())
		return -ENODEV;

	for (pci_index = 0; pci_index < 0xff; pci_index++) {
		u32 pci_id, subsys_id, pci_class_rev;
		u16 pci_command, new_command;
		int chip_idx, irq, pci_flags;
		long pciaddr;
		long ioaddr;
		u32 pci_busaddr;
		u8 pci_irq_line;

		if (pcibios_find_class (drv_id->pci_class, pci_index,
								&pci_bus, &pci_device_fn)
			!= PCIBIOS_SUCCESSFUL)
			break;
		pcibios_read_config_dword(pci_bus, pci_device_fn,
								  PCI_VENDOR_ID, &pci_id);
		pcibios_read_config_dword(pci_bus, pci_device_fn,
								  PCI_SUBSYSTEM_ID, &subsys_id);
		pcibios_read_config_dword(pci_bus, pci_device_fn,
								  PCI_REVISION_ID, &pci_class_rev);

		for (chip_idx = 0; pci_tbl[chip_idx].name; chip_idx++) {
			struct pci_id_info *chip = &pci_tbl[chip_idx];
			if ((pci_id & chip->id.pci_mask) == chip->id.pci
				&& (subsys_id & chip->id.subsystem_mask) == chip->id.subsystem
				&& (pci_class_rev&chip->id.revision_mask) == chip->id.revision)
				break;
		}
		if (pci_tbl[chip_idx].name == 0) 		/* Compiled out! */
			continue;

		pci_flags = pci_tbl[chip_idx].pci_flags;
		pdev = pci_find_slot(pci_bus, pci_device_fn);
		pcibios_read_config_byte(pci_bus, pci_device_fn,
								 PCI_INTERRUPT_LINE, &pci_irq_line);
		irq = pci_irq_line;
		pcibios_read_config_dword(pci_bus, pci_device_fn,
								  ((pci_flags >> 2) & 0x1C) + 0x10,
								  &pci_busaddr);
		pciaddr = pci_busaddr;
#if defined(__alpha__)
		if (pci_flags & PCI_ADDR_64BITS) {
			pcibios_read_config_dword(pci_bus, pci_device_fn,
									  ((pci_flags >> 2) & 0x1C) + 0x14,
									  &pci_busaddr);
			pciaddr |= ((long)pci_busaddr)<<32;
		}
#endif

		if (debug > 2)
			printk(KERN_INFO "Found %s at PCI address %#lx, IRQ %d.\n",
				   pci_tbl[chip_idx].name, pciaddr, irq);

		if ((pciaddr & PCI_BASE_ADDRESS_SPACE_IO)) {
			ioaddr = pciaddr & PCI_BASE_ADDRESS_IO_MASK;
			if (check_region(ioaddr, pci_tbl[chip_idx].io_size))
				continue;
		} else if ((ioaddr = (long)ioremap(pciaddr & PCI_BASE_ADDRESS_MEM_MASK,
										   pci_tbl[chip_idx].io_size)) == 0) {
			printk(KERN_INFO "Failed to map PCI address %#lx.\n",
				   pciaddr);
			continue;
		}

		if ( ! (pci_flags & PCI_NO_ACPI_WAKE))
			acpi_wake(pdev);
		pcibios_read_config_word(pci_bus, pci_device_fn,
								 PCI_COMMAND, &pci_command);
		new_command = pci_command | (pci_flags & 7);
		if (pci_command != new_command) {
			printk(KERN_INFO "  The PCI BIOS has not enabled the"
				   " device at %d/%d!  Updating PCI command %4.4x->%4.4x.\n",
				   pci_bus, pci_device_fn, pci_command, new_command);
			pcibios_write_config_word(pci_bus, pci_device_fn,
									  PCI_COMMAND, new_command);
		}

		newdev = drv_id->probe1(pdev, initial_device,
							   ioaddr, irq, chip_idx, cards_found);

		if (newdev  && (pci_flags & PCI_COMMAND_MASTER)  &&
			! (pci_flags & PCI_NO_MIN_LATENCY)) {
			u8 pci_latency;
			pcibios_read_config_byte(pci_bus, pci_device_fn,
									 PCI_LATENCY_TIMER, &pci_latency);
			if (pci_latency < min_pci_latency) {
				printk(KERN_INFO "  PCI latency timer (CFLT) is "
					   "unreasonably low at %d.  Setting to %d clocks.\n",
					   pci_latency, min_pci_latency);
				pcibios_write_config_byte(pci_bus, pci_device_fn,
										  PCI_LATENCY_TIMER, min_pci_latency);
			}
		}
		initial_device = 0;
		cards_found++;
	}

	if ((drv_id->flags & PCI_HOTSWAP)
		&& register_cb_hook
		&& (*register_cb_hook)(drv_id) == 0) {
		MOD_INC_USE_COUNT;
		return 0;
	} else
		return cards_found ? 0 : -ENODEV;
}
#endif

static inline void pci_drv_unregister(struct drv_id_info *drv_id)
{
	/* We need do something only with CardBus support. */
	if ((unregister_cb_hook =
		 (void *)get_module_symbol(NULL, "do_cb_unregister")) != 0) {
		(*unregister_cb_hook)(drv_id);
		MOD_DEC_USE_COUNT;
	}
	return;
}

/*
  Search PCI configuration space for the specified capability registers.
  Return the index, or 0 on failure.
*/
static inline int pci_find_capability(struct pci_dev *pdev, int findtype)
{
	u16 pci_status, cap_type;
	u8 pci_cap_idx;
	int cap_idx;

	pci_read_config_word(pdev, PCI_STATUS, &pci_status);
	if ( ! (pci_status & PCI_STATUS_CAP_LIST))
		return 0;
	pci_read_config_byte(pdev, PCI_CAPABILITY_LIST, &pci_cap_idx);
	cap_idx = pci_cap_idx;
	for (cap_idx = pci_cap_idx; cap_idx; cap_idx = (cap_type >> 8) & 0xff) {
		pci_read_config_word(pdev, cap_idx, &cap_type);
		if ((cap_type & 0xff) == findtype)
			return cap_idx;
	}
	return 0;
}


/* Change a device from D3 (sleep) to D0 (active).
   Return the old power state.
   This is more complicated than you might first expect since most cards
   forget all PCI config info during the transition! */
static inline int acpi_wake(struct pci_dev *pdev)
{
	u32 base[5], romaddr;
	u16 pci_command, pwr_command;
	u8  pci_latency, pci_cacheline, irq;
	int i, pwr_cmd_idx = pci_find_capability(pdev, PCI_CAP_ID_PM);

	if (pwr_cmd_idx == 0)
		return 0;
	pci_read_config_word(pdev, pwr_cmd_idx + 4, &pwr_command);
	if ((pwr_command & 3) == 0)
		return 0;
	pci_read_config_word(pdev, PCI_COMMAND, &pci_command);
	for (i = 0; i < 5; i++)
		pci_read_config_dword(pdev, PCI_BASE_ADDRESS_0 + i*4,
								  &base[i]);
	pci_read_config_dword(pdev, PCI_ROM_ADDRESS, &romaddr);
	pci_read_config_byte( pdev, PCI_LATENCY_TIMER, &pci_latency);
	pci_read_config_byte( pdev, PCI_CACHE_LINE_SIZE, &pci_cacheline);
	pci_read_config_byte( pdev, PCI_INTERRUPT_LINE, &irq);

	pci_write_config_word(pdev, pwr_cmd_idx + 4, 0x0000);
	for (i = 0; i < 5; i++)
		if (base[i])
			pci_write_config_dword(pdev, PCI_BASE_ADDRESS_0 + i*4,
									   base[i]);
	pci_write_config_dword(pdev, PCI_ROM_ADDRESS, romaddr);
	pci_write_config_byte( pdev, PCI_INTERRUPT_LINE, irq);
	pci_write_config_byte( pdev, PCI_CACHE_LINE_SIZE, pci_cacheline);
	pci_write_config_byte( pdev, PCI_LATENCY_TIMER, pci_latency);
	pci_write_config_word( pdev, PCI_COMMAND, pci_command | 5);
	return pwr_command & 3;
}

static inline int acpi_set_pwr_state(struct pci_dev *pdev, enum acpi_pwr_state new_state)
{
	u16 pwr_command;
	int pwr_cmd_idx = pci_find_capability(pdev, PCI_CAP_ID_PM);

	if (pwr_cmd_idx == 0)
		return 0;
	pci_read_config_word(pdev, pwr_cmd_idx + 4, &pwr_command);
	if ((pwr_command & 3) == ACPI_D3  &&  new_state != ACPI_D3)
		acpi_wake(pdev);		/* The complicated sequence. */
	pci_write_config_word(pdev, pwr_cmd_idx + 4,
							  (pwr_command & ~3) | new_state);
	return pwr_command & 3;
}
#endif

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 * End:
 */
#endif
