/*************************************************************************
  Expand      Expands the current partial word based on match in current file.

  Author:     SemWare (Based on ideas from Mel Hulse and Peter Birch)

  Date:       Jul  8, 1994

  Overview:

  This macro file provides a macro that operates like a limited,
  dynamic Template mode.  It expands, or completes, the "partial"
  word to the left of the cursor based on other words in the file.
  When you execute Expand from the Potpourri PickList, the Expand
  macro is assigned to <Ctrl H>.

  Keys:
              <Ctrl H>  Expand()

  Usage notes:

  With this macro activated, just press <Ctrl H> each time you want
  to complete the partial word immediately preceding the cursor.
  The macro searches backward and forward in the file for words
  that begin with the string that you typed as the partial word.
  If only one match is found, the partial word is replaced with the
  expanded word.  If multiple matches are found, the macro first
  displays a PickList of expanded entries from which you can
  choose.  To select an expanded word from the PickList, move the
  cursor bar over the desired entry and press <Enter>.  Your
  partial word is then replaced in the text with the expanded word
  you selected.

  Copyright 1992-1994 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/

constant
    EXPAND_KEY = <Ctrl h>,
    COMMAND_COMPLETION_KEY = <CtrlShift h>,
    RANGE = 1000

integer work_id, curr_id, maxlen

proc PushString(string s)
    integer i = Length(s)

    while i
        PushKey(Asc(s[i]))
        i = i - 1
    endwhile
end

/************************************************************************
  Gather replacement candidates for abbrev, using the current marked block.
 ************************************************************************/
proc GatherCandidates(string abbrev, string options, integer minlen)
    string s[80]
    integer p

    PushPosition()
    while lFind(abbrev, options)
        p = CurrPos()
        if not Left() or not isWord()
            GotoPos(p + minlen)
            if isWord()
                while isWord() and Right()
                endwhile
                s = GetText(p, CurrPos() - p)
                GotoBufferId(work_id)
                if not lFind(s, "^$g")
                    EndFile()
                    AddLine(s)
                    if Length(s) > maxlen
                        maxlen = Length(s)
                    endif
                endif
                GotoBufferId(curr_id)
            endif
        endif
        GotoPos(p)
    endwhile
    PopPosition()
end

keydef ListKeyDef
<SpaceBar>  Down()
end

proc GrabSpaceBar()
    Enable(ListKeyDef)
    UnHook(GrabSpaceBar)
end

integer proc Expand()
    string abbrev[80], replacement[80]
    integer len, line, ok

    Set(Break, on)
    curr_id = GetBufferId()
    maxlen = 12

    ok = FALSE

// get word to left of cursor
    PushPosition()
    if not Left() or not isWord()
        PopPosition()
        return (ok)
    endif

    PushBlock()
    UnMarkBlock()
    MarkColumn()
    BegWord()
    PushPosition()
    abbrev = GetMarkedText()
    len = Length(abbrev)

// look above
    if CurrLine() > 1
        MarkLine(CurrLine() - RANGE, CurrLine())
        GatherCandidates(abbrev, "bl", len)
    endif

// look below
    if CurrLine() < NumLines()
        MarkLine(CurrLine(), CurrLine() + RANGE)
        GatherCandidates(abbrev, "l+", len)
    endif

    PopPosition()               // at replace point
// Choose which one
    GotoBufferId(work_id)
    if NumLines() == 0
        PopPosition()
        Message("Nothing found...")
    else
        GotoLine(1)
        line = 1
        if NumLines() > 1
            PushString(abbrev)
            Hook(_LIST_STARTUP_, GrabSpaceBar)
            line = List("", maxlen)
        endif
        if line == 0
            PopPosition()
        else
            KillPosition()
            replacement = GetText(1, CurrLineLen())
            GotoBufferId(curr_id)
            ok = lReplace(abbrev, replacement, "1")
        endif
    endif
    EmptyBuffer(work_id)

    PopBlock()
    return (ok)
end

proc WhenLoaded()
    integer id = GetBufferId()

    work_id = CreateTempBuffer()
    GotoBufferId(id)
    if work_id == 0
        PurgeMacro(CurrMacroFilename())
    endif
end

proc WhenPurged()
    AbandonFile(work_id)
end

<EXPAND_KEY>                Expand()
//<COMMAND_COMPLETION_KEY>    CompleteCommand()

