#!/usr/bin/env python


import sys
import string, getpass, re
import logging
import gettext
import codecs
from types import *

import Getline, N1Getline

N1_DELIM = string.whitespace + '='
RE_ARG = re.compile(r'{\d+}')

# Python's codecs module is lacking some codec aliases.  Add new
# aliases here.  This map is used by n1common.enc_alias, which must be
# registered before it takes affect.  (See n1sh.py.)
ENC_ALIASES = {
    'iso88591': 'iso8859_1',
    'iso88592': 'iso8859_2',
    'iso88593': 'iso8859_3',
    'iso88595': 'iso8859_5',
    'iso88596': 'iso8859_6',
    'iso88597': 'iso8859_7',
    'iso88598': 'iso8859_8',
    'iso88599': 'iso8859_9',
    'iso885910': 'iso8859_10',
    'iso885913': 'iso8859_13',
    'iso885914': 'iso8859_14',
    'iso8859-15': 'iso8859_15',
    'iso885915': 'iso8859_15',
    'koi8r': 'koi8_r',
    'koi8u': 'koi8_u',
}


def map_commands_to_nodes(tabdata, all_tokens):
    """Return nodes corresponding to each token in commands."""

    descriptors = []
    node = tabdata
    tokens = all_tokens[:]

    while tokens:
        tok = tokens.pop(0)

        try:
            nnode = node.tokens[tok]
            descriptors.append(N1Getline.TokenDesc(tok, nnode))
            node = nnode
            continue
        except KeyError: pass

        if tok in node.attrs:
            # Enter attribute processing mode.  Return when done.
            tokens.insert(0, tok)
            while tokens:
                tok = tokens.pop(0)
                try:
                    nnode = node.attrs[tok]
                    descriptors.append(N1Getline.AttrNameDesc(tok, nnode))
                    node = nnode
                    continue
                except KeyError: pass

                try:
                    nnode = node.tokens[tok]
                    descriptors.append(N1Getline.AttrValueDesc(tok, nnode))
                    node = nnode
                    continue
                except KeyError: pass

                userval = get_userval_node(node.tokens)
                if not userval:
                    return None

                descriptors.append(N1Getline.AttrValueDesc(tok, userval))
                node = userval

            return descriptors

        userval = get_userval_node(node.tokens)
        if not userval:
            return None

        descriptors.append(N1Getline.ArgumentDesc(tok, userval))
        node = userval

    return descriptors


def possible_completions(matches, base = '', outstream = sys.stdout):
    """Print the possible completions."""

    base = base.strip()
    if base:
        outstream.write("%s (%s):\n" \
                        % (_('hss.ui.console.match.potential'), base))
    else:
        outstream.write("%s:\n" % _('hss.ui.console.match.available'))

    matches.sort()

    #if len(matches) == 1 and self.is_userval(matches[0].name):
    #    outstream.write('    Must enter argument:\n')

    name_len = reduce(max, [len(node.name) for node in matches], 0)

    for n in matches:
        outstream.write('    %-*s  %s\n' % (name_len, n.name, n.help))


def hide_sensitive(line, session):
    """Return line with sensitive data replaced by '?'."""

    descs = map_commands_to_nodes(session.get_tabdata(),
                                  tokenize(line))

    safetoks = []

    if descs:
        for desc in descs:
            if desc.node.sensitive:
                desc.token = '?'

        cmd = join_descs(descs)
    else:
        cmd = line

    return cmd


def get_prompts(descs):
    """Collect userval entries that were marked with '?'."""

    prompts = []

    for d in descs:
        if d.token == '?' \
                and (isinstance(d, N1Getline.ArgumentDesc)
                     or isinstance(d, N1Getline.AttrValueDesc)):
            mystdin = sys.stdin
            try:
                sys.stdin = sys.__stdin__
                prompt = d.node.name[1:-1] + ': '
                val = getpass.getpass(prompt)
                prompts.append(val)
            finally:
                sys.stdin = mystdin

    return prompts


def join_descs(descs, prompts = []):
    """Join descriptors together into a valid commandline."""

    tokens = []
    #logging.debug(__name__)

    for d in descs:
        #logging.debug('%s %s = %s' % (d.__class__, d.node.name, d.token))
        if d.token == '?' \
                and (isinstance(d, N1Getline.ArgumentDesc)
                     or isinstance(d, N1Getline.AttrValueDesc)):
            try:
                tokens.append(quotetok(prompts.pop(0)))
            except IndexError:
                tokens.append(quotetok(d.token))
        else:
            tokens.append(quotetok(d.token))

        if isinstance(d, N1Getline.AttrNameDesc):
            tokens.append('=')
        else:
            tokens.append(' ')

    line = ''.join(tokens)
    return line


def enc_alias(enc):
    """Handle additional encoding aliases."""
    try:
        enc = ENC_ALIASES[enc]
        return codecs.lookup(enc)
    except KeyError:
        return None


def _(msg, args = []):
    """
    Wrap gettext.gettext(msg) and add argument substitution.

    Supports only simplest case of Java's MessageFormat.

    Return localized unicode string with arguments substituted.
    """

    targs = type(args)
    if not (targs == TupleType or targs == ListType):
        args = [ args ]

    msg = gettext.gettext(msg)

    msgargs = [ m for m in RE_ARG.finditer(msg) ]
    maxi = reduce(max, [int(m.group()[1:-1]) for m in msgargs], -1)

    if len(args) != maxi + 1:
        logging.error('Arg number mismatch: msg="%s", args=%s' % (msg, args))
        return msg

    out = []
    lasti = 0
    try:
        for m in msgargs:
            out.append(msg[lasti:m.start()])
            i = int(m.group()[1:-1])
            out.append(str(args[i]))
            lasti = m.end()
        out.append(msg[lasti:])
    except IndexError:
        logging.error('Cannot find index: msg="%s", args=%s' % (msg, args))
        return msg

    return (''.join(out)).decode('utf-8')



#
# Helper functions...
#
def get_userval_node(next):
    """Return the userval child."""

    for n in next.values():
        if is_userval(n.name):
            return n
    return


def is_userval(s):
    return s.startswith('<') and s.endswith('>')


def quotetok(token, whitespace=N1_DELIM):
    """Add quotes around token, as appropriate."""

    if not token:
        # Quote if token is the empty string.  (User entered "" in command.)
        token = '""'
    else:
        # Or if it has whitespace.
        for d in whitespace:
            if d in token:
                token = token.replace('\\', '\\\\')
                token = token.replace('"', '\\"')
                token = '"%s"' % token
                break  # Only need to quote once.

    return token


def tokenize(str):
    """Tokenize the command line based on n1 delimeters."""

    return Getline.tokenize(str, N1_DELIM)




