#!/usr/bin/env python

import sys, os
import string
import logging
import unicodedata


class Getline:
    """
    A readline-like module.

    Especially useful for reading unicode from a decoded stream.
    For example:

        locale.setlocale(locale.LC_ALL, '')
        reader = codecs.getreader(locale.getlocale()[1])
        prompt = Getline()
        prompt.input = reader(sys.__stdin__)

    (See codecs and locale modules for more information.)
    """

    events = {
        "exit" : "\r\n",
        "backspace" : '\b'+unichr(0x7f),
        "del_eof" : unichr(0x04),   # ^D
        "break" : unichr(0x03),        # ^C
        "escape" : unichr(0x1b),       # All escape sequences.

        "home" : unichr(0x01),         # ^A
        "end" :  unichr(0x05),         # ^E
        "left" : unichr(0x02),         # ^B
        "right" : unichr(0x06),        # ^F

        "killtoend" : unichr(0x0b),    # ^K
        "killprevword" : unichr(0x17), # ^W

        "killline" : unichr(0x15),     # ^U

        "prevcommand" : unichr(0x10),  # ^P
        "nextcommand" : unichr(0x0E),  # ^N

        "tab" : u'\t'
    }

    escapes = {
        '[C' : 'do_right',       # right arrow
        '[D' : 'do_left',        # left arrow
        '[A' : 'do_prevcommand', # up arrow
        '[B' : 'do_nextcommand', # down arrow

        '[3~' : 'do_del_eof',
        '[1~' : 'do_home',
        '[4~' : 'do_end',
        '[5~' : 'do_history_top',      # page up
        '[6~' : 'do_history_bottom',   # page down
        'b' : 'do_prevword', 'B' : 'do_prevword',
        'f' : 'do_nextword', 'F' : 'do_nextword',
        chr(0x08) : 'do_killprevword', chr(0x7f) : 'do_killprevword',
    }

    class EOFInterrupt(Exception):
        """Raise when ^D is pressed with nothing on the line."""


    def __init__(self, prompt = u'prompt> '):
        self.prompt = prompt

        self.output = sys.stdout
        self.input = sys.stdin
        self.delims = string.punctuation + string.whitespace
        self.whitespace = string.whitespace

        self.line = u''
        self.cursor = 0
        self.done = False
        self.history = [ self.line ]     # History [0] is reserved for the
        self.history_pos = 0             # current line

        self.prev_char = ''
        self._char = ''


    def getch(self):
        try:
            import tty, termios
        except:
            return

        fd = sys.stdin.fileno()
        savemode = termios.tcgetattr(fd)

        #f = open('/tmp/n1sh.debug', 'a')
        #f.write('\n' + self._encoding + ' ')
        #f.close()

        try:
            tty.setraw(fd, termios.TCSANOW)
            tty.setcbreak(fd, termios.TCSANOW)

            #reader = codecs.getreader(self.in_encoding)
            #r = reader(sys.stdin)
            c = self.input.read(1)
        finally:
            termios.tcsetattr(fd, termios.TCSANOW, savemode)

        self._char = c
        return self._char


    def beep(self):
        self.output.write('\a')


    def getuwidth(self, c):
        """Returns the charcter width of the unicode character."""

        try:
            w = unicodedata.east_asian_width(c)
        except (AttributeError):
            return 1

        if w in 'WF':
            return 2
        else:
            return 1


    def split_at_pos(self):
        """Split the line at the current position and return the two parts."""

        return ( self.line[:self.cursor], self.line[self.cursor:] )


    def write(self, str):
        self.output.write(str)
        self.cursor += len(str)


    def erase(self, count = 1):
        self.output.write(' ' * count)
        self.output.write('\b' * count)


    def print_prompt(self):
        self.output.write(self.prompt)
        self.print_line()


    def print_line(self):
        go_back = len(self.line) - self.cursor
        self.cursor = 0
        self.write(self.line)
        self.do_left(go_back)


    def del_line(self):
        self.do_killtoend()
        while self.line:
            self.do_killprevword()


    def getline(self, prompt = None):
        if prompt:
            self.prompt = prompt

        self.done = False
        self.line = u''
        self.print_prompt()

        while not self.done:
            cmd = self.do_print
            self.prev_char = self._char
            c = self.getch()
            for type in self.events:
                if c in self.events[type]:
                    cmd = getattr(self, "do_" + type)
                    break
            cmd()

        self.output.write('\n')

        return self.line


    def add_history(self, line):
        if line.strip():
            self.history.insert(1, line) # history[0] is the current line
        if len(self.history) > 100:
            self.history.pop()
        self.history_pos = 0

    def do_print(self):
        try:
            unicodedata.name(self._char)
        except TypeError:
            pass
        except ValueError:
            self.beep()
            return

        ( pre, post ) = self.split_at_pos()
        self.write(self._char)
        self.write(post)
        self.line = pre + self._char + post
        self.do_left(len(post))


    def do_escape(self):
        """
        Forwards calls to the correct handler function for
        ANSI escape sequences.
        """

        all_escapes = self.escapes.keys()
        collected = ''

        cmd = None
        while True:
            c = self.getch()
            collected += c
            funcs = filter(lambda f: f.startswith(collected), all_escapes)
            if len(funcs) == 0:
                #if c not in '~':
                #    c = self.getch()
                break
            if len(funcs) == 1 and collected == funcs[0]:
                cmd = getattr(self, self.escapes[funcs[0]])
                break
        try:
            cmd()
        except:
            self.beep()

    def do_left(self, count = 1):
        for i in range(count):
            if self.cursor == 0:
                self.beep()
                break
            w = self.getuwidth(self.line[self.cursor - 1])
            self.output.write('\b'*w)
            self.cursor -= 1

    def do_right(self, count = 1):
        for i in range(count):
            if self.cursor > len(self.line) - 1:
                self.beep()
                break
            c = self.line[self.cursor]
            w = self.getuwidth(c)
            self.output.write(c)
            self.cursor += 1

    def do_prevcommand(self):
        if len(self.history) - 1 > self.history_pos:
            self.move_history(1)
        else:
            self.beep()

    def do_nextcommand(self):
        if self.history_pos:
            self.move_history(-1)
        else:
            self.beep()

    def do_history_top(self):
        if len(self.history) > self.history_pos + 1:
            self.move_history(len(self.history) - (self.history_pos + 1))
        else:
            self.beep()

    def do_history_bottom(self):
        if self.history_pos:
            self.move_history(-1 * self.history_pos)
        else:
            self.beep()

    def move_history(self, count):
        self.history[self.history_pos] = self.line
        self.del_line()
        self.history_pos += count
        self.line = self.history[self.history_pos]
        self.cursor = len(self.line)
        self.print_line()

    def do_tab(self):
        """Override for tab-completion."""

        self.beep()

    def do_backspace(self):
        if self.cursor:
            (pre, post) = self.split_at_pos()

            # w is the width of the removed character
            w = self.getuwidth(pre[-1])
            pre = pre[:-1]

            self.do_left()
            self.write(post)
            self.erase(w)
            self.line = pre + post
            self.do_left(len(post))
        else:
            self.beep()

    def do_del_eof(self):
        if self.cursor == 0 and len(self.line) == 0:
            raise Getline.EOFInterrupt
        elif self.cursor < len(self.line):
            ( pre, post ) = self.split_at_pos()

            # w is the width of the removed character
            w = self.getuwidth(post[0])
            post = post[1:]

            self.write(post)
            self.erase(w)
            self.line = pre + post
            self.do_left(len(post))
        else:
            self.beep()

    def do_home(self):
        self.do_left(self.cursor)

    def do_end(self):
        self.do_right(len(self.line) - self.cursor)

    def do_prevword(self):
        if self.line:
            while self.cursor and self.line[self.cursor - 1] in self.whitespace:
                self.do_left()
            while self.cursor and self.line[self.cursor - 1] not in self.whitespace:
                self.do_left()
        else:
            self.beep()

    def do_nextword(self):
        if self.line:
            while self.cursor < len(self.line) and self.line[self.cursor] in self.whitespace:
                self.do_right()
            while self.cursor < len(self.line) and self.line[self.cursor] not in self.whitespace:
                self.do_right()
        else:
            self.beep()

    def do_killtoend(self):
        while self.cursor != len(self.line):
            self.do_del_eof()

    def do_killline(self):
        self.do_killtoend()
        while self.cursor:
            self.do_killprevword()

    def do_killprevword(self):
        if  self.line:
            while self.cursor and self.line[self.cursor - 1] in self.whitespace:
                self.do_backspace()
            while self.cursor and self.line[self.cursor - 1] not in self.whitespace:
                self.do_backspace()
        else:
            self.beep()

    def do_break(self):
        raise KeyboardInterrupt

    def do_exit(self):
        self.done = True


class UnbalancedQuotes(Exception):
    """Raise when tokenizer detects unbalanced quotes."""


def tokenize(str, delims):
    """Break the string by delims, keeping quoted strings intact."""

    toks = []
    quote = None
    isliteral = False
    currtok = None

    for c in str:
        if isliteral:
            currtok = add_char(currtok, c)
            isliteral = False
        elif c in delims:
            if quote:
                currtok = add_char(currtok, c)
            elif currtok != None:
                toks.append(currtok)
                currtok = None
        elif c in '\'"':
            if quote:
                if c == quote:
                    quote = None
                else:
                    currtok = add_char(currtok, c)
            else:
                # Make sure string is initialized to handle "" case.
                currtok = add_char(currtok, '')
                quote = c
        elif c in '\\':
            isliteral = True
        else:
            currtok = add_char(currtok, c)

    if currtok != None:
        toks.append(currtok)

    if quote:
        raise UnbalancedQuotes

    return toks


def add_char(s, c):
    """Add a character to a string, even if the reference is None."""

    try:
        s += c
    except TypeError:
        s = c

    return s


if __name__ == '__main__':

    p = Getline()
    while 1:
        s = p.getline('prompt> ')
        print "\n", s


