#
# Patch Config Scripts
#
###################################################
#
# patch the configuration files
# takes one argument - the PATCHDIR, typically underinstall/patch/<patchnum>
#

#
####################################################
#
# main program
#
# NEWCONFIG - number of new files
# CONFLICTS - number of new files that have conflicts
# NEWBINS   - number of new files that are binary files
#
####################################################
use Switch;
use File::Compare;
use File::Basename;
use Merge qw(merge diff3 traverse_sequences3);

$basedir=$ARGV[0];
$NEWCONFIGS = 0;
$CONFLICTS = 0;
$NEWBINS = 0;

if (@ARGV != 2) {
	print "Incorrect no: of arguments\n";
	&Usage;
	exit(1);
}

$PATCHDIR = "$ARGV[0]\\install\\patch\\$ARGV[1]";
$tstamp = &timestamp;
$LOGFILE = "$PATCHDIR\\patch-config_$tstamp.log";
&log_init;

if ( ! -d "$PATCHDIR/save" ) {
  print "Invalid PATCHDIR, $PATCHDIR/save is not a directory";
  &Usage;
}
$SAVEDIR="$PATCHDIR\\save";
if ( ! -d "$PATCHDIR/backout" ) {
  print "Invalid PATCHDIR, $PATCHDIR/backout is not a directory";
  &Usage;
}
$BACKOUTDIR="$PATCHDIR\\backout";

# check to make sure config is accessible
&check_configaccess;
&save_movedfiles;
&patch_config;
&special_hacks;

&log_msg ("--");
&log_msg ("-- Number of new config files: $NEWCONFIGS");
&log_msg ("-- Number of config files with conflicts: $CONFLICTS");
&log_msg ("-- Number of config files that are images (gif png ico or jpg): $NEWBINS");
&log_msg ("-- The list of new config files are in $PATCHDIR/newconfig.list");
&log_msg ("--");
&log_msg ("-- Please review the changes in the new config files before installing them.");
&log_msg ("-- Note that you may have to run clbuild, chbuild and cnbuild commands.");
&log_msg ("-- You also need to apply the ldif files under $basedir/lib/patch.");




####################################################
#
# Usage
#
####################################################
sub Usage
{
print <<USAGE;

Usage:
	  patch-config.pl <Server-Root Path> <base patch number>
	 
	  Server-Root Path : Path where MS is being installed
  	  patch number : the base patch number (<patchid>-<patchrev>),
          e.g. 116568-05
	
USAGE
}

########################################
#
# Return the timestamp
#
########################################
sub timestamp
{
$t = time;
$now = localtime($t);
($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($t);
$year += 1900;
$mon = $mon + 1;
$_DATE = sprintf ("%d%02d%02d%02d%02d%02d", $year, $mon, $mday, $hour, $min, $sec);
}

########################################
#
# Initialize the LogFile
#
########################################
sub log_init
{
$CUR_DATE = sprintf ("%d-%02d-%02d::%02d %02d %02d", $mday, $mon, $year, $hour, $min, $sec);
open (OUT, ">>$LOGFILE");
print OUT "============ LOGFILE started $CUR_DATE ==========\n";
}

########################################
#
# Log a message to the LOGFILE
#
########################################
sub log_only
{
my($message) = @_;
print OUT "$message\n";
}

########################################
#
# Log a message to the screen and to the LOGFILE
#
########################################
sub log_msg
{
my($message) = @_;
print "$message\n";
print OUT "$message\n";
}

####################################################
#
# check to make sure config is accessible
#
####################################################
sub check_configaccess
{
if ( ! -w "$basedir/config/imta_tailor" ) {
    &log_msg ("-- Could not access the config area $basedir/config");
    &log_msg ("-- This utility requires access to the config area");
    exit (2);
}  
}

###################################################
#
# Special code for files that have moved locations
# - return_header.opt
#   - moved config location: from config/locale/C/ to config template
#
###################################################
sub save_movedfiles
{
&log_only ("-- save_movedfiles being run");
if ( ! -f "$basedir/config/return_header.opt" && -f "$basedir/config/locale/C/return_header.opt" ) {
    &log_only ("copy $basedir\\config\\locale\\C\\return_header.opt $basedir\\config\\return_header.opt");
    system("copy $basedir\\config\\locale\\C\\return_header.opt $basedir\\config\\return_header.opt");
}
}

####################################################
#
# routine to add a final new line to a file if it does not have one
#
# diff3 will core dump if the file does not end in a newline
# so add a new line if it is missing
#
# - uses the $file variable, and fixes it in-place
#
####################################################
sub add_newline
{
&log_only ("Adding a new line if it doesnt exists");
open(IN,"$file");
open(TEMP,">$file.temp");
$line = <IN>;
while($line ne "") {
	print TEMP $line;
	$line = <IN>;
}
print TEMP "\n";
close(IN);
close(TEMP);
&log_only ("Moving $file.temp to $file\n");
system("move /Y $file.temp $file");
&log_only ("Moving status is $?\n");
}

####################################################
#
# patch config files
# - attempt to patch the user files, but do not actually overwrite
# - These are config files that do not have substitutions in them.
#   so they are "easier" to patch
#
####################################################
sub patch_config
{
&log_only ("-- patch_config being run");
print "-- Applying the Patch Config ........\n";
if( -f "$PATCHDIR/newconfig.list") {
	system("del $PATCHDIR\\newconfig.list");
}
open(NEWCONF,">$PATCHDIR\\newconfig.list");
open(CONF,"$PATCHDIR\\config.list");
$fn = <CONF>;
while($fn ne "") {
    # the "template" file, a path relative to $basedir
    # the "config" file, a path relative to $basedir
    chomp($fn);
    ($tfile,$cfile) = split(/[\t ]+/,$fn);
    # need to do different things depending on:
    # - if the file is a .gif, .jpg .png or .ico file
    # - if the file is a text config file
    switch($tfile) {
    	case /.gif$/	{ &patch_binfile; }
    	case /.jpg$/	{ &patch_binfile; }
    	case /.ico$/	{ &patch_binfile; }
    	case /.png$/	{ &patch_binfile; }
    	case /.jar$/	{ &patch_binfile; }
        else		{ &patch_configfile; }
    }
    $fn = <CONF>;	
}
close(CONF);
}

####################################################
#
# patch a single binary config file
#
####################################################
sub patch_binfile
{
$tmp_copyfile = "C:\\tmplogfile.txt";
open(TEMP,"$tmp_copyfile");
&log_only ("-- Running patch_binfile");
# if (old $tfile != new $tfile && new $tfile != $cfile) do something
$cmp_tfiles = compare("$BACKOUTDIR\\$tfile","$basedir\\$tfile");
$cmp_cfile = compare("$basedir\\$tfile","$basedir\\$cfile");
&log_only ("-- cmp_tfile = $cmp_tfiles cmp_cfile = $cmp_cfile");
# cmp status is 0 if files are identical
if ( $cmp_tfiles != 0 && $cmp_cfile != 0 ) {
# backup the $cfile, do not overwrite an existing backup
	$dir=dirname("$SAVEDIR\\$cfile");
        if ( ! -d "$dir" ) {
      		&log_only ("mkdir  $dir");
      		use File::Path qw(mkpath);
		mkpath $dir;
       	}
    	if ( ! -f "$SAVEDIR/$cfile" ) {
      		&log_only ("copy $basedir\\$cfile $SAVEDIR\\$cfile");
      		system("copy $basedir\\$cfile $SAVEDIR\\$cfile >> $tmp_copyfile");
    	}
    	system("copy $basedir\\$tfile $SAVEDIR\\$cfile.new >> $tmp_copyfile");
    	$NEWCONFIGS++;
    	$NEWBINS++;
    	&log_msg ("-- New: $basedir\$cfile");
    	print NEWCONF "$cfile\n";    	
} else {
    if( -f "$SAVEDIR/$cfile.new" ) { 	
    	system("del $SAVEDIR\\$cfile.new");
    }
    &log_only ("-- No patch required for $basedir\\$cfile");
}
close(TEMP);
}

########################################################
#
# patch a single text config file
# using 3 way diff
#
# diff3 will core dump if the file does not end in a newline
# so add a new line if it is missing. Do not have to do it for 3rd
# file because we will make sure from now on we will not have files like 
# this.
#
########################################################
sub patch_configfile
{
&log_only ("-- Running patch_configfile");
$tmp_copyfile = "C:\\tmplogfile.txt";
open(TEMP,"$tmp_copyfile");
# The 3 files for diff3
$file1="$SAVEDIR\\$cfile.new";
$file2="$BACKOUTDIR\\$tfile";
$file3="$basedir\\$tfile";
$TmpFile = "$SAVEDIR\\$cfile.new.tmp";
open(FILE3,"$basedir\\$tfile");
open(TMP,">$TmpFile");
# backup the $cfile, do not overwrite an existing backup
$dir=dirname("$SAVEDIR\\$cfile");
if ( ! -d "$dir" ) {
    &log_only ("mkdir $dir");
    use File::Path qw(mkpath);
    mkpath $dir;
}  
# create file1: $SAVEDIR/$cfile.new
if ( -f "$basedir/$cfile" ) {
    if ( ! -f "$SAVEDIR/$cfile" ) {
      	&log_only ("copy $basedir\\$cfile $SAVEDIR\\$cfile");
        system("copy $basedir\\$cfile $SAVEDIR\\$cfile >> $tmp_copyfile");
    }
    &log_only ("copy $basedir\\$cfile $SAVEDIR\\$cfile.new");
    system("copy $basedir\\$cfile $SAVEDIR\\$cfile.new >> $tmp_copyfile");
    $file=$file1;
    # &add_newline;
} else {
    if( -f "$SAVEDIR/$cfile.new") { 	
    	system("del $SAVEDIR\\$cfile.new");
    }
}
open(FILE1,"$SAVEDIR\\$cfile.new");
$count = 0;
$tl = <FILE1>;
while($tl ne "") {
	$count++;
	$tl = <FILE1>;
}
seek(FILE1,0,0);
# file2
# if this is a new file
if ( ! -f "$file2" ) {
    if( -f "$PATCHDIR/empty_file" ) {
    	system("del $PATCHDIR\\empty_file");
    }
    open(FILE2, "$PATCHDIR\\empty_file");	
    $file2="$PATCHDIR\\empty_file";
} else {
    $file=$file2;
    # &add_newline;
    open(FILE2,"$BACKOUTDIR\\$tfile");
}  
# see if $file3 exists
if ( ! -f "$file3" ) {
    if( -f "$PATCHDIR/empty_file" ) {
	system("del $PATCHDIR\\empty_file");
    }
    open(FILE3, "$PATCHDIR\\empty_file");	 	
    $file3="$PATCHDIR\\empty_file";
}	
chomp(@f1=<FILE1>);
close FILE1;
chomp(@f2=<FILE2>);
close FILE2;
chomp(@f3=<FILE3>);
close FILE3;
&log_msg ("diff3 -E $file1 $file2 $file3");
@diff = diff3(\@f1, \@f2, \@f3);
$i = 0;
$stat = 0;
$change = 0;
while ($diff[$i][0] eq "o" || $i <= $count) {
	if($diff[$i][0] eq "c") {
		$stat = 1;
		$change = 1;
		print TMP "=======\n";
	} elsif($diff[$i][0] eq "l") {
		$change = 1;
		$count++;
		print TMP "$diff[$i][2]\n";
	} elsif($diff[$i][0] eq "r") {
		$change = 1;
		$count++;
		print TMP "$diff[$i][3]\n";
	} else {
		print TMP "$diff[$i][1]\n";
	}
$i++;
}
close(TMP);
if($change == 1) {
	$NEWCONFIGS++;
	if($stat == 1) {
		&log_msg ("-- New with Conflicts: $basedir\\$cfile");
	        $CONFLICTS++;
	} else {
		&log_msg ("-- New: $basedir\\$cfile");
	}
	print NEWCONF "$cfile\n";
	system("move /Y $SAVEDIR\\$cfile.new.tmp $SAVEDIR\\$cfile.new");
} else {
	system("del $SAVEDIR\\$cfile.new");
	if( -f "$SAVEDIR/$cfile.new.tmp" ) {
		system("del $SAVEDIR\\$cfile.new.tmp");
	}
	&log_only ("-- No patch required for $basedir\\$cfile");
}
close(TEMP);
}

#########################################################
#
# special hard coded stuff for things I do not handle generically
# - msg.conf has a variable that was added in 6.2 local.store.enable
#   Its value is a substitution variable <storeEnable>. 
#   This causes the merge file to have the unsubstituted value in it
#
#########################################################
sub special_hacks
{
&log_only ("-- special_hacks");
$tmp_copyfile = "C:\\tmplogfile.txt";
open(TEMP,"$tmp_copyfile");
if ( -f "$SAVEDIR/config/msg.conf.new" ) {
        &log_only ("move $SAVEDIR\\config\\msg.conf.new $SAVEDIR\\config\\msg.conf.sed");
    	system("move $SAVEDIR\\config\\msg.conf.new $SAVEDIR\\config\\msg.conf.sed");
    	# to preserve permissions
    	system("copy $SAVEDIR\\config\\msg.conf.sed $SAVEDIR\\config\\msg.conf.new >> $tmp_copyfile");
    	&log_only ("s:<storeEnable>:1: from $SAVEDIR\\config\\msg.conf.sed to $SAVEDIR\\config\\msg.conf.new");
    	$TmpFile1 = "$SAVEDIR\\config\\msg.conf.sed";
	$TmpFile2 = "$SAVEDIR\\config\\msg.conf.temp";
	open(IN1,"$TmpFile1");
	open(OUT1,">$TmpFile2");
	$eachline = <IN1>; 
	while($eachline ne "") {
	        $_ = $eachline ;
    		$TmpFile1 = ~s/<storeEnable>/1/g;
    		$TmpFile1 = $_;
    		print OUT1 $TmpFile1;
		$eachline = <IN1>;
	}
	close(IN1);
	close(OUT1);
	system("move $SAVEDIR\\config\\msg.conf.temp $SAVEDIR\\config\\msg.conf.new");
}
close(TEMP);
}
