#
# Install-newconfig Script
#
########################################################
#
# patch the configuration files
# takes one argument - the PATCHDIR, typically underinstall/patch/<patchnum>
#

#
#######################################################
#
# main program
#
########################################################
use Switch;
use File::Compare;
use File::Basename;

$basedir=$ARGV[0];

if (@ARGV < 2) {
	print "Incorrect no: of arguments\n";
	&Usage;
	exit(1);
}

$fflag = 0;
if($ARGV[0] eq "-f") {
	$fflag = 1;
	$ARGV[0] = $ARGV[1];
}

$PATCHDIR = "$ARGV[0]\\install\\patch\\$ARGV[1]";
$tstamp = &timestamp;
$LOGFILE = "$PATCHDIR\\install-newconfig_$tstamp.log";
&log_init;

# save directory created by patchadd and patch-config
if ( ! -d "$PATCHDIR/save" ) {
  print "Invalid PATCHDIR, $PATCHDIR\save is not a directory";
  &Usage;
}
$SAVEDIR="$PATCHDIR\\save";
# backout directory created by patchadd
if ( ! -d "$PATCHDIR/backout" ) {
  print "Invalid PATCHDIR, $PATCHDIR\backout is not a directory";
  &Usage;
}
$BACKOUTDIR="$PATCHDIR\\backout";
# file containing list of new config files
if ( ! -f "$PATCHDIR/newconfig.list" ) {
  print "$PATCHDIR\newconfig.list not found, Please run patch-config first";
  &Usage;
}
# where original config files will be backed up
$CONFIGBACKOUTDIR="$PATCHDIR\\config_$tstamp.log";
use File::Path qw(mkpath);
mkpath $CONFIGBACKOUTDIR;

# check to make sure config is accessible
&check_configaccess;
&isHAconfigured;
if ( $isHAconfigured == 1 ) {
  	&log_msg ("Warning an HA configuration is detected on your systems,");
        &log_msg ("Continue [y,n] :");
  	$ans = <STDIN>;
  	if ( $ans eq "n" || $ans eq "N" || $ans eq "no" || $ans eq "No" || $ans eq "NO" ) {
  		  &log_msg ("-- Okay, terminating. Rerun at your convenience");  
  	  	  exit (3);
  	}
}
&check_servers;
&install_newconfig;
&perform_checks;
&log_msg ("-- The final step is to apply the ldif files");
&log_msg ("-- $basedir/lib/patch/cfgdir_diff.ldif and");
&log_msg ("-- $basedir/lib/patch/ugdir_diff.ldif");
&log_msg ("-- to the config Directory and the user/group Directory respectively");
&log_msg ("-- This is not performed automatically, you must do this manually");



#######################################
#
# Usage
#
########################################
sub Usage
{
print <<USAGE;

Usage:
  
  	install-newconfig.pl <option> <Server-Root Path> <base patch number>
  	
  	<option> : -f : For a forced replacing of files
  	Server-Root Path : Path where MS is being installed
	patch number : the base patch number (<patchid>-<patchrev>),
        e.g. 116568-05
USAGE
}

########################################
#
# Return the timestamp
#
########################################
sub timestamp
{
$t = time;
$now = localtime($t);
($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime($t);
$year += 1900;
$mon = $mon + 1;
$_DATE = sprintf ("%d%02d%02d%02d%02d%02d", $year, $mon, $mday, $hour, $min, $sec);
}

########################################
#
# Initialize the LogFile
#
########################################
sub log_init
{
$CUR_DATE = sprintf ("%d-%02d-%02d::%02d %02d %02d", $mday, $mon, $year, $hour, $min, $sec);
open (LOG_OUT, ">>$LOGFILE");
print LOG_OUT "============ LOGFILE started $CUR_DATE ==========\n";
}

########################################
#
# Log a message to the LOGFILE
#
########################################
sub log_only
{
my($message) = @_;
print LOG_OUT "$message\n";
}

########################################
#
# Log a message to the screen and to the LOGFILE
#
########################################
sub log_msg
{
my($message) = @_;
print "$message\n";
print LOG_OUT "$message\n";
}

########################################
#
# check to make sure config is accessible
#
########################################
sub check_configaccess 
{
  if ( ! -w "$basedir/config/imta_tailor" ) { 
    &log_msg ("-- Could not access the config area $basedir\config");
    &log_msg ("-- This utility requires access to the config area");
    exit (2);
  }
}

########################################
#
# check to make sure servers are stopped
#
########################################
sub check_servers
{
&log_only ("-- check_servers");
if ( -x "$ARGV[0]/sbin/stop-msg.bat" && -x "$ARGV[0]/lib/configutil.exe" ) {
    log_msg ("-- Stopping Messaging Server...");
    system("$ARGV[0]\\sbin\\stop-msg.bat");
    $stat=$?;
    &log_only ("-- stop-msg returned $stat");
}
}

##########################################
#
# determine if configured for HA
# - set the variable isHAconfigured to 0 (false) or 1 (true)
#
##########################################
sub isHAconfigured
{
  $VCSCMD="/opt/VRTSvcs/bin/hares"; 
  $SC3CMD="/usr/cluster/lib/rgm/rtreg/SUNW.ims";
  $IP_INTERFACE = "$basedir\\lib\\tmpfile.txt";
  open(IP_OUT,"$IP_INTERFACE");
  system ("$basedir\\lib\\configutil -o service.listenaddr>$IP_INTERFACE");
  close(IP_OUT);
  open(IP_IN,"$IP_INTERFACE");
  $line = <IP_IN>;
  if ( @ARGV == 0 && ( ! -z "$IP_INTERFACE" || $line eq "INADDR_ANY" ) && ( -f "$VCSCMD" || -f "$SC2CMD" || -f "$SC3CMD" ) ) {
    	$isHAconfigured=1;
  } else {
    	$isHAconfigured=0;
  }
  close (IP_IN);
  &log_only ("-- isHAconfigured = $isHAconfigured");
}

############################################
#
# check_newfile
# - argument passed in $cfile - the config file relative to $basedir
#   $SAVEDIR/$cfile.new will be the "new config file"
# - check if file exists
# - check to see if it has any diff3 conflicts
# If a test fails, increment the variable newfile_error
#
############################################
sub check_newfile
{
$file="$SAVEDIR\\$cfile.new";
$res = 0;
print "File => $file\n";
if ( ! -f "$file" ) {
    	&log_msg ("-- ERROR: $file does not exist");
    	$newfile_error++;
} else {
   	open(MYFILE, "$file");  
        $line = <MYFILE>;
        while ($line ne "") {
         	$stat=grep(/^=======$/, $line);
         	if ( $stat == 1 ) {
              		 $res=1;
         	}
         	$line = <MYFILE>;
        }
        if ($res == 1) {
        	&log_msg ("-- ERROR: $file has diff3 conflicts in it");
                $newfile_error++;
        }
}
}

######################################################    
#
# install newconfig files
# - attempt to patch the user files, but do not actually overwrite
# - These are config files that do not have substitutions in them.
#   so they are "easier" to patch
#
######################################################
sub install_newconfig
{
$newfile_error=0;
$NewConfList = "$PATCHDIR\\newconfig.list";
open(NEWCONF,"$NewConfList");
$line = <NEWCONF>;
while ( $line ne "" ) {
	chomp($line);
	# the "config" file, a path relative to $basedir
	$cfile = $line;
   
	# check if new file exists and has no conflicts
    	# increments newfile_error if any error found
   	&check_newfile;
	$line = <NEWCONF>;
}
if ( $newfile_error != 0 ) {
 	&log_msg ("--");
 	&log_msg ("-- $newfile_error errors found, please correct and rerun the");
	&log_msg ("-- utility. No change was made to the system.");
	exit ($newfile_error);
}

# install the file
$NUMCOPIES=0;
seek(NEWCONF,0,0);
$line = <NEWCONF>;
while ( $line ne "" ) {
    chomp($line);	
    $cfile = $line;
    # check if it is identical first
    $stat=compare("$SAVEDIR\\$cfile.new","$basedir\\$cfile");
    if ( $stat != 0 ) {
   	   # save existing config file
   	   if ( -f "$basedir/$cfile" ) {
   		     $dir=dirname("$CONFIGBACKOUTDIR\\$cfile");
	       	     if ( ! -d "$dir" ) {
	  		use File::Path qw(mkpath);
    			mkpath $dir;
        	     }
	 	     system("copy $basedir\\$cfile $CONFIGBACKOUTDIR\\$cfile");
      	   }
      	   # install new config file
           #   no permissions being enforced by mkdir. Nuts.
           $dir=dirname("$basedir\\$cfile");
           if ( ! -d "$dir" ) {
	 	&log_msg ("-- mkdir $dir");
      		use File::Path qw(mkpath);
    		mkpath $dir;
           }
      	   if ( $fflag != 1 ) {
        	     &log_msg ("copy /-Y $SAVEDIR\\$cfile.new $basedir\\$cfile");
        	     system("copy /-Y $SAVEDIR\\$cfile.new $basedir\\$cfile");
      	   } else {
        	     &log_msg ("copy /Y $SAVEDIR\\$cfile.new $basedir\\$cfile");
        	     system("copy /Y $SAVEDIR\\$cfile.new $basedir\\$cfile");
      	   }
      	   $NUMCOPIES++;
    }
    $line = <NEWCONF>;
}  
&log_msg ("-- Number of different files: $NUMCOPIES");
&log_msg ("-- Original files are backed up under $CONFIGBACKOUTDIR");
}

#######################################################
#
# perform_checks
#
#######################################################
sub perform_checks
{
  &log_only ("-- perform_custom being run");

  &log_msg ("-- Running clbuild, chbuild, cnbuild, test rewrite...");
  
  $tmp_copyfile = "C:\\tmplogfile.txt";
  open(TEMP,"$tmp_copyfile");

  &log_only ("--   $basedir\\sbin\\imsimta.bat clbuild -image_file=IMTA_COMMAND_DATA IMTA_BIN:pmdf.cld");
  system("$basedir\\sbin\\imsimta.bat clbuild -image_file=IMTA_COMMAND_DATA IMTA_BIN:pmdf.cld 2>> $tmp_copyfile");
  &log_only ("--   return status $?");

  &log_only ("--   $basedir\\sbin\\imsimta.bat chbuild");
  system("$basedir\\sbin\\imsimta.bat chbuild >> $tmp_copyfile");
  &log_only ("--   return status $?");

  &log_only ("--   $basedir\\sbin\\imsimta.bat cnbuild");
  system("$basedir\\sbin\\imsimta.bat cnbuild >> $tmp_copyfile");
  &log_only ("--   return status $?");

  &log_only ("--   $basedir\\sbin\\imsimta.bat version");
  system("$basedir\\sbin\\imsimta.bat version >> $tmp_copyfile");
  &log_only ("     return status $?");

  &log_only ("--   $basedir\\sbin\\imsimta.bat test -rewrite -debug postmaster");
  system("$basedir\\sbin\\imsimta.bat test -rewrite -debug postmaster >> $tmp_copyfile");
  &log_only ("--   return status $?");
  
  close(TEMP);
}

