#include <stdio.h>
#include <am_policy.h>

void fail_on_error(am_status_t status, const char *name);

int main(int argc, char **argv)
{
    am_status_t status;
    am_policy_t hdl;
    const char *ssoToken = NULL;
    am_map_t env = NULL, response = NULL;
    am_policy_result_t result = AM_POLICY_RESULT_INITIALIZER;
    am_properties_t properties = AM_PROPERTIES_NULL;
    const char *action = NULL;
    const char *resName = NULL;
    am_map_value_iter_t iter;

    /*
     * resource abstraction structure.  Pass actual implementations of
     * the required abstract routines for the kind of resource name you
     * have implemented.  The am library internally implements URL
     * resource names.  So, we pass references to the resource comparisons
     * inside am library.
     */
    am_resource_traits_t rsrc = {am_policy_compare_urls,
				    am_policy_resource_has_patterns,
				    am_policy_get_url_resource_root,
				    B_FALSE,
				    '/',
				    am_policy_resource_canonicalize,
				    free};

    if(argc < 5) {
	printf("Usage: %s "
               "<properties_file> <sso_token> <resource_name> <action>\n",
               argv[0]);
	return 0;
    }

    ssoToken = argv[2];
    resName = argv[3];
    action = argv[4];

    status = am_properties_create(&properties);
    fail_on_error(status, "am_properties_create");

    /*
     * load the properties file.  This file is the properties file that is
     * used during agent initialization.  If you have installed Identity
     * server or one of its agents, you can pass the path to
     * AMAgent.properties of that installation.  Make sure that the your
     * test program has permissions to write to the log directory specified
     * in the properties file.
     */
    status = am_properties_load(properties, argv[1]);
    fail_on_error(status, "am_properties_load");

    status = am_policy_init(properties);
    fail_on_error(status, "am_init");

    status = am_policy_service_init("iPlanetAMWebAgentService",
				    "UNUSED PARAMETER",
				    rsrc,
				    properties, &hdl);
    fail_on_error(status, "am_policy_init");
    

    /*
     * If the evaluation requires certain environment parameters like
     * IP address of the requester, it may be passed using this structure.
     * Please refer to the documentation for actual key names
     */
    am_map_create(&env);
    am_map_create(&response);

    /*
     * Acutal evaluation routine invoked.
     */
    status = am_policy_evaluate(hdl, ssoToken, resName, action, env, response, &result);
    fail_on_error(status, "am_policy_evaluate");

    /*
     * Policy evaluation is a success.  Now we need to see what is
     * the result of the subject (user) being able to perform the
     * action on the resource (object).
     */

    printf("Policy evalutation successful.\n");
    status = am_map_find(response, action, &iter);
    fail_on_error(status, "am_map_find");

    printf("Remote user = %s\n", result.remote_user);
    printf("Values for the action %s are: ", action);
    while(B_TRUE==am_map_value_iter_is_value_valid(iter)) {
        if(status == AM_SUCCESS) {
            printf("%s ", am_map_value_iter_get(iter)); 
        }
        if(B_TRUE==am_map_value_iter_next(iter)) break;
    }
    printf("\n");

    am_map_value_iter_destroy(iter);

    /* destroy the map. */
    am_map_destroy(env);
    am_map_destroy(response);

    /*
     * Clean up the results structure allocated internally by the
     * evaluation engine.
     */
    am_policy_result_destroy(&result);


    /*
     * Destroy the policy handle.
     */
    am_policy_destroy(hdl);

    /*
     * Library cleanup: last thing to do before exit.
     */
    status = am_cleanup();
    fail_on_error(status, "am_cleanup");

    am_properties_destroy(properties);

    return EXIT_SUCCESS;
}

void fail_on_error(am_status_t status, const char *method_name) {
    if (status != AM_SUCCESS) {
        fprintf(stderr,"\n\n");
        fprintf(stderr,"*** \n");
        fprintf(stderr,"*** ERROR: %s failed with status code=%u\n", method_name, status);
        fprintf(stderr,"*** \n");
        fprintf(stderr,"\n\n");
        exit(EXIT_FAILURE);
    }
}
