/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)CIBValueStatisticImpl.java 1.11     04/07/15 SMI"
 *
 */

package com.sun.mfwk.cib.sdk.statistics;

import java.util.logging.Logger;
import com.sun.mfwk.util.log.MfLogService;

import com.sun.mfwk.cib.statistics.CIBValueStatistic;
import com.sun.mfwk.cib.statistics.CIBStatistic;

import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.OpenDataException;

import com.sun.mfwk.cib.sdk.statistics.CIBStatisticImpl;

/**
 * Provides a default implementation of a CIBValueStatistic
 */
public class CIBValueStatisticImpl
    extends CIBStatisticImpl
    implements CIBValueStatistic {

  static final Logger logger = MfLogService.getLogger("cib-sdk");
  static final String className = "CIBValueStatisticImpl";

// Attributes
  private long myValue = RESET_VALUE;

  // Composite Data support
  protected static String[] itemNames = {
      "Name", "Description", "Unit", "LastSampleTime", "StartTime", "Value"};
  protected static String[] itemDescriptions = {
      "Name", "Description", "Unit", "LastSampleTime", "StartTime", "Value"};
  protected static OpenType[] itemTypes = {
      SimpleType.STRING, SimpleType.STRING, SimpleType.STRING,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG};

  // Composite Type
  private static CompositeType statisticType = null;

  /**
   * Constructor
   * @param name String
   * @param description String
   */
  public CIBValueStatisticImpl(String name, String description) {
    super(name, description, "NUMBER");
    logger.exiting(className, "Constructor");
  }

  /**
   * Set the value of the statsitic object
   * @param value long
   */
  public void setValue(long value) {
    logger.entering(className, "setValue", new Long(value));
    myValue = value;
    logger.exiting(className, "setValue");
  }

  /**
   * Return the current value of the statistic
   * @return long
   */
  public long getValue() {
    logger.exiting(className, "getValue", new Long(myValue));
    return myValue;
  }

  /**
   * Returns a composite data object reprsenting the statistic object
   * @return CompositeData
   */
  public CompositeData toCompositeData() throws OpenDataException {
    logger.entering(className, "toCompositeData");
    try {
      statisticType = new CompositeType("CIBValueStatistic",
                                        "Common monitoring Value Statistic",
                                        itemNames,
                                        itemDescriptions,
                                        itemTypes);
      Object[] itemValues = {
          getName(),
          getDescription(),
          getUnit(),
          new Long(getLastSampleTime()),
          new Long(getStartTime()),
          new Long(getValue())};

      CompositeDataSupport data = null;
      data = new CompositeDataSupport(statisticType,
                                      itemNames, itemValues);
      logger.exiting(className, "toCompositeData", data);
      return data;
    }
    catch (OpenDataException e) {
      logger.throwing(className, "toCompositeData", e);
      logger.exiting(className, "toCompositeData");
      throw e;
    }
  }

  /**
   * Creates a CIBStatistic object from a composite data
   * @return CIBStatistic
   */
  public CIBStatistic fromCompositeData(CompositeData data) {
    logger.entering(className, "fromCompositeData", data);
    String name = (String) data.get("Name");
    String description = (String) data.get("Description");
    String unit = (String) data.get("Unit");
    Long lastSampleTime = (Long) data.get("LastSampleTime");
    Long startTime = (Long) data.get("StartTime");
    Long value = (Long) data.get("Value");

    // Create a new statistic
    CIBValueStatisticImpl statistic = new CIBValueStatisticImpl(name,
        description);

    ( (CIBStatisticImpl) statistic).setLastSampleTime(lastSampleTime.longValue());
    ( (CIBStatisticImpl) statistic).setStartTime(startTime.longValue());

    statistic.setValue(value.longValue());
    logger.exiting(className, "fromCompositeData", statistic);
    return statistic;

  }

  /**
   * Reset the statistic object
   */
  public void reset() {
    logger.entering(className, "reset");
    super.reset();
    myValue = RESET_VALUE;
    logger.exiting(className, "reset");
  }

  /**
   * Returns a printable statistic object
   * @return String
   */
  public String toString() {
    String str = super.toString();
    return new String(str + "Value=" + this.getValue() + super.strSeparator);
  }
}
