/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)CIBCountStatisticImpl.java 1.11     04/07/15 SMI"
 *
 */

package com.sun.mfwk.cib.sdk.statistics;

import com.sun.mfwk.cib.statistics.CIBCountStatistic;
import com.sun.mfwk.cib.statistics.CIBStatistic;
import javax.management.openmbean.CompositeData;

import java.util.logging.Logger;
import com.sun.mfwk.util.log.MfLogService;

import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeData;
import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.OpenDataException;

/**
 * Provides a default implementation of a CIBCountStatistic
 */
public class CIBCountStatisticImpl
    extends CIBStatisticImpl
    implements CIBCountStatistic {

  static final Logger logger = MfLogService.getLogger("cib-sdk");
  static final String className = "CIBCountStatisticImpl";

  private long myCount = RESET_VALUE;

  // Composite Data support
  protected static String[] itemNames = {
      "Name", "Description", "Unit", "LastSampleTime", "StartTime", "Count"};
  protected static String[] itemDescriptions = {
      "Name", "Description", "Unit", "LastSampleTime", "StartTime", "Count"};
  protected static OpenType[] itemTypes = {
      SimpleType.STRING, SimpleType.STRING, SimpleType.STRING,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG};

  // Composite Type
  private static CompositeType statisticType = null;

  /**
   * Constructor
   * @param name String
   * @param description String
   */
  public CIBCountStatisticImpl(String name, String description) {
    super(name, description, "COUNTER");
    logger.exiting(className, "Constructor");
  }

  /**
   * Set the count of the statistic object
   * @param count long
   */
  public void setCount(long count) {
    logger.entering(className, "setCount", new Long(count));
    myCount = count;
    logger.exiting(className, "setCount");
  }

  /**
   * Get the count of the statistic object
   * @return long
   */
  public long getCount() {
    logger.exiting(className, "getCount", new Long(myCount));
    return myCount;
  }

  /**
   * Returns a compsoite data of the statistic object
   * @return CompositeData
   */
  public CompositeData toCompositeData() throws OpenDataException {
    logger.entering(className, "toCompositeData");

    try {
      statisticType = new CompositeType("CIBCountStatistic",
                                        "Common monitoring Count Statistic",
                                        itemNames,
                                        itemDescriptions,
                                        itemTypes);

      Object[] itemValues = {
          getName(),
          getDescription(),
          getUnit(),
          new Long(getLastSampleTime()),
          new Long(getStartTime()),
          new Long(getCount())};

      CompositeDataSupport data = null;

      data = new CompositeDataSupport(statisticType,
                                      itemNames, itemValues);
      logger.exiting(className, "toCompositeData", data);
      return data;
    }
    catch (OpenDataException e) {
      logger.throwing(className, "toCompositeData", e);
      logger.exiting(className, "toCompositeData");
      throw e;
    }
  }

  /**
   * Returns a CIBStatistic object from the composite data
   * @return CompositeData
   */
  public CIBStatistic fromCompositeData(CompositeData data) {
    logger.entering(className, "fromCompositeData", data);
    String name = (String) data.get("Name");
    String description = (String) data.get("Description");
    String unit = (String) data.get("Unit");
    Long lastSampleTime = (Long) data.get("LastSampleTime");
    Long startTime = (Long) data.get("StartTime");
    Long count = (Long) data.get("Count");

    // Create a new statistic
    CIBCountStatisticImpl statistic = new CIBCountStatisticImpl(name,
        description);

    ( (CIBStatisticImpl) statistic).setLastSampleTime(lastSampleTime.longValue());
    ( (CIBStatisticImpl) statistic).setStartTime(startTime.longValue());

    statistic.setCount(count.longValue());
    logger.exiting(className, "fromCompositeData", statistic);
    return statistic;

  }

  /**
   * Reset the statistic object
   */
  public void reset() {
    logger.entering(className, "reset");
    super.reset();
    myCount = RESET_VALUE;
    logger.exiting(className, "reset");
  }

  /**
   * Returns a printable statistic object
   * @return String
   */
  public String toString() {
    String str = super.toString();
    return new String(str + "Count=" + this.getCount() + super.strSeparator);
  }
}
