/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)CIBConnection.java 1.9     04/07/15 SMI"
 *
 */

package com.sun.mfwk.cib.sdk.conn;

// JDK
import java.util.Map;
import java.util.logging.Logger;
import java.io.IOException;

// JMX
import javax.management.Attribute;
import javax.management.ObjectName;
import javax.management.MBeanServerConnection;
import javax.management.InstanceNotFoundException;
import javax.management.AttributeNotFoundException;
import javax.management.MBeanException;
import javax.management.ReflectionException;

// JESMF SDK
import com.sun.mfwk.util.log.MfLogService;
import com.sun.mfwk.cib.CIBIOException;
import com.sun.mfwk.cib.CIBAttributeNotFoundException;

/**
 * Provides an implemetation for attributes mapping.
 */
public class CIBConnection {

  static final Logger logger = MfLogService.getLogger("cib-sdk");
  static final String className = "CIBConnection";

  private MBeanServerConnection myConnection = null;
  private Map myMap = null;

  /**
   * Constructor
   * @param connection MBeanServerConnection
   * @param map Map
   */
  public CIBConnection(MBeanServerConnection connection,
                       Map map) {
    String methodName = "Constructor";
    logger.entering(className, methodName);
    myConnection = connection;
    myMap = map;
    logger.exiting(className, methodName);
  }

  /**
   * Get an Attribute using the connection
   * @param name String
   * @throws CIBIOException
   * @throws CIBAttributeNotFoundException
   * @return Attribute
   */
  public Attribute getAttribute(String name) throws CIBIOException,
      CIBAttributeNotFoundException {

    String methodName = "getAttribute";
    logger.entering(className, methodName, name);

    Object obj = null;
    ObjectName resourceName = null;
    Attribute attribute = null;

    if (myConnection==null) throw new CIBIOException("Null Connection");

    if (myMap.containsKey(name)) {
      resourceName = (ObjectName) myMap.get(name);
      logger.finest(className + " ObjectName=" + resourceName.toString());
      logger.finest(className + " getAttribute");
      try {
        obj = myConnection.getAttribute(resourceName,
                                        name);
        attribute = new Attribute(name, obj);
        logger.exiting(className, "getAttribute", attribute);
        return attribute;
      }
      catch (IOException ex) {
        CIBIOException ex1 = new CIBIOException(ex.getMessage(), ex);
        logger.throwing(className,"getAttribute", ex1);
        throw ex1;
      }
      catch (ReflectionException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            ex.getMessage(), name, ex);
        logger.throwing(className, "getAttribute", ex1);
        throw ex1;
      }
      catch (InstanceNotFoundException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            ex.getMessage(), name, ex);
        logger.throwing(className, "getAttribute", ex1);
        throw ex1;
      }
      catch (AttributeNotFoundException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            ex.getMessage(), name, ex);
        logger.throwing(className, "getAttribute", ex1);
        throw ex1;
      }
      catch (MBeanException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            ex.getMessage(), name, ex);
        logger.throwing(className, "getAttribute", ex1);
        throw ex1;
      }
    }
    else {
      CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
          "CIB to Component product mapping exception", name);
      logger.throwing(className, "getAttribute", ex1);
      throw ex1;
    }
  }

  /**
   * Invoke a method
   * @param method String
   * @throws IOException
   * @throws ReflectionException
   * @throws MBeanException
   * @throws InstanceNotFoundException
   */
  public void invoke(String method) throws CIBIOException,
      CIBAttributeNotFoundException {

    logger.entering(className, "invoke", method);
    Object valOb;
    ObjectName mngtMBean = null;

    if (myMap.containsKey(method)) {
      mngtMBean = (ObjectName) myMap.get(method);
      try {
        valOb = myConnection.invoke(mngtMBean, method, null, null);
        logger.exiting(className, "invoke");
      }
      catch (IOException ex) {
        CIBIOException ex1 = new CIBIOException("", ex);
        logger.throwing(className, "invoke", ex1);
        throw ex1;
      }
      catch (ReflectionException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            "", method, ex);
        logger.throwing(className, "invoke", ex1);
        throw ex1;
      }
      catch (MBeanException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            "", method, ex);
        logger.throwing(className, "invoke", ex1);
        throw ex1;
      }
      catch (InstanceNotFoundException ex) {
        CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
            "", method, ex);
        logger.throwing(className, "invoke", ex1);
        throw ex1;
      }
    }
    else {
      CIBAttributeNotFoundException ex1 = new CIBAttributeNotFoundException(
          "", method);
      logger.throwing(className, "invoke", ex1);
      throw ex1;
    }
  }
}
