/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)mfUtils.c 1.27     04/09/21 SMI"
 *
 */



/*
 *
 * mfUtils.c - utils
 *
 */

#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <strings.h>
#include <plhash.h>
#include <sys/time.h>

#include "mfConstant.h"
#include "mfRserver.h"
#include "mfUtils.h"
#include "mfTransaction.h"


typedef struct resourceMetricsObjectList {
  mfResourceObject_t *resourceMetricsObjectPtr ;
  struct resourceMetricsObjectList *next;
} resourceMetricsObjectList_t;

#define DOMAIN_TRANSACTION_METRICS "MfInstrum:type=MfTransactionMetrics,name="
#define DOMAIN_EXTERNALRESOURCE_METRICS "MfInstrum:type=MfExternalResourceMetrics,name="
#define DOMAIN_RESOURCE_METRICS "MfInstrum:type=MfResourceMetrics,name="
#define DOMAIN_INSTRUM_MANAGEMENT "MfInstrum:type=InstrumentationManagement"
#define DOMAIN_INSTRUM_CONFIG "MfInstrum:type=InstrumentationConfiguration"
#define DOMAIN_INSTRUM_STATUS "MfInstrum:type=MfStatus"


static resourceMetricsObjectList_t *resourceMetricsObjectListHead;



/*
 * Function:resetExternalResourceMetrics   
 *
 */
void mfResetExternalResourceMetrics(/*in */ void *ptr)
{
  
  mfExternalResourceMetrics_t *externalResourceMetrics;

  externalResourceMetrics = (mfExternalResourceMetrics_t *) ptr;
  

 

  externalResourceMetrics->AccumulatedOutboundAssociations= 0;
  externalResourceMetrics->AccumulatedQueriesTime= 0;
  externalResourceMetrics->AccumulatedSqQueriesTime= 0;
  externalResourceMetrics->FailedOutboundAssociations= 0;
  externalResourceMetrics->FailedQueries= 0;
  externalResourceMetrics->LastOutboundActivity= 0;
  externalResourceMetrics->MaxQueriesTime= 0;
  externalResourceMetrics->MinQueriesTime= 0;
  externalResourceMetrics->NbQueries= 0;
  externalResourceMetrics->OutboundAssociations= 0;
   
}
 
/*
 * Function:resetResourceMetrics   
 *
 */
void mfResetResourceMetrics(/*in */ void *ptr)
{
  mfResourceMetrics_t *resourceMetrics;
  
  resourceMetrics = (mfResourceMetrics_t *) ptr;
  
  

   resourceMetrics->AccumulatedOperationTime= 0;
   resourceMetrics->AccumulatedSqOperationTime= 0;
   /* fix BugId : 5069916 (typo)
      FailedOperation --> FailedOperations */
   resourceMetrics->FailedOperations= 0;
   resourceMetrics->MaxOperationTime= 0;
   resourceMetrics->MinOperationTime= 0;
   resourceMetrics->NbOperations= 0; 
}
 
 
/*
 * Function:resetTransactionMetrics   
 *
 */
void mfResetTransactionMetrics(/* in */ void *ptr)
{
  
  mfTransactionMetrics_t  *transactionMetrics;
  transactionMetrics = (mfTransactionMetrics_t *) ptr;

 


  transactionMetrics->AccumulatedResponseTime = 0;
  transactionMetrics->AccumulatedServiceTime = 0;
  transactionMetrics->AccumulatedSqResponseTime = 0;
  transactionMetrics->AccumulatedSqServiceTime = 0;
  transactionMetrics->MaxResponseTime = 0;
  transactionMetrics->MaxServiceTime = 0;
  transactionMetrics->MinResponseTime = 0;
  transactionMetrics->MinServiceTime = 0;
  transactionMetrics->NbFailedRequests = 0;
  transactionMetrics->NbAbortedRequests = 0;
  transactionMetrics->NbInRequests = 0;
  transactionMetrics->NbOutRequests = 0;
  transactionMetrics->SingleAccumulatedServiceTime = 0;
  transactionMetrics->SingleAccumulatedSqServiceTime = 0;
  transactionMetrics->SingleMaxServiceTime = 0;
  transactionMetrics->SingleMinServiceTime = 0;
  transactionMetrics->SingleNbFailedRequests = 0;
  transactionMetrics->SingleNbAbortedRequests = 0;
}
 


/*
 * Function: mfGetInstrumConfigName    
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfGetStatusName(/*out */ char *objectName, int size)
{
if ( strlen(DOMAIN_INSTRUM_STATUS) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_INSTRUM_STATUS);  
    DEBUG_PRINTF2("mfGetInstrumConfigName %s\n",objectName );
    return MF_SUCCESS;
  }
}


/*
 * Function: mfGetInstrumConfigName    
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfGetInstrumConfigName(/*out */ char *objectName, int size)
{
if ( strlen(DOMAIN_INSTRUM_CONFIG) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_INSTRUM_CONFIG);  
    DEBUG_PRINTF2("mfGetInstrumConfigName %s\n",objectName );
    return MF_SUCCESS;
  }
}










/*
 * Function: mfGetInstrumManagementName    
 *
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfGetInstrumManagementName(/*out */ char *objectName, int size)
{
 if ( strlen(DOMAIN_INSTRUM_MANAGEMENT) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_INSTRUM_MANAGEMENT);  
    DEBUG_PRINTF2("mfGetInstrumManagementName %s\n",objectName );
    return MF_SUCCESS;
  }
}


/*
 * Function: mfGetExternalResourceMetricsName    
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfGetExternalResourceMetricsName(/* in */ char *name, /* out */ char *objectName, int size)
{
  if ((strlen(name) + strlen(DOMAIN_EXTERNALRESOURCE_METRICS)) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_EXTERNALRESOURCE_METRICS);
    strcat(objectName,name) ;      
    DEBUG_PRINTF2("mfGetExternalResourceMetricsName %s\n",objectName );
    return MF_SUCCESS;
  }
}


/*
 * Function: mfGetResourceMetricsName    
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfGetResourceMetricsName(/*in */  char *name, /* out */ char *objectName, int size)
{
  if ((strlen(name) + strlen(DOMAIN_RESOURCE_METRICS)) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_RESOURCE_METRICS);
    strcat(objectName,name) ;      
    DEBUG_PRINTF2("mfgetResourceMetricsName %s\n",objectName );
    return MF_SUCCESS;
  }
}

/*
 * Function: mfGetTransactionMetricsName   
 *
 *   
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 *
 */
int mfGetTransactionMetricsName(/* in */ char *name, /* out */ char *objectName, int size)
{ 
  if ((strlen(name) + strlen(DOMAIN_TRANSACTION_METRICS)) >
      size) {
    return MF_FAILURE;
  }  
  else {
    strcpy(objectName,DOMAIN_TRANSACTION_METRICS);
    strcat(objectName,name) ;      
    DEBUG_PRINTF2("mfgetTransactionMetricsName %s\n",objectName );
    return MF_SUCCESS;
  }

}



/*
 * Function: getMetrics    
 *
 *
 *  for a object of type 
 *   -TransactionMetrics_t
 *   -mfResourceMetrics_t
 *   -mfExternalResourceMetrics_t
 *
 * get the:
 *   -opStatus 
 *   -AvailibilityStatus, 
 *   -isMonitoringEnabled  
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int getMetrics(/* in  */ mfMetrics_t *metric,
	       /* in  */ char *attributeName,
	       /* out */ void **value,
	       /* out */ int *type)
{

  DEBUG_PRINTF2("mfCbGetMetrics attributeName = %s\n",attributeName);
   

  if (strcmp(attributeName,"AvailabilityState") ==0) {
    *type = MF_TYPE_INT;    
    *value =  (long *) &(metric->AvailabilityState) ;
    return MF_SUCCESS;
  }

 
  if (strcmp(attributeName,"AvailStateLastChange") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *)&(metric->AvailStateLastChange);
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"DegradedTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *)&(metric->DegradedTime);
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"OperationalState") ==0) {
    *type = MF_TYPE_INT;
    *value =  (long *)&(metric->OperationalState);
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"OpsStateLastChange") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *)&(metric->OpsStateLastChange);
    return MF_SUCCESS;
  }
 
  if (strcmp(attributeName,"RunningTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(metric->RunningTime);
    return MF_SUCCESS;
  }


  if (strcmp(attributeName,"isMonitoringEnabled") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(metric->isMonitoringEnabled);
    return MF_SUCCESS;
  }  
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}


/*
 * Function: setMetrics    
 *
 *
 *  for a object of type 
 *   -TransactionMetrics_t
 *   -mfResourceMetrics_t
 *   -mfExternalResourceMetrics_t
 *
 * set the:
 *   -opStatus 
 *   -AvailibilityStatus, 
 *   -isMonitoringEnabled  
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int setMetrics(/*in */ mfMetrics_t *metric,
	       /*in */ char *attributeName,
	       /*in */ void *value)
{


  DEBUG_PRINTF2("mfCbGetMetrics attributeName = %s\n",attributeName);
  



  if (strcmp(attributeName,"AvailabilityState") ==0) {   
    metric->AvailabilityState = *(long *) value;
    return MF_SUCCESS;
  }

 
  if (strcmp(attributeName,"AvailStateLastChange") ==0) {
    metric->AvailStateLastChange = *(long long *) value;
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"DegradedTime") ==0) {
    metric->DegradedTime = *(long long *) value;
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"OperationalState") ==0) {
    metric->OperationalState = *(long *) value;
    return MF_SUCCESS;
  }

  if (strcmp(attributeName,"OpsStateLastChange") ==0) {
    metric->OpsStateLastChange = *(long long *) value;
    return MF_SUCCESS;
  }



 
  if (strcmp(attributeName,"RunningTime") ==0) {
    metric->RunningTime = *(long long *) value;
    return MF_SUCCESS;
  }


  if (strcmp(attributeName,"isMonitoringEnabled") ==0) {
    metric->isMonitoringEnabled = *(long long *) value;
    return MF_SUCCESS;
  }  
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}


/*
 * Function: mfCbGetTransactionMetrics
 *
 *
 * Transaction metrics getter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbGetTransactionMetrics(void *resource,
			      char *attributeName,
			      void **value,
			      int *type)
{

  
  mfTransactionMetrics_t *transactionMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  transactionMetricsPtr = (mfTransactionMetrics_t *) resource;
  pmetric = (mfMetrics_t *) transactionMetricsPtr ;

   
   rc = getMetrics(pmetric,
		       attributeName,
		       value,
		        type);

   if (rc == MF_SUCCESS) {
     return rc;
   }



  DEBUG_PRINTF2("mfCbGetTransactionMetrics attributeName = %s\n",attributeName);
  /*DEBUG_PRINTF2("mfCbGetTransactionMetrics %x\n",resource); */

  

  if (strcmp(attributeName,"AccumulatedResponseTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(transactionMetricsPtr->AccumulatedResponseTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(transactionMetricsPtr->AccumulatedServiceTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedSqResponseTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(transactionMetricsPtr->AccumulatedSqResponseTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedSqServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(transactionMetricsPtr->AccumulatedSqServiceTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MaxResponseTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(transactionMetricsPtr->MaxResponseTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MaxServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->MaxServiceTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinResponseTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->MinResponseTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->MinServiceTime);
    return MF_SUCCESS; 
  }

 
 
  if (strcmp(attributeName,"NbFailedRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->NbFailedRequests);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"NbAbortedRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->NbAbortedRequests);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"NbInRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->NbInRequests);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"NbOutRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->NbOutRequests);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleAccumulatedServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleAccumulatedServiceTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"SingleAccumulatedSqServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleAccumulatedSqServiceTime);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleMaxServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleMaxServiceTime);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleMinServiceTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleMinServiceTime);
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"SingleNbFailedRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleNbFailedRequests);
    return MF_SUCCESS; 
  }

 
 if (strcmp(attributeName,"SingleNbAbortedRequests") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value = (long long *) &(transactionMetricsPtr->SingleNbAbortedRequests);
    return MF_SUCCESS; 
  }

 /*fix bugId: 5071756
 C SDK getAttributes causing crash of the agent (X86)
 */
  
 if (strcmp(attributeName,"Uri") ==0) {
    *type = MF_TYPE_STRING;
    *value = transactionMetricsPtr->Uri;    
    return MF_SUCCESS; 
  }

  /* resource has not been found */

  DEBUG_PRINTF2("mfCbGetTransactionMetrics %s not found \n",attributeName);
  *type = MF_TYPE_ATTRIBUTE_NOT_EXIST;
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}



/*
 * Function: mfCbSetTransactionMetrics
 *
 *
 * Transaction metrics setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int  mfCbSetTransactionMetrics(void *resource,
			       char *attributeName,
			       void *value)
{
 
  mfTransactionMetrics_t *transactionMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  transactionMetricsPtr = (mfTransactionMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) transactionMetricsPtr ;
  rc = setMetrics(pmetric, attributeName, value);



  DEBUG_PRINTF2("mfCbSetMetrics attributeName = %s\n",attributeName);





 if (rc == MF_SUCCESS) {
     return rc;
   }


  DEBUG_PRINTF2("mfCbSetTransactionMetrics attributeName = %s\n",attributeName);
  /*DEBUG_PRINTF2("mfCbSetTransactionMetrics %x\n",resource); */
  

  if (strcmp(attributeName,"AccumulatedResponseTime") ==0) {    
      transactionMetricsPtr->AccumulatedResponseTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedServiceTime") ==0) {
    transactionMetricsPtr->AccumulatedServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedSqResponseTime") ==0) {
    transactionMetricsPtr->AccumulatedSqResponseTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedSqServiceTime") ==0) {
    transactionMetricsPtr->AccumulatedSqServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MaxResponseTime") ==0) {
    transactionMetricsPtr->MaxResponseTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MaxServiceTime") ==0) {
    transactionMetricsPtr->MaxServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinResponseTime") ==0) {
   transactionMetricsPtr->MinResponseTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinServiceTime") ==0) {
   transactionMetricsPtr->MinServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }

 
 
  if (strcmp(attributeName,"NbFailedRequests") ==0) {
   transactionMetricsPtr->NbFailedRequests= *(long long *) value;
    return MF_SUCCESS; 
  }



  if (strcmp(attributeName,"NbAbortedRequests") ==0) {
   transactionMetricsPtr->NbAbortedRequests= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"NbInRequests") ==0) {
    transactionMetricsPtr->NbInRequests= *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"NbOutRequests") ==0) {
   transactionMetricsPtr->NbOutRequests= *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleAccumulatedServiceTime") ==0) {
    transactionMetricsPtr->SingleAccumulatedServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"SingleAccumulatedSqServiceTime") ==0) {
   transactionMetricsPtr->SingleAccumulatedSqServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleMaxServiceTime") ==0) {
   transactionMetricsPtr->SingleMaxServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleMinServiceTime") ==0) {
   transactionMetricsPtr->SingleMinServiceTime= *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"SingleNbFailedRequests") ==0) {
    transactionMetricsPtr->SingleNbFailedRequests= *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"SingleNbAbortedRequests") ==0) {
    transactionMetricsPtr->SingleNbAbortedRequests= *(long long *) value;
    return MF_SUCCESS; 
  }


 
  if (strcmp(attributeName,"Uri") ==0) {
    transactionMetricsPtr->Uri = (char *) value;   
    return MF_SUCCESS; 
  }

  /* resource has not been found */
  DEBUG_PRINTF2("mfCbSetTransactionMetrics %s not found \n",attributeName);   
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}




/*
 * Function: mfCbInvokeTransactionMetrics
 *
 *
 * Transaction metrics invoker callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbInvokeTransactionMetrics(void *resource, 
				 char* methodName,
				 AttValue_array paramsArray, 
				 String_array signatureArray,
				 void *value, int *type)
{
  
  mfTransactionMetrics_t *transactionMetricsPtr;
  mfMetrics_t *pmetric;

  
  transactionMetricsPtr = (mfTransactionMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) transactionMetricsPtr ;


  DEBUG_PRINTF2("mfCbTransactionMetrics methodName = %s\n", methodName);

  
  if (strcmp(methodName,"disableMonitoring") ==0) {
    pmetric->isMonitoringEnabled = 0;
    
    return MF_SUCCESS;
  }

  if (strcmp(methodName,"enableMonitoring") ==0) {
    mfResetTransactionMetrics(transactionMetricsPtr);
    pmetric->isMonitoringEnabled = 1;
    return MF_SUCCESS;
  }  

  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}


/*
 * Function: mfCbGetResourceMetrics
 *
 *
 *  Resourcemetrics getter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbGetResourceMetrics (/* in */  void *resource,
			    /* in */  char *attributeName,
			    /* out */ void **value,
			    /* out */ int *type) 
{
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) resourceMetricsPtr ;

   
   rc = getMetrics(pmetric,
		       attributeName,
		       value,
		        type);

   if (rc == MF_SUCCESS) {
     return rc;
   }

   
  DEBUG_PRINTF2("mfCbGetResourceMetrics attributeName = %s\n",attributeName);
  

  if (strcmp(attributeName,"AccumulatedOperationTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->AccumulatedOperationTime);
    return MF_SUCCESS; 
  }


 if (strcmp(attributeName,"AccumulatedSqOperationTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->AccumulatedSqOperationTime);
    return MF_SUCCESS; 
  }

 if (strcmp(attributeName,"FailedOperations") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->FailedOperations);
    return MF_SUCCESS; 
  } 

 if (strcmp(attributeName,"MaxOperationTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->MaxOperationTime);
    return MF_SUCCESS; 
  }


 if (strcmp(attributeName,"MinOperationTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->MinOperationTime);
    return MF_SUCCESS; 
  } 

 if (strcmp(attributeName,"NbOperations") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(resourceMetricsPtr->NbOperations);
    return MF_SUCCESS; 
  } 


 return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}



/*
 * Function: mfCbGetStatus
 *
 *
 *  Resourcemetrics getter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbGetStatus (/* in */  void *resource,
		   /* in */  char *attributeName,
		   /* out */ void **value,
		   /* out */ int *type) 
{
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *)  resourceMetricsPtr ;

   
   rc = getMetrics(pmetric,
		       attributeName,
		       value,
		        type);

   if (rc == MF_SUCCESS) {
     return rc;
   }


 return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}











/*
 * Function: mfCbSetResourceMetrics
 *
 *
 *  Resourcemetrics setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int  mfCbSetResourceMetrics(void *resource,
			    char *attributeName,
			    void *value)
{
 
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) resourceMetricsPtr ;
  rc = setMetrics(pmetric, attributeName, value);



  DEBUG_PRINTF2("mfCbSetMetrics attributeName = %s\n",attributeName);



  /* fix bugId 5104353 
     mfCbSetAttribute returns Attribute not found */
   if (rc == MF_SUCCESS) {
     return rc;
   }

  if (strcmp(attributeName,"AccumulatedOperationTime") ==0) {
    resourceMetricsPtr->AccumulatedOperationTime = *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"AccumulatedSqOperationTime") ==0) {
    resourceMetricsPtr->AccumulatedSqOperationTime = *(long long *) value ;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"FailedOperations") ==0) {
    resourceMetricsPtr->FailedOperations = *(long long *) value;
    return MF_SUCCESS; 
  } 

  if (strcmp(attributeName,"MaxOperationTime") ==0) {
    resourceMetricsPtr->MaxOperationTime = *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinOperationTime") ==0) {
    resourceMetricsPtr->MinOperationTime = *(long long *) value;
    return MF_SUCCESS; 
  } 

  if (strcmp(attributeName,"NbOperations") ==0) {
    resourceMetricsPtr->NbOperations = *(long long *) value;
    return MF_SUCCESS; 
  } 



  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}




/*
 * Function: mfCbSetStatus
 *
 *
 *  Resourcemetrics setter callback
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int  mfCbSetStatus(void *resource,
		   char *attributeName,
		   void *value)
{
 
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) resourceMetricsPtr;
  rc = setMetrics(pmetric, attributeName, value);


  return rc;
}









/*
 * Function: mfCbInvokeResourceMetrics
 *
 *
 *  Resourcemetrics invoker callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbInvokeResourceMetrics(void *resource, 
				 char* methodName,
				 AttValue_array paramsArray, 
				 String_array signatureArray,
				 void *value, int *type)
{
  
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;

  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t*) resourceMetricsPtr ;




  DEBUG_PRINTF2("mfCbInvokeResourceMetrics methodName = %s\n", methodName);

  
  if (strcmp(methodName,"disableMonitoring") ==0) {
    pmetric->isMonitoringEnabled = 0;
    
    return MF_SUCCESS;
  }


  if (strcmp(methodName,"enableMonitoring") ==0) {
    mfResetResourceMetrics(resourceMetricsPtr);
    pmetric->isMonitoringEnabled = 1;    
    return MF_SUCCESS;
  }  

  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}


/*
 * Function: mfCbInvokeStatus
 *
 *
 *  Resourcemetrics invoker callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbInvokeStatus(void *resource, 
		     char* methodName,
		     AttValue_array paramsArray, 
		     String_array signatureArray,
		     void *value, int *type)
{
  
  mfResourceMetrics_t *resourceMetricsPtr;
  mfMetrics_t *pmetric;

  
  resourceMetricsPtr = (mfResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) resourceMetricsPtr;




  DEBUG_PRINTF2("mfCbInvokeResourceMetrics methodName = %s\n", methodName);

  
  if (strcmp(methodName,"disableMonitoring") ==0) {
    pmetric->isMonitoringEnabled = 0;
    
    return MF_SUCCESS;
  }


  if (strcmp(methodName,"enableMonitoring") ==0) {
    pmetric->isMonitoringEnabled = 1;    
    return MF_SUCCESS;
  }  

  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}




/*
 * Function:  mfCbGetExternalResourceMetrics
 *
 *
 *  external Resource metrics setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbGetExternalResourceMetrics(/* in */ void *resource,
				   /* in */ char *attributeName,
				   /* out */ void **value,
				   /* out */ int *type)
{

  mfExternalResourceMetrics_t *externalResourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  
  externalResourceMetricsPtr = (mfExternalResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) externalResourceMetricsPtr ;

   
   rc = getMetrics(pmetric,
		       attributeName,
		       value,
		        type);

   if (rc == MF_SUCCESS) {
     return rc;
   }


  if (strcmp(attributeName,"AccumulatedOutboundAssociations") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->AccumulatedOutboundAssociations);
    return MF_SUCCESS; 
  } 
 
 if (strcmp(attributeName,"AccumulatedQueriesTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->AccumulatedQueriesTime);
    return MF_SUCCESS; 
 } 
 if (strcmp(attributeName,"AccumulatedSqQueriesTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->AccumulatedSqQueriesTime);
    return MF_SUCCESS; 
  }
 if (strcmp(attributeName,"FailedOutboundAssociations") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->FailedOutboundAssociations);
    return MF_SUCCESS; 
  } 
 
 if (strcmp(attributeName,"FailedQueries") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->FailedQueries);
    return MF_SUCCESS; 
 } 
 if (strcmp(attributeName,"LastOutboundActivity") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->LastOutboundActivity);
    return MF_SUCCESS; 
  }

 if (strcmp(attributeName,"MaxQueriesTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->MaxQueriesTime);
    return MF_SUCCESS; 
  }


 if (strcmp(attributeName,"MinQueriesTime") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->MinQueriesTime);
    return MF_SUCCESS; 
  }
 if (strcmp(attributeName,"NbQueries") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->NbQueries);
    return MF_SUCCESS; 
  }

 if (strcmp(attributeName,"OutboundAssociations") ==0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(externalResourceMetricsPtr->OutboundAssociations);
    return MF_SUCCESS; 
  }


 return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}




/*
 * Function:  mfCbSetExternalResourceMetrics
 *
 *
 *  external Resource metrics setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int  mfCbSetExternalResourceMetrics(/* in */ void *resource,
				    /* in */ char *attributeName,
				    /* in */ void *value)
{
 
  mfExternalResourceMetrics_t *externalResourceMetricsPtr;
  mfMetrics_t *pmetric;
  int rc;
  
  externalResourceMetricsPtr = (mfExternalResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) externalResourceMetricsPtr;
  rc = setMetrics(pmetric, attributeName, value);


  DEBUG_PRINTF2("mfCbSetMetrics attributeName = %s\n",attributeName);


  /* fix bugId 5104353 
     mfCbSetAttribute returns Attribute not found */
   if (rc == MF_SUCCESS) {
     return rc;
   }

  if (strcmp(attributeName,"AccumulatedOutboundAssociations") ==0) {
    externalResourceMetricsPtr->AccumulatedOutboundAssociations =  *(long long *) value;
    return MF_SUCCESS; 
  } 
 
  if (strcmp(attributeName,"AccumulatedQueriesTime") ==0) {
    externalResourceMetricsPtr->AccumulatedQueriesTime =  *(long long *) value;
    return MF_SUCCESS; 
  } 
  if (strcmp(attributeName,"AccumulatedSqQueriesTime") ==0) {
    externalResourceMetricsPtr->AccumulatedSqQueriesTime =  *(long long *) value;
    return MF_SUCCESS; 
  }
  if (strcmp(attributeName,"FailedOutboundAssociations") ==0) {
    externalResourceMetricsPtr->FailedOutboundAssociations =  *(long long *) value;
    return MF_SUCCESS; 
  } 
 
  if (strcmp(attributeName,"FailedQueries") ==0) {
    externalResourceMetricsPtr->FailedQueries =  *(long long *) value;
    return MF_SUCCESS; 
  } 
  if (strcmp(attributeName,"LastOutboundActivity") ==0) {
    externalResourceMetricsPtr->LastOutboundActivity =  *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"MaxQueriesTime") ==0) {
    externalResourceMetricsPtr->MaxQueriesTime =  *(long long *) value;
    return MF_SUCCESS; 
  }


  if (strcmp(attributeName,"MinQueriesTime") ==0) {
    externalResourceMetricsPtr->MinQueriesTime =  *(long long *) value;
    return MF_SUCCESS; 
  }
  if (strcmp(attributeName,"NbQueries") ==0) {
    externalResourceMetricsPtr->NbQueries =  *(long long *) value;
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"OutboundAssociations") ==0) {
    externalResourceMetricsPtr->OutboundAssociations =  *(long long *) value;
    return MF_SUCCESS; 
  }





  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}







/*
 * Function:  mfCbInvokeExternalResourceMetrics
 *
 *
 *  external Resource metrics setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbInvokeExternalResourceMetrics(void *resource, 
				 char* methodName,
				 AttValue_array paramsArray, 
				 String_array signatureArray,
				 void *value, int *type)
{
  
  mfExternalResourceMetrics_t *externalResourceMetricsPtr;
  mfMetrics_t *pmetric;
  
  externalResourceMetricsPtr = (mfExternalResourceMetrics_t *) resource;
  pmetric =  (mfMetrics_t *) externalResourceMetricsPtr ;


  DEBUG_PRINTF2("mfCbInvokeExternalResourceMetrics methodName = %s\n", methodName);

  
  if (strcmp(methodName,"disableMonitoring") ==0) {
    pmetric->isMonitoringEnabled = 0;
    
    return MF_SUCCESS;
  }


  if (strcmp(methodName,"enableMonitoring") ==0) {
    mfResetExternalResourceMetrics(externalResourceMetricsPtr);
    pmetric->isMonitoringEnabled = 1;
    return MF_SUCCESS;
  }  

  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}




/*
 * Function:  mfCbGetConfig
 *
 *
 *  Configuration getter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */

int mfCbGetConfig (/* in  */ void *resource,
		   /* in  */  char *attributeName,
		   /* out */ void **value,
		   /* out */ int *type)
{
  mfInstrumConfig_t *instrumConfigPtr;

  instrumConfigPtr = (mfInstrumConfig_t *) resource;

  /* fix bugId:5070042
     C SDK : InstrumConfig Attributes not returne
  */

 if (strcmp(attributeName,"AppDirectoryDN") == 0) {
    *type = MF_TYPE_STRING;
    *value =  instrumConfigPtr->AppDirectoryDN;
    return MF_SUCCESS; 
  }

 if (strcmp(attributeName,"AppRole") == 0) {
    *type = MF_TYPE_STRING;
    *value =  instrumConfigPtr->AppRole;
    return MF_SUCCESS; 
  }

 if (strcmp(attributeName,"AppURLDescription") == 0) {
    *type = MF_TYPE_STRING;
    *value = instrumConfigPtr->AppURLDescription;
    return MF_SUCCESS; 
  }

 return MF_RESOURCE_ATTRIBUTE_NOT_EXIST; 
}




/*
 * Function:  mfCbSetConfig
 *
 *
 *  Configuration setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */

int mfCbSetConfig (void *resource,
		   char *attributeName,
		   void *value)
{

  mfInstrumConfig_t *instrumConfigPtr;
  char *valueName = (char *) value;
  

  instrumConfigPtr = (mfInstrumConfig_t *) resource;

  if (strcmp(attributeName,"AppDirectoryDN") == 0) {
    instrumConfigPtr->AppDirectoryDN = malloc(strlen(valueName)+1); 
    bzero(instrumConfigPtr->AppDirectoryDN,strlen(valueName)+1);
    strcpy(instrumConfigPtr->AppDirectoryDN,valueName);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"AppRole") == 0) {
    instrumConfigPtr->AppRole = malloc(strlen(valueName)+1); 
    bzero(instrumConfigPtr->AppRole,strlen(valueName)+1);
    strcpy(instrumConfigPtr->AppRole,valueName);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"AppURLDescription") == 0) {
    instrumConfigPtr->AppURLDescription = malloc(strlen(valueName)+1); 
    bzero(instrumConfigPtr->AppURLDescription,strlen(valueName)+1);
    strcpy(instrumConfigPtr->AppURLDescription,valueName);    
    return MF_SUCCESS; 
  }

  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}





/*
 * Function:  mfCBGetInstrumManagement
 *
 *
 *  Instrum Management getter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbGetInstrumManagement (/* in */  void *resource,
			      /* in */  char *attributeName,
			      /* out */ void **value,
			      /* out */ int *type) 
{

  mfInstrumManagement_t *instrumManagementPtr;
  instrumManagementPtr = (mfInstrumManagement_t *) resource ;
  
  if (strcmp(attributeName,"enableMonitoring") == 0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(instrumManagementPtr->MonitoringEnabled);
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"LoggingLevel") == 0) {
    *type = MF_TYPE_LONGLONG;
    *value =  (long long *) &(instrumManagementPtr->LoggingLevel);
    return MF_SUCCESS; 
  }
 
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
}


 


/*
 * Function: resetMetricObject
 *
 *
 *  Ireset the metric Object 
 *
 *
 */
void resetMetricObject (mfResourceObject_t *resourceObject) 
{
  resetFunc_t *reset;
  
  if ((resourceObject != NULL) && (resourceObject->resource !=NULL)) {
    reset = ((mfMetrics_t *) resourceObject->resource)->reset;
    if (reset != NULL) {
      (*reset)(resourceObject->resource);
    }
  }
}

/*
 * Function: resetMetricObjectList
 *
 *
 *  reset the metric Object List
 *
 *
 */
void resetMetricsObjectList()
{
resourceMetricsObjectList_t *resourceMetricsObjectListPtr;
 mfResourceObject_t *resourceMetricsObjectPtr ;

 resourceMetricsObjectListPtr = resourceMetricsObjectListHead;

 while (resourceMetricsObjectListPtr != NULL) {
   resourceMetricsObjectPtr = resourceMetricsObjectListPtr->resourceMetricsObjectPtr;
   resetMetricObject(resourceMetricsObjectPtr);
   resourceMetricsObjectListPtr = resourceMetricsObjectListPtr->next;
 }
}





/*
 * Function: disableMetricObjectList
 *
 *
 *  disable the metric Object List
 *
 *
 */
void EnableDisableMetricsObjectList(long long value)
{
resourceMetricsObjectList_t *resourceMetricsObjectListPtr;
 mfResourceObject_t *resourceObjectPtr ;
 mfMetrics_t *pmetric;


 resourceMetricsObjectListPtr = resourceMetricsObjectListHead;

 while (resourceMetricsObjectListPtr != NULL) {
   resourceObjectPtr = resourceMetricsObjectListPtr->resourceMetricsObjectPtr;
   pmetric = (mfMetrics_t *) resourceObjectPtr->resource;
   pmetric->isMonitoringEnabled = value;
   resourceMetricsObjectListPtr = resourceMetricsObjectListPtr->next;
 }
}




/*
 * Function: mfCbSetInstrumManagement
 *
 *
 *  Instrum Management setter callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */

int mfCbSetInstrumManagement (void *resource,
			      char *attributeName,
			      void *value) 
{
  mfInstrumManagement_t *instrumManagementPtr;
  instrumManagementPtr = (mfInstrumManagement_t *) resource ;

  if (strcmp(attributeName,"enableMonitoring") == 0) {
    instrumManagementPtr->MonitoringEnabled = *(long long *) value;
    resetMetricsObjectList();
    return MF_SUCCESS; 
  }

  if (strcmp(attributeName,"LoggingLevel") == 0) {
    instrumManagementPtr->LoggingLevel = *(long long *) value ;
    return MF_SUCCESS; 
  }
  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST ;
}
 





/*
 * Function:  mfCbInvokeInstrumManagement
 *
 *
 *  external invoker  callback
 *
 * return Code:
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST
 * MF_SUCCESS
 */
int mfCbInvokeInstrumManagement(void *resource, 
				char* methodName,
				AttValue_array paramsArray, 
				String_array signatureArray,
				void *value, int *type)
{
  mfInstrumManagement_t *instrumManagementPtr;
  instrumManagementPtr = (mfInstrumManagement_t *) resource ;

  if (strcmp(methodName,"enableMonitoring") == 0) {
    instrumManagementPtr->MonitoringEnabled = 1;
    resetMetricsObjectList();
    EnableDisableMetricsObjectList(MF_ENABLED);
    return MF_SUCCESS; 
  }

  if (strcmp(methodName,"disableMonitoring") ==0) {
    instrumManagementPtr->MonitoringEnabled = 0;
    EnableDisableMetricsObjectList(MF_DISABLED);
    return MF_SUCCESS;
  }


  if (strcmp(methodName,"LoggingLevel") == 0) {
    instrumManagementPtr->LoggingLevel = *(long long *) value ;
    return MF_SUCCESS; 
  }


  DEBUG_PRINTF2("mfCbInvokeInstrumManagemnent methodName = %s\n", methodName);



  return MF_RESOURCE_ATTRIBUTE_NOT_EXIST;

}






/*
 * Function:  mfAddInstrumManagementListener
 *
 *
 *  Add Instrum Management Listener
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfAddInstrumManagementListener(mfResourceObject_t *resourceObject) 
{
  resourceMetricsObjectList_t *resourceMetricsObjectListPtr;

  resourceMetricsObjectListPtr = (resourceMetricsObjectList_t *) malloc(sizeof(resourceMetricsObjectList_t));
  bzero(resourceMetricsObjectListPtr, sizeof(resourceMetricsObjectList_t));

  if (resourceMetricsObjectListPtr == NULL) {
    return MF_FAILURE;
  }
  else {

    resourceMetricsObjectListPtr->resourceMetricsObjectPtr = resourceObject;

    if (resourceMetricsObjectListHead == NULL) {
      resourceMetricsObjectListHead = resourceMetricsObjectListPtr;
    }
    else {
      
      resourceMetricsObjectListPtr->next=resourceMetricsObjectListHead;
      resourceMetricsObjectListHead = resourceMetricsObjectListPtr;
    }
    return MF_SUCCESS;
  }

} 


/*
 * Function:  mfRegisterTransactionMetrics
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterResourceTransactionMetrics(/*in */  mfResourceObject_t *resourceObject,
					 /*in */  char *transactionTypeName,
					 /*in */  char *uri,
					 /*in */  mfTransactionMetrics_t *transactionMetrics,
					 /*in */  int objectNameSize,
					 /*out */ char *objectName,
					 /*out */ unsigned short *transactionId )
{
  int rc;
  void *value;
  int type;

  bzero(resourceObject,sizeof(mfResourceObject_t));
  rc = mfGetTransactionMetricsName(transactionTypeName,objectName,objectNameSize);

  if (rc != MF_SUCCESS) {
    DEBUG_PRINTF2("mfRegisterTransactionMetrics failed in mfGetTransactionMetricsName %d\n",rc);
    return rc;
  }

  
  resourceObject->resource = (void *) transactionMetrics;
  resourceObject->getter = &mfCbGetTransactionMetrics;
  resourceObject->setter = &mfCbSetTransactionMetrics ;
  resourceObject->invoker = &mfCbInvokeTransactionMetrics;
  transactionMetrics->reset = &mfResetTransactionMetrics;

  rc = mfRegisterTransaction(transactionTypeName,
			     uri,
			     transactionMetrics,
			     transactionId);
  if (rc <0) {
    DEBUG_PRINTF2("mfRegisterTransactionMetrics failed in mfRegisterTransaction %d\n",rc);
    return rc;
  }



 
 
  rc = mfRegisterResource(resourceObject,objectName);
  if (rc != MF_SUCCESS) {
    DEBUG_PRINTF2("mfRegisterTransactionMetrics failed in mfRegisterResource %d\n",rc);
    return rc;
  }  
 

  rc = mfGetResourceAttributePtr(objectName,"isMonitoringEnabled",&value,&type);
 if (rc != MF_SUCCESS) {
    DEBUG_PRINTF2("mfRegisterTransactionMetrics failed in  mfGetResourceAttributePtr  %d\n",rc);
    return rc;
  }  

 mfSetIsMonitoringEnabledPointer((long long*) value, *transactionId );



 rc = mfAddInstrumManagementListener (resourceObject);
 if (rc != MF_SUCCESS) {
    DEBUG_PRINTF2("mfRegisterTransactionMetrics failed in  mfAddInstrumManagementListener %d\n",rc);
    return rc;
  }  
  return rc;

}


/*
 * Function:  mfRegisterResourceMetrics
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterResourceMetrics(/*in */  mfResourceObject_t *resourceObject,
			      /*in */  char *name, 
			      /*in */  mfResourceMetrics_t *resourceMetrics,
			      /*in */  int objectNameSize,
			      /*out */ char *objectName)
{
  int rc;

  bzero(resourceObject,sizeof(mfResourceObject_t));
  rc = mfGetResourceMetricsName(name,objectName,objectNameSize);

  if (rc != MF_SUCCESS) {
    return rc;
  }

  resourceObject->resource = (void *) resourceMetrics;
  resourceObject->getter = &mfCbGetResourceMetrics;
  resourceObject->setter = &mfCbSetResourceMetrics ;
  resourceObject->invoker = &mfCbInvokeResourceMetrics;
  resourceMetrics->reset = &mfResetResourceMetrics;

  
  rc = mfRegisterResource((mfResourceObject_t *) resourceObject,objectName);
  if (rc != MF_SUCCESS) {
    return rc;
  }  
  rc = mfAddInstrumManagementListener (resourceObject);
  
  return rc;

}




/*
 * Function:  mfRegisterResourceMetrics
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterStatus(/*in */  mfResourceObject_t *resourceObject,
		     /*in */  mfStatus_t *resourceStatus,
		     /*in */  int objectNameSize,
		     /*out */ char *objectName)
{
  int rc;
  

  
  rc = mfGetStatusName(objectName,objectNameSize);
  if (rc != MF_SUCCESS) {
    return rc;
  }

  resourceObject->resource = (void *) resourceStatus;
  resourceObject->getter = &mfCbGetStatus;
  resourceObject->setter = &mfCbSetStatus ;
  resourceObject->invoker = &mfCbInvokeStatus;

  
  rc = mfRegisterResource(resourceObject,objectName);
  if (rc != MF_SUCCESS) {
    return rc;
  }  
  
  
  return rc;

}











/*
 * Function:  mfRegisterExternalResourceMetrics
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterExternalResourceMetrics(/*in */  mfResourceObject_t *resourceObject,
				      /*in */  char *name, 
				      /*in */  mfExternalResourceMetrics_t *externalResourceMetrics,
				      /*in */  int objectNameSize,
				      /*out */ char *objectName)
{
  int rc;


  bzero(resourceObject,sizeof(mfResourceObject_t));
  rc = mfGetExternalResourceMetricsName(name,objectName,objectNameSize);
  if (rc != MF_SUCCESS) {
    return rc;
  }
 
 
  resourceObject->resource = (void *) externalResourceMetrics;
  resourceObject->getter = &mfCbGetExternalResourceMetrics;
  resourceObject->setter = &mfCbSetExternalResourceMetrics ;
  resourceObject->invoker = &mfCbInvokeExternalResourceMetrics;
  externalResourceMetrics->reset = &mfResetExternalResourceMetrics;

  
  rc = mfRegisterResource(resourceObject,objectName);
  if (rc != MF_SUCCESS) {
    return rc;
  }  
  rc = mfAddInstrumManagementListener (resourceObject);
  
  return rc;

}








/*
 * Function:  mfRegisterInstrumConfig
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterInstrumConfig(/*in */  mfResourceObject_t *resourceObject,
			    /*in */  void *resource,
			    /*in */  int objectNameSize,
			    /*out */ char *objectName)
{
  int rc;


  bzero(resourceObject,sizeof(mfResourceObject_t));
  rc = mfGetInstrumConfigName(objectName,objectNameSize);
  if (rc != MF_SUCCESS) {
    return rc;
  }
 
  resourceObject->resource = (void *) resource;
  resourceObject->getter = &mfCbGetConfig;
  resourceObject->setter = &mfCbSetConfig ;
  resourceObject->invoker = NULL;

  
  rc = mfRegisterResource((mfResourceObject_t *) resourceObject,objectName);
  
  return rc;

}





/*
 * Function:  mfRegisterInstrumManagement
 *
 *
 *  Register the transaction metrics resource and
 *  corresponding metrics object
 *
 * return Code:
 * MF_FAILURE
 * MF_SUCCESS
 */
int mfRegisterInstrumManagement(/*in */  mfResourceObject_t *resourceObject,
				/*in */  void *resource,
				/*in */  int objectNameSize,
				/*out */ char *objectName)
{
  int rc;


  bzero(resourceObject,sizeof(mfResourceObject_t));
  rc = mfGetInstrumManagementName(objectName,objectNameSize);
  if (rc != MF_SUCCESS) {
    return rc;
  }
 
  
  resourceObject->resource = (void *) resource;
  resourceObject->getter = &mfCbGetInstrumManagement;
  resourceObject->setter = &mfCbSetInstrumManagement ;
  resourceObject->invoker = &mfCbInvokeInstrumManagement;

  
  rc = mfRegisterResource((mfResourceObject_t *) resourceObject,objectName);
  
  return rc;

}



/*
 * mfGetCurrentTimeMillis
 *
 * return the current time in millisecond
 *
 */
long long mfgetCurrentTimeMillis()
{
  int rc;
  struct timeval tp;
  long long value;

  rc = gettimeofday(&tp, NULL);
  if (rc == 0) {
    value = (((long long) tp.tv_sec) * 1000) + (long long) (tp.tv_usec / 1000) ;
    return value;
  }
  else
    return -1;
}
