/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 * 
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)mfRserver.c 1.12     04/07/05 SMI"
 *
 */



/*
 *
 * mfRserver.c - Resource server
 *                                                                  * 
 */

#include <pthread.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <malloc.h>
#include <strings.h>
#include <plhash.h>



#include "mfConstant.h"
#include "mfRserver.h"
#include "mfSimpleResource.h"


#define SIMPLE_ATTRIBUTE_VALUE 255


static PLHashTable *resourceServer;
static simpleResourceList_t *simpleResourceListHead;


/* static variables */
static int resourceServerCreated  = 0;       /* resource server boolean flag */
static pthread_mutex_t mutexResourceServer;  /* mutex resource server */

extern void initResourceServer();

#ifdef sun
#pragma init (initResourceServer)
#endif

#ifdef linux
__attribute__ ((constructor)) void init() {
    initResourceServer();

}
#endif




void initResourceServer() {
  resourceServer = NULL;
  resourceServerCreated = 0;
  simpleResourceListHead = NULL;
}




/*
 * Function: mfCreateResourceServer 
 *
 * This function is creating a "C" resource server 
 *
 * 
 *  return code:
 *     MF_SUCCESS (success)  
 *     MF_RESOURCE_SERVER_EXIST   (resource server already exist)
 *     MF_FAILURE (resource server not created, errno positioned)    
 *          ENOMEM (cf malloc)
 *          EAGAIN (cf malloc)
 */
int mfCreateResourceServer() 
{
  int rc;
 
   
  pthread_mutex_lock(&mutexResourceServer);
  DEBUG_PRINTF("mfCreateResourceServer\n");

  if (resourceServerCreated == 1) {
    DEBUG_PRINTF("mfCreateResourceServer:  ResourceServer  already exists\n");
    rc = MF_RESOURCE_SERVER_EXIST;
  }
  else {
    resourceServer = PL_NewHashTable( MF_NB_RESOURCE_BUCKETS, 
				      PL_HashString, 
				      PL_CompareStrings,
                           	      PL_CompareValues,
				      NULL,
				      NULL);


    if (resourceServer == NULL) {
      DEBUG_PRINTF("mfCreateResourceServer failed\n");
      rc = MF_FAILED;
    }
    else {
      rc = MF_SUCCESS;
      resourceServerCreated = 1;
      
    }
                                                                 
  }
  pthread_mutex_unlock(&mutexResourceServer);
  return rc;
}




/*
 * lookup_resourceserver 
 *
 * This function is performing a lookup the
 * resource server with the  resourceName (name of the 
 * resource object ) as key.
 *
 * It returns as out parameter a void pointer to the value 
 * if the resource object has been found within the resource
 * server table
 * 
 * The return values are:
 *     MF_RESOURCE_SERVER_NOT_EXIST
 *     MF_RESOURCE_OBJECT_EXIST 
 *     MF_RESOURCE_OBJECT_NOT_EXIST
 *  
 */
int lookupResourceServer(/*in */ char * resourceName, /*out */ void **value )
{
  int rc;
  void *value1;


  if (resourceName == NULL) {
    DEBUG_PRINTF("lookupResourceServer: resourceName parameter is null\n");
    return MF_NULL_PARAM;
  }



  pthread_mutex_lock(&mutexResourceServer);
  if (resourceServerCreated == 0) {
    DEBUG_PRINTF("lookupResourceServer: resource server does not exist \n");
    

    rc =  MF_RESOURCE_SERVER_NOT_EXIST;
  } 
  else {
    value1 = PL_HashTableLookup(resourceServer, resourceName);	
    if (value1 != NULL) {
      *value = (char *) value1;
      DEBUG_PRINTF2("lookupResourceServer %s found\n",resourceName);
      rc = MF_RESOURCE_OBJECT_EXIST;
    }

    else {
      DEBUG_PRINTF2("lookupResourceServer %s not found\n",resourceName);
      rc = MF_RESOURCE_OBJECT_NOT_EXIST;
    }
  }
  pthread_mutex_unlock(&mutexResourceServer);
  
  return rc;
}



/*
 * Function: mfRegisterResource 
 *
 *
 *
 *  This function is used to resgister a register object within 
 *  the resource table
 *
 *
 *  input: resource_object 
 *  
 *  return code:
 *               MF_SUCCESS (success)
 *               MF_RESOURCE_SERVER_NOT_EXIST (resource server does not exist)
 *               MF_RESOURCE_OBJECT_EXIST (The resource object already exists)
 *               MF_NOMORE  (cannot allocate more resources)
 *
 *
 */ 
int mfRegisterResource(/*in */ mfResourceObject_t *resourceObject, 
		       /*in */ char *resourceName)
{
  int rc;
  PLHashEntry *entry;
  void *value;

  /* fix of bugid 5025887 
     Robustness of communication API:
     jesmf_register_resource makes application crash
     when null pointer parameter */


  if (resourceObject == NULL) {
    DEBUG_PRINTF("mfRegisterResource: resourceObject parameter is null\n");
    return MF_NULL_PARAM;
  }



  if (resourceName == NULL) {
    DEBUG_PRINTF("mfRegisterResource: resourceName parameter is null\n");    
    return MF_NULL_PARAM;
  }

  /* perform the lookup of the resource */
  rc = lookupResourceServer(resourceName,&value);
   
  /* test if resource to be added to resource server */
  if ((rc != MF_RESOURCE_SERVER_NOT_EXIST) && 
      (rc != MF_RESOURCE_OBJECT_EXIST)){
    pthread_mutex_lock(&mutexResourceServer);
    entry = PL_HashTableAdd(resourceServer, 
			    resourceName, 
			    resourceObject);
    pthread_mutex_unlock(&mutexResourceServer);
  
  
    if (entry == NULL) {
      DEBUG_PRINTF2("mfRegisterResource failed %s\n",resourceName);
      rc = MF_NOMORE;
    }
    else {
      rc = MF_SUCCESS;
    }
  }
  return rc;
}



/*
 * Function: AllocateSimpleResource  
 *
 * This functions allocates a simple Resource  
 * strcuture to copy the attaibute Name into it
 * 
 *
 */
simpleResource_t *AllocateSimpleResource(char *attributeName, int type, void *value) 
{
  simpleResource_t *simpleResourcePtr ;
  simpleResourcePtr = NULL;
 
  simpleResourcePtr = (simpleResource_t *) malloc(sizeof(simpleResource_t));
  
  if (simpleResourcePtr !=NULL) {
    bzero(simpleResourcePtr,sizeof(simpleResource_t));
    simpleResourcePtr->type = type ; 
    simpleResourcePtr->value = value;
    
    /* fix bugId: 5070434
       Segmentation fault*/
       
    simpleResourcePtr->attributeName = (char *) malloc(strlen(attributeName)+1);
    bzero(simpleResourcePtr->attributeName,(strlen(attributeName)+1));  
    strcpy(simpleResourcePtr->attributeName, attributeName);
    simpleResourcePtr->next = NULL;
  }
  return simpleResourcePtr;
}



/*
 * Function:  mfCbGetSimpleResource
 *
 * This primitive is the getter callback  
 * for simple resource
 *
 *
 * return values:
 *     -MF_SUCCESS
 *     -MF_ATTRIBUTE_NOT_EXIST
 *     -MF_FAILURE
 *
 */
int mfCbGetSimpleResource(/* in  */ void *resource,
			  /* in  */ char *attributeName,
			  /* out */ void **value,
			  /* out */ int *type)
{
  simpleResource_t *simpleResourcePtr;
  int rc;
  rc = MF_FAILURE;

  simpleResourcePtr = (simpleResource_t *) resource;
  while (simpleResourcePtr !=NULL) {

    if (strcmp(simpleResourcePtr->attributeName,attributeName) != 0) {
      simpleResourcePtr = simpleResourcePtr->next;
    }
    else {
      *value = simpleResourcePtr->value;
      *type =  simpleResourcePtr->type;
      rc = MF_SUCCESS;
      break;
    }    
  }
  
  if (simpleResourcePtr == NULL) {
    rc = MF_RESOURCE_ATTRIBUTE_NOT_EXIST;
  }

  return rc;

}


/*
 * Function: lookupSimpleResourceList 
 *
 * lookup a resource obejct name within the 
 * simple resource object list
 *
 */
int lookupSimpleResourceList(char *resourceName) 
{
 simpleResourceList_t *simpleResourceListPtr;
 int found;

  
 found = 0;
 simpleResourceListPtr = simpleResourceListHead ; 

 while (simpleResourceListPtr != NULL) {
   if (strcmp(simpleResourceListPtr->resourcename,resourceName) == 0) {
     found = 1;
     break;
   }
   simpleResourceListPtr = simpleResourceListPtr->next;
 }

 if (found == 0) {
   return MF_FAILURE;
 }
 else {
   return MF_SUCCESS;
 }



}


/*
 * Function: lookupSimpleResourceList 
 *
 * add a simple resource to the simple resource
 * object list
 *
 */
int AddSimpleResourceList(char *resourceName) 
{
  simpleResourceList_t *simpleResourceListPtr;
  

  if (resourceName == NULL) {
    return MF_NULL_PARAM;

  }

  simpleResourceListPtr = (simpleResourceList_t *) malloc(sizeof(simpleResourceList_t));
  if (simpleResourceListPtr == NULL) {
    return MF_FAILURE;
  }
  
  bzero(simpleResourceListPtr,sizeof(simpleResourceList_t));
  
  simpleResourceListPtr->resourcename= (char *) malloc(strlen(resourceName)+1);
  if (simpleResourceListPtr->resourcename == NULL) {
    return MF_FAILURE;
  }
  
  bzero(simpleResourceListPtr->resourcename,strlen(resourceName)+1);
  strcpy(simpleResourceListPtr->resourcename,resourceName);

  if (simpleResourceListHead == NULL) {
    simpleResourceListHead = simpleResourceListPtr;
  }
  else {
    simpleResourceListPtr->next=simpleResourceListHead;
    simpleResourceListHead = simpleResourceListPtr;
  }
  
  return MF_SUCCESS;
}


/*
 * Function: mfRegisterSimpleResource 
 *
 *
 * This primitive is used to register a simple 
 * resource within the resource server which are 
 * of elementary type (char, short, int, long, 
 * long long, float, double, string). 
 *
 * This primitive alleviates the user to use a 
 * resource object, and thus having to write/specify 
 * getter/invoker callbacks. This primitive has to 
 * be used when the user wants to register a basic 
 * type variable. Internally is created a proper 
 * resource object.
 * 
 * It is possible to register several simple resources 
 * within the same resource object, ny this primtive 
 * each time with the same resourceName, but different attributeName . 
 * 
 * Parameters:
 * resourceAddr: is the address of the resource to register. 
 * resourceName: is the name of the resource object name 
 * attributeName: is the name of the attribute 
 * type is the type of the object:
 *              MF_TYPE_BOOLEAN  (boolean)
 *              MF_TYPE_CHAR  (char)
 *              MF_TYPE_DOUBLE  (double)
 *              MF_TYPE_FLOAT  (float)
 *              MF_TYPE_LONG    (long) 
 *              MF_TYPE_LONGLONG (long long
 *              MF_TYPE_SHORT (short)
 *              MF_TYPE_INT (int) 
 *              MF_TYPE_STRING (string)
 * Return Value:
 * MF_SUCCESS (success)
 * MF_RESOURCESERVER_NOTEXIST (resource server does not exist)
 * MF_RESOURCEOBJECT_NOTEXIST  (resource object does not exists)
 * MF_ATTRIBUTE_EXIST (attribute ha s already been registered within the 
 * resource object
 * MF_NOMORE  (cannot allocate more resources)
 */
int mfRegisterSimpleResource(/*in*/ void *resourceAddr,
				    /*in */ char *resourceName, 
				    /*in */ char  *attributeName,  
				    /*in */ int type)
{
 

  int rc;
  PLHashEntry *entry;
  void *value;
  mfResourceObject_t *resourceObjectPtr;
  simpleResource_t *simpleResourcePtr;
    


  if (resourceAddr == NULL) {
    DEBUG_PRINTF("mfRegisterResource: resourceAddr parameter is null\n");    
    return MF_NULL_PARAM;
  }


  if (resourceName == NULL) {
    DEBUG_PRINTF("mfRegisterResource: resourceObject parameter is null\n");
    return MF_NULL_PARAM;
  }



  if (attributeName == NULL) {
    DEBUG_PRINTF("mfRegisterResource: resourceName parameter is null\n");    
    return MF_NULL_PARAM;
  }

  if (type <=0) {
    DEBUG_PRINTF2("mfRegisterResource: type parameter is inconsistent (%d)\n",
	   type);
    return MF_NULL_PARAM;
  }



  /* perform the lookup of the resource */
  rc = lookupResourceServer(resourceName,(void *) &resourceObjectPtr);
   
  /* test if resource to be added to resource server */
  if (rc == MF_RESOURCE_SERVER_NOT_EXIST) {
    return MF_RESOURCE_SERVER_NOT_EXIST;
  }
  
  pthread_mutex_lock(&mutexResourceServer);
  

  /* the Resource object does not exist and neds to be created */
  if (rc != MF_RESOURCE_OBJECT_EXIST){
    resourceObjectPtr = (mfResourceObject_t *) malloc(sizeof(mfResourceObject_t));
    bzero(resourceObjectPtr,sizeof(mfResourceObject_t));
    resourceObjectPtr->getter = mfCbGetSimpleResource; 
    entry = PL_HashTableAdd(resourceServer, 
			    resourceName, 
			    resourceObjectPtr);   
    if (entry == NULL) {
      DEBUG_PRINTF2("mfRegisterSimpleResource failed %s\n",resourceName);
      rc = MF_NOMORE;
      pthread_mutex_unlock(&mutexResourceServer);
      return rc;
    }
    AddSimpleResourceList(resourceName);
  }
  else {
    /* check teh resource that teh resource object is
       not in use (within a stanadrd resource 
     */
    rc = lookupSimpleResourceList(resourceName);
    if (rc == MF_FAILURE) {
      DEBUG_PRINTF2("mfRegisterSimpleResource failed (resource alredy exists) %s\n",resourceName);
      pthread_mutex_unlock(&mutexResourceServer);
      return MF_RESOURCE_OBJECT_EXIST;
    }
    

    /* check that the attributename is not in use within 
       a simple resource */
    rc = mfCbGetSimpleResource(resourceObjectPtr->resource,
			       attributeName,
			       &value,
			       &type);    
    if (rc != MF_RESOURCE_ATTRIBUTE_NOT_EXIST) {
       pthread_mutex_unlock(&mutexResourceServer);
      DEBUG_PRINTF2("mfRegisterSimpleResource failed (attribute already exists) %s\n",attributeName);
       return MF_RESOURCE_ATTRIBUTE_EXIST;
    }

  }
 
  /* Allocate the Simple Resource */
  simpleResourcePtr = AllocateSimpleResource(attributeName,type, resourceAddr);
  if (simpleResourcePtr == NULL) {
    rc = MF_FAILURE;
  }
  else {
    /* Hook the simple Resource at the end of 
       the resource field of the ResourceObject
    */
    if (resourceObjectPtr->resource != NULL) {
      simpleResourcePtr->next = (simpleResource_t *) resourceObjectPtr->resource;
      resourceObjectPtr->resource = (void *) simpleResourcePtr ;
    }
    else {
      resourceObjectPtr->resource = (void *) simpleResourcePtr;
    }
    rc = MF_SUCCESS;
  }

  pthread_mutex_unlock(&mutexResourceServer);  
  return rc;


}

int mfUnRegisterResource(/*in */char *resourceName)
{

  return MF_FAILURE;
}

int mfUnRegisterSimpleResource(/*in */ char *resourceNname, 
			       /*in */ char *attributeName)
{
return MF_FAILURE;
}





/*
 * Function: mfGetResourceAttributePtr 
 *
 *
 *  This primitive is returning a pointer towards the attribute value 
 *  of a given resource object identified by resourceName, 
 *  with an attribute attributeName.
 *
 *  input: resource_object 
 *  
 *  return code
 *  MF_SUCCESS (success)
 *  MF_NULL_PARAM
 *  MF_FAILURE
 *  MF_RESOURCE_OBJECT_NOT_EXIST ( the resource object does not exist)
 *  MF_RESOURCE_ATTRIBUTE_NOT_EXIST (the resource attribute does not exist)
 *  MF_GETTER_CALLBACK_NOT_EXIST (the seeter callback does not exist)
 *
 */ 
int mfGetResourceAttributePtr(/*in */ char *resourceName,
			      /*in */ char *attributeName, 
			      /*out */ void **value, 
			      /*out*/ int *type)
{

  mfResourceObject_t *resourceObjectPtr;
  int rc;
  


  if (resourceName == NULL) {
    DEBUG_PRINTF("mfGetResourceAttributePtr: resourceName null parameter\n");
    return MF_NULL_PARAM;
  }

  /* fix of bugId 5026611:
     application crashes with NULL attribute parameter */

  if (attributeName == NULL) {

    DEBUG_PRINTF("mfGetResourceAttributePtr: attribute name null parameter\n");
    return MF_NULL_PARAM;
  }

  
  rc = lookupResourceServer(resourceName,(void *) &resourceObjectPtr);

  if (rc != MF_RESOURCE_OBJECT_EXIST) {
    DEBUG_PRINTF2("mfGetResourceAttributePtr: resourceName %s not found\n", resourceName);
    return rc;
  } 



  /* fix of bugId 5022087 
     Communication API: inconsistency between C part and java 
     in getAttribute() call
     the exception raised is InstanceNotFound instead of AttributeNotFound.
     Problem is that, in Service_server, when attribute has not been found, 
     the error returned is ATTR_TYPE_OBJECT_NOT_EXIST which corresponds 
     in the java part to AttValue.OBJECT_NOT_EXIST. But this error is 
     interpreted in java as  if the resource has not been found.

  */
  if (resourceObjectPtr == NULL) {
    DEBUG_PRINTF2("mfGetResourceAttributePtr: resourceName %s not found\n", 
	   resourceName);
    return MF_FAILURE;
  }


  
  


  if (resourceObjectPtr->getter == NULL) {
    DEBUG_PRINTF2("mfGetResourceAttributePtr: getter callback NULL for resource %s \n", resourceName);
    return MF_RESOURCE_GETTER_CALLBACK_NOT_EXIST;    
  }


   /* DEBUG_PRINTF2("mfGetResourceAttributePtr: resourceObjectPtr %x\n",resourceObjectPtr); */



  if (resourceObjectPtr->resource == NULL) {
    DEBUG_PRINTF("mfGetResourceAttributePtr: resource not exist \n");
    return MF_RESOURCE_NOT_EXIST;    
  }




  /*invoke resource callback to access to the resource */
  rc = (resourceObjectPtr->getter)(resourceObjectPtr->resource,
				   attributeName,
				   value,
				   type) ;

 
  return rc;
}




/*
 * Function: mfSetResource 
 *
 *
 * This primitive is setting  the attribute value 
 * of a given resource object identified by resourceName,  
 * and attributeName
 *
 * Parameters:
 * resourceName: the name of the resource object
 * attributeName: the name of the attribute on which one to perform the lookup.
 * value: void pointer to the value to set for the attribute
 * attributeType: type of the attribute value of the value to be set
 *
 * Return Value:
 * MF_SUCCESS (success)
 * MF_FAILURE
 * MF_RESOURCE_OBJECT_NOT_EXIST ( the resource object does not exist)
 * MF_RESOURCE_ATTRIBUTE_NOT_EXIST (the resource attribute does not exist)
 * MF_SETTER_CALLBACK_NOT_EXIST (the setter callback does not exist)
 *
 */
int mfSetResourceAttribute(/*in */ char *resourceName, 
			   /*in */ char *attributeName, 
			   /*in */ void *value)
{
  mfResourceObject_t *resourceObjectPtr;
  int rc;
  


  if (resourceName == NULL) {
    DEBUG_PRINTF("mfSetResourceAttribute: resourceName null parameter\n");
    return MF_NULL_PARAM;
  }

  /* fix of bugId 5026611:
     application crashes with NULL attribute parameter */

  if (attributeName == NULL) {
    DEBUG_PRINTF("mfSetResourceAttribute: attribute name null parameter\n");
    return MF_NULL_PARAM;
  }

  
  rc = lookupResourceServer(resourceName,(void *) &resourceObjectPtr);

  if (rc != MF_RESOURCE_OBJECT_EXIST) {
    DEBUG_PRINTF2("mfSetResourceAttributePtr: resourceName %s not found\n", 
	   resourceName);
    return rc;
  } 



  /* fix of bugId 5022087 
     Communication API: inconsistency between C part and java in 
     getAttribute() call

     the exception raised is InstanceNotFound instead of AttributeNotFound.
     Problem is that, in Service_server, when attribute has not been found, 
     the error returned is ATTR_TYPE_OBJECT_NOT_EXIST which corresponds 
     in the java part to AttValue.OBJECT_NOT_EXIST. But this error is 
     interpreted in java as if the resource has not been found.

  */
  if (resourceObjectPtr == NULL) {

    DEBUG_PRINTF2("mfSetResourceAttribute: resourceName %s not found\n", 
	   resourceName);
    return MF_FAILURE;
  }


  if (resourceObjectPtr->setter == NULL) {
    DEBUG_PRINTF2("mfSetResourceAttribute: getter callback NULL for resource %s \n",
	   resourceName);
    return MF_RESOURCE_SETTER_CALLBACK_NOT_EXIST;    
  }


  if (resourceObjectPtr->resource == NULL) {
    DEBUG_PRINTF("mfGetResourceAttributePtr: resource not exist \n");
    return MF_RESOURCE_NOT_EXIST;    
  }



  /*invoke resource callback to access to the resource */
  rc = (resourceObjectPtr->setter)(resourceObjectPtr->resource,
				   attributeName,
				   value) ;

 
  return rc;
}







/*
 * Function: mfInvokeResource 
 *
 *
 * This primitive is invoking the method methodName 
 * on the resource object with the paramters param 
 * which signature is defined in signature.
 * 
 *
 * Parameters:
 * resourceName: the name of the resource object
 * methodName: the name of the method to invoke
 * params: parmeters passed to the method
 * signature: signature of the params
 * value: value returned by the method 
 *         (The value needs to be of a basic type: char,short,long,
 *	  flota,double,long long string)
 * type: type of the of the returned value
 *
 * Return Value:
 * MF_SUCCESS (success)
 * MF_FAILURE
 * MF_RESOURCE_OBJECT_NOT_EXIST ( the resource object does not exist)
 * MF_RESOURCE_METHOD_NOT_EXIST (the resource attribute does not exist)
 * MF_INVOKER_CALLBACK_NOT_EXIST (the invoker callback does not exist)
 */ 
int mfInvokeResource(/*in */ char *resourceName, 
			    /*in */  char *methodName, 
			    /*in */ AttValue_array paramsArray, 
			    /*in */ String_array signatureArray, 
			    /*out */ void *value,  
			    /*out*/ int *type)
{
  mfResourceObject_t *resourceObjectPtr;
  int rc;  

  if (resourceName == NULL) {
    DEBUG_PRINTF("mfInvokeResource: resourceName null parameter\n");
    return MF_NULL_PARAM;
  }

  /* fix of bugId 5026611:
     application crashes with NULL attribute parameter */

  if (methodName == NULL) {
    DEBUG_PRINTF("mfInvokeResource: attribute name null parameter\n");
    return MF_NULL_PARAM;
  }

  
  rc = lookupResourceServer(resourceName,(void *) &resourceObjectPtr);

  if (rc != MF_RESOURCE_OBJECT_EXIST) {
    DEBUG_PRINTF2("mfInvokeResource: resourceName %s not found\n", resourceName);
    return rc;
  } 


  if (resourceObjectPtr == NULL) {
    DEBUG_PRINTF2("mfInvokeResource: resourceName %s not found\n", resourceName);
    return MF_FAILURE;
  }


  if (resourceObjectPtr->invoker == NULL) {
    DEBUG_PRINTF2("mfInvokeResource: invoker callback NULL for resource %s \n", 
	   resourceName);
    return MF_RESOURCE_GETTER_CALLBACK_NOT_EXIST;    
  }



  if (resourceObjectPtr->resource == NULL) {
    DEBUG_PRINTF("mfGetResourceAttributePtr: resource not exist \n");
    return MF_RESOURCE_NOT_EXIST;    
  }




  /*invoke resource callback to access to the resource */
  rc = (resourceObjectPtr->invoker)(resourceObjectPtr->resource,
				    methodName,
				    paramsArray,
				    signatureArray,
				    value,
				    type) ;

 
  return rc;



}
