/*
 * Copyright 2004 Sun Microsystems, Inc.  All rights reserved.
 * This software is the proprietary information of Sun Microsystems, Inc.
 * Use is subject to license terms.
 *
 * Copyright 2004 Sun Microsystems, Inc.  Tous droits rservs.
 * Ce logiciel est proprit de Sun Microsystems, Inc.
 * Distribu par des licences qui en restreignent l'utilisation.
 *
 * ident        "@(#)CIBTimeStatisticImpl.java 1.11     04/07/15 SMI"
 *
 */

package com.sun.mfwk.cib.sdk.statistics;

import java.util.logging.Logger;
import com.sun.mfwk.util.log.MfLogService;

import com.sun.mfwk.cib.statistics.CIBTimeStatistic;
import com.sun.mfwk.cib.statistics.CIBStatistic;

import javax.management.openmbean.CompositeDataSupport;
import javax.management.openmbean.CompositeData;

import javax.management.openmbean.CompositeType;
import javax.management.openmbean.OpenType;
import javax.management.openmbean.SimpleType;
import javax.management.openmbean.OpenDataException;

/**
 * Provides a default implementation of a CIBTimeStatistic
 */
public class CIBTimeStatisticImpl
    extends CIBStatisticImpl
    implements CIBTimeStatistic {

  static final Logger logger = MfLogService.getLogger("cib-sdk");
  static final String className = "CIBTimeStatisticImpl";

  private long myCount = RESET_VALUE;
  private long myMinTime = RESET_VALUE;
  private long myMaxTime = RESET_VALUE;
  private long myTotalTime = RESET_VALUE;
  private long myTotalSquaredTime = RESET_VALUE;

  private static String[] itemNames = {
      "Name",
      "Description",
      "Unit",
      "LastSampleTime",
      "StartTime",
      "MinTime",
      "MaxTime",
      "TotalTime",
      "TotalSquaredTime",
      "Count"};

  private static String[] itemDescriptions = {
      "Name",
      "Description",
      "Unit",
      "LastSampleTime",
      "StartTime",
      "MinTime",
      "MaxTime",
      "TotalTime",
      "TotalSquaredTime",
      "Count"};

  private static OpenType[] itemTypes = {
      SimpleType.STRING,
      SimpleType.STRING,
      SimpleType.STRING,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG,
      SimpleType.LONG};

  private static CompositeType statisticType = null;

  /**
   *
   * @param name String
   * @param description String
   */
  public CIBTimeStatisticImpl(String name, String description) {
    super(name, description, "MILLISECONDS");
    logger.exiting(className, "Constructor");
  }

  /**
   * Set the count on the statistic object
   * @param count long
   */
  public void setCount(long count) {
    logger.entering(className, "setCount", new Long(count));
    myCount = count;
    logger.exiting(className, "setCount");
  }

  /**
   * Set the MaxTime of the statistic object
   * @param maxTime long
   */
  public void setMaxTime(long maxTime) {
    logger.entering(className, "setMaxTime" + new Long(maxTime));
    myMaxTime = maxTime;
    logger.exiting(className, "setMaxTime");
  }

  /**
   * Set the mMinTime of the statistic object
   * @param minTime long
   */
  public void setMinTime(long minTime) {
    logger.entering(className, "setMinTime" + new Long(minTime));
    myMinTime = minTime;
    logger.exiting(className, "setMinTime");
  }

  /**
   * Set the totalTime of the statistic object
   * @param totalTime long
   */
  public void setTotalTime(long totalTime) {
    logger.entering(className, "setTotalTime" + new Long(totalTime));
    myTotalTime = totalTime;
    logger.exiting(className, "setTotalTime");
  }

  /**
   * Set the total of squared time of the statistic object
   * @param totalSquaredTime long
   */
  public void setTotalSquaredTime(long totalSquaredTime) {
    logger.entering(className,
                    "setTotalSquaredTime" + new Long(totalSquaredTime));
    myTotalSquaredTime = totalSquaredTime;
    logger.exiting(className, "setTotalSquaredTime");

  }

  /**
   * Get the count of the statistic object
   * @return long
   */
  public long getCount() {
    logger.exiting(className, "getCount" + new Long(myCount));
    return myCount;
  }

  /**
   * Get the MaxTime of the statistic object
   * @return long
   */
  public long getMaxTime() {
    logger.exiting(className, "getMaxTime" + new Long(myMaxTime));
    return myMaxTime;
  }

  /**
   * Get the MinTime of the statistic object
   * @return long
   */
  public long getMinTime() {
    logger.exiting(className, "getMinTime" + new Long(myMinTime));
    return myMinTime;
  }

  /**
   * Get the totalTime of the statistic object
   *
   * @return long
   */
  public long getTotalTime() {
    logger.exiting(className, "getTotalTime" + new Long(myTotalTime));
    return myTotalTime;
  }

  /**
   * Get the toal of the squared time of the statistic object
   *
   * @return long
   */
  public long getTotalSquaredTime() {
    logger.exiting(className,
                   "getTotalSquaredTime" + new Long(myTotalSquaredTime));
    return myTotalSquaredTime;
  }

  /**
   * Returns a composite data reprenting the statistic object
   * @return CompositeData
   */
  public CompositeData toCompositeData() throws OpenDataException {
    logger.entering(className, "toCompositeData");

    try {
      statisticType = new CompositeType("CIBTimeStatistic",
                                        "Common monitoring Time Statistic",
                                        itemNames,
                                        itemDescriptions,
                                        itemTypes);

      Object[] itemValues = {
          getName(),
          getDescription(),
          getUnit(),
          new Long(getLastSampleTime()),
          new Long(getStartTime()),
          new Long(getMinTime()),
          new Long(getMaxTime()),
          new Long(getTotalTime()),
          new Long(getTotalSquaredTime()),
          new Long(getCount())
      };

      CompositeDataSupport data = null;
      data = new CompositeDataSupport(statisticType, itemNames, itemValues);
      logger.exiting(className, "toCompositeData", data);
      return data;
    }
    catch (OpenDataException e) {
      logger.throwing(className, "toCompositeData", e);
      logger.exiting(className, "toCompositeData");
      throw e;
    }
  }

  /**
   * Returns a CIBStatistic object from a composite data
   * @param data CompositeData
   * @return CIBStatistic
   */
  public CIBStatistic fromCompositeData(CompositeData data) {
    logger.entering(className, "fromCompositeData", data);
    String name = (String) data.get("Name");
    String description = (String) data.get("Description");
    String unit = (String) data.get("Unit");
    Long lastSampleTime = (Long) data.get("LastSampleTime");
    Long startTime = (Long) data.get("StartTime");
    Long min = (Long) data.get("MinTime");
    Long max = (Long) data.get("MaxTime");
    Long total = (Long) data.get("TotalTime");
    Long totalSquared = (Long) data.get("TotalSquaredTime");
    Long count = (Long) data.get("Count");

    // Create a new statistic
    CIBTimeStatisticImpl statistic = new CIBTimeStatisticImpl(name,
        description);

    ( (CIBStatisticImpl) statistic).setLastSampleTime(lastSampleTime.longValue());
    ( (CIBStatisticImpl) statistic).setStartTime(startTime.longValue());

    statistic.setMinTime(min.longValue());
    statistic.setMaxTime(max.longValue());
    statistic.setTotalTime(total.longValue());
    statistic.setTotalSquaredTime(totalSquared.longValue());
    statistic.setCount(count.longValue());
    logger.exiting(className, "fromCompositeData", statistic);
    return statistic;

  }

  /**
   * Reset the statistic object
   */
  public void reset() {
    logger.entering(className, "reset");
    super.reset();
    myCount = RESET_VALUE;
    myMinTime = RESET_VALUE;
    myMaxTime = RESET_VALUE;
    myTotalTime = RESET_VALUE;
    myTotalSquaredTime = RESET_VALUE;
    logger.exiting(className, "reset");
  }

  /**
   * Returns a printable statistic object
   * @return String
   */
  public String toString() {
    String str = super.toString();
    String strMin = "MinTime=" + this.getMinTime() + super.strSeparator;
    String strMax = "MaxTime=" + this.getMaxTime() + super.strSeparator;
    String strTotal = "TotalTime=" + this.getTotalTime() + super.strSeparator;
    String strCount = "Count=" + this.getCount() + super.strSeparator;
    String strAcc = "AccumulatedSquaredTime=" + this.getTotalSquaredTime() +
        super.strSeparator;
    return new String(str + strMin + strMax + strTotal + strCount + strAcc);

  }

}
