
/*
 * Copyright (c) 2004
 * All rights reserved.
 * These functions provide support for circular log files.  A circfile file
 * contains the most recently added entries, where an entry is any opaque
 * chunk of data supplied by the application.  When the file becomes full,
 * new entries overwrite the oldest entries in a circular fashion.
 *
 * Each circfile file has a fixed size and is accessed using mmap for
 * efficiency. * 
 */

#ifndef _CIRCFILE_H_
#define _CIRCFILE_H_

/*
 * Support for on-disk circular log files. Each file contains the
 * N most recent entries, where an entry is any opaque chunk of
 * data supplied by the application. New entries overwrite the
 * oldest entries in a circular fashion.
 */

struct circfile;

#ifdef  __cplusplus
extern "C" {
#endif


/*
 * Open/create a new circfile.
 *
 * "maxent" sets the maximum number of entries the file can hold.
 * "maxdata" sets the maximum amount of entry data the file can hold.
 *
 * The size of the file will be approx.: maxdata + (8 * maxent) + 20
 *
 * If "path" already exists, it must be a valid circfile and the
 * "maxent" and "maxdata" arguments are ignored.
 *
 * "flags" may be zero, O_SHLOCK or O_EXLOCK.
 *
 * If "path" is NULL, a memory-only circfile is created.
 */
extern struct	circfile *circfile_open(const char *path,
			int flags, uint32_t maxent, uint32_t maxdata);

/*
 * Close a circfile.
 */
extern void	circfile_close(struct circfile **cfp);

/*
 * Get the number of valid entries in a circfile.
 */
extern uint32_t circfile_num_entries(struct circfile *cf);

/*
 * Retrieve a circfile entry.
 *
 * "which" must be a negative number.
 *
 * Entry -1 is the most recent, -2 the second most recent, etc.
 * If "lenp" is not NULL, the length of the entry is returned.
 */
extern const	void *circfile_get(struct circfile *cf, int which, int *lenp);

/*
 * Add an entry.
 */
extern int	circfile_put(struct circfile *cf, const void *data, int len);

/*
 * Trim the number of stored entries.
 */
extern void	circfile_trim(struct circfile *cf, int num);

/*
 * Sync entries to disk.
 */
extern void	circfile_sync(struct circfile *cf);


#ifdef  __cplusplus
}
#endif

#endif	/* _CIRCFILE_H_ */

