
/*
 * Copyright (c) 2004
 * All rights reserved.
 * 
 */

#ifndef _ALOG_H_
#define _ALOG_H_

#include <sys/types.h>
#include <sys/time.h>

#include <stdarg.h>
#include <regex.h>


/*
 * Simple support for logging channels. Each channel can log to
 * standard error, local syslog, or remote syslog, and has a minimum
 * log severity filter.
 */

#define ALOG_MAX_CHANNELS	16	/* max number of channels */

/*
 * This structure is used to configure a channel.
 */
struct alog_config {
   const char     *path;         /* circfile filename, or NULL for none */
   const char     *name;         /* syslog id, or null to disable */
   const char     *facility;     /* syslog facility, null for stderr */
   struct in_addr remote_server; /* remote server, or 0.0.0.0 local */
   int            min_severity;  /* min severity to actually log */
   int            histlen;       /* how many history entries to save */
};

/* Entries in the log history are returned in this form */
struct alog_entry {
   time_t when;   /* when event was logged */
   int    sev;    /* entry log severity */
#if defined(__SUNPRO_CC) || defined(__SUNPRO_C)
   char	  msg[1]; /* entry contents (including NUL) */
#else
   char	  msg[0]; /* entry contents (including NUL) */
#endif
};

/*
 * Use this to get 'elems' declared as the right type
 */
#define DEFINE_STRUCTS_ARRAY(name, etype)                         \
struct name {                                                     \
   u_int   length;         /* number of elements in array */      \
   etype   *elems;         /* array elements */                   \
}

DEFINE_STRUCTS_ARRAY(alog_history, struct alog_entry *);

#ifdef  __cplusplus
extern "C" {
#endif


/*
 * Initialize or reconfigure a logging channel.
 *
 *   channel   Between zero and ALOG_MAX_CHANNELS - 1.
 *   conf      Channel configuration.
 */
extern int     alog_configure(int channel, const struct alog_config *conf);

/*
 * Reset a logging channel.
 */
extern int     alog_shutdown(int channel);

/*
 * Set current logging channel.
 */
extern int     alog_set_channel(int channel);

/*
 * Enable/disable debugging on a channel. Everything logged to the
 * channel will be logged to stderr as well.
 */
extern void    alog_set_debug(int channel, int enabled);

/*
 * Get a selection from the log history.
 *
 * The caller's structs array is filled in and is an array of
 * pointers to struct alog_entry.
 *
 * Caller should free the returned array by calling
 * "structs_free(&alog_history_type, NULL, list)".
 */
extern int     alog_get_history(int channel, int min_severity,
			int max_entries, time_t max_age,
			const regex_t *preg, struct alog_history *list);

/*
 * Clear (i.e., forget) log history.
 */
extern int     alog_clear_history(int channel);

/*
 * Log to the currently active logging channel. Preserves errno.
 */
extern void    alog(int sev, const char *fmt, ...);
extern void    valog(int sev, const char *fmt, va_list args);

/*
 * Convert between numeric syslog facility and string.
 */
extern int	alog_facility(const char *name);
const char	*alog_facility_name(int facility);

/*
 * Convert between numeric syslog severity and string.
 */
extern int	alog_severity(const char *name);
const char	*alog_severity_name(int sev);

/*
 * Expand '%m' in a format string.
 *
 * Returns a pointer to a static buffer.
 */
extern void	alog_expand(const char *fmt,
			int errnum, char *buf, size_t bufsize);


#ifdef  __cplusplus
}
#endif



#endif	/* _ALOG_H_ */
